package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`.G#]F$``````````/``+@(+`@(E`$0```#,````$@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!P`0``!```F9T!``,``(``
M`"`````````0```````````0````````$``````````````0`````!`!`+T!
M````(`$`I`H````P`0!0*P```-```-0!``````````````!@`0`P`````,``
M`!P`````````````````````````````````````````````````````````
MN"(!`&@"```````````````````````````````````N=&5X=````.A"````
M$````$0````$``````````````````!@``!@+F1A=&$```!@`````&`````"
M````2```````````````````0```P"YR9&%T80``7$,```!P````1````$H`
M`````````````````$```$`N8G5I;&1I9#4`````P`````(```".````````
M``````````!```!`+G!D871A``#4`0```-`````"````D```````````````
M````0```0"YX9&%T80``P`$```#@`````@```)(``````````````````$``
M`$`N8G-S`````.`1````\`````````````````````````````"```#`+F5D
M871A``"]`0```!`!```"````E```````````````````0```0"YI9&%T80``
MI`H````@`0``#````)8``````````````````$```,`N<G-R8P```%`K````
M,`$`4"L```"B``````````````````!```#`+G)E;&]C```P`````&`!```"
M````S@``````````````````0```0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0UU0```
MZ,`Z``!%,<`QTC')Z-0Z``!%,<`QTC')Z-@Z``!%,<`QTC')Z-PZ``!%,<`Q
MTC')2(/$*.G<.@``D)"0D)"0D)"0D)"03(T%J4\``#'22(T-$````.G[-P``
M9F8N#Q^$``````##D)"0D)"0D)"0D)"0D)"0055!5%575E-(@^PH28G-2(G7
M2(7)#X2U````2(L%#I\``$B+*$B%[0^$H@````^V`83`=`0\/757,?9,BV4`
M387D=#)(8_9(B>MF#Q]$``!)B?!,B>I,B>'HLCD``(7`=0=!@#PT/71'3(MC
M"$B#PPA-A>1UVDR)X$B#Q"A;7E]=05Q!7<-F+@\?A```````2(G.#[9&`4B#
MQ@&$P'0$/#UU\$0I[NN3#Q^$``````!(*>M-C60T`4C!^P-,B>")'TB#Q"A;
M7E]=05Q!7<-FD$4QY$R)X$B#Q"A;7E]=05Q!7<-F9BX/'X0``````&:0059!
M54%455=64TB![+````!(BQT(G@``2(L#2(F$)*@````QP$F)S$B%R70%@#D`
M=29(BX0DJ````$@K`P^%'`$``$B!Q+````!;7E]=05Q!74%>PP\?`.@#.```
M28G%2(7`=,U(C2TD7@``2(U\)"!,B>GH!S@``$B%P`^$G@```(!X&"Y(C7`8
M=0R`?@$`=-]F#Q]$``"`>!@N#X26````3(GAZ&8X``!(B?%)B<;H6S@``$&-
M3`8"2&/)Z#XW``!)B?%-B>!(B>I)B<9(B<'HZC<``$R)\4B)^NCO-P``3(GQ
M@_C_=!"+1"0P)0#P```]`$```'1?Z%(X``!,B?'HFC8``$R)Z>AR-P``2(7`
M#X5K____9@\?A```````3(GIZ$@V``!,B>'H@#<``.G]_O__#Q\`@'X!+@^%
M8/___X!^`@`/A"7____I4?___P\?@`````#HH_[__^N?Z,PU``"09F8N#Q^$
M``````!!5%-(@>RX````2(L=GYP``$B+`TB)A"2H````,<!(C50D($F)S.A5
M-@``A<!U,8M4)#"!X@#P``"!^@!```!T"('Z`*```'49N@(```!,B>'HBC4`
M`(7`#Y3`#[;`ZP(QP$B+E"2H````2"L3=0M(@<2X````6T%<P^A!-0``D$%4
MN#`0``!55U93Z+`T``!(*<1(BST6G```0;C_#P``2(L'2(F$)"@0```QP$B-
M="0@28G,2(G32(T-9UP``$B)\NA9-@``A<!^38U0_TACZH!\+"``=7B%TG0\
M28GHNB\```!(B?'HM#4``$B%P'0G2(L%B)L``$B+".@P-```28GI28GP3(GB
M2(G!Z*<S``#K(@\?1```2(L%89L``$B+".@)-```28G83(GB2(G!Z(LS``!(
MBX0D*!```$@K!W432('$,!```%M>7UU!7,-(8^CKA^AF-```D`\?1```05=!
M5D%505155U932(/L.$B+'3F;``!(BP-(B40D*#'`,<"`.CU(B=8/E,!(B<](
M`<9(B?'H!C8``$B-5"0D2(GY2(G%Z/;[__])B<1(A<`/A/H```!(B<'HXC4`
M`$@YQ0^&R0```$QC?"0D3(LM_9H``$G!YP,/M@>$P`^$C@$``#P]#X2&`0``
M2(GX#Q\`#[90`4B#P`&$TG0%@/H]=>](*?A(F$V+90!(C4PH`NA\-```30'\
M28D$)$B%P'1028M%`$B-3P%*BQ0X#[8'B`*$P'4<ZQYF#Q^$```````/M@%(
M@\(!2(/!`8@"A,!T!#P]=>O&`CTQP&8/'T0```^V#`:(3`(!2(/``83)=?!(
MBT0D*$@K`P^%!@$``$B#Q#A;7E]=05Q!74%>05_#D#'`9@\?1```#[84!D&(
M%`1(@\`!A-)U\.O&9@\?1```3(LM&9H``$V+=0!)@SX`#X2R````3(GP13'D
M#Q]$``!(@\`(1(GB08/$`4B#.`!U[X/"`TACTDC!X@.+!2_J``"%P'53QP4A
MZ@```0```$B)T>B),P``2(G!2(7`#X1=____36/\3(GR2<'G`TV)^.B;,P``
M28E%`$B)P4&-1"0!1(ED)"1(F$C'!,$`````Z9+^__\/'P!,B?'HX#,``$F)
M10!(B<%(A<`/A!#___]-8_Q)P><#Z\,/'X``````,<#IC_[__[H0````13'D
MZ6?____H-S(``)!F#Q]$``!32(/L,$B+'>28``!(BPOHC#$``$B+"^B$,0``
M3(T-15P``$R-!>8P``!(C15T60``2(G!Z`\Q``!(BPOH7S$``$R-#7E9``#'
M1"0H`````$R-!5D2``!,B4PD($B)P4B-%5Q9``#HU3```)!(@\0P6\-F9BX/
M'X0```````\?`$B#["A(BP5EF```2(L(Z`TQ``!,C0W.6P``3(T%;S```$B)
MP4B-%?I8``!(@\0HZ90P``!F9BX/'X0``````)`QP,-F9BX/'X0``````&:0
M05>XV!```$%6055!5%575E/HRC```$@IQ$B+/3"8``!(BP=(B80DR!```#'`
M2(UL)$!(B<Y)B=5(C0W*6```2(GJZ/GX__^Z+P```$B)\4B)P^BY,@``2(7`
M=$1(B?)(C0VL6```Z)7\__](B?!(BS7;EP``2(N4),@0``!(*Q8/A6<!``!(
M@<38$```6UY?74%<05U!7D%?PV8/'T0``$B)\4B-/758``#HD3(``$R)Z4&)
MQ.AV,@``2(GZ2(G!Z*LR``!)B<9(A<!TCT&`/@!,C2U06```#X2R````9I!(
MA=MT#TR)\DB)V>@P,@``A<!T?$R)\>A$,@``ZP5FD,8"`(G!@^@!2&/03`'R
M@#HO=.U"C40A`3W_#P``#X\Y____2(ET)#`QTDR-O"3`````3(T-Z%<``$R)
M;"0H3(GY0;@`$```3(ET)"#H_R\``$B)ZDR)^>BD,0``A<!U$(M$)%`E`/``
M`#T`@```=%!(B?HQR>CV,0``28G&2(7`#X36_O__08`^``^%4/___TB%VW0%
M@#LN=!9,C35F5P``N0$```#I:/___P\?1```@'L!`'2ZZ^(/'X0``````+H!
M````3(GYZ+,O``"%P'6?3(GZ2(T--U<``.@@^___3(GYZ$@Q``#I@?[__^AN
M+P``D&9F+@\?A```````9I`/M@%)B<B$P'079@\?1```2(/!`3PO#[8!3`]$
MP83`=>],B<##9F8N#Q^$``````!!5$B#["!)B<Q(A<D/A+X```"`.0`/A+4`
M``#H\#```$F-1`3_23G$<@_ILP```)!(@^@!23G$="6`."]T\DDYQ`^#FP``
M`$B#Z`%).<1T#H`X+W0P2(/H`4DYQ'7R08`\)"]T:TB-#8Q6``!(@\0@05SI
MB3```&8/'X0``````(`X+W4)2(/H`4DYQ'+R3"G@13'`2(U0`DB!^@`0``!W
M'$R-0`%!N0`0``!,B>)(C0W;U0``Z(8N``!)B<!,B<!(@\0@05S#9@\?A```
M````2(T-`%8``$B#Q"!!7.D>,```23G$=9[I<O___P\?0`!!5T%6055!5%57
M5E-(@^QH2(LM&94``$B+-3*5``!(NE)?0TQ%04Y?2(T=VU4``$B+10!(B40D
M6$BX7U]%3E9?4$%(B50D.$R-9"0L2+H@("`@("`@($B)5"1(2(E$)#!(N%\@
M("`@("`@2(E$)$`QP&:)1"103(GB2(G9Z)_U__](A<!T.DAC5"0L2(L&2(T$
MT$B+4`A(B1!(A=)TV&:02(M0$$B#P`A(B1!(A=)U\$R)XDB)V>AE]?__2(7`
M=<9(BS6)E```2(T=154``$R)XDB)V>A']?__2(7`=$)(8U0D+$B+!DB-!-!(
MBU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)XDB)V>@%
M]?__2(7`=;Y(BS4IE```2(T=[50``$R)XDB)V>CG]/__2(7`=$)(8U0D+$B+
M!DB-!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)
MXDB)V>BE]/__2(7`=;Y(BS7)DP``2(T=EE0``$R)XDB)V>B']/__2(7`=$)(
M8U0D+$B+!DB-!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(
MA=)U\$R)XDB)V>A%]/__2(7`=;Y(BQUIDP``3(TM/50``$R)XDR)Z>@G]/__
M2(7`=$)(8U0D+$B+`TB-!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#
MP`A(B1!(A=)U\$R)XDR)Z>CE\___2(7`=;Y(BS4)DP``2(T=[5,``$R)XDB)
MV>C'\___2(7`=$)(8U0D+$B+!DB-!-!(BU`(2(D02(72=-AF+@\?A```````
M2(M0$$B#P`A(B1!(A=)U\$R)XDB)V>B%\___2(7`=;Y(BQVID@``3(TU.U,`
M`$R)XDR)\>AG\___2(7`=$)(8U0D+$B+`TB-!-!(BU`(2(D02(72=-AF+@\?
MA```````2(M0$$B#P`A(B1!(A=)U\$R)XDR)\>@E\___2(7`=;Y(BS5)D@``
M2(T=.5,``$R)XDB)V>@'\___2(7`=$)(8U0D+$B+!DB-!-!(BU`(2(D02(72
M=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)XDB)V>C%\O__2(7`=;Y(
MBS7ID0``3(T]XU(``$R)XDR)^>BG\O__2(7`=$)(8U0D+$B+!DB-!-!(BU`(
M2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)XDR)^>AE\O__
M2(7`=;Y(BSV)D0``2(TUC5(``$R)XDB)\>A'\O__2(7`=$)(8U0D+$B+!TB-
M!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U\$R)XDB)
M\>@%\O__2(7`=;Y(BSTID0``2(TUQ%$``$R)XDB)\>CG\?__2(7`=$)(8U0D
M+$B+!TB-!-!(BU`(2(D02(72=-AF+@\?A```````2(M0$$B#P`A(B1!(A=)U
M\$R)XDB)\>BE\?__2(7`=;Y,B>)(C0W;40``Z)'Q__](B<)(A<!T"$R)^>A!
M]?__3(GB2(T-S5$``.AR\?__2(G"2(7`=`Q(C0W+40``Z![U__],B>)(C0W'
M40``Z$_Q__](B<)(A<!T,TR)\>C_]/__2(T5UU$``$R)Z>CP]/__2(M$)%A(
M*T4`=5-(@\1H6UY?74%<05U!7D%?PTR)XDB-#8U1``#H!?'__TB)PDB%P'0*
M2(G9Z+7T___KM$BX4$%27T-,14%(.40D076C9H%\)$E./76:2(U4)$OKU>CM
M*```D&9F+@\?A```````D%-(@^PP2(L=Q(\``$B+`TB)1"0HBP4V0```@_C_
M=!%(BU0D*$@K$W4\2(/$,%O#D$B-5"0D2(T-JU```.A_\/__2(G",<!(A=)T
M$@^V`CPP#Y7"A,`/E<`/ML`AT(D%[3\``.NZZ'8H``"0#Q]$``!(@^PHZ`<I
M``")P>CP*```2(7`=`-(BP!(@\0HPP\?`$%6055!5%932(/L0$B+-2V/``!,
MC2VW4```2(L&2(E$)#@QP$F)S$B-5"0T3(GIZ/SO__](B<-(A<!T!8`X`'4O
M2(M$)#A(*P8/A:$```!,B>),B>E(@\1`6UY!7$%=05[IBO/__V8N#Q^$````
M``!,B>)(B<'HU2D``$B%P'0@2(M$)#A(*P9U9DB#Q$!;7D%<05U!7L-F#Q^$
M``````!,B>'HB"D``$B)V4F)QNA]*0``28U,!@+H8R@``$B)7"0@38G@3(T-
M0T\``$B-%3Y/``!(B<%)B<;H`BD``$B+1"0X2"L&=0A,B?+I8O___^A;)P``
MD&8N#Q^$``````!!5[BH@0``059!54%455=64^BZ)@``2"G$2(L]((X``$R-
M-=).``!(BP=(B80DF($``$B-!;Y.``!(B<Y,C6PD4$R)\4C'A"3@````````
M`$R)ZDC'A"0X`0```````$C'A"0``0```````$B)A"38````2(T%;4\``$B)
MA"08`0``2(T%94\``$B)A"0@`0``2(T%7D\``$B)A"0H`0``2(T%5$\``$B)
MA"0P`0``2(T%24\``$B)A"3X````Z&WN__])B<1(A<!T"8`X``^%;`0``.@7
M)P``B<'H`"<``$B%P`^$UP,``$B+*$B%[0^$RP,``$B)Z>@S*```2(U,``'H
M&2<``$0/MD4`2(G#183`="U)B=],C27J3@``9I!%#[;`3(GY3(GB2(/%`>BM
M)P``1`^V10!)@\<"183`==](C:PD$`$``$B-#4A.``#K%V8/'T0``$B+30A(
M@\4(2(7)#X0'`0``3(GJZ+_M__])B<1(A<!TWX`X`'3:2(G!Z`KP__^%P'3.
M3(GAZ(XG``!(B<=(A<`/A-(```!FD$B)^>B()P``2(G92(G%Z'TG``!$C:0%
M!`0``$UCY$R)X>A:)@``2(E<)"A(C14[3@``28GX28G'2(T%-TX``$R-#3)-
M``!(B40D($R)^>CM)@``NL`!``!,B?GH8"8``(/X_P^$;P(``$R)ZDR)^>C\
M)0``B<.%P`^$P@```$B+/NAZ)```38GY28GX,>U(BT@82(T5)TX``.AR)0``
M2(LU^XL``$B+A"28@0``2"L&#X5E!0``2(GH2('$J($``%M>7UU!7$%=05Y!
M7\,/'P`Q[4R-)6E-``#K'`\?1```2&/%3(NDQ-````!-A>1T0$&`/"0`=#E,
MB>&#Q0'H[N[__X7`=-I,B>'H<B8``$B)QTB%P`^%YO[__TACQ4R+I,30````
M387D=<9F#Q]$```Q_^G)_O__9@\?A```````BT0D8"4`\```/0!````/A2K_
M__^+;"1HZ/,D```YQ0^%&?___XM$)&!(BSXE_P$``#W``0``#X4%____3(GA
MZ/PD``!,B>I(C0VC3```2(G%Z/KK__](B<)(A<`/A#X"``!(B?GHIO+__TB)
MQTB%P`^$*@(``$B)^>C2)0``28G$@_@'?CI(F$B-%41-``!(C4P'^.B7)0``
MA<!U(T&#_`@/A(4#``!!C40D]TB8@#P'+P^$=`,```\?A```````BP4:.P``
M@_C_#X09`P``2(T-H4L``(7`#X34`0``2(T5"DP``$B--15+``#H'^___TR-
M+4!+``!,C27_3```Z.PC``!(B70D*$V)^$B)Z8E$)"!,C0T>2P``2(T5STP`
M`.C:)```ZS@/'X0``````.@+(P``@S@1=3CHL2,``(/#`4B)="0P38GIB5PD
M*$V)^$R)XDB)Z8E$)"#HH"0``+K``0``2(GIZ!,D``"#^/]TODR)^>AF(P``
M3(GQ2(GJZ(ON__](B>GHD_K__^G,_?__9@\?1```Z*,B``"#.!$/A(/]___H
ME2(``$B+'HLHZ`LB``!-B?E)B=A(C15^2P``2(M(&(EL)"`Q[>C_(@``Z8C]
M__]F+@\?A```````,=M(C0T-2P``ZQ4/'T0``$ACPTB+C,3P````2(7)=#E,
MB>J#PP'H1>K__TB%P'3@@#@`=-M(B<'H(R0``$B)Q4B%P`^%[/O__TACPTB+
MC,3P````2(7)=<>Y#0```$B-+;9*``#H]B(``$&X4P```$B)P^G<^___#Q^$
M``````!(B<'HN/G__TB+-?&(``!(BX0DF($``$@K!@^%6P(``$R)X4B!Q*B!
M``!;7E]=05Q!74%>05_II2,```\?1```2(L^Z<[]___'!2XY````````2(GY
MN@```0#HT2(``$B-#;%)``!!B<2%P`^$#?[__T&X`@```$C'PN[___^)P>@[
M(@``2(V<))`!``!!N`8```!$B>%(B=KHLB(``("\))`!````=0Z`O"21`0``
M0P^$40$``$4QP#'21(GAZ/TA``#HV!L``$F)Q>L1#Q\`08G`2(G:3(GIZ"(<
M``!!N`"```!(B=I$B>'H82(``(7`?]U$B>%(C9PD0`$``$B-O"1@`0``Z#4A
M``!,B>I(B=E)B=WH=QT``$B)^TB-M"2(`0``3(TEG4D```\?1```10^V10!(
MB=E,B>)(@\,"28/%`>A8(@``2#G>=>-(C05-2```QH0DB`$```!(B40D*$B)
M?"0@3(T-:T@``$V)^$B-%0I*``!(B>GH(2(``.F._?__#Q]``$B-#8A(``!,
MB>I(B4PD2.A4Z/__2(7`#X2C_O__#[8`A,`/A)C^__\\,`^$D/[__\<%OC<`
M``$```!(BTPD2.FR_/__2(MV"$B%]@^$A_S__TB)\>@/(@``2(/X`P^&=?S_
M_TB-3`;\2(T5A4D``.C4(0``A<!(#T3^Z5G\__^`O"22`0``00^%H?[__X"\
M)),!``!##X63_O__@+PDE`$``$@/A87^__^`O"25`0``10^%=_[__T&X`@``
M`$C'PL;___]$B>'H;"```$&X*0```$B)VD2)X>CK(```2(T%-4<``$B)7"0@
M2(E$)"CIZ_[__^B0'P``D&9F+@\?A```````#Q]``$%6055!5%=64TB#[#A(
MBQTLA@``2(L]588``$B+"TB+!TB)1"0H,<!,C60D).B^'@``2(L+2(G&Z+,>
M```/MI:X````2(T-($<``(B0N0```$R)XN@'Y___2(7`=`F`.#$/A)D!``#H
MU/#__TB+"^A\'@``0;D#````0;@!````2(T5F$@``$B)P>@9'@``2(G&2(7`
M=#-(BT`02(L+3(LHZ$H>``!,C05T2```2(G!3(GJZ,@=``!(BT803(LH0?9%
M#D`/A6X!``!(BPOH'AX``$&Y`P```$&X`0```$B-%4%(``!(B<'HNQT``$B)
MQDB%P'0X2(T--D8``$R)XNA<YO__2(L+28G%2(M&$$V%[0^$^0```$R+,.C1
M'0``38GH2(G!3(GRZ%,=``!(BPOHNQT``$&X`0```$B-%>9'``!(B<'H9AT`
M`$B+"TF)Q>B;'0``3(T%X4<``$R)ZDB)P>@9'0``2(L+Z($=``!(C0VS10``
M3(GBQH"Y`````.C;Y?__2(D%7-4``$B)P4B%P'0\NL`!``#H\AX``(U0`8/Z
M`78)@_@1#X6A````2(M$)"A(*P</A;,```!(@\0X6UY?05Q!74%>PP\?1```
M2(L-&=4``.A4]O__2(D%!=4``$B)P4B%P'6I2(T-5D<``.C!'```9@\?A```
M````@'@!``^$8O[__^E8_O__D$B+%=G4``!,BRA,BS+HSAP``$R)Z4R)\N@3
MZ/__Z?O^__]F#Q]$``!(BPOHL!P``$R)ZDB)P>A5'```Z7K^__]!B<%(BP66
MU```3(L%A]0``$B-#0A'``!(BQ#H2!P``.@+'0``D&8N#Q^$``````!!54%4
M5E-(@^PX2(LUWX,``$B+!DB)1"0H,<!)B<WHK>W__P^V$$F)Q(32=!4/'P!(
M@\`!@/HO#[803`]$X(32=>Z+!2@T``!-A>T/E<.#^/]T;87`#Y7`(<.$VW4B
M2(M$)"A(*P8/A9,```!(@\0X6UY!7$%=PV8/'X0``````$&`?0``=-=!N`0`
M``!(C14R10``3(GAZ%0>``"%P'6^2(M$)"A(*P9U54R)Z4B#Q#A;7D%<05WI
M%.7__P\?0`!(C50D)$B-#3M$``#H#^3__TB%P'0+#[8`A,!T!#PP=0_'!84S
M````````Z6[____'!78S```!````Z5O____H_!L``)"0D)"0D)"0D)"0D$%7
M059!54%455=64TB![+@"``!(BSW&@@``2(L'2(F$)*@"```QP$B)RDR-63A(
MC4PD($B-G"2@````28G*9I!!BP--BPM)@\(028/#"`_(B<!-B<A)B4+P3(G(
M2<'H($C!Z"A)P>DX)0#_``!,"<A-B<%)P>`(2<'A&$&!X```_P!%B<E,"<A,
M"<!)B4+X23G:=:Q,C8D``@``#Q]$``!(BT%H2#-!0$B#P0A(,T$(2#-!^$R-
M!`!(P>@?3`G`@^#_2(E!>$PYR7753(L:2(MR$$B+6B!(BWH83(G83(G92(MJ
M"$R+5"0H2,'I&TC!X`5("<@#1"0@3HV4%YEY@EJ-C!B9>8):2(GP2#'X2"'H
M2#'X1(T,`4B)Z4B)Z$C!Z`)(P>$>3(G+2`G!2(GP2,'K&T&)R$PQP$PAV$@Q
M\$P!T$V)RDG!X@5)"=I&C2003(G838G:2<'J`DC!X!Y-B>5,"=!,BU0D,$G!
M[1M(,<&)PTPAR4Z-E!:9>8):3#'!3`'138GB2<'B!4T)ZDR+;"0X00'*3(G)
M2<'I`DC!X1Y%B=)/C80HF7F"6DP)R4V)U4@QR$&)R4PAX$G![1M(,=A,`<!-
MB=!)P>`%30GH3(ML)$!!`<!,B>!)P>P"2,'@'D6)P$J-G"N9>8):3`G@38G%
M2#'!2<'M&T&)Q$PAT4PQR4@!V4R)PTC!XP5,">M,BVPD2`'+3(G12<'J`DC!
MX1Z)VT^-C"F9>8):3`G128G=2#'(2<'M&T&)RDPAP$PQX$P!R$F)V4G!X05-
M">E,BVPD4$$!P4R)P$G!Z`)(P>`>18G)3XVD+)EY@EI,"<!-B<U(,<%)P>T;
M08G`2"'93#'13`'A38G,2<'D!4T)[$R+;"1800',2(G92,'K`DC!X1Y%B>1/
MC90JF7F"6D@)V4V)Y4@QR$G![1N)RTPAR$PQP$P!T$V)XDG!X@5-">I,BVPD
M8$$!PDR)R$G!Z0)(P>`>18G23XV$*)EY@EI,"<A-B=5(,<%)P>T;08G!3"'A
M2#'93`'!38G02<'@!4T)Z$R+;"1H00'(3(GA2<'L`DC!X1Y%B<!*C9PKF7F"
M6DP)X4V)Q4@QR$G![1M!B<Q,(=!,,<A(`=A,B<-(P>,%3`GK3(ML)'`!PTR)
MT$G!Z@)(P>`>B=M/C8PIF7F"6DP)T$F)W4@QP4G![1M!B<),(<%,,>%,`<E)
MB=E)P>$%30GI3(ML)'A!`<E,B<%)P>@"2,'A'D6)R4^-I"R9>8):3`G!38G-
M2#'(2<'M&T&)R$@AV$PQT$P!X$V)S$G!Y`5-">Q,BZPD@````$$!Q$B)V$C!
MX!Y(P>L"18GD3XV4*IEY@EI("=A-B>5(,<%)P>T;B<-,(<E,,<%,`=%-B>))
MP>(%30GJ3(NL)(@```!!`<I,B<E)P>D"2,'A'D6)TD^-A"B9>8):3`G)38G5
M2#'(2<'M&T&)R4PAX$@QV$P!P$V)T$G!X`5-">A,BZPDD````$$!P$R)X$G!
M[`)(P>`>18G`2HV<*YEY@EI,">!-B<5(,<%)P>T;08G$3"'13#')2`'93(G#
M2,'C!4P)ZTR+K"28`````<M,B=%(P>$>2<'J`HG;3XV,*9EY@EI,"=%)B=U(
M,<A)P>T;08G*3"'`3#'@3`'(28G92<'A!4T)Z4R+K"2@````00'!3(G`2<'H
M`DC!X!Y%B<E/C:0LF7F"6DP)P$V)S4@QP4G![1M!B<!((=E,,=%,`>%-B<Q)
MP>0%30GL1HTT(4B)V4C!ZP)(P>$>2`G92(N<)*@```!(,<A!B<Q,(<A-C90:
MF7F"6DR)\TPQP$C!ZQM,`=!-B?))P>(%20G:2(N<)+````!!`<),B<A(P>`>
M2<'I`D6)TDV-A!B9>8):3`G(3(G32#'!2,'K&T&)P4PA\4PQX4D!R$R)T4C!
MX05("=E,B?-)P>X"2,'C'HG)3`GS20'(2(N,)+@```!(,=A%B<5,(=!)C8P,
MF7F"6DV)[$PQR$G![!M(`<A,B>E(P>$%3`GAB<E(`<A,B=%)P>H"2,'A'D&)
MQ$P)T4F)VHG;23'*13'03(N4),````!/C8P1H>O9;DV)XDT!P4V)X$G!ZAM)
MP>`%30G018G`30'!38GH2<'@'DG![0)%B<I(`YPDR````$T)Z$F)S8G)2`.,
M)-````!-,<5$,>A-B=5(C9P#H>O9;DR)T$G![1M(P>`%3`GHB<!(`<-,B>!)
MP>P"2,'@'D&)W4P)X$V)Q$6)P$P#A"38````23'$13'A38GL3HV,":'KV6Y,
MB>E)P>P;2,'A!4P)X8G)20')3(G12<'J`DC!X1Y%B<Q,"=%)B<*)P$DQRD$Q
MVDR)XT^-E!"AZ]EN38G@2,'K&TG!X`5)"=A%B<!-`<)-B>A)P>T"2<'@'D2)
MTTT)Z$F)S4@#A"3@````B<E-,<5(`XPDZ````$4QZ4F)W4Z-C`BAZ]EN2(G8
M2<'M&TC!X`5,">B)P$D!P4R)X$G![`)(P>`>18G-3`G@38G$18G`3`.$)/``
M``!),<1%,>)-B>Q.C901H>O9;DR)Z4G![!M(P>$%3`GAB<E)`<I(B=E(P>L"
M2,'A'D6)U$@)V4B)PXG`2`.$)/@```!(,<M!,=E,B>-/C8P(H>O9;DV)X$C!
MZQM)P>`%20G818G`30'!38GH2<'M`DG!X!Y$B<M-">A)B<V)R4TQQ4@#C"0`
M`0``13'J28G=3HV4$*'KV6Y(B=A)P>T;2,'@!4P)Z(G`20'"3(G@2<'L`DC!
MX!Y%B=5,">!-B<1%B<!,`X0D"`$``$DQQ$4QX4V)[$Z-C`FAZ]EN3(GI2<'L
M&TC!X05,">&)R4D!R4B)V4C!ZP)(P>$>18G,2`G92(G#B<!(`X0D$`$``$@Q
MRT$QVDR)XT^-E!"AZ]EN38G@2,'K&TG!X`5)"=A%B<!-`<)-B>A)P>T"2<'@
M'D2)TTT)Z$F)S8G)33'%13'I28G=3HV,"*'KV6Y(B=A(P>`%2<'M&T@#C"08
M`0``3`GHB<!)`<%,B>!)P>P"2,'@'D6)S4P)X$V)Q$6)P$P#A"0@`0``23'$
M13'B38GL3HV4$:'KV6Y,B>E)P>P;2,'A!4P)X8G)20'*2(G92,'K`DC!X1Y%
MB=1("=E(B<.)P$@#A"0H`0``2#'+03'93(GC3XV,"*'KV6Y-B>!(P>L;2<'@
M!4D)V$6)P$T!P4V)Z$G![0))P>`>1(G+30GH28G-B<E-,<5%,>I)B=U.C900
MH>O9;DB)V$G![1M(P>`%3`GH2`.,)#`!``")P$D!PDR)X$G![`)(P>`>18G5
M3`G@38G$18G`3`.$)#@!``!),<1%,>%-B>Q.C8P)H>O9;DR)Z4G![!M(P>$%
M3`GAB<E)`<E(B=E(P>L"2,'A'D6)S$@)V4B)PXG`2`.$)$`!``!(,<M!,=I,
MB>-/C900H>O9;DV)X$C!ZQM)P>`%20G818G`30'"38GH2<'M`DG!X!Y$B=--
M">A)B<V)R4TQQ44QZ4F)W4Z-C`BAZ]EN2(G82<'M&TC!X`5,">B)P$D!P4R)
MX$C!X!Y)P>P"18G-2`.,)$@!``!,">!-B<1%B<!,`X0D4`$``$DQQ$4QXDZ-
MI!&AZ]EN3(GI38GJ2<'J&TC!X05,"=&)R4D!S$B)V4C!ZP)(P>$>18GB2`G9
M2(G#B<!(`X0D6`$``$@QRT$QV4R)TT^-C`BAZ]EN38G02,'K&TG!X`5)"=A,
MB>M)P>T"2,'C'D4!P4F)R(G)3`GK18G)23'828G>08G=13'@38G,3HV$`*'K
MV6Y,B<A)P>P;2,'@!4P)X$6-)`!-B=!)P>H"N-R\&X])P>`>30G030G&18G"
M3"'332'.20G>2(N<)&`!``!(`<-(`=E,B>-,`?%-B>9(P>,%2<'N&TP)\P'+
M3(G)2<'I`DC!X1Z)VTP)R4D)R$F)SD&)R4TAX$TAUDT)QDR+A"1H`0``20'`
M30'H28G=30'P28G>2<'E!4G![AM-"?5)B<Y%`<5-B>!)P>P"2<'@'D6)[4T)
MX$6)Q$T)QDDAWDPAX4P)\4R+M"1P`0``20'&30'638GJ3`'Q38GN2<'B!4G!
M[AM-"?)!`<I(B=E(P>L"2,'A'D6)TD@)V4D)R$F)SHG+32'H32'F30G&3(N$
M)'@!``!)`<!-`<A-B=%-`?!-B=9)P>$%2<'N&TT)\4F)SD4!P4V)Z$G![0))
MP>`>18G)30GH18G%30G&32'63"'I3`GQ3(NT)(`!``!)`<9-`>9-B<Q,`?%-
MB<Y)P>0%2<'N&TT)]$$!S$R)T4G!Z@)(P>$>18GD3`G120G(28G.08G*32'(
M32'N30G&3(N$)(@!``!)`<!)`=A,B>--`?!-B>9(P>,%2<'N&TP)\TF)SD0!
MPTV)R$G!Z0))P>`>B=M-"<A%B<%-"<9-(>9,(<E,"?%,B[0DD`$``$D!QDT!
M[DF)W4P!\4F)WDG!Y05)P>X;30GU00'-3(GA2,'A'DG![`)%B>U,">%)"<A)
MB<Y!B<Q)(=A-(<Y-"<9,BX0DF`$``$D!P$T!T$V)ZDT!\$V)[DG!X@5)P>X;
M30GR28G.10'"28G82,'K`DG!X!Y%B=))"=A$B<--"<9-(>Y((=E,"?%,B[0D
MH`$``$D!QDT!SDV)T4P!\4V)UDG!X05)P>X;30GQ00')3(GI2<'M`DC!X1Y%
MB<E,">E)"<A)B<Y!B<U-(=!)(=Y-"<9,BX0DJ`$``$D!P$T!X$V)S$T!\$V)
MSDG!Y`5)P>X;30GT28G.10'$38G02<'J`DG!X!Y%B>1-"=!%B<)-"<9-(<Y,
M(=%,"?%,B[0DL`$``$D!QDD!WDR)XTP!\4V)YDC!XP5)P>X;3`GS`<M,B<E)
MP>D"2,'A'HG;3`G)20G(28G.08G)32'@32'630G&3(N$)+@!``!)`<!-`>A)
MB=U-`?!)B=Y)P>4%2<'N&TT)]4F)SD4!Q4V)X$G![`))P>`>18GM30G@18G$
M30G&22'>3"'A3`GQ3(NT),`!``!)`<9-`=9-B>I,`?%-B>Y)P>(%2<'N&TT)
M\D$!RDB)V4C!ZP)(P>$>18G22`G920G(28G.B<M-(>A-(>9-"<9,BX0DR`$`
M`$D!P$T!R$V)T4T!\$V)UDG!X05)P>X;30GQ28G.10'!38GH2<'M`DG!X!Y%
MB<E-">A%B<5-"<9-(=9,(>E,"?%,B[0DT`$``$D!QDT!YDV)S$P!\4V)SDG!
MY`5)P>X;30GT00',3(G12<'J`DC!X1Y%B>1,"=%)"<A)B<Y!B<I-(<A-(>Y-
M"<9,BX0DV`$``$D!P$D!V$R)XTT!\$V)YDC!XP5)P>X;3`GS28G.1`'#38G(
M2<'I`DG!X!Z)VTT)R$T)QD6)P4TAYDPAR4P)\4R+M"3@`0``20'&30'N28G=
M3`'Q28G>2<'E!4G![AM-"?5!`<U,B>%)P>P"2,'A'D6)[4P)X4D)R$F)S$&)
MSTDAV$TAS$T)Q$R+A"3H`0``20'`30'038GJ30'@38GL2<'B!4G![!M-">)%
M`<))B=A(P>L"2<'@'D6)TDD)V$B)RT6)Q$P)PTPAZTPAX4@)V4B+G"3P`0``
M2`'#2`.$)/@!``!)`=E,B=-,`?A,B[PD&`(``$D!R4R)T4C!ZQM(P>$%2`G9
MB<E)`<E,B>E)P>T"2,'A'D2)RTP)Z4D)R$F)S4&)SDTAT$TAY4T)Q4V-1`4`
M2(G828G=2<'M&TC!X`5,">B)P$D!P$R)T$G!Z@)(P>`>18G%20G"N-;!8LI,
M,=%$,<E,BXPD``(``$D!P4T!X4V)[$P!R4V)Z4G![!M)P>$%30GA28G<18G)
M3`')2<'D'DD!QTC!ZP)!B<E,">--B=1%B=)),=Q%,<1,BX0D"`(``$D!P$T!
M\$V)SDT!X$V)S$G![AM)P>0%30GT38GN2<'M`DG!YAY%B>1-"?5,B[0D$`(`
M`$T!X$6)Q$D!QDT!\DF)WDTQ[D$QSDR)X4T!\DV)YDC!X05)P>X;3`GQB<E)
M`<I,B<E)P>D"2,'A'D6)UDD)R8G93(GK3#'+3`'Y3(N\)"`"``!$,<-,C009
M3(GQ3(GS2,'A!4C!ZQM)`<=("=F)R4D!R$R)X4G![`)(P>$>1(G#3`GA18GL
M38G-18G)23'-30'\3(N\)#@"``!%,=5/C10L28G<28G=2<'M&TG!Y`5-">Q-
MB?5)P>X"2<'E'D6)Y$T)[DR+K"0H`@``30'B18G420'%30'I28G-B<E-,?5%
M,<5-B>!-`>E-B>5)P>`%2<'M&TT)Z$6)P$T!P4F)V$C!ZP))P>`>18G-20G8
M2(N<)#`"``!(`<-(`=E,B?-,,<-$,=-)`<=,C1093(GI3(GK2,'K&TC!X05(
M"=F)R4D!RDR)X4G![`)(P>$>1(G320G,1(GQ38G&18G`33'F3`'Y3(N\)$@"
M``!%,<Y.C0PQ2(G928G>20''2<'N&TC!X05,"?&)R4D!R4R)Z4G![0)(P>$>
M18G.3`GI3(NL)$`"``!)`<5-`>A-B>5),<U%,=5/C10H38GP38GU2<'M&TG!
MX`5-">A%B<!-`<))B=A(P>L"2<'@'D6)U4P)PT6)X$F)S(G)30'X23'<3(N\
M)%@"``!%,<Q/C0P@38GH38GL20''2<'L&TG!X`5-">!%B<!-`<%-B?!)P>X"
M2<'@'D6)S$T)\$R+M"10`@``20'&3`'Q28G>33'&13'63HT4,4R)X4V)YDG!
M[AM(P>$%3`GQB<E)`<I,B>E)P>T"2,'A'D6)UDD)S8G93(G#18G`3#'K3`'Y
M3(N\)&@"``!$,<M,C0P93(GQ3(GS2,'K&TC!X05("=F)R4D!R4R)X4G![`)(
MP>$>1(G+3`GA3(ND)&`"``!)`<1-`>!-B>Q)`<=),<Q%,=1/C10@28G828G<
M2<'L&TG!X`5-">!%B<!-`<)-B?!)P>X"2<'@'D6)U$T)QD6)Z$F)S8G)33'U
M30'X3(N\)'@"``!%,<U/C0PH38G@38GE2<'M&TG!X`5-">A%B<!-`<%)B=A(
MP>L"2<'@'D6)S4D)V$B+G"1P`@``2`'#2`'93(GS3#'#1#'33(T4&4R)Z4R)
MZTC!ZQM(P>$%2`G9B<E)`<I,B>%)P>P"2,'A'D2)TTD)S$2)\4V)QDD!QTTQ
MYDP!^46)P$R+O"2(`@``13'.28G920'.2(G92<'I&TD!QTC!X05,"<F)R4D!
MSDR)Z4G![0)(P>$>18GQ3`GI3(NL)(`"``!)`<5-`>A-B>5),<U%,=5/C10H
M38G(38G-2<'M&TG!X`5-">A%B<!-`<))B=A(P>L"2<'@'D6)U4P)PT6)X$F)
MS(G)23'<30'X13'T38GN30'@38GL2<'N&TG!Y`5-"?1-B<Y%B>1-`>!)P>8>
M2<'I`D6)Q$T)\4R+M"20`@``20'&2`.$))@"``!,`?%)B=Y,`=B)VTTQSD@!
MV$4QUDV)XDP!\4V)YDG!X@5)P>X;30GR1`'138GJ2<'M`DG!XAZ)R4T)U4V)
MRD0#2B!-,>I!`?U,B4H@13'028G*3(EJ&$P!P$F)R$G!ZAL!Z4G!X`5(B4H(
M30G01`'`2(D"3(G@2<'L`DC!X!Y,">`!\$B+-9AL``!(B4(02(N$)*@"``!(
M*P9U%$B!Q+@"``!;7E]=05Q!74%>05_#Z'X%``"09F8N#Q^$``````!FD$B#
M["BY@````.@R!@``NHFKS>^Y_MRZF$B)4`A(@>J9R?HK2,<``2-%9TB)2!!(
MQT`8=E0R$$B)4"!(QT`H`````$C'0#``````QT!X`````$B#Q"C#9BX/'X0`
M`````$%505155U932(/L*$ECP$R+02A(B<](B=-(BU<P28G$2(UW.$&-#,!,
M.<%(B4\H2&-/>$B#T@!(P>@=2`'02(E',(7)=$B]0````$B)VBG-1#GE00]/
M[$@!\4QC[4V)Z.B;!0``BT=X`>B)1WB#^$!T#DB#Q"A;7E]=05Q!7<.02(GY
M02GL3`'KZ)+H__]!@_P_#X[8````08/L0$6)Y4'![09!C6T!2,'E!D@!W6:0
M\P]O`TB)^4B#PT`/$0;S#V]+T`\13A#S#V]3X`\15B#S#V];\`\17C#H0NC_
M_T@YZW7-0<'E!D4I[$ECQ$B#^`AR74B+50!(C4X(2(/A^$B)5SA(BU0%^$B)
M5`;X2"G.2`'P2"GU2(/@^$B#^`AR&$B#X/@QTDR+1!4`3(D$$4B#P@A(.<)R
M[D2)9WA(@\0H6UY?74%<05W##Q^``````*@$=21(A<!TWP^V50"(5SBH`G34
M#[=4!?YFB50&_NO(2(G=Z7+___^+50")5SB+5`7\B50&_.NP#Q^$``````!!
M5%575E-(@^P@2(M"*(MR*(MZ,$C!Z`,/SH/@/P_/2(G+C4@!2)A)B=3&1`(X
M@$B-:CA(8\%(`>B#^3@/CDX!``!!N$````!!*<AT$#'2B=&#P@'&!`@`1#G"
M<O),B>'H*^?__TG'1"0X`````$G'1"1``````$C'11``````2,=%&`````!(
MQT4@`````$C'12@`````2,=%,`````!!B7PD<$R)X4&)="1TZ-_F__])BP0D
M3(GA2,'H&(@#28L$)$C!Z!"(0P%)BP0DB&,"28L$)(A#`TF+1"0(2,'H&(A#
M!$F+1"0(2,'H$(A#!4F+1"0(B&,&28M$)`B(0P=)BT0D$$C!Z!B(0PA)BT0D
M$$C!Z!"(0PE)BT0D$(AC"DF+1"00B$,+28M$)!A(P>@8B$,,28M$)!A(P>@0
MB$,-28M$)!B(8PY)BT0D&(A##TF+1"0@2,'H&(A#$$F+1"0@2,'H$(A#$4F+
M1"0@B&,228M$)""(0Q-(@\0@6UY?74%<Z6H"``!F+@\?A```````NC@```!%
M,<DIRH/Z"'))B=%(QP``````2,=$"/@`````2(U("$B#X?A(*<@!PH/B^(/Z
M"`^"U?[__X/B^#'`08G`@\`(3HD,`3G0<O+IO?[__V8/'T0``/;"!'4CA=(/
MA*K^___&``#VP@(/A)[^__\QR6:)3!#^Z9+^__\/'P#'``````#'1!#\````
M`.E\_O__D)"0D)"0D)"0D)"0D/\EJMT``)"0_R6:W0``D)#_)8K=``"0D/\E
M>MT``)"0_R5JW0``D)#_)5K=``"0D/\E2MT``)"0_R4ZW0``D)#_)2K=``"0
MD/\E&MT``)"0_R4*W0``D)#_)?K<``"0D/\EZMP``)"0_R7:W```D)#_)<K<
M``"0D/\ENMP``)"0_R6JW```D)#_)9K<``"0D/\EBMP``)"0_R5ZW```D)#_
M)=+:``"0D)"0D)"0D)"0_R7"VP``D)"0D)"0D)"0D/\ENML``)"0D)"0D)"0
MD)!14$@]`!```$B-3"08<AE(@>D`$```2(,)`$@M`!```$@]`!```'?G2"G!
M2(,)`%A9PY"0D)"0D)"0D)"0D)"0_R52V@``D)"0D)"0D)"0D/\E2MH``)"0
MD)"0D)"0D)#_)4K:``"0D)"0D)"0D)"0_R5"V@``D)"0D)"0D)"0D/\E.MH`
M`)"0D)"0D)"0D)#_)3K:``"0D)"0D)"0D)"0_R5"V@``D)"0D)"0D)"0D/\E
M0MH``)"0D)"0D)"0D)#_)3K:``"0D)"0D)"0D)"0_R52V@``D)"0D)"0D)"0
MD/\E2MH``)"0D)"0D)"0D)#_)4+:``"0D)"0D)"0D)"0_R4ZV@``D)"0D)"0
MD)"0D/\E,MH``)"0D)"0D)"0D)#_)2K:``"0D)"0D)"0D)"0_R4BV@``D)"0
MD)"0D)"0D/\E&MH``)"0D)"0D)"0D)#_)1+:``"0D)"0D)"0D)"0_R4*V@``
MD)"0D)"0D)"0D/\E`MH``)"0D)"0D)"0D)#_)?K9``"0D)"0D)"0D)"0_R7R
MV0``D)"0D)"0D)"0D/\EZMD``)"0D)"0D)"0D)#_)>+9``"0D)"0D)"0D)"0
M_R7RV0``D)"0D)"0D)"0D/\EZMD``)"0D)"0D)"0D)#_)>+9``"0D)"0D)"0
MD)"0_R7:V0``D)"0D)"0D)"0D/\ETMD``)"0D)"0D)"0D)#_)<K9``"0D)"0
MD)"0D)"0_R7"V0``D)"0D)"0D)"0D/\ENMD``)"0D)"0D)"0D)#_);+9``"0
MD)"0D)"0D)"0_R6JV0``D)"0D)"0D)"0D/\EHMD``)"0D)"0D)"0D)#_)9K9
M``"0D)"0D)"0D)"0_R62V0``D)"0D)"0D)"0D/\EBMD``)"0D)"0D)"0D)#_
M)8+9``"0D)"0D)"0D)"02(/L*#'2Z%4```#_%1?8``"0D)"0D)"0D)"0D)"0
MD)##D)"0D)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0
MD)"0D)##D)"0D)"0D)"0D)"0D)"05E-(@^PH2(G.2(G3N0@```#HR@0``$B%
MVP^$$0(``$BZ`````%4!``!(B7-`2(TUC/___TB-#54'``!(B9/P````2(T5
M7P<``$B)4U!(BQ5\UP``2(FS@````$B--6[___](B4M(2(T-<____TB)4QA(
MC15XM```2(FSB````$B--6K___](B8N0````2+E(`0``NPL``$B)LY@```!(
MC36;_?__2(E3.$B-%3#]__](B4L(2(T-)?[__TB)<R!(C37Z`P``2(E3*$B-
M%1\$``!(QP,`````QT,0`P```$B)2S!(B7-X2(F3&`$``$B%P`^$O0```$B+
M@.````!(A<`/A*T```"+%?.S``"%T@^%GP```$B#/=,3````#X0-`@``2(,]
MS1,````/A&\"``!(@SW'$P````^$40(``$B#/<$3````#X0S`@``2(,]NQ,`
M```/A!4"``!(@SVU$P````^$]P$``$B#/:\3````#X39`0``2(,]J1,````/
MA+L!``!F#V\%8Q,```\1`&8/;PUH$P``#Q%($&8/;Q5L$P``#Q%0(&8/;QUP
M$P``#Q%8,#')Z(4#```QR4B)@^@```!(C04%$P``2(E#6$B-!5H3``!(B4-@
M2(T%[Z(``$B)0VA(C07$M```2(E#<$B-!?EE``!(B8,@`0``2(T%)V8``$B)
M@R@!``!(C06]LO__2(F#,`$``.B1`@``N`$```!(@\0H6U[##Q]$``!(B7!`
M2(TUA?W__TBZ`````%4!``!(C0U<!0``2(FP@````$B--7;]__](B9#P````
M2(T5*`4``$B)L(@```!(C35J_?__2(E02$B+%4_5``!(B4A02(N(0`$``$B)
ML)````!(C356_?__2(FPF````$B--4BR``!(B0I(C16.^___2(T-)_O__TB)
M<#A(OD@!``"["P``2(EP"$B--0[\__](B5`@2(T5XP$``$B)2"A(C0T(`@``
M2,<``````,=`$`,```!(B7`P2(E0>$B)B!@!``!(A<`/A*;^__](B<-(BX#@
M````2(7`#X7F_?__Z8[^__]F+@\?A```````2(L02(D5MA$``.GD_?__D$B+
M4#A(B17=$0``Z37^__](BU`P2(D5Q1$``.D7_O__2(M0*$B)%:T1``#I^?W_
M_TB+4"!(B165$0``Z=O]__](BU`82(D5?1$``.F]_?__2(M0$$B)%641``#I
MG_W__TB+4`A(B15-$0``Z8']__],BP59L0``BQ5;L0``2(L-7+$``.G7````
M#Q^``````$B#[#B#^@)T=W=5A=(/A'T````QP$V%P$B-%<'___](B0TJL0``
M#Y3`3(D%$+$``,<%#K$```$```")!?"P``#HZP```$B#^/](B07HL```#Y7`
M#[;`2(/$.,,/'T0``+@!````@_H#=>RZ`P```$B#Q#CI6`````\?A```````
MN@(```!(@\0XZ4(```!FD#'2Z#D```"%P'2[2(L-EK```(E$)"SH50```(M$
M)"Q(QP5^L```_____^N:#Q]``.D[____D)"0D)"0D)"0D)"X`0```,.0D)"0
MD)"0D)"0PY"0D)"0D)"0D)"0D)"0D/\E&M,``)"0D)"0D)"0D)#_)2+3``"0
MD)"0D)"0D)"0_R4:TP``D)"0D)"0D)"0D/\EFM,``)"0D)"0D)"0D)!!54%4
M2(/L*$R-+2&P``!)B<Q(B=%,B>KHX_K__TR)ZDR)X4B#Q"A!7$%=Z0````#_
M)=+2``"0D)"0D)"0D)"0_R6:U```D)`/'X0``````$%505164TB#[%B)C"2`
M````2(E4)#A,B40D,.C@]___2(U4)#A(BQWD7@``3(U$)#!(C8PD@````$B+
M`TB)1"1(,<#HR/;__TB+%;%>``!(BPV:7@``3(T%`_?__^@>]___2(L%9UX`
M`$B+-7!>``"`.`!U,^A^]O__2(D%%Z\``$B)P4B%P`^$C`$``.A>]O__2(LU
M1UX``$B+#NCO]O__QH`A"@```$B+#NC@]O__2(L.@(AR!````NC1]O__@(AR
M!````8N$)(````"-2`2-<`-(8\E(P>$#Z$GV__],BU0D.$F)P4B)!:*N``!)
MBP))B0%(C056(0``28E!"$B+!9(.``!)B4$02(T%0R$``$F)01B+A"2`````
M@_@!#X[J````28/""(/``[H$````#Q^``````$V+`DACRH/"`4F#P@A-B03)
M.=!UZTB82(L-0JX``$&)\$B-%9#$__])QP3!`````$C'1"0@`````.AB]?__
M08G$A<!T3DB+#12N``!,C6PD1.A:]?__3(GJ2(T-@QX``.ACOO__2(7`=#I(
MBPWOK0``Z#+U___H5?7__TB+1"1(2"L#=69$B>!(@\186UY!7$%=PTB+#<:M
M``#H^?3__T&)Q.NA2(L-I:T``$B%R70'Z#/9___KLTB-#=,=``!,B>KH`K[_
M_TB)!8.M``!(B<%(A<!TE>O9N`0```#I-/___[D!````Z$WV___H^/7__Y"0
MD)"0D)"0Z9N]__^0D)"0D)"0D)"0D/__________L%)```$`````````````
M`/__________````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````````/__
M__\`````Z')```$`````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````E
M<R\E<P`O<')O8R]S96QF+V5X90!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D
M97(``$EN=&5R;F%L<SHZ4$%2.CI"3T]4`"X`4$%27U1%35``4$%27U!23T=.
M04U%`#H`)7,E<R5S`"\`4$523#5,24(`4$523$Q)0@!015),-4]05`!015),
M24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@!005)?
M1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"54<`4$%27T=,3T)!3%]4
M35!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#
M3$5!3@!005)?0TQ%04X]`#$`4$%42`!54T52`%-94U1%30`O=&UP`%1-4$1)
M4@!414U01$E2`%1%35``5$U0`%5315).04U%`"4P,G@`)7,E<R5S)7,`<&%R
M+0```````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E
M<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@<')I=F%T92!S=6)D:7)E
M8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R
M>2!Y;W5R(&]P97)A=&EO;BD*`'!A<FPN97AE`"YP87(`)7,E<V-A8VAE+25S
M)7,`)7,E<W1E;7`M)74E<P`E<R5S=&5M<"TE=2TE=25S`!@`<&5R;``P`%!!
M4CHZ4&%C:V5R.CI615)324].`#$N,#4T````````56YA8FQE('1O(&-R96%T
M92!C86-H92!D:7)E8W1O<GD``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV
M871E(&-A8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`
M+64`+2T``'!E<FQX<VDN8P````````!E=F%L("=E>&5C("]U<W(O8FEN+W!E
M<FP@("U3("0P("1[,2LB)$`B?2<*:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E
M<B!S;VUE('-H96QL"G!A8VMA9V4@7U]P87)?<&P["FUY("@D4$%27TU!1TE#
M+"`D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I;&4L("5-;V1U;&5#86-H
M92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q=6ER92!&:6QE
M.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U:7)E($9I;&4Z
M.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D
M<&%R7W1E;7`I.PIO=71S*'%Q6U)E;6]V:6YG(&9I;&5S(&EN("(D<&%R7W1E
M;7`B72D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD("D@/R!R
M;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D<&%R7W1E;7`[
M"FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R)RD@>PIM>2`D
M=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%(#T^("=T;7!86%A86"<L
M"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER*2P*4U5&
M1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD["FUY("1F:6QE;F%M92`]
M("1T;7`M/F9I;&5N86UE.PIP<FEN="`D=&UP(#P\(BXN+B(["G@],3L@=VAI
M;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A<E]T96UP)PII9B!;
M(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E86L*9FD*<VQE97`@,0IX
M/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R1F:6QE;F%M92<*+BXN"F-L;W-E
M("1T;7`["F-H;6]D(#`W,#`L("1F:6QE;F%M93L*;7D@)&-M9"`]("(D9FEL
M96YA;64@/B]D978O;G5L;"`R/B8Q("8B.PIS>7-T96TH)&-M9"D["F]U=',H
M<7%;4W!A=VYE9"!B86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R9F]R;2!C;&5A
M;G5P.B`D9FEL96YA;65=*3L*?0I]('T*0D5'24X@>PHD4$%27TU!1TE#(#T@
M(EQN4$%2+G!M7&XB.PI);G1E<FYA;',Z.E!!4CHZ0D]/5"@I(&EF(&1E9FEN
M960@)DEN=&5R;F%L<SHZ4$%2.CI"3T]4.PIE=F%L('L*7W!A<E]I;FET7V5N
M=B@I.PIM>2`D<75I970@/2`A)$5.5GM005)?1$5"54=].PIM>2`E<WES(#T@
M*`IP871H7W-E<"`@("`]/B`H)%Y/(#U^("]>35-7:6XO(#\@)SLG(#H@)SHG
M*2P*7V5X92`@("`@("`@/3X@*"1>3R`]?B`O7B@_.DU35VEN?$]3,GQC>6=W
M:6XI+R`_("<N97AE)R`Z("<G*2P*7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O
M7DU35VEN?$]3,B\@/R`G7%PG(#H@)R\G*2P**3L*7W-E=%]P<F]G;F%M92@I
M.PIO=71S*'%Q6UPD<')O9VYA;64@/2`B)'!R;V=N86UE(ETI.PI?<V5T7W!A
M<E]T96UP*"D["F]U=',H<7%;7"1%3E9[4$%27U1%35!](#T@(B1%3E9[4$%2
M7U1%35!](ETI.PIM>2`H)'-T87)T7W!O<RP@)&1A=&%?<&]S*3L*34%'24,Z
M('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E
M;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E
M860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O;B=T('5S92`D(2!H
M97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@34%'24,["GT*;7D@
M)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@
M+7,@7T9(.PIM>2`D;6%G:6-?<&]S(#T@)'-I>F4@+2`D<VEZ92`E("1C:'5N
M:U]S:7IE.R`C($Y/5$4Z("1M86=I8U]P;W,@:7,@82!M=6QT:7!L92!O9B`D
M8VAU;FM?<VEZ90IW:&EL92`H)&UA9VEC7W!O<R`^/2`P*2!["G-E96L@7T9(
M+"`D;6%G:6-?<&]S+"`P.PIR96%D(%]&2"P@)&)U9BP@)&-H=6YK7W-I>F4@
M*R!L96YG=&@H)%!!4E]-04=)0RD["FEF("@H;7D@)&D@/2!R:6YD97@H)&)U
M9BP@)%!!4E]-04=)0RDI(#X](#`I('L*)&UA9VEC7W!O<R`K/2`D:3L*;&%S
M=#L*?0HD;6%G:6-?<&]S("T]("1C:'5N:U]S:7IE.PI]"FEF("@D;6%G:6-?
M<&]S(#P@,"D@>PIO=71S*'%Q6T-A;B=T(&9I;F0@;6%G:6,@<W1R:6YG("(D
M4$%27TU!1TE#(B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'24,[
M"GT*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`T+"`P.PIR96%D(%]&2"P@)&)U
M9BP@-#L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`T("T@=6YP86-K*").(BP@
M)&)U9BDL(#`["B1D871A7W!O<R`]('1E;&P@7T9(.PIM>2`E<F5Q=6ER95]L
M:7-T.PIR96%D(%]&2"P@)&)U9BP@-#L@("`@("`@("`@("`@("`@("`@("`@
M("`@("`C(')E860@=&AE(&9I<G-T(")&24Q%(@IW:&EL92`H)&)U9B!E<2`B
M1DE,12(I('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U
M;G!A8VLH(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*;W5T<RAQ
M<5M5;G!A8VMI;F<@1DE,12`B)&9U;&QN86UE(BXN+ETI.PIM>2`D8W)C(#T@
M*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z('5N
M9&5F.PIM>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\*#\Z+BHO
M*3\H+BHI*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@
M)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E>'0I(&%N
M9"`D97AT("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`D9FEL96YA
M;64@/2!?<V%V95]A<R@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*)%!!4CHZ
M2&5A=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE.PHD4$%2
M.CI(96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA;64["GT*
M96QS:68@*"`D9G5L;&YA;64@/7X@;7Q>+S]S:&QI8B]\(&%N9"!D969I;F5D
M("1%3E9[4$%27U1%35!]("D@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B
M)&)A<V5N86UE)&5X="(L("1B=68L(#`W-34I.PIO=71S*")32$Q)0CH@)&9I
M;&5N86UE7&XB*3L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@
M/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA
M;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(
M+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D
M;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U
M;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@
M=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*
M)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H
M)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@
M>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG
M="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M
M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N
M("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O
M+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D
M9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F
M:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO
M9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR
M97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I
M<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A
M9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@
M/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.
M0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H
M)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64[
M"GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S
M:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[
M)&9I;&5N86UE?3L*?0IU;FQE<W,@*"1B=68@97$@(E!+7#`P,UPP,#0B*2![
M"F]U=',H<7%;3F\@>FEP(&9O=6YD(&%F=&5R($9)3$4@<V5C=&EO;B!I;B!F
M:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'24,@.PI]"B1S=&%R=%]P;W,@
M/2`H=&5L;"!?1D@I("T@-#L@("`@("`@("`@("`@("`@(R!S=&%R="!O9B!Z
M:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T+"`D8G5N9&QE+"`D;&]G9F@L
M("1C86-H95]N86UE*3L*9&5L971E("1%3E9[4$%27T%04%]2155317T[(",@
M<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y('!R;V)L96TI"B1Q
M=6EE="`](#`@=6YL97-S("1%3E9[4$%27T1%0E5'?3L*:68@*"$D<W1A<G1?
M<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I
M*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN
M<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@
M=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E
M=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H
M:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM
M>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>
M+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G
M22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-
M)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS
M:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U
M="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E
M)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE
M;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q
M("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T
M(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I
M('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D
M8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G
M?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES
M=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B
M?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D
M7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*
M<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*
M;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@
M)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;
M0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G
M24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M
M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII
M<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H
M:79E("(D<&%R(ET["GT*;7D@)65N=B`](&1O('L*:68@*"1Z:7`@86YD(&UY
M("1M971A(#T@)'II<"T^8V]N=&5N=',H)TU%5$$N>6UL)RDI('L*)&UE=&$@
M/7X@<R\N*EYP87(Z)"\O;7,["B1M971A(#U^(',O7EQ3+BHO+VUS.PHD;65T
M82`]?B`O7B`@*%M>.ETK*3H@*"XK*20O;6<["GT*?3L*:68@*&1E9FEN960@
M)'!A<BD@>PIO<&5N(&UY("1P:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;
M0V%N)W0@<F5A9"!P87(@9FEL92`B)'!A<B(Z("0A73L*;7D@)&)U9CL*<F5A
M9"`D<&@L("1B=68L(#0["F1I92!Q<5LB)'!A<B(@:7,@;F]T(&$@<&%R(&9I
M;&5=('5N;&5S<R`D8G5F(&5Q(")02UPP,#-<,#`T(CL*8VQO<V4@)'!H.PI]
M"D-R96%T95!A=&@H)&]U="D@.PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`HD
M;W5T+`I)3SHZ1FEL93HZ3U]#4D5!5"@I('P@24\Z.D9I;&4Z.D]?5U)/3DQ9
M*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#
M86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D[
M"G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D9&%T
M85]P;W,I('L*<F5A9"!?1D@L("1L;V%D97(L("1D871A7W!O<SL*?2!E;'-E
M('L*;&]C86P@)"\@/2!U;F1E9CL*)&QO861E<B`](#Q?1D@^.PI]"FEF("@A
M)$5.5GM005)?5D520D%424U](&%N9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R
M96TI+RD@>PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*4$%2.CI&
M:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&QO861E<BP@)#`I.PI]"F9O<F5A
M8V@@;7D@)&ME>2`H<V]R="!K97ES("5E;G8I('L*;7D@)'9A;"`]("1E;G9[
M)&ME>7T@;W(@;F5X=#L*)'9A;"`](&5V86P@)'9A;"!I9B`D=F%L(#U^("]>
M6R<B72\["FUY("1M86=I8R`](")?7T5.5E]005)?(B`N('5C*"1K97DI("X@
M(E]?(CL*;7D@)'-E="`](")005)?(B`N('5C*"1K97DI("X@(CTD=F%L(CL*
M)&QO861E<B`]?B!S>R1M86=I8R@@*RE]>PHD;6%G:6,@+B`D<V5T("X@*"<@
M)R!X("AL96YG=&@H)#$I("T@;&5N9W1H*"1S970I*2D*?65G.PI]"B1F:"T^
M<')I;G0H)&QO861E<BD["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(
M96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN
M8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)3
M1%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@
M)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG
M.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D
M?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@
M0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*
M;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@
M("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E
M86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D
M;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL
M0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD[
M"GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]
M?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K
M97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK
M+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L*
M*"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B
M)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D[
M"FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X
M="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A
M;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H
M<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^
M>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I
M*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D
M+R`]('5N9&5F.PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE
M('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD9F@^
M.PIC;&]S92`D9F@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C
M;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"
M051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&
M:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F
M:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L
M96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E
M.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G
M3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D["F]U=',H<7%;5W)I
M='1E;B!A<R`B)&YA;64B72D["GT*?0HD>FEP+3YW<FET951O1FEL94AA;F1L
M92@D9F@I(&EF("1Z:7`["B1C86-H95]N86UE(#T@<W5B<W1R("1C86-H95]N
M86UE+"`P+"`T,#L*:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@)&UT:6UE(#T@
M*'-T870H)&]U="DI6SE=*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W
M*#$I.PIO<&5N(&UY("1F:"P@(CPZ<F%W(BP@)&]U=#L*)&-T>"T^861D9FEL
M92@D9F@I.PIC;&]S92`D9F@["B1C86-H95]N86UE(#T@)&-T>"T^:&5X9&EG
M97-T.PI]"B1C86-H95]N86UE("X](")<,"(@>"`H-#$@+2!L96YG=&@@)&-A
M8VAE7VYA;64I.PHD8V%C:&5?;F%M92`N/2`B0T%#2$4B.PHD9F@M/G!R:6YT
M*"1C86-H95]N86UE*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&9H+3YT96QL
M("T@;&5N9W1H*"1L;V%D97(I*2D["B1F:"T^<')I;G0H)%!!4E]-04=)0RD[
M"B1F:"T^8VQO<V4["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T
M('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*
M<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*
M>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]
M"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@
M("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N
M9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF
M9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I
M>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z
M:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P<F]G;F%M92D@/3T@07)C
M:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@
M>FEP(&%R8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H
M=6YK4VEZ92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP
M.PHD4$%2.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*)'%U:65T(#T@
M(21%3E9[4$%27T1%0E5'?3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI
M('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@
M)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^
M:7-$:7)E8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.
M86UE.PIN97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB
M+PHH/SHD0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\**#\Z)$-O;F9I9SHZ
M0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T
M7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I
M;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E
M<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H
M*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET
M(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO
M=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA
M;64B72D["B1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N86UE
M*3L*8VAM;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*
M?0I]"GT*?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*
M9&EE(#P\("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A
M<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G
M<F%M+G!L(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD
M14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!
M4D=6*3L*?0IS=6(@0W)E871E4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q
M=6ER92!&:6QE.CI"87-E;F%M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E
M>'0I(#T@1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ
M)RDI.PIR97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D
M<&%T:"D@=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R
M<F]R"GT*<W5B(')E<75I<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U
M:7)E($1Y;F%,;V%D97(["G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I
M8W0["G)E<75I<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#
M87)P.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U
M:7)E($5X<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER
M92!&8VYT;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P
M96,["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E
M($E/.CI(86YD;&4["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E
M<W,Z.EIL:6(["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S
M=#HZ4TA!.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U
M:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`[
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q=6ER92!A
M='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*979A;"![(')E<75I
M<F4@5VEN,S(@?3L*979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL('T["F5V
M86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE=F%L('L@
M<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!].PIE=F%L('L@<F5Q
M=6ER92!097)L24\[(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*979A;"![
M(')E<75I<F4@=71F."!].PI]"G-U8B!?<V5T7W!A<E]T96UP('L*:68@*&1E
M9FEN960@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H
M+BLI+RD@>PHD<&%R7W1E;7`@/2`D,3L*<F5T=7)N.PI]"F9O<F5A8V@@;7D@
M)'!A=&@@*`HH;6%P("1%3E9[)%]]+"!Q=R@@4$%27U1-4$1)4B!435!$25(@
M5$5-4$1)4B!414U0(%1-4"`I*2P*<7<H($,Z7%Q414U0("]T;7`@+B`I"BD@
M>PIN97AT('5N;&5S<R!D969I;F5D("1P871H(&%N9"`M9"`D<&%T:"!A;F0@
M+7<@)'!A=&@["FUY("1U<V5R;F%M93L*;7D@)'!W=6ED.PIE=F%L('LH)'!W
M=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D969I;F5D("0^.WT["FEF("@@9&5F
M:6YE9"@F5VEN,S(Z.DQO9VEN3F%M92D@*2!["B1U<V5R;F%M92`]("97:6XS
M,CHZ3&]G:6Y.86UE.PI]"F5L<VEF("AD969I;F5D("1P=W5I9"D@>PHD=7-E
M<FYA;64@/2`D<'=U:60["GT*96QS92!["B1U<V5R;F%M92`]("1%3E9[55-%
M4DY!345]('Q\("1%3E9[55-%4GT@?'P@)U-94U1%32<["GT*)'5S97)N86UE
M(#U^(',O7%<O7R]G.PIM>2`D<W1M<&1I<B`]("(D<&%T:"1S>7-[7V1E;&EM
M?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA;64I.PIM:V1I<B`D<W1M<&1I
M<BP@,#<U-3L*:68@*"$D14Y6>U!!4E]#3$5!3GT@86YD(&UY("1M=&EM92`]
M("AS=&%T*"1P<F]G;F%M92DI6SE=*2!["F]P96X@;7D@)&9H+"`B/#IR87<B
M+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z
M("0A73L*<V5E:R`D9F@L("TQ."P@,CL*;7D@)&)U9CL*<F5A9"`D9F@L("1B
M=68L(#8["FEF("@D8G5F(&5Q(")<,$-!0TA%(BD@>PIS965K("1F:"P@+34X
M+"`R.PIR96%D("1F:"P@)&)U9BP@-#$["B1B=68@/7X@<R]<,"\O9SL*)'-T
M;7!D:7(@+CT@(B1S>7-[7V1E;&EM?6-A8VAE+21B=68B.PI]"F5L<V4@>PIM
M>2`D9&EG97-T(#T@979A;`I["G)E<75I<F4@1&EG97-T.CI32$$["FUY("1C
M='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIO<&5N(&UY("1F:"P@(CPZ<F%W
M(BP@)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D<')O9VYA;64B
M.B`D(5T["B1C='@M/F%D9&9I;&4H)&9H*3L*8VQO<V4H)&9H*3L*)&-T>"T^
M:&5X9&EG97-T.PI]('Q\("1M=&EM93L*)'-T;7!D:7(@+CT@(B1S>7-[7V1E
M;&EM?6-A8VAE+21D:6=E<W0B.PI]"F-L;W-E*"1F:"D["GT*96QS92!["B1%
M3E9[4$%27T-,14%.?2`](#$["B1S=&UP9&ER("X]("(D<WES>U]D96QI;7UT
M96UP+20D(CL*?0HD14Y6>U!!4E]414U0?2`]("1S=&UP9&ER.PIM:V1I<B`D
M<W1M<&1I<BP@,#<U-3L*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6
M>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U
M8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`
M7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H
M+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B
M.PIO<&5N(&UY("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE(&]R(&1I92!Q<5M#
M86XG="!W<FET92`B)'1E;7!N86UE(CH@)"%=.PIP<FEN="`D9F@@)&-O;G1E
M;G1S.PIC;&]S92`D9F@["F-H;6]D("1M;V1E+"`D=&5M<&YA;64@:68@9&5F
M:6YE9"`D;6]D93L*<F5N86UE*"1T96UP;F%M92P@)&9U;&QN86UE*2!O<B!U
M;FQI;FLH)'1E;7!N86UE*3L*?0IR971U<FX@)&9U;&QN86UE.PI]"G-U8B!?
M<V5T7W!R;V=N86UE('L*:68@*&1E9FEN960@)$5.5GM005)?4%)/1TY!345]
M(&%N9"`D14Y6>U!!4E]04D]'3D%-17T@/7X@+R@N*RDO*2!["B1P<F]G;F%M
M92`]("0Q.PI]"B1P<F]G;F%M92!\?#T@)#`["FEF("@D14Y6>U!!4E]414U0
M?2!A;F0@:6YD97@H)'!R;V=N86UE+"`D14Y6>U!!4E]414U0?2D@/CT@,"D@
M>PHD<')O9VYA;64@/2!S=6)S='(H)'!R;V=N86UE+"!R:6YD97@H)'!R;V=N
M86UE+"`D<WES>U]D96QI;7TI("L@,2D["GT*:68@*"$D14Y6>U!!4E]04D]'
M3D%-17T@;W(@:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#X](#`I
M('L*:68@*&]P96X@;7D@)&9H+"`G/"<L("1P<F]G;F%M92D@>PIR971U<FX@
M:68@+7,@)&9H.PI]"FEF("@M<R`B)'!R;V=N86UE)'-Y<WM?97AE?2(I('L*
M)'!R;V=N86UE("X]("1S>7-[7V5X97T["G)E='5R;CL*?0I]"F9O<F5A8V@@
M;7D@)&1I<B`H<W!L:70@+UQ1)'-Y<WMP871H7W-E<'U<12\L("1%3E9[4$%4
M2'TI('L*;F5X="!I9B!E>&ES=',@)$5.5GM005)?5$5-4'T@86YD("1D:7(@
M97$@)$5.5GM005)?5$5-4'T["B1D:7(@/7X@<R]<421S>7-[7V1E;&EM?5Q%
M)"\O.PHH*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE
M)'-Y<WM?97AE?2(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P
M<F]G;F%M921S>7-[7V5X97TB.PHH*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?
M9&5L:6U])'!R;V=N86UE(BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L
M:6U])'!R;V=N86UE(CL*?0I]"G-U8B!?9FEX7W!R;V=N86UE('L*)#`@/2`D
M<')O9VYA;64@?'P]("1%3E9[4$%27U!23T=.04U%?3L*:68@*&EN9&5X*"1P
M<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`\(#`I('L*)'!R;V=N86UE(#T@(BXD
M<WES>U]D96QI;7TD<')O9VYA;64B.PI]"FUY("1P=V0@/2`H9&5F:6YE9"`F
M0W=D.CIG971C=V0I(#\@0W=D.CIG971C=V0H*0HZ("@H9&5F:6YE9"`F5VEN
M,S(Z.D=E=$-W9"D@/R!7:6XS,CHZ1V5T0W=D*"D@.B!@<'=D8"D["F-H;VUP
M*"1P=V0I.PHD<')O9VYA;64@/7X@<R]>*#\]7"Y<+C]<421S>7-[7V1E;&EM
M?5Q%*2\D<'=D)'-Y<WM?9&5L:6U]+SL*)$5.5GM005)?4%)/1TY!345](#T@
M)'!R;V=N86UE.PI]"G-U8B!?<&%R7VEN:71?96YV('L*:68@*"`D14Y6>U!!
M4E])3DE424%,25I%1'TK*R`]/2`Q("D@>PIR971U<FX["GT@96QS92!["B1%
M3E9[4$%27TE.251)04Q)6D5$?2`](#(["GT*9F]R("AQ=R@@4U!!5TY%1"!4
M14U0($-,14%.($1%0E5'($-!0TA%(%!23T=.04U%("D@*2!["F1E;&5T92`D
M14Y6>R=005)?)RXD7WT["GT*9F]R("AQ=R\@5$U01$E2(%1%35`@0TQ%04X@
M1$5"54<@+RD@>PHD14Y6>R=005)?)RXD7WT@/2`D14Y6>R=005)?1TQ/0D%,
M7R<N)%]](&EF(&5X:7-T<R`D14Y6>R=005)?1TQ/0D%,7R<N)%]].PI]"FUY
M("1P87)?8VQE86X@/2`B7U]%3E9?4$%27T-,14%.7U\@("`@("`@("`@("`@
M("`B.PII9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,
M14%.?3L*?0IE;'-I9B`H(65X:7-T<R`D14Y6>U!!4E]'3$]"04Q?0TQ%04Y]
M*2!["FUY("1V86QU92`]('-U8G-T<B@D<&%R7V-L96%N+"`Q,B`K(&QE;F=T
M:"@B0TQ%04XB*2D["B1%3E9[4$%27T-,14%.?2`]("0Q(&EF("1V86QU92`]
M?B`O7E!!4E]#3$5!3CTH7%,K*2\["GT*?0IS=6(@;W5T<R!["G)E='5R;B!I
M9B`D<75I970["FEF("@D;&]G9F@I('L*<')I;G0@)&QO9V9H(")`7UQN(CL*
M?0IE;'-E('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B(&EN:71?:6YC('L*<F5Q
M=6ER92!#;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D969I;F5D+"!M87`@)$-O
M;F9I9SHZ0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E>'`@<')I=FQI8F5X<"!S
M:71E87)C:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R8VAE>'`@=F5N9&]R;&EB
M97AP"BD["GT*<&%C:V%G92!M86EN.PIR97%U:7)E(%!!4CL*=6YS:&EF="!`
M24Y#+"!<)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T*$!P87)?87)G<RD[
M"F1I92!Q<5MP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S8W)I<'0@(B1P<F]G
M;F%M92(Z($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<;ET*=6YL97-S("UE
M("1P<F]G;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%.CIE>&ET*"0Q*2!I9B`H
M)$`@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)$`@:68@)$`["GT[
M"B0Z.E]?15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%.CIE>&ET*"0Q*2!I9B`H
M)#HZ7U]%4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D.CI?
M7T524D]2(&EF("0Z.E]?15)23U(["C$["E]?14Y$7U\*````X$\`````````
M`````````%@D00`!``````````````!@)$$``0``````````````Z$=```$`
M`````````````.!'0``!``````````````#H(D$``0``````````````@$=`
M``$``````````````$`C00`!``````````````!'0T,Z("A'3E4I(#$Q+C(N
M,"`R,#(Q,#<R."`H1F5D;W)A($-Y9W=I;B`Q,2XR+C`M,BD`````````````
M````1T-#.B`H1TY5*2`Q,2XR+C````````````````````!'0T,Z("A'3E4I
M(#$Q+C(N,````````````````````$=#0SH@*$=.52D@,3$N,BXP````````
M````````````1T-#.B`H1TY5*2`Q,2XR+C`@,C`R,3`W,C@@*$9E9&]R82!#
M>6=W:6X@,3$N,BXP+3(I`````````````````$=#0SH@*$=.52D@,3$N,BXP
M(#(P,C$P-S(X("A&961O<F$@0WEG=VEN(#$Q+C(N,"TR*0``````````````
M``!'0T,Z("A'3E4I(#$Q+C(N,"`R,#(Q,#<R."`H1F5D;W)A($-Y9W=I;B`Q
M,2XR+C`M,BD`````````````````1T-#.B`H1TY5*2`Q,2XR+C`@,C`R,3`W
M,C@@*$9E9&]R82!#>6=W:6X@,3$N,BXP+3(I`````````````````$=#0SH@
M*$=.52D@,3$N,BXP(#(P,C$P-S(X("A&961O<F$@0WEG=VEN(#$Q+C(N,"TR
M*0````````````````!'0T,Z("A'3E4I(#$Q+C(N,"`R,#(Q,#<R."`H1F5D
M;W)A($-Y9W=I;B`Q,2XR+C`M,BD`````````````````1T-#.B`H1TY5*2`Q
M,2XR+C`@,C`R,3`W,C@@*$9E9&]R82!#>6=W:6X@,3$N,BXP+3(I````````
M`````````$=#0SH@*$=.52D@,3$N,BXP(#(P,C$P-S(X("A&961O<F$@0WEG
M=VEN(#$Q+C(N,"TR*0````````````````!'0T,Z("A'3E4I(#$Q+C(N,"`R
M,#(Q,#<R."`H1F5D;W)A($-Y9W=I;B`Q,2XR+C`M,BD`````````````````
M1T-#.B`H1TY5*2`Q,2XR+C`@,C`R,3`W,C@@*$9E9&]R82!#>6=W:6X@,3$N
M,BXP+3(I`````````````````$=#0SH@*$=.52D@,3$N,BXP````````````
M````````1T-#.B`H1TY5*2`Q,2XR+C``````````````````````````````
M``$```!8)`$`4*\``$````!@)`$`8*\``$````#H(@$`D*\``$````!`(P$`
ML*\``$``````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````(````9````',```!R.``!24T13\8&.9?9)+E8GF5:5KOX+Y0$`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````!```$00````X```4!```&40```(X```<!```'$0```,X```
M@!```&,1```0X```<!$``-42```DX```X!(``&`3```\X```8!,``#L4``!(
MX```0!0``&H6``!<X```<!8``.(6``!TX```\!8``"07``!\X```,!<``#,7
M``"$X```0!<``#,9``"(X```0!D``&49``"@X```<!D``%P:``"DX```8!H`
M`+0?``"LX```P!\``"L@``#$X```,"```$T@``#,X```4"```$8A``#4X```
M4"$``!$I``#DX```("D``)8K``#\X```H"L``*4L```0X0``L"P``"-#```P
MX0``,$,``(9#``!(X0``D$,``!A%``!0X0``($4``$-'``!DX0``T$H``.)*
M``!TX0``\$H``/%*``!\X0```$L```%+``"`X0``$$L``!%+``"$X0``($L`
M`"%+``"(X0``,$L``.!.``",X0``X$X``/E.``"8X0```$\``+Q/``"<X0``
MP$\``,5/``"DX0``T$\``-9/``"HX0``X$\``.%/``"LX0``,%```&!0``"P
MX0``@%```*E2```@X0``L%(``+52``"\X0``````````````````````````
M```````````````````````````````````!!`$`!$(```$````!`````0P'
M``Q""#`'8`9P!5`$P`+0```!$0D`$0$6``HP"6`(<`=0!L`$T`+@```!"@0`
M"@$7``,P`L`!$P<`$P$&`@LP"F`)<`A0`L````$0"0`08@PP"V`*<`E0",`&
MT`3@`O````$%`@`%4@$P`00!``1"```!`````1D*`!D!&P(1,!!@#W`.4`W`
M"]`)X`+P`0````$&`@`&,@+``1`)`!#"##`+8`IP"5`(P`;0!.`"\````04"
M``52`3`!!`$`!$(```$,!@`,<@@P!V`&P`30`N`!&0H`&0$U$!$P$&`/<`Y0
M#<`+T`G@`O`!#0<`#6(),`A@!W`&P`30`N````$*!0`*8@8P!6`$P`+0```!
M"@4`"J(&,`5@!,`"T````1,*`!,!5P`,,`M@"G`)4`C`!M`$X`+P`00!``1"
M```!#`<`#$((,`=@!G`%4`3``M````$*!@`*,@8P!6`$<`-0`L`!!`$`!$(`
M``$````!`````0````$````!!@,`!D(",`%@```!`````00!``1B```!````
M`0````$````!"`,`"$($P`+0```!````````````````````````````````
M````````````````````````````````````````````````````````````
M``#IP_9A`````-(0`0`!````$0```!$````H$`$`;!`!`+`0`0`@*0``,"``
M`(!0```88```0!D``*`K```P%P``<!D``,`?``!`%P``8!H``%`A``!0(```
M($4``#!#``"00P``\!8``-H0`0#P$`$`"Q$!`!`1`0`:$0$`)Q$!`#,1`0!$
M$0$`4!$!`%X1`0!K$0$`>!$!`(41`0"7$0$`H1$!`*H1`0"U$0$````!``(`
M`P`$``4`!@`'``@`"0`*``L`#``-``X`#P`0`'!A<BYE>&4`6%-?26YT97)N
M86QS7U!!4E]"3T]4`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`&UA:6X`
M;7E?<&%R7W!L`'!A<E]B87-E;F%M90!P87)?8VQE86YU<`!P87)?8W5R<F5N
M=%]E>&5C`'!A<E]D:7)N86UE`'!A<E]E;G9?8VQE86X`<&%R7V9I;F1P<F]G
M`'!A<E]I;FET7V5N=@!P87)?;6MT;7!D:7(`<&%R7W-E='5P7VQI8G!A=&@`
M<VAA7V9I;F%L`'-H85]I;FET`'-H85]U<&1A=&4`>'-?:6YI=```````````
M````````````````````````````````````````````````````````````
M````````````````````4"`!````````````'"H!`+@B`0#P(0$`````````
M``"`*@$`6"0!`*@B`0```````````)0J`0`0)0$`````````````````````
M```````@)0$``````#`E`0``````/"4!``````!,)0$``````%@E`0``````
M:"4!``````!\)0$``````)`E`0``````H"4!``````"L)0$``````+PE`0``
M````R"4!``````#4)0$``````-PE`0``````Z"4!``````#\)0$``````!`F
M`0``````("8!```````L)@$``````#0F`0``````0"8!``````!()@$`````
M`%0F`0``````8"8!``````!L)@$``````'0F`0``````?"8!``````"()@$`
M`````)0F`0``````H"8!``````"L)@$``````+0F`0``````O"8!``````#(
M)@$``````-PF`0``````\"8!```````()P$``````!`G`0``````'"<!````
M```H)P$``````#0G`0``````/"<!``````!()P$``````%`G`0``````7"<!
M``````!H)P$``````'0G`0``````@"<!``````",)P$``````)@G`0``````
MI"<!`````````````````+`G`0``````P"<!``````#0)P$``````.0G`0``
M````_"<!```````4*`$``````"0H`0``````."@!``````!(*`$``````%@H
M`0``````;"@!``````"$*`$``````)0H`0``````J"@!``````"\*`$`````
M`,PH`0``````X"@!``````#P*`$```````0I`0``````%"D!```````@*0$`
M`````#`I`0`````````````````\*0$`````````````````("4!```````P
M)0$``````#PE`0``````3"4!``````!8)0$``````&@E`0``````?"4!````
M``"0)0$``````*`E`0``````K"4!``````"\)0$``````,@E`0``````U"4!
M``````#<)0$``````.@E`0``````_"4!```````0)@$``````"`F`0``````
M+"8!```````T)@$``````$`F`0``````2"8!``````!4)@$``````&`F`0``
M````;"8!``````!T)@$``````'PF`0``````B"8!``````"4)@$``````*`F
M`0``````K"8!``````"T)@$``````+PF`0``````R"8!``````#<)@$`````
M`/`F`0``````""<!```````0)P$``````!PG`0``````*"<!```````T)P$`
M`````#PG`0``````2"<!``````!0)P$``````%PG`0``````:"<!``````!T
M)P$``````(`G`0``````C"<!``````"8)P$``````*0G`0``````````````
M``"P)P$``````,`G`0``````T"<!``````#D)P$``````/PG`0``````%"@!
M```````D*`$``````#@H`0``````2"@!``````!8*`$``````&PH`0``````
MA"@!``````"4*`$``````*@H`0``````O"@!``````#,*`$``````.`H`0``
M````\"@!```````$*0$``````!0I`0``````("D!```````P*0$`````````
M````````/"D!``````````````````\`7U]C>&%?871E>&ET```6`%]?97)R
M;F\````C`%]?9V5T<F5E;G0`````+@!?7VUA:6X`````3`!?7W-P<FEN=&9?
M8VAK`$\`7U]S=&%C:U]C:&M?9F%I;```40!?7W-T86-K7V-H:U]G=6%R9`!7
M`%]?<W1R;F-P>5]C:&L`;`!?9&QL7V-R=#``<P!?:6UP=7)E7W!T<@```(,`
M86-C97-S``````(!8V%L;&]C`````$,!8VQO<V4`1`%C;&]S961I<@``?`%C
M>6=W:6Y?9&5T86-H7V1L;`!^`6-Y9W=I;E]I;G1E<FYA;````)H!9&QL7V1L
M;&-R=#````"W`65N=FER;VX```#4`65X:70``#X"9G!R:6YT9@```$H"9G)E
M90``L`)G971P:60`````NP)G971P=W5I9```R0)G971U:60`````C0-L<V5E
M:P"/`VQS=&%T`)0#;6%L;&]C`````*,#;65M8VAR`````*4#;65M8W!Y````
M`*<#;65M;6]V90```*L#;6MD:7(`[0-O<&5N``#Q`V]P96YD:7(````$!'!O
M<VEX7VUE;6%L:6=N`````"D$<'1H<F5A9%]A=&9O<FL`````7P1P=&AR96%D
M7V=E='-P96-I9FEC````MP1R96%D``"X!')E861D:7(```"Z!')E861L:6YK
M``"]!')E86QL;V,```#K!')M9&ER`(`%<W!R:6YT9@```(@%<W1A=```DP5S
M=')C:'(`````E05S=')C;7``````F@5S=')D=7``````I`5S=')L96X`````
MJ05S=')N8VUP````M05S=')S='(`````NP5S=')T;VL`````+09U;FQI;FL`
M````%P!03%]D;U]U;F1U;7```&T`4$Q?=&AR7VME>0````!K`5!E<FQ?871F
M;W)K7VQO8VL``&P!4&5R;%]A=&9O<FM?=6YL;V-K`````.4!4&5R;%]C<F]A
M:U]N;V-O;G1E>'0``'@"4&5R;%]G971?<W8```"G`E!E<FQ?9W9?9F5T8VAP
M=@```&D#4&5R;%]M9U]S970```#?`U!E<FQ?;F5W6%,`````X0-097)L7VYE
M=UA37V9L86=S``"G!5!E<FQ?<V%F97-Y<VUA;&QO8P````"&!E!E<FQ?<W9?
M<V5T<'8`CP9097)L7W-V7W-E='!V;@````"[!E!E<FQ?<WES7VEN:70S````
M`+P&4&5R;%]S>7-?=&5R;0`Y!V)O;W1?1'EN84QO861E<@```$H'<&5R;%]A
M;&QO8P````!,!W!E<FQ?8V]N<W1R=6-T`````$T'<&5R;%]D97-T<G5C=`!.
M!W!E<FQ?9G)E90!/!W!E<FQ?<&%R<V4`````4`=P97)L7W)U;@``BP)'971-
M;V1U;&5(86YD;&5!````(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!
M```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$`
M`"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``
M(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@
M`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0!C>6=W:6XQ
M+F1L;``4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`
M%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4
M(`$`8WEG<&5R;#5?,S(N9&QL`"@@`0!+15).14PS,BYD;&P`````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````$``,`
M```P``"`#@```,```(`0````\```@!@````@`0"````````````````````$
M``$```!@``"``@```'@``(`#````D```@`0```"H``"`````````````````
M```!``D$``!0`0`````````````````````!``D$``!@`0``````````````
M```````!``D$``!P`0`````````````````````!``D$``"``0``````````
M`````````0```,`!`(#8``"````````````````````!``D$``"0`0``````
M```````````````!``$````(`0"````````````````````!``D$``"@`0``
M```````````````````!``$````X`0"````````````````````!``D$``"P
M`0``T#$!`&@%`````````````#@W`0"H"`````````````#@/P$`:`4`````
M````````2$4!`*@(`````````````/!-`0`^```````````````P3@$`6`(`
M````````````B%`!`-H%``````````````8`5P!)`$X`10!8`$4````H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\```$`
M!``0$````0`(`&@%```!`"`@```!``@`J`@```(`$!````$`"`!H!0```P`@
M(````0`(`*@(```$````6`(T````5@!3`%\`5@!%`%(`4P!)`$\`3@!?`$D`
M3@!&`$\``````+T$[_X```$``````````````````````#\`````````!``$
M``$```````````````````"X`0```0!3`'0`<@!I`&X`9P!&`&D`;`!E`$D`
M;@!F`&\```"4`0```0`P`#``,``P`#``-`!"`#`````D``(``0!#`&\`;0!P
M`&$`;@!Y`$X`80!M`&4``````"`````L``(``0!&`&D`;`!E`$0`90!S`&,`
M<@!I`'``=`!I`&\`;@``````(````#``"``!`$8`:0!L`&4`5@!E`'(`<P!I
M`&\`;@``````,``N`#``+@`P`"X`,````"0``@`!`$D`;@!T`&4`<@!N`&$`
M;`!.`&$`;0!E````(````"@``@`!`$P`90!G`&$`;`!#`&\`<`!Y`'(`:0!G
M`&@`=````"`````L``(``0!,`&4`9P!A`&P`5`!R`&$`9`!E`&T`80!R`&L`
M<P``````(````"P``@`!`$\`<@!I`&<`:0!N`&$`;`!&`&D`;`!E`&X`80!M
M`&4````@````)``"``$`4`!R`&\`9`!U`&,`=`!.`&$`;0!E```````@````
M-``(``$`4`!R`&\`9`!U`&,`=`!6`&4`<@!S`&D`;P!N````,``N`#``+@`P
M`"X`,````$0````!`%8`80!R`$8`:0!L`&4`20!N`&8`;P``````)``$````
M5`!R`&$`;@!S`&P`80!T`&D`;P!N`````````+`$/#]X;6P@=F5R<VEO;CTB
M,2XP(B!E;F-O9&EN9STB551&+3@B('-T86YD86QO;F4](GEE<R(_/@T*/&%S
M<V5M8FQY('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV
M,2(@;6%N:69E<W1697)S:6]N/2(Q+C`B/@T*("`@(#QA<W-E;6)L>4ED96YT
M:71Y('9E<G-I;VX](C`N,"XP+C`B('1Y<&4](G=I;C,R(B!N86UE/2)005(M
M4&%C:V5R+D%P<&QI8V%T:6]N(B`O/@T*("`@(#QD97-C<FEP=&EO;CY005(M
M4&%C:V5R($%P<&QI8V%T:6]N/"]D97-C<FEP=&EO;CX-"B`@("`\9&5P96YD
M96YC>3X-"B`@("`@("`@/&1E<&5N9&5N=$%S<V5M8FQY/@T*("`@("`@("`@
M("`@/&%S<V5M8FQY261E;G1I='D@='EP93TB=VEN,S(B(&YA;64](DUI8W)O
M<V]F="Y7:6YD;W=S+D-O;6UO;BU#;VYT<F]L<R(@=F5R<VEO;CTB-BXP+C`N
M,"(@<'5B;&EC2V5Y5&]K96X](C8U.35B-C0Q-#1C8V8Q9&8B(&QA;F=U86=E
M/2(J(B!P<F]C97-S;W)!<F-H:71E8W1U<F4](BHB("\^#0H@("`@("`@(#PO
M9&5P96YD96YT07-S96UB;'D^#0H@("`@/"]D97!E;F1E;F-Y/@T*("`@(#QT
M<G5S=$EN9F\@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O<V]F="UC;VTZ87-M
M+G8S(CX-"B`@("`@("`@/'-E8W5R:71Y/@T*("`@("`@("`@("`@/')E<75E
M<W1E9%!R:79I;&5G97,^#0H@("`@("`@("`@("`@("`@/')E<75E<W1E9$5X
M96-U=&EO;DQE=F5L(&QE=F5L/2)A<TEN=F]K97(B('5I06-C97-S/2)F86QS
M92(@+SX-"B`@("`@("`@("`@(#PO<F5Q=65S=&5D4')I=FEL96=E<SX-"B`@
M("`@("`@/"]S96-U<FET>3X-"B`@("`\+W1R=7-T26YF;SX-"B`@("`\8V]M
M<&%T:6)I;&ET>2!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IC
M;VUP871I8FEL:71Y+G8Q(CX-"B`@("`@("`@/&%P<&QI8V%T:6]N/@T*("`@
M("`@("`@("`@/"$M+2!7:6YD;W=S(#@N,2`M+3X-"B`@("`@("`@("`@(#QS
M=7!P;W)T961/4R!)9#TB>S%F-C<V8S<V+3@P93$M-#(S.2TY-6)B+3@S9#!F
M-F0P9&$W.'TB+SX-"B`@("`@("`@("`@(#PA+2T@5VEN9&]W<R!6:7-T82`M
M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>V4R,#$Q-#4W+3$U
M-#8M-#-C-2UA-69E+3`P.&1E964S9#-F,'TB+SX-"B`@("`@("`@("`@(#PA
M+2T@5VEN9&]W<R`W("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED
M/2)[,S4Q,SAB.6$M-60Y-BTT9F)D+3AE,F0M83(T-#`R,C5F.3-A?2(O/@T*
M("`@("`@("`@("`@/"$M+2!7:6YD;W=S(#@@+2T^#0H@("`@("`@("`@("`\
M<W5P<&]R=&5D3U,@260](GLT83)F,CAE,RTU,V(Y+30T-#$M8F$Y8RUD-CED
M-&$T839E,SA](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@,3`@+2T^
M#0H@("`@("`@("`@("`\<W5P<&]R=&5D3U,@260](GLX93!F-V$Q,BUB9F(S
M+31F93@M8CEA-2TT.&9D-3!A,35A.6%](B\^#0H@("`@("`@(#PO87!P;&EC
M871I;VX^#0H@("`@/"]C;VUP871I8FEL:71Y/@T*/"]A<W-E;6)L>3X-"@``
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````4```
M#````,BB````8```#````!B@````H```&````%"O8*]PKX"OD*^@K["O````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
-````````````````````
