package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M``?='V```````````/``+@(+`@(C`$0```#,````$@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!P`0``!```='T!``,``(``
M`"`````````0```````````0````````$``````````````0`````!`!`+T!
M````(`$`I`H````P`0!0*P```-```.`!``````````````!@`0`P`````,``
M`!P`````````````````````````````````````````````````````````
MN"(!`&@"```````````````````````````````````N=&5X=````.A"````
M$````$0````$``````````````````!@`%!@+F1A=&$```!@`````&`````"
M````2```````````````````0`!@P"YR9&%T80``C$(```!P````1````$H`
M`````````````````$``4$`N8G5I;&1I9#4`````P`````(```".````````
M``````````!``#!`+G!D871A``#@`0```-`````"````D```````````````
M````0``P0"YX9&%T80``N`$```#@`````@```)(``````````````````$``
M,$`N8G-S`````.`1````\`````````````````````````````"``&#`+F5D
M871A``"]`0```!`!```"````E```````````````````0``P0"YI9&%T80``
MI`H````@`0``#````)8``````````````````$``,,`N<G-R8P```%`K````
M,`$`4"L```"B``````````````````!``##`+G)E;&]C```P`````&`!```"
M````S@``````````````````0``P0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0UU0```
MZ,`Z``!%,<`QTC')Z-0Z``!%,<`QTC')Z-@Z``!%,<`QTC')Z-PZ``!%,<`Q
MTC')2(/$*.G<.@``D)"0D)"0D)"0D)"03(T%J4\``#'22(T-$````.G[-P``
M9F8N#Q^$``````##D)"0D)"0D)"0D)"0D)"0055!5%575E-(@^PH28G-2(G7
M2(7)#X2U````2(L%/IX``$B+*$B%[0^$H@````^V`83`=`0\/757,?9,BV4`
M387D=#)(8_9(B>MF#Q]$``!)B?!,B>I,B>'HLCD``(7`=0=!@#PT/71'3(MC
M"$B#PPA-A>1UVDR)X$B#Q"A;7E]=05Q!7<-F+@\?A```````2(G.#[9&`4B#
MQ@&$P'0$/#UU\$0I[NN3#Q^$``````!(*>M-C60T`4C!^P-,B>")'TB#Q"A;
M7E]=05Q!7<-FD$4QY$R)X$B#Q"A;7E]=05Q!7<-F9BX/'X0``````&:0059!
M54%455=64TB![+````!(BQTXG0``2(L#2(F$)*@````QP$F)S$B%R70%@#D`
M=29(BX0DJ````$@K`P^%'`$``$B!Q+````!;7E]=05Q!74%>PP\?`.@#.```
M28G%2(7`=,U(C2TD7@``2(U\)"!,B>GH!S@``$B%P`^$G@```(!X&"Y(C7`8
M=0R`?@$`=-]F#Q]$``"`>!@N#X26````3(GAZ&8X``!(B?%)B<;H6S@``$&-
M3`8"2&/)Z#XW``!)B?%-B>!(B>I)B<9(B<'HZC<``$R)\4B)^NCO-P``3(GQ
M@_C_=!"+1"0P)0#P```]`$```'1?Z%(X``!,B?'HFC8``$R)Z>AR-P``2(7`
M#X5K____9@\?A```````3(GIZ$@V``!,B>'H@#<``.G]_O__#Q\`@'X!+@^%
M8/___X!^`@`/A"7____I4?___P\?@`````#HH_[__^N?Z,PU``"09F8N#Q^$
M``````!!5%-(@>RX````2(L=SYL``$B+`TB)A"2H````,<!(C50D($F)S.A5
M-@``A<!U,8M4)#"!X@#P``"!^@!```!T"('Z`*```'49N@(```!,B>'HBC4`
M`(7`#Y3`#[;`ZP(QP$B+C"2H````2"L+=0M(@<2X````6T%<P^A!-0``D$%4
M2(/L($F)S.@2-P``28U$!/]).<1R$>F>````#Q\`2(/H`4DYQ'0E@#@O=/))
M.<0/@X0```!(@^@!23G$=`Z`."]T,$B#Z`%).<1U\D&`/"0O='-(C0U,7```
M2(/$($%<Z:DV``!F#Q^$``````"`."]U"4B#Z`%).<1R\DPIX$4QP$B-4`)(
M@?H`$```=R!,C4`!0;D`$```3(GB2(T-^]L``.BF-```3(T%[]L``$R)P$B#
MQ"!!7,-).<1UM>N,9@\?1```2(T-UUL``$B#Q"!!7.DV-@``9@\?1```056X
M.!```$%455=64^C.,P``2"G$2(L]9)H``$&X_P\``$B+!TB)A"0H$```,<!(
MC70D($F)S$B)TTB-#8E;``!(B?+H=S4``(7`?G.-4/](BPWYF0``2&/J@'PL
M(`!,BREU5(72=&))B>BZ+P```$B)\>C(-```2(7`=$U,B>GH2S,``$F)Z4F)
M\$R)XDB)P>C",@``2(N$)"@0``!(*P=U/TB!Q#@0``!;7E]=05Q!7<,/'T0`
M`$ACZ.NK#Q\`2(L%B9D``$R+*$R)Z>C^,@``28G83(GB2(G!Z(`R``#KM.AY
M,P``D`\?A```````05=!5D%505155U932(/L.$B+'7F9``!(BP-(B40D*#'`
M,<"`.CU(B=8/E,!(B<](`<9(B?'H%C4``$B-5"0D2(GY2(G%Z`;[__])B<1(
MA<`/A/H```!(B<'H\C0``$@YQ0^&R0```$QC="0D3(LM/9D``$G!Y@,/M@>$
MP`^$C@$``#P]#X2&`0``2(GX#Q\`#[90`4B#P`&$TG0%@/H]=>](*?A(F$B-
M3"@"30-U`.B,,P``28D&2(7`=%1)BT4`2&-4)"1(C4\!2(L4T`^V!X@"A,!U
M&^L=#Q^$```````/M@%(@\(!2(/!`8@"A,!T!#P]=>O&`CTQP&8/'T0```^V
M#`:(3`(!2(/``83)=?!(BT0D*$@K`P^%!@$``$B#Q#A;7E]=05Q!74%>05_#
MD#'`9@\?1```#[84!D&(%`1(@\`!A-)U\.O&9@\?1```3(LM69@``$V+?0!)
M@S\`#X2R````3(GX13'D#Q]$``!(@\`(1(GB08/$`4B#.`!U[X/"`TACTDC!
MX@.+!3_I``"%P'53QP4QZ0```0```$B)T>B9,@``2(G!2(7`#X1=____36/T
M3(GZ2<'F`TV)\.BK,@``28E%`$B)P4&-1"0!2)A(QP3!`````$2)9"0DZ9+^
M__\/'P!,B?GH\#(``$F)10!(B<%(A<`/A!#___]-8_1)P>8#Z\,/'X``````
M,<#IC_[__[H0````13'DZ6?____H1S$``)!F#Q]$``!32(/L,$B+'227``!(
MBPOHG#```$B+"^B4,```3(T-55L``$R-!?8O``!(C16(6```2(G!Z!\P``!(
MBPOH;S```$R-#8U8``#'1"0H`````$R-!7D1``!,B4PD($B)P4B-%7!8``#H
MY2\``)!(@\0P6\-F9BX/'X0```````\?`$B#["A(BP6EE@``2(L(Z!TP``!,
MC0W>6@``3(T%?R\``$B)P4B-%0Y8``!(@\0HZ:0O``!F9BX/'X0``````)`Q
MP,-F9BX/'X0``````&:005>XV!```$%6055!5%575E/HVB\``$@IQ$R++7"6
M``!)BT4`2(F$),@0```QP$B-?"1`2(G.28G42(T-VU<``$B)^N@(^/__NB\`
M``!(B?%(B</HR#$``$B%P'0[2(GR2(T-O5<``.B4_/__2(GP2(N\),@0``!)
M*WT`#X5L`0``2('$V!```%M>7UU!7$%=05Y!7\,/'P!(B?'HL#$``$B-%8I7
M``!,B>&)Q>C/,0``28G'2(7`=*1!@#\`3(TE%%<```^$R0```&8/'T0``$B%
MVW033(GZ2(G9Z%`Q``"%P`^$AP```$R)^>A@,0``C5#_B<%(8\),`?B`."]U
M$XG1@^H!Q@``2&/"3`'X@#@O=.V-1"D!/?\/```/CSO___](B70D,#'23(VT
M),````!,C0W[5@``3(ED)"A,B?%!N``0``!,B7PD(.@0+P``2(GZ3(GQZ+4P
M``"%P'40BT0D4"4`\```/0"```!T44B-%;M6```QR>@#,0``28G'2(7`#X34
M_O__08`_``^%/?___TB%VW0%@#LN=!-,C3TR5@``N0$```#I9?___V:0@'L!
M`'2YZ^4/'X0``````+H!````3(GQZ,,N``"%P'6>3(GR2(T-258``.@@^___
M3(GQZ%@P``#I@O[__^A^+@``D&9F+@\?A```````9I`/M@%)B<B$P'079@\?
M1```2(/!`3PO#[8!3`]$P83`=>],B<##9F8N#Q^$``````!(A<ET"X`Y`'0&
MZ?'X__^02(T-CU4``.GT+P``#Q]``$%45U932(/L:$B+'3"4``!(BSU)E```
M2+I27T-,14%.7TB--<)5``!(BP-(B40D6$BX7U]%3E9?4$%(B50D.$R-9"0L
M2+H@("`@("`@($B)5"1(2(E$)#!(N%\@("`@("`@2(E$)$`QP&:)1"103(GB
M2(GQZ(?U__](A<!T0DAC5"0L2(L'2(T$T$B+4`A(B1!(A=)TV&8N#Q^$````
M``!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ$7U__](A<!UODB+/9F3``!(C34E
M50``3(GB2(GQZ"?U__](A<!T0DAC5"0L2(L'2(T$T$B+4`A(B1!(A=)TV&8N
M#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ.7T__](A<!UODB+/3F3
M``!(C37-5```3(GB2(GQZ,?T__](A<!T0DAC5"0L2(L'2(T$T$B+4`A(B1!(
MA=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ(7T__](A<!U
MODB+/=F2``!(C35V5```3(GB2(GQZ&?T__](A<!T0DAC5"0L2(L'2(T$T$B+
M4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ"7T
M__](A<!UODB+/7F2``!(C34=5```3(GB2(GQZ`?T__](A<!T0DAC5"0L2(L'
M2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB
M2(GQZ,7S__](A<!UODB+/1F2``!(C37-4P``3(GB2(GQZ*?S__](A<!T0DAC
M5"0L2(L'2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%
MTG7P3(GB2(GQZ&7S__](A<!UODB+/;F1``!(C34=4P``3(GB2(GQZ$?S__](
MA<!T0DAC5"0L2(L'2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`
M"$B)$$B%TG7P3(GB2(GQZ`7S__](A<!UODB+/5F1``!(C3494P``3(GB2(GQ
MZ.?R__](A<!T0DAC5"0L2(L'2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(
MBU`02(/`"$B)$$B%TG7P3(GB2(GQZ*7R__](A<!UODB+/?F0``!(C37#4@``
M3(GB2(GQZ(?R__](A<!T0DAC5"0L2(L'2(T$T$B+4`A(B1!(A=)TV&8N#Q^$
M``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ$7R__](A<!UODB+/9F0``!(
MC35M4@``3(GB2(GQZ"?R__](A<!T0DAC5"0L2(L'2(T$T$B+4`A(B1!(A=)T
MV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ.7Q__](A<!UODB+
M/3F0``!(C36F40``3(GB2(GQZ,?Q__](A<!T0DAC5"0L2(L'2(T$T$B+4`A(
MB1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ(7Q__](
MA<!UODR)XDB-#;M1``#H<?'__TB)PDB%P'0,2(T-DU$``.@-]O__3(GB2(T-
MJ5$``.A.\?__2(G"2(7`=`Q(C0VG40``Z.KU__],B>)(C0VC40``Z"OQ__](
MB<)(A<!T,TB-#>=0``#HQ_7__TB-%:]1``!(C0T440``Z+3U__](BT0D6$@K
M`W502(/$:%M>7T%<PTR)XDB-#6E1``#HX?#__TB)PDB%P'0.2(T-^5```.A]
M]?__Z[1(N%!!4E]#3$5!2#E$)$%UHV:!?"1)3CUUFDB-5"1+Z]'HQ2@``)`/
M'T``4TB#[#!(BQW4C@``2(L#2(E$)"B+!19```"#^/]T$4B+3"0H2"L+=3Q(
M@\0P6\.02(U4)"1(C0V+4```Z%_P__](B<(QP$B%TG02#[82@/HP#Y7!,<"$
MT@^5P"'(B07-/P``Z[KH5B@``)`/'T0``$B#["CHYR@``(G!Z-`H``!(A<!T
M`TB+`$B#Q"C##Q\`055!5%932(/L2$B+-3^.``!(BP9(B40D.#'`28G,2(U4
M)#1(C0V'4```Z.'O__](B<-(A<!T!8`X`'4L2(M$)#A(*P8/A98```!,B>)(
MC0U=4```2(/$2%M>05Q!7>E=]/__#Q]$``!,B>)(B<'HO2D``$B%P'082(M$
M)#A(*P9U7DB#Q$A;7D%<05W##Q\`3(GAZ'@I``!(B=E)B<7H;2D``$F-3`4"
MZ%,H``!(B5PD($V)X$R-#35/``!(C14P3P``2(G!28G%Z/(H``!(BT0D.$@K
M!G4(3(GJZ6W____H2R<``)!F+@\?A```````05>XF($``$%6055!5%575E/H
MJB8``$@IQ$B+!4"-``!(BSA(B;PDB($``#'_2(G+2(T%FT\``$R-;"1`2(F$
M),````!,B>I(C0583@``2(T-EDX``$B)A"3(````2(T%<T\``$B)A"0(`0``
M2(T%:T\``$B)A"00`0``2(T%9$\``$B)A"08`0``2(T%6D\``$B)A"0@`0``
M2(T%3T\``$C'A"30`````````$C'A"0H`0```````$B)A"3H````2,>$)/``
M````````Z$_N__])B<1(A<!T"8`X``^%;@0``.CY)@``B<'HXB8``$B%P`^$
MV0,``$B+,$B%]@^$S0,``$B)\>@5*```2(U,``'H^R8``$0/M@9(B<=%A,!T
M*TF)_$R--<U.```/'T0``$R)X4R)\DB#Q@%)@\0"Z(TG``!$#[8&183`=>1(
MC;0D``$``$B-#2U.``#K%`\?`$B+3@A(@\8(2(7)#X0/`0``3(GJZ*?M__])
MB<1(A<!TWX`X`'3:2(G!Z/+O__^%P'3.3(GAZ'8G``!)B<1(A<`/A-H```!F
M+@\?A```````3(GAZ&@G``!(B?E(B<;H72<``$2-O`8$!```36/_3(GYZ#HF
M``!(B7PD*$B-%1M.``!-B>!)B<9(C0473@``3(T-LDP``$B)1"0@3(GQZ,TF
M``"ZP`$``$R)\>A`)@``@_C_#X1W`@``3(GJ3(GQZ-PE``!!B<2%P`^$P0``
M`$B+&T4Q_^A6)```38GQ2(T5#$X``$B+2!A)B=CH4"4``$B+!0F+``!(B[PD
MB($``$@K.`^%8`4``$R)^$B!Q)B!``!;7E]=05Q!74%>05_#D$R+I"3`````
M,?;K%`\?0`!(8\9,BZ3$P````$V%Y'1`08`\)`!T.4R)X8/&`>C.[O__A<!T
MVDR)X>A2)@``28G$2(7`#X7F_O__2&/&3(NDQ,````!-A>1UQF8/'T0``$4Q
MY.G(_O__#Q^$``````"+1"10)0#P```]`$````^%*____XMT)%CHTR0``#G&
M#X4:____BT0D4"7_`0``/<`!```/A0;___],B?GHWR0``$B-#8E,``!,B>I)
MB<?HW>O__TB)P4B%P`^$00(``.B\)0``2(L+2(G"Z''S__](B<5(A<`/A"4"
M``!(B>GHK24``(/X!WY`2&/0N0D```!(C3T<30``2(UT%?CSI@^7PH#:`(32
M=2"#^`@/A'$#``"#Z`E(F(!\!0`O#X1A`P``#Q^``````(L%\CH``(/X_P^$
M$0,``(7`#X33`0``2(T5Z4L``$B-#6I+``#H[N___TB-'>Q*``!(C3VX2@``
MZ,LC``!(B5PD*$V)\$R)^8E$)"!,C0V=2@``2(T5KDP``$B--;5,``#HLB0`
M`.LRZ.LB``"#.!%U.NB1(P``08/$`4F)^4V)\$B)7"0P2(GR3(GY1(ED)"B)
M1"0@Z'XD``"ZP`$``$R)^>CQ(P``@_C_=+Q,B?'H1",``$R)^DB-#75*``#H
M5>___TR)^>B-^O__Z<C]__\/'X0``````.A[(@``@S@1#X1[_?__Z&TB``!(
MBQM%,?^+,.C@(0``38GQ28G82(T54TL``$B+2!B)="0@Z-8B``#I@?W__Y`Q
M_TB-#>U*``#K%0\?1```2&/'2(N,Q.````!(A<ET.4R)ZH/'`>@EZO__2(7`
M=."`.`!TVTB)P>@#)```2(G&2(7`#X7J^___2&/'2(N,Q.````!(A<EUQ[D-
M````2(TUEDH``.C6(@``0;A3````2(G'Z=G[__\/'X0``````$B)P>BX^?__
M2(L%`8@``$B+O"2(@0``2"LX#X58`@``3(GA2('$F($``%M>7UU!7$%=05Y!
M7^F%(P``#Q]$``!(BROIT_W__\<%#CD```````"Z```!`$B)Z>BQ(@``08G%
MA<`/A!7^__]!N`(```!(Q\+N____B<'H(B(``$B-G"2``0``0;@&````1(GI
M2(G:Z)DB``"`O"2``0```'4.@+PD@0$``$,/A%4!``!%,<`QTD2)Z>CD(0``
MZ*\;``!)B<3K&&8N#Q^$``````!!B<!(B=I,B>'H\AL``$&X`(```$B)VD2)
MZ>A!(@``A<!_W42)Z4B-M"0P`0``3(VL)%`!``#H%2$``$R)XDB)\4R)Z^A'
M'0``3(VD)'@!``!(C3V`20``#Q^$``````!$#[8&2(G92(GZ2(/#`DB#Q@'H
M.2(``$DYW'7D2(T%,D@``,:$)'@!````2(E$)"A,B6PD($R-#>Q'``!-B?!(
MC17K20``3(GYZ`(B``#ID?W__P\?1```3(GJ2(T-94@``.@YZ/__2(7`#X2H
M_O__#[8`A,`/A)W^__\\,`^$E?[__\<%HS<```$```#IO?S__TB+6PA(A=L/
MA)G\__](B=GH^2$``$B#^`,/AH?\__](C70#_+D%````2(T]:DD``/.F#Y?`
M'`"$P$@/1.OI9/S__X"\)((!``!!#X6=_O__@+PD@P$``$,/A8_^__^`O"2$
M`0``2`^%@?[__X"\)(4!``!%#X5S_O__0;@"````2,?"QO___T2)Z>A/(```
M0;@I````2(G:1(GIZ,X@``!(C04<1P``2(E<)"!(B40D*.GM_O__Z',?``"0
M9I!!5D%505175E-(@^PX2(L=3(4``$B+/76%``!(BPM(BP=(B40D*#'`3(UD
M)"3HKAX``$B+"TB)QNBC'@``#[:6N````$B-#1!'``"(D+D```!,B>+H]^;_
M_TB%P'0)@#@Q#X29`0``Z.3P__](BPOH;!X``$&Y`P```$&X`0```$B-%8A(
M``!(B<'H"1X``$B)QDB%P'0S2(M`$$B+"TR+*.@Z'@``3(T%9$@``$B)P4R)
MZNBX'0``2(M&$$R+*$'V10Y`#X5V`0``2(L+Z`X>``!!N0,```!!N`$```!(
MC14Q2```2(G!Z*L=``!(B<9(A<!T.$R)XDB-#25&``#H3.;__TF)Q4B%P`^$
M``$``$B+1A!(BPM,BS#HP1T``$V)Z$B)P4R)\NA#'0``2(L+Z*L=``!!N`$`
M``!(C1761P``2(G!Z%8=``!(BPM)B<7HBQT``$R-!=%'``!,B>I(B<'H"1T`
M`$B+"^AQ'0``2(T-I44``$R)XL:`N0````#HR^7__TB)!4S5``!(B<%(A<!T
M/+K``0``Z.(>``"-4`&#^@%V"8/X$0^%J0```$B+1"0H2"L'#X6[````2(/$
M.%M>7T%<05U!7L,/'T0``$B+#0G5``#H5/;__TB)!?74``!(B<%(A<!UJ4B-
M#49'``#HL1P``&8/'X0``````(!X`0`/A&+^___I6/[__Y!(BP7)U```2(L+
M3(LP2(M&$$R+*.BW'```3(GR3(GIZ-SH___I]/[__P\?@`````!(BPOHF!P`
M`$R)ZDB)P>@]'```Z7+^__]!B<%(BP5^U```3(L%;]0``$B-#?!&``!(BQ#H
M,!P``.CS'```D&:005175E-(@^PX2(L=`(,``$B+`TB)1"0H,<!)B<Q(A<D/
MA*H```"`.0`/A*$```#H?.?__P^V$$B)QH32=!1FD$B#P`&`^B\/MA!(#T3P
MA-)U[HL%"#0``$V%Y$`/E<>#^/\/A(````"%P`^5P"''0(3_=1Q(BT0D*$@K
M`P^%I0```$B#Q#A;7E]!7,,/'T``08`\)`!TW;D$````2(T]$T4``/.F#Y?`
M'`"$P'7&2(M$)"A(*P-U;TR)X4B#Q#A;7E]!7.GWY/__#Q^``````$B-#7]#
M``#HY!T``.E3____#Q^``````$B-5"0D2(T-`T0``.C7X___2(7`=`L/M@"$
MP'0$/#!U#\<%33,```````#I7/___\<%/C,```$```#I2/___^C$&P``D)"0
MD$%7059!54%455=64TB![+@"``!(BP7&@0``2(LX2(F\)*@"```Q_TB-5"0@
M3(U9.$B-G"2@````28G2#Q]$``!!BP--BPM)@\(028/#"`_(B<!-B<A)B4+P
M3(G(2<'H($C!Z"A)P>DX)0#_``!,"<A-B<%)P>`(2<'A&$&!X```_P!%B<E,
M"<A,"<!)B4+X23G:=:Q,C8H``@``#Q]$``!(BT)H2#-"0$B#P@A(,T((2#-"
M^$R-!`!(P>@?3`G`@^#_2(E">$PYRG753(L92(MQ$$B+62!(BWD83(G83(G:
M2(MI"$C!ZAM(P>`%2`G0`T0D($F)Z(V4&)EY@EI(B?!)P>`>2#'X2"'H2#'X
M1(TL`DB)Z$B+5"0H2,'H`DV)Z4D)P$B)\$B-E!>9>8):2<'I&T6)PDPQT$PA
MV$@Q\$@!T$R)ZDC!X@5,"<J-'!!,B=I,B=A(P>@"2,'B'DF)W$@)PDB+1"0P
M2<'L&TDQT$&)T4TAZ$B-A`:9>8):33'020'`2(G82,'@!4P)X$6-)`!,B>A)
MP>T"2,'@'DP)Z$R+;"0X2#'"08G`2"':3XV4*IEY@EI-B>5,,<I)P>T;3`'2
M38GB2<'B!4T)ZD:-+!))B=I(P>L"2<'B'DD)VDB+7"1`3#'01(G23"'@38V,
M&9EY@EI,B>M,,<!(P>L;3`'(38GI2<'A!4D)V4*-'`A-B>%)P>P"2<'A'DT)
MX4R+9"1(33'*1(G(32'J3XV$()EY@EI)B=Q),=))P>P;30'028G:2<'B!4T)
MXD>-)!!-B>A)P>T"2<'@'DT)Z$R+;"1033'!18G"22'92HV4*IEY@EI-B>5)
M,<%,`<I-B>%)P>$%2<'M&TT)Z4:-+`I(B=I(P>L"2,'B'D@)VDB+7"1823'0
M08G132'@2(V$&)EY@EI,B>M-,=!(P>L;3`'`38GH2<'@!4D)V$*-'`!,B>!)
MP>P"2,'@'DP)X$R+9"1@2#'"08G`3"'J3XV4(IEY@EI)B=Q,,<I)P>P;3`'2
M28G:2<'B!4T)XD:-)!)-B>I)P>T"2<'B'DT)ZDR+;"1H3#'01(G22"'83XV,
M*9EY@EI-B>5,,<!)P>T;3`'(38GA2<'A!4T)Z4:-+`A)B=E(P>L"2<'A'DD)
MV4B+7"1P33'*1(G(32'B38V$&)EY@EI,B>M),=)(P>L;30'038GJ2<'B!4D)
MVD.-'!!-B>!)P>P"2<'@'DT)X$R+9"1X33'!18G"32'I2HV4(IEY@EI)B=Q)
M,<%)P>P;3`'*28G92<'A!4T)X4:-)`I,B>I)P>T"2,'B'DP)ZDR+K"2`````
M23'008G122'82HV$*)EY@EI-B>5-,=!)P>T;3`'`38G@2<'@!4T)Z$:-+`!(
MB=A(P>`>2,'K`D@)V$B+G"2(````2#'"08G`3"'B38V4&IEY@EI,B>M,,<I(
MP>L;3`'238GJ2<'B!4D)VD*-'!)-B>))P>P"2<'B'DT)XDR+I"20````3#'0
M1(G23"'H3XV,(9EY@EI)B=Q,,<!)P>P;3`'(28G92<'A!4T)X4:-)`A-B>E)
MP>T"2<'A'DT)Z4R+K"28````33'*1(G(22':3XV$*)EY@EI-B>5),=))P>T;
M30'038GB2<'B!4T)ZDR+K"2@````10'"28G82,'K`DG!X!Y*C90JF7F"6DV)
MU4D)V$G![1M-,<%$B<--(>%),<%,`<I-B=%)P>$%30GI1HTL"DR)XDG![`)(
MP>(>3`GB3(ND)*@```!),=!!B=%-(=!*C80@F7F"6DV)[$DQV$G![!M,`<!-
MB>A)P>`%30G@1HTD`$R)T$G!Z@),BX0DL````$C!X!Y,"=!.C80#F7F"6DR)
MXT@QPDC!ZQM!B<),(>I,,<I,`<)-B>!)P>`%20G818G`3`'"38GH2<'M`DG!
MX!Z)TTT)Z$R+K"2X````3#'`3"'@3XV,*9EY@EI)B=U,,=!)P>T;3`'(28G9
M2<'A!4T)Z46)R4P!R$V)X4G![`))P>$>08G&30GA38G$18G`3`.$),@```!-
M,<Q$,>),BZ0DP````$^-E"*AZ]EN38GT20'23(GR2<'L&TC!X@5,">*)TDV-
M)!))B=I(P>L"2<'B'D2)XDD)VDR)RT6)R4P#C"30````3#'3,=A(B=--C80`
MH>O9;DB)T$C!ZQM(P>`%2`G8B<!)C1P`38GP2<'N`DR)T$G!X!Y!B=U%B=)-
M"?!,,<!!,<1,B>A/C8PAH>O9;DV)[$C!X`5,`Y0DV````$G![!M,">")P$P!
MR$F)T4C!Z@))P>$>08G&20G13(G"18G`3`.$).````!,,<HQTTR)\DV-E!JA
MZ]EN3(GS2,'B!4C!ZQM("=J)TDF-'!),B>I)P>T"38G*2,'B'D&)W$6)R4P#
MC"3H````3`GJ23'21#'038GB38V$`*'KV6Y,B>!)P>H;2,'@!4P)T(G`38T4
M`$R)\$G![@))B=!(P>`>18G5B=),"?!),<!$,<--B>A-C8P9H>O9;DR)ZTG!
MX`5(P>L;20G818G`2XT<`4V)X$F)P8G`2<'@'DG![`)!B=Y(`Y0D\````$T)
MX$@#A"3X````33'!13'*3HV,$J'KV6Y,B?)-B?))P>H;2,'B!4P)THG238T4
M$4V)Z4G![0),B<))P>$>18G418G`3`.$)``!``!-">E,,<HQTTB-E!BAZ]EN
M3(G@3(GC2,'K&TC!X`5("=B)P$B-'`),B?))P>X"3(G(2,'B'D&)W46)R4P#
MC"0(`0``3`GR2#'003'"3(GH3XV$$*'KV6Y-B>I(P>`%2<'J&TP)T(G`38T4
M`$R)X$G![`))B=!(P>`>18G6B=),">!),<!$,<--B?!-C8P9H>O9;DR)\TG!
MX`5(`Y0D$`$``$C!ZQM)"=A%B<!+C1P!38GH2<'M`DF)P4G!X!Y!B=R)P$@#
MA"08`0``30GH33'!13'*3HV,$J'KV6Y,B>)-B>))P>H;2,'B!4P)THG238T4
M$4V)\4G![@),B<))P>$>18G518G`3`.$)"`!``!-"?%,,<HQTTB-E!BAZ]EN
M3(GH3(GK2,'K&TC!X`5("=B)P$B-'`),B>))P>P"3(G(2,'B'D&)WD6)R4P)
MXD@QT$$QPDR)\$^-A!"AZ]EN38GR2,'@!4G!ZAM,"=")P$V-%`!,B>A)P>T"
M28G02,'@'D6)U(G23`GH3`.,)"@!``!(`Y0D,`$``$DQP$0QPTV)X$V-C!FA
MZ]EN3(GC2<'@!4C!ZQM)"=A%B<!+C1P!38GP2<'N`DF)P4G!X!Y!B=V)P$@#
MA"0X`0``30GP33'!13'*3HV,$J'KV6Y,B>I-B>I)P>H;2,'B!4P)THG238T4
M$4V)X4G![`),B<))P>$>18G618G`3`.$)$`!``!-">%-B?1,,<I)P>P;,=-(
MC908H>O9;DR)\$C!X`5,">")P$R-)`),B>I)P>T"3(G(2,'B'D2)XT6)R4P)
MZD@QT$$QPDB)V$^-A!"AZ]EN28G:2,'@!4G!ZAM,`XPD2`$``$P)T(G`38T4
M`$R)\$G![@))B=!(P>`>18G5B=)(`Y0D4`$``$P)\$DQP$4QQ$V)Z$^-C"&A
MZ]EN38GL2<'@!4G![!M-">!%B<!/C20!28G82,'K`DF)P4G!X!Y%B>>)P$@#
MA"18`0``20G833'!13'*3HV,$J'KV6Y,B?I-B?I)P>H;2,'B!4P)TDV)ZDG!
M[0))P>(>08T<$4R)PD6)P$T)ZDPQTDV)UD6)T4$QU$J-E""AZ]EN2(G828G<
M2<'L&TC!X`5,">!-B?Q)P>\"2<'D'D2-+`*XW+P;CTT)_$T)YD2)XDDATDDA
MWDT)UDR+E"1@`0``20'"30'038GJ30'&38GH2<'J&TG!X`5-"=!'C10&28G8
M2,'K`DG!X!Y)"=A-"<1-B<9$B<-)(=9-(>Q-"?1,B[0D:`$``$D!QDT!\4V)
MUDT!X4V)U$G![AM)P>0%30GT1XTT(4V)Z4G![0)-B<1)P>$>30GI18G-30G,
M32'432'H30G@3(ND)'`!``!)`<1,`>)-B?1)`=!,B?))P>P;2,'B!4P)XD6-
M)!!,B=))P>H"2,'B'DP)TDD)T4F)T$&)TDTAZ$TA\4T)P4R+A"1X`0``20'`
M20'83(GC30'(38GA2,'K&TG!X05)"=E,B?-)P>X"2,'C'D>-/`A)B=!,"?-!
MB=E)"=A-(>!,(<I,"<),BX0D@`$``$D!P$T!Z$V)_4P!PDV)^$G![1M)P>`%
M30GH1HTL`DV)X$G![`))P>`>30G@3`G#3(G"18G$3"'*3"'[2`G32(N4)(@!
M``!(`<),`=)-B>I(`=I,B>M)P>(%2,'K&TD)VD:--!)-B?I)P>\"3(G"2<'B
M'DT)^D2)TTP)TDPAZDDAV$D)T$B+E"20`0``2`'"3`'*38GQ20'03(GR2<'I
M&TC!X@5,"<I%C0P03(GJ2<'M`DC!XAY,">I)"=))B=!!B=5)(=A-(?)-"<),
MBX0DF`$``$D!P$T!X$V)S$T!T$V)RDG![!M)P>(%30GB38GT2<'N`DG!Y!Y'
MC3P028G030GT18GB30G@32'(3"'23`G"3(N$)*`!``!)`<!)`=A,B?M,`<)-
MB?A(P>L;2<'@!4D)V$*-'`)-B<A)P>D"2<'@'DT)R$T)Q$R)PD6)P4PATDTA
M_$D)U$B+E"2H`0``2`'"3`'J28G=3`'B28G<2<'M&TG!Y`5-">Q-B?U)P>4>
M2<'O`D:--"),B<)-"?U%B>Q,">I((=I-(>!)"=!(BY0DL`$``$@!PDP!TDV)
M\DD!T$R)\DG!ZAM(P>(%3`G218T4$$B)VDC!ZP)(P>(>2`G:20G528G0B=--
M(>!-(?5-"<5,BX0DN`$``$D!P$T!R$V)T4T!Z$V)U4G!X05)P>T;30GI1XT\
M"$V)\4G![@))B=!)P>$>30GQ18G.30G(32'03"'R3`G"3(N$),`!``!)`<!-
M`>!-B?Q,`<)-B?A)P>P;2<'@!4T)X$:-)`)-B=!)P>H"2<'@'DT)T$T)P4R)
MPD6)PDPA\DTA^4D)T4B+E"3(`0``2`'"2`':3(GC3`'*38GA2,'K&TG!X05)
M"=E,B?M)P>\"2,'C'D:-+`I,B<),"?M!B=E("=I,(>)-(<A)"=!(BY0DT`$`
M`$@!PDP!\DV)[DD!T$R)ZDG![AM(P>(%3`GR18TT$$R)XDC!XAY)P>P"3`GB
M2`G328G408G032',3"'K3`GC3(ND)-@!``!)`<1-`>)-B?1)`=I,B?-)P>P;
M2,'C!4P)XT6-)!I-B>I)P>T"2<'B'DT)ZDF)U42)TTT)U4TA]4@AVDP)ZDR+
MK"3@`0``20'%30'I38GE3`'*38GA2<'M&TG!X05-">E&C2P*38GQ2<'N`DG!
MX1Y-"?%-"<I,B<I%B<]((=I-(>))"=)(BY0DZ`$``$@!PDP!PDV)Z$P!TDV)
MZDG!X`5)P>H;30G01HT4`DV)X$G![`),B<I)P>`>30G@18G$3`G"3"'J32'A
M20G12(N4)/`!``!(`<)(`X0D^`$``$@!VDR)TTP!^$D!T4R)TDC!ZQM(P>(%
M2`G:B=))`=%,B>I)P>T"2,'B'D2)RTP)ZDD)T$F)U4&)UDTAY4TAT$T)Z$F)
MW4D!P$B)V$C!X`5)P>T;3`GHB<!)`<!,B=!)P>H"2,'@'D6)Q4D)PKC6P6+*
M3#'203'12(N4)``"``!(`<),`>)-B>Q)`=%,B>I)P>P;2,'B!4P)XHG220'1
M2(G:2,'K`DC!XAY%B<Q("=-,B=)%B=)(,=I!,=!(BY0D"`(``$@!PDP!\DV)
MYDD!T$R)XDG![AM(P>(%3`GRB=))`=!,B>I)P>T"2,'B'D6)QDD)U4B+E"00
M`@``2`'"3`'228G:B=M-,>I%,<I-B?%)`=),B?))P>D;2,'B!4P)RDV)Z46)
M[8G220'23(GB2<'L`DC!XAY%B==)"=1(BY0D&`(``$TQX4@!PD4QR$V)^4@!
MVDG!Z1M)`=!,B?I(P>(%3`G*3(N,)"`"``")TDD!T$R)\DD!P4G![@)(P>(>
M30'I38GE1(G#3`GR18GD23'513'J28G=30'*28G92<'M&TG!X05-">E%B<E-
M`<I-B?E)P>$>2<'O`D6)U4T)STR+C"0H`@``20'!30'A28G4B=)-,?Q%,<1-
MB>A-`>%-B>Q)P>`%2<'L&TT)Q$R+A"0P`@``18GD30',28G920'`2,'K`DG!
MX1Y,`<)-B?A%B>9)"=E%B?]-,<A%,<)-B?!)`=),B?))P>@;2,'B!4P)PDV)
MR$6)R8G220'23(GJ2<'M`DC!XAY$B=-)"=5(BY0D.`(``$TQZ$@!PD4QQ$F)
MV$P!^DD!U$B)VDC!X@5)P>@;3`G"38GP2<'N`HG22<'@'DD!U$B+E"1``@``
M30GP18GG2`'"3`'*38GI18GM33'!13'*38GY20'23(GZ2<'I&TC!X@5,"<I-
MB<%%B<")TDD!TDB)VDC!ZP)(P>(>18G62`G32(N4)$@"``!),=E(`<)%,<Q-
MB?%,`>I)P>D;20'43(GR2,'B!4P)RDR+C"10`@``B=))`=1,B?I)`<%)P>\"
M2,'B'DT!R$F)V46)Y4P)^HG;23'113'*38GI30'"38GH2<'I&TG!X`5-"<A%
MB<!-`<)-B?!)P>X"2<'@'D6)UTT)QDR+A"18`@``38GY2<'I&TD!P$D!V$B)
MTXG23#'S1#'C3`'#38GX2<'@!4T)R$V)Z4G![0)%B<!)P>$>3`'#3(N$)&`"
M``!-">E!B=Q)`<!,`<)-B?!%B?9-,<A%,<)-B>!)`=),B>))P>@;2,'B!4P)
MPDV)R$6)R8G220'23(GZ2,'B'DG![P)%B=5)"==(BY0D:`(``$TQ^$@!PD0Q
MPTV)Z$P!\DG!Z!M(`=-,B>I(P>(%3`G"38G@2<'L`HG22<'@'D@!TTB+E"1P
M`@``30G@08G>38G$2`'"3`'*38GY18G_33'!18G`13'138GR20'13(GR2<'J
M&TC!X@5,"=*)TDD!T4R)ZDG![0)(P>(>18G*20G52(N4)'@"``!-,>Q(`<)!
M,=Q,B=-,`?I)`=1,B=)(P>(%2,'K&T@)VHG220'43(GR2<'N`DC!XAY$B>-,
M"?),B[0D@`(``$D!QDT!\$V)[D6)[4DQUD4Q\4F)WDT!P4F)V$G![AM)P>`%
M30GP18G`30'!38G02<'J`DG!X!Y%B<Y-"<),BX0DB`(``$D!P$T!Z$F)U8G2
M33'513'L38GU30'@38GT2<'M&TG!Y`5-">Q)B=U(P>L"2<'E'D6)Y$P)ZTR+
MK"20`@``30'@18G$20'%3`'J38G518G223'=2`.$))@"``!%,>E-B>5,`<I-
MB>%)P>T;2<'A!4T)Z40!RDV)\4G![@))P>$>30G.3HT,&$B)V`-9($PQ\$T!
MT40!]TB)62!!,<!(B=!(B7D830'(28G12,'@!0'J2<'I&TB)40A,"<A$`<!(
MB0%,B>!)P>P"2,'@'DD)Q$B+!=%K``!$`>9(B7$02(NT)*@"``!(*S!U%$B!
MQ+@"``!;7E]=05Q!74%>05_#Z(0%``"0#Q\`2(/L*+F`````Z$(&``"ZB:O-
M[[G^W+J82(E0"$B!ZIG)^BM(QP`!(T5G2(E($$C'0!AV5#(02(E0($C'0"@`
M````2,=`,`````#'0'@`````2(/$*,-F+@\?A```````055!5%575E-(@^PH
M26/`3(M!*$B)STB)TTB+5S!)B<1(C7<X08T,P$PYP4B)3RA(8T]X2(/2`$C!
MZ!U(`=!(B4<PA<ET2+U`````2(G:*<U!.>Q!#T[L2`'Q3&/M38GHZ*L%``"+
M1W@!Z(E'>(/X0'0.2(/$*%M>7UU!7$%=PY!(B?E!*>Q,`>OHTNC__T&#_#\/
MCM@```!!@^Q`18GE0<'M!D&-;0%(P>4&2`'=9I#S#V\#2(GY2(/#0`\1!O,/
M;TO0#Q%.$/,/;U/@#Q%6(/,/;UOP#Q%>,.B"Z/__2#GK=<U!P>4&12GL26/$
M2(/X"')=2(M5`$B)5SA(BU0%^$B)5`;X2(U6"$B#XOA(*=9(*?5(`<9(@^;X
M2(/^"'(82(/F^#'`2(M,!0!(B0P"2(/`"$@Y\'+N1(EG>$B#Q"A;7E]=05Q!
M7<,/'X``````J`1U)$B%P'3?#[95`(A7.*@"=-0/MU0%_F:)5`;^Z\A(B=WI
M<O___XM5`(E7.(M4!?R)5`;\Z[`/'X0``````$%455=64TB#["!(BT(HBW(H
MBWHP2,'H`P_.@^`_#\](B<N-2`%(F$F)U,9$`CB`2(UJ.$ACP4@!Z(/Y.`^.
M7@$``$&X0````$$IR'00,=*)T8/"`<8$"`!$.<)R\DR)X>AKY___2<=$)#@`
M````2<=$)$``````2,=%$`````!(QT48`````$C'12``````2,=%*`````!(
MQT4P`````$&)?"1P3(GA08ET)'3H'^?__TF+!"1,B>%(P>@8B`-)BP0D2,'H
M$(A#`4F+!"1(P>@(B$,"28L$)(A#`TF+1"0(2,'H&(A#!$F+1"0(2,'H$(A#
M!4F+1"0(2,'H"(A#!DF+1"0(B$,'28M$)!!(P>@8B$,(28M$)!!(P>@0B$,)
M28M$)!!(P>@(B$,*28M$)!"(0PM)BT0D&$C!Z!B(0PQ)BT0D&$C!Z!"(0PU)
MBT0D&$C!Z`B(0PY)BT0D&(A##TF+1"0@2,'H&(A#$$F+1"0@2,'H$(A#$4F+
M1"0@2,'H"(A#$DF+1"0@B$,32(/$(%M>7UU!7.EF`@``9@\?1```NC@```!%
M,<DIRH/Z"'))B=%(QP``````2,=$"/@`````2(U("$B#X?A(*<@!PH/B^(/Z
M"`^"Q?[__X/B^#'`08G`@\`(3HD,`3G0<O+IK?[__V8/'T0``/;"!'4CA=(/
MA)K^___&``#VP@(/A([^__\QR6:)3!#^Z8+^__\/'P#'``````#'1!#\````
M`.EL_O__D)"0D)"0D)"0D)"0D/\EJMT``)"0_R6:W0``D)#_)8K=``"0D/\E
M>MT``)"0_R5JW0``D)#_)5K=``"0D/\E2MT``)"0_R4ZW0``D)#_)2K=``"0
MD/\E&MT``)"0_R4*W0``D)#_)?K<``"0D/\EZMP``)"0_R7:W```D)#_)<K<
M``"0D/\ENMP``)"0_R6JW```D)#_)9K<``"0D/\EBMP``)"0_R5ZW```D)#_
M)=+:``"0D)"0D)"0D)"0_R7"VP``D)"0D)"0D)"0D/\ENML``)"0D)"0D)"0
MD)!14$@]`!```$B-3"08<AE(@>D`$```2(,)`$@M`!```$@]`!```'?G2"G!
M2(,)`%A9PY"0D)"0D)"0D)"0D)"0_R52V@``D)"0D)"0D)"0D/\E2MH``)"0
MD)"0D)"0D)#_)4K:``"0D)"0D)"0D)"0_R5"V@``D)"0D)"0D)"0D/\E.MH`
M`)"0D)"0D)"0D)#_)3K:``"0D)"0D)"0D)"0_R5"V@``D)"0D)"0D)"0D/\E
M0MH``)"0D)"0D)"0D)#_)3K:``"0D)"0D)"0D)"0_R52V@``D)"0D)"0D)"0
MD/\E2MH``)"0D)"0D)"0D)#_)4+:``"0D)"0D)"0D)"0_R4ZV@``D)"0D)"0
MD)"0D/\E,MH``)"0D)"0D)"0D)#_)2K:``"0D)"0D)"0D)"0_R4BV@``D)"0
MD)"0D)"0D/\E&MH``)"0D)"0D)"0D)#_)1+:``"0D)"0D)"0D)"0_R4*V@``
MD)"0D)"0D)"0D/\E`MH``)"0D)"0D)"0D)#_)?K9``"0D)"0D)"0D)"0_R7R
MV0``D)"0D)"0D)"0D/\EZMD``)"0D)"0D)"0D)#_)>+9``"0D)"0D)"0D)"0
M_R7RV0``D)"0D)"0D)"0D/\EZMD``)"0D)"0D)"0D)#_)>+9``"0D)"0D)"0
MD)"0_R7:V0``D)"0D)"0D)"0D/\ETMD``)"0D)"0D)"0D)#_)<K9``"0D)"0
MD)"0D)"0_R7"V0``D)"0D)"0D)"0D/\ENMD``)"0D)"0D)"0D)#_);+9``"0
MD)"0D)"0D)"0_R6JV0``D)"0D)"0D)"0D/\EHMD``)"0D)"0D)"0D)#_)9K9
M``"0D)"0D)"0D)"0_R62V0``D)"0D)"0D)"0D/\EBMD``)"0D)"0D)"0D)#_
M)8+9``"0D)"0D)"0D)"02(/L*#'2Z%4```#_%1?8``"0D)"0D)"0D)"0D)"0
MD)##D)"0D)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0
MD)"0D)##D)"0D)"0D)"0D)"0D)"05E-(@^PH2(G.2(G3N0@```#HR@0``$B%
MVP^$$0(``$BZ`````%0!``!(B7-`2(TUC/___TB-#54'``!(B9/P````2(T5
M7P<``$B)4U!(BQ5\UP``2(FS@````$B--6[___](B4M(2(T-<____TB)4QA(
MC15XM```2(FSB````$B--6K___](B8N0````2+E(`0``N0L``$B)LY@```!(
MC36;_?__2(E3.$B-%3#]__](B4L(2(T-)?[__TB)<R!(C37Z`P``2(E3*$B-
M%1\$``!(QP,`````QT,0!P```$B)2S!(B7-X2(F3&`$``$B%P`^$O0```$B+
M@.````!(A<`/A*T```"+%?.S``"%T@^%GP```$B#/=,3````#X0-`@``2(,]
MS1,````/A&\"``!(@SW'$P````^$40(``$B#/<$3````#X0S`@``2(,]NQ,`
M```/A!4"``!(@SVU$P````^$]P$``$B#/:\3````#X39`0``2(,]J1,````/
MA+L!``!F#V\%8Q,```\1`&8/;PUH$P``#Q%($&8/;Q5L$P``#Q%0(&8/;QUP
M$P``#Q%8,#')Z(4#```QR4B)@^@```!(C04%$P``2(E#6$B-!5H3``!(B4-@
M2(T%[Z(``$B)0VA(C07$M```2(E#<$B-!2EE``!(B8,@`0``2(T%5V4``$B)
M@R@!``!(C06]LO__2(F#,`$``.B1`@``N`$```!(@\0H6U[##Q]$``!(B7!`
M2(TUA?W__TBZ`````%0!``!(C0U<!0``2(FP@````$B--7;]__](B9#P````
M2(T5*`4``$B)L(@```!(C35J_?__2(E02$B+%4_5``!(B4A02(N(0`$``$B)
ML)````!(C356_?__2(FPF````$B--4BR``!(B0I(C16.^___2(T-)_O__TB)
M<#A(OD@!``"Y"P``2(EP"$B--0[\__](B5`@2(T5XP$``$B)2"A(C0T(`@``
M2,<``````,=`$`<```!(B7`P2(E0>$B)B!@!``!(A<`/A*;^__](B<-(BX#@
M````2(7`#X7F_?__Z8[^__]F+@\?A```````2(L02(D5MA$``.GD_?__D$B+
M4#A(B17=$0``Z37^__](BU`P2(D5Q1$``.D7_O__2(M0*$B)%:T1``#I^?W_
M_TB+4"!(B165$0``Z=O]__](BU`82(D5?1$``.F]_?__2(M0$$B)%641``#I
MG_W__TB+4`A(B15-$0``Z8']__],BP59L0``BQ5;L0``2(L-7+$``.G7````
M#Q^``````$B#[#B#^@)T=W=5A=(/A'T````QP$V%P$B-%<'___](B0TJL0``
M#Y3`3(D%$+$``,<%#K$```$```")!?"P``#HZP```$B#^/](B07HL```#Y7`
M#[;`2(/$.,,/'T0``+@!````@_H#=>RZ`P```$B#Q#CI6`````\?A```````
MN@(```!(@\0XZ4(```!FD#'2Z#D```"%P'2[2(L-EK```(E$)"SH50```(M$
M)"Q(QP5^L```_____^N:#Q]``.D[____D)"0D)"0D)"0D)"X`0```,.0D)"0
MD)"0D)"0PY"0D)"0D)"0D)"0D)"0D/\E&M,``)"0D)"0D)"0D)#_)2+3``"0
MD)"0D)"0D)"0_R4:TP``D)"0D)"0D)"0D/\EFM,``)"0D)"0D)"0D)!!5$B#
M["!)B<Q(B=%(C14=L```Z.CZ__](C141L```3(GA2(/$($%<Z0,```"0D)#_
M)=+2``"0D)"0D)"0D)"0_R6:U```D)`/'X0``````$%505164TB#[%B)C"2`
M````2(E4)#A,B40D,.C@]___2(U4)#A(BQT47@``3(U$)#!(C8PD@````$B+
M`TB)1"1(,<#HR/;__TR-!1'W__](C0T2]___3(G"Z"+W__](BP6;70``2(LU
MI%T``(`X`'4SZ(+V__](B04;KP``2(G!2(7`#X2/`0``Z&+V__](BS5[70``
M2(L.Z//V___&@"$*````2(L.Z.3V__](BPZ`B'($```"Z-7V__^`B'($```!
MBX0D@````(U(!(UP`TACR4C!X0/H3?;__TB+3"0X1(N4)(````!)B<%(B06>
MK@``2(L!28D!2(T%4B$``$F)00A(BP6.#@``28E!$$B-!3\A``!)B4$808/Z
M`0^.Z@```$B#P0A!@\(#N`0```!,BP%(8]"#P`%(@\$(38D$T4$YPG7J36/2
M2<'B`TO'!!$`````2(L-/*X``$&)\$B-%7K%__](QT0D(`````#H9/7__T&)
MQ(7`=$Y(BPT6K@``3(UL)$3H7/7__TR)ZDB-#84>``#H9;[__TB%P'0Z2(L-
M\:T``.@T]?__Z%?U__](BT0D2$@K`W5G1(G@2(/$6%M>05Q!7<-(BPW(K0``
MZ/OT__]!B<3KH4B+#:>M``!(A<ET!^A%V?__Z[-(C0W7'0``3(GJZ`2^__](
MB06%K0``2(G!2(7`=)7KV4&Z(````.DU____N0$```#H3O;__^CY]?__D)"0
MD)"0D)"0Z9N]__^0D)"0D)"0D)"0D/__________L%)```$`````````````
M`/__________````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````````/__
M__\`````Z')```$`````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````E
M<R\E<P`N`"\`+W!R;V,O<V5L9B]E>&4`1'EN84QO861E<CHZ8F]O=%]$>6YA
M3&]A9&5R``!);G1E<FYA;',Z.E!!4CHZ0D]/5`!005)?5$5-4`!005)?4%)/
M1TY!344`.@`E<R5S)7,`4$523#5,24(`4$523$Q)0@!015),-4]05`!015),
M24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@!005)?
M1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"54<`4$%27T=,3T)!3%]4
M35!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#
M3$5!3@!005)?0TQ%04X]`#$`4$%42`!54T52`%-94U1%30`O=&UP`%1-4$1)
M4@!414U01$E2`%1%35``5$U0`%5315).04U%`"4P,G@`)7,E<R5S)7,`<&%R
M+0```````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E
M<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@<')I=F%T92!S=6)D:7)E
M8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R
M>2!Y;W5R(&]P97)A=&EO;BD*`'!A<FPN97AE`"YP87(`)7,E<V-A8VAE+25S
M)7,`)7,E<W1E;7`M)74E<P`E<R5S=&5M<"TE=2TE=25S`!@`<&5R;``P`%!!
M4CHZ4&%C:V5R.CI615)324].`#$N,#4R````````56YA8FQE('1O(&-R96%T
M92!C86-H92!D:7)E8W1O<GD``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV
M871E(&-A8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`
M+64`+2T``'!E<FQX<VDN8P````````!E=F%L("=E>&5C("]U<W(O8FEN+W!E
M<FP@("U3("0P("1[,2LB)$`B?2<*:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E
M<B!S;VUE('-H96QL"G!A8VMA9V4@7U]P87)?<&P["FUY("@D4$%27TU!1TE#
M+"`D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I;&4L("5-;V1U;&5#86-H
M92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q=6ER92!&:6QE
M.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U:7)E($9I;&4Z
M.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D
M<&%R7W1E;7`I.PIO=71S*'%Q6U)E;6]V:6YG(&9I;&5S(&EN("(D<&%R7W1E
M;7`B72D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD("D@/R!R
M;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D<&%R7W1E;7`[
M"FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R)RD@>PIM>2`D
M=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%(#T^("=T;7!86%A86"<L
M"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER*2P*4U5&
M1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD["FUY("1F:6QE;F%M92`]
M("1T;7`M/F9I;&5N86UE.PIP<FEN="`D=&UP(#P\(BXN+B(["G@],3L@=VAI
M;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A<E]T96UP)PII9B!;
M(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E86L*9FD*<VQE97`@,0IX
M/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R1F:6QE;F%M92<*+BXN"F-L;W-E
M("1T;7`["F-H;6]D(#`W,#`L("1F:6QE;F%M93L*;7D@)&-M9"`]("(D9FEL
M96YA;64@/B]D978O;G5L;"`R/B8Q("8B.PIS>7-T96TH)&-M9"D["F]U=',H
M<7%;4W!A=VYE9"!B86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R9F]R;2!C;&5A
M;G5P.B`D9FEL96YA;65=*3L*?0I]('T*0D5'24X@>PI);G1E<FYA;',Z.E!!
M4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"3T]4.PHD
M4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PIE=F%L('L*7W!A<E]I;FET7V5N
M=B@I.PIM>2`D<75I970@/2`A)$5.5GM005)?1$5"54=].PIM>2`E0V]N9FEG
M(#T@*`IP871H7W-E<"`@("`]/B`H)%Y/(#U^("]>35-7:6XO(#\@)SLG(#H@
M)SHG*2P*7V5X92`@("`@("`@/3X@*"1>3R`]?B`O7B@_.DU35VEN?$]3,GQC
M>6=W:6XI+R`_("<N97AE)R`Z("<G*2P*7V1E;&EM("`@("`@/3X@*"1>3R`]
M?B`O7DU35VEN?$]3,B\@/R`G7%PG(#H@)R\G*2P**3L*7W-E=%]P<F]G;F%M
M92@I.PI?<V5T7W!A<E]T96UP*"D["FUY("@D<W1A<G1?<&]S+"`D9&%T85]P
M;W,I.PI["FQO8V%L("1324=[7U]705).7U]](#T@<W5B('M].PIO<&5N(%]&
M2"P@)SPZ<F%W)RP@)'!R;V=N86UE(&]R(&QA<W0["FUY("1B=68["FUY("1S
M:7IE(#T@+7,@)'!R;V=N86UE.PIM>2`D8VAU;FM?<VEZ92`](#8T("H@,3`R
M-#L*;7D@)&UA9VEC7W!O<SL*:68@*"1S:7IE(#P]("1C:'5N:U]S:7IE*2![
M"B1M86=I8U]P;W,@/2`P.PI](&5L<VEF("@H;7D@)&T@/2`D<VEZ92`E("1C
M:'5N:U]S:7IE*2`^(#`I('L*)&UA9VEC7W!O<R`]("1S:7IE("T@)&T["GT@
M96QS92!["B1M86=I8U]P;W,@/2`D<VEZ92`M("1C:'5N:U]S:7IE.PI]"G=H
M:6QE("@D;6%G:6-?<&]S(#X](#`I('L*<V5E:R!?1D@L("1M86=I8U]P;W,L
M(#`["G)E860@7T9(+"`D8G5F+"`D8VAU;FM?<VEZ92`K(&QE;F=T:"@D4$%2
M7TU!1TE#*3L*:68@*"AM>2`D:2`](')I;F1E>"@D8G5F+"`D4$%27TU!1TE#
M*2D@/CT@,"D@>PHD;6%G:6-?<&]S("L]("1I.PIL87-T.PI]"B1M86=I8U]P
M;W,@+3T@)&-H=6YK7W-I>F4["GT*;&%S="!I9B`D;6%G:6-?<&]S(#P@,#L*
M<V5E:R!?1D@L("1M86=I8U]P;W,@+2`T+"`P.PIR96%D(%]&2"P@)&)U9BP@
M-#L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`T("T@=6YP86-K*").(BP@)&)U
M9BDL(#`["B1D871A7W!O<R`]('1E;&P@7T9(.PIM>2`E<F5Q=6ER95]L:7-T
M.PIR96%D(%]&2"P@)&)U9BP@-#L@("`@("`@("`@("`@("`@("`@("`@("`@
M("`C(')E860@=&AE(&9I<G-T(")&24Q%(@IW:&EL92`H)&)U9B!E<2`B1DE,
M12(I('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A
M8VLH(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*;W5T<RAQ<5M5
M;G!A8VMI;F<@1DE,12`B)&9U;&QN86UE(BXN+ETI.PIM>2`D8W)C(#T@*"`D
M9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z('5N9&5F
M.PIM>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\*#\Z+BHO*3\H
M+BHI*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U
M9BP@=6YP86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E>'0I(&%N9"`D
M97AT("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`D9FEL96YA;64@
M/2!?<V%V95]A<R@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*)%!!4CHZ2&5A
M=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE.PHD4$%2.CI(
M96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA;64["GT*96QS
M:68@*"`D9G5L;&YA;64@/7X@;7Q>+S]S:&QI8B]\(&%N9"!D969I;F5D("1%
M3E9[4$%27U1%35!]("D@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&)A
M<V5N86UE)&5X="(L("1B=68L(#`W-34I.PIO=71S*")32$Q)0CH@)&9I;&5N
M86UE7&XB*3L*?0IE;'-E('L*)')E<75I<F5?;&ES='LD9G5L;&YA;65](#T*
M)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@
M/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["GT*<F5A9"!?1D@L("1B
M=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U
M;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*
M;7D@)&EN9F\@/2!D96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?2!O<B!R
M971U<FX["B1)3D-[)&UO9'5L97T@/2`B+VQO861E<B\D:6YF;R\D;6]D=6QE
M(CL*:68@*"1%3E9[4$%27T-,14%.?2!A;F0@9&5F:6YE9"@F24\Z.D9I;&4Z
M.FYE=RDI('L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=U]T;7!F:6QE(&]R(&1I
M92`B0V%N)W0@8W)E871E('1E;7`@9FEL93H@)"$B.PHD9F@M/F)I;FUO9&4H
M*3L*)&9H+3YP<FEN="@D:6YF;RT^>V)U9GTI.PHD9F@M/G-E96LH,"P@,"D[
M"G)E='5R;B`D9F@["GT*96QS92!["FUY("1F:6QE;F%M92`](%]S879E7V%S
M*"(D:6YF;RT^>V-R8WTN<&TB+"`D:6YF;RT^>V)U9GTI.PIO<&5N(&UY("1F
M:"P@)SPZ<F%W)RP@)&9I;&5N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D
M9FEL96YA;64B.B`D(5T["G)E='5R;B`D9F@["GT*9&EE(")";V]T<W1R87!P
M:6YG(&9A:6QE9#H@8V%N)W0@9FEN9"!M;V1U;&4@)&UO9'5L92$B.PI]+"!`
M24Y#*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z.DAE879Y.PIR
M97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U:7)E($E/.CI&
M:6QE.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y<R`E<F5Q=6ER
M95]L:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE?2!O<B`D9FEL
M96YA;64@/7X@+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]?B`O7"YP;6,_
M)"]I*2!["G)E<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID;R`D9FEL96YA
M;64@=6YL97-S("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ95PN<&PD+SL*
M?0I]"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T["GT*;&%S="!U
M;FQE<W,@)&)U9B!E<2`B4$M<,#`S7#`P-"(["B1S=&%R=%]P;W,@/2`H=&5L
M;"!?1D@I("T@-#L@("`@("`@("`@("`@("`@(R!S=&%R="!O9B!Z:7`*?0IM
M>2!`<&%R7V%R9W,["FUY("@D;W5T+"`D8G5N9&QE+"`D;&]G9F@L("1C86-H
M95]N86UE*3L*9&5L971E("1%3E9[4$%27T%04%]2155317T[(",@<V%N:71I
M>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y('!R;V)L96TI"B1Q=6EE="`]
M(#`@=6YL97-S("1%3E9[4$%27T1%0E5'?3L*:68@*"$D<W1A<G1?<&]S(&]R
M("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY
M("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?
M<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y
M7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@
M>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H:69T($!!
M4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM>2!`861D
M7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>+2A;04E-
M3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G22<I('L*
M<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE
M=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS:&EF="!`
M<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U="`]("0R
M.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E)SL*?0IE
M;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H
M)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q("=,)RD@
M>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@
M;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A
M8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D8VUD(#T@
M)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI
M=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M
M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@
M9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D7W1O7VEN
M8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER
M92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*;7D@)'!A
M<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A<BD@
M>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@
M9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I
M;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E861&
M<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?
M3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D
M<&%R(ET["GT*;7D@)65N=B`](&1O('L*:68@*"1Z:7`@86YD(&UY("1M971A
M(#T@)'II<"T^8V]N=&5N=',H)TU%5$$N>6UL)RDI('L*)&UE=&$@/7X@<R\N
M*EYP87(Z)"\O;7,["B1M971A(#U^(',O7EQ3+BHO+VUS.PHD;65T82`]?B`O
M7B`@*%M>.ETK*3H@*"XK*20O;6<["GT*?3L*:68@*&1E9FEN960@)'!A<BD@
M>PIO<&5N(&UY("1P:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@
M<F5A9"!P87(@9FEL92`B)'!A<B(Z("0A73L*;7D@)&)U9CL*<F5A9"`D<&@L
M("1B=68L(#0["F1I92!Q<5LB)'!A<B(@:7,@;F]T(&$@<&%R(&9I;&5=('5N
M;&5S<R`D8G5F(&5Q(")02UPP,#-<,#`T(CL*8VQO<V4@)'!H.PI]"D-R96%T
M95!A=&@H)&]U="D@.PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`HD;W5T+`I)
M3SHZ1FEL93HZ3U]#4D5!5"@I('P@24\Z.D9I;&4Z.D]?5U)/3DQ9*"D@?"!)
M3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C
M<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@
M7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D9&%T85]P;W,I
M('L*<F5A9"!?1D@L("1L;V%D97(L("1D871A7W!O<SL*?2!E;'-E('L*;&]C
M86P@)"\@/2!U;F1E9CL*)&QO861E<B`](#Q?1D@^.PI]"FEF("@A)$5.5GM0
M05)?5D520D%424U](&%N9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R96TI+RD@
M>PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*4$%2.CI&:6QT97(Z
M.E!O9%-T<FEP+3YA<'!L>2A<)&QO861E<BP@)#`I.PI]"F9O<F5A8V@@;7D@
M)&ME>2`H<V]R="!K97ES("5E;G8I('L*;7D@)'9A;"`]("1E;G9[)&ME>7T@
M;W(@;F5X=#L*)'9A;"`](&5V86P@)'9A;"!I9B`D=F%L(#U^("]>6R<B72\[
M"FUY("1M86=I8R`](")?7T5.5E]005)?(B`N('5C*"1K97DI("X@(E]?(CL*
M;7D@)'-E="`](")005)?(B`N('5C*"1K97DI("X@(CTD=F%L(CL*)&QO861E
M<B`]?B!S>R1M86=I8R@@*RE]>PHD;6%G:6,@+B`D<V5T("X@*"<@)R!X("AL
M96YG=&@H)#$I("T@;&5N9W1H*"1S970I*2D*?65G.PI]"B1F:"T^<')I;G0H
M)&QO861E<BD["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR
M97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@
M?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF
M:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R
M96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,
M;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI
M8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!8
M6%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY
M("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D[
M"FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS
M92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<
M421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q
M*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD
M?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE
M+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D
M,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0[
M"GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D
M=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I
M;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C
M:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65]
M(#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C
M;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N
M9&5F.PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD9F@^.PIC;&]S
M92`D9F@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT
M+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@
M86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z
M.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D
M;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H
M)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z
M.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE
M;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D["F]U=',H<7%;5W)I='1E;B!A
M<R`B)&YA;64B72D["GT*?0HD>FEP+3YW<FET951O1FEL94AA;F1L92@D9F@I
M(&EF("1Z:7`["B1C86-H95]N86UE(#T@<W5B<W1R("1C86-H95]N86UE+"`P
M+"`T,#L*:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@)&UT:6UE(#T@*'-T870H
M)&]U="DI6SE=*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIO
M<&5N(&UY("1F:"P@(CPZ<F%W(BP@)&]U=#L*)&-T>"T^861D9FEL92@D9F@I
M.PIC;&]S92`D9F@["B1C86-H95]N86UE(#T@)&-T>"T^:&5X9&EG97-T.PI]
M"B1C86-H95]N86UE("X](")<,"(@>"`H-#$@+2!L96YG=&@@)&-A8VAE7VYA
M;64I.PHD8V%C:&5?;F%M92`N/2`B0T%#2$4B.PHD9F@M/G!R:6YT*"1C86-H
M95]N86UE*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&9H+3YT96QL("T@;&5N
M9W1H*"1L;V%D97(I*2D["B1F:"T^<')I;G0H)%!!4E]-04=)0RD["B1F:"T^
M8VQO<V4["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S
M<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER
M92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U
M:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F
M:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@
M("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F
M:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH
M*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@
M7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E
M861&<F]M1FEL94AA;F1L92@D9F@L("1P<F]G;F%M92D@/3T@07)C:&EV93HZ
M6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R
M8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ
M92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP.PHD4$%2
M.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*)'%U:65T(#T@(21%3E9[
M4$%27T1%0E5'?3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM0
M05)?5$5-4'TB72D["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@
M<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB
M97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER
M96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*
M;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z
M)$-O;F9I9SHZ0V]N9FEG>W9E<G-I;VY]+RD_"B@_.B1#;VYF:6<Z.D-O;F9I
M9WMA<F-H;F%M97TO*3\**%M>+UTK*0HD"GUX.PIM>2`D97AT<F%C=%]N86UE
M(#T@)#$["FUY("1D97-T7VYA;64@/2!&:6QE.CI3<&5C+3YC871F:6QE*"1%
M3E9[4$%27U1%35!]+"`D97AT<F%C=%]N86UE*3L*:68@*"UF("1D97-T7VYA
M;64@)B8@+7,@7R`]/2`D;65M8F5R+3YU;F-O;7!R97-S9613:7IE*"DI('L*
M;W5T<RAQ<5M3:VEP<&EN9R`B)&UE;6)E<E]N86UE(B!S:6YC92!I="!A;')E
M861Y(&5X:7-T<R!A="`B)&1E<W1?;F%M92)=*3L*?2!E;'-E('L*;W5T<RAQ
M<5M%>'1R86-T:6YG("(D;65M8F5R7VYA;64B('1O("(D9&5S=%]N86UE(ETI
M.PHD;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D["F-H
M;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]
M"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\
M/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;
M("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP
M;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM0
M05)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD[
M"GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@
M1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]
M($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*
M<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I
M('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]
M"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$
M>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR
M97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*
M<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%
M>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N
M=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR
M97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ
M2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:
M;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(
M03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!0
M05(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U
M:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB
M=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I
M;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@
M<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I
M<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@
M4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U
M:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D
M("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I
M('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H
M("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$
M25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X
M="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P
M871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@
M/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H
M)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO
M9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE
M(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%
M?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"B1U<V5R;F%M92`]?B!S
M+UQ7+U\O9SL*;7D@)'-T;7!D:7(@/2`B)'!A=&@D0V]N9FEG>U]D96QI;7UP
M87(M(BYU;G!A8VLH(D@J(BP@)'5S97)N86UE*3L*;6MD:7(@)'-T;7!D:7(L
M(#`W-34["FEF("@A)$5.5GM005)?0TQ%04Y](&%N9"!M>2`D;71I;64@/2`H
M<W1A="@D<')O9VYA;64I*5LY72D@>PIO<&5N(&UY("1F:"P@(CPZ<F%W(BP@
M)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D<')O9VYA;64B.B`D
M(5T["G-E96L@)&9H+"`M,3@L(#(["FUY("1B=68["G)E860@)&9H+"`D8G5F
M+"`V.PII9B`H)&)U9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L("TU."P@
M,CL*<F5A9"`D9F@L("1B=68L(#0Q.PHD8G5F(#U^(',O7#`O+V<["B1S=&UP
M9&ER("X]("(D0V]N9FEG>U]D96QI;7UC86-H92TD8G5F(CL*?0IE;'-E('L*
M;7D@)&1I9V5S="`](&5V86P*>PIR97%U:7)E($1I9V5S=#HZ4TA!.PIM>2`D
M8W1X(#T@1&EG97-T.CI32$$M/FYE=R@Q*3L*;W!E;B!M>2`D9F@L("(\.G)A
M=R(L("1P<F]G;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)'!R;V=N86UE
M(CH@)"%=.PHD8W1X+3YA9&1F:6QE*"1F:"D["F-L;W-E*"1F:"D["B1C='@M
M/FAE>&1I9V5S=#L*?2!\?"`D;71I;64["B1S=&UP9&ER("X]("(D0V]N9FEG
M>U]D96QI;7UC86-H92TD9&EG97-T(CL*?0IC;&]S92@D9F@I.PI]"F5L<V4@
M>PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PHD<W1M<&1I<B`N/2`B)$-O;F9I9WM?
M9&5L:6U]=&5M<"TD)"(["GT*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*
M;6MD:7(@)'-T;7!D:7(L(#`W-34["FQA<W0["GT*)'!A<E]T96UP(#T@)#$@
M:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI
M+SL*?0IS=6(@7W-A=F5?87,@>PIM>2`H)&YA;64L("1C;VYT96YT<RP@)&UO
M9&4I(#T@0%\["FUY("1F=6QL;F%M92`]("(D<&%R7W1E;7`O)&YA;64B.PIU
M;FQE<W,@*"UE("1F=6QL;F%M92D@>PIM>2`D=&5M<&YA;64@/2`B)&9U;&QN
M86UE+B0D(CL*;W!E;B!M>2`D9F@L("<^.G)A=R<L("1T96UP;F%M92!O<B!D
M:64@<7%;0V%N)W0@=W)I=&4@(B1T96UP;F%M92(Z("0A73L*<')I;G0@)&9H
M("1C;VYT96YT<SL*8VQO<V4@)&9H.PIC:&UO9"`D;6]D92P@)'1E;7!N86UE
M(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D=&5M<&YA;64L("1F=6QL;F%M
M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*<F5T=7)N("1F=6QL;F%M93L*
M?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD969I;F5D("1%3E9[4$%27U!2
M3T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!345](#U^("\H+BLI+RD@>PHD
M<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@?'P]("0P.PII9B`H)$5.5GM0
M05)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?5$5-4'TI
M(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R*"1P<F]G;F%M92P@<FEN9&5X
M*"1P<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`K(#$I.PI]"FEF("@A)$5.
M5GM005)?4%)/1TY!345](&]R(&EN9&5X*"1P<F]G;F%M92P@)$-O;F9I9WM?
M9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY("1F:"P@)SPG+"`D<')O9VYA
M;64I('L*<F5T=7)N(&EF("US("1F:#L*?0II9B`H+7,@(B1P<F]G;F%M921#
M;VYF:6=[7V5X97TB*2!["B1P<F]G;F%M92`N/2`D0V]N9FEG>U]E>&5].PIR
M971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@*'-P;&ET("]<421#;VYF:6=[
M<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE>'0@:68@97AI<W1S("1%
M3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%35!].PHD9&ER
M(#U^(',O7%$D0V]N9FEG>U]D96QI;7U<120O+SL**"@D<')O9VYA;64@/2`B
M)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M921#;VYF:6=[7V5X97TB*2P@
M;&%S="D*:68@+7,@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA;64D0V]N
M9FEG>U]E>&5](CL**"@D<')O9VYA;64@/2`B)&1I<B1#;VYF:6=[7V1E;&EM
M?21P<F]G;F%M92(I+"!L87-T*0II9B`M<R`B)&1I<B1#;VYF:6=[7V1E;&EM
M?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M92!["B0P(#T@)'!R
M;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-17T["FEF("AI;F1E>"@D<')O
M9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M92`]("(N
M)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@*&1E9FEN
M960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E9FEN960@
M)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W9&`I.PIC
M:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D0V]N9FEG
M>U]D96QI;7U<12DO)'!W9"1#;VYF:6=[7V1E;&EM?2\["B1%3E9[4$%27U!2
M3T=.04U%?2`]("1P<F]G;F%M93L*?0IS=6(@7W!A<E]I;FET7V5N=B!["FEF
M("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@,2`I('L*<F5T=7)N.PI]
M(&5L<V4@>PHD14Y6>U!!4E])3DE424%,25I%1'T@/2`R.PI]"F9O<B`H<7<H
M(%-005=.140@5$5-4"!#3$5!3B!$14)51R!#04-(12!04D]'3D%-12`I("D@
M>PID96QE=&4@)$5.5GLG4$%27R<N)%]].PI]"F9O<B`H<7<O(%1-4$1)4B!4
M14U0($-,14%.($1%0E5'("\I('L*)$5.5GLG4$%27R<N)%]](#T@)$5.5GLG
M4$%27T=,3T)!3%\G+B1??2!I9B!E>&ES=',@)$5.5GLG4$%27T=,3T)!3%\G
M+B1??3L*?0IM>2`D<&%R7V-L96%N(#T@(E]?14Y67U!!4E]#3$5!3E]?("`@
M("`@("`@("`@("`@(CL*:68@*"1%3E9[4$%27U1%35!]*2!["F1E;&5T92`D
M14Y6>U!!4E]#3$5!3GT["GT*96QS:68@*"%E>&ES=',@)$5.5GM005)?1TQ/
M0D%,7T-,14%.?2D@>PIM>2`D=F%L=64@/2!S=6)S='(H)'!A<E]C;&5A;BP@
M,3(@*R!L96YG=&@H(D-,14%.(BDI.PHD14Y6>U!!4E]#3$5!3GT@/2`D,2!I
M9B`D=F%L=64@/7X@+UY005)?0TQ%04X]*%Q3*RDO.PI]"GT*<W5B(&]U=',@
M>PIR971U<FX@:68@)'%U:65T.PII9B`H)&QO9V9H*2!["G!R:6YT("1L;V=F
M:"`B0%]<;B(["GT*96QS92!["G!R:6YT(")`7UQN(CL*?0I]"G-U8B!I;FET
M7VEN8R!["G)E<75I<F4@0V]N9FEG.PIP=7-H($!)3D,L(&=R97`@9&5F:6YE
M9"P@;6%P("1#;VYF:6<Z.D-O;F9I9WLD7WTL('%W*`IA<F-H;&EB97AP('!R
M:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI8F5X<`IV96YD;W)A<F-H97AP
M('9E;F1O<FQI8F5X<`HI.PI]"G!A8VMA9V4@;6%I;CL*<F5Q=6ER92!005([
M"G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?<&%R.PI005(M/FEM<&]R="A`
M<&%R7V%R9W,I.PID:64@<7%;<&%R+G!L.B!#86XG="!O<&5N('!E<FP@<V-R
M:7!T("(D<')O9VYA;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&Y=
M"G5N;&5S<R`M92`D<')O9VYA;64["F1O("1P<F]G;F%M93L*0T]213HZ97AI
M="@D,2D@:68@*"1`(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`
M(&EF("1`.PI].PHD.CI?7T524D]2(#T@)$`@:68@)$`["GT*0T]213HZ97AI
M="@D,2D@:68@*"0Z.E]?15)23U(@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I
M.PID:64@)#HZ7U]%4E)/4B!I9B`D.CI?7T524D]2.PHQ.PI?7T5.1%]?"@``
M``````````#@3P``````````````````6"1!``$``````````````&`D00`!
M``````````````#H1T```0``````````````X$=```$``````````````.@B
M00`!``````````````"`1T```0``````````````0"-!``$`````````````
M`$=#0SH@*$=.52D@.2XS+C`@,C`R,#`S,3(@*$9E9&]R82!#>6=W:6X@.2XS
M+C`M,2D```````````````````!'0T,Z("A'3E4I(#$P+C(N,```````````
M`````````$=#0SH@*$=.52D@,3`N,BXP````````````````````1T-#.B`H
M1TY5*2`Q,"XR+C````````````````````!'0T,Z("A'3E4I(#DN,RXP(#(P
M,C`P,S$R("A&961O<F$@0WEG=VEN(#DN,RXP+3$I````````````````````
M1T-#.B`H1TY5*2`Y+C,N,"`R,#(P,#,Q,B`H1F5D;W)A($-Y9W=I;B`Y+C,N
M,"TQ*0```````````````````$=#0SH@*$=.52D@.2XS+C`@,C`R,#`S,3(@
M*$9E9&]R82!#>6=W:6X@.2XS+C`M,2D```````````````````!'0T,Z("A'
M3E4I(#DN,RXP(#(P,C`P,S$R("A&961O<F$@0WEG=VEN(#DN,RXP+3$I````
M````````````````1T-#.B`H1TY5*2`Y+C,N,"`R,#(P,#,Q,B`H1F5D;W)A
M($-Y9W=I;B`Y+C,N,"TQ*0```````````````````$=#0SH@*$=.52D@.2XS
M+C`@,C`R,#`S,3(@*$9E9&]R82!#>6=W:6X@.2XS+C`M,2D`````````````
M``````!'0T,Z("A'3E4I(#DN,RXP(#(P,C`P,S$R("A&961O<F$@0WEG=VEN
M(#DN,RXP+3$I````````````````````1T-#.B`H1TY5*2`Y+C,N,"`R,#(P
M,#,Q,B`H1F5D;W)A($-Y9W=I;B`Y+C,N,"TQ*0```````````````````$=#
M0SH@*$=.52D@.2XS+C`@,C`R,#`S,3(@*$9E9&]R82!#>6=W:6X@.2XS+C`M
M,2D```````````````````!'0T,Z("A'3E4I(#DN,RXP(#(P,C`P,S$R("A&
M961O<F$@0WEG=VEN(#DN,RXP+3$I````````````````````1T-#.B`H1TY5
M*2`Q,"XR+C````````````````````!'0T,Z("A'3E4I(#$P+C(N,```````
M`````````````````````````0```%@D`0"`K@``0````&`D`0"0K@``0```
M`.@B`0#`K@``0````$`C`0#@K@``0```````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````(````9````',```!R.``!24T13CM-!XF]D78O#G38-W#3D^`$`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````!```$00````X```4!```&40```(X```<!```'$0```,X```
M@!```&,1```0X```<!$``-42```DX```X!(``&`3```\X```8!,``#H4``!(
MX```0!0``"@5``!0X```,!4``%H7``!DX```8!<``-(7``!\X```X!<``!08
M``"$X```(!@``",8``",X```,!@``",:``"0X```,!H``%4:``"HX```8!H`
M`'P:``"LX```@!H``-P?``"PX```X!\``$L@``#`X```4"```&T@``#(X```
M<"```%8A``#0X```8"$``"XI``#@X```,"D``*XK``#XX```L"L``-TL```,
MX0``X"P``!U#```LX0``($,``'9#``!$X0``@$,```A%``!,X0``$$4``$-'
M``!@X0``T$H``.)*``!PX0``\$H``/%*``!XX0```$L```%+``!\X0``$$L`
M`!%+``"`X0``($L``"%+``"$X0``,$L``.!.``"(X0``X$X``/E.``"4X0``
M`$\``+Q/``"8X0``P$\``,5/``"@X0``T$\``-9/``"DX0``X$\``.%/``"H
MX0``,%```%U0``"LX0``@%```*A2```<X0``L%(``+52``"TX0``````````
M```````````````````````````````````!!`$`!$(```$````!`````0P'
M``Q""#`'8`9P!5`$P`+0```!$0D`$0$6``HP"6`(<`=0!L`$T`+@```!"@0`
M"@$7``,P`L`!!@(`!C("P`$5"``5`0<"#3`,8`MP"E`)P`+0`1`)`!!B##`+
M8`IP"5`(P`;0!.`"\````04"``52`3`!!`$`!$(```$````!&0H`&0$;`A$P
M$&`/<`Y0#<`+T`G@`O`!`````0````$)!0`)P@4P!&`#<`+````!!0(`!5(!
M,`$$`0`$0@```0H%``J"!C`%8`3``M````$9"@`9`3,0$3`08`]P#E`-P`O0
M">`"\`$-!P`-8@DP"&`'<`;`!-`"X````0D%``EB!3`$8`-P`L````$*!0`*
MH@8P!6`$P`+0```!$PH`$P%7``PP"V`*<`E0",`&T`3@`O`!!`$`!$(```$,
M!P`,0@@P!V`&<`50!,`"T````0H&``HR!C`%8`1P`U`"P`$$`0`$0@```0``
M``$````!`````0````$&`P`&0@(P`6````$````!!`$`!&(```$````!````
M`0````$&`@`&,@+``0``````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```'W1]@`````-(0`0`!````$0```!$````H$`$`;!`!`+`0`0`P*0``4"``
M`(!0```88```,!H``+`K```@&```8!H``.`?```P&```@!H``&`A``!P(```
M$$4``"!#``"`0P``X!<``-H0`0#P$`$`"Q$!`!`1`0`:$0$`)Q$!`#,1`0!$
M$0$`4!$!`%X1`0!K$0$`>!$!`(41`0"7$0$`H1$!`*H1`0"U$0$````!``(`
M`P`$``4`!@`'``@`"0`*``L`#``-``X`#P`0`'!A<BYE>&4`6%-?26YT97)N
M86QS7U!!4E]"3T]4`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`&UA:6X`
M;7E?<&%R7W!L`'!A<E]B87-E;F%M90!P87)?8VQE86YU<`!P87)?8W5R<F5N
M=%]E>&5C`'!A<E]D:7)N86UE`'!A<E]E;G9?8VQE86X`<&%R7V9I;F1P<F]G
M`'!A<E]I;FET7V5N=@!P87)?;6MT;7!D:7(`<&%R7W-E='5P7VQI8G!A=&@`
M<VAA7V9I;F%L`'-H85]I;FET`'-H85]U<&1A=&4`>'-?:6YI=```````````
M````````````````````````````````````````````````````````````
M````````````````````4"`!````````````'"H!`+@B`0#P(0$`````````
M``"`*@$`6"0!`*@B`0```````````)0J`0`0)0$`````````````````````
M```````@)0$``````#`E`0``````/"4!``````!,)0$``````%@E`0``````
M:"4!``````!\)0$``````)`E`0``````H"4!``````"L)0$``````+PE`0``
M````R"4!``````#4)0$``````-PE`0``````Z"4!``````#\)0$``````!`F
M`0``````("8!```````L)@$``````#0F`0``````0"8!``````!()@$`````
M`%0F`0``````8"8!``````!L)@$``````'0F`0``````?"8!``````"()@$`
M`````)0F`0``````H"8!``````"L)@$``````+0F`0``````O"8!``````#(
M)@$``````-PF`0``````\"8!```````()P$``````!`G`0``````'"<!````
M```H)P$``````#0G`0``````/"<!``````!()P$``````%`G`0``````7"<!
M``````!H)P$``````'0G`0``````@"<!``````",)P$``````)@G`0``````
MI"<!`````````````````+`G`0``````P"<!``````#0)P$``````.0G`0``
M````_"<!```````4*`$``````"0H`0``````."@!``````!(*`$``````%@H
M`0``````;"@!``````"$*`$``````)0H`0``````J"@!``````"\*`$`````
M`,PH`0``````X"@!``````#P*`$```````0I`0``````%"D!```````@*0$`
M`````#`I`0`````````````````\*0$`````````````````("4!```````P
M)0$``````#PE`0``````3"4!``````!8)0$``````&@E`0``````?"4!````
M``"0)0$``````*`E`0``````K"4!``````"\)0$``````,@E`0``````U"4!
M``````#<)0$``````.@E`0``````_"4!```````0)@$``````"`F`0``````
M+"8!```````T)@$``````$`F`0``````2"8!``````!4)@$``````&`F`0``
M````;"8!``````!T)@$``````'PF`0``````B"8!``````"4)@$``````*`F
M`0``````K"8!``````"T)@$``````+PF`0``````R"8!``````#<)@$`````
M`/`F`0``````""<!```````0)P$``````!PG`0``````*"<!```````T)P$`
M`````#PG`0``````2"<!``````!0)P$``````%PG`0``````:"<!``````!T
M)P$``````(`G`0``````C"<!``````"8)P$``````*0G`0``````````````
M``"P)P$``````,`G`0``````T"<!``````#D)P$``````/PG`0``````%"@!
M```````D*`$``````#@H`0``````2"@!``````!8*`$``````&PH`0``````
MA"@!``````"4*`$``````*@H`0``````O"@!``````#,*`$``````.`H`0``
M````\"@!```````$*0$``````!0I`0``````("D!```````P*0$`````````
M````````/"D!``````````````````\`7U]C>&%?871E>&ET```6`%]?97)R
M;F\````C`%]?9V5T<F5E;G0`````+@!?7VUA:6X`````3`!?7W-P<FEN=&9?
M8VAK`$\`7U]S=&%C:U]C:&M?9F%I;```40!?7W-T86-K7V-H:U]G=6%R9`!7
M`%]?<W1R;F-P>5]C:&L`;`!?9&QL7V-R=#``<P!?:6UP=7)E7W!T<@```(,`
M86-C97-S``````$!8V%L;&]C`````$(!8VQO<V4`0P%C;&]S961I<@``=0%C
M>6=W:6Y?9&5T86-H7V1L;`!W`6-Y9W=I;E]I;G1E<FYA;````),!9&QL7V1L
M;&-R=#````"P`65N=FER;VX```#-`65X:70``#<"9G!R:6YT9@```$,"9G)E
M90``J0)G971P:60`````M`)G971P=W5I9```P@)G971U:60`````A@-L<V5E
M:P"(`VQS=&%T`(T#;6%L;&]C`````)P#;65M8VAR`````)X#;65M8W!Y````
M`*`#;65M;6]V90```*0#;6MD:7(`X`-O<&5N``#D`V]P96YD:7(```#W`W!O
M<VEX7VUE;6%L:6=N`````!P$<'1H<F5A9%]A=&9O<FL`````401P=&AR96%D
M7V=E='-P96-I9FEC````I@1R96%D``"G!')E861D:7(```"I!')E861L:6YK
M``"L!')E86QL;V,```#:!')M9&ER`&T%<W!R:6YT9@```'4%<W1A=```?P5S
M=')C:'(`````@05S=')C;7``````A@5S=')D=7``````D`5S=')L96X`````
ME05S=')N8VUP````H05S=')S='(`````IP5S=')T;VL`````#09U;FQI;FL`
M````%P!03%]D;U]U;F1U;7```&T`4$Q?=&AR7VME>0````!K`5!E<FQ?871F
M;W)K7VQO8VL``&P!4&5R;%]A=&9O<FM?=6YL;V-K`````.4!4&5R;%]C<F]A
M:U]N;V-O;G1E>'0``'@"4&5R;%]G971?<W8```"G`E!E<FQ?9W9?9F5T8VAP
M=@```&D#4&5R;%]M9U]S970```#?`U!E<FQ?;F5W6%,`````X0-097)L7VYE
M=UA37V9L86=S``"G!5!E<FQ?<V%F97-Y<VUA;&QO8P````"&!E!E<FQ?<W9?
M<V5T<'8`CP9097)L7W-V7W-E='!V;@````"[!E!E<FQ?<WES7VEN:70S````
M`+P&4&5R;%]S>7-?=&5R;0`Y!V)O;W1?1'EN84QO861E<@```$H'<&5R;%]A
M;&QO8P````!,!W!E<FQ?8V]N<W1R=6-T`````$T'<&5R;%]D97-T<G5C=`!.
M!W!E<FQ?9G)E90!/!W!E<FQ?<&%R<V4`````4`=P97)L7W)U;@``BP)'971-
M;V1U;&5(86YD;&5!````(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!
M```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$`
M`"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``
M(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@
M`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0!C>6=W:6XQ
M+F1L;``4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`
M%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4
M(`$`8WEG<&5R;#5?,S(N9&QL`"@@`0!+15).14PS,BYD;&P`````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````$``,`
M```P``"`#@```,```(`0````\```@!@````@`0"````````````````````$
M``$```!@``"``@```'@``(`#````D```@`0```"H``"`````````````````
M```!``D$``!0`0`````````````````````!``D$``!@`0``````````````
M```````!``D$``!P`0`````````````````````!``D$``"``0``````````
M`````````0```,`!`(#8``"````````````````````!``D$``"0`0``````
M```````````````!``$````(`0"````````````````````!``D$``"@`0``
M```````````````````!``$````X`0"````````````````````!``D$``"P
M`0``T#$!`&@%`````````````#@W`0"H"`````````````#@/P$`:`4`````
M````````2$4!`*@(`````````````/!-`0`^```````````````P3@$`6`(`
M````````````B%`!`-H%``````````````8`5P!)`$X`10!8`$4````H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\```$`
M!``0$````0`(`&@%```!`"`@```!``@`J`@```(`$!````$`"`!H!0```P`@
M(````0`(`*@(```$````6`(T````5@!3`%\`5@!%`%(`4P!)`$\`3@!?`$D`
M3@!&`$\``````+T$[_X```$``````````````````````#\`````````!``$
M``$```````````````````"X`0```0!3`'0`<@!I`&X`9P!&`&D`;`!E`$D`
M;@!F`&\```"4`0```0`P`#``,``P`#``-`!"`#`````D``(``0!#`&\`;0!P
M`&$`;@!Y`$X`80!M`&4``````"`````L``(``0!&`&D`;`!E`$0`90!S`&,`
M<@!I`'``=`!I`&\`;@``````(````#``"``!`$8`:0!L`&4`5@!E`'(`<P!I
M`&\`;@``````,``N`#``+@`P`"X`,````"0``@`!`$D`;@!T`&4`<@!N`&$`
M;`!.`&$`;0!E````(````"@``@`!`$P`90!G`&$`;`!#`&\`<`!Y`'(`:0!G
M`&@`=````"`````L``(``0!,`&4`9P!A`&P`5`!R`&$`9`!E`&T`80!R`&L`
M<P``````(````"P``@`!`$\`<@!I`&<`:0!N`&$`;`!&`&D`;`!E`&X`80!M
M`&4````@````)``"``$`4`!R`&\`9`!U`&,`=`!.`&$`;0!E```````@````
M-``(``$`4`!R`&\`9`!U`&,`=`!6`&4`<@!S`&D`;P!N````,``N`#``+@`P
M`"X`,````$0````!`%8`80!R`$8`:0!L`&4`20!N`&8`;P``````)``$````
M5`!R`&$`;@!S`&P`80!T`&D`;P!N`````````+`$/#]X;6P@=F5R<VEO;CTB
M,2XP(B!E;F-O9&EN9STB551&+3@B('-T86YD86QO;F4](GEE<R(_/@T*/&%S
M<V5M8FQY('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV
M,2(@;6%N:69E<W1697)S:6]N/2(Q+C`B/@T*("`@(#QA<W-E;6)L>4ED96YT
M:71Y('9E<G-I;VX](C`N,"XP+C`B('1Y<&4](G=I;C,R(B!N86UE/2)005(M
M4&%C:V5R+D%P<&QI8V%T:6]N(B`O/@T*("`@(#QD97-C<FEP=&EO;CY005(M
M4&%C:V5R($%P<&QI8V%T:6]N/"]D97-C<FEP=&EO;CX-"B`@("`\9&5P96YD
M96YC>3X-"B`@("`@("`@/&1E<&5N9&5N=$%S<V5M8FQY/@T*("`@("`@("`@
M("`@/&%S<V5M8FQY261E;G1I='D@='EP93TB=VEN,S(B(&YA;64](DUI8W)O
M<V]F="Y7:6YD;W=S+D-O;6UO;BU#;VYT<F]L<R(@=F5R<VEO;CTB-BXP+C`N
M,"(@<'5B;&EC2V5Y5&]K96X](C8U.35B-C0Q-#1C8V8Q9&8B(&QA;F=U86=E
M/2(J(B!P<F]C97-S;W)!<F-H:71E8W1U<F4](BHB("\^#0H@("`@("`@(#PO
M9&5P96YD96YT07-S96UB;'D^#0H@("`@/"]D97!E;F1E;F-Y/@T*("`@(#QT
M<G5S=$EN9F\@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O<V]F="UC;VTZ87-M
M+G8S(CX-"B`@("`@("`@/'-E8W5R:71Y/@T*("`@("`@("`@("`@/')E<75E
M<W1E9%!R:79I;&5G97,^#0H@("`@("`@("`@("`@("`@/')E<75E<W1E9$5X
M96-U=&EO;DQE=F5L(&QE=F5L/2)A<TEN=F]K97(B('5I06-C97-S/2)F86QS
M92(@+SX-"B`@("`@("`@("`@(#PO<F5Q=65S=&5D4')I=FEL96=E<SX-"B`@
M("`@("`@/"]S96-U<FET>3X-"B`@("`\+W1R=7-T26YF;SX-"B`@("`\8V]M
M<&%T:6)I;&ET>2!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IC
M;VUP871I8FEL:71Y+G8Q(CX-"B`@("`@("`@/&%P<&QI8V%T:6]N/@T*("`@
M("`@("`@("`@/"$M+2!7:6YD;W=S(#@N,2`M+3X-"B`@("`@("`@("`@(#QS
M=7!P;W)T961/4R!)9#TB>S%F-C<V8S<V+3@P93$M-#(S.2TY-6)B+3@S9#!F
M-F0P9&$W.'TB+SX-"B`@("`@("`@("`@(#PA+2T@5VEN9&]W<R!6:7-T82`M
M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>V4R,#$Q-#4W+3$U
M-#8M-#-C-2UA-69E+3`P.&1E964S9#-F,'TB+SX-"B`@("`@("`@("`@(#PA
M+2T@5VEN9&]W<R`W("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED
M/2)[,S4Q,SAB.6$M-60Y-BTT9F)D+3AE,F0M83(T-#`R,C5F.3-A?2(O/@T*
M("`@("`@("`@("`@/"$M+2!7:6YD;W=S(#@@+2T^#0H@("`@("`@("`@("`\
M<W5P<&]R=&5D3U,@260](GLT83)F,CAE,RTU,V(Y+30T-#$M8F$Y8RUD-CED
M-&$T839E,SA](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@,3`@+2T^
M#0H@("`@("`@("`@("`\<W5P<&]R=&5D3U,@260](GLX93!F-V$Q,BUB9F(S
M+31F93@M8CEA-2TT.&9D-3!A,35A.6%](B\^#0H@("`@("`@(#PO87!P;&EC
M871I;VX^#0H@("`@/"]C;VUP871I8FEL:71Y/@T*/"]A<W-E;6)L>3X-"@``
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````4```
M#````,BB````8```#````!B@````H```&````("ND*Z@KK"NP*[0KN"N````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
-````````````````````
