package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.30.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.30.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`$*1.5X``````````/``+@(+`@(=`$(```#(````$@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!P`0``!```,I(!``,``(``
M`"`````````0```````````0````````$``````````````0`````!`!`+T!
M````(`$`%`L````P`0!0*P```-```.`!``````````````!@`0`P`````,``
M`!P`````````````````````````````````````````````````````````
MT"(!`(`"```````````````````````````````````N=&5X=`````A`````
M$````$(````$``````````````````!@`%!@+F1A=&$```!@`````&`````"
M````1@``````````````````0`!@P"YR9&%T80``G$$```!P````0@```$@`
M`````````````````$``4$`N8G5I;&1I9#4`````P`````(```"*````````
M``````````!``#!`+G!D871A``#@`0```-`````"````C```````````````
M````0``P0"YX9&%T80``T`$```#@`````@```(X``````````````````$``
M,$`N8G-S`````.`1````\`````````````````````````````"``&#`+F5D
M871A``"]`0```!`!```"````D```````````````````0``P0"YI9&%T80``
M%`L````@`0``#````)(``````````````````$``,,`N<G-R8P```%`K````
M,`$`4"L```">``````````````````!``##`+G)E;&]C```P`````&`!```"
M````R@``````````````````0``P0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0VE/0``
MZ-`W``!%,<`QTC')Z.0W``!%,<`QTC')Z.@W``!%,<`QTC')Z.PW``!%,<`Q
MTC')2(/$*.GL-P``D)"0D)"0D)"0D)"054B)Y4R-!:5/``!(C0T.````,=)=
MZ08U``!F#Q]$``!52(GE7<.0D)"0D)"0D)"0055!5%575E-(@^PH2(7)2(G-
M28G4#X2E````2(L%3IT``$R+*$V%[0^$D@````^V`83`=`0\/75+,?])BUT`
M2(7;=#!(8_],B>YF#Q]$``!)B?A(B>I(B=GHPC8``(7`=0:`/#L]=#A(@\8(
M2(L>2(7;==Q(B=A(@\0H6UY?74%<05W#2(G/2(/'`0^V!X3`=`0\/77Q*>_K
MH68/'T0``$B-7#L!3"GN2,'^`TB)V$&)-"1(@\0H6UY?74%<05W#,=OKL&:0
M9BX/'X0``````%575E-(@^PX2(LM<9P``$B+/8J<``!(C5PD)$B)SDB+10!(
MB40D*#'`2(G:2(GQZ/S^__](A<!T.$AC5"0D2(L'2(T$T$B+4`A(A=)(B1!T
MV$B#P`A(BU`(2(722(D0=?!(B=I(B?'HQ/[__TB%P'7(2(M$)"A(,T4`=0E(
M@\0X6UY?7</HUC,``)`/'T0``$%7059!54%455=64TB![,@```!(BRW6FP``
M2(G+2(M%`$B)A"2X````,<!(A<ET!8`Y`'4K2(N$)+@```!(,T4`#X45`0``
M2('$R````%M>7UU!7$%=05Y!7\,/'T0``.B;-```2(7`28G$=,A,C2VL70``
M3(TUIUT``$R-?"0P3(GAZ)@T``!(A<`/A*\```!,C4@8N0(```!,B>],B<[S
MI@^7P!P`A,!TU;D#````3(G.3(GW\Z8/E\`<`(3`=+](B=E,B4PD*.CB-```
M3(M,)"A(B<9,B<GHTC0``(U,!@)(8\GHMC,``$R+3"0H2(T5+UT``$B)QDF)
MV$B)P>A<-```2(GQ3(GZZ&$T``"#^/](B?%T$(M$)$`E`/```#T`0```="[H
MQ#0``$B)\>@,,P``Z4?___\/'X``````3(GAZ,@R``!(B=GH`#0``.G@_O__
MZ*;^___KT.AO,@``D`\?0`!F+@\?A```````05=!5D%505155U932(/L2$B+
M-6F:``!(B=-(B<](BP9(B40D.#'`,<"`.CT/E,!(`<-(B=GH!C0``$B-5"0T
M2(GY2(G%Z.;\__](A<!)B<0/A/H```!(B<'HXC,``$@YZ`^#R@```$QC="0T
M3(LE+9H``$G!Y@,/M@>$P`^$CP$``#P]#X2'`0``2(GXZP:0@/H]=`M(@\`!
M#[80A-)U\$@I^$B830,T)$B-3"@"Z'TR``!(A<!)B08/A&X!``!(8U0D-$F+
M!"1(C4\!2(L4T`^V!SP]B`)U&>L;#Q]$``!(@\$!#[9!_TB#P@$\/8@"=`2$
MP'7JQ@(],<`/'T0```^V#`.(3`(!2(/``83)=?`QP$B+?"0X2#,^#X47`0``
M2(/$2%M>7UU!7$%=05Y!7\-(@\,!#[9#_TF#Q`&$P$&(1"3_=>OKR`\?A```
M````3(LE29D``$V+/"1)@S\`#X2[````3(GX,=+K!P\?0`!$B>I(@\`(1(UJ
M`4B#.`!U[X/"`TACTDC!X@.+!1_K``"%P'5;2(G1QP4.ZP```0```.B),0``
M2(7`#X1]````36/53(GZ2(G!3HTTU0````!,B50D*$V)\.B2,0``3(M4)"A)
MB<%)B00D2\=$T0@`````1(EL)#3IBO[__P\?`$R)^>C8,0``2(7`28G!28D$
M)'0I36/53HTTU0````#KR3'`Z8W^__\/'X0``````+H0````13'MZ5[___^X
M_____^G;_O__Z"0P``"0#Q\`5E-(@>RX````2(L=,)@``$B-5"0@2(G.2(L#
M2(F$)*@````QP.BV,```A<!U,HM4)#"!X@#P``"!^@!```!T)('Z`*```'0<
M2(N,)*@```!(,PMU)DB!Q+@```!;7L.0,<#KY+H"````2(GQZ,\O``"%P`^4
MP`^VP.O-Z*`O``"0#Q]$``!F+@\?A```````055!5+@X$```55=64^C^+@``
M2"G$2(LUE)<``$&X_P\``$B-7"0@2(G-2(T-NED``$F)U4B)VDB+!DB)A"0H
M$```,<#HIS```(7`#XZ/````C5#_2(L-)9<``$AC^H!\/"``3(LA=%A(8_A)
MB?BZ+P```$B)V>CU+P``2(7`=$1,B>'H>"X``$F)^4F)V$B)ZDB)P>C/+0``
M2(N$)"@0``!(,P9U1DB!Q#@0``!;7E]=05Q!7<-F+@\?A```````A=)UITR)
MX>@T+@``38GH2(GJ2(G!Z)8M``#KO0\?0`!(BP69E@``3(L@Z]CHGRX``)`/
M'T``9BX/'X0``````$B#["A(BP5UE@``2(L(Z.TM``!,C0W&6P``3(T%+RT`
M`$B-%=)8``!(B<%(@\0HZ5PM``!FD&8N#Q^$``````!32(/L,$B+'326``!(
MBPOHK"T``$B)P>BD____2(L+Z)PM``!,C0VK6```3(T%[@T``$B-%9Y8``#'
M1"0H`````$B)P4R)3"0@Z/HL``"02(/$,%O#D&8N#Q^$```````QP,,/'P!F
M+@\?A```````05=!5KC8$```055!5%575E/H2BT``$@IQ$B+!>"5``!,C6PD
M0$B)RTB-#4U8``!(B=9,B>I(BSA(B;PDR!```#'_Z&GX__^Z+P```$B)V4B)
MQ>@Y+P``2(7`=$1(C0TB6```2(G:Z!7[__](B=A(BSV+E0``2(N<),@0``!(
M,Q\/A7@!``!(@<38$```6UY?74%<05U!7D%?PV8/'T0``$B)V>@8+P``2(T5
MXU<``$B)\4&)QN@V+P``2(7`2(G&=)J`/@!,C:0DP````$R-/<=7```/A+\`
M``!F#Q]$``!(A>UT$TB)\DB)Z>BP+@``A<`/A'X```!(B?'HP"X``(G"2)B`
M?`;_+W46#Q]$``"#Z@%(8\*`?`;_+\8$!@!T[T&-1!8!/?\/```/CRO___],
MC0U<5P``,=)!N``0``!,B>%(B5PD,$R)?"0H2(ET)"#H>2P``$R)ZDR)X>@>
M+@``A<!U$(M$)%`E`/```#T`@```=%M(C1455P``,<GH;"X``$B%P$B)Q@^$
MS/[__X`^``^%1____TB%[70B2(T]BU8``+D"````2(GN\Z8/E\`<`(3`=+X/
M'X0``````$B--6E6``"Z`0```.E0____N@$```!,B>'H(BP``(7`=91(C0V<
M5@``3(GBZ(_Y__],B>'HMRT``.EP_O__Z-TK``"09I!F+@\?A```````#[81
M2(G(A-)T&&8/'T0``$B#P0&`^B\/MA%(#T3!A-)U[L,/'P!F+@\?A```````
M5E-(@^PH2(7)2(G+=%(/MC%`A/9T2NAE+0``2(U$`_](.<-R%.LG9@\?A```
M````2(/H`4@YPW0:@#@O=/+K!8`X+W0R2(/H`4@YPW+R2#G#=21`@/XO2(T-
M_54``'4'2(T-B54``$B#Q"A;7NG^+```@#@O=0E(@^@!2#G#<O)(*=A(C5`"
M2('Z`!```'<P3(U``4B-#575``!!N0`0``!(B=KH!RL``$B-!4#5``!(@\0H
M6U[#9@\?A```````,<#K[&:09BX/'X0``````%575E-(@^QH2(LMX9(``$B+
M%3I6``!(C0UP50``2(U<)"Q(BT4`2(E$)%@QP$B+!116``!(B50D.$B+%2!6
M``!(B40D,$B+!0Q6``!(B50D2$B)1"1`#[<%"U8``&:)1"10Z`GV__](C0TH
M50``Z/WU__](C0TD50``Z/'U__](C0TA50``Z.7U__](C0T<50``Z-GU__](
MC0T@50``Z,WU__](C0W"5```Z,'U__](C0T450``Z+7U__](C0T250``Z*GU
M__](C0T050``Z)WU__](C0V;5```Z)'U__](C0T"50``2(G:Z++T__](A<!T
M#TB-#=I4``!(B<+H;O?__TB-#?!4``!(B=KHC_3__TB%P'0/2(T-[E0``$B)
MPNA+]___2(T-ZE0``$B)VNAL]/__2(7`=#=(C0TL5```2(G"Z"CW__](C17S
M5```2(T-6%0``.@5]___2(M$)%A(,T4`=5Y(@\1H6UY?7<.02(T-KU0``$B)
MVN@A]/__2(7`=2](C70D04B-/:=4``"Y"@```/.F#Y?`'`"$P'6H2(U4)$M(
MC0T>5```Z+_V___KE4B-#1!4``!(B<+HKO;__^N$Z`<I``"09@\?1```4TB#
M[#!(BQT4D0``2(L#2(E$)"@QP(L%1$,``(/X_W002(M,)"A(,PMU.TB#Q#!;
MPTB-5"0D2(T-O5,``.B.\___2(G",<!(A=)T$@^V$H#Z,`^5P3'`A-(/E<`A
MR(D%_$(``.N[Z)4H``"0#Q]``$B#["CH)RD``(G!Z!`I``!(A<!T"TB+`$B#
MQ"C##Q\`,<#K]&:09BX/'X0``````%575E-(@^Q(2(LU<9```$B)STB-5"0T
M2(T-W%,``$B+!DB)1"0X,<#H`_/__TB%P$B)PTB)^G18@#@`=%-(B<'H&RH`
M`$B%P'522(GYZ.XI``!(B=E(B<7HXRD``$B-3`4"Z,DH``!(C16F4@``3(T-
MG5(``$B)Q4B)7"0@28GX2(G!Z&@I``!(B>H/'T0``$B-#6-3``#H9/7__TB+
M1"0X2#,&=0E(@\1(6UY?7</HK"<``)"09BX/'X0``````$%705:XJ($``$%5
M05155U93Z`HG``!(*<1(BRV@CP``2(U\)%!)B<Q(C0T-4@``2,>$).``````
M````2,>$)#@!````````2(M%`$B)A"28@0``,<!(C07O4@``2(GZ2,>$)``!
M````````2(F$)-````!(C05V40``2(F$)-@```!(C0524@``2(F$)!`!``!(
MC06X4@``2(F$)!@!``!(C06P4@``2(F$)"`!``!(C06I4@``2(F$)"@!``!(
MC06?4@``2(F$)#`!``!(C05K4@``2(F$)/````!(C06%4@``2(F$)/@```#H
M@/'__TB%P'0)@#@`#X4B`@``Z`W^__](A<!(B<,/A&$"``!(B=GH:2@``$B-
M3``!Z$\G``"`.P!)B<5T)DR--4%2``!,B>Y$#[8#2(GQ3(GR2(/#`4B#Q@+H
MYB<``(`[`'7D2(V<)!`!``!(C0U]40``ZQ!(@\,(2(L+2(7)#X13`0``2(GZ
MZ/CP__](A<!(B<9TX(`X`'3;2(G!Z//U__^%P'3/2(GQZ-<G``!(A<!(B<,/
MA!X!``!(B=J+"DB#P@2-@?_^_O[WT2'()8"`@(!TZ8G!P>D0J8"````/1,%(
MC4H"B<9(#T310`#&3(GI2(/:`T@IVD2+`4B#P01!C8#__O[^0??01"'`)8"`
M@(!TY4&)P$'!Z!"I@(```$$/1,!,C4$"B<9)#T3(0`#&2(/9`TPIZ8VT"@0$
M``!(8_9(B?'H/"8``$F)QDB-!4%1``!,C0T64```2(T5*E$``$F)V$R)\4R)
M;"0H2(E$)"#HSR8``+K``0``3(GQZ$(F``"#^/\/A%\!``!(B?I,B?'HWB4`
M`(7`08G%=12+1"1@)0#P```]`$````^$>`$``$F+'"3H2B0``$B+2!A(C147
M40``38GQ28G8,=OH0B4``.E]````,?9(8\9(BYS$T````$B%VW0[@#L`=#9(
MB=F#Q@'HF_3__X7`=-Q(B=GH?R8``$B%P$B)PP^%J/[__TACQDB+G,30````
M2(7;=<@/'P`QV^F._O__9@\?A```````2(G!2(E$)$CH$_S__TB+1"1(2(G!
MZ#8F``!(B<,/'P!(B[PDF($``$@S?0!(B=@/A14$``!(@<2H@0``6UY?74%<
M05U!7D%?PP\?@`````!(C0W(3P``,?9(A<ET)V:02(GZ@\8!Z.7N__](A<!T
M!8`X`'4K2&/&2(N,Q/````!(A<EUV[D-````2(T=DT\``.BZ)```28G%Z6O]
M__]FD$B)P>BH)0``2(7`2(G##X4[_?__Z[_HE2,``(,X$0^$D_[__^B'(P``
M28L<)(LPZ/PB``!(BT@82(T5B4\``$V)\4F)V(ET)"`QV^CP(P``Z2O___^+
M7"1HZ"(D```YPP^%=_[__XM$)&`E_P$``#W``0``#X5C_O__2(GQZ"XD``!(
MC0WQ3@``2(GZ2(G#Z!SN__](A<`/A$,!``!(B<'H"R4``$F+#"1(B<+H3_7_
M_TB%P$F)QP^$(P$``$R)^>C[)```@_@'?CI(8]!(C3V'3P``N0D```!)C707
M^/.F#Y?"@-H`A-)U&H/X"`^$`@(``(/H"$B808!\!_\O#X3Q`0``Z-?Y__^%
MP'47N@```0!,B?GHYB,``(7`B<8/A<P```!(C14G3@``2(T-J$T``$B--2U-
M``!,C25;30``2(T]/T\``.@T\/__Z`\C``!,C0U#30``2(T5&4\``$B)="0H
MB40D($V)\$B)V>C](P``ZS+H-B(``(,X$74Z08/%`>C8(@``2(ET)#!$B6PD
M*$V)X8E$)"!-B?!(B?I(B=GHR2,``+K``0``2(G9Z#PC``"#^/]TO$R)\>B/
M(@``2(T-M$P``$B)VNBP[___2(G9Z*CY___IH_W__P\?`$V+/"3IU/[__P\?
M@`````!(C;PDD`$``$&X`@```$C'PN[___^)P>B$(@``0;@&````2(GZB?'H
M!",``("\))`!````=0Z`O"21`0``0P^$!`$``$4QP#'2B?'H4"(``.C;&P``
M28G$ZQ1F#Q]$``!!B<!(B?I,B>'H(AP``$&X`(```$B)^HGQZ+(B``"%P'_>
M3(V\)&`!``")\4B-O"1``0``Z(<A``!-C6\H3(GB3(TE&DT``$B)^4R)_NAN
M'0``1`^V!TB)\4R)XDB#Q@)(@\<!Z+<B``!).?5UY$B-!:%+``#&A"2(`0``
M`$R)?"0@2(E$)"A,C0V]2P``2(T5A$T``$V)\$B)V>B`(@``Z<3^__]-BV0D
M"$V%Y`^$`?[__TR)X>BU(@``2(/X`P^&[_W__TF-=`3\2(T]0TT``+D%````
M\Z8/E\`<`(3`30]$_.G,_?__@+PDD@$``$$/A>[^__^`O"23`0``0P^%X/[_
M_X"\))0!``!(#X72_O__@+PDE0$``$4/A<3^__]!N`(```!(Q\+&____B?'H
M#"$``$&X*0```$B)^HGQZ(PA``!(C07+2@``2(E\)"!(B40D*.DM____Z#$@
M``"005=!5D%505155U932(/L.$B+-0F(``!,BS4RB```3(UL)"1(BPY)BP9(
MB40D*#'`Z&L?``!(BPY(B</H8!\```^VD[@```!(C0W`2@``B)"Y````3(GJ
MZ*3J__](A<!T"8`X,0^$U@(``.CQ]/__2(L.Z"D?``!(C15I3```0;D+````
M0;@!````2(G!Z*X>``!(A<`/A+<```"#/>[9```!2(M`$$B+'DR+8"@/CJ(`
M```Q[;\!````ZS</'T``28M7$$B+!;W9``!(8\](BPS(Z!DA``"%P(L5L=D`
M``^%Y`$``(/'`8/%`3G7#XW6`0``2(G9Z*4>``!%,<E,8\5,B>)(B<'H5!X`
M`$B%P`^$JP$``$R+.$B+'D&+1PPE``0@`#T`!```=)5(B=GH;1X``$R)^D&Y
M`@```$4QP$B)P>C1'0``2(L>2(G"Z7+___](BQY(B=GH0QX``$B-%8A+``!!
MN0,```!!N`$```!(B<'HR!T``$B%P$B)PW0R2(M`$$B+#DB+..@1'@``3(T%
M6$L``$B)P4B)^NAO'0``2(M#$$B+&/9##D`/A<X!``!(BP[HYAT``$B-%3)+
M``!!N0,```!!N`$```!(B<'H:QT``$B%P$B)PW0X2(T-\4@``$R)ZN@4Z?__
M2(7`2(G'#X18`0``2(M#$$B+#DB+&.B9'0``28GX2(G!2(G:Z/L<``!(BP[H
M@QT``$B-%=%*``!!N`$```!(B<'H%AT``$B+#DB)P^AC'0``3(T%QDH``$B)
MVDB)P>C!'```2(L.Z$D=``!(C0UN2```QH"Y`````$R)ZNB3Z/__2(7`2(D%
M(=@```^$DP```+K``0``2(G!Z+8>``"-4`&#^@%V"8/X$0^%`@$``$B+1"0H
M23,&#X7O````2(/$.%M>7UU!7$%=05Y!7\-F#Q]$``"+%<K7``!(BQZ-0O\Y
MZ`^%@?[__X/Z`0^.>/[__[\!````2(G9@\<!Z+,<``!,B>)(B<'H8!P``#D]
MDM<``$B+'G_?Z4W^__\/'X0``````$B+#7'7``#H=/7__TB%P$B)!7+7```/
MA5'___](C0W=20``Z!@<```/'X0``````(!X`0`/A"7]___I&_W__Y!(BP4Q
MUP``2(L.2(LX2(M#$$B+&.@W'```2(GZ2(G9Z"SM___IG/[__P\?@`````!(
MBP[H&!P``$B)VDB)P>BE&P``Z1K^___HDQP``$&)P4B+!>'6``!,BP7JU@``
M2(T-@TD``$B+$.B3&P``D&:05U932(/L($B)R^C!\/__#[802(G&A-)T&0\?
M@`````!(@\`!@/HO#[802`]$\(32=>[H.?/__TB%VW0@A<!T'(`[`'072(T]
M`T@``+D$````\Z8/E\`<`(3`=`A(@\0@6UY?PTB)V4B#Q"!;7E_I+>C__Y"0
MD)"0D)"0D)"0D)!!5T%6055!5%575E-(@>RX`@``2(L%]H,``$4QTDR-63A,
MC40D($B+.$B)O"2H`@``,?]#BP033XL,$P_(3(G*B<!(P>H@2XD$4$R)R$G!
MZ3A(P>@H)0#_``!,"<A)B=%(P>((2<'A&('B``#_`$6)R4P)R$@)T$N)1%`(
M28/""$F#^D!UK4V-B``"```/'P!)BT!H23-`0$F#P`A),T`(23-`^$B-%`!(
MP>@?2`G0@^#_28E`>$TYR'752(L92(MY$$B+<2!(BVD83(M9"$B)V$B)VDC!
MZAM(P>`%2`G0`T0D($V)V$G!X!Z-E#"9>8):2(GX2#'H3"'82#'H1(TD`DR)
MV$B+5"0H2,'H`DD)P$B)^$V)X46)PDB-E!69>8):2<'I&TPQT$@AV$@Q^$@!
MT$R)XDC!X@5,"<J--`)(B=I(B=A(P>@"2,'B'D@)PDB+1"0P28GU23'02<'M
M&T&)T4TAX$B-A`>9>8):33'020'`2(GP2,'@!4P)Z$:-+`!,B>!)P>P"2,'@
M'DP)X$R+9"0X2#'"08G`2"'R3XV4(IEY@EI,,<I-B>Q)P>P;3`'238GJ2<'B
M!4T)XD6-)!))B?)(P>X"2<'B'DD)\DB+="1`3#'01(G23"'H38V,,9EY@EI,
M,<!,B>9(P>X;3`'(38GA2<'A!4D)\4&--`%-B>E)P>T"2<'A'DT)Z4R+;"1(
M33'*1(G(32'B3XV$*)EY@EI),=))B?5)P>T;30'028GR2<'B!4T)ZD>-+`)-
MB>!)P>P"2<'@'DT)X$R+9"1033'!18G"22'Q2HV4(IEY@EI),<%-B>Q,`<I-
MB>E)P>$%2<'L&TT)X46-)!%(B?)(P>X"2,'B'D@)\DB+="1823'008G132'H
M2(V$,)EY@EI-,=!,B>9(P>X;3`'`38G@2<'@!4D)\$&--`!,B>A)P>T"2,'@
M'DP)Z$R+;"1@2#'"08G`3"'B3XV4*IEY@EI,,<I)B?5)P>T;3`'228GR2<'B
M!4T)ZD6-+!)-B>))P>P"2<'B'DT)XDR+9"1H3#'01(G22"'P3XV,(9EY@EI,
M,<!-B>Q)P>P;3`'(38GI2<'A!4T)X46-)`%)B?%(P>X"2<'A'DD)\4B+="1P
M33'*1(G(32'J38V$,)EY@EI),=),B>9(P>X;30'038GB2<'B!4D)\D.--`)-
MB>A)P>T"2<'@'DT)Z$R+;"1X33'!18G"32'A2HV4*IEY@EI),<%)B?5)P>T;
M3`'*28GQ2<'A!4T)Z46-+!%,B>))P>P"2,'B'DP)XDR+I"2`````23'008G1
M22'P2HV$()EY@EI-,=!-B>Q)P>P;3`'`38GH2<'@!4T)X$6-)`!(B?!(P>`>
M2,'N`D@)\$B+M"2(````2#'"08G`3"'J38V4,IEY@EI,,<I,B>9(P>X;3`'2
M38GB2<'B!4D)\D&--!)-B>I)P>T"2<'B'DT)ZDR+K"20````3#'01(G23"'@
M3XV,*9EY@EI,,<!)B?5)P>T;3`'(28GQ2<'A!4T)Z46--`%-B>%)P>P"2<'A
M'DT)X4R+I"28````33'*1(G(22'R3XV$()EY@EI),=)-B?1)P>P;30'038GR
M2<'B!4T)XD>-)`))B?!,BY0DH````$C![@))P>`>20GP33'!2HV4$IEY@EI-
MB>)-(?%)P>H;1(G&23'!3`'*38GA2<'A!4T)T4R+E"2H````18TL$4R)\DG!
M[@)(P>(>3`GR2HV$$)EY@EI-B>I),=!)P>H;08G132'@23'P3`'`38GH2<'@
M!4T)T$6-%`!,B>!)P>P"2,'@'DP)X$R+I"2P````2#'"08G`3"'J2HVT)IEY
M@EI,,<I-B=1)P>P;2`'R3(G62,'F!4P)YHGV2`'R3(GN2<'M`DC!YAY!B=1,
M">Y,BZPDN````$@Q\$PAT$^-C"F9>8):3#'`38GE2<'M&TP!R$V)X4G!X05-
M">E)B?5%B<E,`<A-B=%)P>H"2<'A'D&)QDT)T4R+E"3`````33'-1#'J38GU
M3XV$$*'KV6Y)P>T;38GB2<'B'DG![`))`=!,B?)-">)(P>(%3`GJB=)-C2P0
MB?)(`Y0DR````$R)SDPQUD6)Z#'P38G$2(V4`J'KV6Y,B<!)P>P;2,'@!4P)
MX(G`3(TD`DR)\D2)R$G![@)(`X0DT````$C!XAY,"?)-B=%$B>9),=%%,<U.
MC8PHH>O9;DB)\$F)]4C!X`5)P>T;3`GHB<!,`<A-B<%)P>@"2<'A'D&)Q4T)
MP46)T$P#A"38````28G2B=)(`Y0DX````$TQRD4QU$V)ZD^-I""AZ]EN38GH
M2<'J&TG!X`5-"=!%B<!/C10$28GP2,'N`DG!X!Y)"?!,B<Y%B=1,,<8Q\$R)
MYDB-E`*AZ]EN3(G@2,'N&TC!X`5("?")P$B--`),B>A$B<I)P>T"2`.4).@`
M``!(P>`>3`GH38G!08GV23'!18G`13'*3HV,$J'KV6Y,B?)-B?))P>H;2,'B
M!4P)THG238T4$4R)XDG![`)(P>(>28G!B<!,">),`X0D\````$6)U4DQT4@#
MA"3X````1#'.38V,,*'KV6Y-B>A,B>Y(P>X;2<'@!4D)\$6)P$N--`%-B?%)
MP>X"2<'A'DF)T$T)\4&)]$TQR$4QPDZ-A!"AZ]EN3(G@38GB2<'J&TC!X`5,
M"=")P$V-%`!-B>B)T$G![0)(`X0D``$``$G!X!Y-">A,B<I%B=9,,<(QUDB-
ME#"AZ]EN3(GP3(GV2,'N&TC!X`5("?")P$B--`),B>!$B<I)P>P"2`.4)`@!
M``!(P>`>3`G@38G!08GU23'!18G`13'*3HV,$J'KV6Y,B>I-B>I(P>(%2<'J
M&TP#A"00`0``3`G2B=)-C1013(GR2<'N`DC!XAY)B<&)P$P)\D6)U$@#A"08
M`0``23'11#'.38V,,*'KV6Y-B>!,B>9(P>X;2<'@!4D)\$6)P$N--`%-B>E)
MP>T"2<'A'DF)T$T)Z4&)]DTQR$4QPDZ-A!"AZ]EN3(GP38GR2<'J&TC!X`5,
M"=")P$V-%`!-B>")T$G![`)(`X0D(`$``$G!X!Y-">!,B<I%B=5,,<)-B>PQ
MUDG![!M(C90PH>O9;DR)Z$C!X`5,">")P$R-)`),B?!)P>X"2,'@'D2)RDV)
MP4P)\$@#E"0H`0``1(GF23'!18G`3`.$)#`!``!%,<I.C8P2H>O9;DB)\DF)
M\DG!ZAM(P>(%3`G2B=)-C1013(GJ2<'M`DC!XAY)B<&)P$P)ZD6)UD@#A"0X
M`0``23'138GU13',2<'M&T^-C""AZ]EN38GP2<'@!4T)Z$6)P$^-+`%)B?%(
MP>X"2<'A'DF)T$D)\46)[$TQR$4QPDZ-A!"AZ]EN3(G@38GB2<'J&TC!X`5,
M"=")P$V-%`!-B?")T$G![@)(`X0D0`$``$G!X!Y-"?!,B<I$B=9,,<)%B<E!
M,=5(B?).C:PHH>O9;DB)\$C!ZAM(P>`%2`G03`.,)$@!``")P$F-5`4`3(G@
M2<'L`DC!X!Y,">!-B<1!B=9),<1%B<!,`X0D4`$``$4QXD^-I!&AZ]EN38GQ
M38GR2<'J&TG!X05-"=%%B<E/C10,28GT2,'N`DG!Y!Y)B<&)P$D)]$6)U4@#
MA"18`0``33'A1#'*38GI38V$$*'KV6Y,B>I)P>D;2,'B!4P)RD*--`),B?))
MP>X"2,'B'DV)X$6)Y$P)\DDQT$&)T44QPDZ-A!"AZ]EN2(GP28GR2<'J&TC!
MX`5,"=!-B>I)P>T"2<'B'D:--`!,B<A-">I,BZPD8`$``$6)T$P)TD@A\DPA
MP$@)PKC<O!N/20'%30'L38GU3`'B38GT2<'M&TG!Y`5-">Q-B<5!`=1(B?)(
MP>X"2,'B'D@)\HG620G222'U32'R30GJ3(NL)&@!``!)`<5-`>E-B>5-`=%-
MB>))P>T;2<'B!4T)ZD>-+`I-B?%)P>X"2<'A'DT)\4F)]D6)RDP)RDTAUDPA
MXDP)\DR+M"1P`0``20'&30'P38GN3`'"38GH2<'N&TG!X`5-"?!%C30038G@
M2<'L`DG!X!Y-">!-B=1$B<)-"<%)(=1-(>E-">%,BZ0D>`$``$D!Q$P!YDV)
M]$D!\4R)]DG![!M(P>8%3`GF1HTD#DR)[DG![0)(P>8>3`GN28G508GQ20GP
M32'-32'P30GH3(NL)(`!``!)`<5-`>I-B>5-`=!-B>))P>T;2<'B!4T)ZD>-
M+`)-B?))P>X"2<'B'DT)\DV)SD6)T$P)UDPAYDTAQDP)]DR+M"2(`0``20'&
M3`'R38GN2`'R3(GN2<'N&TC!Y@5,"?9$C3063(GB2<'L`DC!XAY,">)-B<2)
MUDD)TDDA]$TAZDT)XDR+I"20`0``20'$30'A38GT30'138GR2<'L&TG!X@5-
M">)'C20*38GI2<'M`DG!X1Y-">E)B?5%B<I,"<I-(=5,(?),">I,BZPDF`$`
M`$D!Q4T!Z$V)Y4P!PDV)X$G![1M)P>`%30GH18TL$$V)\$G![@))P>`>30GP
M38G61(G"30G!22'632'A30GQ3(NT)*`!``!)`<9,`?9-B>Y)`?%,B>Y)P>X;
M2,'F!4P)]D:--`Y,B>9)P>P"2,'F'DP)YDF)U$&)\4D)\$TAS$TAZ$T)X$R+
MI"2H`0``20'$30'B38GT30'038GR2<'L&TG!X@5-">)'C20"38GJ2<'B'DG!
M[0)-">I-B<U%B=!,"=9-(<5,(?9,">Y,BZPDL`$``$D!Q4P!ZDV)Y4@!\DR)
MYDG![1M(P>8%3`GN1(TL%DR)\DG![@)(P>(>3`GR38G&B=9)"=))(?9-(>)-
M"?),B[0DN`$``$D!QDT!\4V)[DT!T4V)ZDG![AM)P>(%30GR1XTT"DV)X4G!
M[`))P>$>30GA28GT18G*3`G*32'43"'J3`GB3(ND),`!``!)`<1-`>!-B?1,
M`<)-B?!)P>P;2<'@!4T)X$6-)!!-B>A)P>T"2<'@'DT)Z$V)U42)PDT)P4DA
MU4TA\4T)Z4R+K"3(`0``20'%3`'N38GE20'Q3(GF2<'M&TC!Y@5,">Y&C2P.
M3(GV2<'N`DC!YAY,"?9)B=9!B?%)"?!-(<Y-(>!-"?!,B[0DT`$``$D!QDT!
M\DV)[DT!T$V)ZDG![AM)P>(%30GR1XTT`DV)XDG!XAY)P>P"30GB38G,18G0
M3`G632'$3"'N3`GF3(ND)-@!``!)`<1,`>)-B?1(`?),B?9)P>P;2,'F!4P)
MYD2-)!9,B>I)P>T"2,'B'DP)ZDV)Q8G620G222'U32'R30GJ3(NL).`!``!)
M`<5-`>E-B>5-`=%-B>))P>T;2<'B!4T)ZD>-+`I-B?%)P>X"2<'A'DF)\DT)
M\46)SDP)RDTA\DPAXDP)TDR+E"3H`0``20'"30'038GJ3`'"38GH2<'J&TG!
MX`5-"=!%C10038G@2<'L`DG!X!Y,B?)-">!%B<1-"<%,(>)-(>E)"=%(BY0D
M\`$``$@!PD@#A"3X`0``2`'R3(G620'13(G22,'N&TC!X@5("?),`?!,B[0D
M"`(``(G220'13(GJ2<'M`DC!XAY$B<Y,">I-B>5!B==)"=!-(?U-(=!-">A)
MB?5,`<!)B?!)P>`%2<'M&TT)Z$6)P$P!P$V)T$G!Z@))P>`>08G%30G"0;C6
MP6+*3#'230'&03'12(N4)``"``!,`<),`>)-B>Q,`<I-B>E)P>P;2<'A!4T)
MX4F)]$C![@))P>0>18G)3`GF38G43`'*23'T08G118G21#'@3XTD/DV)SDG!
M[AM)`<1,B<A(P>`%3`GPB<!)`<1,B>A)P>T"2,'@'D6)YDD)Q4B+A"00`@``
M3`'`3`'028GRB?9-,>I$,=)-B?)(`<),B?!)P>H;2,'@!4P)T(G`2`'"3(G(
M2<'I`DC!X!Y!B=))"<%(BX0D&`(``$P!P$@!\$R)[D6)[4PQSD0QYDV)U$@!
MQDR)T$G![!M(P>`%3`G@B<!(`<9,B?!)P>X"2,'@'D&)]$P)\$R+M"0@`@``
M30'&30'U38G.18G)23'&1#'R38GF20'53(GB2<'N&TC!X@5,"?*)TDD!U4R)
MTDG!Z@)(P>(>18GN20G22(N4)"@"``!,`<),`<I)B<&)P$TQT40QSDV)\4@!
MUDR)\DG!Z1M(P>(%3`G*38GA2<'L`HG22<'A'D@!UDB+E"0P`@``30GA38G4
M08GW18G233',3`'"13'L2`'03(GZ20'$3(GX2,'J&TC!X`5("=")P$D!Q$R)
M\$G![@)(P>`>18GE20G&2(N$)#@"``!,B>I(P>H;3`'`3`'038G*18G)33'R
M03'R20'"3(GH2,'@!4@)T$R)^DG![P*)P$C!XAY)`<)(BX0D0`(``$P)^D2)
MUDP!P$P!R$V)\46)]DDQT44QS$F)\4D!Q$B)\$G!Z1M(P>`%3`G(28G1B=*)
MP$D!Q$R)Z$G![0)(P>`>18GG20G%2(N$)$@"``!-,>E%,<I-B?E,`<!)P>D;
M3`'P20'"3(GX2,'@!4P)R$R+C"10`@``B<!)`<)(B?!(P>X"2,'@'DT!P46)
MUD@)\$R)[DP!RD@QQDV)\46)[40QYDG!Z1M(`=9,B?)(P>(%3`G*28G!B<")
MTD@!UDR)^DG![P)(P>(>08GT20G72(N4)%@"``!-,?E%,<I-B>%,`<))P>D;
M3`'J20'23(GB2,'B!4P)RDV)\4G![@*)TDG!X1Y)`=)(BY0D8`(``$T)\46)
MU4P!PD@!T$R)^D6)_TPQRC'63(GJ2`'&3(GH2,'J&TC!X`5("=!,B<I%B<F)
MP$@!QDR)X$G![`)(P>`>08GV20G$2(N$)&@"``!,,>)!,=),B?),`<!(P>H;
M3`'X20'"3(GP2,'@!4@)T$R)ZDG![0*)P$C!XAY)`<)(BX0D<`(``$P)ZD6)
MUTR+K"2``@``3`'`3`'(38GA18GD23'103'Q3(G^20'!3(GX2,'N&TC!X`5(
M"?")P$D!P4R)\$G![@)(P>`>1(G.20G&2(N$)'@"``!,`<!,`>!)B=2)TDTQ
M]$4QU$F)\DD!Q$B)\$G!ZAM(P>`%3`G030'%B<!,`>I-B?5)`<1,B?A)P>\"
M2,'@'D6)XD6)]DP)^$DQQ44QZ4V)U4D!T4R)TDG![1M(P>(%3`GJB=))`=%(
MB?)(P>X"2,'B'D6)S4@)UDB+E"2(`@``3`'"3`'R28G&B<!),?9%,?1-B>Y,
M`>)-B>Q)P>X;2<'D!4T)]$V)UDG!Z@))P>8>18GD30GR3(NT))`"``!,`>)!
MB=1-`<9,`?!)B?9-,=9%,?%-B>9,`X0DF`(``$P!R$V)X4G![AM)P>$%30GQ
M00'!3(GH2<'M`DC!X!Y)"<5,B>!)P>P"2,'@'D0![4D)Q(GP2(EI&$D!P$R)
MT$0#42!,,>A$`><QPDB)>1!)C0003(G*38G(2,'B!4G!Z!M%`=E,"<),B4D(
M3(E1(`'0`<-(BP43;@``2(D92(N<)*@"``!(,QAU%$B!Q+@"``!;7E]=05Q!
M74%>05_#Z,H%``"09@\?A```````2(/L*+F`````Z((&``"ZB:O-[[G^W+J8
M2,<``2-%9TB)4`A(@>J9R?HK2(E($$C'0!AV5#(02(E0($C'0"@`````2,=`
M,`````#'0'@`````2(/$*,-F+@\?A```````055!5%575E-(@^PH26/`3(M!
M*$B)STB)TTB+5S!(B<5(C7<X08T,P$PYP4B)3RA(8T]X2(/2`$C!Z!U(`=!(
MB4<PA<ET2D&]0````$B)VD$IS40Y[40/3NU(`?%-8^5-B>#HZ04``(M'>$0!
MZ(/X0(E'>'0-2(/$*%M>7UU!7$%=PTB)^40I[4P!X^C@Z/__@_T_?G&#[4!!
MB>U!P>T&18GL28/$`4G!Y`9)`=P/'P!(BP-(BU,(2(GY2(/#0$B)!DB)5@A(
MBT/02(M3V$B)1A!(B5882(M#X$B+4^A(B48@2(E6*$B+0_!(BU/X2(E&,$B)
M5CCH=NC__TPYXW6Q0<'E!D0I[4ACQ4B#^`AR4TB+$TB)5SA(BU0#^$B)5`;X
M2(U6"$B#XOA(*=9(*?-(`<9(@^;X2(/^"'(72(/F^#'`2(L,`TB)#`)(@\`(
M2#GP<N^);WA(@\0H6UY?74%<05W#J`1U&TB%P'3G#[83J`*(5SATW0^W5`/^
M9HE4!O[KT8L3B5<XBU0#_(E4!OSKPD%455=64TB#["!(BWHH2(MJ,$B)SDR-
M8CA(B=-(B?A(P>@#@^`_C4@!2)C&1`(X@$ACP4P!X(/Y.`^.@0$``+I`````
M*<IT$3')08G(@\$!.=%"Q@0``'+Q2(G9Z'_G__](QT,X`````$C'0T``````
M2<=$)!``````2<=$)!@`````2<=$)"``````2<=$)"@`````2<=$)#``````
M2(GH2(G90(AK<TC!Z!A`B'MWB$-P2(GH2,'H$(A#<4B)Z$C!Z`B(0W)(B?A(
MP>@8B$-T2(GX2,'H$(A#=4B)^$C!Z`B(0W;H]N;__TB+`TB)V4C!Z!B(!DB+
M`TC!Z!"(1@%(BP-(P>@(B$8"2(L#B$8#2(M#"$C!Z!B(1@1(BT,(2,'H$(A&
M!4B+0PA(P>@(B$8&2(M#"(A&!TB+0Q!(P>@8B$8(2(M#$$C!Z!"(1@E(BT,0
M2,'H"(A&"DB+0Q"(1@M(BT,82,'H&(A&#$B+0QA(P>@0B$8-2(M#&$C!Z`B(
M1@Y(BT,8B$8/2(M#($C!Z!B(1A!(BT,@2,'H$(A&$4B+0R!(P>@(B$822(M#
M((A&$TB#Q"!;7E]=05SI@0(``)"Z.````$4QP"G*@_H(<D.)T4C'``````!(
MQT0(^`````!(C4@(2(/A^$@IR`'"@^+X@_H(#X*D_O__@^+X,<!!B<&#P`@Y
MT$Z)!`ER\NF,_O__]L($=2F%T@^$?_[___;"`L8```^$<_[__S')9HE,$/[I
M9_[__V8/'X0``````,<``````,=$$/P`````Z4O^__^0D)"0D)"0D)"0D)"0
M_R7JX```D)#_)=K@``"0D/\ERN```)"0_R6ZX```D)#_):K@``"0D/\EFN``
M`)"0_R6*X```D)#_)7K@``"0D/\E:N```)"0_R5:X```D)#_)4K@``"0D/\E
M.N```)"0_R4JX```D)#_)1K@``"0D/\E"N```)"0_R7ZWP``D)#_)>K?``"0
MD/\EVM\``)"0_R7*WP``D)#_);K?``"0D/\EJM\``)"0_R6:WP``D)#_)8K?
M``"0D`\?A```````_R7:W0``D)"0D)"0D)"0D/\ERMX``)"0D)"0D)"0D)#_
M)<+>``"0D)"0D)"0D)"045!(/0`0``!(C4PD&'(92('I`!```$B#"0!(+0`0
M``!(/0`0``!WYT@IP4B#"0!86<.0D)"0D)"0D)"0D)"0D/\E6MT``)"0D)"0
MD)"0D)#_)5+=``"0D)"0D)"0D)"0_R52W0``D)"0D)"0D)"0D/\E2MT``)"0
MD)"0D)"0D)#_)4+=``"0D)"0D)"0D)"0_R5"W0``D)"0D)"0D)"0D/\E2MT`
M`)"0D)"0D)"0D)#_)4K=``"0D)"0D)"0D)"0_R5"W0``D)"0D)"0D)"0D/\E
M6MT``)"0D)"0D)"0D)#_)5+=``"0D)"0D)"0D)"0_R5*W0``D)"0D)"0D)"0
MD/\E0MT``)"0D)"0D)"0D)#_)3K=``"0D)"0D)"0D)"0_R4RW0``D)"0D)"0
MD)"0D/\E*MT``)"0D)"0D)"0D)#_)2+=``"0D)"0D)"0D)"0_R4:W0``D)"0
MD)"0D)"0D/\E$MT``)"0D)"0D)"0D)#_)0K=``"0D)"0D)"0D)"0_R4"W0``
MD)"0D)"0D)"0D/\E^MP``)"0D)"0D)"0D)#_)?+<``"0D)"0D)"0D)"0_R7J
MW```D)"0D)"0D)"0D/\E^MP``)"0D)"0D)"0D)#_)?+<``"0D)"0D)"0D)"0
M_R7JW```D)"0D)"0D)"0D/\EXMP``)"0D)"0D)"0D)#_)=K<``"0D)"0D)"0
MD)"0_R72W```D)"0D)"0D)"0D/\ERMP``)"0D)"0D)"0D)#_)<+<``"0D)"0
MD)"0D)"0_R6ZW```D)"0D)"0D)"0D/\ELMP``)"0D)"0D)"0D)#_):K<``"0
MD)"0D)"0D)"0_R6BW```D)"0D)"0D)"0D/\EFMP``)"0D)"0D)"0D)#_)9+<
M``"0D)"0D)"0D)"0_R6*W```D)"0D)"0D)"0D$B#["@QTNA5````_Q4?VP``
MD)"0D)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0
MD)##D)"0D)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D%932(/L*$B)TTB)
MSKD(````Z.H$``!(A=L/A#0"``!(B7-`2(TUEO___TB-%6\'``!(N0````!4
M`0``2(7`2,<#`````$B)LX````!(C35]____2(E32$B+%7K:``!(B8OP````
M2(T-3`<``$B)LX@```!(C35F____QT,0`@```$B)2U!(N4@!``"Y"P``2(E3
M&$B)LY````!(C15?MP``2(TU2/___TB)2PA(C0TM_O__2(E3.$B-%2+]__](
MB;.8````2(TU=/W__TB)2S!(B5,H2(T5-00``$B)<R!(C37Z`P``2(F3&`$`
M`$B)<W@/A.4```!(BY#@````2(72#X35````BP4#MP``A<`/A<<```!(@SW#
M%@````^$+0(``$B#/;T6````#X2/`@``2(,]MQ8````/A'$"``!(@SVQ%@``
M``^$4P(``$B#/:L6````#X0U`@``2(,]I18````/A!<"``!(@SV?%@````^$
M^0$``$B#/9D6````#X3;`0``3(L-5!8``$R+%546``!,B0I,B5((3(L-3Q8`
M`$R+%5`6``!,B4H03(E2&$R+#4D6``!,BQ5*%@``3(E*($R)4BA,BPU#%@``
M3(L51!8``$R)2C!,B5(X,<GH?0,``$B)@^@```!(C07/%0``,<E(B4-82(T%
M(A8``$B)0V!(C06WI0``2(E#:$B-!8RW``!(B4-P2(T%`6<``$B)@R`!``!(
MC04O9P``2(F#*`$``$B-!86U__](B8,P`0``Z(D"``"X`0```$B#Q"A;7L-(
MN4@!``"Y"P``2+H`````5`$``$B)<$!(B4@(2(F0\````$B-#2,%``!(C14T
M!0``2(TU-?W__TB%P$B)2$A(BXA``0``2(E04$B+%338``!(B;"`````2(TU
M'OW__\=`$`(```!(B0I(B;"(````2(T-%OW__TB--1_]__](C14HM0``2,<`
M`````$B)B)````!(C0U3^___2(FPF````$B-->7Z__](B5`X2(E(($B-%=;[
M__](C0W/`0``2(EP*$B--?0!``!(B5`P2(E(>$B)L!@!```/A*O^__](BY#@
M````2(G#2(72#X7#_?__Z9/^__\/'X``````2(L"2(D%AA0``.G$_?__D$B+
M0CA(B06M%```Z17^__](BT(P2(D%E10``.GW_?__2(M"*$B)!7T4``#IV?W_
M_TB+0B!(B05E%```Z;O]__](BT(82(D%310``.F=_?__2(M"$$B)!344``#I
M?_W__TB+0@A(B04=%```Z6']__],BP5)M```BQ5+M```2(L-3+0``.G7````
M#Q^``````$B#[#B#^@%T9W(U@_H"=!.#^@-U'+H#````2(/$..FM````N@(`
M``!(@\0XZ9\```"X`0```$B#Q#C##Q]$```QTNB)````A<!TZTB+#=:S``")
M1"0LZ*4```!(QP7"LP``_____XM$)"SKR@\?0``QP$B-%6?___]-A<`/E,!(
MB0VZLP``3(D%H[,``,<%H;,```$```")!8.S``#HC@```$B#^/](B05[LP``
M#Y7`#[;`2(/$.,,/'X0``````.D[____D)"0D)"0D)"0D)"X`0```,.0D)"0
MD)"0D)"0PY"0D)"0D)"0D)"0D)"0D/\E`M8``)"0D)"0D)"0D)#_)0K6``"0
MD)"0D)"0D)"0_R4"U@``D)"0D)"0D)"0D/\E@M8``)"0D)"0D)"0D)!32(/L
M($B)RTB)T4B-%0ZS``#HR?K__TB-%0*S``!(B=E(@\0@6^D%````D)"0D)#_
M);K5``"0D)"0D)"0D)"0_R6:UP``D)`/'X0``````%=64TB#[%")3"1P2(E4
M)#A,B40D,.C&]___2(L=_U\``$B-5"0X3(U$)#!(C4PD<$B+`TB)1"1(,<#H
MD?;__TR-!?+V__](C0WS]O__3(G"Z`OW__](BP6$7P``2(LUC5\``(`X`'4S
MZ$OV__](A<!(B07YL0``#X2.`0``2(G!Z"OV__](BS5D7P``2(L.Z-SV___&
M@`D*````2(L.Z,WV__^`B'($```"2(L.Z+[V__^`B'($```!BT0D<(U(!4AC
MR4C!X0/H)/;__TB+3"0X28G!2(D%?;$``$2+7"1PQP5VL0``!````$B+`4&#
M^P%%C5,$28D!2(T%2B0``$F)00A(BP5>$0``28E!$$B-!3<D``!)B4$8?BQ%
MC5,$2(/!"+H%````#Q^$``````!,BP%(8\*#P@%(@\$(03G238E$P?AUZ4B+
M#2*Q``!-8])%C4,#2(T55,C__TO'1-'X`````$C'1"0@`````.@E]?__A<")
MQG1=2(L-\+```$B-?"1$Z![U__](C0U=(0``2(GZZ#?!__](A<!T)4B+#<NP
M``#H]O3__^@9]?__2(M\)$A(,SN)\'5+2(/$4%M>7\-(BPV>L```2(7)=!?H
MO-G__^O(2(L-D[```.BN]/__B<;KDTB-#:\@``!(B?KHV\#__TB%P$B)P4B)
M!6:P``!TFNO)Z/7U__^Y`0```.@[]O__D)"0D)"0D)"0D)!52(GE7>EVP/__
MD)"0D)"0___________03T```0``````````````__________\`````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````#_____`````!!S0``!````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````+@`N+@`E<R\E<P`O<')O8R]S
M96QF+V5X90!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D97(``$EN=&5R;F%L
M<SHZ4$%2.CI"3T]4`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"5S)7,E<P`O
M`%!%4DPU3$E"`%!%4DQ,24(`4$523#5/4%0`4$523$E/`%!!4E])3DE424%,
M25I%1`!005)?4U!!5TY%1`!005)?0TQ%04X`4$%27T1%0E5'`%!!4E]#04-(
M10!005)?1TQ/0D%,7T1%0E5'`%!!4E]'3$]"04Q?5$U01$E2`%!!4E]435!$
M25(`4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`4$%27T-,14%.
M/0`Q``````!?7T5.5E]005)?0TQ%04Y?7R`@("`@("`@("`@("`@(```4$%4
M2`!54T52`%-94U1%30`O=&UP`%1-4$1)4@!414U01$E2`%1%35``5$U0`%53
M15).04U%`"4P,G@`)7,E<R5S)7,`<&%R+0``````)7,Z(&-R96%T:6]N(&]F
M('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@``
M`````"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I<R!U;G-A9F4@*'!L
M96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E<F%T:6]N*0H`<&%R
M;"YE>&4`+G!A<@`E<R5S8V%C:&4M)7,E<P`E<R5S=&5M<"TE=25S`"5S)7-T
M96UP+25U+25U)7,`05)'5@`8`'!E<FP`,`!005(Z.E!A8VME<CHZ5D524TE/
M3@`Q+C`T.0!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``````
M````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y
M("5S(&9A:6QE9"`H97)R;F\])6DI"@`M90`M+0``<&5R;'AS:2YC````````
M`&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII
M9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?
M7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1P87)?=&5M<"P@)'!R;V=N86UE
M+"!`=&UP9FEL92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q
M=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U
M:7)E($9I;&4Z.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ
M9&ER;F%M92@D<&%R7W1E;7`I.PIO=71S*'%Q>U)E;6]V:6YG(&9I;&5S(&EN
M("(D<&%R7W1E;7`B?2D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H
M("UD("D@/R!R;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D
M<&%R7W1E;7`["FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R
M)RD@>PIM>2`D=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%(#T^("=T
M;7!86%A86"<L"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P
M9&ER*2P*4U5&1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD["G!R:6YT
M("1T;7`@(B,A+V)I;B]S:`IX/3$[('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D
M;PIR;2`M<F8@)R1P87)?=&5M<"<*:68@6R!<(2`M9"`G)'!A<E]T96UP)R!=
M.R!T:&5N"F)R96%K"F9I"G-L965P(#$*>#U@97AP<B!<)'@@*R`Q8`ID;VYE
M"G)M("<B("X@)'1M<"T^9FEL96YA;64@+B`B)PHB.PIC:&UO9"`P-S`P+"1T
M;7`M/F9I;&5N86UE.PIM>2`D8VUD(#T@)'1M<"T^9FEL96YA;64@+B`G(#XO
M9&5V+VYU;&P@,CXF,2`F)SL*8VQO<V4@)'1M<#L*<WES=&5M*"1C;60I.PIO
M=71S*'%Q*%-P87=N960@8F%C:V=R;W5N9"!P<F]C97-S('1O('!E<F9O<FT@
M8VQE86YU<#H@*0HN("1T;7`M/F9I;&5N86UE*3L*?0I]('T*0D5'24X@>PI)
M;G1E<FYA;',Z.E!!4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ
M4$%2.CI"3T]4.PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PIE=F%L('L*
M7W!A<E]I;FET7V5N=B@I.PIM>2`D<75I970@/2`A)$5.5GM005)?1$5"54=]
M.PIM>2`E0V]N9FEG(#T@*`IP871H7W-E<"`@("`]/B`H)%Y/(#U^("]>35-7
M:6XO(#\@)SLG(#H@)SHG*2P*7V5X92`@("`@("`@/3X@*"1>3R`]?B`O7B@_
M.DU35VEN?$]3,GQC>6=W:6XI+R`_("<N97AE)R`Z("<G*2P*7V1E;&EM("`@
M("`@/3X@*"1>3R`]?B`O7DU35VEN?$]3,B\@/R`G7%PG(#H@)R\G*2P**3L*
M7W-E=%]P<F]G;F%M92@I.PI?<V5T7W!A<E]T96UP*"D["FUY("@D<W1A<G1?
M<&]S+"`D9&%T85]P;W,I.PI["FQO8V%L("1324=[7U]705).7U]](#T@<W5B
M('M].PIO<&5N(%]&2"P@)SPG+"`D<')O9VYA;64@;W(@;&%S=#L*8FEN;6]D
M92A?1D@I.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US("1P<F]G;F%M93L*;7D@
M)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1M86=I8U]P;W,["FEF("@D
M<VEZ92`\/2`D8VAU;FM?<VEZ92D@>PHD;6%G:6-?<&]S(#T@,#L*?2!E;'-I
M9B`H*&UY("1M(#T@)'-I>F4@)2`D8VAU;FM?<VEZ92D@/B`P*2!["B1M86=I
M8U]P;W,@/2`D<VEZ92`M("1M.PI](&5L<V4@>PHD;6%G:6-?<&]S(#T@)'-I
M>F4@+2`D8VAU;FM?<VEZ93L*?0IW:&EL92`H)&UA9VEC7W!O<R`^/2`P*2![
M"G-E96LH7T9(+"`D;6%G:6-?<&]S+"`P*3L*<F5A9"A?1D@L("1B=68L("1C
M:'5N:U]S:7IE("L@;&5N9W1H*"1005)?34%'24,I*3L*:68@*"AM>2`D:2`]
M(')I;F1E>"@D8G5F+"`D4$%27TU!1TE#*2D@/CT@,"D@>PHD;6%G:6-?<&]S
M("L]("1I.PIL87-T.PI]"B1M86=I8U]P;W,@+3T@)&-H=6YK7W-I>F4["GT*
M;&%S="!I9B`D;6%G:6-?<&]S(#P@,#L*<V5E:R!?1D@L("1M86=I8U]P;W,@
M+2`T+"`P.PIR96%D(%]&2"P@)&)U9BP@-#L*<V5E:R!?1D@L("1M86=I8U]P
M;W,@+2`T("T@=6YP86-K*").(BP@)&)U9BDL(#`["B1D871A7W!O<R`]('1E
M;&P@7T9(.PIM>2`E<F5Q=6ER95]L:7-T.PIR96%D(%]&2"P@)&)U9BP@-#L@
M("`@("`@("`@("`@("`@("`@("`@("`@("`C(')E860@=&AE(&9I<G-T(")&
M24Q%(@IW:&EL92`H)&)U9B!E<2`B1DE,12(I('L*<F5A9"!?1D@L("1B=68L
M(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*;7D@)&9U
M;&QN86UE(#T@)&)U9CL*;W5T<RAQ<2A5;G!A8VMI;F<@9FEL92`B)&9U;&QN
M86UE(BXN+BDI.PIM>2`D8W)C(#T@*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<
M9%U[.'TI+WQ\("D@/R`D,2`Z('5N9&5F.PIM>2`H)&)A<V5N86UE+"`D97AT
M*2`]("@D8G5F(#U^(&U\*#\Z+BHO*3\H+BHI*%PN+BHI?"D["G)E860@7T9(
M+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD[
M"FEF("AD969I;F5D*"1E>'0I(&%N9"`D97AT("%^("]<+B@_.G!M?'!L?&EX
M?&%L*20O:2D@>PIM>2`D9FEL96YA;64@/2!?=&5M<&9I;&4H(B1C<F,D97AT
M(BP@)&)U9BP@,#<U-2D["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9U;&QN
M86UE?2`]("1F:6QE;F%M93L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL
M96YA;65](#T@)&9U;&QN86UE.PI]"F5L<VEF("@@)&9U;&QN86UE(#U^(&U\
M7B\_<VAL:6(O?"!A;F0@9&5F:6YE9"`D14Y6>U!!4E]414U0?2`I('L*;7D@
M)&9I;&5N86UE(#T@7W1E;7!F:6QE*"(D8F%S96YA;64D97AT(BP@)&)U9BP@
M,#<U-2D["F]U=',H(E-(3$E".B`D9FEL96YA;65<;B(I.PI]"F5L<V4@>PHD
M<F5Q=6ER95]L:7-T>R1F=6QL;F%M97T@/0HD4$%2.CI(96%V>3HZ36]D=6QE
M0V%C:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C
M+`IN86UE(#T^("1F=6QL;F%M92P*?3L*?0IR96%D(%]&2"P@)&)U9BP@-#L*
M?0IL;V-A;"!`24Y#(#T@*'-U8B!["FUY("@D<V5L9BP@)&UO9'5L92D@/2!`
M7SL*<F5T=7)N(&EF(')E9B`D;6]D=6QE(&]R("$D;6]D=6QE.PIM>2`D:6YF
M;R`](&1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5](&]R(')E='5R;CL*
M)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H
M)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@
M>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE("0A.PIB
M:6YM;V1E*"1F:"D["G!R:6YT("1F:"`D:6YF;RT^>V)U9GT["G-E96LH)&9H
M+"`P+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@
M7W1E;7!F:6QE*"(D:6YF;RT^>V-R8WTN<&TB+"`D:6YF;RT^>V)U9GTI.PIO
M<&5N(&UY("1F:"P@)SPG+"`D9FEL96YA;64@;W(@9&EE(")C86XG="!R96%D
M("1F:6QE;F%M93H@)"$B.PIB:6YM;V1E*"1F:"D["G)E='5R;B`D9F@["GT*
M9&EE(")";V]T<W1R87!P:6YG(&9A:6QE9#H@8V%N;F]T(&9I;F0@)&UO9'5L
M92%<;B(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!
M4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T
M97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E
M<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K
M97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA
M;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE
M(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*96QS92![
M"F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E8W5S=&]M
M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I;&5N86UE
M?3L*?0IL87-T('5N;&5S<R`D8G5F(&5Q(")02UPP,#-<,#`T(CL*)'-T87)T
M7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T
M(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L
M;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%
M?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE
M;2D*)'%U:65T(#T@,"!U;FQE<W,@)$5.5GM005)?1$5"54=].PII9B`H(21S
M=&%R=%]P;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S
M:&EF="DI('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II
M("`@:6YS=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R
M"G8@("!V97)I9GE?<&%R"BD["FEF("A`05)'5B!A;F0@)$%21U9;,%T@97$@
M)RTM<F5U<V4G*2!["G-H:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?4D554T5]
M(#T@<VAI9G0@0$%21U8["GT*96QS92![(",@;F]R;6%L('!A<FP@8F5H879I
M;W5R"FUY($!A9&1?=&]?:6YC.PIW:&EL92`H0$%21U8I('L*)$%21U9;,%T@
M/7X@+UXM*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q
M(&5Q("=))RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@
M97$@)TTG*2!["F5V86P@(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@
M>PIU;G-H:69T($!P87)?87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@
M>PHD;W5T(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@
M)W-I=&4G.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L92`]("=A;&PG
M.PI]"F5L<VEF("@D,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H
M)#$@97$@)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D:64@(EA8
M6#H@0V%N;F]T(&]P96X@;&]G.B`D(2(["GT*96QS:68@*"0Q(&5Q("=4)RD@
M>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY("1C
M;60@/2`D9&ES=%]C;61[)#%]*2!["F1E;&5T92`D14Y6>R=005)?5$5-4"=]
M.PII;FET7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T
M.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]
M*"1?*2!F;W(@0$%21U8["F5X:70["GT*?0IU;G-H:69T($!)3D,L($!A9&1?
M=&]?:6YC.PI]"GT*:68@*"1O=70I('L*>PIR97%U:7)E($E/.CI&:6QE.PIR
M97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM
M>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D
M<&%R*2!["F]P96X@;7D@)&9H+"`G/"<L("1P87(@;W(@9&EE(")#86YN;W0@
M9FEN9"`G)'!A<B<Z("0A(CL*8FEN;6]D92@D9F@I.PIB;&5S<R@D9F@L("=)
M3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^
M<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP
M.CI!6E]/2R@I("D*;W(@9&EE(")296%D("<D<&%R)R!E<G)O<CH@)"$B.PI]
M"FUY("5E;G8@/2!D;R!["FEF("@D>FEP(&%N9"!M>2`D;65T82`]("1Z:7`M
M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R.B0O
M+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;7CI=
M*RDZ("@N*RDD+VUG.PI]"GT["FQO8V%L("0O(#T@7#0["FEF("AD969I;F5D
M("1P87(I('L*;W!E;B!005(L("<\)RP@)'!A<B!O<B!D:64@(B0A.B`D<&%R
M(CL*8FEN;6]D92A005(I.PID:64@(B1P87(@:7,@;F]T(&$@4$%2(&9I;&4B
M('5N;&5S<R`\4$%2/B!E<2`B4$M<,#`S7#`P-"(["GT*0W)E871E4&%T:"@D
M;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE
M.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]74D].3%DH*2!\($E/.CI&:6QE
M.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE("0A.PIB:6YM;V1E*"1F:"D[
M"B0O(#T@*&1E9FEN960@)&1A=&%?<&]S*2`_(%PD9&%T85]P;W,@.B!U;F1E
M9CL*<V5E:R!?1D@L(#`L(#`["FUY("1L;V%D97(@/2!S8V%L87(@/%]&2#X[
M"FEF("@A)$5.5GM005)?5D520D%424U](&%N9"`D;&]A9&5R(#U^("]>*#\Z
M(R%\7$!R96TI+RD@>PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*
M4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YN97<M/F%P<&QY*%PD;&]A9&5R+"`D
M,"D*?0IF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E96YV*2!["FUY("1V
M86P@/2`D96YV>R1K97E](&]R(&YE>'0["B1V86P@/2!E=F%L("1V86P@:68@
M)'9A;"`]?B`O7ELG(ETO.PIM>2`D;6%G:6,@/2`B7U]%3E9?4$%27R(@+B!U
M8R@D:V5Y*2`N(")?7R(["FUY("1S970@/2`B4$%27R(@+B!U8R@D:V5Y*2`N
M("(])'9A;"(["B1L;V%D97(@/7X@<WLD;6%G:6,H("LI?7L*)&UA9VEC("X@
M)'-E="`N("@G("<@>"`H;&5N9W1H*"0Q*2`M(&QE;F=T:"@D<V5T*2DI"GUE
M9SL*?0HD9F@M/G!R:6YT*"1L;V%D97(I.PHD+R`]('5N9&5F.PII9B`H)&)U
M;F1L92D@>PIR97%U:7)E(%!!4CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET
M7V1Y;F%L;V%D97(H*3L*:6YI=%]I;F,H*3L*<F5Q=6ER95]M;V1U;&5S*"D[
M"FUY($!I;F,@/2!G<F5P('L@(2]"4T1004XO('T*9W)E<"!["B@D8G5N9&QE
M(&YE("=S:71E)RD@;W(**"1?(&YE("1#;VYF:6<Z.D-O;F9I9WMA<F-H;&EB
M97AP?2!A;F0*)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>W!R:79L:6)E>'!]*3L*
M?2!`24Y#.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D
M97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E
M>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T["FUY("5W<FET=&5N.PIF
M;W)E86-H("AS;W)T(&ME>7,@)69I;&5S*2!["FUY("@D;F%M92P@)&9I;&4I
M.PIF;W)E86-H(&UY("1D:7(@*$!I;F,I('L*:68@*"1N86UE(#T@)%!!4CHZ
M2&5A=GDZ.D9U;&Q#86-H97LD7WTI('L*)&9I;&4@/2`D7SL*;&%S=#L*?0IE
M;'-I9B`H+UXH7%$D9&ER7$5<+R@N*EM>0V-=*2E<6B]I*2!["B@D9FEL92P@
M)&YA;64I(#T@*"0Q+"`D,BD["FQA<W0["GT*96QS:68@*&TA7B]L;V%D97(O
M6UXO72LO*"XJ6UY#8UTI7%HA*2!["FEF("AM>2`D<F5F(#T@)%!!4CHZ2&5A
M=GDZ.DUO9'5L94-A8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F
M+"`D,2D["FQA<W0["GT*96QS:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L
M("1N86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*;F5X="!U
M;FQE<W,@9&5F:6YE9"`D;F%M92!A;F0@;F]T("1W<FET=&5N>R1N86UE?2LK
M.PIN97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?
M97AT7$4D+SL*;W5T<R@@:F]I;B`B(BP*<7$H4&%C:VEN9R`B*2P@<F5F("1F
M:6QE(#\@)&9I;&4M/GMN86UE?2`Z("1F:6QE+`IQ<2@B+BXN*0HI.PIM>2`D
M8V]N=&5N=#L*:68@*')E9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^
M>V)U9GT["GT*96QS92!["F]P96X@1DE,12P@)SPG+"`D9FEL92!O<B!D:64@
M(D-A;B=T(&]P96X@)&9I;&4Z("0A(CL*8FEN;6]D92A&24Q%*3L*)&-O;G1E
M;G0@/2`\1DE,13X["F-L;W-E($9)3$4["E!!4CHZ1FEL=&5R.CI0;V13=')I
M<"T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE*0II9B`A)$5.5GM005)?
M5D520D%424U](&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["E!!
M4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT96YT
M+"`D9FEL92P@)&YA;64I.PI]"F]U=',H<7$H5W)I='1E;B!A<R`B)&YA;64B
M*2D["B1F:"T^<')I;G0H(D9)3$4B*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@
M;&5N9W1H*"1N86UE*2`K(#DI*3L*)&9H+3YP<FEN="AS<')I;G1F*`HB)3`X
M>"\E<R(L($%R8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@
M)&YA;64**2D["B1F:"T^<')I;G0H<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N
M="DI*3L*)&9H+3YP<FEN="@D8V]N=&5N="D["GT*?0HD>FEP+3YW<FET951O
M1FEL94AA;F1L92@D9F@I(&EF("1Z:7`["B1C86-H95]N86UE(#T@<W5B<W1R
M("1C86-H95]N86UE+"`P+"`T,#L*:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@
M)&UT:6UE(#T@*'-T870H)&]U="DI6SE=*2!["FUY("1C='@@/2!$:6=E<W0Z
M.E-(02T^;F5W*#$I.PIO<&5N*&UY("1F:"P@(CPB+"`D;W5T*3L*8FEN;6]D
M92@D9F@I.PHD8W1X+3YA9&1F:6QE*"1F:"D["F-L;W-E*"1F:"D["B1C86-H
M95]N86UE(#T@)&-T>"T^:&5X9&EG97-T.PI]"B1C86-H95]N86UE("X](")<
M,"(@>"`H-#$@+2!L96YG=&@@)&-A8VAE7VYA;64I.PHD8V%C:&5?;F%M92`N
M/2`B0T%#2$4B.PHD9F@M/G!R:6YT*"1C86-H95]N86UE*3L*)&9H+3YP<FEN
M="AP86-K*"=.)RP@)&9H+3YT96QL("T@;&5N9W1H*"1L;V%D97(I*2D["B1F
M:"T^<')I;G0H)%!!4E]-04=)0RD["B1F:"T^8VQO<V4["F-H;6]D(#`W-34L
M("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P
M;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ
M.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E
M<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[
M("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O
M<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?
M1D@I+"`G<B<I(&]R(&1I92`B)"$Z("1`(CL*07)C:&EV93HZ6FEP.CIS971#
M:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W
M.PHD>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]
M($%R8VAI=F4Z.EII<#HZ05I?3TLH*2!O<B!D:64@(B0A.B`D0"(["D%R8VAI
M=F4Z.EII<#HZ<V5T0VAU;FM3:7IE*#8T("H@,3`R-"D["G!U<V@@0%!!4CHZ
M3&EB0V%C:&4L("1Z:7`["B1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D
M>FEP.PHD<75I970@/2`A)$5.5GM005)?1$5"54=].PIO=71S*'%Q*%PD14Y6
M>U!!4E]414U0?2`]("(D14Y6>U!!4E]414U0?2(I*3L*:68@*&1E9FEN960@
M)$5.5GM005)?5$5-4'TI('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I
M;G0A"F9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE
M>'0@:68@)&UE;6)E<BT^:7-$:7)E8W1O<GD["FUY("1M96UB97)?;F%M92`]
M("1M96UB97(M/F9I;&5.86UE.PIN97AT('5N;&5S<R`D;65M8F5R7VYA;64@
M/7X@;7L*7@HO/W-H;&EB+PHH/SHD0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO
M*3\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*
M?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I
M;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA
M;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N
M8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q*%-K:7!P:6YG("(D;65M8F5R
M7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE
M(BDI.PI](&5L<V4@>PIO=71S*'%Q*$5X=')A8W1I;F<@(B1M96UB97)?;F%M
M92(@=&\@(B1D97-T7VYA;64B*2D["B1M96UB97(M/F5X=')A8W14;T9I;&5.
M86UE9"@D9&5S=%]N86UE*3L*8VAM;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@
M)%Y/(&5Q(")H<'5X(CL*?0I]"GT*?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE
M>R1P<F]G;F%M97TI('L*9&EE(#P\("(N(B!U;FQE<W,@0$%21U8["E5S86=E
M.B`D,"!;("U!;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S
M<F,N<&%R(%T@6R!P<F]G<F%M+G!L(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I
M;&5=('-R8RYP87(*+@HD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64@
M/2`D,"`]('-H:69T*$!!4D=6*3L*?0IS=6(@0W)E871E4&%T:"!["FUY("@D
M;F%M92D@/2!`7SL*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*;7D@*"1B87-E
M;F%M92P@)'!A=&@L("1E>'0I(#T@1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S
M92@D;F%M92P@*"=<+BXJ)RDI.PIR97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z
M.E!A=&@Z.FUK<&%T:"@D<&%T:"D@=6YL97-S*"UE("1P871H*3L@(R!M:W!A
M=&@@9&EE<R!W:71H(&5R<F]R"GT*<W5B(')E<75I<F5?;6]D=6QE<R!["G)E
M<75I<F4@;&EB.PIR97%U:7)E($1Y;F%,;V%D97(["G)E<75I<F4@:6YT96=E
M<CL*<F5Q=6ER92!S=')I8W0["G)E<75I<F4@=V%R;FEN9W,["G)E<75I<F4@
M=F%R<SL*<F5Q=6ER92!#87)P.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U
M:7)E($5R<FYO.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%
M>'!O<G1E<CL*<F5Q=6ER92!&8VYT;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR
M97%U:7)E($9I;&4Z.E-P96,["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@
M0V]N9FEG.PIR97%U:7)E($E/.CI(86YD;&4["G)E<75I<F4@24\Z.D9I;&4[
M"G)E<75I<F4@0V]M<')E<W,Z.EIL:6(["G)E<75I<F4@07)C:&EV93HZ6FEP
M.PIR97%U:7)E($1I9V5S=#HZ4TA!.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!0
M05(Z.DAE879Y.PIR97%U:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I
M;'1E<CHZ4&]D4W1R:7`["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N
M=&5N=#L*<F5Q=6ER92!A='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@
M?3L*979A;"![(')E<75I<F4@5VEN,S(@?3L*979A;"![(')E<75I<F4@4V-A
M;&%R.CI5=&EL('T["F5V86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"
M=7)S="!].PIE=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R
M92!].PIE=F%L('L@<F5Q=6ER92!097)L24\[(')E<75I<F4@4&5R;$E/.CIS
M8V%L87(@?3L*979A;"![(')E<75I<F4@=71F."!].PI]"G-U8B!?<V5T7W!A
M<E]T96UP('L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'T@86YD("1%3E9[
M4$%27U1%35!](#U^("\H+BLI+RD@>PHD<&%R7W1E;7`@/2`D,3L*<F5T=7)N
M.PI]"F9O<F5A8V@@;7D@)'!A=&@@*`HH;6%P("1%3E9[)%]]+"!Q=R@@4$%2
M7U1-4$1)4B!435!$25(@5$5-4$1)4B!414U0(%1-4"`I*2P*<7<H($,Z7%Q4
M14U0("]T;7`@+B`I"BD@>PIN97AT('5N;&5S<R!D969I;F5D("1P871H(&%N
M9"`M9"`D<&%T:"!A;F0@+7<@)'!A=&@["FUY("1U<V5R;F%M93L*;7D@)'!W
M=6ED.PIE=F%L('LH)'!W=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D969I;F5D
M("0^.WT["FEF("@@9&5F:6YE9"@F5VEN,S(Z.DQO9VEN3F%M92D@*2!["B1U
M<V5R;F%M92`]("97:6XS,CHZ3&]G:6Y.86UE.PI]"F5L<VEF("AD969I;F5D
M("1P=W5I9"D@>PHD=7-E<FYA;64@/2`D<'=U:60["GT*96QS92!["B1U<V5R
M;F%M92`]("1%3E9[55-%4DY!345]('Q\("1%3E9[55-%4GT@?'P@)U-94U1%
M32<["GT*)'5S97)N86UE(#U^(',O7%<O7R]G.PIM>2`D<W1M<&1I<B`]("(D
M<&%T:"1#;VYF:6=[7V1E;&EM?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA
M;64I.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*:68@*"$D14Y6>U!!4E]#3$5!
M3GT@86YD(&UY("1M=&EM92`]("AS=&%T*"1P<F]G;F%M92DI6SE=*2!["F]P
M96X@*&UY("1F:"P@(CPB+B`D<')O9VYA;64I.PIS965K("1F:"P@+3$X+"`R
M.PIS>7-R96%D("1F:"P@;7D@)&)U9BP@-CL*:68@*"1B=68@97$@(EPP0T%#
M2$4B*2!["G-E96L@)&9H+"`M-3@L(#(["G-Y<W)E860@)&9H+"`D8G5F+"`T
M,3L*)&)U9B`]?B!S+UPP+R]G.PHD<W1M<&1I<B`N/2`B)$-O;F9I9WM?9&5L
M:6U]8V%C:&4M(B`N("1B=68["GT*96QS92!["FUY("1D:6=E<W0@/2!E=F%L
M"GL*<F5Q=6ER92!$:6=E<W0Z.E-(03L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!
M+3YN97<H,2D["F]P96XH;7D@)&9H+"`B/"(L("1P<F]G;F%M92D["F)I;FUO
M9&4H)&9H*3L*)&-T>"T^861D9FEL92@D9F@I.PIC;&]S92@D9F@I.PHD8W1X
M+3YH97AD:6=E<W0["GT@+R\@)&UT:6UE.PHD<W1M<&1I<B`N/2`B)$-O;F9I
M9WM?9&5L:6U]8V%C:&4M)&1I9V5S="(["GT*8VQO<V4H)&9H*3L*?0IE;'-E
M('L*)$5.5GM005)?0TQ%04Y](#T@,3L*)'-T;7!D:7(@+CT@(B1#;VYF:6=[
M7V1E;&EM?71E;7`M)"0B.PI]"B1%3E9[4$%27U1%35!](#T@)'-T;7!D:7([
M"FUK9&ER("1S=&UP9&ER+"`P-S4U.PIL87-T.PI]"B1P87)?=&5M<"`]("0Q
M(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK
M*2\["GT*<W5B(%]T96UP9FEL92!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D
M;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92([
M"G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L
M;&YA;64N)"0B.PIO<&5N(&UY("1F:"P@)SXG+"`D=&5M<&YA;64@;W(@9&EE
M(")C86XG="!W<FET92`D=&5M<&YA;64Z("0A(CL*8FEN;6]D92`D9F@["G!R
M:6YT("1F:"`D8V]N=&5N=',["F-L;W-E("1F:#L*8VAM;V0@)&UO9&4L("1T
M96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D
M9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L
M;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6
M>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK
M*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@
M*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%2
M7U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L
M(')I;F1E>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@*R`Q*3L*?0II
M9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1#
M;VYF:6=[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@
M)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O
M9VYA;64D0V]N9FEG>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)$-O;F9I9WM?
M97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D
M0V]N9FEG>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X
M:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0
M?3L*)&1I<B`]?B!S+UQ1)$-O;F9I9WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N
M86UE(#T@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA;64D0V]N9FEG>U]E
M>&5](BDL(&QA<W0I"FEF("US("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N
M86UE)$-O;F9I9WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D0V]N9FEG
M>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D0V]N9FEG
M>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD
M,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD
M97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA
M;64@/2`B+B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]
M("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD
M969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P
M=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1
M)$-O;F9I9WM?9&5L:6U]7$4I+R1P=V0D0V]N9FEG>U]D96QI;7TO.PHD14Y6
M>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E
M;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E
M='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF
M;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!
M344@*2`I('L*9&5L971E("1%3E9[)U!!4E\G+B1??3L*?0IF;W(@*'%W+R!4
M35!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B1%3E9[)U!!4E\G+B1??2`]
M("1%3E9[)U!!4E]'3$]"04Q?)RXD7WT@:68@97AI<W1S("1%3E9[)U!!4E]'
M3$]"04Q?)RXD7WT["GT*;7D@)'!A<E]C;&5A;B`](")?7T5.5E]005)?0TQ%
M04Y?7R`@("`@("`@("`@("`@("(["FEF("@D14Y6>U!!4E]414U0?2D@>PID
M96QE=&4@)$5.5GM005)?0TQ%04Y].PI]"F5L<VEF("@A97AI<W1S("1%3E9[
M4$%27T=,3T)!3%]#3$5!3GTI('L*;7D@)'9A;'5E(#T@<W5B<W1R*"1P87)?
M8VQE86XL(#$R("L@;&5N9W1H*")#3$5!3B(I*3L*)$5.5GM005)?0TQ%04Y]
M(#T@)#$@:68@)'9A;'5E(#U^("]>4$%27T-,14%./2A<4RLI+SL*?0I]"G-U
M8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN
M="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS
M=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P
M(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI
M8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R
M87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I
M<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI
M;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q*'!A<BYP;#H@0V%N)W0@;W!E;B!P
M97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C
M=&]R>5QN*0IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/
M4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD[
M"F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/
M4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<
M9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%
M3D1?7PH`````````````````````$$T``````````````````'`D00`!````
M``````````!X)$$``0``````````````\$1```$``````````````.A$0``!
M````````````````(T$``0``````````````<$1```$``````````````%@C
M00`!``````````````!'0T,Z("A'3E4I(#<N-"XP(#(P,3@Q,C`V("A&961O
M<F$@0WEG=VEN(#<N-"XP+3$I````````````````````1T-#.B`H1TY5*2`W
M+C0N,`````````````````````!'0T,Z("A'3E4I(#<N-"XP````````````
M`````````$=#0SH@*$=.52D@-RXT+C``````````````````````1T-#.B`H
M1TY5*2`W+C0N,"`R,#$X,3(P-B`H1F5D;W)A($-Y9W=I;B`W+C0N,"TQ*0``
M`````````````````$=#0SH@*$=.52D@-RXT+C`@,C`Q.#$R,#8@*$9E9&]R
M82!#>6=W:6X@-RXT+C`M,2D```````````````````!'0T,Z("A'3E4I(#<N
M-"XP(#(P,3@Q,C`V("A&961O<F$@0WEG=VEN(#<N-"XP+3$I````````````
M````````1T-#.B`H1TY5*2`W+C0N,"`R,#$X,3(P-B`H1F5D;W)A($-Y9W=I
M;B`W+C0N,"TQ*0```````````````````$=#0SH@*$=.52D@-RXT+C`@,C`Q
M.#$R,#8@*$9E9&]R82!#>6=W:6X@-RXT+C`M,2D```````````````````!'
M0T,Z("A'3E4I(#<N-"XP(#(P,3@Q,C`V("A&961O<F$@0WEG=VEN(#<N-"XP
M+3$I````````````````````1T-#.B`H1TY5*2`W+C0N,"`R,#$X,3(P-B`H
M1F5D;W)A($-Y9W=I;B`W+C0N,"TQ*0```````````````````$=#0SH@*$=.
M52D@-RXT+C`@,C`Q.#$R,#8@*$9E9&]R82!#>6=W:6X@-RXT+C`M,2D`````
M``````````````!'0T,Z("A'3E4I(#<N-"XP(#(P,3@Q,C`V("A&961O<F$@
M0WEG=VEN(#<N-"XP+3$I````````````````````1T-#.B`H1TY5*2`W+C0N
M,"`R,#$X,3(P-B`H1F5D;W)A($-Y9W=I;B`W+C0N,"TQ*0``````````````
M`````$=#0SH@*$=.52D@-RXT+C``````````````````````1T-#.B`H1TY5
M*2`W+C0N,`````````````````````````````````$```!P)`$`D*T``$``
M``!X)`$`H*T``$``````(P$`T*T``$````!8(P$`\*T``$``````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````@```!D````<P```'(H``%)31%-6>EJ^[.'O(>PK
M?9C^82/.`0``````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````$```1!````#@``!0$```:A````C@``!P
M$```=A```!#@``"`$```1!$``!C@``!0$0``VQ$``"S@``#@$0``0A,``#S@
M``!0$P``C14``%3@``"0%0``$18``&S@```@%@``$A<``'C@```@%P``5!<`
M`(S@``!@%P``M1<``)3@``#`%P``PQ<``)S@``#0%P``U!D``*#@``#@&0``
M`QH``+C@```0&@``U!H``+S@``#@&@``JAP``,C@``"P'```'!T``-C@```@
M'0``1!T``.#@``!0'0``!1X``.C@```0'@``@"4``/C@``"`)0``/BD``!#A
M``!`*0``LRD``"CA``#`*0``YS\``$#A``#P/P``1D```%CA``!00```X$$`
M`&#A``#@00``,T0``'3A``#@1P``\D<``(3A````2````4@``(SA```02```
M$4@``)#A```@2```(4@``)3A```P2```,4@``)CA``!`2```$$P``)SA```0
M3```*4P``*CA```P3```Z$P``*SA``#P3```]4P``+3A````30``!DT``+CA
M```030``$4T``+SA``!@30``BTT``,#A``"P30``QD\``#3A``#03P``VD\`
M`,CA``````````````````````````````````````````````$$`0`$0@``
M`00"!00#`5`!!`(%!`,!4`$,!P`,0@@P!V`&<`50!,`"T````0@%``AB!#`#
M8`)P`5````$3"@`3`1D`##`+8`IP"5`(P`;0!.`"\`$0"0`0@@PP"V`*<`E0
M",`&T`3@`O````$)!``)`1<``C`!8`$5"``5`0<"#3`,8`MP"E`$P`+0`00!
M``1"```!!0(`!5(!,`$````!&0H`&0$;`A$P$&`/<`Y0#<`+T`3@`O`!````
M`08#``9"`C`!8````0@%``C"!#`#8`)P`5````$%`@`%4@$P`00!``1"```!
M"`4`"(($,`-@`G`!4````1D*`!D!-1`1,!!@#W`.4`W`"]`$X`+P`1`)`!!B
M##`+8`IP"5`(P`;0!.`"\````0<$``<R`S`"8`%P`0<$``>2`S`"8`%P`1,*
M`!,!5P`,,`M@"G`)4`C`!M`$X`+P`00!``1"```!#`<`#$((,`=@!G`%4`3`
M`M````$*!@`*,@8P!6`$<`-0`L`!!`$`!$(```$````!`````0````$````!
M!@,`!D(",`%@```!`````00!``1B```!`````0````$````!!0(`!3(!,`$$
M`@4$`P%0````````````````````````````````````````````````````
M`````````````````$*1.5X`````TA`!``$````1````$0```"@0`0!L$`$`
ML!`!`(`E```@'0``L$T``!A@``#@&0``0"D``,`7```0&@``L!P``-`7``#@
M&@``$!X``%`=``#@00``\#\``%!````@%P``VA`!`/`0`0`+$0$`$!$!`!H1
M`0`G$0$`,Q$!`$01`0!0$0$`7A$!`&L1`0!X$0$`A1$!`)<1`0"A$0$`JA$!
M`+41`0````$``@`#``0`!0`&``<`"``)``H`"P`,``T`#@`/`!``<&%R+F5X
M90!84U]);G1E<FYA;'-?4$%27T)/3U0`9V5T7W5S97)N86UE7V9R;VU?9V5T
M<'=U:60`;6%I;@!M>5]P87)?<&P`<&%R7V)A<V5N86UE`'!A<E]C;&5A;G5P
M`'!A<E]C=7)R96YT7V5X96,`<&%R7V1I<FYA;64`<&%R7V5N=E]C;&5A;@!P
M87)?9FEN9'!R;V<`<&%R7VEN:71?96YV`'!A<E]M:W1M<&1I<@!P87)?<V5T
M=7!?;&EB<&%T:`!S:&%?9FEN86P`<VAA7VEN:70`<VAA7W5P9&%T90!X<U]I
M;FET````````````````````````````````````````````````````````
M``````````````````````````````````!0(`$```````````"`*@$`T"(!
M`/`A`0```````````/`J`0!P)`$`P"(!````````````!"L!`$`E`0``````
M`````````````````````%`E`0``````8"4!``````!L)0$``````'PE`0``
M````B"4!``````"8)0$``````*PE`0``````P"4!``````#0)0$``````-PE
M`0``````["4!``````#X)0$```````0F`0``````#"8!```````8)@$`````
M`"PF`0``````0"8!``````!0)@$``````%PF`0``````9"8!``````!P)@$`
M`````'@F`0``````A"8!``````"0)@$``````)PF`0``````I"8!``````"L
M)@$``````+@F`0``````Q"8!``````#0)@$``````-PF`0``````Y"8!````
M``#L)@$``````/@F`0``````#"<!```````@)P$``````#@G`0``````0"<!
M``````!,)P$``````%@G`0``````9"<!``````!L)P$``````'@G`0``````
M@"<!``````",)P$``````)@G`0``````I"<!``````"P)P$``````+PG`0``
M````R"<!``````#4)P$`````````````````X"<!``````#P)P$````````H
M`0``````%"@!```````L*`$``````#PH`0``````3"@!``````!D*`$`````
M`'0H`0``````B"@!``````"8*`$``````*@H`0``````O"@!``````#4*`$`
M`````.@H`0``````^"@!```````,*0$``````"`I`0``````,"D!``````!$
M*0$``````%0I`0``````:"D!``````!X*0$``````(0I`0``````E"D!````
M`````````````*`I`0````````````````!0)0$``````&`E`0``````;"4!
M``````!\)0$``````(@E`0``````F"4!``````"L)0$``````,`E`0``````
MT"4!``````#<)0$``````.PE`0``````^"4!```````$)@$```````PF`0``
M````&"8!```````L)@$``````$`F`0``````4"8!``````!<)@$``````&0F
M`0``````<"8!``````!X)@$``````(0F`0``````D"8!``````"<)@$`````
M`*0F`0``````K"8!``````"X)@$``````,0F`0``````T"8!``````#<)@$`
M`````.0F`0``````["8!``````#X)@$```````PG`0``````("<!```````X
M)P$``````$`G`0``````3"<!``````!8)P$``````&0G`0``````;"<!````
M``!X)P$``````(`G`0``````C"<!``````"8)P$``````*0G`0``````L"<!
M``````"\)P$``````,@G`0``````U"<!`````````````````.`G`0``````
M\"<!````````*`$``````!0H`0``````+"@!```````\*`$``````$PH`0``
M````9"@!``````!T*`$``````(@H`0``````F"@!``````"H*`$``````+PH
M`0``````U"@!``````#H*`$``````/@H`0``````#"D!```````@*0$`````
M`#`I`0``````1"D!``````!4*0$``````&@I`0``````>"D!``````"$*0$`
M`````)0I`0````````````````"@*0$`````````````````#P!?7V-X85]A
M=&5X:70``!8`7U]E<G)N;P```",`7U]G971R965N=``````N`%]?;6%I;@``
M``!,`%]?<W!R:6YT9E]C:&L`3P!?7W-T86-K7V-H:U]F86EL``!1`%]?<W1A
M8VM?8VAK7V=U87)D`%<`7U]S=')N8W!Y7V-H:P!L`%]D;&Q?8W)T,`!S`%]I
M;7!U<F5?<'1R````@P!A8V-E<W,``````0%C86QL;V,`````0@%C;&]S90!#
M`6-L;W-E9&ER``!U`6-Y9W=I;E]D971A8VA?9&QL`'<!8WEG=VEN7VEN=&5R
M;F%L````DP%D;&Q?9&QL8W)T,````+`!96YV:7)O;@```,T!97AI=```-P)F
M<')I;G1F````0P)F<F5E``"I`F=E='!I9`````"T`F=E='!W=6ED``#"`F=E
M='5I9`````"&`VQS965K`(@#;'-T870`C0-M86QL;V,`````G`-M96UC:'(`
M````G@-M96UC<'D`````H`-M96UM;W9E````I`-M:V1I<@#@`V]P96X``.0#
M;W!E;F1I<@```/<#<&]S:7A?;65M86QI9VX`````'`1P=&AR96%D7V%T9F]R
M:P````!1!'!T:')E861?9V5T<W!E8VEF:6,```"F!')E860``*<$<F5A9&1I
M<@```*D$<F5A9&QI;FL``*P$<F5A;&QO8P```-H$<FUD:7(`;05S<')I;G1F
M````=05S=&%T``!_!7-T<F-H<@````"!!7-T<F-M<`````"&!7-T<F1U<```
M``"0!7-T<FQE;@````"5!7-T<FYC;7````"A!7-T<G-T<@````"G!7-T<G1O
M:P`````-!G5N;&EN:P`````5`%!,7V1O7W5N9'5M<```:0!03%]T:')?:V5Y
M`````'`!4&5R;%]A=&9O<FM?;&]C:P``<0%097)L7V%T9F]R:U]U;FQO8VL`
M````>@%097)L7V%V7V9E=&-H`(,!4&5R;%]A=E]S:&EF=`#D`5!E<FQ?8W)O
M86M?;F]C;VYT97AT``!O`E!E<FQ?9V5T7W-V````G0)097)L7V=V7V9E=&-H
M<'8```"1`U!E<FQ?;6=?<V5T````!@1097)L7VYE=UA3``````@$4&5R;%]N
M97=84U]F;&%G<P``R@5097)L7W-A9F5S>7-M86QL;V,`````,09097)L7W-V
M7S)P=E]F;&%G<P"D!E!E<FQ?<W9?<V5T<'8`K09097)L7W-V7W-E='!V;@``
M``#:!E!E<FQ?<WES7VEN:70S`````-L&4&5R;%]S>7-?=&5R;0!:!V)O;W1?
M1'EN84QO861E<@```&L'<&5R;%]A;&QO8P````!M!W!E<FQ?8V]N<W1R=6-T
M`````&X'<&5R;%]D97-T<G5C=`!P!W!E<FQ?9G)E90!Q!W!E<FQ?<&%R<V4`
M````<@=P97)L7W)U;@``BP)'971-;V1U;&5(86YD;&5!````(`$``"`!```@
M`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!
M```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$`
M`"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``
M(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@
M`0``(`$``"`!```@`0!C>6=W:6XQ+F1L;``4(`$`%"`!`!0@`0`4(`$`%"`!
M`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`
M%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`8WEG<&5R;#5?
M,S`N9&QL`"@@`0!+15).14PS,BYD;&P`````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````0``P```#```(`.````P```@!````#P``"`&````"`!`(``````
M``````````````0``0```&```(`"````>```@`,```"0``"`!````*@``(``
M``````````````````$`"00``%`!``````````````````````$`"00``&`!
M``````````````````````$`"00``'`!``````````````````````$`"00`
M`(`!```````````````````!````P`$`@-@``(````````````````````$`
M"00``)`!``````````````````````$``0````@!`(``````````````````
M``$`"00``*`!``````````````````````$``0```#@!`(``````````````
M``````$`"00``+`!``#0,0$`:`4`````````````.#<!`*@(````````````
M`.`_`0!H!0````````````!(10$`J`@`````````````\$T!`#X`````````
M`````#!.`0!8`@````````````"(4`$`V@4`````````````!@!7`$D`3@!%
M`%@`10```"@````0````(`````$`"`````````$```````````````$````!
M```&!@8`!`0$``("`@``````0D)"`)V=G0!N;FX`@8&!`+&QL0#___\`<W-S
M`)^?GP!86%@`LK*R`/W]_0!K:VL`_/S\`-?7UP`6%A8`VMK:`):6E@".CHX`
M1D9&`,O+RP`+"PL`H*"@`/[^_@#)R<D`P<'!``,#`P`1$1$`DY.3`+.SLP#N
M[NX`]/3T`./CXP#EY>4`HJ*B`+JZN@"OKZ\`!P<'`'%Q<0"UM;4`]?7U`(F)
MB0"*BHH`-#0T`.CHZ``U-34`6EI:`+>WMP#8V-@`T-#0`!T='0`,#`P`CX^/
M`+Z^O@!M;6T`"0D)`)N;FP`O+R\`S<W-`-G9V0`9&1D`34U-`$1$1`!04%``
M/#P\`&1D9`#"PL(`,C(R``@("``-#0T`#P\/````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````4%!04%!04%!
M04%!04%!`0$!-4`C!4$Q0T%!1$4N`0$!`0$!`04!/!<7`0$3`0$!`0$!`0$5
M`0$F%0$!#P$!`0$!`0$!$P$1`04U`28!`0$!`0$!`2(2$#$/!0$@`0$!`0$!
M`0$0"B`B%14N(2X!`0$!`1\@(1`B$",C)28G`0$!`240$!`0$!`0$!`710$!
M`4$0$!`0$!`0$!`0$!<!`0$/$!`5$!`0$!`0$!`!`0$!$A`0$Q\0$!`0$!`'
M`0$!`2`0$!`!#Q`0$!`1`0$!`0$'(!`*`0$%$!`0,0$!`0$!`0$!`0$!`4$%
M#P$!`0$!`0$!`0$!`0$!`0$!`0$!`8`!``#``0``X`,``/`#``#P`P``\`,`
M`.`#``#``0``@`$``(`!``"``P``@`,````'````!P``C@\``/\?```H````
M(````$`````!``@````````$```````````````!`````0````````$!`0`"
M`@(``P,#``0$!``%!04`!@8&``<'!P`("`@`"0D)``H*"@`+"PL`#`P,``T-
M#0`.#@X`#P\/`!`0$``1$1$`$A(2`!,3$P`4%!0`%145`!86%@`7%Q<`&!@8
M`!D9&0`:&AH`&QL;`!P<'``='1T`'AX>`!\?'P`@("``(2$A`"(B(@`C(R,`
M)"0D`"4E)0`F)B8`)R<G`"@H*``I*2D`*BHJ`"LK*P`L+"P`+2TM`"XN+@`O
M+R\`,#`P`#$Q,0`R,C(`,S,S`#0T-``U-34`-C8V`#<W-P`X.#@`.3DY`#HZ
M.@`[.SL`/#P\`#T]/0`^/CX`/S\_`$!`0`!!04$`0D)"`$-#0P!$1$0`145%
M`$9&1@!'1T<`2$A(`$E)20!*2DH`2TM+`$Q,3`!-34T`3DY.`$]/3P!04%``
M45%1`%)24@!34U,`5%14`%5550!65E8`5U=7`%A86`!965D`6EI:`%M;6P!<
M7%P`75U=`%Y>7@!?7U\`8&!@`&%A80!B8F(`8V-C`&1D9`!E964`9F9F`&=G
M9P!H:&@`:6EI`&IJ:@!K:VL`;&QL`&UM;0!N;FX`;V]O`'!P<`!Q<7$`<G)R
M`'-S<P!T='0`=75U`'9V=@!W=W<`>'AX`'EY>0!Z>GH`>WM[`'Q\?`!]?7T`
M?GY^`']_?P"`@(``@8&!`(*"@@"#@X,`A(2$`(6%A0"&AH8`AX>'`(B(B`")
MB8D`BHJ*`(N+BP",C(P`C8V-`(Z.C@"/CX\`D)"0`)&1D0"2DI(`DY.3`)24
ME`"5E94`EI:6`)>7EP"8F)@`F9F9`)J:F@";FYL`G)R<`)V=G0">GIX`GY^?
M`*"@H`"AH:$`HJ*B`*.CHP"DI*0`I:6E`*:FI@"GIZ<`J*BH`*FIJ0"JJJH`
MJZNK`*RLK`"MK:T`KJZN`*^OKP"PL+``L;&Q`+*RL@"SL[,`M+2T`+6UM0"V
MMK8`M[>W`+BXN`"YN;D`NKJZ`+N[NP"\O+P`O;V]`+Z^O@"_O[\`P,#``,'!
MP0#"PL(`P\/#`,3$Q`#%Q<4`QL;&`,?'QP#(R,@`R<G)`,K*R@#+R\L`S,S,
M`,W-S0#.SLX`S\_/`-#0T`#1T=$`TM+2`-/3TP#4U-0`U=75`-;6U@#7U]<`
MV-C8`-G9V0#:VMH`V]O;`-S<W`#=W=T`WM[>`-_?WP#@X.``X>'A`.+BX@#C
MX^,`Y.3D`.7EY0#FYN8`Y^?G`.CHZ`#IZ>D`ZNKJ`.OKZP#L[.P`[>WM`.[N
M[@#O[^\`\/#P`/'Q\0#R\O(`\_/S`/3T]`#U]?4`]O;V`/?W]P#X^/@`^?GY
M`/KZ^@#[^_L`_/S\`/W]_0#^_OX`____`#0T-#P\/#P\/#P\/#P\/#P\/#P\
M/#P\/"P,!`0$!`0$7+34U-34U-34U-34W-34U-34U-34U-34M#11!`0$!`1D
MS.ST]/3\]/ST]/3\]/ST_/3\_/ST].S$-%%1!`0$!&3$Y.ST]/3T]/3T]/3T
M]/3T]/3T]/3L[+PT45%1!`0$9,3D[.SLY.SL[.3DY.3L[.3LY.SL[.SDO#11
M45%1!`1DO.3L[*R,K-3$E)2<E(R,E)24G)R<[-R\-%%145%1!&2\W.3LY,11
ME*QDY*1,-"Q<Q,Q1M-3DW+PT45%145%19+S<Y.3LY.11A'3DC&RDG&R,O'2L
MU.3<O#0$!`0$!`1DO-SDY.SDY+Q1C-34C*S,=,S<I%'$Y-R\-`0$!`0$!&2\
MW.3DY.3DW(Q\G+1\A)QLS.2\=,3DW+0T!`0$!`0$7+S<W.3DY.34G*1<E(S,
M/)1DS+R,4=S4M#0$!`0$!`1DM-3<W-SDW(R,K"R\M,Q:Q&2<;)2DW-2T-`0$
M!`0$!%RTU-S<Y-S<=)343*3TC#S4=)1TG$S<S+0T!`0$!`0$9+34W-S<Y-QT
M4?1TA.S,9.1\;*2T;*34M#0$!`0$!`1DM,S<W,Q\C(2T_,S$_/2T_+QDS*R$
M4;RL-`0$!`0$!%RTS-3,C*3<W.S\_/S\_/3\[+3LK'R44:PT!`0$!`0$7*S,
MS&R<]/S\_/S\_/S\_/S\_/S$E(Q1K#0$!`0$!`1<K,2\?.3\_/S\_/S\_/S\
M_/S\_.S,K%&L-`0$!`0$!%RLM$S$_/S\[/3\_/S\_/S\_/S\]+Q1Q*PT!`0$
M!`0$7*RT3-3\_/2TW/S\_/S\_/S\_/S,4<S$I#0$!`0$!`1<K,2D4?S\_+2<
M]/S\_/S\_/S\])11S,2D+`0$!`0$!%RDI%&$[/S\S&2T[/S\_/S\_/S4;*S,
MO*0T!`0$!`0$7*1D=+ST_/S$482<Y/S\_/S\Y(Q1S,S$I#0$!`0$!`1<I&R,
MY/3\_)Q1S%&4[/S\_/R\=,S,Q+R<+`0$!`0$!%RDK&1<9*2D4:S,Q'24Y/3\
MY%&\S,S$O)PT!`0$!`0$7*2TO+R,45'$Q,S,K%%\?)1L?+S,Q,2\G"P$!`0$
M!`1<G+S$Q,3$Q,3$Q,3$Q%%145&\Q,3$Q+2<-`0$!`0$!%2<M,3$Q,3$Q,3$
MQ,3$Q,3$Q,3$Q,2\M)PL!`0$!`0$5)R\Q,2\O+R\O+R\O+R\O+R\O+R\O+RT
MG"P$!`0$!`1<G+3$O+2TM+2TM+2TM+2TM+2TM+2TM*R<+`0$!`0$!%2$E)R<
ME)24E)24E)24E)24E)24E)24E(0D!`0$!`0$1%1<7%Q<7%Q<7%Q<7%Q<7%Q<
M7%Q<7%Q<5#0$!`0$!`0````_````'P````\````'`````P````$`````````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\````_````/P```#\````_````/P```#\````_````/P```#\`
M```_````/R@````0````(`````$`"`````````$```````````````$````!
M```&!@8`!`0$``("`@``````0D)"`)V=G0!N;FX`@8&!`+&QL0#___\`<W-S
M`)^?GP!86%@`LK*R`/W]_0!K:VL`_/S\`-?7UP`6%A8`VMK:`):6E@".CHX`
M1D9&`,O+RP`+"PL`H*"@`/[^_@#)R<D`P<'!``,#`P`1$1$`DY.3`+.SLP#N
M[NX`]/3T`./CXP#EY>4`HJ*B`+JZN@"OKZ\`!P<'`'%Q<0"UM;4`]?7U`(F)
MB0"*BHH`-#0T`.CHZ``U-34`6EI:`+>WMP#8V-@`T-#0`!T='0`,#`P`CX^/
M`+Z^O@!M;6T`"0D)`)N;FP`O+R\`S<W-`-G9V0`9&1D`34U-`$1$1`!04%``
M/#P\`&1D9`#"PL(`,C(R``@("``-#0T`#P\/````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````4%!04%!04%!
M04%!04%!`0$!-4`C!4$Q0T%!1$4N`0$!`0$!`04!/!<7`0$3`0$!`0$!`0$5
M`0$F%0$!#P$!`0$!`0$!$P$1`04U`28!`0$!`0$!`2(2$#$/!0$@`0$!`0$!
M`0$0"B`B%14N(2X!`0$!`1\@(1`B$",C)28G`0$!`240$!`0$!`0$!`710$!
M`4$0$!`0$!`0$!`0$!<!`0$/$!`5$!`0$!`0$!`!`0$!$A`0$Q\0$!`0$!`'
M`0$!`2`0$!`!#Q`0$!`1`0$!`0$'(!`*`0$%$!`0,0$!`0$!`0$!`0$!`4$%
M#P$!`0$!`0$!`0$!`0$!`0$!`0$!`8`!``#``0``X`,``/`#``#P`P``\`,`
M`.`#``#``0``@`$``(`!``"``P``@`,````'````!P``C@\``/\?```H````
M(````$`````!``@````````$```````````````!`````0````````$!`0`"
M`@(``P,#``0$!``%!04`!@8&``<'!P`("`@`"0D)``H*"@`+"PL`#`P,``T-
M#0`.#@X`#P\/`!`0$``1$1$`$A(2`!,3$P`4%!0`%145`!86%@`7%Q<`&!@8
M`!D9&0`:&AH`&QL;`!P<'``='1T`'AX>`!\?'P`@("``(2$A`"(B(@`C(R,`
M)"0D`"4E)0`F)B8`)R<G`"@H*``I*2D`*BHJ`"LK*P`L+"P`+2TM`"XN+@`O
M+R\`,#`P`#$Q,0`R,C(`,S,S`#0T-``U-34`-C8V`#<W-P`X.#@`.3DY`#HZ
M.@`[.SL`/#P\`#T]/0`^/CX`/S\_`$!`0`!!04$`0D)"`$-#0P!$1$0`145%
M`$9&1@!'1T<`2$A(`$E)20!*2DH`2TM+`$Q,3`!-34T`3DY.`$]/3P!04%``
M45%1`%)24@!34U,`5%14`%5550!65E8`5U=7`%A86`!965D`6EI:`%M;6P!<
M7%P`75U=`%Y>7@!?7U\`8&!@`&%A80!B8F(`8V-C`&1D9`!E964`9F9F`&=G
M9P!H:&@`:6EI`&IJ:@!K:VL`;&QL`&UM;0!N;FX`;V]O`'!P<`!Q<7$`<G)R
M`'-S<P!T='0`=75U`'9V=@!W=W<`>'AX`'EY>0!Z>GH`>WM[`'Q\?`!]?7T`
M?GY^`']_?P"`@(``@8&!`(*"@@"#@X,`A(2$`(6%A0"&AH8`AX>'`(B(B`")
MB8D`BHJ*`(N+BP",C(P`C8V-`(Z.C@"/CX\`D)"0`)&1D0"2DI(`DY.3`)24
ME`"5E94`EI:6`)>7EP"8F)@`F9F9`)J:F@";FYL`G)R<`)V=G0">GIX`GY^?
M`*"@H`"AH:$`HJ*B`*.CHP"DI*0`I:6E`*:FI@"GIZ<`J*BH`*FIJ0"JJJH`
MJZNK`*RLK`"MK:T`KJZN`*^OKP"PL+``L;&Q`+*RL@"SL[,`M+2T`+6UM0"V
MMK8`M[>W`+BXN`"YN;D`NKJZ`+N[NP"\O+P`O;V]`+Z^O@"_O[\`P,#``,'!
MP0#"PL(`P\/#`,3$Q`#%Q<4`QL;&`,?'QP#(R,@`R<G)`,K*R@#+R\L`S,S,
M`,W-S0#.SLX`S\_/`-#0T`#1T=$`TM+2`-/3TP#4U-0`U=75`-;6U@#7U]<`
MV-C8`-G9V0#:VMH`V]O;`-S<W`#=W=T`WM[>`-_?WP#@X.``X>'A`.+BX@#C
MX^,`Y.3D`.7EY0#FYN8`Y^?G`.CHZ`#IZ>D`ZNKJ`.OKZP#L[.P`[>WM`.[N
M[@#O[^\`\/#P`/'Q\0#R\O(`\_/S`/3T]`#U]?4`]O;V`/?W]P#X^/@`^?GY
M`/KZ^@#[^_L`_/S\`/W]_0#^_OX`____`#0T-#P\/#P\/#P\/#P\/#P\/#P\
M/#P\/"P,!`0$!`0$7+34U-34U-34U-34W-34U-34U-34U-34M#11!`0$!`1D
MS.ST]/3\]/ST]/3\]/ST_/3\_/ST].S$-%%1!`0$!&3$Y.ST]/3T]/3T]/3T
M]/3T]/3T]/3L[+PT45%1!`0$9,3D[.SLY.SL[.3DY.3L[.3LY.SL[.SDO#11
M45%1!`1DO.3L[*R,K-3$E)2<E(R,E)24G)R<[-R\-%%145%1!&2\W.3LY,11
ME*QDY*1,-"Q<Q,Q1M-3DW+PT45%145%19+S<Y.3LY.11A'3DC&RDG&R,O'2L
MU.3<O#0$!`0$!`1DO-SDY.SDY+Q1C-34C*S,=,S<I%'$Y-R\-`0$!`0$!&2\
MW.3DY.3DW(Q\G+1\A)QLS.2\=,3DW+0T!`0$!`0$7+S<W.3DY.34G*1<E(S,
M/)1DS+R,4=S4M#0$!`0$!`1DM-3<W-SDW(R,K"R\M,Q:Q&2<;)2DW-2T-`0$
M!`0$!%RTU-S<Y-S<=)343*3TC#S4=)1TG$S<S+0T!`0$!`0$9+34W-S<Y-QT
M4?1TA.S,9.1\;*2T;*34M#0$!`0$!`1DM,S<W,Q\C(2T_,S$_/2T_+QDS*R$
M4;RL-`0$!`0$!%RTS-3,C*3<W.S\_/S\_/3\[+3LK'R44:PT!`0$!`0$7*S,
MS&R<]/S\_/S\_/S\_/S\_/S$E(Q1K#0$!`0$!`1<K,2\?.3\_/S\_/S\_/S\
M_/S\_.S,K%&L-`0$!`0$!%RLM$S$_/S\[/3\_/S\_/S\_/S\]+Q1Q*PT!`0$
M!`0$7*RT3-3\_/2TW/S\_/S\_/S\_/S,4<S$I#0$!`0$!`1<K,2D4?S\_+2<
M]/S\_/S\_/S\])11S,2D+`0$!`0$!%RDI%&$[/S\S&2T[/S\_/S\_/S4;*S,
MO*0T!`0$!`0$7*1D=+ST_/S$482<Y/S\_/S\Y(Q1S,S$I#0$!`0$!`1<I&R,
MY/3\_)Q1S%&4[/S\_/R\=,S,Q+R<+`0$!`0$!%RDK&1<9*2D4:S,Q'24Y/3\
MY%&\S,S$O)PT!`0$!`0$7*2TO+R,45'$Q,S,K%%\?)1L?+S,Q,2\G"P$!`0$
M!`1<G+S$Q,3$Q,3$Q,3$Q%%145&\Q,3$Q+2<-`0$!`0$!%2<M,3$Q,3$Q,3$
MQ,3$Q,3$Q,3$Q,2\M)PL!`0$!`0$5)R\Q,2\O+R\O+R\O+R\O+R\O+R\O+RT
MG"P$!`0$!`1<G+3$O+2TM+2TM+2TM+2TM+2TM+2TM*R<+`0$!`0$!%2$E)R<
ME)24E)24E)24E)24E)24E)24E(0D!`0$!`0$1%1<7%Q<7%Q<7%Q<7%Q<7%Q<
M7%Q<7%Q<5#0$!`0$!`0````_````'P````\````'`````P````$`````````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\````_````/P```#\````_````/P```#\````_````/P```#\`
M```_````/P```0`$`!`0```!``@`:`4```$`("````$`"`"H"````@`0$```
M`0`(`&@%```#`"`@```!``@`J`@```0```!8`C0```!6`%,`7P!6`$4`4@!3
M`$D`3P!.`%\`20!.`$8`3P``````O03O_@```0``````````````````````
M/P`````````$``0``0```````````````````+@!```!`%,`=`!R`&D`;@!G
M`$8`:0!L`&4`20!N`&8`;P```)0!```!`#``,``P`#``,``T`$(`,````"0`
M`@`!`$,`;P!M`'``80!N`'D`3@!A`&T`90``````(````"P``@`!`$8`:0!L
M`&4`1`!E`',`8P!R`&D`<`!T`&D`;P!N```````@````,``(``$`1@!I`&P`
M90!6`&4`<@!S`&D`;P!N```````P`"X`,``N`#``+@`P````)``"``$`20!N
M`'0`90!R`&X`80!L`$X`80!M`&4````@````*``"``$`3`!E`&<`80!L`$,`
M;P!P`'D`<@!I`&<`:`!T````(````"P``@`!`$P`90!G`&$`;`!4`'(`80!D
M`&4`;0!A`'(`:P!S```````@````+``"``$`3P!R`&D`9P!I`&X`80!L`$8`
M:0!L`&4`;@!A`&T`90```"`````D``(``0!0`'(`;P!D`'4`8P!T`$X`80!M
M`&4``````"`````T``@``0!0`'(`;P!D`'4`8P!T`%8`90!R`',`:0!O`&X`
M```P`"X`,``N`#``+@`P````1`````$`5@!A`'(`1@!I`&P`90!)`&X`9@!O
M```````D``0```!4`'(`80!N`',`;`!A`'0`:0!O`&X`````````L`0\/WAM
M;"!V97)S:6]N/2(Q+C`B(&5N8V]D:6YG/2)55$8M."(@<W1A;F1A;&]N93TB
M>65S(C\^#0H\87-S96UB;'D@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O<V]F
M="UC;VTZ87-M+G8Q(B!M86YI9F5S=%9E<G-I;VX](C$N,"(^#0H@("`@/&%S
M<V5M8FQY261E;G1I='D@=F5R<VEO;CTB,"XP+C`N,"(@='EP93TB=VEN,S(B
M(&YA;64](E!!4BU086-K97(N07!P;&EC871I;VXB("\^#0H@("`@/&1E<V-R
M:7!T:6]N/E!!4BU086-K97(@07!P;&EC871I;VX\+V1E<V-R:7!T:6]N/@T*
M("`@(#QD97!E;F1E;F-Y/@T*("`@("`@("`\9&5P96YD96YT07-S96UB;'D^
M#0H@("`@("`@("`@("`\87-S96UB;'E)9&5N=&ET>2!T>7!E/2)W:6XS,B(@
M;F%M93TB36EC<F]S;V9T+E=I;F1O=W,N0V]M;6]N+4-O;G1R;VQS(B!V97)S
M:6]N/2(V+C`N,"XP(B!P=6)L:6-+97E4;VME;CTB-C4Y-6(V-#$T-&-C9C%D
M9B(@;&%N9W5A9V4](BHB('!R;V-E<W-O<D%R8VAI=&5C='5R93TB*B(@+SX-
M"B`@("`@("`@/"]D97!E;F1E;G1!<W-E;6)L>3X-"B`@("`\+V1E<&5N9&5N
M8WD^#0H@("`@/'1R=7-T26YF;R!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S
M;V9T+6-O;3IA<VTN=C,B/@T*("`@("`@("`\<V5C=7)I='D^#0H@("`@("`@
M("`@("`\<F5Q=65S=&5D4')I=FEL96=E<SX-"B`@("`@("`@("`@("`@("`\
M<F5Q=65S=&5D17AE8W5T:6]N3&5V96P@;&5V96P](F%S26YV;VME<B(@=6E!
M8V-E<W,](F9A;'-E(B`O/@T*("`@("`@("`@("`@/"]R97%U97-T9610<FEV
M:6QE9V5S/@T*("`@("`@("`\+W-E8W5R:71Y/@T*("`@(#PO=')U<W1);F9O
M/@T*("`@(#QC;VUP871I8FEL:71Y('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R
M;W-O9G0M8V]M.F-O;7!A=&EB:6QI='DN=C$B/@T*("`@("`@("`\87!P;&EC
M871I;VX^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@."XQ("TM/@T*("`@
M("`@("`@("`@/'-U<'!O<G1E9$]3($ED/2)[,68V-S9C-S8M.#!E,2TT,C,Y
M+3DU8F(M.#-D,&8V9#!D83<X?2(O/@T*("`@("`@("`@("`@/"$M+2!7:6YD
M;W=S(%9I<W1A("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED/2)[
M93(P,3$T-3<M,34T-BTT,V,U+6$U9F4M,#`X9&5E93-D,V8P?2(O/@T*("`@
M("`@("`@("`@/"$M+2!7:6YD;W=S(#<@+2T^#0H@("`@("`@("`@("`\<W5P
M<&]R=&5D3U,@260](GLS-3$S.&(Y82TU9#DV+31F8F0M.&4R9"UA,C0T,#(R
M-68Y,V%](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@."`M+3X-"B`@
M("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>S1A,F8R.&4S+34S8CDM-#0T
M,2UB83EC+60V.60T831A-F4S.'TB+SX-"B`@("`@("`@("`@(#PA+2T@5VEN
M9&]W<R`Q,"`M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>SAE
M,&8W83$R+6)F8C,M-&9E."UB.6$U+30X9F0U,&$Q-6$Y87TB+SX-"B`@("`@
M("`@/"]A<'!L:6-A=&EO;CX-"B`@("`\+V-O;7!A=&EB:6QI='D^#0H\+V%S
M<V5M8FQY/@T*````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````!````,````Z*\```!@```,````&*````"@```8````D*V@K;"M
MP*W0K>"M\*T`````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
8````````````````````````````````
