package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.26.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.26.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`-*1[ED``````````/``+@(+`@(=`$(```#(````$@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!P`0``!```:/4```,``(``
M`"`````````0```````````0````````$``````````````0`````!`!`($!
M````(`$`2`L````P`0"P*````-```,@!``````````````!@`0`P`````,``
M`!P`````````````````````````````````````````````````````````
M["(!`(@"```````````````````````````````````N=&5X=````/A!````
M$````$(````$``````````````````!@`%!@+F1A=&$```!H`````&`````"
M````1@``````````````````0`!@P"YR9&%T80``'$(```!P````1````$@`
M`````````````````$``4$`N8G5I;&1I9#4`````P`````(```",````````
M``````````!``#!`+G!D871A``#(`0```-`````"````C@``````````````
M````0``P0"YX9&%T80``K`$```#@`````@```)```````````````````$``
M,$`N8G-S`````.`1````\`````````````````````````````"``&#`+F5D
M871A``"!`0```!`!```"````D@``````````````````0``P0"YI9&%T80``
M2`L````@`0``#````)0``````````````````$``,,`N<G-R8P```+`H````
M,`$`L"@```"@``````````````````!``##`+G)E;&]C```P`````&`!```"
M````R@``````````````````0``P0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0VE/P``
MZ,`Y``!%,<`QTC')Z-0Y``!%,<`QTC')Z-@Y``!%,<`QTC')Z-PY``!%,<`Q
MTC')2(/$*.G<.0``D)"0D)"0D)"0D)"054B)Y5W#9BX/'X0``````%5(B>5(
M@^P@2(,]\$\```!T,$B-#8=?``#_%=T4`0!(A<!T/$B-%8-?``!(B<'_%=`4
M`0!(A<!T"4B-#<!/``#_T$R-!5=/``!(C0T@````,=)(@\0@7>G$-@``#Q]`
M`$B-!8G____KT`\?@`````!52(GE7<.0D)"0D)"0D)"005=!5D%505155U93
M2('LN`(``$B++=:>``!%,=),C5DX3(U$)"!(BT4`2(F$)*@"```QP$.+!!-/
MBPP3#\A,B<J)P$C!ZB!+B0103(G(2<'I.$C!Z"@E`/\``$P)R$F)T4C!X@A)
MP>$8@>(``/\`18G)3`G(2`G02XE$4`A)@\((28/Z0'6M38V(``(``&:028M`
M:$DS0$!)@\`(23-`"$DS0/A(C10`2,'H'T@)T(/@_TF)0'A-.<AUU4B+&4R+
M>2!(BW$02(MY&$R+60A,BTPD*$B)V$B)VDF)W4C!ZAM(P>`%2<'M`D@)T$B+
M5"0@38G8B<!)P>`>3HV,#YEY@EI)C907F7F"6D@!T$B)\D@Q^DPAVD@Q^D@!
MPDR)V$C!Z`)!B=1)"<!(B?!-B>)%B<!)P>H;3#'`2"'82#'P3`'(38GA2<'A
M!4T)T46)R4P!R$F)V4G!X1Y!B<)-">E-B>5-B=9%B<E)]]5)P>X;32'%3"'*
M3`GJ3(ML)#!.C:PNF7F"6DP!ZDV)U4G!Y05-"?5%B>U,`>I-B>5)P>P"2<'E
M'D&)UDT)[$6)Y4V)U$GWU$PAZ$TAS$P)X$R+9"0X3XV$()EY@EI-B?1)P>P;
M3`'`38GP2<'@!4T)X$6)P$P!P$V)T$G!Z@))P>`>08G$30G"18G038GR2??2
M3"'"32'J3`G23(M4)$!/C8P1F7F"6DV)XDG!ZAM,`<I-B>%)P>$%30G118G)
M3`'*38GQ2<'N`DG!X1Y!B=)-"<Y%B?%-B>9)]]9,(<A-(<9,"?!,BW0D2$^-
MK#69>8):38G62<'N&TP!Z$V)U4G!Y05-"?5%B>U,`>A-B>5)P>4>2<'L`D&)
MQDT)[$6)Y4V)U$GWU$PAZDTAS$P)XDR+9"103XV$()EY@EI-B?1)P>P;3`'"
M38GP2<'@!4T)X$6)P$P!PDV)T$G!Z@))P>`>08G430G"18G038GR2??23"'`
M32'J3`G03(M4)%A/C8P1F7F"6DV)XDG!ZAM,`<A-B>%)P>$%30G118G)3`'(
M38GQ2<'N`DG!X1Y!B<)-"<Y%B?%-B>9)]]9,(<I-(<9,"?),BW0D8$^-K#69
M>8):38G62<'N&TP!ZDV)U4G!Y05-"?5%B>U,`>I-B>5)P>4>2<'L`D&)UDT)
M[$6)Y4V)U$GWU$PAZ$TAS$P)X$R+9"1H3XV$()EY@EI-B?1)P>P;3`'`38GP
M2<'@!4T)X$6)P$P!P$V)T$G!Z@))P>`>08G$30G"18G038GR2??23"'"32'J
M3`G23(M4)'!/C8P1F7F"6DV)XDG!ZAM,`<I-B>%)P>$%30G118G)3`'*38GQ
M2<'N`DG!X1Y!B=)-"<Y%B?%-B>9)]]9,(<A-(<9,"?!,BW0D>$^-K#69>8):
M38G62<'N&TP!Z$V)U4G!Y05-"?5%B>U,`>A-B>5)P>P"2<'E'D&)QDT)[$6)
MY4V)U$GWU$PAZDTAS$P)XDR+I"2`````3XV$()EY@EI-B?1)P>P;3`'"38GP
M2<'@!4T)X$6)P$P!PDV)T$G!Z@))P>`>08G430G"18G038GR2??23"'`32'J
M3`G03(N4)(@```!/C8P1F7F"6DV)XDG!ZAM,`<A-B>%)P>$%30G118G)3`'(
M38GQ2<'N`DG!X1Y!B<)-"<Y%B?%-B>9)]]9,(<I-(<9,"?),B[0DD````$^-
MK#69>8):38G62<'N&TP!ZDV)U4G!Y05-"?5%B>U,`>I-B>5)P>P"2<'E'D&)
MUDT)[$6)Y4V)U$GWU$PAZ$TAS$P)X$R+I"28````3XV$()EY@EI-B?1)P>P;
M3`'`38GP2<'@!4T)X$6)P$P!P$V)T$G!Z@))P>`>08G$30G"38GG18G038GR
M2??23"'"32'J3`G23(N4)*````!/C8P1F7F"6DV)XDG!ZAM,`<I-B>%)P>$%
M30G118G)3`'*38GQ2<'N`DG!X1Y!B=)-"<Y%B?%-B>9)]]9,(<A-(<9,"?!,
MB[0DJ````$^-K#69>8):38G63`'H38G52<'E!4G![AM)P><>2<'L`DT)]4T)
M_$6)[46)YTV)U$P!Z$GWU$PA^D&)Q4TAS$P)XDR+I"2P````3XV$()EY@EI-
MB>Q)P>P;3`'"38GH2<'@!4T)X$6)P$P!PDV)T$G!Z@))P>`>08G630G038GJ
M18G$2??232'Z3"'@3`G03(N4)+@```!/C8P1F7F"6DV)\DG!ZAM)`<%,B?!(
MP>`%3`G0B<!)`<%,B>A)P>T"2,'@'D6)RDD)Q4R)P$V)T$PQZ#'"2(N$),``
M``!)C80'H>O9;D@!T$R)TDC!X@5)P>@;3`G"38GP2<'N`HG22<'@'DT)\$@!
MT$R)ZDPQPD&)QT$QT4B+E"3(````38VD%*'KV6Y,B?I(P>(%30',38GY2<'I
M&TP)RHG238T,%$V)U$G!Z@))P>0>30G418GJ3`.4)-````!-B<5$B<I-,>5$
M,>A)B=5-C90"H>O9;DB)T$G![1M(P>`%3`GHB<!-C2P"38GZ1(G`2<'O`D@#
MA"38````2<'B'DT)^DV)X$6)[DTQT$4QP4V)\$Z-C`BAZ]EN3(GP2<'H&TC!
MX`5,"<")P$V-!`%)B=%(P>H"2<'A'DD)T42)XD@#E"3@````38G41(G`18G2
M33',3`.4).@```!%,>5)B<1.C:PJH>O9;DB)PDG![!M(P>(%3`GBB=)-C605
M`$R)\DG![@)(P>(>3`GR38G.18GE23'613'P3XVT`J'KV6Y-B>A-B>I)P>H;
M2<'@!4T)T$6)P$^-%`9)B<!(P>@"2<'@'DD)P$2)R$@#A"3P````28G118G6
MB=)-,<%(`Y0D^````$4QS$V)\4Z-I""AZ]EN3(GP2<'I&TC!X`5,"<B)P$V-
M#`1,B>A)P>T"2,'@'DP)Z$V)Q46)S$DQQ46)P$4QZDV)Y4Z-E!*AZ]EN3(GB
M2,'B!4G![1M,`X0D``$``$P)ZHG23`'238GR2<'N`DG!XAY!B=5-"?))B<:)
MP$TQUD@#A"0(`0``13'Q38GN3XV,"*'KV6Y-B>A)P>X;2<'@!4T)\$6)P$T!
MR$V)X4G![`))P>$>18G&30GA38G418G233',3`.4)!`!``!$,>)-B?1(C900
MH>O9;DR)\$G![!M(P>`%3`G@B<!(`=!,B>I)P>T"2,'B'D&)Q$P)ZDV)S46)
MR4DQU44QZ$^-K`*AZ]EN38G@38GB2<'J&TG!X`5-"=!%B<!/C50%`$V)\$G!
MX!Y,`XPD&`$``$G![@)-"?!)B=9%B=5-,<:)TD@#E"0@`0``1#'P38VT`:'K
MV6Y,B>A-B>E)P>D;2,'@!4P)R(G`38T,!DR)X$G![`)(P>`>3`G@38G$18G.
M23'$18G`3`.$)"@!``!%,>)-B?1.C902H>O9;DR)\DG![!M(P>(%3`GBB=),
M`=)-B>I)P>T"2<'B'D&)U$T)ZDF)Q8G`33'513'I38GE3XV,"*'KV6Y-B>!)
MP>T;2<'@!4T)Z$6)P$T!R$V)\4G![@))P>$>18G%30GQ2`.$)#`!``!-B=9-
M,<Y%B=),`Y0D.`$``$0Q\DV)[DG![AM(C900H>O9;DR)Z$C!X`5,"?")P$@!
MT$R)XDG![`)(P>(>08G'3`GB38G,18G)23'43`.,)$`!``!%,>!/C:0"H>O9
M;DV)^$V)^DG!ZAM)P>`%30G018G`3XT4!$V)Z$G![0))P>`>30GH28G518G4
M33'%B=)(`Y0D2`$``$0QZ$V)Y4V-C`&AZ]EN3(G@2<'M&TC!X`5,">B)P$V-
M+`%,B?A)P>\"2,'@'DV)P46)P$P)^$6)[DDQP44QRDZ-C!*AZ]EN3(GR38GR
M2,'B!4G!ZAM,`X0D4`$``$P)THG23`'*38GA2<'L`DG!X1Y!B=)-">%)B<2)
MP$TQS$@#A"18`0``13'E38G43XVL**'KV6Y-B=!)P>P;2<'@!4T)X$6)P$^-
M9`4`38GU2<'N`DG!Y1Y-"?5-B<Y%B>!-,>Y%B<E$,?)-B<9(C900H>O9;DR)
MP$G![AM(P>`%3`GPB<!,C30"3(G22<'J`DC!XAZXW+P;CTP)TDV)ZD6)]TD)
MTDTAU$V)ZD6)[4DATD4)U$F)PDP#E"1@`0``30'*38GY30'B38G\2<'A!4G!
M[!M-">%%B<E-`<I-B<%)P>@"2<'A'D6)U$T)P4F)T$T)R$TAQDF)T(G232'(
M10G&28G`3`.$)&@!``!-`<5-B>!-`>Y-B>5)P>`%2<'M&TT)Z$6)P$T!QDV)
M^$G![P))P>`>18GU30GX38G/30G'32'Z38G/18G)32''10GZ28G'3`.\)'`!
M``!,`?I-B>])`=),B>I)P>\;2,'B!4P)^HG220'23(GB2,'B'DG![`)%B==)
M"=1,B<),">))(=9,B<)%B<!,(>)!"=9(B<)(`Y0D>`$``$D!T4R)^DT!SDV)
M^4C!X@5)P>D;3`G*B=))`=9,B>I)P>T"2,'B'D6)\4P)ZDV)Y4D)U4TAZDV)
MY46)Y$DAU44)ZDF)Q4P#K"2``0``30'H38G-30'"38G(2<'M&TG!X`5-">A%
MB<5-`=5-B?I)P>\"2<'B'D6)Z$T)^DF)UTT)UTTA_DF)UXG232'710G^28G'
M3`.\)(@!``!-`?Q-B<=-`>9-B<1)P>\;2<'D!4T)_$6)Y$T!YDV)S$G!Z0))
MP>0>18GW30GA38G430G,32'E38G418G232',10GE28G$3`.D))`!``!,`>)-
MB?Q)`=5,B?I)P>P;2,'B!4P)XHG220'53(G"2<'H`DC!XAY%B>Q)"=!,B<I,
M"<))(=9,B<I%B<E,(<)!"=9(B<)(`Y0DF`$``$D!TDR)XDT!UDV)XDC!X@5)
MP>H;3`G2B=))`=9,B?I)P>\"2,'B'D6)\DP)^DV)QTD)UTTA_4V)QT6)P$DA
MUT4)_4F)QTP#O"2@`0``30'Y38G730'-38G12<'O&TG!X05-"?E%B<E-`<U-
MB>%)P>P"2<'A'D6)[TT)S$F)T4T)X4TASDF)T8G232'A10G.28G!3`.,)*@!
M``!-`<%-B?A-`?%-B?Y)P>`%2<'N&TT)\$6)P$T!P4V)T$G!Z@))P>`>18G.
M30G"38G@30G032'%38G@18GD32'010G%28G`3`.$)+`!``!,`<)-B?!)`=5,
MB?))P>@;2,'B!4P)PDV)^$G![P))P>`>B=)-"?A-B==)`=5-"<=$B>I-(?E-
MB==%B=)-(<=%"?E)B<=,`[PDN`$``$T!YTF)U$T!STF)T4G![!M)P>$%30GA
M18G)30'/38GQ2<'N`DG!X1Y%B?Q-"?%-B<9-"<Y-(?5-B<9%B<!-(<Y%"?5)
MB<9,`[0DP`$``$T!\DV)YDT!U4V)XDG![AM)P>(%30GR18G230'528G22,'J
M`DG!XAY%B>Y,"=)-B<I)"=)-(==-B<I%B<E)(=)%"==)B<),`Y0DR`$``$T!
MPDV)\$T!^DV)]TG!X`5)P>\;30GX18G`30'"38G@2<'L`DG!X!Y%B==-"<1)
MB=!-">!-(<5)B=")TDTAX$4)Q4F)P$P#A"30`0``30'!38GX30'-38GY2<'@
M!4G!Z1M-"<A%B<!-`<5-B?!)P>X"2<'@'D6)Z4T)\$V)YDT)QDTA\DV)YD6)
MY$TAQD4)\DF)QDP#M"38`0``3`'R38G.20'23(G*2<'N&TC!X@5,"?)!B=9-
M`=9-B?I)P>\"2<'B'D2)\DT)^DV)QTT)UTTA_4V)QT6)P$TAUT4)_4F)QTP#
MO"3@`0``30'\28G730'E28G42<'D!4G![QM-"?Q%B>1-`>5-B<Q)P>D"2<'D
M'D6)[TT)X4V)U$T)S$TAYDV)U$6)TDTAS$4)YDF)Q$P#I"3H`0``30'$38GX
M30'T38G^2<'@!4G![AM-"?!%B<!-`<1)B=!(P>H"2<'@'D6)YDP)PDV)R$D)
MT$TAQ4V)R$6)R4DAT$4)Q4F)P$P#A"3P`0``30'"38GP30'J38GU2<'@!4G!
M[1M-">A%B<!-`<)-B?A)P>`>2<'O`D@#A"3X`0``30GX28G718G530G'32'\
M28G7B=)-(<=)`<%,B>A%"?Q(P>`%30',38GI2<'I&TP)R(G`38T,!$R)\$G!
M[@)(P>`>20G&N-;!8LI%B<Q)B<=,`[PD``(``$P!^DV)QT6)P$TQ]T4Q^DV)
MYTD!TDR)XDG![QM(P>(%3`GZB=))`=),B>I)P>T"2,'B'D6)UTD)U4B)PD@#
ME"0(`@``3`'"38GP18GV33'H13'!38GX20'13(GZ2,'B!4G!Z!M,"<)-B>!)
MP>P"2<'@'HG230G@28G$3`.D)!`"``!)`=%$B<I-`?1-B>Y%B>U-,<9%,?))
MB=9-`>))B=1)P>X;2<'D!4T)]$V)_DG![P))P>8>18GD30GW28G&3`.T)!@"
M``!-`>)%B=1-`?5-B<9%B<!-,?Y%,?%-B>9-`<U-B>%)P>X;2<'A!4T)\46)
MR4T!S4F)T4C!Z@))P>$>18GN3`G*28G!3`.,)"`"``!-`<A-B?E%B?]),=%%
M,<I-B?%-`<)-B?!)P>D;2<'@!4T)R$V)X4G![`))P>$>18G`30G,28G!3`.,
M)"@"``!-`<)%B=!-`?E)B=>)TDTQYT4Q_4V)QTT!S4V)P4G![QM)P>$%30GY
M18G)30'-38GQ2<'N`DG!X1Y%B>]-"?%)B<9,`[0D,`(``$P!\DV)YD6)Y$TQ
MSD4Q\DV)_DD!TDR)^DG![AM(P>(%3`GRB=))`=),B<)(P>(>2<'H`D6)UDP)
MPDF)P$P#A"0X`@``30'@38G,18G)23'413'L38GU30'$38GP2<'M&TG!X`5-
M">A%B<!-`<1-B?A)P>\"2<'@'D6)Y4T)^$F)QTP#O"1``@``30'Y28G7B=)-
M,<=%,?I-B>]-`<I-B>E)P>\;2<'A!4T)^46)R4T!RDV)\4G![@))P>$>18G7
M30G.28G!3`.,)$@"``!,`<I-B<%%B<!-,?%%,<Q-B?E)`=1,B?I(P>(%2<'I
M&TP)RHG220'43(GJ2<'M`DC!XAY%B>%,">I)B<5,`ZPD4`(``$T!Z$V)]46)
M]DDQU44QZDV)S4T!PDV)R$G![1M)P>`%30GH18G`30'"38GX2<'O`DG!X!Y%
MB=5-"<=)B<!,`X0D6`(``$T!\$F)UHG233'^13'T38GN30'$38GH2<'N&TG!
MX`5-"?!%B<!-`<1-B<A)P>D"2<'@'D6)YDT)P4F)P$P#A"1@`@``3`'"38GX
M18G_33'(13'"38GP20'23(GR2<'H&TC!X@5,"<)-B>A)P>T"2<'@'HG230G%
M28G`3`.$)&@"``!)`=)$B=)-`?A-B<]%B<E-,>]%,?Q)B==-`<1)B=!)P>\;
M2<'@!4T)^$6)P$T!Q$V)\$G![@))P>`>18GG30GP28G&3`.T)'`"``!-`?%-
MB>Y%B>U-,<9%,?)-B?Y-`=%-B?I)P>X;2<'B!4T)\DF)UD6)TDT!T4G!YAY(
MP>H"20G62(G"2`.4)'@"``!%B<I,`>I-B<5%B<!-,?5%,>5-B=1)`=5,B=))
MP>P;2,'B!4P)XHG220'53(GZ2<'O`DC!XAY%B>Q,"?I)B<=,`[PD@`(``$T!
M^$V)]T6)]DDQUT4Q^4V)YTT!P4V)X$G![QM)P>`%30GX18G`30'!38G02<'J
M`DG!X!Y%B<]-"<))B<!,`X0DB`(``$T!\$F)UHG233'613'U38G^30'H38G]
M2<'E!4G![AM-"?5-B>9)P>P"2<'F'D6)[4T)]$F)QDP#M"20`@``30'H2`.$
M))@"``!%B<5,`?)-B=9-,>9%,?%-B>Y,`<I-B>E)P>X;2<'A!4T)\40!RDV)
M^4G![P))P>$>30G/38GI2<'M`DG!X1Y-"<U%B=%)`<%,B>!,,?A!,<!+C00!
M28G028G12<'@!4G!Z1M$`=I-"<A(B5$(1`'``<-$`>Y$`V$@2(D91`'_2(EQ
M$$B+A"2H`@``2#-%`$B)>1A,B6$@=11(@<2X`@``6UY?74%<05U!7D%?P^BS
M'@``D&:0055!5%575E-(@^PH2(7)2(G-28G4#X2E````2(L%3H<``$R+*$V%
M[0^$D@````^V`83`=%\\/71;2(G/2(/'`0^V!X3`=`0\/77Q*>])BUT`2(7;
M="Y(8_],B>X/'T``28GX2(GJ2(G9Z)(@``"%P'4&@#P[/70H2(/&"$B+'DB%
MVW7<2(G82(/$*%M>7UU!7$%=PS'_Z[4/'X0``````$B-7#L!3"GN2,'^`TB)
MV$&)-"1(@\0H6UY?74%<05W#,=OKP&:09BX/'X0``````%575E-(@^PX2(LM
M<88``$B+/8J&``!(C5PD)$B)SDB+10!(B40D*#'`2(G:2(GQZ/S^__](A<!T
M1P\?@`````!(8U0D)$B+!TB-!-!(BU`(2(722(D0=-$/'X0``````$B#P`A(
MBU`(2(722(D0=?!(B=I(B?'HM?[__TB%P'7`2(M$)"A(,T4`=0E(@\0X6UY?
M7</H1QT``)!F#Q]$``!!5T%6055!5%575E-(@^Q(2(LUR84``$B)TTB)STB+
M!DB)1"0X,<`QP(`Z/0^4P$@!PTB)V>A&'P``2(U4)#1(B?E(B<7H1O[__TB%
MP$F)Q`^$^@```$B)P>@B'P``2#G%#X;*````3&-T)#1,BR6-A0``2<'F`P^V
M!X3`#X21`0``/#T/A(D!``!(B?CK!I"`^CUT"TB#P`$/MA"$TG7P2"GX2)A-
M`S0D2(U,*`+HO1T``$B%P$F)!@^$:`$``$AC5"0T28L$)$B-3P%(BQ30#[8'
MA,"(`G49ZQL/'T0``$B#P0$/MD'_2(/"`83`B`)T!#P]=>K&`CTQP`\?1```
M#[8,`XA,`@%(@\`!A,EU\#'`2(M\)#A(,SX/A1$!``!(@\1(6UY?74%<05U!
M7D%?PTB#PP$/MD/_28/$`83`08A$)/]UZ^O(#Q^$``````!,BR6IA```38L\
M)$F#/P`/A+4```!,B?@QTNL'#Q]``$2)ZDB#P`A$C6H!2(,X`'7O@\(#2&/2
M2,'B`XL%?]0``(7`=#M,B?GH<QT``$B%P$F)!"0/A'T```!-8]5)B<%.C335
M`````$O'1-$(`````$2);"0TZ:S^__\/'T0``$B)T<<%,]0```$```#HCAP`
M`$B%P'1`36/53(GZ2(G!3HTTU0````!,B50D*$V)\.B;'```3(M4)"A)B<%)
MB00DZZ4QP.F+_O__NA````!%,>WI9/___[C_____Z>'^___H"AL``)!F#Q^$
M``````!!5T%6055!5%575E-(@>S(````3(LEAH,``$B)RTF+!"1(B80DN```
M`#'`2(7)=`6`.0!U*TB+A"2X````23,$)`^%$`$``$B!Q,@```!;7E]=05Q!
M74%>05_##Q]$``#H*QP``$B%P$B)Q73(2(U$)#!,C2V'0P``3(TU@D,``$B)
M1"0H2(GIZ",<``!(A<`/A)P```!,C7@8N0(```!,B>],B?[SIG3<N0,```!,
MB?Y,B??SIG3-2(G9Z(`<``!,B?E(B<;H=1P``(U,!@)(8\GH61L``$B-%2=#
M``!-B?E)B=A(B<%(B<;H`1P``$B+5"0H2(GQZ`0<``"#^/](B?%T$(M$)$`E
M`/```#T`0```=#SH=QP``$B)\>BO&@``2(GIZ(<;``!(A<`/A63___](B>GH
M9AH``$B)V>B>&P``Z>[^__]F#Q^$``````#HJ_[__^O"Z*09``"0#Q\`5E-(
M@^PH2(L=`X(``$B)SDB+"^AX&0``2(V0``(``$B)\>B9&@``A<!T"3'`2(/$
M*%M>PTB+"^A4&0``BX`0`@``)0#P```]`$```'0:2(L+Z#H9``"+@!`"```E
M`/```#T`H```=<.Z`@```$B)\>BK&0``A<`/E,`/ML!(@\0H6U[#9I!F+@\?
MA```````0515N#`0``!75E/H$!D``$@IQ$B+/9:!``!!N/\/``!(C5PD($B)
MS4B-#>Q!``!)B=1(B=I(BP=(B80D*!```#'`Z(D:``"%P'YIC5#_2&/R@'PT
M(`!T6$AC\$F)\+HO````2(G9Z.49``!(A<!T1$B+!0F!``!(BPCH@1@``$F)
M\4F)V$B)ZDB)P>C8%P``2(N$)"@0``!(,P=U-DB!Q#`0``!;7E]=05S##Q]$
M``"%TG6G2(L%Q8```$B+".@]&```38G@2(GJ2(G!Z)\7``#KO>@X&```D`\?
M@`````!(@^PH2(L%E8```$B+".@-&```3(T-%D0``$R-!4\7``!(C14B00``
M2(G!2(/$*.E\%P``9I!F+@\?A```````4TB#[#!(BQU4@```2(L+Z,P7``!(
MB<'HI/___TB+"^B\%P``3(T-^T```$R-!;X2``!(C17N0```QT0D*`````!(
MB<%,B4PD(.@:%P``D$B#Q#!;PY!F+@\?A```````,<###Q\`9BX/'X0`````
M`$%705:X6!```$%505155U93Z'H7``!(*<1,BST`@```2(G+2(G62(T-GT``
M`$B-5"0\28L'2(F$)$@0```QP.B,^/__NB\```!(B=E(B<7H/!D``$B%P'0W
M2(T-=4```$B)VNC8^?__2(G82(N\)$@0``!),S\/A8(!``!(@<18$```6UY?
M74%<05U!7D%?PTB)V>@H&0``2(T50T```$B)\4&)QNA6&0``2(7`2(G&=*=!
M@\8!@#X`3(UD)$!,BRTK?P``#X3-````#Q]$``!(A>UT$TB)\DB)Z>C`&```
MA<`/A(T```!(B?'HT!@``(G"2)B`?`;_+W46#Q]$``"#Z@%(8\*`?`;_+\8$
M!@!T[T0!\H'Z_P\```^/.?___TR-#;T_``!(C16X/P``28GP3(GA2(E<)"#H
M,A@``$F+30#H*18``$B-D``"``!,B>'H*A@``(7`=1M)BTT`Z`T6``"+@!`"
M```E`/```#T`@```=%Q(C15F/P``,<GH?1@``$B%P$B)Q@^$RO[__X`^``^%
M./___TB%[70C2(T%W#X``+D"````2(GN2(G'\Z9TPDB)QKH!````Z5'___](
MC36Y/@``N@$```#I0/___[H!````3(GAZ"(6``"%P'632(T-[#X``$R)XNA/
M^/__3(GAZ+<7``#I;?[__^A]%0``D&:09BX/'X0``````$0/M@%(B<A%A,!T
M&4B-40%!@/@O1`^V`D@/1,)(@\(!183`=>OSPV8/'X0``````%932(/L*$B%
MR4B)RW14#[8Q0(3V=$SH91<``$R-1`/_3#G#<A3K*68/'X0``````$F#Z`%,
M.<-T'$&`."]T\>L&08`X+W0S28/H`4PYPW+Q3#G#=25`@/XO2(T-1#X``'4'
M2(T-UST``$B#Q"A;7NG\%@``08`X+W4)28/H`4PYPW+Q22G8,<!)C5`"2('Z
M`!```'<:2(T-=+T``$F#P`%(B=KH^!8``$B-!6&]``!(@\0H6U[#9BX/'X0`
M`````%575E-(@^QH2(LM$7T``$B-#=<]``!(C5PD+$B+10!(B40D6#'`2(L%
M>SX``$B)1"0P2(L%=SX``$B)1"0X2(L%<SX``$B)1"1`2(L%;SX``$B)1"1(
M#[<%:SX``&:)1"10Z#GV__](C0V(/0``Z"WV__](C0V$/0``Z"'V__](C0V!
M/0``Z!7V__](C0U\/0``Z`GV__](C0V`/0``Z/WU__](C0TB/0``Z/'U__](
MC0UT/0``Z.7U__](C0UR/0``Z-GU__](C0UP/0``Z,WU__](C0W[/```Z,'U
M__](C0UB/0``2(G:Z.+T__](A<!T#TB-#3H]``!(B<+H/O;__TB-#5`]``!(
MB=KHO_3__TB%P'0/2(T-3CT``$B)PN@;]O__2(T-2CT``$B)VNB<]/__2(7`
M=#=(C0V,/```2(G"Z/CU__](C153/0``2(T-N#P``.CE]?__2(M$)%A(,T4`
M=5=(@\1H6UY?7<.02(T-#ST``$B)VNA1]/__2(7`=!%(C0V?/```2(G"Z*WU
M___KLTB-="1!2(T]]CP``+D*````\Z9UGDB-5"1+2(T-=#P``.B%]?__ZXOH
MOA(``)`/'P!F+@\?A```````4TB#[#!(BQU$>P``2(L#2(E$)"@QP(L%="L`
M`(/X_W002(M,)"A(,PMU.TB#Q#!;PTB-5"0D2(T-'3P``.B^\___2(7`=!L/
MMA"`^C`/E<"$T@^5PB'0#[;`B04P*P``Z[\QP.OQZ$42``"0#Q]``$B#["CH
M-Q,``(G!Z"`3``!(A<!T"TB+`$B#Q"C##Q\`,<#K]&:09BX/'X0``````%57
M5E-(@^Q(2(LUH7H``$B)STB-5"0T2(T-/#P``$B+!DB)1"0X,<#H,_/__TB%
MP$B)PTB)^G1I@#@`=&1(B<'H.Q0``$B%P'062(M$)#A(,P9U6TB#Q$A;7E]=
MPP\?`$B)^>CH$P``2(G92(G%Z-T3``!(C4P%`NC#$@``2(T5\CH``$R-#><Z
M``!(B<5(B5PD($F)^$B)P>AB$P``2(GJ2(T-LCL``.@C]/__ZYOH7!$``)"0
M9BX/'X0``````$%705:XN($``$%505155U93Z$H1``!(*<1(BP70>0``28G,
M2(T-<CH``$C'1"1P`````$C'A"3(`````````$C'A"20`````````$B+,$B)
MM"2H@0``,?9(C05,.P``2(UT)%Q(B40D8$B-!>`Y``!(B?)(B40D:$B-!;PZ
M``!(B80DH````$B-!2([``!(B80DJ````$B-!1H[``!(B80DL````$B-!1,[
M``!(B80DN````$B-!0D[``!(B80DP````$B-!=4Z``!(B80D@````$B-!>\Z
M``!(B80DB````.BZ\?__2(7`=`F`.``/A4P$``#H%_[__TB%P$B)PP^$700`
M`$B)V>B#$@``2(U,``'H:1$``(`[`$F)Q70M2(TMJSH``$R)[P\?@`````!$
M#[8#2(GY2(GJ2(/#`4B#QP+H^1$``(`[`'7D2(V\)*````!(C0W@.0``ZP\/
M'P!(@\<(2(L/2(7)=#%(B?+H+/'__TB%P$B)PW3D@#@`=-](B<'H)_;__X7`
M=--(B=GHZQ$``$B%P$B)PW532(M<)&`Q_^L>9BX/'X0``````(/'`4ACQTB+
M7,1@2(7;#X1\`0``@#L`#X1S`0``2(G9Z-OU__^%P'372(G9Z)\1``!(A<!(
MB<-TQP\?@`````!(B=J+"DB#P@2-@?_^_O[WT2'()8"`@(!TZ8G!P>D0J8"`
M```/1,%(C4H"B<=(#T310`#'3(GI2(/:`T@IVD2+`4B#P01!C8#__O[^0??0
M1"'`)8"`@(!TY4&)P$'!Z!"I@(```$$/1,!,C4$"B<=)#T3(0`#'2(/9`TPI
MZ8V\"@0$``!(8_](B?GH`1```$B)Q4B-!58Y``!,C0TD.```2(T5/SD``$F)
MV$B)Z4R);"0H2(E$)"#HE!```+K``0``2(GIZ`<0``"#^/\/A(4```!(BQWW
M=@``2(L+Z&\.``!(C9```@``2(GIZ)`/``"%P$&)Q0^$F````$F+'"3H+`X`
M`$B+2!A(C15A.0``28GI28G8,=OH!`\``$B+-=UV``!(B=A(B[PDJ($``$@S
M/@^%&0D``$B!Q+B!``!;7E]=05Q!74%>05_##Q]$```QV^FI_O__Z&0.``"#
M.!$/A&W____H5@X``$F+'"2+,.B[#0``2(M(&$B-%7@X``!)B>E)B=B)="0@
M,=OHCPX``.N)2(L+Z+4-``"+@!`"```E`/```#T`0```#X5*____2(L+Z)<-
M``!$B[`8`@``Z)L.``!!.<8/A2W___](BPOH>@T``(N`$`(``"7_`0``/<`!
M```/A0____](B?GHG`X``$B-#:\W``!(B?)(B</HJN[__TB%P`^$Q`$``$B)
MP>AY#P``28L,)$B)PNB]]?__2(7`28G&#X2D`0``3(GQZ&D/``"#^`=^-$AC
MT$B-/>TW``"Y"0```$F-=!;X\Z9U'(/X"`^$>08``(/H"$B808!\!O\O#X1H
M!@``9I#H._K__X7`=1FZ```!`$R)\>A:#@``A<")1"1(#X5*`0``2(T5Z38`
M`$B-#6HV``!(C37O-0``3(TE%C8``$B-/:DW``#H9N___^B!#0``3(T-_C4`
M`$B-%8,W``!(B70D*(E$)"!)B>A(B=GH;PX``.LRZ-@,``"#.!%U.D&#Q0'H
M2@T``$B)="0P1(EL)"A-B>&)1"0@28GH2(GZ2(G9Z#L.``"ZP`$``$B)V>BN
M#0``@_C_=+Q(B>GH`0T``$B-#78U``!(B=KHXN[__TB)V>@*^O__Z=']__\/
M'T0``$B)P4B)1"1(Z//Y__](BT0D2$B)P>@F#@``2(G#Z:K]__^_`0```$B-
M#3$V``#K$TACQX/'`4B+C,2`````2(7)="=(B?+H%>W__TB%P'3@@#@`=-M(
MB<'HXPT``$B%P$B)PW3+Z5O[__^Y#0```$B-'>XU``#HQ0P``$F)Q>E<^___
M38LT).E3_O__3(VT)*`!``!!N`(```!(Q\+N____B<:)P>AV#```0;@&````
M3(GRB?'H]@P``("\)*`!````=0Z`O"2A`0``0P^$D04``(M,)$A%,<`QTDR-
MI"30````Z#@,``"XB:O-[TC'A"30`````2-%9TC'A"3H````=E0R$$B)A"38
M````N/[<NIA(QX0D^`````````!(B80DX````+CPX=+#2,>$)``!````````
M2(F$)/````#'A"1(`0```````&8/'T0``(M,)$A!N`"```!,B?+H3@P``(7`
M28G%B<</CJ\!``!(BXPD^````$ECQ8T4P4@YT0^'6@$``$B+C"0``0``2,'H
M'4B)E"3X````2`'(2(F$)``!``!(8X0D2`$``(7`#X0>`0``OT````!)C4P$
M.$R)\BG'03G]00]._4QC_TV)^.B*"P``BX0D2`$```'X@_A`B80D2`$```^%
M8?___T2)Z$R)X4T!]RGXB<?HO]/__X/_/P^.CP```(/O0$&)_4'![09$B>Y(
M@\8!2,'F!DP!_DF+%TR)X4F#QT!(B90D"`$``$F+5\A(B90D$`$``$F+5]!(
MB90D&`$``$F+5]A(B90D(`$``$F+5^!(B90D*`$``$F+5^A(B90D,`$``$F+
M5_!(B90D.`$``$F+5_A(B90D0`$``.@ST___23GW=9!!P>4&1"GO2&/738U$
M)#A(@_H(<TKVP@0/A9P#``!(A=)T$D$/M@_VP@)!B$PD.`^%"00``(F\)$@!
M``#I??[__TV)]^DF____#Q]$``!(B[0D``$``$B-3@'IG?[__TF+3!?X28E,
M$/A(@^H!2(/Z"'+"2(/B^#')38L,#TV)#`A(@\$(2#G1<N_KJ8M,)$CH>0D`
M`$B+M"3X````3(NL)``!``!(B?!(P>@#@^`_C5`!2)C&A`0(`0``@(/Z.`^/
M:P(``+@X````*=!(8]*#^`A)C504.`^#E@(``*@$#X5J`P``A<!T"Z@"Q@(`
M#X5N`P``3(GH3(GA1(BL)$,!``!(P>@80(BT)$<!``!,C;0D<`$``(B$)$`!
M``!,B>A(C;PD40$``$C!Z!!,C27,,@``B(0D00$``$R)Z$V-;BA(P>@(B(0D
M0@$``$B)\$C!Z!B(A"1$`0``2(GP2,'H$(B$)$4!``!(B?!,B?9(P>@(B(0D
M1@$``.BPT?__2(N$)-````!(B<*(A"13`0``2,'J&$0/ML*(E"10`0``2(G"
M2,'J$(B4)%$!``!(B<)(BX0DV````$C!Z@B(E"12`0``2(G"B(0D5P$``$C!
MZAB(E"14`0``2(G"2,'J$(B4)%4!``!(B<)(BX0DX````$C!Z@B(E"16`0``
M2(G"B(0D6P$``$C!ZAB(E"18`0``2(G"2,'J$(B4)%D!``!(B<)(BX0DZ```
M`$C!Z@B(E"1:`0``2(G"B(0D7P$``$C!ZAB(E"1<`0``2(G"2,'J$(B4)%T!
M``!(B<)(BX0D\````$C!Z@B(E"1>`0``2(G"B(0D8P$``$C!ZAB(E"1@`0``
M2(G"2,'J$(B4)&$!``!(B<)(P>H(B)0D8@$``.L(1`^V!TB#QP%(B?%,B>)(
M@\8"Z+8(``!).?5UY,:$))@!````2(T%Z"\``$R-#0\P``!(C16%,0``3(ET
M)"!)B>A(B=E(B40D*.A_"```Z5'Z__]-BV0D"$V%Y`^$C/G__TR)X>BT"```
M2(/X`P^&>OG__TF-=`3\2(T].C$``+D%````\Z9-#T3TZ5[Y__^X0````$AC
MRBG028U,##B)PG01,<!!B<"#P`$YT$+&!`$`<O%,B>'HTL___TF-5"0XN0<`
M```QP$B)U_-(J^F!_?__B<%(QP(`````2,=$"O@`````2(U*"$B#X?A(*<H!
MT(/@^(/X"`^"5OW__X/@^#'208G0@\((.<)*QP0!`````'+NZ3K]__]!BP]!
MB4PD.$&+1!?\08E$$/SI9/S__X"\)*(!``!!#X5A^O__@+PDHP$``$,/A5/Z
M__^`O"2D`0``2`^%1?K__X"\)*4!``!%#X4W^O__BW0D2$&X`@```$C'PL;_
M__^)\>AM!@``0;@I````3(GRB?'H[08``.F/_O__#Q^$``````!!#[=$%_YF
M08E$$/[IYOO__\<"`````,=$`OP`````Z9+\__\QR6:)3`+^Z8;\___H"P4`
M`)!F+@\?A```````059!54%455=64TB#[#!(BSU;;0``3(LMA&T``$R-9"0D
M2(L/28M%`$B)1"0H,<#HO`0``$B+#TB)P^BQ!```#[:`N````$B-#4$N``!,
MB>*(@[D```#H]>7__TB%P'0)@#@Q#X1Z`@``Z!+P__](BP_H>@0``$B-%>8O
M``!!N0L```!!N`$```!(B<'H_P,``$B%P`^$MP```(,]/[T```%(BT`02(MH
M*`^.H@```#'VNP$```#K,P\?A```````28M6$$B+!0V]``!(8\M(BPS(Z$D&
M``"%P'5EBP7_O```@\,!@\8!.=A^6TB+#^C]`P``13')3&/&2(GJ2(G!Z*P#
M``!(A<!T-TR+,$&+1@PE``0@`#T`!```=*1(BP_HS`,``$R)\D&Y`@```$4Q
MP$B)P>@P`P``2(G"ZX</'P"+!9J\``"-4/\Y\@^$8`$``$B+#^B7`P``2(T5
M""\``$&Y`P```$&X`0```$B)P>@<`P``2(7`2(G#=#)(BT`02(L/2(LPZ&4#
M``!,C078+@``2(G!2(GRZ,,"``!(BT,02(L8]D,.0`^%FP$``$B+#^@Z`P``
M2(T5LBX``$&Y`P```$&X`0```$B)P>B_`@``2(7`2(G#=#A(C0UU+```3(GB
MZ&CD__](A<!(B<8/A"P!``!(BT,02(L/2(L8Z.T"``!)B?!(B<%(B=KH3P(`
M`$B+#^C7`@``2(T542X``$&X`0```$B)P>AJ`@``2(L/2(G#Z+<"``!,C05&
M+@``2(G:2(G!Z!4"``!(BP_HG0(``$B-#?(K``#&@+D`````3(GBZ.?C__](
MA<!(B05UNP``='VZP`$``$B)P>CN`P``C5`!@_H!=@F#^!$/A=,```!(BT0D
M*$DS10`/A;\```!(@\0P6UY?74%<05U!7L.#^`$/CI?^__^[`0```)!(BP^#
MPP'H)0(``$B)ZDB)P>C2`0``.1T$NP``?^+I;O[__X!X`0`/A(']___I=_W_
M_TB+#=^Z``#HLO#__TB%P$B)!>"Z```/A6?___](C0U[+0``Z(8!``!F#Q]$
M``!(BP6QN@``2(L/2(LP2(M#$$B+&.BW`0``2(GR2(G9Z*SH___IR/[__TB+
M#^B?`0``2(G:2(G!Z"P!``#I3?[__^B:`0``08G!2(L%:+H``$R+!7&Z``!(
MC0TZ+0``2(L0Z!H!``"09@\?A```````5E-(@^PH2(G.Z"+L__\/M@B$R71+
M2(U0`4B)PP\?0`"`^2\/M@I(#T3:2(/"`83)=>[HB>[__TB%]G0=A<!T&8`^
M`'042(T5LRL``$B)V>B,`P``2#G8=`Q(@\0H6U[#2(G#Z\U(B?%(@\0H6U[I
M_>7__Y"0D)"0D)"0D)"0D)#_)0;?``"0D/\E]MX``)"0_R7FW@``D)#_)=;>
M``"0D/\EQMX``)"0_R6VW@``D)#_):;>``"0D/\EEMX``)"0_R6&W@``D)#_
M)7;>``"0D/\E9MX``)"0_R56W@``D)#_)4;>``"0D/\E-MX``)"0_R4FW@``
MD)#_)1;>``"0D/\E!MX``)"0_R7VW0``D)#_)>;=``"0D/\EUMT``)"0_R7&
MW0``D)#_);;=``"0D/\EIMT``)"0#Q^$``````#_)?;;``"0D)"0D)"0D)"0
M_R7&W```D)"0D)"0D)"0D/\EOMP``)"0D)"0D)"0D)#_)3[=``"0D`\?A```
M````45!(/0`0``!(C4PD&'(92('I`!```$B#"0!(+0`0``!(/0`0``!WYT@I
MP4B#"0!86<.0D)"0D)"0D)"0D)"0D/\E9ML``)"0D)"0D)"0D)#_)5[;``"0
MD)"0D)"0D)"0_R5>VP``D)"0D)"0D)"0D/\E9ML``)"0D)"0D)"0D)#_)6;;
M``"0D)"0D)"0D)"0_R5>VP``D)"0D)"0D)"0D/\E=ML``)"0D)"0D)"0D)#_
M)6[;``"0D)"0D)"0D)"0_R5FVP``D)"0D)"0D)"0D/\E7ML``)"0D)"0D)"0
MD)#_)5;;``"0D)"0D)"0D)"0_R5.VP``D)"0D)"0D)"0D/\E1ML``)"0D)"0
MD)"0D)#_)3[;``"0D)"0D)"0D)"0_R4VVP``D)"0D)"0D)"0D/\E+ML``)"0
MD)"0D)"0D)#_)2;;``"0D)"0D)"0D)"0_R4>VP``D)"0D)"0D)"0D/\E%ML`
M`)"0D)"0D)"0D)#_)0[;``"0D)"0D)"0D)"0_R4&VP``D)"0D)"0D)"0D/\E
M%ML``)"0D)"0D)"0D)#_)0[;``"0D)"0D)"0D)"0_R4&VP``D)"0D)"0D)"0
MD/\E_MH``)"0D)"0D)"0D)#_)?;:``"0D)"0D)"0D)"0_R7NV@``D)"0D)"0
MD)"0D/\EYMH``)"0D)"0D)"0D)#_)=[:``"0D)"0D)"0D)"0_R76V@``D)"0
MD)"0D)"0D/\ESMH``)"0D)"0D)"0D)#_)<;:``"0D)"0D)"0D)"0_R6^V@``
MD)"0D)"0D)"0D/\EMMH``)"0D)"0D)"0D)#_):[:``"0D)"0D)"0D)"0_R6F
MV@``D)"0D)"0D)"0D/\EGMH``)"0D)"0D)"0D)!(@^PH,=+H50```/\5*]D`
M`)"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0D)##D)"0D)"0D)"0D)"0
MD)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0D)!64TB#["A(B=-(
MB<ZY"````.CZ!```2(7;#X1!`@``2(ES0$B--9;___](C0UO!P``2(L5K-@`
M`$B%P,=#"$@!``!(B;.`````2(TU@/___TB)2TA(C0U=!P``QT,,V0<``,=#
M$`````!(B;.(````2(TU:?___TB)4QA(B4M02(T52OW__TB-#8.U``!(B;.0
M````2(TU5?___\>#\`````````!(B4LX2(E3*$B-#1S^__](B;.8````2(TU
M;OW__TB-%5<$``#'@_0````^`0``2,<#`````$B)<R!(C34+!```2(E+,$B)
MDQ@!``!(B7-X#X3E````2(N0X````$B%T@^$U0```(L%$+4``(7`#X7'````
M2(,]T!0````/A*H"``!(@SW*%`````^$C`(``$B#/<04````#X1N`@``2(,]
MOA0````/A%`"``!(@SVX%`````^$,@(``$B#/;(4````#X04`@``2(,]K!0`
M```/A/8!``!(@SVF%`````^$V`$``$B+!6$4``!(B0)(BP5?%```2(E""$B+
M!5P4``!(B4(02(L%610``$B)0AA(BP56%```2(E"($B+!5,4``!(B4(H2(L%
M4!0``$B)0C!(BP5-%```2(E".#')Z)(#``!(B8/H````2(T%W!,``#')2(E#
M6$B-!3<4``!(B4-@2(T%Q*,``$B)0VA(C069M0``2(E#<$B-!8YE``!(B8,@
M`0``2(T%O&4``$B)@R@!``!(C062L___2(F#,`$``.B6`@``N`$```!(@\0H
M6U[#9BX/'X0``````$B-#3D%``!(BQ5VU@``2(EP0$B--4?]__](A<#'0`A(
M`0``2(E(2$B-#2H%``#'0`S9!P``2(FP@````$B--3W]___'0!``````2(E(
M4$B+B$`!``!(B;"0````2(TU;?O__\>`\`````````#'@/0````^`0``2(D*
M2(T5[_S__TB-#0C]__](B7`@2(TU_0$``$C'``````!(B9"(````2(F(F```
M`$B-%0&S``!(C0VZ^O__2(EP>$B)4#A(C16K^___2(E(*$B-#?`!``!(B5`P
M2(F(&`$```^$GO[__TB+D.````!(B<-(A=(/A;;]___IAO[__P\?@`````!(
MBT(X2(D%O1(``.D8_O__2(M",$B)!:42``#I^OW__TB+0BA(B06-$@``Z=S]
M__](BT(@2(D%=1(``.F^_?__2(M"&$B)!5T2``#IH/W__TB+0A!(B05%$@``
MZ8+]__](BT((2(D%+1(``.ED_?__2(L"2(D%%A(``.E'_?__D$R+!4FR``"+
M%4NR``!(BPU,L@``Z=<````/'X``````2(/L.(/Z`71G<C6#^@)T$X/Z`W4<
MN@,```!(@\0XZ:T```"Z`@```$B#Q#CIGP```+@!````2(/$.,,/'T0``#'2
MZ(D```"%P'3K2(L-UK$``(E$)"SHI0```$C'!<*Q``#_____BT0D+.O*#Q]`
M`#'`2(T59____TV%P`^4P$B)#;JQ``!,B06CL0``QP6AL0```0```(D%@[$`
M`.B.````2(/X_TB)!7NQ```/E<`/ML!(@\0XPP\?A```````Z3O___^0D)"0
MD)"0D)"0D+@!````PY"0D)"0D)"0D)##D)"0D)"0D)"0D)"0D)"0_R7^TP``
MD)"0D)"0D)"0D/\E!M0``)"0D)"0D)"0D)#_)?[3``"0D)"0D)"0D)"0_R5^
MU```D)"0D)"0D)"0D%-(@^P@2(G+2(G12(T5#K$``.BY^O__2(T5`K$``$B)
MV4B#Q"!;Z04```"0D)"0D/\EMM,``)"0D)"0D)"0D)#_);[5``"0D/\EKM4`
M`)"05U932(/L4(E,)'!(B50D.$R)1"0PZ-;W__](BQW_7P``2(U4)#A,C40D
M,$B-3"1P2(L#2(E$)$@QP.B1]O__3(T%\O;__TB-#?/V__],B<+H"_?__TB+
M!81?``!(BS6-7P``@#@`=3/H2_;__TB%P$B)!?FO```/A'L!``!(B<'H*_;_
M_TB+-61?``!(BP[HW/;__\:`^0@```!(BP[HS?;__X"(T@0```)(BP[HOO;_
M_X"(T@0```&+1"1PC4@%2&/)2,'A`^@D]O__2(M,)#A)B<%(B05]KP``1(M<
M)'#'!7:O```$````N@4```!(BP%(@\$(08/[`4F)`4B-!74B``!)B4$(2(L%
M60\``$F)01!(C05B(@``28E!&$&-0P1^%DR+$4QCPH/"`4B#P0@YPD^)5,'X
M=>I(BPTOKP``2)A%C4,#2(T50M[__TG'1,'X`````$C'1"0@`````.@S]?__
MA<")QG1*2(L-_JX``$B-?"1$Z"SU__](C0V;'P``2(GZZ$77__](A<!T-4B+
M#=FN``#H!/7__^@G]?__2(M\)$A(,SN)\'552(/$4%M>7\-(BPVTK@``Z,_T
M__^)QNNF2(L-G*X``$B%R70'Z#KT___KN$B-#>T>``!(B?KHZ=;__TB%P$B)
MP4B)!72N``!TFNO9N0$```#H+O;__^AY]?__D)"0D)"0D)"054B)Y5WIEK[_
M_Y"0D)"0D/__________P%%```$``````````````/__________````````
M``````````````````````````````````#_____`````$!S0``!````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````8WEG9V-J+3$V+F1L;`!?2G9?
M4F5G:7-T97)#;&%S<V5S````````````````````+@`N+@`E<R\E<P`O<')O
M8R]S96QF+V5X90!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D97(``$EN=&5R
M;F%L<SHZ4$%2.CI"3T]4`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"\`)7,E
M<R5S`%!%4DPU3$E"`%!%4DQ,24(`4$523#5/4%0`4$523$E/`%!!4E])3DE4
M24%,25I%1`!005)?4U!!5TY%1`!005)?0TQ%04X`4$%27T1%0E5'`%!!4E]#
M04-(10!005)?1TQ/0D%,7T1%0E5'`%!!4E]'3$]"04Q?5$U01$E2`%!!4E]4
M35!$25(`4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`4$%27T-,
M14%./0`Q``````!?7T5.5E]005)?0TQ%04Y?7R`@("`@("`@("`@("`@(```
M4$%42`!54T52`%-94U1%30`O=&UP`%1-4$1)4@!414U01$E2`%1%35``5$U0
M`%5315).04U%`"4P,G@`)7,E<R5S)7,`<&%R+0``````)7,Z(&-R96%T:6]N
M(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI
M"@!P87)L+F5X90`N<&%R`"5S)7-C86-H92TE<R5S`"5S)7-T96UP+25U)7,`
M)7,E<W1E;7`M)74M)74E<P``)7,Z('!R:79A=&4@<W5B9&ER96-T;W)Y("5S
M(&ES('5N<V%F92`H<&QE87-E(')E;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O
M<&5R871I;VXI"@!!4D=6`!@`<&5R;``P`%!!4CHZ4&%C:V5R.CI615)324].
M`#$N,#0P``````!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``
M````````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T
M;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@`M90`M+0``<&5R;'AS:2YC````
M`````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")]
M)PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G
M92!?7W!A<E]P;#L*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL
M92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q=6ER92!&:6QE
M.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U:7)E($9I;&4Z
M.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D
M<&%R7W1E;7`I.PIO=71S*'%Q>U)E;6]V:6YG(&9I;&5S(&EN("(D<&%R7W1E
M;7`B?2D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD("D@/R!R
M;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D<&%R7W1E;7`[
M"FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R)RD@>PIM>2`D
M=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%(#T^("=T;7!86%A86"<L
M"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER*2P*4U5&
M1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD["G!R:6YT("1T;7`@(B,A
M+V)I;B]S:`IX/3$[('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D;PIR;2`M<F8@
M)R1P87)?=&5M<"<*:68@6R!<(2`M9"`G)'!A<E]T96UP)R!=.R!T:&5N"F)R
M96%K"F9I"G-L965P(#$*>#U@97AP<B!<)'@@*R`Q8`ID;VYE"G)M("<B("X@
M)'1M<"T^9FEL96YA;64@+B`B)PHB.PIC:&UO9"`P-S`P+"1T;7`M/F9I;&5N
M86UE.PIM>2`D8VUD(#T@)'1M<"T^9FEL96YA;64@+B`G(#XO9&5V+VYU;&P@
M,CXF,2`F)SL*8VQO<V4@)'1M<#L*<WES=&5M*"1C;60I.PIO=71S*'%Q*%-P
M87=N960@8F%C:V=R;W5N9"!P<F]C97-S('1O('!E<F9O<FT@8VQE86YU<#H@
M*0HN("1T;7`M/F9I;&5N86UE*3L*?0I]('T*0D5'24X@>PI);G1E<FYA;',Z
M.E!!4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"3T]4
M.PIE=F%L('L*7W!A<E]I;FET7V5N=B@I.PIM>2`D<75I970@/2`A)$5.5GM0
M05)?1$5"54=].PIM>2`E0V]N9FEG(#T@*`IP871H7W-E<"`@("`]/B`H)%Y/
M(#U^("]>35-7:6XO(#\@)SLG(#H@)SHG*2P*7V5X92`@("`@("`@/3X@*"1>
M3R`]?B`O7B@_.DU35VEN?$]3,GQC>6=W:6XI+R`_("<N97AE)R`Z("<G*2P*
M7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O7DU35VEN?$]3,B\@/R`G7%PG(#H@
M)R\G*2P**3L*7W-E=%]P<F]G;F%M92@I.PI?<V5T7W!A<E]T96UP*"D["FUY
M("@D<W1A<G1?<&]S+"`D9&%T85]P;W,I.PI["FQO8V%L("1324=[7U]705).
M7U]](#T@<W5B('M].PIO<&5N(%]&2"P@)SPG+"`D<')O9VYA;64@;W(@;&%S
M=#L*8FEN;6]D92A?1D@I.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US("1P<F]G
M;F%M93L*;7D@)&]F9G-E="`](#4Q,CL*;7D@)&ED>"`]("TQ.PIW:&EL92`H
M,2D*>PHD;V9F<V5T(#T@)'-I>F4@:68@)&]F9G-E="`^("1S:7IE.PIS965K
M(%]&2"P@+21O9F9S970L(#(@;W(@9&EE('%Q6W-E96L@9F%I;&5D(&]N("(D
M<')O9VYA;64B.B`D(5T["FUY("1N<F5A9"`](')E860@7T9(+"`D8G5F+"`D
M;V9F<V5T.PID:64@<7%;<F5A9"!F86EL960@;VX@(B1P<F]G;F%M92(Z("0A
M72!U;FQE<W,@)&YR96%D(#T]("1O9F9S970["B1I9'@@/2!R:6YD97@H)&)U
M9BP@(EQN4$%2+G!M7&XB*3L*;&%S="!I9B`D:61X(#X](#`@?'P@)&]F9G-E
M="`]/2`D<VEZ92!\?"`D;V9F<V5T(#X@,3(X("H@,3`R-#L*)&]F9G-E="`J
M/2`R.PI]"FQA<W0@=6YL97-S("1I9'@@/CT@,#L*)&]F9G-E="`M/2`D:61X
M("T@-#L*<V5E:R!?1D@L("TD;V9F<V5T+"`R.PIR96%D(%]&2"P@)&)U9BP@
M-#L*<V5E:R!?1D@L("TD;V9F<V5T("T@=6YP86-K*").(BP@)&)U9BDL(#([
M"G)E860@7T9(+"`D8G5F+"`T.PHD9&%T85]P;W,@/2`H=&5L;"!?1D@I("T@
M-#L*;7D@)7)E<75I<F5?;&ES=#L*=VAI;&4@*"1B=68@97$@(D9)3$4B*2![
M"G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").
M(BP@)&)U9BD["FUY("1F=6QL;F%M92`]("1B=68["F]U=',H<7$H56YP86-K
M:6YG(&9I;&4@(B1F=6QL;F%M92(N+BXI*3L*;7D@)&-R8R`]("@@)&9U;&QN
M86UE(#U^('-\7BA;82UF7&1=>SA]*2]\?"`I(#\@)#$@.B!U;F1E9CL*;7D@
M*"1B87-E;F%M92P@)&5X="D@/2`H)&)U9B`]?B!M?"@_.BXJ+RD_*"XJ*2A<
M+BXJ*7PI.PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N
M<&%C:R@B3B(L("1B=68I.PII9B`H9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A
M?B`O7"XH/SIP;7QP;'QI>'QA;"DD+VDI('L*;7D@)&9I;&5N86UE(#T@7W1E
M;7!F:6QE*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ
M1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y
M.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-I9B`H
M("1F=6QL;F%M92`]?B!M?%XO/W-H;&EB+WP@86YD(&1E9FEN960@)$5.5GM0
M05)?5$5-4'T@*2!["FUY("1F:6QE;F%M92`](%]T96UP9FEL92@B)&)A<V5N
M86UE)&5X="(L("1B=68L(#`W-34I.PIO=71S*")32$Q)0CH@)&9I;&5N86UE
M7&XB*3L*?0IE;'-E('L*)')E<75I<F5?;&ES='LD9G5L;&YA;65](#T*)%!!
M4CHZ2&5A=GDZ.DUO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D
M8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["GT*<F5A
M9"!?1D@L("1B=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E
M;&8L("1M;V1U;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A
M)&UO9'5L93L*;7D@)&9I;&5N86UE(#T@9&5L971E("1R97%U:7)E7VQI<W1[
M)&UO9'5L97T@?'P@9&\@>PIM>2`D:V5Y.PIF;W)E86-H("AK97ES("5R97%U
M:7)E7VQI<W0I('L*;F5X="!U;FQE<W,@+UQ1)&UO9'5L95Q%)"\["B1K97D@
M/2`D7SL@;&%S=#L*?0ID96QE=&4@)')E<75I<F5?;&ES='LD:V5Y?2!I9B!D
M969I;F5D*"1K97DI.PI](&]R(')E='5R;CL*)$E.0WLD;6]D=6QE?2`]("(O
M;&]A9&5R+R1F:6QE;F%M92\D;6]D=6QE(CL*:68@*"1%3E9[4$%27T-,14%.
M?2!A;F0@9&5F:6YE9"@F24\Z.D9I;&4Z.FYE=RDI('L*;7D@)&9H(#T@24\Z
M.D9I;&4M/FYE=U]T;7!F:6QE(&]R(&1I92`D(3L*8FEN;6]D92@D9F@I.PIP
M<FEN="`D9F@@)&9I;&5N86UE+3Y[8G5F?3L*<V5E:R@D9F@L(#`L(#`I.PIR
M971U<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?=&5M<&9I;&4H
M(B1F:6QE;F%M92T^>V-R8WTN<&TB+"`D9FEL96YA;64M/GMB=69]*3L*;W!E
M;B!M>2`D9F@L("<\)RP@)&9I;&5N86UE(&]R(&1I92`B8V%N)W0@<F5A9"`D
M9FEL96YA;64Z("0A(CL*8FEN;6]D92@D9F@I.PIR971U<FX@)&9H.PI]"F1I
M92`B0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;FYO="!F:6YD("1M;V1U;&4A
M7&XB.PI]+"!`24Y#*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z
M.DAE879Y.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R
M.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U
M:7)E($E/.CI&:6QE.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y
M<R`E<F5Q=6ER95]L:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE
M?2!O<B`D9FEL96YA;64@/7X@+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]
M?B`O7"YP;6,_)"]I*2!["G)E<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID
M;R`D9FEL96YA;64@=6YL97-S("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ
M95PN<&PD+SL*?0I]"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T[
M"GT*;&%S="!U;FQE<W,@)&)U9B!E<2`B4$M<,#`S7#`P-"(["B1S=&%R=%]P
M;W,@/2`H=&5L;"!?1D@I("T@-#L*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T
M+"`D8G5N9&QE+"`D;&]G9F@L("1C86-H95]N86UE*3L*9&5L971E("1%3E9[
M4$%27T%04%]2155317T[(",@<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E
M8W5R:71Y('!R;V)L96TI"B1Q=6EE="`](#`@=6YL97-S("1%3E9[4$%27T1%
M0E5'?3L*:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M
M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@
M8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R
M"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD
M("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM0
M05)?05!07U)%55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA
M;"!P87)L(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6
M*2!["B1!4D=66S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R
M(&QA<W0["FEF("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#([
M"GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF
M("@D,2!E<2`G02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF
M("@D,2!E<2`G3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I
M('L*)&)U;F1L92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B
M=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`]
M(#$["GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@
M)#(@;W(@9&EE(")86%@Z($-A;FYO="!O<&5N(&QO9SH@)"$B.PI]"F5L<VEF
M("@D,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'
M5BD["FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.
M5GLG4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0[
M"B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z
M.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF
M="!`24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER
M92!)3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@)'!A<B`]
M('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A<BD@>PI!
M<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@Q,C@J,3`R-"D["F]P96X@;7D@
M)&9H+"`G/"<L("1P87(@;W(@9&EE(")#86YN;W0@9FEN9"`G)'!A<B<Z("0A
M(CL*8FEN;6]D92@D9F@I.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP
M(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N
M9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@
M9&EE(")296%D("<D<&%R)R!E<G)O<CH@)"$B.PI]"FUY("5E;G8@/2!D;R![
M"FEF("@D>FEP(&%N9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-151!
M+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R.B0O+VUS.PHD;65T82`]?B!S
M+UY<4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;7CI=*RDZ("@N*RDD+VUG.PI]
M"GT["FQO8V%L("0O(#T@7#0["FEF("AD969I;F5D("1P87(I('L*;W!E;B!0
M05(L("<\)RP@)'!A<B!O<B!D:64@(B0A.B`D<&%R(CL*8FEN;6]D92A005(I
M.PID:64@(B1P87(@:7,@;F]T(&$@4$%2(&9I;&4B('5N;&5S<R`\4$%2/B!E
M<2`B4$M<,#`S7#`P-"(["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]
M($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)
M3SHZ1FEL93HZ3U]74D].3%DH*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W
M-S<L"BD@;W(@9&EE("0A.PIB:6YM;V1E*"1F:"D["B0O(#T@*&1E9FEN960@
M)&1A=&%?<&]S*2`_(%PD9&%T85]P;W,@.B!U;F1E9CL*<V5E:R!?1D@L(#`L
M(#`["FUY("1L;V%D97(@/2!S8V%L87(@/%]&2#X["FEF("@A)$5.5GM005)?
M5D520D%424U](&%N9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R96TI+RD@>PIR
M97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*4$%2.CI&:6QT97(Z.E!O
M9%-T<FEP+3YN97<M/F%P<&QY*%PD;&]A9&5R+"`D,"D*?0IF;W)E86-H(&UY
M("1K97D@*'-O<G0@:V5Y<R`E96YV*2!["FUY("1V86P@/2`D96YV>R1K97E]
M(&]R(&YE>'0["B1V86P@/2!E=F%L("1V86P@:68@)'9A;"`]?B`O7ELG(ETO
M.PIM>2`D;6%G:6,@/2`B7U]%3E9?4$%27R(@+B!U8R@D:V5Y*2`N(")?7R([
M"FUY("1S970@/2`B4$%27R(@+B!U8R@D:V5Y*2`N("(])'9A;"(["B1L;V%D
M97(@/7X@<WLD;6%G:6,H("LI?7L*)&UA9VEC("X@)'-E="`N("@G("<@>"`H
M;&5N9W1H*"0Q*2`M(&QE;F=T:"@D<V5T*2DI"GUE9SL*?0HD9F@M/G!R:6YT
M*"1L;V%D97(I.PHD+R`]('5N9&5F.PII9B`H)&)U;F1L92D@>PIR97%U:7)E
M(%!!4CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*
M:6YI=%]I;F,H*3L*<F5Q=6ER95]M;V1U;&5S*"D["FUY($!I;F,@/2!G<F5P
M('L@(2]"4T1004XO('T*9W)E<"!["B@D8G5N9&QE(&YE("=S:71E)RD@;W(*
M*"1?(&YE("1#;VYF:6<Z.D-O;F9I9WMA<F-H;&EB97AP?2!A;F0*)%\@;F4@
M)$-O;F9I9SHZ0V]N9FEG>W!R:79L:6)E>'!]*3L*?2!`24Y#.PIM>2`E9FEL
M97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O
M8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#
M;VYF:6=[;&EB7V5X='T["FUY("5W<FET=&5N.PIF;W)E86-H("AS;W)T(&ME
M>7,@)69I;&5S*2!["FUY("@D;F%M92P@)&9I;&4I.PIF;W)E86-H(&UY("1D
M:7(@*$!I;F,I('L*:68@*"1N86UE(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H
M97LD7WTI('L*)&9I;&4@/2`D7SL*;&%S=#L*?0IE;'-I9B`H+UXH7%$D9&ER
M7$5<+R@N*EM>0V-=*2E<6B]I*2!["B@D9FEL92P@)&YA;64I(#T@*"0Q+"`D
M,BD["FQA<W0["GT*96QS:68@*&TA7B]L;V%D97(O6UXO72LO*"XJ6UY#8UTI
M7%HA*2!["FEF("AM>2`D<F5F(#T@)%!!4CHZ2&5A=GDZ.DUO9'5L94-A8VAE
M>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0["GT*
M96QS:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I
M<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D
M;F%M92!A;F0@;F]T("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R968H
M)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+SL*;W5T<R@@
M:F]I;B`B(BP*<7$H4&%C:VEN9R`B*2P@<F5F("1F:6QE(#\@)&9I;&4M/GMN
M86UE?2`Z("1F:6QE+`IQ<2@B+BXN*0HI.PIM>2`D8V]N=&5N=#L*:68@*')E
M9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS92![
M"F]P96X@1DE,12P@)SPG+"`D9FEL92!O<B!D:64@(D-A;B=T(&]P96X@)&9I
M;&4Z("0A(CL*8FEN;6]D92A&24Q%*3L*)&-O;G1E;G0@/2`\1DE,13X["F-L
M;W-E($9)3$4["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^;F5W+3YA<'!L>2A<
M)&-O;G1E;G0L("1F:6QE*0II9B`A)$5.5GM005)?5D520D%424U](&%N9"`D
M;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["E!!4CHZ1FEL=&5R.CI0871C
M:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D9FEL92P@)&YA;64I
M.PI]"F]U=',H<7$H5W)I='1E;B!A<R`B)&YA;64B*2D["B1F:"T^<')I;G0H
M(D9)3$4B*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@;&5N9W1H*"1N86UE*2`K
M(#DI*3L*)&9H+3YP<FEN="AS<')I;G1F*`HB)3`X>"\E<R(L($%R8VAI=F4Z
M.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64**2D["B1F:"T^
M<')I;G0H<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI*3L*)&9H+3YP<FEN
M="@D8V]N=&5N="D["GT*?0HD>FEP+3YW<FET951O1FEL94AA;F1L92@D9F@I
M(&EF("1Z:7`["B1C86-H95]N86UE(#T@<W5B<W1R("1C86-H95]N86UE+"`P
M+"`T,#L*:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@)&UT:6UE(#T@*'-T870H
M)&]U="DI6SE=*2!["FUY("1C='@@/2!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z
M.E-(03L@1&EG97-T.CI32$$M/FYE=R@Q*2!]"GQ\(&5V86P@>R!R97%U:7)E
M($1I9V5S=#HZ4TA!,3L@1&EG97-T.CI32$$Q+3YN97<@?0I\?"!E=F%L('L@
M<F5Q=6ER92!$:6=E<W0Z.DU$-3L@1&EG97-T.CI-1#4M/FYE=R!].PIM>2`D
M<VAA7W9E<G-I;VX@/2!E=F%L('L@)$1I9V5S=#HZ4TA!.CI615)324].('T@
M?'P@,#L*:68@*"1S:&%?=F5R<VEO;B!E<2`G-2XS."<@;W(@)'-H85]V97)S
M:6]N(&5Q("<U+C,Y)RD@>PHD8W1X+3YA9&1F:6QE*"1O=70L(")B(BD@:68@
M*"1C='@I.PI]"F5L<V4@>PII9B`H)&-T>"!A;F0@;W!E;BAM>2`D9F@L("(\
M)&]U="(I*2!["F)I;FUO9&4H)&9H*3L*)&-T>"T^861D9FEL92@D9F@I.PIC
M;&]S92@D9F@I.PI]"GT*)&-A8VAE7VYA;64@/2`D8W1X(#\@)&-T>"T^:&5X
M9&EG97-T(#H@)&UT:6UE.PI]"B1C86-H95]N86UE("X](")<,"(@>"`H-#$@
M+2!L96YG=&@@)&-A8VAE7VYA;64I.PHD8V%C:&5?;F%M92`N/2`B0T%#2$4B
M.PHD9F@M/G!R:6YT*"1C86-H95]N86UE*3L*)&9H+3YP<FEN="AP86-K*"=.
M)RP@)&9H+3YT96QL("T@;&5N9W1H*"1L;V%D97(I*2D["B1F:"T^<')I;G0H
M(EQN4$%2+G!M7&XB*3L*)&9H+3YC;&]S93L*8VAM;V0@,#<U-2P@)&]U=#L*
M97AI=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I
M>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?
M9'EN86QO861E<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!
M<F-H:79E.CI::7`["GT*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PIM
M>2`D9F@@/2!)3SHZ1FEL92T^;F5W.PHD9F@M/F9D;W!E;BAF:6QE;F\H7T9(
M*2P@)W(G*2!O<B!D:64@(B0A.B`D0"(["B1Z:7`M/G)E861&<F]M1FEL94AA
M;F1L92@D9F@L("1P<F]G;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I
M(&]R(&1I92`B)"$Z("1`(CL*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*
M)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["B1Q=6EE="`]("$D
M14Y6>U!!4E]$14)51WT["F]U=',H<7$H7"1%3E9[4$%27U1%35!](#T@(B1%
M3E9[4$%27U1%35!](BDI.PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@
M>R`C('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D
M;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X="!I9B`D;65M8F5R+3YI
M<T1I<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA
M;64["FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O
M"B@_.B1#;VYF:6<Z.D-O;F9I9WMV97)S:6]N?2\I/PHH/SHD0V]N9FEG.CI#
M;VYF:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?
M;F%M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL
M92@D14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S
M=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I
M*2!["F]U=',H<7$H4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@
M86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B*2D["GT@96QS92!["F]U
M=',H<7$H17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M
M92(I*3L*)&UE;6)E<BT^97AT<F%C=%1O1FEL94YA;65D*"1D97-T7VYA;64I
M.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE*2!I9B`D7D\@97$@(FAP=7@B.PI]
M"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2D@>PID
M:64@/#P@(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R
M(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@72!;('-R8RYP87(@72!;('!R;V=R
M86TN<&P@70HD,"!;("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C+G!A<@HN"B1%
M3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M92`]("0P(#T@<VAI9G0H0$%2
M1U8I.PI]"G-U8B!#<F5A=&50871H('L*;7D@*"1N86UE*2`]($!?.PIR97%U
M:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H)&)A<V5N86UE+"`D<&%T:"P@)&5X
M="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N86UE+"`H)UPN+BHG
M*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*1FEL93HZ4&%T:#HZ;6MP871H*"1P
M871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK<&%T:"!D:65S('=I=&@@97)R
M;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S('L*<F5Q=6ER92!L:6(["G)E<75I
M<F4@1'EN84QO861E<CL*<F5Q=6ER92!I;G1E9V5R.PIR97%U:7)E('-T<FEC
M=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q=6ER92!V87)S.PIR97%U:7)E($-A
M<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17)R;F\["G)E<75I
M<F4@17AP;W)T97(Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.PIR97%U:7)E
M($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ4W!E
M8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!#;VYF:6<["G)E<75I<F4@
M24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!#;VUP<F5S
M<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@4$%2.PIR
M97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@4$%2.CI$:7-T.PIR97%U:7)E
M(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ
M4&%T8VA#;VYT96YT.PIR97%U:7)E(&%T=')I8G5T97,["F5V86P@>R!R97%U
M:7)E($-W9"!].PIE=F%L('L@<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@<F5Q
M=6ER92!38V%L87(Z.E5T:6P@?3L*979A;"![(')E<75I<F4@07)C:&EV93HZ
M56YZ:7`Z.D)U<G-T('T["F5V86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ3F%M
M961#87!T=7)E('T["F5V86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER92!0
M97)L24\Z.G-C86QA<B!].PIE=F%L('L@<F5Q=6ER92!U=&8X('T["GT*<W5B
M(%]S971?<&%R7W1E;7`@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2!A
M;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]("0Q
M.PIR971U<FX["GT*9F]R96%C:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD7WTL
M('%W*"!005)?5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI+`IQ
M=R@@0SI<7%1%35`@+W1M<"`N("D**2!["FYE>'0@=6YL97-S(&1E9FEN960@
M)'!A=&@@86YD("UD("1P871H(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N86UE
M.PIM>2`D<'=U:60["F5V86P@>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF
M(&1E9FEN960@)#X[?3L*:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE
M*2`I('L*)'5S97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64["GT*96QS:68@
M*&1E9FEN960@)'!W=6ED*2!["B1U<V5R;F%M92`]("1P=W5I9#L*?0IE;'-E
M('L*)'5S97)N86UE(#T@)$5.5GM54T523D%-17T@?'P@)$5.5GM54T52?2!\
M?"`G4UE35$5-)SL*?0HD=7-E<FYA;64@/7X@<R]<5R]?+V<["FUY("1S=&UP
M9&ER(#T@(B1P871H)$-O;F9I9WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L
M("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PII9B`H(21%3E9[
M4$%27T-,14%.?2!A;F0@;7D@)&UT:6UE(#T@*'-T870H)'!R;V=N86UE*2E;
M.5TI('L*;W!E;B`H;7D@)&9H+"`B/"(N("1P<F]G;F%M92D["G-E96L@)&9H
M+"`M,3@L(#(["G-Y<W)E860@)&9H+"!M>2`D8G5F+"`V.PII9B`H)&)U9B!E
M<2`B7#!#04-(12(I('L*<V5E:R`D9F@L("TU."P@,CL*<WES<F5A9"`D9F@L
M("1B=68L(#0Q.PHD8G5F(#U^(',O7#`O+V<["B1S=&UP9&ER("X]("(D0V]N
M9FEG>U]D96QI;7UC86-H92TB("X@)&)U9CL*?0IE;'-E('L*;7D@)&-T>"`]
M(&5V86P@>R!R97%U:7)E($1I9V5S=#HZ4TA!.R!$:6=E<W0Z.E-(02T^;F5W
M*#$I('T*?'P@979A;"![(')E<75I<F4@1&EG97-T.CI32$$Q.R!$:6=E<W0Z
M.E-(03$M/FYE=R!]"GQ\(&5V86P@>R!R97%U:7)E($1I9V5S=#HZ340U.R!$
M:6=E<W0Z.DU$-2T^;F5W('T["FUY("1S:&%?=F5R<VEO;B`](&5V86P@>R`D
M1&EG97-T.CI32$$Z.E9%4E-)3TX@?2!\?"`P.PII9B`H)'-H85]V97)S:6]N
M(&5Q("<U+C,X)R!O<B`D<VAA7W9E<G-I;VX@97$@)S4N,SDG*2!["B1C='@M
M/F%D9&9I;&4H)'!R;V=N86UE+"`B8B(I(&EF("@D8W1X*3L*?0IE;'-E('L*
M:68@*"1C='@@86YD(&]P96XH;7D@)&9H+"`B/"1P<F]G;F%M92(I*2!["F)I
M;FUO9&4H)&9H*3L*)&-T>"T^861D9FEL92@D9F@I.PIC;&]S92@D9F@I.PI]
M"GT*)'-T;7!D:7(@+CT@(B1#;VYF:6=[7V1E;&EM?6-A8VAE+2(@+B`H("1C
M='@@/R`D8W1X+3YH97AD:6=E<W0@.B`D;71I;64@*3L*?0IC;&]S92@D9F@I
M.PI]"F5L<V4@>PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PHD<W1M<&1I<B`N/2`B
M)$-O;F9I9WM?9&5L:6U]=&5M<"TD)"(["GT*)$5.5GM005)?5$5-4'T@/2`D
M<W1M<&1I<CL*;6MD:7(@)'-T;7!D:7(L(#`W-34["FQA<W0["GT*)'!A<E]T
M96UP(#T@)#$@:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!]
M(#U^("\H+BLI+SL*?0IS=6(@7W1E;7!F:6QE('L*;7D@*"1N86UE+"`D8V]N
M=&5N=',L("1M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T96UP
M+R1N86UE(CL*=6YL97-S("@M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N86UE
M(#T@(B1F=6QL;F%M92XD)"(["F]P96X@;7D@)&9H+"`G/B<L("1T96UP;F%M
M92!O<B!D:64@(F-A;B=T('=R:71E("1T96UP;F%M93H@)"$B.PIB:6YM;V1E
M("1F:#L*<')I;G0@)&9H("1C;VYT96YT<SL*8VQO<V4@)&9H.PIC:&UO9"`D
M;6]D92P@)'1E;7!N86UE(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D=&5M
M<&YA;64L("1F=6QL;F%M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*<F5T
M=7)N("1F=6QL;F%M93L*?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD969I
M;F5D("1%3E9[4$%27U!23T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!345]
M(#U^("\H+BLI+RD@>PHD<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@?'P]
M("0P.PII9B`H)$5.5GM005)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@
M)$5.5GM005)?5$5-4'TI(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R*"1P
M<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`K
M(#$I.PI]"FEF("@A)$5.5GM005)?4%)/1TY!345](&]R(&EN9&5X*"1P<F]G
M;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY("1F
M:"P@)SPG+"`D<')O9VYA;64I('L*<F5T=7)N(&EF("US("1F:#L*?0II9B`H
M+7,@(B1P<F]G;F%M921#;VYF:6=[7V5X97TB*2!["B1P<F]G;F%M92`N/2`D
M0V]N9FEG>U]E>&5].PIR971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@*'-P
M;&ET("]<421#;VYF:6=[<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE
M>'0@:68@97AI<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[
M4$%27U1%35!].PHD9&ER(#U^(',O7%$D0V]N9FEG>U]D96QI;7U<120O+SL*
M*"@D<')O9VYA;64@/2`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M921#
M;VYF:6=[7V5X97TB*2P@;&%S="D*:68@+7,@(B1D:7(D0V]N9FEG>U]D96QI
M;7TD<')O9VYA;64D0V]N9FEG>U]E>&5](CL**"@D<')O9VYA;64@/2`B)&1I
M<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M92(I+"!L87-T*0II9B`M<R`B)&1I
M<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P<F]G
M;F%M92!["B0P(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-17T[
M"FEF("AI;F1E>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@/"`P*2![
M"B1P<F]G;F%M92`]("(N)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE(CL*?0IM
M>2`D<'=D(#T@*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D
M*"D*.B`H*&1E9FEN960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W
M9"@I(#H@8'!W9&`I.PIC:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O7B@_
M/5PN7"X_7%$D0V]N9FEG>U]D96QI;7U<12DO)'!W9"1#;VYF:6=[7V1E;&EM
M?2\["B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*?0IS=6(@7W!A
M<E]I;FET7V5N=B!["FEF("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@
M,2`I('L*<F5T=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])3DE424%,25I%1'T@
M/2`R.PI]"F9O<B`H<7<H(%-005=.140@5$5-4"!#3$5!3B!$14)51R!#04-(
M12!04D]'3D%-12!!4D=#($%21U9?,"`I("D@>PID96QE=&4@)$5.5GLG4$%2
M7R<N)%]].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I
M('L*)$5.5GLG4$%27R<N)%]](#T@)$5.5GLG4$%27T=,3T)!3%\G+B1??2!I
M9B!E>&ES=',@)$5.5GLG4$%27T=,3T)!3%\G+B1??3L*?0IM>2`D<&%R7V-L
M96%N(#T@(E]?14Y67U!!4E]#3$5!3E]?("`@("`@("`@("`@("`@(CL*:68@
M*"1%3E9[4$%27U1%35!]*2!["F1E;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*
M96QS:68@*"%E>&ES=',@)$5.5GM005)?1TQ/0D%,7T-,14%.?2D@>PIM>2`D
M=F%L=64@/2!S=6)S='(H)'!A<E]C;&5A;BP@,3(@*R!L96YG=&@H(D-,14%.
M(BDI.PHD14Y6>U!!4E]#3$5!3GT@/2`D,2!I9B`D=F%L=64@/7X@+UY005)?
M0TQ%04X]*%Q3*RDO.PI]"GT*<W5B(&]U=',@>PIR971U<FX@:68@)'%U:65T
M.PII9B`H)&QO9V9H*2!["G!R:6YT("1L;V=F:"`B0%]<;B(["GT*96QS92![
M"G!R:6YT(")`7UQN(CL*?0I]"G-U8B!I;FET7VEN8R!["G)E<75I<F4@0V]N
M9FEG.PIP=7-H($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O
M;F9I9WLD7WTL('%W*`IA<F-H;&EB97AP('!R:79L:6)E>'`@<VET96%R8VAE
M>'`@<VET96QI8F5X<`IV96YD;W)A<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]
M"G!A8VMA9V4@;6%I;CL*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.0RP@7"90
M05(Z.F9I;F1?<&%R.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PID:64@<7$H
M<&%R+G!L.B!#86XG="!O<&5N('!E<FP@<V-R:7!T("(D<')O9VYA;64B.B!.
M;R!S=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&XI"G5N;&5S<R`M92`D<')O9VYA
M;64["F1O("1P<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`(#U^+UY?
M5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PI].PHD.CI?7T52
M4D]2(#T@)$`@:68@)$`["GT*0T]213HZ97AI="@D,2D@:68@*"0Z.E]?15)2
M3U(@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I
M9B`D.CI?7T524D]2.PHQ.PI?7T5.1%]?"@`````````````````03P``````
M````````````C"1!``$``````````````)0D00`!``````````````#P1D``
M`0``````````````Z$9```$``````````````'PD00`!``````````````!P
M1D```0``````````````5"-!``$``````````````$=#0SH@*$=.52D@-BXT
M+C``````````````````````1T-#.B`H1TY5*2`V+C0N,```````````````
M``````!'0T,Z("A'3E4I(#8N-"XP`````````````````````$=#0SH@*$=.
M52D@-BXT+C``````````````````````1T-#.B`H1TY5*2`V+C0N,```````
M``````````````!'0T,Z("A'3E4I(#8N-"XP`````````````````````$=#
M0SH@*$=.52D@-BXT+C``````````````````````1T-#.B`H1TY5*2`V+C0N
M,`````````````````````!'0T,Z("A'3E4I(#8N-"XP````````````````
M`````$=#0SH@*$=.52D@-BXT+C``````````````````````1T-#.B`H1TY5
M*2`V+C0N,`````````````````````!'0T,Z("A'3E4I(#8N-"XP````````
M`````````````$=#0SH@*$=.52D@-BXT+C``````````````````````1T-#
M.B`H1TY5*2`V+C0N,`````````````````````!'0T,Z("A'3E4I(#8N-"XP
M`````````````````````````````````0```(PD`0"0KP``0````)0D`0"@
MKP``0````%0C`0#PKP``0````'PD`0#0KP``0```````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````"````&0```!S````<
MC```4E-$4Y[L:`9NI]\`0P(*S[WOPPD!````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````0``!$$```
M`.```%`0``!6$```".```&`0``#)$```$.```-`0``#6$```'.```.`0``!^
M*```).```(`H``!$*0``/.```%`I``#J*0``4.```/`I```G+```8.```#`L
M``"-+0``>.```)`M```4+@``D.```"`N``#Y+@``G.`````O```T+P``L.``
M`$`O``"5+P``N.```*`O``"C+P``P.```+`O``"T,0``Q.```,`Q``#G,0``
MW.```/`Q``"F,@``X.```+`R``!S-```[.```(`T``#L-```_.```/`T```4
M-0``!.$``"`U``#5-0``#.$``.`U```F0@``'.$``#!"``"W10``-.$``,!%
M```S1@``2.$``-!)``#B20``8.$``/!)``#Q20``:.$```!*```!2@``;.$`
M`!!*```12@``<.$``"!*```A2@``=.$``#!*```/3@``>.$``!!.```I3@``
MA.$``#!.``#H3@``B.$``/!.``#U3@``D.$```!/```&3P``E.$``!!/```1
M3P``F.$``&!/``"+3P``G.$``+!/``"X40``5.$``,!1``#*40``I.$`````
M````````````````````````````````````````````````````````````
M`````````````00!``1"```!!`(%!`,!4`$(`P4(,@0#`5````$$`@4$`P%0
M`1,*`!,!5P`,,`M@"G`)4`C`!M`$X`+P`0P'``Q""#`'8`9P!5`$P`+0```!
M"`4`"&($,`-@`G`!4````1`)`!""##`+8`IP"5`(P`;0!.`"\````1,*`!,!
M&0`,,`M@"G`)4`C`!M`$X`+P`08#``9"`C`!8````1,'`!,!!@(+,`I@"7`#
M4`+````!!`$`!$(```$%`@`%4@$P`0````$9"@`9`0L"$3`08`]P#E`-P`O0
M!.`"\`$````!!@,`!D(",`%@```!"`4`",($,`-@`G`!4````04"``52`3`!
M!`$`!$(```$(!0`(@@0P`V`"<`%0```!&0H`&0$W$!$P$&`/<`Y0#<`+T`3@
M`O`!#@@`#E(*,`E@"'`'4`;`!-`"X`$&`P`&0@(P`6````$'!``'D@,P`F`!
M<`$$`0`$0@```0````$````!`````0````$&`P`&0@(P`6````$````!!`$`
M!&(```$````!`````0````$%`@`%,@$P`00"!00#`5``````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````TI'N60````"T$`$``0``
M``X````.````*!`!`&`0`0"8$`$`,$(``/`T``"P3P``&&```,`Q``#`10``
MH"\``/`Q``"`-```L"\``+`R``#@-0``(#4````O``"\$`$`TA`!`.T0`0#R
M$`$`_!`!``D1`0`5$0$`)A$!`#(1`0!`$0$`31$!`%H1`0!G$0$`>1$!````
M`0`"``,`!``%``8`!P`(``D`"@`+``P`#0!P87(N97AE`%A37TEN=&5R;F%L
M<U]005)?0D]/5`!G971?=7-E<FYA;65?9G)O;5]G971P=W5I9`!M86EN`&UY
M7W!A<E]P;`!P87)?8F%S96YA;64`<&%R7V-L96%N=7``<&%R7V-U<G)E;G1?
M97AE8P!P87)?9&ER;F%M90!P87)?96YV7V-L96%N`'!A<E]F:6YD<')O9P!P
M87)?:6YI=%]E;G8`<&%R7VUK=&UP9&ER`'!A<E]S971U<%]L:6)P871H`'AS
M7VEN:70`````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````````&0@
M`0```````````)@J`0#L(@$`["$!````````````K"H!`'0D`0`$(@$`````
M```````@*P$`C"0!`-0B`0```````````#@K`0!<)0$`````````````````
M``````````!T)0$``````(0E`0``````D"4!``````"@)0$``````*PE`0``
M````N"4!``````#()0$``````-0E`0``````X"4!``````#H)0$``````/0E
M`0``````""8!```````<)@$``````"PF`0``````."8!``````!`)@$`````
M`$PF`0``````5"8!``````!@)@$``````&PF`0``````>"8!``````"`)@$`
M`````(@F`0``````E"8!``````"@)@$``````*PF`0``````N"8!``````#`
M)@$``````,@F`0``````U"8!``````#H)@$``````/PF`0``````%"<!````
M```<)P$``````"@G`0``````-"<!``````!`)P$``````$@G`0``````5"<!
M``````!<)P$``````&@G`0``````="<!``````"`)P$``````(PG`0``````
MF"<!``````"D)P$``````+`G`0``````O"<!`````````````````,@G`0``
M````W"<!`````````````````/`G`0```````"@!```````0*`$``````"0H
M`0``````/"@!``````!,*`$``````%PH`0``````="@!``````"$*`$`````
M`)@H`0``````J"@!``````"X*`$``````,PH`0``````Y"@!``````#X*`$`
M``````@I`0``````'"D!```````P*0$``````$`I`0``````5"D!``````!D
M*0$``````'@I`0``````B"D!``````"4*0$``````*0I`0``````````````
M``"P*0$``````,0I`0````````````````!T)0$``````(0E`0``````D"4!
M``````"@)0$``````*PE`0``````N"4!``````#()0$``````-0E`0``````
MX"4!``````#H)0$``````/0E`0``````""8!```````<)@$``````"PF`0``
M````."8!``````!`)@$``````$PF`0``````5"8!``````!@)@$``````&PF
M`0``````>"8!``````"`)@$``````(@F`0``````E"8!``````"@)@$`````
M`*PF`0``````N"8!``````#`)@$``````,@F`0``````U"8!``````#H)@$`
M`````/PF`0``````%"<!```````<)P$``````"@G`0``````-"<!``````!`
M)P$``````$@G`0``````5"<!``````!<)P$``````&@G`0``````="<!````
M``"`)P$``````(PG`0``````F"<!``````"D)P$``````+`G`0``````O"<!
M`````````````````,@G`0``````W"<!`````````````````/`G`0``````
M`"@!```````0*`$``````"0H`0``````/"@!``````!,*`$``````%PH`0``
M````="@!``````"$*`$``````)@H`0``````J"@!``````"X*`$``````,PH
M`0``````Y"@!``````#X*`$```````@I`0``````'"D!```````P*0$`````
M`$`I`0``````5"D!``````!D*0$``````'@I`0``````B"D!``````"4*0$`
M`````*0I`0````````````````"P*0$``````,0I`0`````````````````.
M`%]?8WAA7V%T97AI=```%0!?7V5R<FYO````(@!?7V=E=')E96YT`````"P`
M7U]M86EN`````%@`7V1L;%]C<G0P`%T`7VEM<'5R95]P='(```!M`&%C8V5S
M<P````#D`&-A;&QO8P`````A`6-L;W-E`"(!8VQO<V5D:7(``%0!8WEG=VEN
M7V1E=&%C:%]D;&P`5@%C>6=W:6Y?:6YT97)N86P```!H`61L;%]D;&QC<G0P
M````A0%E;G9I<F]N````H@%E>&ET```,`F9P<FEN=&8````8`F9R964``'X"
M9V5T<&ED`````(D"9V5T<'=U:60``)<"9V5T=6ED`````%H#;'-E96L`7`-L
M<W1A=`!A`VUA;&QO8P````!Q`VUE;6-H<@````!S`VUE;6-P>0````!U`VUE
M;6UO=F4```!Y`VUK9&ER`+4#;W!E;@``N0-O<&5N9&ER````S`-P;W-I>%]M
M96UA;&EG;@````#Q`W!T:')E861?871F;W)K`````"4$<'1H<F5A9%]G971S
M<&5C:69I8P```'<$<F5A9```>`1R96%D9&ER````>@1R96%D;&EN:P``?01R
M96%L;&]C````JP1R;61I<@`X!7-P<FEN=&8```!`!7-T870``$H%<W1R8VAR
M`````$P%<W1R8VUP`````%$%<W1R9'5P`````%L%<W1R;&5N`````&`%<W1R
M;F-M<````&$%<W1R;F-P>0```&P%<W1R<W1R`````'(%<W1R=&]K`````-,%
M=6YL:6YK``````L`7U]S=&%C:U]C:&M?9F%I;```#0!?7W-T86-K7V-H:U]G
M=6%R9``1`%!,7V1O7W5N9'5M<```60!03%]T:')?:V5Y`````%<!4&5R;%]A
M=&9O<FM?;&]C:P``6`%097)L7V%T9F]R:U]U;FQO8VL`````80%097)L7V%V
M7V9E=&-H`&D!4&5R;%]A=E]S:&EF=`#)`5!E<FQ?8W)O86M?;F]C;VYT97AT
M``!2`E!E<FQ?9V5T7W-V````@`)097)L7V=V7V9E=&-H<'8```!K`U!E<FQ?
M;6=?<V5T````VP-097)L7VYE=UA3`````-T#4&5R;%]N97=84U]F;&%G<P``
MH`5097)L7W-A9F5S>7-M86QL;V,`````_P5097)L7W-V7S)P=E]F;&%G<P!Q
M!E!E<FQ?<W9?<V5T<'8`>@9097)L7W-V7W-E='!V;@````"E!E!E<FQ?<WES
M7VEN:70S`````*8&4&5R;%]S>7-?=&5R;0`A!V)O;W1?1'EN84QO861E<@``
M`"P'<&5R;%]A;&QO8P`````N!W!E<FQ?8V]N<W1R=6-T`````"\'<&5R;%]D
M97-T<G5C=``P!W!E<FQ?9G)E90`Q!W!E<FQ?<&%R<V4`````,@=P97)L7W)U
M;@``)0)'971-;V1U;&5(86YD;&5!``!7`D=E=%!R;V-!9&1R97-S```````@
M`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!
M```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$`
M`"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``
M(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@
M`0``(`$``"`!`&-Y9W=I;C$N9&QL`!0@`0`4(`$`8WEG<W-P+3`N9&QL````
M`"@@`0`H(`$`*"`!`"@@`0`H(`$`*"`!`"@@`0`H(`$`*"`!`"@@`0`H(`$`
M*"`!`"@@`0`H(`$`*"`!`"@@`0`H(`$`*"`!`"@@`0`H(`$`*"`!`"@@`0`H
M(`$`*"`!`"@@`0!C>6=P97)L-5\R-BYD;&P`/"`!`#P@`0!+15).14PS,BYD
M;&P`````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````!``#````,```@`X```#```"`$```
M`/```(`8````(`$`@```````````````````!``!````8```@`(```!X``"`
M`P```)```(`$````J```@````````````````````0`)!```4`$`````````
M`````````````0`)!```8`$``````````````````````0`)!```<`$`````
M`````````````````0`)!```@`$```````````````````$```#``0"`V```
M@````````````````````0`)!```D`$``````````````````````0`!````
M"`$`@````````````````````0`)!```H`$``````````````````````0`!
M````.`$`@````````````````````0`)!```L`$``-`Q`0!H!0``````````
M```X-P$`J`@`````````````X#\!`&@%`````````````$A%`0"H"```````
M``````#P30$`/@``````````````,$X!`%@"`````````````(A0`0`^`P``
M```````````&`%<`20!.`$4`6`!%````*````!`````@`````0`(````````
M`0```````````````0````$```8&!@`$!`0``@("``````!"0D(`G9V=`&YN
M;@"!@8$`L;&Q`/___P!S<W,`GY^?`%A86`"RLK(`_?W]`&MK:P#\_/P`U]?7
M`!86%@#:VMH`EI:6`(Z.C@!&1D8`R\O+``L+"P"@H*``_O[^`,G)R0#!P<$`
M`P,#`!$1$0"3DY,`L[.S`.[N[@#T]/0`X^/C`.7EY0"BHJ(`NKJZ`*^OKP`'
M!P<`<7%Q`+6UM0#U]?4`B8F)`(J*B@`T-#0`Z.CH`#4U-0!:6EH`M[>W`-C8
MV`#0T-``'1T=``P,#`"/CX\`OKZ^`&UM;0`)"0D`FYN;`"\O+P#-S<T`V=G9
M`!D9&0!-34T`1$1$`%!04``\/#P`9&1D`,+"P@`R,C(`"`@(``T-#0`/#P\`
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````!04%!04%!04%!04%!04$!`0$U0",%03%#04%$12X!`0$!`0$!
M!0$\%Q<!`1,!`0$!`0$!`14!`285`0$/`0$!`0$!`0$3`1$!!34!)@$!`0$!
M`0$!(A(0,0\%`2`!`0$!`0$!`1`*("(5%2XA+@$!`0$!'R`A$"(0(R,E)B<!
M`0$!)1`0$!`0$!`0$!=%`0$!01`0$!`0$!`0$!`0%P$!`0\0$!40$!`0$!`0
M$`$!`0$2$!`3'Q`0$!`0$`<!`0$!(!`0$`$/$!`0$!$!`0$!`0<@$`H!`040
M$!`Q`0$!`0$!`0$!`0$!004/`0$!`0$!`0$!`0$!`0$!`0$!`0$!@`$``,`!
M``#@`P``\`,``/`#``#P`P``X`,``,`!``"``0``@`$``(`#``"``P````<`
M```'``".#P``_Q\``"@````@````0`````$`"`````````0`````````````
M``$````!`````````0$!``("`@`#`P,`!`0$``4%!0`&!@8`!P<'``@("``)
M"0D`"@H*``L+"P`,#`P`#0T-``X.#@`/#P\`$!`0`!$1$0`2$A(`$Q,3`!04
M%``5%14`%A86`!<7%P`8&!@`&1D9`!H:&@`;&QL`'!P<`!T='0`>'AX`'Q\?
M`"`@(``A(2$`(B(B`",C(P`D)"0`)24E`"8F)@`G)R<`*"@H`"DI*0`J*BH`
M*RLK`"PL+``M+2T`+BXN`"\O+P`P,#``,3$Q`#(R,@`S,S,`-#0T`#4U-0`V
M-C8`-S<W`#@X.``Y.3D`.CHZ`#L[.P`\/#P`/3T]`#X^/@`_/S\`0$!``$%!
M00!"0D(`0T-#`$1$1`!%144`1D9&`$='1P!(2$@`24E)`$I*2@!+2TL`3$Q,
M`$U-30!.3DX`3T]/`%!04`!145$`4E)2`%-34P!45%0`5555`%965@!75U<`
M6%A8`%E960!:6EH`6UM;`%Q<7`!=75T`7EY>`%]?7P!@8&``86%A`&)B8@!C
M8V,`9&1D`&5E90!F9F8`9V=G`&AH:`!I:6D`:FIJ`&MK:P!L;&P`;6UM`&YN
M;@!O;V\`<'!P`'%Q<0!R<G(`<W-S`'1T=`!U=74`=G9V`'=W=P!X>'@`>7EY
M`'IZ>@![>WL`?'Q\`'U]?0!^?GX`?W]_`("`@`"!@8$`@H*"`(.#@P"$A(0`
MA86%`(:&A@"'AX<`B(B(`(F)B0"*BHH`BXN+`(R,C`"-C8T`CHZ.`(^/CP"0
MD)``D9&1`)*2D@"3DY,`E)24`)65E0"6EI8`EY>7`)B8F`"9F9D`FIJ:`)N;
MFP"<G)P`G9V=`)Z>G@"?GY\`H*"@`*&AH0"BHJ(`HZ.C`*2DI`"EI:4`IJ:F
M`*>GIP"HJ*@`J:FI`*JJJ@"KJZL`K*RL`*VMK0"NKJX`KZ^O`+"PL`"QL;$`
MLK*R`+.SLP"TM+0`M;6U`+:VM@"WM[<`N+BX`+FYN0"ZNKH`N[N[`+R\O`"]
MO;T`OKZ^`+^_OP#`P,``P<'!`,+"P@##P\,`Q,3$`,7%Q0#&QL8`Q\?'`,C(
MR`#)R<D`RLK*`,O+RP#,S,P`S<W-`,[.S@#/S\\`T-#0`-'1T0#2TM(`T]/3
M`-34U`#5U=4`UM;6`-?7UP#8V-@`V=G9`-K:V@#;V]L`W-S<`-W=W0#>WMX`
MW]_?`.#@X`#AX>$`XN+B`./CXP#DY.0`Y>7E`.;FY@#GY^<`Z.CH`.GIZ0#J
MZNH`Z^OK`.SL[`#M[>T`[N[N`._O[P#P\/``\?'Q`/+R\@#S\_,`]/3T`/7U
M]0#V]O8`]_?W`/CX^`#Y^?D`^OKZ`/O[^P#\_/P`_?W]`/[^_@#___\`-#0T
M/#P\/#P\/#P\/#P\/#P\/#P\/#P\+`P$!`0$!`1<M-34U-34U-34U-3<U-34
MU-34U-34U-2T-%$$!`0$!&3,[/3T]/ST_/3T]/ST_/3\]/S\_/3T[,0T45$$
M!`0$9,3D[/3T]/3T]/3T]/3T]/3T]/3T].SLO#1145$$!`1DQ.3L[.SD[.SL
MY.3DY.SLY.SD[.SL[.2\-%%145$$!&2\Y.SLK(RLU,24E)R4C(R4E)2<G)SL
MW+PT45%145$$9+S<Y.SDQ%&4K&3DI$PT+%S$S%&TU.3<O#1145%145%DO-SD
MY.SDY%&$=.2,;*2<;(R\=*S4Y-R\-`0$!`0$!&2\W.3D[.3DO%&,U-2,K,QT
MS-RD4<3DW+PT!`0$!`0$9+S<Y.3DY.3<C'R<M'R$G&S,Y+QTQ.3<M#0$!`0$
M!`1<O-S<Y.3DY-2<I%R4C,P\E&3,O(Q1W-2T-`0$!`0$!&2TU-S<W.3<C(RL
M++RTS%K$9)QLE*3<U+0T!`0$!`0$7+34W-SDW-QTE-1,I/2,/-1TE'2<3-S,
MM#0$!`0$!`1DM-3<W-SDW'11]'2$[,QDY'QLI+1LI-2T-`0$!`0$!&2TS-S<
MS'R,A+3\S,3\]+3\O&3,K(11O*PT!`0$!`0$7+3,U,R,I-S<[/S\_/S\]/SL
MM.RL?)11K#0$!`0$!`1<K,S,;)ST_/S\_/S\_/S\_/S\_,24C%&L-`0$!`0$
M!%RLQ+Q\Y/S\_/S\_/S\_/S\_/S\[,RL4:PT!`0$!`0$7*RT3,3\_/SL]/S\
M_/S\_/S\_/STO%'$K#0$!`0$!`1<K+1,U/S\]+3<_/S\_/S\_/S\_,Q1S,2D
M-`0$!`0$!%RLQ*11_/S\M)ST_/S\_/S\_/STE%',Q*0L!`0$!`0$7*2D483L
M_/S,9+3L_/S\_/S\_-1LK,R\I#0$!`0$!`1<I&1TO/3\_,11A)SD_/S\_/SD
MC%',S,2D-`0$!`0$!%RD;(SD]/S\G%',493L_/S\_+QTS,S$O)PL!`0$!`0$
M7*2L9%QDI*11K,S$=)3D]/SD4;S,S,2\G#0$!`0$!`1<I+2\O(Q14<3$S,RL
M47Q\E&Q\O,S$Q+R<+`0$!`0$!%R<O,3$Q,3$Q,3$Q,3$45%14;S$Q,3$M)PT
M!`0$!`0$5)RTQ,3$Q,3$Q,3$Q,3$Q,3$Q,3$Q+RTG"P$!`0$!`14G+S$Q+R\
MO+R\O+R\O+R\O+R\O+R\O+2<+`0$!`0$!%R<M,2\M+2TM+2TM+2TM+2TM+2T
MM+2TK)PL!`0$!`0$5(24G)R4E)24E)24E)24E)24E)24E)24A"0$!`0$!`1$
M5%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q4-`0$!`0$!````#\````?````#P``
M``<````#`````0`````````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\````_````/P```#\````_*````!`````@`````0`(````````
M`0```````````````0````$```8&!@`$!`0``@("``````!"0D(`G9V=`&YN
M;@"!@8$`L;&Q`/___P!S<W,`GY^?`%A86`"RLK(`_?W]`&MK:P#\_/P`U]?7
M`!86%@#:VMH`EI:6`(Z.C@!&1D8`R\O+``L+"P"@H*``_O[^`,G)R0#!P<$`
M`P,#`!$1$0"3DY,`L[.S`.[N[@#T]/0`X^/C`.7EY0"BHJ(`NKJZ`*^OKP`'
M!P<`<7%Q`+6UM0#U]?4`B8F)`(J*B@`T-#0`Z.CH`#4U-0!:6EH`M[>W`-C8
MV`#0T-``'1T=``P,#`"/CX\`OKZ^`&UM;0`)"0D`FYN;`"\O+P#-S<T`V=G9
M`!D9&0!-34T`1$1$`%!04``\/#P`9&1D`,+"P@`R,C(`"`@(``T-#0`/#P\`
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````!04%!04%!04%!04%!04$!`0$U0",%03%#04%$12X!`0$!`0$!
M!0$\%Q<!`1,!`0$!`0$!`14!`285`0$/`0$!`0$!`0$3`1$!!34!)@$!`0$!
M`0$!(A(0,0\%`2`!`0$!`0$!`1`*("(5%2XA+@$!`0$!'R`A$"(0(R,E)B<!
M`0$!)1`0$!`0$!`0$!=%`0$!01`0$!`0$!`0$!`0%P$!`0\0$!40$!`0$!`0
M$`$!`0$2$!`3'Q`0$!`0$`<!`0$!(!`0$`$/$!`0$!$!`0$!`0<@$`H!`040
M$!`Q`0$!`0$!`0$!`0$!004/`0$!`0$!`0$!`0$!`0$!`0$!`0$!@`$``,`!
M``#@`P``\`,``/`#``#P`P``X`,``,`!``"``0``@`$``(`#``"``P````<`
M```'``".#P``_Q\``"@````@````0`````$`"`````````0`````````````
M``$````!`````````0$!``("`@`#`P,`!`0$``4%!0`&!@8`!P<'``@("``)
M"0D`"@H*``L+"P`,#`P`#0T-``X.#@`/#P\`$!`0`!$1$0`2$A(`$Q,3`!04
M%``5%14`%A86`!<7%P`8&!@`&1D9`!H:&@`;&QL`'!P<`!T='0`>'AX`'Q\?
M`"`@(``A(2$`(B(B`",C(P`D)"0`)24E`"8F)@`G)R<`*"@H`"DI*0`J*BH`
M*RLK`"PL+``M+2T`+BXN`"\O+P`P,#``,3$Q`#(R,@`S,S,`-#0T`#4U-0`V
M-C8`-S<W`#@X.``Y.3D`.CHZ`#L[.P`\/#P`/3T]`#X^/@`_/S\`0$!``$%!
M00!"0D(`0T-#`$1$1`!%144`1D9&`$='1P!(2$@`24E)`$I*2@!+2TL`3$Q,
M`$U-30!.3DX`3T]/`%!04`!145$`4E)2`%-34P!45%0`5555`%965@!75U<`
M6%A8`%E960!:6EH`6UM;`%Q<7`!=75T`7EY>`%]?7P!@8&``86%A`&)B8@!C
M8V,`9&1D`&5E90!F9F8`9V=G`&AH:`!I:6D`:FIJ`&MK:P!L;&P`;6UM`&YN
M;@!O;V\`<'!P`'%Q<0!R<G(`<W-S`'1T=`!U=74`=G9V`'=W=P!X>'@`>7EY
M`'IZ>@![>WL`?'Q\`'U]?0!^?GX`?W]_`("`@`"!@8$`@H*"`(.#@P"$A(0`
MA86%`(:&A@"'AX<`B(B(`(F)B0"*BHH`BXN+`(R,C`"-C8T`CHZ.`(^/CP"0
MD)``D9&1`)*2D@"3DY,`E)24`)65E0"6EI8`EY>7`)B8F`"9F9D`FIJ:`)N;
MFP"<G)P`G9V=`)Z>G@"?GY\`H*"@`*&AH0"BHJ(`HZ.C`*2DI`"EI:4`IJ:F
M`*>GIP"HJ*@`J:FI`*JJJ@"KJZL`K*RL`*VMK0"NKJX`KZ^O`+"PL`"QL;$`
MLK*R`+.SLP"TM+0`M;6U`+:VM@"WM[<`N+BX`+FYN0"ZNKH`N[N[`+R\O`"]
MO;T`OKZ^`+^_OP#`P,``P<'!`,+"P@##P\,`Q,3$`,7%Q0#&QL8`Q\?'`,C(
MR`#)R<D`RLK*`,O+RP#,S,P`S<W-`,[.S@#/S\\`T-#0`-'1T0#2TM(`T]/3
M`-34U`#5U=4`UM;6`-?7UP#8V-@`V=G9`-K:V@#;V]L`W-S<`-W=W0#>WMX`
MW]_?`.#@X`#AX>$`XN+B`./CXP#DY.0`Y>7E`.;FY@#GY^<`Z.CH`.GIZ0#J
MZNH`Z^OK`.SL[`#M[>T`[N[N`._O[P#P\/``\?'Q`/+R\@#S\_,`]/3T`/7U
M]0#V]O8`]_?W`/CX^`#Y^?D`^OKZ`/O[^P#\_/P`_?W]`/[^_@#___\`-#0T
M/#P\/#P\/#P\/#P\/#P\/#P\/#P\+`P$!`0$!`1<M-34U-34U-34U-3<U-34
MU-34U-34U-2T-%$$!`0$!&3,[/3T]/ST_/3T]/ST_/3\]/S\_/3T[,0T45$$
M!`0$9,3D[/3T]/3T]/3T]/3T]/3T]/3T].SLO#1145$$!`1DQ.3L[.SD[.SL
MY.3DY.SLY.SD[.SL[.2\-%%145$$!&2\Y.SLK(RLU,24E)R4C(R4E)2<G)SL
MW+PT45%145$$9+S<Y.SDQ%&4K&3DI$PT+%S$S%&TU.3<O#1145%145%DO-SD
MY.SDY%&$=.2,;*2<;(R\=*S4Y-R\-`0$!`0$!&2\W.3D[.3DO%&,U-2,K,QT
MS-RD4<3DW+PT!`0$!`0$9+S<Y.3DY.3<C'R<M'R$G&S,Y+QTQ.3<M#0$!`0$
M!`1<O-S<Y.3DY-2<I%R4C,P\E&3,O(Q1W-2T-`0$!`0$!&2TU-S<W.3<C(RL
M++RTS%K$9)QLE*3<U+0T!`0$!`0$7+34W-SDW-QTE-1,I/2,/-1TE'2<3-S,
MM#0$!`0$!`1DM-3<W-SDW'11]'2$[,QDY'QLI+1LI-2T-`0$!`0$!&2TS-S<
MS'R,A+3\S,3\]+3\O&3,K(11O*PT!`0$!`0$7+3,U,R,I-S<[/S\_/S\]/SL
MM.RL?)11K#0$!`0$!`1<K,S,;)ST_/S\_/S\_/S\_/S\_,24C%&L-`0$!`0$
M!%RLQ+Q\Y/S\_/S\_/S\_/S\_/S\[,RL4:PT!`0$!`0$7*RT3,3\_/SL]/S\
M_/S\_/S\_/STO%'$K#0$!`0$!`1<K+1,U/S\]+3<_/S\_/S\_/S\_,Q1S,2D
M-`0$!`0$!%RLQ*11_/S\M)ST_/S\_/S\_/STE%',Q*0L!`0$!`0$7*2D483L
M_/S,9+3L_/S\_/S\_-1LK,R\I#0$!`0$!`1<I&1TO/3\_,11A)SD_/S\_/SD
MC%',S,2D-`0$!`0$!%RD;(SD]/S\G%',493L_/S\_+QTS,S$O)PL!`0$!`0$
M7*2L9%QDI*11K,S$=)3D]/SD4;S,S,2\G#0$!`0$!`1<I+2\O(Q14<3$S,RL
M47Q\E&Q\O,S$Q+R<+`0$!`0$!%R<O,3$Q,3$Q,3$Q,3$45%14;S$Q,3$M)PT
M!`0$!`0$5)RTQ,3$Q,3$Q,3$Q,3$Q,3$Q,3$Q+RTG"P$!`0$!`14G+S$Q+R\
MO+R\O+R\O+R\O+R\O+R\O+2<+`0$!`0$!%R<M,2\M+2TM+2TM+2TM+2TM+2T
MM+2TK)PL!`0$!`0$5(24G)R4E)24E)24E)24E)24E)24E)24A"0$!`0$!`1$
M5%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q4-`0$!`0$!````#\````?````#P``
M``<````#`````0`````````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\````_````/P```#\````_```!``0`$!````$`"`!H!0```0`@
M(````0`(`*@(```"`!`0```!``@`:`4```,`("````$`"`"H"```!````%@"
M-````%8`4P!?`%8`10!2`%,`20!/`$X`7P!)`$X`1@!/``````"]!._^```!
M```````````````````````_``````````0`!``!````````````````````
MN`$```$`4P!T`'(`:0!N`&<`1@!I`&P`90!)`&X`9@!O````E`$```$`,``P
M`#``,``P`#0`0@`P````)``"``$`0P!O`&T`<`!A`&X`>0!.`&$`;0!E````
M```@````+``"``$`1@!I`&P`90!$`&4`<P!C`'(`:0!P`'0`:0!O`&X`````
M`"`````P``@``0!&`&D`;`!E`%8`90!R`',`:0!O`&X``````#``+@`P`"X`
M,``N`#`````D``(``0!)`&X`=`!E`'(`;@!A`&P`3@!A`&T`90```"`````H
M``(``0!,`&4`9P!A`&P`0P!O`'``>0!R`&D`9P!H`'0````@````+``"``$`
M3`!E`&<`80!L`%0`<@!A`&0`90!M`&$`<@!K`',``````"`````L``(``0!/
M`'(`:0!G`&D`;@!A`&P`1@!I`&P`90!N`&$`;0!E````(````"0``@`!`%``
M<@!O`&0`=0!C`'0`3@!A`&T`90``````(````#0`"``!`%``<@!O`&0`=0!C
M`'0`5@!E`'(`<P!I`&\`;@```#``+@`P`"X`,``N`#````!$`````0!6`&$`
M<@!&`&D`;`!E`$D`;@!F`&\``````"0`!````%0`<@!A`&X`<P!L`&$`=`!I
M`&\`;@````````"P!#P_>&UL('9E<G-I;VX](C$N,"(@96YC;V1I;F<](E54
M1BTX(B!S=&%N9&%L;VYE/2)Y97,B/SX-"CQA<W-E;6)L>2!X;6QN<STB=7)N
M.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IA<VTN=C$B(&UA;FEF97-T5F5R<VEO
M;CTB,2XP(CX-"B`@("`\87-S96UB;'E)9&5N=&ET>2!V97)S:6]N/2(P+C`N
M,"XP(B!T>7!E/2)W:6XS,B(@;F%M93TB4$%2+5!A8VME<BY!<'!L:6-A=&EO
M;B(@+SX-"B`@("`\9&5S8W)I<'1I;VX^4$%2+5!A8VME<B!!<'!L:6-A=&EO
M;CPO9&5S8W)I<'1I;VX^#0H@("`@/&1E<&5N9&5N8WD^#0H@("`@("`@(#QD
M97!E;F1E;G1!<W-E;6)L>3X-"B`@("`@("`@("`@(#QA<W-E;6)L>4ED96YT
M:71Y('1Y<&4](G=I;C,R(B!N86UE/2)-:6-R;W-O9G0N5VEN9&]W<RY#;VUM
M;VXM0V]N=')O;',B('9E<G-I;VX](C8N,"XP+C`B('!U8FQI8TME>51O:V5N
M/2(V-3DU8C8T,30T8V-F,61F(B!L86YG=6%G93TB*B(@<')O8V5S<V]R07)C
M:&ET96-T=7)E/2(J(B`O/@T*("`@("`@("`\+V1E<&5N9&5N=$%S<V5M8FQY
M/@T*("`@(#PO9&5P96YD96YC>3X-"B`@("`\=')U<W1);F9O('AM;&YS/2)U
M<FXZ<V-H96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV,R(^#0H@("`@("`@(#QS
M96-U<FET>3X-"B`@("`@("`@("`@(#QR97%U97-T9610<FEV:6QE9V5S/@T*
M("`@("`@("`@("`@("`@(#QR97%U97-T961%>&5C=71I;VY,979E;"!L979E
M;#TB87-);G9O:V5R(B!U:4%C8V5S<STB9F%L<V4B("\^#0H@("`@("`@("`@
M("`\+W)E<75E<W1E9%!R:79I;&5G97,^#0H@("`@("`@(#PO<V5C=7)I='D^
M#0H@("`@/"]T<G5S=$EN9F\^#0H\+V%S<V5M8FQY/@T*````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````!0```,````V*$```!@```,````&*````"@```8````D*^@K["O
MP*_0K^"O\*\`````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
8````````````````````````````````
