/* Library libcerf:
 *   A self-contained numeric library that provides an efficient and accurate implementation
 *   of complex error functions, along with Dawson, Faddeeva, and Voigt functions.
 *
 * File cerf_real.h:
 *   Declare exported functions with real argument(s) and real return value.
 *
 * Copyright:
 *   (C) 2012 Massachusetts Institute of Technology
 *   (C) 2013 Forschungszentrum Jülich GmbH
 *
 * Licence:
 *   Permission is hereby granted, free of charge, to any person obtaining
 *   a copy of this software and associated documentation files (the
 *   "Software"), to deal in the Software without restriction, including
 *   without limitation the rights to use, copy, modify, merge, publish,
 *   distribute, sublicense, and/or sell copies of the Software, and to
 *   permit persons to whom the Software is furnished to do so, subject to
 *   the following conditions:
 *
 *   The above copyright notice and this permission notice shall be
 *   included in all copies or substantial portions of the Software.
 *
 *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *   NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *   LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *   OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *   WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *   Steven G. Johnson, Massachusetts Institute of Technology, 2012, core author
 *   Joachim Wuttke, Forschungszentrum Jülich, 2013, package maintainer
 *
 * Website:
 *   http://apps.jcns.fz-juelich.de/libcerf
 *
 * Revision history:
 *   ../CHANGELOG
 *
 * Man pages:
 *   w_of_z(3), dawson(3), voigt(3), cerf(3), erfcx(3), erfi(3)
 */

#ifndef CERF_CERFREAL_H
#define CERF_CERFREAL_H

#if _WIN32
#define EXPORT __declspec(dllexport)
#else
#define EXPORT
#endif

//! compute Re exp(-z^2) erfc(-iz), z=x+iy, Faddeeva's scaled complex error function
EXPORT double re_w_of_z(double x, double y);

//! compute Im exp(-z^2) erfc(-iz), z=x+iy, Faddeeva's scaled complex error function
EXPORT double im_w_of_z(double x, double y);

//! compute Im exp(-x^2) erfc(-ix), Faddeeva's scaled complex error function
EXPORT double im_w_of_x(double x);

//! compute erfcx(x) = exp(x^2) erfc(x), an underflow-compensated version of erfc
EXPORT double erfcx(double x);

//! compute erfi(x) = -i erf(ix), the imaginary error function
EXPORT double erfi (double x);

//! compute dawson(x) = sqrt(pi)/2 * exp(-x^2) * erfi(x), Dawson's integral
EXPORT double dawson(double x);

//! compute voigt(x,...), the convolution of a Gaussian and a Lorentzian
EXPORT double voigt(double x, double sigma, double gamma);

//! compute the full width at half maximum of the Voigt function
EXPORT double voigt_hwhm(double sigma, double gamma);

// EXPERIMENTAL
EXPORT double cerf_experimental_imw(double x, double y);
EXPORT double cerf_experimental_rew(double x, double y);

#endif // CERF_CERFREAL_H
