/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Setmiss    setmissval      Set a new missing value
      Setmiss    setctomiss      Set constant to missing value
      Setmiss    setmisstoc      Set missing value to constant
      Setmiss    setrtomiss      Set range to missing value
      Setmiss    setvrange       Set range of valid value
*/

#include <cmath>
#include <cdi.h>

#include "process_int.h"
#include "param_conversion.h"

template <typename T>
static size_t
set_missval(size_t gridsize, Varray<T> &array, T missval, T new_missval)
{
  size_t nmiss = 0;
  for (size_t i = 0; i < gridsize; i++)
    if (DBL_IS_EQUAL(array[i], missval) || DBL_IS_EQUAL(array[i], (float) missval) || DBL_IS_EQUAL(array[i], new_missval)
        || DBL_IS_EQUAL(array[i], (float) new_missval))
      {
        array[i] = new_missval;
        nmiss++;
      }

  return nmiss;
}

static void
set_missval(Field &field, double new_missval)
{
  if (field.memType == MemType::Float)
    field.nmiss = set_missval(field.size, field.vec_f, (float)field.missval, (float)new_missval);
  else
    field.nmiss = set_missval(field.size, field.vec_d, field.missval, new_missval);
}

template <typename T>
static size_t
set_const_to_miss(size_t gridsize, Varray<T> &array, T missval, T rconst)
{
  size_t nmiss = 0;
  if (std::isnan(rconst))
    {
      for (size_t i = 0; i < gridsize; i++)
        if (std::isnan(array[i]))
          {
            array[i] = missval;
            nmiss++;
          }
    }
  else
    {
      for (size_t i = 0; i < gridsize; i++)
        if (DBL_IS_EQUAL(array[i], rconst) || DBL_IS_EQUAL(array[i], (float) rconst))
          {
            array[i] = missval;
            nmiss++;
          }
    }

  return nmiss;
}

static void
set_const_to_miss(Field &field, double rconst)
{
  if (field.memType == MemType::Float)
    field.nmiss += set_const_to_miss(field.size, field.vec_f, (float)field.missval, (float)rconst);
  else
    field.nmiss += set_const_to_miss(field.size, field.vec_d, field.missval, rconst);
}

template <typename T>
static size_t
set_miss_to_const(size_t gridsize, Varray<T> &array, T missval, T rconst)
{
  for (size_t i = 0; i < gridsize; i++)
    if (DBL_IS_EQUAL(array[i], missval) || DBL_IS_EQUAL(array[i], (float) missval))
      {
        array[i] = rconst;
      }

  return 0;
}

static void
set_miss_to_const(Field &field, double rconst)
{
  if (field.memType == MemType::Float)
    field.nmiss = set_miss_to_const(field.size, field.vec_f, (float)field.missval, (float)rconst);
  else
    field.nmiss = set_miss_to_const(field.size, field.vec_d, field.missval, rconst);
}

template <typename T>
static size_t
set_range_to_miss(size_t gridsize, Varray<T> &array, T missval, T rmin, T rmax)
{
  size_t nmiss = 0;
  for (size_t i = 0; i < gridsize; i++)
    if (array[i] >= rmin && array[i] <= rmax)
      {
        array[i] = missval;
        nmiss++;
      }

  return nmiss;
}

static void
set_range_to_miss(Field &field, double rmin, double rmax)
{
  if (field.memType == MemType::Float)
    field.nmiss += set_range_to_miss(field.size, field.vec_f, (float)field.missval, (float)rmin, (float)rmax);
  else
    field.nmiss += set_range_to_miss(field.size, field.vec_d, field.missval, rmin, rmax);
}

template <typename T>
static size_t
set_valid_range(size_t gridsize, Varray<T> &array, T missval, T rmin, T rmax)
{
  for (size_t i = 0; i < gridsize; i++)
    if (array[i] < rmin || array[i] > rmax) array[i] = missval;

  const auto nmiss = varrayNumMV(gridsize, array, missval);

  return nmiss;
}

static void
set_valid_range(Field &field, double rmin, double rmax)
{
  if (field.memType == MemType::Float)
    field.nmiss = set_valid_range(field.size, field.vec_f, (float)field.missval, (float)rmin, (float)rmax);
  else
    field.nmiss = set_valid_range(field.size, field.vec_d, field.missval, rmin, rmax);
}

void *
Setmiss(void *process)
{
  cdoInitialize(process);

  // clang-format off
  const auto SETMISSVAL = cdoOperatorAdd("setmissval", 0, 0, "missing value");
  const auto SETCTOMISS = cdoOperatorAdd("setctomiss", 0, 0, "constant");
  const auto SETMISSTOC = cdoOperatorAdd("setmisstoc", 0, 0, "constant");
  const auto SETRTOMISS = cdoOperatorAdd("setrtomiss", 0, 0, "range (min, max)");
  const auto SETVRANGE  = cdoOperatorAdd("setvrange",  0, 0, "range (min, max)");
  // clang-format on

  const auto operatorID = cdoOperatorID();

  double new_missval = 0.0;
  double rconst = 0.0, rmin = 0.0, rmax = 0.0;

  if (operatorID == SETMISSVAL)
    {
      operatorCheckArgc(1);
      new_missval = parameter2double(cdoOperatorArgv(0));
    }
  else if (operatorID == SETCTOMISS || operatorID == SETMISSTOC)
    {
      operatorCheckArgc(1);
      rconst = parameter2double(cdoOperatorArgv(0));
    }
  else
    {
      operatorCheckArgc(2);
      rmin = parameter2double(cdoOperatorArgv(0));
      rmax = parameter2double(cdoOperatorArgv(1));
    }

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  VarList varList;
  varListInit(varList, vlistID1);

  Field field;

  if (operatorID == SETMISSVAL)
    {
      const auto nvars = vlistNvars(vlistID2);
      for (int varID = 0; varID < nvars; varID++) vlistDefVarMissval(vlistID2, varID, new_missval);
    }
  else if (operatorID == SETMISSTOC)
    {
      const auto nvars = vlistNvars(vlistID2);
      for (int varID = 0; varID < nvars; varID++)
        {
          const auto missval = varList[varID].missval;
          if (DBL_IS_EQUAL(rconst, missval))
            {
              cdoWarning("Missing value and constant have the same value!");
              break;
            }
        }
    }

  /*
  if (operatorID == SETVRANGE)
    {
      double range[2] = {rmin, rmax};
      nvars = vlistNvars(vlistID2);
      for (varID = 0; varID < nvars; varID++)
        cdiDefAttFlt(vlistID2, varID, "valid_range", CDI_DATATYPE_FLT64, 2, range);
    }
  */
  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdoInqRecord(streamID1, &varID, &levelID);
          field.init(varList[varID]);
          cdoReadRecord(streamID1, field);

          // clang-format off
          if      (operatorID == SETMISSVAL) set_missval(field, new_missval);
          else if (operatorID == SETCTOMISS) set_const_to_miss(field, rconst);
          else if (operatorID == SETMISSTOC) set_miss_to_const(field, rconst);
          else if (operatorID == SETRTOMISS) set_range_to_miss(field, rmin, rmax);
          else if (operatorID == SETVRANGE)  set_valid_range(field, rmin, rmax);
          // clang-format on

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, field);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
