/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "process_int.h"
#include "cdo_zaxis.h"
#include "param_conversion.h"
#include "cdi_lockedIO.h"

void *
Seloperator(void *process)
{
  int varID, levelID;
  bool selfound = false;

  cdoInitialize(process);

  const auto lcopy = unchangedRecord();

  operatorInputArg("code, ltype, level");

  const auto scode = parameter2int(cdoOperatorArgv(0));
  const auto sltype = parameter2int(cdoOperatorArgv(1));

  const auto slevel = (operatorArgc() == 3) ? parameter2double(cdoOperatorArgv(2)) : 0.0;

  const auto streamID1 = cdoOpenRead(0);
  const auto vlistID1 = cdoStreamInqVlist(streamID1);

  const auto nvars = vlistNvars(vlistID1);
  for (varID = 0; varID < nvars; varID++)
    {
      const auto code = vlistInqVarCode(vlistID1, varID);
      const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
      const auto nlevels = zaxisInqSize(zaxisID);
      const auto ltype = zaxis2ltype(zaxisID);

      for (int levID = 0; levID < nlevels; levID++)
        {
          const auto level = cdoZaxisInqLevel(zaxisID, levID);
          const auto sellevel = (operatorArgc() == 3) ? IS_EQUAL(level, slevel) : true;
          const auto selcode = (scode == -1 || scode == code);
          const auto selltype = (sltype == -1 || sltype == ltype);

          if (selcode && selltype && sellevel)
            {
              vlistDefFlag(vlistID1, varID, levID, true);
              selfound = true;
            }
        }
    }

  if (!selfound) cdoWarning("Code %d, ltype %d, level %g not found!", scode, sltype, slevel);

  const auto vlistID2 = vlistCreate();
  cdoVlistCopyFlag(vlistID2, vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  Field field;

  VarList varList1;
  varListInit(varList1, vlistID1);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          if (vlistInqFlag(vlistID1, varID, levelID) == true)
            {
              const auto varID2 = vlistFindVar(vlistID2, varID);
              const auto levelID2 = vlistFindLevel(vlistID2, varID, levelID);
              cdoDefRecord(streamID2, varID2, levelID2);

              if (lcopy)
                {
                  cdoCopyRecord(streamID2, streamID1);
                }
              else
                {
                  field.init(varList1[varID]);
                  cdoReadRecord(streamID1, field);
                  cdoWriteRecord(streamID2, field);
                }
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
