/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Seasmonstat   seasmonmean        Seasonal mean from monthly data
      Seasmonstat   seasmonavg         Seasonal average from monthly data
*/

#include <cdi.h>

#include "cdo_options.h"
#include "functs.h"
#include "process_int.h"
#include "calendar.h"
#include "datetime.h"
#include "printinfo.h"
#include "cdo_season.h"

void *
Seasmonstat(void *process)
{
  TimeStat timestat_date = TimeStat::MEAN;
  int64_t vdate0 = 0, vdate1 = 0;
  int vtime0 = 0, vtime1 = 0;
  int nrecs;
  int seas0 = 0;
  int oldmon = 0;
  int nseason = 0;
  int month0 = 0;
  int year, month, day;
  const char *seas_name[4];

  cdoInitialize(process);

  // clang-format off
  cdoOperatorAdd("seasmonmean",  func_mean, 0, nullptr);
  cdoOperatorAdd("seasmonavg",   func_avg,  0, nullptr);
  // clang-format on

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);

  operatorCheckArgc(0);

  const auto season_start = getSeasonStart();
  getSeasonName(seas_name);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID2);
  if (taxisInqType(taxisID2) == TAXIS_FORECAST) taxisDefType(taxisID2, TAXIS_RELATIVE);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const auto calendar = taxisInqCalendar(taxisID1);
  DateTimeList dtlist;
  dtlist.setStat(timestat_date);
  dtlist.setCalendar(calendar);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  field.resize(gridsizemax);

  FieldVector2D samp1, vars1;
  fieldsFromVlist(vlistID1, samp1);
  fieldsFromVlist(vlistID1, vars1, FIELD_VEC);

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      long nsets = 0;
      double dsets = 0.0;
      bool newseas = false;
      while (true)
        {
          nrecs = cdoStreamInqTimestep(streamID1, tsID);
          if (nrecs <=0) break;

          dtlist.taxisInqTimestep(taxisID1, nsets);
          const auto vdate = dtlist.getVdate(nsets);
          const auto vtime = dtlist.getVtime(nsets);
          cdiDecodeDate(vdate, &year, &month, &day);

          auto newmon = month;
          if (season_start == START_DEC && newmon == 12) newmon = 0;

          const auto seas = monthToSeason(month);

          if (nsets > 0 && month == month0)
            {
              cdoWarning("   last timestep: %s %s", dateToString(vdate0).c_str(), timeToString(vtime0).c_str());
              cdoWarning("current timestep: %s %s", dateToString(vdate).c_str(), timeToString(vtime).c_str());
              cdoAbort("Month does not change!");
            }

          const auto dpm = days_per_month(calendar, year, month);

          if (nsets == 0)
            {
              nseason++;
              vdate0 = vdate;
              vtime0 = vtime;
              seas0 = seas;
              oldmon = newmon;
            }

          if (newmon < oldmon) newseas = true;

          if ((seas != seas0) || newseas) break;

          oldmon = newmon;

          for (int recID = 0; recID < nrecs; recID++)
            {
              int varID, levelID;
              cdoInqRecord(streamID1, &varID, &levelID);

              if (tsID == 0)
                {
                  recList[recID].varID = varID;
                  recList[recID].levelID = levelID;
                  recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
                }

              auto &rsamp1 = samp1[varID][levelID];
              auto &rvars1 = vars1[varID][levelID];

              const auto fieldsize = rvars1.size;

              if (nsets == 0)
                {
                  cdoReadRecord(streamID1, rvars1.vec_d.data(), &rvars1.nmiss);

                  vfarcmul(rvars1, dpm);

                  if (rvars1.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(fieldsize);

                      for (size_t i = 0; i < fieldsize; i++)
                        rsamp1.vec_d[i] = (DBL_IS_EQUAL(rvars1.vec_d[i], rvars1.missval)) ? 0 : dpm;
                    }
                }
              else
                {
                  cdoReadRecord(streamID1, field.vec_d.data(), &field.nmiss);
                  field.size = fieldsize;
                  field.grid = rvars1.grid;
                  field.missval = rvars1.missval;

                  vfarcmul(field, dpm);

                  if (field.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(fieldsize, dsets);

                      for (size_t i = 0; i < fieldsize; i++)
                        if (!DBL_IS_EQUAL(field.vec_d[i], rvars1.missval)) rsamp1.vec_d[i] += dpm;
                    }

                  vfarfun(rvars1, field, operfunc);
                }
            }

          month0 = month;
          vdate1 = vdate;
          vtime1 = vtime;
          nsets++;
          dsets += dpm;
          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (recList[recID].lconst) continue;

          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          auto &rsamp1 = samp1[varID][levelID];
          auto &rvars1 = vars1[varID][levelID];

          if (rsamp1.empty())
            vfarcdiv(rvars1, dsets);
          else
            vfardiv(rvars1, rsamp1);
        }

      if (Options::cdoVerbose)
        cdoPrint("season: %3d %3s  start: %s %s  end: %s %s ntimesteps: %ld", nseason, seas_name[seas0],
                 dateToString(vdate0).c_str(), timeToString(vtime0).c_str(), dateToString(vdate1).c_str(),
                 timeToString(vtime1).c_str(), nsets);

      dtlist.statTaxisDefTimestep(taxisID2, nsets);
      cdoDefTimestep(streamID2, otsID);

      if (nsets < 3)
        cdoWarning("Season %3d (%s) has only %d input time step%s!", otsID + 1, dateToString(vdate0).c_str(), nsets,
                   nsets == 1 ? "" : "s");

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recList[recID].lconst) continue;

          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          auto &rvars1 = vars1[varID][levelID];

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, rvars1.vec_d.data(), rvars1.nmiss);
        }

      if (nrecs == 0) break;
      otsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
