package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``3`$*
M`-[G'V```````````.``+@,+`0(C`%8```#D````$@```!`````0````<```
M``!````0`````@``!`````$````$``````````!P`0``!```;',!``,``(``
M`"```!``````$```$````````!``````$`$`\P$````@`0`0"0```#`!`%`K
M`````````````````````````&`!``0#````T```'```````````````````
M``````````````````````````````````````"D(0$`5`$`````````````
M`````````````````````"YT97AT````)%0````0````5@````0`````````
M`````````&``4&`N9&%T80```$``````<`````(```!:````````````````
M``!``&#`+G)D871A``"H0P```(````!$````7```````````````````0``P
M0"YB=6EL9&ED-0````#0`````@```*```````````````````$``,$`N96A?
M9G)A;00(````X`````H```"B``````````````````!``#!`+F)S<P`````P
M$0```/``````````````````````````````@`!@P"YE9&%T80``\P$````0
M`0```@```*P``````````````````$``,$`N:61A=&$``!`)````(`$```H`
M``"N``````````````````!``##`+G)S<F,```!0*P```#`!`%`K````N```
M````````````````0``PP"YR96QO8P``!`,```!@`0``!````.0`````````
M`````````$``,$(`````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````%6)Y8/D\(/L$,<$
M)`!B0`#H^T\``,=$)`@`````QT0D!`````#'!"0`````Z`]1``#'1"0(````
M`,=$)`0`````QP0D`````.@#40``QT0D"`````#'1"0$`````,<$)`````#H
M]U```,=$)`@`````QT0D!`````#'!"0`````Z.M0``#)PY"0D)"0D)"0D%6)
MY5=64X/L',<$)`"`0`#_%>@B00"#[`2%P`^$?P```(G#QP0D`(!``/\5\")!
M`(L][")!`(/L!*,`\$``QT0D!`^`0`")'"3_UX/L"(G&QT0D!"6`0`")'"3_
MUZ,`<$``@^P(A?9T$<=$)`0$\$``QP0D..!``/_6QT0D"`1P0`#'1"0$````
M`,<$)%`10`#H;DH``(UE]%M>7UW#9I#'!0!P0```````O@````#KM(VT)@``
M``"-M"8`````D%6)Y8/L&*$`<$``A<!T"<<$)#C@0`#_T*$`\$``A<!T#(D$
M)/\5Y")!`(/L!,G#D%575E.#["R)5"0<A<`/A)4```")QJ%$`$$`B40D&(7`
M#X2"````#[8&A,!T!#P]=3\QVXM\)!B++X7M="B-=@")7"0(B70D!(DL).C<
M2@``A<!U!X!\'0`]="V+;P2#QP2%[77;@\0LB>A;7E]=PY")\P^V0P&#PP&$
MP'0$/#UU\2GSZZZ-=@"+1"0<B?DK3"08C6P=`<'Y`HD(@\0LB>A;7E]=PXUV
M`(/$+#'M6XGH7E]=PXVT)@````"-="8`D%575E.)PX'LC````*&\(4$`B40D
M?#'`A=MT!8`[`'4@BT0D?"L%O"%!``^%$0$``('$C````%M>7UW#C70F`)")
M'"3HQ$D``(G&A<!TTHDT).C&20``A<`/A*(```"`>!0NC7@4=0F`?P$`=.&-
M=@"`>!0N#X2>````B1PDZ.))``")/"2)Q>C820``C40%`HD$).A$20``B7PD
M#(E<)`B)Q<=$)`1`@$``B00DZ(I)``"-1"08B2PDB40D!.B"20``@_C_=!"+
M1"0H)0#P```]`$```'1AB2PDZ*5)``")+"3HQ4@``(DT).@M20``A<`/A6?_
M__^-M"8`````9I")-"3HC$@``(D<).@D20``Z0O___^-=@"`?P$N#X58____
M@'\"``^$(O___^E)____C;0F`````(GHZ,'^___KGN@N2```C;0F`````(UV
M`%.)PX'LB````*&\(4$`B40D?#'`C40D&(D<)(E$)`3H8$@``(7`=3B+5"0H
M@>(`\```@?H`0```=`B!^@"@``!U(,=$)`0"````B1PDZ.)'``"%P`^4P`^V
MP.L&C70F`#'`BTPD?"L-O"%!`'4(@<2(````6\/HJ4<``(UT)@"04XG#@^P8
MB00DZ(9(``"-1`/_.<-R$>F;````C70F`)"#Z`$YPW0E@#@O=/0YPP^#@@``
M`(/H`3G#=!&-="8`D(`X+W0P@^@!.<-U](`[+W1OQP0D2(!``.@O2```B<*#
MQ!B)T%O#C;0F`````(UV`(`X+W4'@^@!.<-R]"G8,=*-2`*!^0`0``!WU(/`
M`8E<)`3'1"0,`!```(E$)`C'!"0@\$``Z`)'``"Z(/!``(/$&(G06\,YPW6[
MZY"-="8`QP0D1H!``.C`1P``@\08B<);B=##C;0F`````(UT)@!55U934+@X
M$```Z&E&```IQ(G6BP0$C7PD+,=$)`C_#P``B7PD!(G#QP0D2H!``*&\(4$`
MB80D+!```#'`Z#)'``"%P'YR@'P$*P"+%8PB00"-:/]U6H7M=&2);"0(QT0D
M!"\```")/"2)5"0<Z,)&``"+5"0<A<!T1(D4).CN10``B6PD#(E\)`B)7"0$
MB00DZ')%``"+A"0L$```*P6\(4$`=3.!Q#P0``!;7E]=PXG%ZZ:-="8`BQ6,
M(D$`B10DZ*I%``")="0(B5PD!(D$).@Z10``Z[[HYT4``(UV`%57B<=64XG3
M@^P\H;PA00")1"0L,<`QP(`Z/0^4P`'#B1PDZ*Q&``"-5"0HB<6)^.A[^___
MA<`/A/,```")!"2)QNB-1@``.<4/AL$```"+1"0HC0R%``````^V!X3`#X2;
M`0``/#T/A),!``")^)`/ME`!@\`!A-)T!8#Z/77P*?B-1"@"BS5$`$$`B00D
M`<[HMD4``(D&A<!T4Z%$`$$`BU0D*(U/`8L4D`^V!X@"A,!U&^L=C;0F````
M`(UV``^V`8/"`8/!`8@"A,!T!#P]=>W&`CTQP(VT)@````"0#[8,`XA,`@&#
MP`&$R77QBT0D+"L%O"%!``^%%`$``(/$/%M>7UW#C;0F`````&:0,<"-M@``
M```/MA0#B!0&@\`!A-)U\NO'C;0F`````)"+%40`00"+,H7V#X3'````,?:-
MM@````")\(/&`8L,LH7)=?2-!(4,````BPT@`$$`A<EU:XD$)(E4)!S'!2``
M00`!````Z-%$``")1"04A<`/A&C___^+5"0<C0RU`````(D$)(E,)`B)5"0$
MB4PD&.B^1```BT0D%(M,)!BC1`!!`(M$)!3'1+`$`````(ET)"CIDO[__XVT
M)@````"0B40D!(D4).C`1```B40D%*-$`$$`A<`/A`+___^-#+4`````Z[V-
MM"8`````C78`,<#I>_[__[@(````,?;I2/___^C,0P``C;0F`````)"#["RA
MC")!`(D$).A@0P``H8PB00")!"3H4T,``,=$)`P(@T``QT0D".!:0`#'1"0$
M68!``(D$).CK0@``H8PB00")!"3H)D,``,=$)!0`````QT0D$'6`0`#'1"0,
M=8!``,=$)`C@)T``QT0D!':`0`")!"3HID(``(/$+,-FD(/L'*&,(D$`B00D
MZ.!"``#'1"0,"(-``,=$)`C@6D``QT0D!%F`0`")!"3H>$(``(/$',.-M"8`
M````C70F`)`QP,.-M"8`````C;8`````5;BL$```5U93Z)I"```IQ*&\(4$`
MB80DG!```#'`C6PD.(N\),`0``"XBX!``(GJB[0DQ!```.B$^/__QT0D!"\`
M``")/"2)P^A^0P``A<!T+KB4@$``B?KHLOS__XGXBXPDG!```"L-O"%!``^%
M:@$``('$K!```%M>7UW#9I")/"3H7$,``(DT),=$)`2A@$``B40D+.A@0P``
MB<:%P'2PBT0D+(/``8`^`(E$)"P/A-8```"-M"8`````C78`A=MT%(ET)`2)
M'"3H!$,``(7`#X2-````B30DZ`1#``"`?`;_+W48C5;_C;8`````@^@!QD0"
M`0"`?`;_+W3Q`T0D+#W_#P``#X]$____B70D$(VT))P```")?"08QT0D%$B`
M0`#'1"0,HX!``,=$)`@`$```QT0D!`````")-"3HIT$``(EL)`2)-"3H<T(`
M`(7`=1"+1"1()0#P```]`(```'1+QT0D!*&`0`#'!"0`````Z(-"``")QH7`
M#X3/_O__@#X`#X4T____A=MT!8`[+G03OD:`0`"X`0```.E;____C70F`(![
M`0!TM^OEQT0D!`$```")-"3H1$$``(7`=:&)\KB4@$``Z$#[__^)-"3H!$(`
M`.F#_O__Z!)!``"-M"8`````C;0F`````(M$)`0/MA")P832=!.-=@"#P`&`
M^B\/MA`/1,B$TG7PB<C#C;0F`````(VV`````(M$)`2%P'00@#@`=`OI'OG_
M_XVV`````,=$)`1&@$``Z9=!``"-M"8`````C70F`(/L/*&\(4$`B40D+#'`
M,<#'1"0*7U]%3L=$)`Y67U!!QT0D$E)?0TS'1"0614%.7\=$)!I?("`@QT0D
M'B`@("#'1"0B("`@(,=$)"8@("`@9HE$)"J-=@"-5"0$N*J`0`#H"O;__X7`
M=":+5"0$H40`00"-!)"+4`2)$(72=-F0BU`(@\`$B1"%TG7TZ\IFD(U4)`2X
MLX!``.C2]?__A<!T+HM4)`2A1`!!`(T$D(M0!(D0A=)TV8VT)@````!FD(M0
M"(/`!(D0A=)U].O"9I"-5"0$N+N`0`#HDO7__X7`="Z+5"0$H40`00"-!)"+
M4`2)$(72=-F-M"8`````9I"+4`B#P`2)$(72=?3KPF:0C50D!+C$@$``Z%+U
M__^%P'0NBU0D!*%$`$$`C020BU`$B1"%TG39C;0F`````&:0BU`(@\`$B1"%
MTG7TZ\)FD(U4)`2XRX!``.@2]?__A<!T+HM4)`2A1`!!`(T$D(M0!(D0A=)T
MV8VT)@````!FD(M0"(/`!(D0A=)U].O"9I"-5"0$N-N`0`#HTO3__X7`="Z+
M5"0$H40`00"-!)"+4`2)$(72=-F-M"8`````9I"+4`B#P`2)$(72=?3KPF:0
MC50D!+B+@$``Z)+T__^%P'0NBU0D!*%$`$$`C020BU`$B1"%TG39C;0F````
M`&:0BU`(@\`$B1"%TG7TZ\)FD(U4)`2XYX!``.A2]/__A<!T+HM4)`2A1`!!
M`(T$D(M0!(D0A=)TV8VT)@````!FD(M0"(/`!(D0A=)U].O"9I"-5"0$N/&`
M0`#H$O3__X7`="Z+5"0$H40`00"-!)"+4`2)$(72=-F-M"8`````9I"+4`B#
MP`2)$(72=?3KPF:0C50D!+C[@$``Z-+S__^%P'0NBU0D!*%$`$$`C020BU`$
MB1"%TG39C;0F`````&:0BU`(@\`$B1"%TG7TZ\)FD(U4)`2XE(!``.B2\___
MA<!T+HM4)`2A1`!!`(T$D(M0!(D0A=)TV8VT)@````!FD(M0"(/`!(D0A=)U
M].O"9I"-5"0$N`6!0`#H4O/__XG"A<!T"KCQ@$``Z)+W__^-5"0$N!:!0`#H
M-//__XG"A<!T"K@H@4``Z'3W__^-5"0$N#.!0`#H%O/__XG"A<!T*;B+@$``
MZ%;W__^XRX!``+I?@4``Z$?W__^+1"0L*P6\(4$`=4>#Q#S#C50D!+A#@4``
MZ-GR__^)PH7`=`RXYX!``.@9]___Z\&!?"0;4$%27W6W@7PD'T-,14%UK6:!
M?"0C3CUUI(U4)"7KT>C3/```C;0F`````(VT)@````"0@^P<H;PA00")1"0,
MH0AP0`"#^/]T&HM,)`PK#;PA00!U/8/$',.-M"8`````C78`C50D"+CG@$``
MZ%+R__^)PC'`A=)T$@^V$H#Z,`^5P3'`A-(/E<`AR*,(<$``Z[?H8#P``(VT
M)@````"-="8`D(/L'.B</```B00DZ(P\``"%P'0"BP"#Q!S#C;8`````5U93
M@^PPH;PA00")1"0L,<"-5"0HN&&!0`"+="1`Z-WQ__^%P'0'@#@`B<-U(HM$
M)"PK!;PA00`/A8T```")\H/$,+AA@4``6UY?Z0#V__^)="0$B00DZ-@\``"%
MP'08BT0D+"L%O"%!`'5?@\0P6UY?PXUT)@"0B30DZ*0\``")'"2)Q^B:/```
MC40'`HD$).@&/```B5PD$,=$)`RA@$``B<>)="0(QT0D!*.`0`")!"3H1#P`
M`(M$)"PK!;PA00!U!XGZZ77____H9#L``%6X'($``%=64^@*.P``*<2+A"0P
M@0``C50D.,>$))P```!R@4``QX0DH````$:`0`")1"0HH;PA00")A"0,@0``
M,<"XBX!``,>$)*0`````````QX0DN````'>!0`#'A"2\````?H%``,>$),``
M``"&@4``QX0DQ````(N!0`#'A"3(`````````,>$)*@```!F@4``QX0DK```
M`(^!0`#'A"2P`````````.B`\/__A<!T"8`X``^%&P0``.CR.@``B00DZ.(Z
M``"%P`^$G@,``(L8A=L/A)0#``")'"3H<#L``(U$``&)!"3HW#H``(G&#[8#
MA,!T*(GWC70F`)")/"2#PP&#QP*)1"0(QT0D!)B!0`#H$CL```^V`X3`==^-
MO"2X````N"B!0`#K"I"+!X/'!(7`="^-5"0XZ.[O__^)PX7`=.B`.`!TX^CN
M\?__A<!TVHD<).CN.@``B<.%P'5.C70F`(N<))P```"-O"2<````ZQ*-M"8`
M````D(M?!(7;="F#QP2`.P!T(8G8Z*KQ__^%P'3FB1PDZ*HZ``"%P'3:B</K
M"(VV`````#';B1PDZ)HZ``")-"2)Q^B0.@``C;P'!`0``(D\).CY.0``B70D
M%,=$)!"=@4``B<;'1"0,2(!``(E<)`C'1"0$HH%``(D$)(E$)"3H*SH``,=$
M)`3``0``B30DZ-LY``"#^/\/A`8"``"-1"0XB40D!(M$)"2)!"3HECD``(G#
MA<!T3(M$)"@Q[8L8Z+\X``"+3"0DQT0D!.B!0`")7"0(B4PD#(M`#(D$).@T
M.0``BX0D#($``"L%O"%!``^%%P4``('$'($``(GH6UY?7<.+1"1()0#P```]
M`$```'6DBW0D4.@;.0``.<9UEXM$)$@E_P$``#W``0``=8>)/"3H%SD``(U4
M)#B)Q;AA@4``Z&ON__^%P`^$2P(``(D$).AW.0``B40D!(M$)"B+`(D$).B)
M]?__B40D+(7`#X0E`@``BW0D+(DT).A5.0``@_@'=CR-=`;XOSN"0`"Y"0``
M`/.F#Y?"@-H`A-)U(H/X"`^$F0,``(MT)"R`?`;W+P^$B@,``(VT)@````"-
M=@"A"'!``(/X_P^$.@,``(7`#X3;`0``NE^!0`"XYX!``.@;\O__Z#HX``"+
M="0DB2PDQT0D%'6`0`")1"00QT0D#$B`0`")="0(QT0D!%B"0`#HEC@``.M&
MC;0F`````)#HIS<``(,X$75)Z/4W``"#PP&)="0(QT0D&'6`0`")7"04B40D
M$,=$)`Q(@$``QT0D!&:"0`")+"3H3C@``,=$)`3``0``B2PDZ/XW``"#^/]T
MK8M$)"2)!"3HG3<``(GJN(N`0`#H=?'__XDL).@=^___Z4C^__^-M"8`````
MD.@G-P``@S@1#X3L_?__Z!DW```Q[8LPBT0D*(L8Z+XV``"+3"0DB70D$(E<
M)`B)3"0,QT0D!*R!0`"+0`R)!"3H+S<``.GV_?__C;8`````C;0DJ````+AF
M@4``ZPQFD(M&!(7`="Z#Q@2-5"0XZ)WL__^%P'3I@#@`=.2)!"3HJ#<``(G#
MA<`/A2[\__^+1@2%P'72QP0D#0```+MK@4``Z`8W``")QKA3````Z2?\__^-
MM@````")!"2)1"0DZ$SZ__^+A"0,@0``*P6\(4$`BT0D)`^%CP(``(F$)#"!
M``"!Q!R!``!;7E]=Z3LW``"-M"8`````BT0D*(L`B40D+.G,_?__QP4(<$``
M`````(M$)"S'1"0$```!`(D$).BO-@``B<:%P`^$!_[__\=$)`P"````C9PD
M#`$``,=$)`3N____QT0D"/____^)!"3H1C8``,=$)`@&````B5PD!(DT).AZ
M-@``@+PD#`$```!U#H"\)`T!``!##X1K`0``QT0D#`````#'1"0$`````,=$
M)`@`````B30DZ/HU``#H:3```(G'ZQ6-="8`D(E$)`B)7"0$B3PDZ-`P``#'
M1"0(`(```(E<)`2)-"3H$#8``(7`?]B)-"2-M"3C````B?/H<S4``(E\)`2-
MO"3/````B3PDZ&0R``"-="8`#[8'B30D@\<!@\8"QT0D!)B!0`")1"0(Z.\U
M```YWW7?C80DXP```,:$)`L!````QT0D%'6`0`")1"00BT0D),=$)`Q(@$``
MQT0D!$F"0`")1"0(B2PDZ+`U``#I<OW__XVT)@````"-5"0XN.>`0`#HFNK_
M_X7`#X2)_O__#[8`A,`/A'[^__\\,`^$=O[__\<%"'!```$```#IEOS__XM$
M)"B+4`2%T@^$<?S__XD4)(E4)"CH>34``(/X`P^&7/S__XM4)"B_1()``+D%
M````C70"_/.F#Y?`'`"$P`]%5"0LB50D+.DS_/__@+PD#@$``$$/A8?^__^`
MO"0/`0``0P^%>?[__X"\)!`!``!(#X5K_O__@+PD$0$``$4/A5W^___'1"0,
M`@```,=$)`3&____QT0D"/____^)-"3H5S0``,=$)`@I````B5PD!(DT).B+
M-```QT0D%'6`0`")7"00Z=+^___HU3,``)!64X/L)*&\(4$`B40D'*&,(D$`
MB00DZ&4S``")PZ&,(D$`B00DZ%8S```/ME-HB%!IC50D&+C+@$``Z&'I__^%
MP'0)@#@Q#X2L`0``Z-_R__^AC")!`(D$).@B,P``QT0D#`,```#'1"0(`0``
M`,=$)`1W@D``B00DZ,HR``")PX7`=#6+0!"+,*&,(D$`B00DZ.HR``#'1"0(
M>8)``(D$)(ET)`3H=C(``(M#$(L8]D,*0`^%AP$``*&,(D$`B00DZ+HR``#'
M1"0,`P```,=$)`@!````QT0D!'Z"0`")!"3H8C(``(G#A<!T.HU4)!BXE(!`
M`.BFZ/__B<:%P`^$#`$``(M#$(L8H8PB00")!"3H:C(``(ET)`B)7"0$B00D
MZ/HQ``"AC")!`(D$).A-,@``QT0D"`$```#'1"0$@()``(D$).@%,@``B<.A
MC")!`(D$).@F,@``QT0D")6"0`")!"2)7"0$Z+(Q``"AC")!`(D$).@%,@``
MC50D&,9`:0"XBX!``.@3Z/__HR0`00"%P'0ZQT0D!,`!``")!"3HMC(``(U0
M`8/Z`78)@_@1#X6G````BT0D'"L%O"%!``^%NP```(/$)%M>PXUV`*$H`$$`
MB00DZ)/V__^C)`!!`(7`=;#'!"2<@D``Z&8Q``"-M@````"`>`$`#X1/_O__
MZ47^__^-M"8`````9I"A*`!!`(LPBT,0BQBAC")!`(D$).A7,0``B?*)V.C>
MZO__Z>_^__^-M"8`````9I"AC")!`(D$).@S,0``B5PD!(D$).CG,```Z5O^
M__^)1"0,H20`00")1"0(H2@`00"+`,<$),""0`")1"0$Z-8P``#H33$``(VT
M)@````!FD%=64X/L,*&\(4$`B40D+#'`BUPD0(7;#X2C````@#L`#X2:````
MB=CH<^G__P^V$(G&A-)T&HVT)@````"-=@"#P`&`^B\/MA`/1/"$TG7PH0AP
M0`"%VP^5P8/X_W1YA<`/E<`AP83)=1Z+1"0L*P6\(4$`#X6B````@\0P6UY?
MPXVT)@````"`.P!TW;^=@4``N00```#SI@^7P!P`A,!UR(M$)"PK!;PA00!U
M;H/$,(G86UY?Z17G__^-="8`D,<$)$:`0`#H8#$``.E<____C;0F`````(U4
M)"BXYX!``(A,)!_H)N;__P^V3"0?A<!T"P^V`(3`=`0\,'4/QP4(<$``````
M`.E=____QP4(<$```0```.E*____Z",P``"0D)"0D)"0D)"0D)"0D)!5C5`X
M5U93@>P,`P``BQV\(4$`B9PD_`(``#';B70D"(VL)(````")1"1HC40D>(E\
M)`R)QXUV`(M*!(L"QT7\`````(/%$,=%]`````"#P@@/R(E%Z(G(P>@(B0PD
M)0#_``#'1"0$`````(G#B<C!Z!@)V(L<)(G9P>,(P>$8"<B)V8'A``#_``G!
MC80D``$``(E-\#G%=9^-E"1X`@``B?V)%"2-=@"+56B+16R#Q0C'17P`````
M,U4X,T4\,U4(,T4,,U7X,T7\B<>)T8G0`=$/K/@?"<B)17@[+"1UR(ML)&B+
M=12+10"+30B+?1R+502+70R)="0$BW48B4PD,(G!B7PD#(M](`^LT1N)="0(
MB<:)7"0TP>8%BUT0BVTD"<Z)?"18B50D+(M\)%@QT@-\)'B);"1<B?F+;"1<
M$VPD?(D<)('!F7F"6HE$)"B)ZXGP@],``<B+3"0P$=J+'"2)1"00B50D%(M4
M)`B)WS'7(<^)^HM\)`B+1"00QT0D)`````"+;"0TBTPD#,=$)#P`````,?J+
M?"0PQT0D1``````!T(GZB?Z)1"0@BP0DP>(>#ZSN`HML)"2)TXM4)`@)\P.4
M)(`````3C"2$````B5PD.(MT)#B)7"08B<N)T8M4)"R!P9EY@EJ#TP`QQHM$
M)"B)3"00B?>)7"04(<>)^XL\)#'[BWPD((G8`T0D$(G^B?F)[P^L[AO!X06)
M="00BVPD$,'O&XE\)!0)S<=$)!0`````B>X!QHM$)"B)="00B<$/K-`"P>$>
MB<L)PXG=B5PD0(L<)`.<)(@```"+3"08BW0D!(G8$[0DC````(GR!9EY@EJ+
M="0@BWPD%(/2`#'I,=LASC-T)#@!\(MT)!`1VHE$)!B+7"0@B?$/K/X;B50D
M','A!8GRP>\;QT0D'``````)RHM,)#B)?"1,B=8#="08B=J+?"0<B70D&(MT
M)"3!XA['1"1,``````^L\P*+="1`"=H#C"20````B<B)5"1(!9EY@EHQU2-L
M)!")5"10,>Z)_0'PBW0D&(GWB?$/K.\;P>T;B7PD(,'A!8EL)"2+;"0@"<V)
M[@'&B70D((M\)!"+;"04BTPD0,=$)"0`````B?N+="08QT0D/`````#!XQX/
MK.\"B=J+7"1$"?H#C"28````$YPDG````(G(BTPD4(E4)#B)U069>8):B=J+
M?"0D@](`,>DQVR'.,W0D2`'PBW0D(!':B40D$(M<)!B)\0^L_AN)5"04P>$%
MB?+![QO'1"04``````G*BTPD2(E\)$2)U@-T)!")VL=$)$0`````B70D$(MT
M)!S!XAX/K/,"BW0D.`G:`XPDH````(G(B50D0`69>8):,=4C;"0@B50D4#'N
M`?"+="00BWPD%,=$)$P`````B?V)]XGQ#ZSO&\'M&XE\)!C!X06+?"0@B6PD
M'(ML)!B)^\=$)!P`````"<V+3"0XB>Z+;"0D`<;!XQX/K.\"B=J+7"0\B70D
M&`GZ`XPDJ````!.<)*P```")R(M,)%")U8E4)$@%F7F"6HMT)!")VHM\)!R#
MT@`QZ3';(<XS="1``?"+="08$=J)1"0@BUPD$(GQ#ZS^&XE4)"3!X06)\L'O
M&\=$)"0`````"<J)?"0\B=8#="0@B=K'1"0\`````(ET)""+="04P>(>#ZSS
M`@G:B50D.(M,)$`#C"2P````BW0D2(E4)%")R(M\)"3'1"1$``````69>8):
M,=4C;"08,>Z)_0'PBW0D((GWB?$/K.\;P>T;B7PD$,'A!8M\)!B);"04BVPD
M$(G[QT0D%``````)S8M,)$B)[HML)!P!QL'C'@^L[P*)VHM<)$R)="00"?H#
MC"2X````$YPDO````(G(BTPD4(G5B50D0`69>8):BW0D((G:BWPD%(/2`#'I
M,=LASC-T)#@!\(MT)!`1VHE$)!B)\0^L_AN)5"0<P>$%B?+![QL)RHE\)$R)
MU@-T)!B)="08BUPD((MT)"2+3"0XQT0D'`````")VHM\)!S'1"1,``````^L
M\P+!XAZ+="1`QT0D/``````)V@.,),````")R(E4)$@%F7F"6C'5(VPD$(E4
M)%`Q[HG]`?"+="08B?>)\0^L[QO![1N)?"0@P>$%BWPD$(EL)"2+;"0@B?O'
M1"0D``````G-BTPD0(GNBVPD%`'&P>,>#ZSO`HG:BUPD1(ET)"`)^@.,),@`
M```3G"3,````B<B+3"10B=6)5"0X!9EY@EJ+="08B=J#T@`QZ3';(<XS="1(
M`?`1VHE$)!")5"04BW0D((M\)"2+7"08QT0D%`````")\<=$)$P`````#ZS^
M&\'A!8GRP>\;"<J+3"1(B7PD1(G6`W0D$(G:BWPD%(ET)!"+="0<P>(>QT0D
M1``````/K/,"BW0D.`G:`XPDT````(G(B50D0`69>8):,=4C;"0@B50D4#'N
MB?T!\(MT)!")]XGQ#ZSO&\'M&XE\)!C!X06+?"0@B6PD'(ML)!B)^\=$)!P`
M````"<V+3"0XB>Z+;"0D`<;!XQX/K.\"B=J)="08BUPD/`GZ`XPDV````(E4
M)$B)R!.<)-P```")U8M,)%`%F7F"6HMT)!")VH/2`(M\)!PQVS'I(<XS="1`
M`?"+="08$=J)1"0@BUPD$(GQ#ZS^&XE4)"3!X06)\L'O&\=$)"0`````"<J+
M3"1`B7PD/(G6`W0D((G:BWPD)(ET)""+="04P>(>QT0D/``````/K/,"BW0D
M2`G:`XPDX````(G(B50D.`69>8):,=4C;"08B50D4#'NB?T!\(MT)"")]XGQ
M#ZSO&\'M&XE\)!#!X06+?"08B6PD%(ML)!")^\=$)!0`````"<V)[HML)!P!
MQL'C'@^L[P*)VHET)!`)^HE4)$"+3"1(B=6+7"1,BW0D(,=$)$0``````XPD
MZ````!.<).P```")R(M,)%")VHM\)!0%F7F"6H/2`#'I,=LASC-T)#@!\(MT
M)!`1VHE$)!B+7"0@B?$/K/X;B50D','A!8GRP>\;QT0D'``````)RHM,)#B)
M?"1,B=8#="08B=J+?"0<B70D&(MT)"3!XA['1"1,``````^L\P*+="1`"=H#
MC"3P````B<B)5"1(!9EY@EHQU2-L)!")5"10,>Z)_0'PBW0D&(GWB?$/K.\;
MP>T;B7PD(,'A!8EL)"2+;"0@BWPD$,=$)"0`````"<V+3"1`QT0D/`````")
M[HG[BVPD%`'&P>,>#ZSO`HG:BUPD1(ET)"`)^@.,)/@````3G"3\````B<B+
M3"10B=6)5"0X!9EY@EJ+="08B=J+?"0D@](`,>DQVR'.,W0D2`'PBW0D(!':
MB40D$(M<)!B)\0^L_AN)5"04P>$%B?+![QO'1"04``````G*BTPD2(E\)$2)
MU@-T)!")VL=$)$0`````B70D$(MT)!S!XAX/K/,""=H#C"0``0``B<B)5"1`
M!9EY@EJ)5"10,=4C;"0@BW0D.(M\)!3'1"1,`````#'NB?T!\(MT)!")]XGQ
M#ZSO&\'M&XE\)!C!X06+?"0@B6PD'(ML)!B)^\=$)!P`````"<V+3"0XB>Z+
M;"0D`<;!XQX/K.\"B=J+7"0\B70D&`GZ`XPD"`$``!.<)`P!``")R(M,)%")
MU8E4)$@%F7F"6HMT)!")VH/2`#'I,=LASC-T)$")\0'!BT0D&!'3BU0D'(E,
M)"")QHE<)"2+3"0@#ZS0&\'F!8M4)!3'1"0D`````(G#BT0D$`GS`=F)SHE,
M)"")P0^LT`+!X1Z)RXM,)$`)PXE<)%`#C"00`0``B<B+5"0DQT0D%``````%
MF7F"6C-L)%`C;"08,VPD2(GI`<&+1"0@P>`%B40D$(M$)""+;"00#ZS0&XM4
M)!R)1"0XBT0D&`ML)#@!Z8ML)%")3"1@B4PD$(G!#ZS0`HM4)$C!X1Z)RXM,
M)$P)PP.4)!@!```3C"0<`0``B5PD.(G0BUPD.(G*!:'KV6Z#T@`QZXE$)!B)
M5"0<B=HQVS'RB=$#3"08$UPD'(E,)!B)7"0<BT0D$(M4)!3'1"0<`````(G&
MP>8%B=&)PHM$)"")="1`BUPD0`^LRAN+3"08B<8)TXM4)"0!V8G##ZS6`L'C
M'HE,)$@)\XNT)"0!``")3"08B=^+G"0@`0``B7PD0('#H>O9;H/6`(G8,=N)
M\@'H$=J+7"0X,?LQ_XG9BUPD8#'+BTPD'(G>`<:)RQ'7BU0D&(ET)"")?"0D
MB=>)T<=$)"0`````P><%#ZS9&XM<)"")_HM\)!0)S@'SBW0D$(E<)"")\0^L
M_@*)7"10BUPD2,'A'HGPB<Z+C"0H`0``"<:!P:'KV6Z)R(M,)#B)="0X`<B+
M3"1`,?$QRXM,)"")WHM<)"0!QHG(P>`%#ZS9&XE$)!"+;"00BWPD'(N4)#`!
M``#'1"04``````G-BT0D0`'NB70D2(ET)!"+="08B?$/K/X"BWPD4,'A'HG+
MBXPD-`$```GS@<*AZ]ENBW0D.(/1`(G=B<N)T3'2`<@1VC'N,=LQ]XGY`<&+
M1"00$=.+5"04B4PD&(G&B5PD'`^LT!O!Y@6+5"0DQT0D'`````")PPGSBW0D
M2(G8BUPD&`'#BT0D((E<)!B)P0^LT`*+5"0<B5PD0,'A'@G!BT0D.(G/BXPD
M.`$``(G[@<&AZ]EN`<B)^3'I,<Z)\0'!BT0D&,'@!8E$)""+1"08BW0D(`^L
MT!N)1"0X"W0D.`'QB4PD2(E,)""+1"00BU0D%,=$)"0`````B<&)?"10BWPD
M2,'A'@^LT`*+5"0DQT0D%`````")SHN,)$`!```)QHGH@<&AZ]ENB70D.`'(
M,?.+="1`,=X!QHM$)"")P<'A!8E,)!")T8G"BUPD$`^LRAN+1"08B?&)5"1`
M"UPD0`'9BU0D'(G&B4PD$(E,)$")P0^LU@*+1"10P>$>B<L)\XNT)$P!``")
MW8N<)$@!``"!PZ'KV6Z#U@`QT@'8$?*+="0X,=LQ[C'WB?D!P8M$)!`1TXM4
M)!2)3"08B<:)7"0<BTPD&`^LT!O!Y@7'1"0<`````(G#BT0D(`GS`=F)3"1(
MB4PD&(M4)"2)P<'A'HMT)$#'1"0D``````^LT`*)RXN4)%`!```)PXM$)#B!
MPJ'KV6Z)WP'0,>N+5"0<B7PD4#'>B?N)\0'!BT0D&,'@!8E$)""+1"08BW0D
M(`^LT!N+5"04QT0D%`````")1"0XBT0D$`MT)#@!\8E,)"")3"1`B<$/K-`"
MBY0D6`$``,'A'HG."<:!PJ'KV6Z)Z`'0B70D.#'SBW0D2(M4)"0QWHN<)&`!
M```!QHM$)"")P<'A!8E,)!")T8G"#ZS*&XM,)!")5"1("TPD2(G(B?&+5"0<
M`<&+1"08B4PD2(E,)!")QHG!#ZS6`L'A'HGPB<X)QH'#H>O9;HGUB[0D9`$`
M`(M$)%"+?"1`@]8`,=(!V!'RBW0D.#';,>XQ]XGY`<&+1"00$=.+5"04B4PD
M&(G&B5PD'(M,)!@/K-`;P>8%BU0D),=$)!P`````B<.+1"0@QT0D)``````)
M\XMT)$@!V8E,)!B)3"1`B<$/K-`"BY0D:`$``,'A'HG+"<.+1"0X@<*AZ]EN
MB=\!T#'KBU0D'(E\)%`QWHG[B?$!P8M$)!C!X`6)1"0@BT0D&(MT)"`/K-`;
MBU0D%(E$)#B+1"00"W0D.`'QB4PD2(E,)"")P0^LT`*+E"1P`0``P>$>B<X)
MQH'"H>O9;HGH`=")="0X,?.+="1`BU0D)(M\)$C'1"04`````#'>BYPD>`$`
M``'&BT0D((G!P>$%B4PD$(G1B<(/K,H;BTPD$(E4)$`+3"1`B<B)\8M4)!P!
MP8M$)!B)3"1`B4PD$(G&B<$/K-8"P>$>B?")S@G&BT0D4('#H>O9;HGUB[0D
M?`$``(/6`#'2`=@1\HMT)#@QVS'N,?>)^0'!BT0D$!'3BU0D%(E,)!B)QHE<
M)!R+3"08#ZS0&\'F!8M4)"3'1"0<`````(G#BT0D(`GSBW0D0`'9B4PD&(E,
M)$B)P0^LT`*+E"2``0``P>$>B<L)PXM$)#B!PJ'KV6Z)WP'0,>LQWHG[B?$!
MP8M$)!C!X`6)1"0@BT0D&(M4)!R+="0@QT0D)`````")?"10#ZS0&XM4)!3'
M1"04`````(E$)#B+1"00"W0D.`'QB4PD0(M\)$")3"0@B<$/K-`"BY0DB`$`
M`,'A'HG."<:!PJ'KV6Z)Z`'0B70D.#'SBW0D2(M4)"0QWHN<))`!```!QHM$
M)"")P<'A!8E,)!")T8G"#ZS*&XM,)!")5"1("TPD2(G(B?&+5"0<`<&+1"08
MB4PD2(E,)!")QHG!#ZS6`L'A'HGPB<X)QHM$)%"!PZ'KV6Z)]8NT))0!``"#
MU@`QT@'8$?*+="0X,=LQ[C'WB?D!P1'3B4PD&(E<)!R+1"00BU0D%(M,)!C'
M1"0<`````(G&#ZS0&\'F!8M4)"3'1"0D`````(G#BT0D(`GSBW0D2`'9B4PD
M&(E,)%")P0^LT`*+E"28`0``P>$>B<L)PXM$)#B!PJ'KV6Z)WP'0,>N+5"0<
MB7PD;#'>B?N)\0'!BT0D&,'@!8E$)""+1"08BW0D(`^LT!N+5"04B40D.(M$
M)!`+="0X`?&)3"0@B4PD2(G!#ZS0`HN4)*`!``#!X1Z)S@G&@<*AZ]ENB>@!
MT(ET)$`Q\XMT)%"+5"0D,=X!QHM$)"")P<'A!8E,)!")T8G"#ZS*&XM,)!")
M5"0X"TPD.(M\)!B+;"0<B<B)\0'!BYPDJ`$``,=$)#P`````B?J)_HE,)&"+
M?"0\#ZSN`L'B'HE,)#B+3"1LB?")UHG]QT0D3``````)QH'#H>O9;HG8BUPD
M0(ET)%`!R#'SBW0D.(G9BUPD2(GW,<N)\0'8#ZSO&XN<)+`!``#![1N)?"00
MP>$%BWPD((EL)!2+;"00QT0D%``````)S8GYB>Z+;"0D`<;!X1X/K.\"B70D
M$(NT)+0!```)^8'#H>O9;HG-B4PD2(M,)$"#U@")V(GR,=N+="00`<B+3"10
M$=J+7"1@,?\QZ3'+B?$!V!'ZBWPD%,'A!8E$)!B)5"0<#ZS^&\'O&XGRB7PD
M)`G*B=8#="08BWPD/,=$)!P`````B70D&(MT)#B+1"10QT0D/`````")\0^L
M_@*+?"0<QT0D1`````#!X1Z)RXN,)+@!```)\X'!W+P;CXE<)#B)W@'(B5PD
M4`G=BUPD2"-L)!`A\XMT)!@)W0'HB?V)]XGQ#ZSO&\'M&XE\)"#!X06+?"00
MB6PD)(ML)"")^\=$)"0`````"<V+3"10B>Z+;"04`<;!XQZ+1"08#ZSO`HG:
MB70D(#';"?H)T8G5B50D0(M4)#@AR"'JB<:+1"1("=8#A"3``0``BU0D3!.4
M),0!```%W+P;CX/2``'PB40D$(MT)"`1VHM\)"2+7"08B50D%(GQQT0D%```
M```/K/X;P>$%QT0D/`````")\L'O&PG*B7PD3(M\)!2)U@-T)!")VL=$)$P`
M````B70D$(MT)!S!XAX/K/,"BW0D.`G:BUPD0`G5(VPD((E4)$@ATXE4)%")
MZHG]"=H#M"3(`0``B?"+="00!=R\&X^)]P'0B?$/K.\;P>T;B7PD&,'A!8M\
M)"");"0<BVPD&(G[QT0D'``````)S8M,)%")[HML)"0!QL'C'HM$)!`/K.\"
MB=J)="08,=L)^@G1B=6)5"0XBU0D2"'(BW0D0(M\)$0AZ@G0`[0DT`$``!.\
M)-0!``")P8GPB?J+="08BWPD'`7<O!N/@](``<B)\1':#ZS^&XE$)""+7"00
MB50D),'A!8GRP>\;"<J)?"1$B=8#="0@B=K'1"0D`````(ET)""+="04P>(>
MBWPD),=$)$0`````#ZSS`HMT)$@)VHM<)#@)U2-L)!B)5"1`(=.)5"10B>J)
M_0G:`[0DV`$``(GPBW0D(`7<O!N/B?<!T(GQ#ZSO&\'M&XE\)!#!X06+?"08
MB6PD%(ML)!")^\=$)!0`````"<V)[@'&P>,>B70D$(ML)!R)VC';BTPD4(M$
M)"#'1"1,``````^L[P*+="0X"?J+?"0\"=&)5"1(B=6+5"1`(<@AZ@G0`[0D
MX`$``!.\).0!``")P8GPB?J+="00BWPD%`7<O!N/@](``<B)\1':#ZS^&XE$
M)!B+7"0@B50D','A!8GRP>\;"<J)?"0\B=8#="08B=K'1"0<`````(ET)!B+
M="0DP>(>BWPD',=$)#P`````#ZSS`HMT)$`)VHM<)$@)U2-L)!")5"0X(=.)
M5"10B>J)_0G:`[0DZ`$``(GPBW0D&`7<O!N/B?<!T(GQ#ZSO&\'M&XE\)"#!
MX06+?"00B6PD)(ML)"")^\=$)"0`````"<V+3"10QT0D1`````")[HML)!0!
MQL'C'HM$)!@/K.\"B=J)="0@,=L)^HMT)$B+?"1,"=&)5"1`B=6+5"0X(<@A
MZ@G0`[0D\`$``!.\)/0!``")P8GPB?J+="0@BWPD)`7<O!N/@](``<B)\1':
M#ZS^&XE$)!"+7"08B50D%,'A!8GRP>\;"<J)?"1,B=8#="00B=K'1"04````
M`(ET)!"+="0<P>(>QT0D3``````/K/,""=J+7"1`"=4C;"0@B50D2"'3B50D
M4(GJBW0D.`G:`[0D^`$``(M\)!3'1"0\`````(GPBW0D$(G]!=R\&X^)]P'0
MB?$/K.\;P>T;B7PD&,'A!8M\)"");"0<BVPD&(G[QT0D'``````)S8M,)%")
M[HML)"0!QL'C'HM$)!`/K.\"B=J)="08,=L)^HMT)$"+?"1$"=&)U8E4)#B+
M5"1((<@AZ@G0`[0D``(``!.\)`0"``")P8GPB?J+="08BWPD'`7<O!N/@](`
M`<B)\1':#ZS^&XE$)""+7"00B50D),'A!8GRP>\;"<J)?"1$B=8#="0@QT0D
M)`````")VHET)""+="04P>(>BWPD),=$)$0`````#ZSS`HMT)$C'1"1,````
M``G:BUPD.`G5(VPD&(E4)$`ATXE4)%")ZHG]"=H#M"0(`@``B?"+="0@!=R\
M&X^)]P'0B?$/K.\;P>T;B7PD$,'A!8M\)!B);"04BVPD$(G[QT0D%``````)
MS8M,)%")[HML)!P!QL'C'HM$)"`/K.\"B=J)="00,=L)^HMT)#B+?"0\"=&)
MU8E4)$B+5"1`(<@AZ@G0`[0D$`(``!.\)!0"``")P8GPB?J+="00BWPD%`7<
MO!N/@](``<B)\1':#ZS^&XE$)!C!X06)5"0<B?+![QL)RHE\)#R+7"0@B=8#
M="08QT0D'`````")="08BW0D)(G:P>(>BWPD',=$)#P`````#ZSS`HMT)$#'
M1"1$``````G:BUPD2`G5(VPD$(E4)#@ATXE4)%")ZHG]"=H#M"08`@``B?"+
M="08!=R\&X^)]P'0B?$/K.\;P>T;B7PD(,'A!8M\)!");"0DBVPD((G[QT0D
M)``````)S8M,)%")[HML)!0!QL'C'HM$)!@/K.\"B=J)="0@,=L)^HMT)$B+
M?"1,"=&)U8E4)$"+5"0X(<@AZ@G0`[0D(`(``!.\)"0"``#'1"0\`````(G!
MB?")^HMT)""+?"0D!=R\&X^#T@`!R(GQ$=H/K/X;B40D$(M<)!B)5"04P>$%
MB?+![QL)RHE\)$R)U@-T)!")VL=$)!0`````B70D$(MT)!S!XAZ+?"04QT0D
M3``````/K/,"BW0D.`G:BUPD0`G5(VPD((E4)%`ATXE4)$B)ZHG]"=H#M"0H
M`@``B?"+="00!=R\&X^)]P'0B?$/K.\;P>T;B7PD&,'A!8M\)"");"0<BVPD
M&(G[QT0D'``````)S8M,)%")[HML)"0!QL'C'@^L[P*)VHET)!@QVPGZ"=&)
MU8E4)#B+1"00BU0D2(MT)$`AR(M\)$0AZ@G0`[0D,`(``!.\)#0"``")P8GP
MB?J+="08BWPD'`7<O!N/@](``<B)\1':#ZS^&XE$)""+7"00B50D),'A!8GR
MP>\;"<J)?"1$B=8#="0@B=K'1"0D`````(ET)""+="04P>(>BWPD),=$)$0`
M````#ZSS`HMT)$@)VHM<)#@)U2-L)!B)5"1`(=.)5"10B>J)_0G:`[0D.`(`
M`(GPBW0D(`7<O!N/B?<!T(GQ#ZSO&\'M&XE\)!#!X06);"04BVPD$`G-B>X!
MQHET)!#'1"04`````(M\)!B+;"0<BTPD4,=$)$P`````B?N+1"0@P>,>#ZSO
M`HG:,=L)^HM\)#P)T8E4)$B)U8M4)$`AR"'JB<8)UHGQBW0D.`.T)$`"``")
M\!.\)$0"``"+="00B?H%W+P;CXM\)!2#T@`!R(GQ$=H/K/X;B40D&(M<)"")
M5"0<P>$%B?+![QL)RHM,)$B+1"1`B7PD/(G6`W0D&(G:QT0D'`````")="08
MBW0D),'B'L=$)"0`````#ZSS`HG>,=L)U@GU(VPD$"'QB70D((GJ"<H#A"1(
M`@``B=&+5"1$$Y0D3`(```7<O!N/@](``<&)3"0X$=.+5"08BTPD'(M$)!B)
M7"0\B=?'1"0\`````,'G!8G+B=&)\@^LV1N)_8M\)!#'1"1$``````G-BTPD
M.`'IBVPD%(E,)%")3"0XB?D/K.\"P>$>B<L)^XM\)$R)W8E<)$")\PGK(>HA
MV#';B<8)UHGQBW0D2`.T)%`"``")\!.\)%0"``")^@7<O!N/@](``<&+1"0X
M$=.+5"0\B4PD$(G&B5PD%`^LT!O!Y@6+5"0<QT0D%`````")PPGSBW0D4(G8
MBUPD$`'#BT0D&(E<)!")P8E<)$@/K-`"P>$>B<N+3"0@"<,#C"18`@``B<@%
MUL%BRC'=,>Z)W8GQ`<&+1"00P>`%B40D&(M$)!"+5"04BW0D&,=$)!P`````
MBWPD%,=$)"0`````#ZS0&XM4)#R)1"0@BT0D.`MT)"`!\8E,)%")3"08B<$/
MK-`"BU0D','A'HG.BTPD0`G&`XPD8`(``(G(B70D.`76P6+*,?.+="1(,=Z)
M\0'!BT0D&(G.B<'!X06)3"0@B=&)PHM<)"`/K,H;B?&+="00B50D0`M<)$")
M\@'9#ZS^`L'B'HE,)$B)TXE,)"`)\XNT)&P"``")WXN<)&@"``")?"1`@</6
MP6+*@]8`B=@QVXGR`>@1VHM<)#@Q^S'_B=F+7"10,<N)W@'&$=>)="00B7PD
M%(M4)""+3"0DQT0D%`````")U\'G!8G+B=$/K-D;B?Z+7"00BWPD'`G.QT0D
M'``````!\XMT)!B)7"10B?$/K/X"B5PD$(M<)$C!X1Z)\(M\)"2)SHN,)'`"
M```)QH'!UL%BRHG(BTPD.(ET)#@!R(M,)$`Q\3'+BTPD$(G>BUPD%`'&B<C!
MX`4/K-D;B40D&(ML)!B+1"1`"<T![HET)!B)="1(BW0D((GQ#ZS^`HM\)%#!
MX1Z)RXN,)'@"```)\XMT)#B)W8N<)'P"``"!P=;!8LJ#TP`QT@'($=HQ[C';
M,?>)^0'!BT0D&!'3BU0D'(E,)"")QHE<)"0/K-`;P>8%B<,)\XG8BUPD(`'#
MB5PD4(MT)!"+?"04BT0D.(E<)"")\8N<)(0"``#'1"0D``````^L_@+!X1Z+
M?"1("?&)SHN,)(`"``")="1`@<'6P6+*@],`,=(!R(GQ$=HQZ3';,<^)^0'!
MBT0D(!'3BU0D)(E,)!")QHE<)!0/K-`;P>8%BU0D'(G!"?&)R(M,)!`!P8M$
M)!@Q_XG.B4PD2(G!#ZS0`L'A'HG+BXPDB`(```G#B>B)7"0XBYPDC`(``('!
MUL%BRH/3`#'2`<B+3"1`$=HS3"0XB40D$(GPB50D%(M4)%`QRHG1`TPD$,'@
M!8E$)!")\(ML)!`/K/@;B40D&`ML)!@!Z8E,)%")3"00BT0D((M4)"3'1"04
M`````(G!P>$>#ZS0`HN4))`"``")RXN,))0"```)PX'"UL%BRHM$)$"#T0")
MW8G+B=$QT@'($=J+7"0XB40D&(E4)!R+5"1(,>N+1"00,=HQVXG1`TPD&(M4
M)!2)3"08B<$37"0<P>$%B5PD'(N<))@"``")3"0@B=&)P@^LRAN+3"0@QT0D
M'`````")5"1`"TPD0(G(BTPD&`'!B4PD0(E,)!B)\0^L_@+!X1Z)\(G."<:+
M1"0X@</6P6+*B?>+M"2<`@``B7PD.(/6`#'2`=@1\HG^BWPD4#';,>XQ]XGY
M`<$1TXE,)"")7"0DBT0D&(M4)!R+3"0@B<8/K-`;P>8%BU0D%(G#BT0D$`GS
M`=DQ_XG.B4PD2(G!#ZS0`L'A'HN4)*`"``")RXN,)*0"```)PX'"UL%BRHGH
M@]$`B5PD((G+B=$QT@'($=J+7"0X,UPD((E$)!")5"04BU0D0(GPQT0D%```
M```QVHG1`TPD$,'@!8M4)!R)1"00B?"+;"00#ZSX&XE$)$"+1"08"VPD0`'I
MB4PD0(E,)!")P0^LT`*+E"2H`@``P>$>B<N+C"2L`@``"<.!PM;!8LJ+1"0X
M@]$`B=V)RXG1,=(!R!':BUPD((E$)!B)5"0<BU0D2#'K,=HQVXG1`TPD&(M$
M)!"+5"04$UPD'(E,)!B)P8E<)!R+G"2P`@``P>$%QT0D'`````")3"0XB=&)
MP@^LRAN+3"0XB50D2`M,)$B)R(M,)!@!P8E,)!B)3"1(B?$/K/X"P>$>B?")
MS@G&BT0D(('#UL%BRHGWB[0DM`(``(E\)#B#U@`QT@'8$?*)_HM\)$`QVS'N
M,?>)^0'!BT0D&!'3BU0D'(E,)"")QHM,)"")7"0D#ZS0&\'F!8M4)!2)PXM$
M)!`)\P'9,?^)SHE,)$")P0^LT`+!X1Z+E"2X`@``B<N+C"2\`@``"<.!PM;!
M8LJ)Z(/1`(E<)"")RXG1,=(!R!':B40D$(GPB50D%(M4)$B+7"0X,UPD(,=$
M)!0`````,=J)T0-,)!#!X`6+5"0<B40D$(GPBVPD$`^L^!N)1"1(BT0D&`ML
M)$@!Z8E,)!")3"1(B<$/K-`"BY0DP`(``,'A'HG+BXPDQ`(```G#@<+6P6+*
MBT0D.(/1`(G=B<N)T3'2`<@1VHM<)"")1"08B50D'(M4)$`QZXM$)!`QVC';
MB=$#3"08BU0D%(E,)!B)P1-<)!S!X06)7"0<B4PD.(G1B<(/K,H;BTPD.(E4
M)$`+3"1`B<B+3"08BYPDR`(``,=$)!P``````<&)3"1`B4PD&(GQ#ZS^`L'A
M'HGPB<X)QHM$)""!P];!8LJ)]XNT),P"``")?"0@@]8`,=(!V!'RB?Z+?"1(
M,=LQ[C'WB?D!P8M$)!@1TXM4)!R)3"0XB<:)7"0\BTPD.`^LT!O!Y@4QTHG#
M"?.)V(M<)#P!P8M$)!`1TXM4)!2)SHE,)$B)P8E<)$PQ_P^LT`+!X1Z+E"30
M`@``B<N+C"34`@``"<.!PM;!8LJ)Z(/1`(E<)#B)RXG1,=(!R!':BUPD(#-<
M)#B)1"00B50D%(M4)$")\#':B=$#3"00P>`%B?J)1"00B?#!ZAL/K/@;B50D
M1(E$)$"+;"00BT0D&`ML)$#'1"04``````'IBU0D'(ML)#B)3"00B4PD4(G!
M#ZS0`HN4)-@"``#!X1Z)RXN,)-P"```)PX'"UL%BRHM$)""#T0")7"1`B<N)
MT3'2`<@1VHM<)$")1"08B50D'(M4)$@QZXM$)!`QVC';B=$#3"08BU0D%(E,
M)!B)P1-<)!S!X06)7"0<B4PD((G1B<*+7"0@#ZS*&XM,)!B)5"0X"UPD.#'2
MB=B+7"0<`<$1TXGR#ZS^`HE,)&#!XAZ)7"1DB=.)3"0@"?/'1"0D`````(NT
M).0"``")WXN<).`"``#'1"0\`````(E\)$B!P];!8LJ#U@")V#';B?(!Z!':
MBUPD0#'[,?^)V8M<)%`QRXM,)"2)W@'&B<L1UXM4)"")="08B7PD'(G7B=$Q
MTL'G!0^LV1N+7"08B?Z+?"04"<Z)\(MT)!P!PQ'6B5PD.(ET)%2+="00B5PD
M4(G[B?$/I/,>P>$>#ZS^`HGPB<[![P*+C"3H`@``"<:+1"1`B70D&(G>BUPD
M8`G^BWPD/('!UL%BRHET)!R+="0X`<B+3"1(,TPD&(G],<N)]XGQ`=@/K.\;
MP>T;B7PD$,'A!8M\)"");"04BVPD$,=$)!0`````"<V)^8GN`<;!X1Z)="00
MBVPD)(G.BT0D*(M4)#R)ZP^D^QX/K.\"B=G![0()_HM\)%`)Z0.$)/`"``"+
M;"04B70D((E,)"2+3"1(!=;!8LH!R(M,)!@Q\3'/`?B+?"00B?Z)^8GO#ZSN
M&\'A!8ET)"B+;"0HP>\;B7PD+`G-B>Z+;"1H`<:+1"00`T0D,(E%"(M$)#B)
M=0")QL=%!``````/K-`"P>8>QT4,`````(GU"<6+1"0@B>F+;"1H`PPD`T0D
M"(E-$,=%%`````")11B+1"08`T0D6,=%'`````")12#'120`````BX0D_`(`
M`"L%O"%!`'4+@<0,`P``6UY?7</H0`4``(VT)@````"-="8`D(/L',<$)(``
M``#HC04``,<``2-%9\=`!`````#'0`B)J\WOQT`,`````,=`$/[<NIC'0!0`
M````QT`8=E0R$,=`'`````#'0"#PX=+#QT`D`````,=`*`````#'0"P`````
MQT`P`````,=`-`````#'0'@`````@\0<PXUT)@"055=64X/L+(M,)$B+;"1`
MBUPD1(G(BW4PBWTTB4PD$)G!X`,#12@[12B)5"04N@`````;52QS!H/&`8/7
M`(M4)!2)12B+1"00QT4L``````^LT!W!ZAT!\(E%,(M%>!'ZC7TXB54TA<!T
M4;Y`````B5PD!"G&B4PD2#GQ#T[Q`?B)!"2)="0(Z*`$``"+17B+3"1(`?"#
M^$")17AT"X/$+%M>7UW#C78`*?&)Z`'SB4PD2.CIT___BTPD2(/Y/P^._```
M`(U!P(E$)!S!Z`:-<`&)1"00P>8&`=Z-M@````"+`X/#0(D'BT/$B4<$BT/(
MB4<(BT/,B4<,BT/0B4<0BT/4B4<4BT/8B4<8BT/<B4<<BT/@B4<@BT/DB4<D
MBT/HB4<HBT/LB4<LBT/PB4<PBT/TB4<TBT/XB4<XBT/\B4<\B>CH6-/__SGS
M=92+1"00BTPD','@!BG!@_D$<D*+!HGZB44XBT0._(E$#_R-1P2#X/PIPBG6
M`<J#XOR#^@1R$H/B_#';BSP>B3P8@\,$.=-R\XE->(/$+%M>7UW#9I"%R73O
M#[8&B$4X]L$"=.0/MT0._F:)1`_^Z]B-="8`D(G>ZY6-M"8`````C70F`)!5
M5U93@^P<BUPD-(MT)#"+:RR+>RB+0RB+4S`/K.\#C6LX@^<_#\B)1"0(#\J-
M1P'&1#LX@(U,!0"#^#@/CB8!``"_0````"G'=`TQP,8$`0"#P`$Y^'+UC7T$
MB=B)5"0,Z&K2__^#Y_S'0S@`````,<#'130`````*?V-33C!Z0+SJXM4)`R+
M1"0(B5-PB4-TB=CH.-+__XM3!(L##ZS0&(@&BU,$BP,/K-`0B$8!BP.(9@*+
M`XA&`XM3#(M#"`^LT!B(1@2+4PR+0P@/K-`0B$8%BT,(B&8&BT,(B$8'BU,4
MBT,0#ZS0&(A&"(M3%(M#$`^LT!"(1@F+0Q"(9@J+0Q"(1@N+4QR+0Q@/K-`8
MB$8,BU,<BT,8#ZS0$(A&#8M#&(AF#HM#&(A&#XM3)(M#(`^LT!B(1A"+0R"+
M4R0/K-`0B$81BT,@B&82BT,@B$83B5PD,(/$'%M>7UWINP$``(VT)@````"_
M.````#'M*<>#_P1R.HU!!,<!`````(/@_,=$.?P`````*<$!SX/G_(/_!`^"
M[?[__X/G_#')B2P(@\$$.?ER]NG9_O__9I"%_P^$S_[__\8!`/?'`@````^$
MP/[__S'M9HEL.?[IM/[__Y"0D)"0D)"0D)"0D)#_)=PB00"0D/\EV")!`)"0
M_R74(D$`D)#_)=`B00"0D/\ES")!`)"0_R7((D$`D)#_)<0B00"0D/\EP")!
M`)"0_R6\(D$`D)#_);@B00"0D/\EM")!`)"0_R6P(D$`D)#_):PB00"0D/\E
MJ")!`)"0_R6D(D$`D)#_):`B00"0D/\EG")!`)"0_R68(D$`D)#_)90B00"0
MD/\ED")!`)"0_R6L(4$`D)#_)4`B00"0D/\E1")!`)"045`]`!```(U,)`QR
M%8'I`!```(,)`"T`$```/0`0``!WZRG!@PD`6%G#D)#_):0A00"0D/\EJ"%!
M`)"0_R6P(4$`D)#_);0A00"0D/\EN"%!`)"0_R7`(4$`D)#_)>0A00"0D/\E
M["%!`)"0_R7P(4$`D)#_)0`B00"0D/\E!")!`)"0_R4((D$`D)#_)0PB00"0
MD/\ER"%!`)"0_R7,(4$`D)#_)=0A00"0D/\EV"%!`)"0_R4@(D$`D)#_)20B
M00"0D/\E*")!`)"0_R4L(D$`D)#_)3`B00"0D/\EW"%!`)"0_R4X(D$`D)#_
M)4@B00"0D/\E3")!`)"0_R50(D$`D)#_)50B00"0D/\E6")!`)"0_R5<(D$`
MD)#_)>`A00"0D/\E9")!`)"0_R5H(D$`D)#_)6PB00"0D/\E<")!`)"0_R5T
M(D$`D)#_)7@B00"0D/\E?")!`)"0_R6`(D$`D)!FD&:04X/L&,<$)`@```"+
M7"0DZ+P$``"%VP^$W`$``(L5T"%!`(M,)"#'0P2H````QT,(N0L``,=##`<`
M``#'@X``````````QX.$````5`$``,=#+!!D0`#'0S`<9$``QT,41`!!`(E3
M$(E+*,=#2$!A0`#'0TQ084``QT-08&%``,=#5'!A0`#'0R1``$$`QP,`````
MQT,8'%Q``,=#'.Q;0`#'0R!L7$``QT-$D&%``,>#D````*AA0`"%P`^$T```
M`(M`>(7`#X3%````BQ5(`$$`A=(/A;<```"+#2!P0`"%R0^$T`$``(L5)'!`
M`(72#X0Z`@``BPTH<$``A<D/A!P"``"+%2QP0`"%T@^$_@$``(L-,'!``(7)
M#X3@`0``BQ4T<$``A=(/A,(!``"+#3AP0`"%R0^$I`$``(L5/'!``(72#X1^
M`0``BQ4@<$``B1"+%21P0`")4`2+%2AP0`")4`B+%2QP0`")4`R+%3!P0`")
M4!"+%31P0`")4!2+%3AP0`")4!B+%3QP0`")4!S'!"0`````Z(L#``#'0S0`
M<$``@^P$B4-\QT,X0'!``,=#/`#P0`#'0T`P`4$`QX.4````.,%``,>#F```
M`*C#0`#'@YP``````$``QP0D`````.C'`@``@\08N`$```!;P@@`C78`BXBD
M````BQ70(4$`QT`$J````,=`"+D+``")"HM,)"#'0`P'````QX"`````````
M`,>`A````%0!``#'0"P09$``QT`P'&1``,=`%$0`00")2"C'0$A`84``QT!,
M4&%``,=`4&!A0`#'0%1P84``QT`D0`!!`,<``````,=`&!Q<0`#'0!SL6T``
MQT`@;%Q``,=`1)!A0`#'@)````"H84``A<`/A._^__^)PXM`>(7`#X4=_O__
MZ=W^__^-="8`BQ")%2!P0`#I(_[__XUV`(M0'(D5/'!``.ET_O__C;0F````
M`(UV`(M0&(D5.'!``.E._O__9I"+4!2)%31P0`#I,/[__V:0BU`0B14P<$``
MZ1+^__]FD(M0#(D5+'!``.GT_?__9I"+4`B)%2AP0`#IUOW__V:0BU`$B14D
M<$``Z;C]__^0D(/L'(M$)"#'1"0$`````(D$).BI_/__@^P(_Q7$(4$`@^P<
MH5``00")1"0(H50`00")1"0$H5@`00")!"3HW@```(/L#(/$',.-M"8`````
M@^PLBT0D-(M4)#"+3"0X@_@"=%%W2H7`=&;'1"0$,&!``#'`A<F)%"0/E,")
M%5@`00#'!50`00`!````B0U0`$$`HT@`00#H!`$``(/X_Z-,`$$`#Y7`#[;`
M@\0LP@P`@_@#=0N#Q"SI8P```(UV`+@!````@\0LP@P`C70F`)")3"0(QT0D
M!`````")%"3H/````(/L#(7`=+^+%4P`00")1"0<B10DZ(L```"+1"0<QP5,
M`$$`_____^N=C78`Z3O___^0D)"0D)"0D)"0D+@!````P@P`D)"0D)"0D)##
MD)"0D)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0
MD)##D)"0D)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D/\EZ"%!`)"0_R7T
M(4$`D)#_)?@A00"0D/\E/")!`)"04X/L&(M$)"2+7"0@QT0D!&``00")!"3H
M!/O__X/L"(E<)"#'1"0D8`!!`(/$&%OI#````)"0D)"0D)"0D)"0D/\E_"%!
M`)"0_R7H(D$`D)!5B>575E.#Y/"#[$"+10R)1"0LBT40B40D*.B%^?__H;PA
M00")1"0\,<"-1"0HB40D"(U$)"R)1"0$C44(B00DZ*OX___'1"0(0%M``,=$
M)`1`6T``QP0D2%M``.CW^/__@#V((D$``'4NZ&GX__^C+`!!`(7`#X1R`0``
MB00DZ$SX__^AC")!`(D$).C/^/__QH"]!@```*&,(D$`B00DZ+OX__^`B`(#
M```"H8PB00")!"3HI_C__X"(`@,```&+10B->`.-!(40````B00DB7PD).@O
M^/__BTPD+(M]"(G#HR@`00"+`<=#!`&#0`#'0PP$@T``B0.A#'!``(E#"(/_
M`0^.V@```,'G`HU!!(U3$`'YC78`BS"#P`2#P@2)<OPYR'7Q@\<,BT0D),<$
M.P````")7"0,B40D"*$L`$$`QT0D$`````#'1"0$\!=``(D$).A>]___B<.%
MP'1'H2P`00")!"3H6_?__XU4)#BXVX!``.@%KO__A<!T.:$L`$$`B00DZ#3W
M___H5_?__XM$)#PK!;PA00!U78UE](G86UY?7<.A+`!!`(D$).C\]O__B</K
MJ*$D`$$`A<!T"HD$).BGQO__Z[2-5"0XN(N`0`#HIZW__Z,D`$$`A<!TG>O=
MOQ````#I//___\<$)`$```#HXO?__^BU]___D.F+K/__D)"0D)"0D)"0D)#_
M____`&1```````#_____````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````#_____%(-`````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````!C>6=G8V-?<RTQ+F1L;`!?7W)E9VES=&5R7V9R
M86UE7VEN9F\`7U]D97)E9VES=&5R7V9R86UE7VEN9F\`````)7,O)7,`+@`O
M`"]P<F]C+W-E;&8O97AE`$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@``
M26YT97)N86QS.CI005(Z.D)/3U0`4$%27U1%35``4$%27U!23T=.04U%`#H`
M)7,E<R5S`%!%4DPU3$E"`%!%4DQ,24(`4$523#5/4%0`4$523$E/`%!!4E])
M3DE424%,25I%1`!005)?4U!!5TY%1`!005)?0TQ%04X`4$%27T1%0E5'`%!!
M4E]#04-(10!005)?1TQ/0D%,7T1%0E5'`%!!4E]'3$]"04Q?5$U01$E2`%!!
M4E]435!$25(`4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`4$%2
M7T-,14%./0`Q`%!!5$@`55-%4@!365-414T`+W1M<`!435!$25(`5$5-4$1)
M4@!414U0`%1-4`!54T523D%-10`E,#)X`'!A<BT`)7,E<R5S)7,``"5S.B!C
M<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R
M<FYO/25I*0H``"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I<R!U;G-A
M9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E<F%T:6]N
M*0H`<&%R;"YE>&4`+G!A<@`E<R5S8V%C:&4M)7,E<P`E<R5S=&5M<"TE=25S
M`"5S)7-T96UP+25U+25U)7,`&`!P97)L`#``4$%2.CI086-K97(Z.E9%4E-)
M3TX`,2XP-3(``%5N86)L92!T;R!C<F5A=&4@8V%C:&4@9&ER96-T;W)Y````
M`"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A8VAE('-U8F1I<F5C=&]R>2`E
M<R!F86EL960@*&5R<FYO/25I*0H`+64`+2T``'!E<FQX<VDN8P```&5V86P@
M)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C
M(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P
M;#L*;7D@*"1005)?34%'24,L("1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP
M9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.
M?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA
M;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A
M<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@
M9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H
M*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D[
M"G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@
M)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!
M5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N
M86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P*
M*3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@
M/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F
M("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB
M<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I
M;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE
M.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y
M<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S
M<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)
M3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N
M86QS.CI005(Z.D)/3U0["B1005)?34%'24,@/2`B7&Y005(N<&U<;B(["F5V
M86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$
M14)51WT["FUY("5#;VYF:6<@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@
M+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^
M("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L
M:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I
M+`HI.PI?<V5T7W!R;V=N86UE*"D["E]S971?<&%R7W1E;7`H*3L*;7D@*"1S
M=&%R=%]P;W,L("1D871A7W!O<RD["GL*;&]C86P@)%-)1WM?7U=!4DY?7WT@
M/2!S=6(@>WT["F]P96X@7T9(+"`G/#IR87<G+"`D<')O9VYA;64@;W(@;&%S
M=#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D<')O9VYA;64["FUY("1C:'5N
M:U]S:7IE(#T@-C0@*B`Q,#(T.PIM>2`D;6%G:6-?<&]S.PII9B`H)'-I>F4@
M/#T@)&-H=6YK7W-I>F4I('L*)&UA9VEC7W!O<R`](#`["GT@96QS:68@*"AM
M>2`D;2`]("1S:7IE("4@)&-H=6YK7W-I>F4I(#X@,"D@>PHD;6%G:6-?<&]S
M(#T@)'-I>F4@+2`D;3L*?2!E;'-E('L*)&UA9VEC7W!O<R`]("1S:7IE("T@
M)&-H=6YK7W-I>F4["GT*=VAI;&4@*"1M86=I8U]P;W,@/CT@,"D@>PIS965K
M(%]&2"P@)&UA9VEC7W!O<RP@,#L*<F5A9"!?1D@L("1B=68L("1C:'5N:U]S
M:7IE("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X
M*"1B=68L("1005)?34%'24,I*2`^/2`P*2!["B1M86=I8U]P;W,@*ST@)&D[
M"FQA<W0["GT*)&UA9VEC7W!O<R`M/2`D8VAU;FM?<VEZ93L*?0IL87-T(&EF
M("1M86=I8U]P;W,@/"`P.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0L(#`[
M"G)E860@7T9(+"`D8G5F+"`T.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0@
M+2!U;G!A8VLH(DXB+"`D8G5F*2P@,#L*)&1A=&%?<&]S(#T@=&5L;"!?1D@[
M"FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@
M("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H
M:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A
M9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@
M/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN
M72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO
M?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B
M=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L
M(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E
M9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I
M*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L
M(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D
M9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]
M("1F=6QL;F%M93L*?0IE;'-I9B`H("1F=6QL;F%M92`]?B!M?%XO/W-H;&EB
M+WP@86YD(&1E9FEN960@)$5.5GM005)?5$5-4'T@*2!["FUY("1F:6QE;F%M
M92`](%]S879E7V%S*"(D8F%S96YA;64D97AT(BP@)&)U9BP@,#<U-2D["F]U
M=',H(E-(3$E".B`D9FEL96YA;65<;B(I.PI]"F5L<V4@>PHD<F5Q=6ER95]L
M:7-T>R1F=6QL;F%M97T@/0HD36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*
M8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*
M?3L*?0IR96%D(%]&2"P@)&)U9BP@-#L*?0IL;V-A;"!`24Y#(#T@*'-U8B![
M"FUY("@D<V5L9BP@)&UO9'5L92D@/2!`7SL*<F5T=7)N(&EF(')E9B`D;6]D
M=6QE(&]R("$D;6]D=6QE.PIM>2`D:6YF;R`](&1E;&5T92`D<F5Q=6ER95]L
M:7-T>R1M;V1U;&5](&]R(')E='5R;CL*)$E.0WLD;6]D=6QE?2`]("(O;&]A
M9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D
M969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^
M;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D
M(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D[
M"B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I
M;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[
M8G5F?2D["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE
M('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*
M?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L
M92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U
M:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@
M17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R
M*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H
M<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD
M9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I
M;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*
M96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E
M8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I
M;&5N86UE?3L*?0IL87-T('5N;&5S<R`D8G5F(&5Q(")02UPP,#-<,#`T(CL*
M)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C
M('-T87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD
M;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!0
M7U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@
M<')O8FQE;2D*)'%U:65T(#T@,"!U;FQE<W,@)$5.5GM005)?1$5"54=].PII
M9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS
M)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O
M7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P87(*<R`@('-I
M9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF("A`05)'5B!A;F0@)$%21U9;
M,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?
M4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS92![(",@;F]R;6%L('!A<FP@
M8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW:&EL92`H0$%21U8I('L*)$%2
M1U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*
M:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I
M9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q
M("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q
M("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N
M9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L92`]
M("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE
M;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D
M:64@<7%;0V%N)W0@;W!E;B!L;V<@9FEL92`B)#(B.B`D(5T["GT*96QS:68@
M*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H:69T*$!!4D=6
M*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2!["F1E;&5T92`D14Y6
M>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*
M)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ
M1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X:70["GT*?0IU;G-H:69T
M($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I('L*>PIR97%U:7)E
M($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E
M<W0Z.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII
M9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R
M(&]R(&1I92!Q<5M#86XG="!F:6YD('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB
M;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN
M97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@
M07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I
M;F<@>FEP(&%R8VAI=F4@(B1P87(B73L*?0IM>2`E96YV(#T@9&\@>PII9B`H
M)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG
M*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N
M*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII
M9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R
M(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM
M>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N
M;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@(E!+7#`P,UPP,#0B.PIC
M;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&
M:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL
M93HZ3U]74D].3%DH*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@
M;W(@9&EE('%Q6T-A;B=T(&-R96%T92!F:6QE("(D;W5T(CH@)"%=.PHD9F@M
M/F)I;FUO9&4H*3L*<V5E:R!?1D@L(#`L(#`["FUY("1L;V%D97(["FEF("AD
M969I;F5D("1D871A7W!O<RD@>PIR96%D(%]&2"P@)&QO861E<BP@)&1A=&%?
M<&]S.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&
M2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)37T@86YD("1L;V%D97(@/7X@
M+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T
M<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D
M,"D["GT*9F]R96%C:"!M>2`D:V5Y("AS;W)T(&ME>7,@)65N=BD@>PIM>2`D
M=F%L(#T@)&5N=GLD:V5Y?2!O<B!N97AT.PHD=F%L(#T@979A;"`D=F%L(&EF
M("1V86P@/7X@+UY;)R)=+SL*;7D@)&UA9VEC(#T@(E]?14Y67U!!4E\B("X@
M=6,H)&ME>2D@+B`B7U\B.PIM>2`D<V5T(#T@(E!!4E\B("X@=6,H)&ME>2D@
M+B`B/21V86PB.PHD;&]A9&5R(#U^('-[)&UA9VEC*"`K*7U["B1M86=I8R`N
M("1S970@+B`H)R`G('@@*&QE;F=T:"@D,2D@+2!L96YG=&@H)'-E="DI*0I]
M96<["GT*)&9H+3YP<FEN="@D;&]A9&5R*3L*:68@*"1B=6YD;&4I('L*<F5Q
M=6ER92!005(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R
M*"D["FEN:71?:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@
M9W)E<"![("$O0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I
M(&]R"B@D7R!N92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?
M(&YE("1#;VYF:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*<WPO
M*WPO?&<L('-\+R1\?"!F;W)E86-H($!I;F,["FUY("5F:6QE<SL*)&9I;&5S
M>R1??2LK(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A
M;'5E<R`E24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?
M97AT?3L@("`@("`@("`C(%A86"!L:6)?97AT('9S(&1L97AT(#\*;7D@)7=R
M:71T96X["F9O<F5A8V@@;7D@)&ME>2`H<V]R="!K97ES("5F:6QE<RD@>PIM
M>2`H)&9I;&4L("1N86UE*3L*:68@*&1E9FEN960H;7D@)&9C(#T@)%!!4CHZ
M2&5A=GDZ.D9U;&Q#86-H97LD:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@/2`H
M)&ME>2P@)&9C*3L*?0IE;'-E('L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2![
M"FEF("@D:V5Y(#U^(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE+"`D
M;F%M92D@/2`H)&ME>2P@)#$I.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\7B]L
M;V%D97(O6UXO72LO*"XJ*21\*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L94-A
M8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0[
M"GT*:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I
M<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*?0IN97AT('5N;&5S<R!D969I;F5D
M("1N86UE.PIN97AT(&EF("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R
M968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+VD["F]U
M=',H<W!R:6YT9BAQ<5M086-K:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D9FEL
M92`_("1F:6QE+3Y[;F%M97T@.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*:68@
M*')E9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS
M92!["FQO8V%L("0O(#T@=6YD968["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D
M9FEL92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C;VYT
M96YT(#T@/"1F:#X["F-L;W-E("1F:#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M+3YA<'!L>2A<)&-O;G1E;G0L("(\96UB961D960^+R1N86UE(BD*:68@(21%
M3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI
M)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY*%PD
M8V]N=&5N="P@)&9I;&4L("1N86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%(BP*
M<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X>"\E
M<R(L($%R8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA
M;64I+`IP86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT*3L*
M;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"B1Z:7`M/G=R:71E
M5&]&:6QE2&%N9&QE*"1F:"D@:68@)'II<#L*)&-A8VAE7VYA;64@/2!S=6)S
M='(@)&-A8VAE7VYA;64L(#`L(#0P.PII9B`H(21C86-H95]N86UE(&%N9"!M
M>2`D;71I;64@/2`H<W1A="@D;W5T*2E;.5TI('L*;7D@)&-T>"`]($1I9V5S
M=#HZ4TA!+3YN97<H,2D["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D;W5T.PHD
M8W1X+3YA9&1F:6QE*"1F:"D["F-L;W-E("1F:#L*)&-A8VAE7VYA;64@/2`D
M8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@+CT@(EPP(B!X("@T,2`M
M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1C86-H95]N86UE("X](")#04-(12([
M"B1F:"T^<')I;G0H)&-A8VAE7VYA;64I.PHD9F@M/G!R:6YT*'!A8VLH)TXG
M+"`D9F@M/G1E;&P@+2!L96YG=&@H)&QO861E<BDI*3L*)&9H+3YP<FEN="@D
M4$%27TU!1TE#*3L*)&9H+3YC;&]S93L*8VAM;V0@,#<U-2P@)&]U=#L*97AI
M=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P
M<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN
M86QO861E<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H
M:79E.CI::7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@
M("`@("`@("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N
M(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@
M;W(@9&EE('%Q6V9D;W!E;B@I(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z
M.G-E=$-H=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP
M+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N
M86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO<B!D:64@<7%;17)R
M;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!R;V=N86UE(ET["D%R8VAI=F4Z
M.EII<#HZ<V5T0VAU;FM3:7IE*#8T("H@,3`R-"D["G!U<V@@0%!!4CHZ3&EB
M0V%C:&4L("1Z:7`["B1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D>FEP
M.PHD<75I970@/2`A)$5.5GM005)?1$5"54=].PIO=71S*'%Q6UPD14Y6>U!!
M4E]414U0?2`]("(D14Y6>U!!4E]414U0?2)=*3L*:68@*&1E9FEN960@)$5.
M5GM005)?5$5-4'TI('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A
M"F9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@
M:68@)&UE;6)E<BT^:7-$:7)E8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M
M96UB97(M/F9I;&5.86UE.PIN97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@
M;7L*7@HO/W-H;&EB+PHH/SHD0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\*
M*#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@[
M"FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z
M.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I
M.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M
M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA
M;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI
M.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@
M=&\@(B1D97-T7VYA;64B72D["B1M96UB97(M/F5X=')A8W14;T9I;&5.86UE
M9"@D9&5S=%]N86UE*3L*8VAM;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/
M(&5Q(")H<'5X(CL*?0I]"GT*?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P
M<F]G;F%M97TI('L*9&EE(#P\("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D
M,"!;("U!;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N
M<&%R(%T@6R!P<F]G<F%M+G!L(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=
M('-R8RYP87(*+@HD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D
M,"`]('-H:69T*$!!4D=6*3L*?0IS=6(@0W)E871E4&%T:"!["FUY("@D;F%M
M92D@/2!`7SL*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*;7D@*"1B87-E;F%M
M92P@)'!A=&@L("1E>'0I(#T@1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S92@D
M;F%M92P@*"=<+BXJ)RDI.PIR97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A
M=&@Z.FUK<&%T:"@D<&%T:"D@=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@
M9&EE<R!W:71H(&5R<F]R"GT*<W5B(')E<75I<F5?;6]D=6QE<R!["G)E<75I
M<F4@;&EB.PIR97%U:7)E($1Y;F%,;V%D97(["G)E<75I<F4@:6YT96=E<CL*
M<F5Q=6ER92!S=')I8W0["G)E<75I<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R
M<SL*<F5Q=6ER92!#87)P.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E
M($5R<FYO.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O
M<G1E<CL*<F5Q=6ER92!&8VYT;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U
M:7)E($9I;&4Z.E-P96,["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@0V]N
M9FEG.PIR97%U:7)E($E/.CI(86YD;&4["G)E<75I<F4@24\Z.D9I;&4["G)E
M<75I<F4@0V]M<')E<W,Z.EIL:6(["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR
M97%U:7)E($1I9V5S=#HZ4TA!.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z
M.DAE879Y.PIR97%U:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E
M<CHZ4&]D4W1R:7`["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N
M=#L*<F5Q=6ER92!A='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*
M979A;"![(')E<75I<F4@5VEN,S(@?3L*979A;"![(')E<75I<F4@4V-A;&%R
M.CI5=&EL('T["F5V86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S
M="!].PIE=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!]
M.PIE=F%L('L@<F5Q=6ER92!097)L24\[(')E<75I<F4@4&5R;$E/.CIS8V%L
M87(@?3L*979A;"![(')E<75I<F4@=71F."!].PI]"G-U8B!?<V5T7W!A<E]T
M96UP('L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%2
M7U1%35!](#U^("\H+BLI+RD@>PHD<&%R7W1E;7`@/2`D,3L*<F5T=7)N.PI]
M"F9O<F5A8V@@;7D@)'!A=&@@*`HH;6%P("1%3E9[)%]]+"!Q=R@@4$%27U1-
M4$1)4B!435!$25(@5$5-4$1)4B!414U0(%1-4"`I*2P*<7<H($,Z7%Q414U0
M("]T;7`@+B`I"BD@>PIN97AT('5N;&5S<R!D969I;F5D("1P871H(&%N9"`M
M9"`D<&%T:"!A;F0@+7<@)'!A=&@["FUY("1U<V5R;F%M93L*;7D@)'!W=6ED
M.PIE=F%L('LH)'!W=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D969I;F5D("0^
M.WT["FEF("@@9&5F:6YE9"@F5VEN,S(Z.DQO9VEN3F%M92D@*2!["B1U<V5R
M;F%M92`]("97:6XS,CHZ3&]G:6Y.86UE.PI]"F5L<VEF("AD969I;F5D("1P
M=W5I9"D@>PHD=7-E<FYA;64@/2`D<'=U:60["GT*96QS92!["B1U<V5R;F%M
M92`]("1%3E9[55-%4DY!345]('Q\("1%3E9[55-%4GT@?'P@)U-94U1%32<[
M"GT*)'5S97)N86UE(#U^(',O7%<O7R]G.PIM>2`D<W1M<&1I<B`]("(D<&%T
M:"1#;VYF:6=[7V1E;&EM?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA;64I
M.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*:68@*"$D14Y6>U!!4E]#3$5!3GT@
M86YD(&UY("1M=&EM92`]("AS=&%T*"1P<F]G;F%M92DI6SE=*2!["F]P96X@
M;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E
M860@(B1P<F]G;F%M92(Z("0A73L*<V5E:R`D9F@L("TQ."P@,CL*;7D@)&)U
M9CL*<F5A9"`D9F@L("1B=68L(#8["FEF("@D8G5F(&5Q(")<,$-!0TA%(BD@
M>PIS965K("1F:"P@+34X+"`R.PIR96%D("1F:"P@)&)U9BP@-#$["B1B=68@
M/7X@<R]<,"\O9SL*)'-T;7!D:7(@+CT@(B1#;VYF:6=[7V1E;&EM?6-A8VAE
M+21B=68B.PI]"F5L<V4@>PIM>2`D9&EG97-T(#T@979A;`I["G)E<75I<F4@
M1&EG97-T.CI32$$["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIO
M<&5N(&UY("1F:"P@(CPZ<F%W(BP@)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG
M="!R96%D("(D<')O9VYA;64B.B`D(5T["B1C='@M/F%D9&9I;&4H)&9H*3L*
M8VQO<V4H)&9H*3L*)&-T>"T^:&5X9&EG97-T.PI]('Q\("1M=&EM93L*)'-T
M;7!D:7(@+CT@(B1#;VYF:6=[7V1E;&EM?6-A8VAE+21D:6=E<W0B.PI]"F-L
M;W-E*"1F:"D["GT*96QS92!["B1%3E9[4$%27T-,14%.?2`](#$["B1S=&UP
M9&ER("X]("(D0V]N9FEG>U]D96QI;7UT96UP+20D(CL*?0HD14Y6>U!!4E]4
M14U0?2`]("1S=&UP9&ER.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*;&%S=#L*
M?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM0
M05)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M
M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P
M87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T
M96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIO<&5N(&UY("1F:"P@)SXZ<F%W
M)RP@)'1E;7!N86UE(&]R(&1I92!Q<5M#86XG="!W<FET92`B)'1E;7!N86UE
M(CH@)"%=.PIP<FEN="`D9F@@)&-O;G1E;G1S.PIC;&]S92`D9F@["F-H;6]D
M("1M;V1E+"`D=&5M<&YA;64@:68@9&5F:6YE9"`D;6]D93L*<F5N86UE*"1T
M96UP;F%M92P@)&9U;&QN86UE*2!O<B!U;FQI;FLH)'1E;7!N86UE*3L*?0IR
M971U<FX@)&9U;&QN86UE.PI]"G-U8B!?<V5T7W!R;V=N86UE('L*:68@*&1E
M9FEN960@)$5.5GM005)?4%)/1TY!345](&%N9"`D14Y6>U!!4E]04D]'3D%-
M17T@/7X@+R@N*RDO*2!["B1P<F]G;F%M92`]("0Q.PI]"B1P<F]G;F%M92!\
M?#T@)#`["FEF("@D14Y6>U!!4E]414U0?2!A;F0@:6YD97@H)'!R;V=N86UE
M+"`D14Y6>U!!4E]414U0?2D@/CT@,"D@>PHD<')O9VYA;64@/2!S=6)S='(H
M)'!R;V=N86UE+"!R:6YD97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI
M("L@,2D["GT*:68@*"$D14Y6>U!!4E]04D]'3D%-17T@;W(@:6YD97@H)'!R
M;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI(#X](#`I('L*:68@*&]P96X@;7D@
M)&9H+"`G/"<L("1P<F]G;F%M92D@>PIR971U<FX@:68@+7,@)&9H.PI]"FEF
M("@M<R`B)'!R;V=N86UE)$-O;F9I9WM?97AE?2(I('L*)'!R;V=N86UE("X]
M("1#;VYF:6=[7V5X97T["G)E='5R;CL*?0I]"F9O<F5A8V@@;7D@)&1I<B`H
M<W!L:70@+UQ1)$-O;F9I9WMP871H7W-E<'U<12\L("1%3E9[4$%42'TI('L*
M;F5X="!I9B!E>&ES=',@)$5.5GM005)?5$5-4'T@86YD("1D:7(@97$@)$5.
M5GM005)?5$5-4'T["B1D:7(@/7X@<R]<421#;VYF:6=[7V1E;&EM?5Q%)"\O
M.PHH*"1P<F]G;F%M92`]("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE
M)$-O;F9I9WM?97AE?2(I+"!L87-T*0II9B`M<R`B)&1I<B1#;VYF:6=[7V1E
M;&EM?21P<F]G;F%M921#;VYF:6=[7V5X97TB.PHH*"1P<F]G;F%M92`]("(D
M9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE(BDL(&QA<W0I"FEF("US("(D
M9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE(CL*?0I]"G-U8B!?9FEX7W!R
M;V=N86UE('L*)#`@/2`D<')O9VYA;64@?'P]("1%3E9[4$%27U!23T=.04U%
M?3L*:68@*&EN9&5X*"1P<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`\(#`I
M('L*)'!R;V=N86UE(#T@(BXD0V]N9FEG>U]D96QI;7TD<')O9VYA;64B.PI]
M"FUY("1P=V0@/2`H9&5F:6YE9"`F0W=D.CIG971C=V0I(#\@0W=D.CIG971C
M=V0H*0HZ("@H9&5F:6YE9"`F5VEN,S(Z.D=E=$-W9"D@/R!7:6XS,CHZ1V5T
M0W=D*"D@.B!@<'=D8"D["F-H;VUP*"1P=V0I.PHD<')O9VYA;64@/7X@<R]>
M*#\]7"Y<+C]<421#;VYF:6=[7V1E;&EM?5Q%*2\D<'=D)$-O;F9I9WM?9&5L
M:6U]+SL*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE.PI]"G-U8B!?
M<&%R7VEN:71?96YV('L*:68@*"`D14Y6>U!!4E])3DE424%,25I%1'TK*R`]
M/2`Q("D@>PIR971U<FX["GT@96QS92!["B1%3E9[4$%27TE.251)04Q)6D5$
M?2`](#(["GT*9F]R("AQ=R@@4U!!5TY%1"!414U0($-,14%.($1%0E5'($-!
M0TA%(%!23T=.04U%("D@*2!["F1E;&5T92`D14Y6>R=005)?)RXD7WT["GT*
M9F]R("AQ=R\@5$U01$E2(%1%35`@0TQ%04X@1$5"54<@+RD@>PHD14Y6>R=0
M05)?)RXD7WT@/2`D14Y6>R=005)?1TQ/0D%,7R<N)%]](&EF(&5X:7-T<R`D
M14Y6>R=005)?1TQ/0D%,7R<N)%]].PI]"FUY("1P87)?8VQE86X@/2`B7U]%
M3E9?4$%27T-,14%.7U\@("`@("`@("`@("`@("`B.PII9B`H)$5.5GM005)?
M5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,14%.?3L*?0IE;'-I9B`H(65X
M:7-T<R`D14Y6>U!!4E]'3$]"04Q?0TQ%04Y]*2!["FUY("1V86QU92`]('-U
M8G-T<B@D<&%R7V-L96%N+"`Q,B`K(&QE;F=T:"@B0TQ%04XB*2D["B1%3E9[
M4$%27T-,14%.?2`]("0Q(&EF("1V86QU92`]?B`O7E!!4E]#3$5!3CTH7%,K
M*2\["GT*?0IS=6(@;W5T<R!["G)E='5R;B!I9B`D<75I970["FEF("@D;&]G
M9F@I('L*<')I;G0@)&QO9V9H(")`7UQN(CL*?0IE;'-E('L*<')I;G0@(D!?
M7&XB.PI]"GT*<W5B(&EN:71?:6YC('L*<F5Q=6ER92!#;VYF:6<["G!U<V@@
M0$E.0RP@9W)E<"!D969I;F5D+"!M87`@)$-O;F9I9SHZ0V]N9FEG>R1??2P@
M<7<H"F%R8VAL:6)E>'`@<')I=FQI8F5X<"!S:71E87)C:&5X<"!S:71E;&EB
M97AP"G9E;F1O<F%R8VAE>'`@=F5N9&]R;&EB97AP"BD["GT*<&%C:V%G92!M
M86EN.PIR97%U:7)E(%!!4CL*=6YS:&EF="!`24Y#+"!<)E!!4CHZ9FEN9%]P
M87(["E!!4BT^:6UP;W)T*$!P87)?87)G<RD["F1I92!Q<5MP87(N<&PZ($-A
M;B=T(&]P96X@<&5R;"!S8W)I<'0@(B1P<F]G;F%M92(Z($YO('-U8V@@9FEL
M92!O<B!D:7)E8W1O<GE<;ET*=6YL97-S("UE("1P<F]G;F%M93L*9&\@)'!R
M;V=N86UE.PI#3U)%.CIE>&ET*"0Q*2!I9B`H)$`@/7XO7E]42U]%6$E47UPH
M*%QD*RE<*2\I.PID:64@)$`@:68@)$`["GT["B0Z.E]?15)23U(@/2`D0"!I
M9B`D0#L*?0I#3U)%.CIE>&ET*"0Q*2!I9B`H)#HZ7U]%4E)/4B`]?B]>7U1+
M7T58251?7"@H7&0K*5PI+RD["F1I92`D.CI?7T524D]2(&EF("0Z.E]?15)2
M3U(["C$["E]?14Y$7U\*``"`80``1T-#.B`H1TY5*2`Y+C,N,"`R,#(P,#,Q
M,B`H1F5D;W)A($-Y9W=I;B`Y+C,N,"TQ*0```$=#0SH@*$=.52D@,3`N,BXP
M````1T-#.B`H1TY5*2`Q,"XR+C````!'0T,Z("A'3E4I(#$P+C(N,````$=#
M0SH@*$=.52D@.2XS+C`@,C`R,#`S,3(@*$9E9&]R82!#>6=W:6X@.2XS+C`M
M,2D```!'0T,Z("A'3E4I(#DN,RXP(#(P,C`P,S$R("A&961O<F$@0WEG=VEN
M(#DN,RXP+3$I````1T-#.B`H1TY5*2`Y+C,N,"`R,#(P,#,Q,B`H1F5D;W)A
M($-Y9W=I;B`Y+C,N,"TQ*0```$=#0SH@*$=.52D@.2XS+C`@,C`R,#`S,3(@
M*$9E9&]R82!#>6=W:6X@.2XS+C`M,2D```!'0T,Z("A'3E4I(#DN,RXP(#(P
M,C`P,S$R("A&961O<F$@0WEG=VEN(#DN,RXP+3$I````1T-#.B`H1TY5*2`Y
M+C,N,"`R,#(P,#,Q,B`H1F5D;W)A($-Y9W=I;B`Y+C,N,"TQ*0```$=#0SH@
M*$=.52D@.2XS+C`@,C`R,#`S,3(@*$9E9&]R82!#>6=W:6X@.2XS+C`M,2D`
M``!'0T,Z("A'3E4I(#DN,RXP(#(P,C`P,S$R("A&961O<F$@0WEG=VEN(#DN
M,RXP+3$I````1T-#.B`H1TY5*2`Y+C,N,"`R,#(P,#,Q,B`H1F5D;W)A($-Y
M9W=I;B`Y+C,N,"TQ*0```$=#0SH@*$=.52D@.2XS+C`@,C`R,#`S,3(@*$9E
M9&]R82!#>6=W:6X@.2XS+C`M,2D```!'0T,Z("A'3E4I(#$P+C(N,````$=#
M0SH@*$=.52D@,3`N,BXP```````````````!````O"$!`$T2```@````O"$!
M`&82```@````O"$!`)H3```@````O"$!`/@3```@````O"$!`"`5```@````
MC"(!`#T5```@````O"$!`(\5```@````C"(!`*H5```@````O"$!`-P5```@
M````O"$!`,46```@````C"(!`/07```@````C"(!``$8```@````C"(!`"X8
M```@````C"(!`'08```@````O"$!`-$8```@````O"$!`"D9```@````O"$!
M`!0;```@````O"$!`(\>```@````O"$!`/0>```@````O"$!``P?```@````
MO"$!`(<?```@````O"$!`+0?```@````O"$!`.8?```@````O"$!`#X@```@
M````O"$!`(8@```@````O"$!`+DB```@````O"$!`#TE```@````O"$!`.8G
M```@````C"(!`.\G```@````C"(!`/XG```@````C"(!`#(H```@````C"(!
M`&HH```@````C"(!`)HH```@````C"(!`.HH```@````C"(!``<I```@````
MC"(!`"XI```@````C"(!`$\I```@````O"$!`)TI```@````C"(!`/TI```@
M````C"(!`"$J```@````O"$!`'<J```@````O"$!`.`J```@````O"$!`!@K
M```@````O"$!`"!B```@````B"(!`&-B```@````C"(!`(5B```@````C"(!
M`)EB```@````C"(!`*UB```@````O"$!`)=C```@````O"$!`*\K```@````
MO"$!`%Y6```@````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````(````9````'-```!R@``!24T13@O7@
MX4"?%FJ,<.HSG6ZI!0$`````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````%``````````!>E(``7P(`1L,
M!`2(`0``'````!P```#@+___AP````!!#@B%`D(-!0*#Q0P$!``4````````
M``%Z4@`!?`@!&PP$!(@!```L````'````#@P__^Q`````$$."(4"0@T%1H<#
MA@2#!0*1"L-!QD''0<4,!`1#"P`<````3````,@P__\O`````$$."(4"0@T%
M:\4,!`0``!0``````````7I2``%\"`$;#`0$B`$``&0````<````P##__[0`
M````00X(A0)!#@R'`T$.$(8$00X4@P5##D`"80H.%$/##A!!Q@X,0<<."$'%
M#@1""VX*#A1#PPX00<8.#$''#@A!Q0X$1`M##A1#PPX00\8.#$''#@A!Q0X$
M/````(0````8,?__1@$```!!#@B%`D$.#(<#00X0A@1!#A2#!4@.H`%J"@X4
M0<,.$$'&#@Q!QPX(0<4.!$8+`"0```#$````*#+__WL`````00X(@P)(#I`!
M`FL*#@A!PPX$00L````T````[````(`R___5`````$$."(,"10X@`EP*#@A#
MPPX$2PL"0@H."$/##@1!"UD."$/##@0``$`````D`0``*#/__]T`````00X(
MA0)!#@R'`T$.$(8$00X4@P5!#AA,#M`@`IH*#A1!PPX00<8.#$''#@A!Q0X$
M00L`/````&@!``#$,___&`(```!!#@B%`D$.#(<#0PX0A@1!#A2#!44.4`+W
M"@X40<,.$$'&#@Q!QPX(0<4.!$H+`!0```"H`0``I#7__WX`````0PXP`GH.
M!!0```#``0``##;__S0`````0PX@<`X$`!````#8`0``-#;__P,`````````
M/````.P!```P-O__X@$```!!#@B%`D8.#(<#00X0A@1!#A2#!4<.P"$":0H.
M%$'##A!!Q@X,0<<."$'%#@1#"Q`````L`@``X#?__R,`````````$````$`"
M``#\-___)0`````````8````5`(``!@X___1`P```$,.0`.%`PH.!$$+&```
M`'`"``#<.___9`````!##B!B"@X$2PL``!0```",`@``,#S__QH`````0PX@
M5@X$`$0```"D`@``.#S__]``````00X(AP)!#@R&`T$.$(,$0PY`?0H.$$;#
M#@Q!Q@X(0<<.!$4+7PH.$$'##@Q!Q@X(0<<.!$8+`%@```#L`@``P#S__X\'
M````00X(A0)&#@R'`T$.$(8$00X4@P5'#K""`@-I`@H.%$/##A!!Q@X,0<<.
M"$'%#@1!"P.(`@H.%$'##A!!Q@X,0<<."$'%#@1,"P``+````$@#``#T0___
MAP(```!!#@B&`D$.#(,#0PXP`\4!"@X,0<,."$'&#@1$"P``1````'@#``!4
M1O__(0$```!!#@B'`D$.#(8#00X0@P1##D`"=PH.$$'##@Q!Q@X(0<<.!$@+
M:0H.$$/##@Q!Q@X(0<<.!$H++````,`#``"<??___P$```!!#@B%`D(-!4F'
M`X8$@P4#EP$*PT'&0<=!Q0P$!$$+%``````````!>E(``7P(`1L,!`2(`0``
M0````!P```#T1O__U"H```!!#@B%`D0.#(<#00X0A@1!#A2#!48.H`8#O2H*
M#A1!PPX00<8.#$''#@A!Q0X$00L````4````8````)!Q__][`````$,.(`)W
M#@14````>````/AQ___$`0```$$."(4"00X,AP-!#A"&!$$.%(,%0PY``ID*
M#A1!PPX00<8.#$''#@A!Q0X$1`L"\0H.%$'##A!!Q@X,0<<."$'%#@1#"P``
M/````-````!P<___TP$```!!#@B%`D$.#(<#00X0A@1!#A2#!4,.,`-1`0H.
M%$'##A!!Q@X,0<<."$'%#@1,"Q0``````````7I2``%\"`$;#`0$B`$``"@`
M```<````&'?__SX#````00X(@P)##B`#F0$.'$H.(`)%"@X(1L,.!$8+%```
M```````!>E(``7P(`1L,!`2(`0``&````!P````4>O__(`````!##B!4#AA#
M#B```!0``````````7I2``%\"`$;#`0$B`$``!P````<`````'K__RD`````
M0PX@7PX40PX@0PX$````+````#P````0>O__O0````!##C`"6@H.!$,+2`H.
M!$@+2`H.!$@+5`XD0PXP````$````&P```"@>O__!0`````````4````````
M``%Z4@`!?`@!&PP$!(@!```0````'````(1Z__\(`````````!0`````````
M`7I2``%\"`$;#`0$B`$``!`````<````:'K__P$`````````%``````````!
M>E(``7P(`1L,!`2(`0``$````!P```!,>O__`0`````````4``````````%Z
M4@`!?`@!&PP$!(@!```0````'````#!Z__\!`````````!0``````````7I2
M``%\"`$;#`0$B`$``!`````<````%'K__P$`````````%``````````!>E(`
M`7P(`1L,!`2(`0``$````!P```#X>?__`0`````````4``````````%Z4@`!
M?`@!&PP$!(@!```D````'````/QY__\T`````$$."(,"0PX@6`X80PX@3PX(
M0<,.!````````!0``````````7I2``%\"`$;#`0$B`$``!`````<````"'S_
M_P4`````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````#>YQ]@`````.80`0`!````$P``
M`!,````H$`$`=!`!`,`0`0#@)P``S"4!`-PE`0!@'P```&(```QP``"P&@``
M<"H``+`8``#@&@``\!X``,`8```0&P``4"```(`?``#06```@%8```!7``!P
M&```[A`!``01`0`6$0$`)A$!`$$1`0!&$0$`4!$!`%T1`0!I$0$`>A$!`(81
M`0"4$0$`H1$!`*X1`0"[$0$`S1$!`-<1`0#@$0$`ZQ$!`````0`"``,`!``%
M``8`!P`(``D`"@`+``P`#0`.``\`$``1`!(`<&%R+F5X90!84U]);G1E<FYA
M;'-?4$%27T)/3U0`7VYM7U]03%]D;U]U;F1U;7``7VYM7U]03%]T:')?:V5Y
M`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`&UA:6X`;7E?<&%R7W!L`'!A
M<E]B87-E;F%M90!P87)?8VQE86YU<`!P87)?8W5R<F5N=%]E>&5C`'!A<E]D
M:7)N86UE`'!A<E]E;G9?8VQE86X`<&%R7V9I;F1P<F]G`'!A<E]I;FET7V5N
M=@!P87)?;6MT;7!D:7(`<&%R7W-E='5P7VQI8G!A=&@`<VAA7V9I;F%L`'-H
M85]I;FET`'-H85]U<&1A=&4`>'-?:6YI=```````````````````4"`!````
M````````?"@!`*0A`0`T(0$```````````#@*`$`B"(!`)`A`0``````````
M```I`0#D(@$```````````````````````````#X(@$`"",!`!0C`0`D(P$`
M,",!`$`C`0!4(P$`:",!`'@C`0"((P$`F",!`*0C`0"T(P$`P",!`,PC`0#8
M(P$`Y",!`/`C`0#\(P$`!"0!`!`D`0`D)`$`."0!`$@D`0!0)`$`7"0!`&0D
M`0!P)`$`?"0!`(@D`0"0)`$`F"0!`*0D`0"P)`$`O"0!`,@D`0#0)`$`V"0!
M`.0D`0#X)`$`#"4!`"0E`0`L)0$`."4!`$0E`0!0)0$`6"4!`&0E`0!L)0$`
M>"4!`(0E`0"0)0$`G"4!`*@E`0"T)0$`P"4!``````#,)0$`W"4!`.PE`0``
M)@$`&"8!`#`F`0!`)@$`5"8!`&0F`0!T)@$`B"8!`*`F`0"P)@$`Q"8!`-@F
M`0#H)@$`_"8!``PG`0`@)P$`,"<!`#PG`0!,)P$``````%@G`0!F)P$`>B<!
M`(PG`0``````^"(!``@C`0`4(P$`)",!`#`C`0!`(P$`5",!`&@C`0!X(P$`
MB",!`)@C`0"D(P$`M",!`,`C`0#,(P$`V",!`.0C`0#P(P$`_",!``0D`0`0
M)`$`)"0!`#@D`0!()`$`4"0!`%PD`0!D)`$`<"0!`'PD`0"()`$`D"0!`)@D
M`0"D)`$`L"0!`+PD`0#()`$`T"0!`-@D`0#D)`$`^"0!``PE`0`D)0$`+"4!
M`#@E`0!$)0$`4"4!`%@E`0!D)0$`;"4!`'@E`0"$)0$`D"4!`)PE`0"H)0$`
MM"4!`,`E`0``````S"4!`-PE`0#L)0$``"8!`!@F`0`P)@$`0"8!`%0F`0!D
M)@$`="8!`(@F`0"@)@$`L"8!`,0F`0#8)@$`Z"8!`/PF`0`,)P$`("<!`#`G
M`0`\)P$`3"<!``````!8)P$`9B<!`'HG`0",)P$``````!L`7U]C>&%?871E
M>&ET```I`%]?97)R;F\```!(`%]?9V5T<F5E;G0`````4P!?7VUA:6X`````
M<0!?7W-P<FEN=&9?8VAK`'0`7U]S=&%C:U]C:&M?9F%I;```=@!?7W-T86-K
M7V-H:U]G=6%R9`!\`%]?<W1R;F-P>5]C:&L`X0!?9&QL7V-R=#!`,````(8!
M7V=E='!W=6ED,S(```"-`5]G971U:60S,@":`5]I;7!U<F5?<'1R````U0%?
M;'-E96LV-```UP%?;'-T870V-```]P%?;W!E;C8T````7@)?<W1A=#8T````
MQ@)A8V-E<W,`````10-C86QL;V,`````A@-C;&]S90"'`V-L;W-E9&ER``#+
M`V-Y9W=I;E]D971A8VA?9&QL`,T#8WEG=VEN7VEN=&5R;F%L````[0-D;&Q?
M9&QL8W)T,````"<$97AI=```D@1F<')I;G1F````G@1F<F5E```&!6=E='!I
M9``````2!6=E='!W=6ED```@!6=E='5I9`````#E!6QS965K`.<%;'-T870`
M[`5M86QL;V,`````^P5M96UC:'(`````_05M96UC<'D`````_P5M96UM;W9E
M`````P9M:V1I<@`_!F]P96X``$,&;W!E;F1I<@```%8&<&]S:7A?;65M86QI
M9VX`````?P9P=&AR96%D7V%T9F]R:P````"T!G!T:')E861?9V5T<W!E8VEF
M:6,````)!W)E860```H'<F5A9&1I<@````P'<F5A9&QI;FL```\'<F5A;&QO
M8P```#T'<FUD:7(`W`=S<')I;G1F````Y0=S=&%T``#O!W-T<F-H<@````#Q
M!W-T<F-M<`````#V!W-T<F1U<```````"'-T<FQE;@`````%"'-T<FYC;7``
M```1"'-T<G-T<@`````8"'-T<G1O:P````!^"'5N;&EN:P`````7`%!,7V1O
M7W5N9'5M<```;0!03%]T:')?:V5Y`````&L!4&5R;%]A=&9O<FM?;&]C:P``
M;`%097)L7V%T9F]R:U]U;FQO8VL`````Y0%097)L7V-R;V%K7VYO8V]N=&5X
M=```>`)097)L7V=E=%]S=@```*<"4&5R;%]G=E]F971C:'!V````:0-097)L
M7VUG7W-E=````-\#4&5R;%]N97=84P````#A`U!E<FQ?;F5W6%-?9FQA9W,`
M`*<%4&5R;%]S869E<WES;6%L;&]C`````(8&4&5R;%]S=E]S971P=@"/!E!E
M<FQ?<W9?<V5T<'9N`````+L&4&5R;%]S>7-?:6YI=#,`````O`9097)L7W-Y
M<U]T97)M`#D'8F]O=%]$>6YA3&]A9&5R````2P=P97)L7V%L;&]C`````$T'
M<&5R;%]C;VYS=')U8W0`````3@=P97)L7V1E<W1R=6-T`$\'<&5R;%]F<F5E
M`%`'<&5R;%]P87)S90````!1!W!E<FQ?<G5N``"Q`49R965,:6)R87)Y`'T"
M1V5T36]D=6QE2&%N9&QE00``M@)'9710<F]C061D<F5S<P``T0-,;V%D3&EB
M<F%R>4$````@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@
M`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!
M```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$`
M`"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``
M(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@
M`0``(`$`8WEG=VEN,2YD;&P`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!
M`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`%"`!`!0@`0`4(`$`
M%"`!`!0@`0`4(`$`%"`!`&-Y9W!E<FPU7S,R+F1L;``H(`$`*"`!`"@@`0`H
M(`$`2T523D5,,S(N9&QL````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````!``#````,```@`X```#```"`$````/```(`8````(`$`@```````````
M````````!``!````8```@`(```!X``"``P```)```(`$````J```@```````
M`````````````0`)!```4`$``````````````````````0`)!```8`$`````
M`````````````````0`)!```<`$``````````````````````0`)!```@`$`
M``````````````````$```#``0"`V```@````````````````````0`)!```
MD`$``````````````````````0`!````"`$`@````````````````````0`)
M!```H`$``````````````````````0`!````.`$`@```````````````````
M`0`)!```L`$``-`Q`0!H!0`````````````X-P$`J`@`````````````X#\!
M`&@%`````````````$A%`0"H"`````````````#P30$`/@``````````````
M,$X!`%@"`````````````(A0`0#:!0`````````````&`%<`20!.`$4`6`!%
M````*````!`````@`````0`(`````````0```````````````0````$```8&
M!@`$!`0``@("``````!"0D(`G9V=`&YN;@"!@8$`L;&Q`/___P!S<W,`GY^?
M`%A86`"RLK(`_?W]`&MK:P#\_/P`U]?7`!86%@#:VMH`EI:6`(Z.C@!&1D8`
MR\O+``L+"P"@H*``_O[^`,G)R0#!P<$``P,#`!$1$0"3DY,`L[.S`.[N[@#T
M]/0`X^/C`.7EY0"BHJ(`NKJZ`*^OKP`'!P<`<7%Q`+6UM0#U]?4`B8F)`(J*
MB@`T-#0`Z.CH`#4U-0!:6EH`M[>W`-C8V`#0T-``'1T=``P,#`"/CX\`OKZ^
M`&UM;0`)"0D`FYN;`"\O+P#-S<T`V=G9`!D9&0!-34T`1$1$`%!04``\/#P`
M9&1D`,+"P@`R,C(`"`@(``T-#0`/#P\`````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````!04%!04%!04%!04%!
M04$!`0$U0",%03%#04%$12X!`0$!`0$!!0$\%Q<!`1,!`0$!`0$!`14!`285
M`0$/`0$!`0$!`0$3`1$!!34!)@$!`0$!`0$!(A(0,0\%`2`!`0$!`0$!`1`*
M("(5%2XA+@$!`0$!'R`A$"(0(R,E)B<!`0$!)1`0$!`0$!`0$!=%`0$!01`0
M$!`0$!`0$!`0%P$!`0\0$!40$!`0$!`0$`$!`0$2$!`3'Q`0$!`0$`<!`0$!
M(!`0$`$/$!`0$!$!`0$!`0<@$`H!`040$!`Q`0$!`0$!`0$!`0$!004/`0$!
M`0$!`0$!`0$!`0$!`0$!`0$!@`$``,`!``#@`P``\`,``/`#``#P`P``X`,`
M`,`!``"``0``@`$``(`#``"``P````<````'``".#P``_Q\``"@````@````
M0`````$`"`````````0```````````````$````!`````````0$!``("`@`#
M`P,`!`0$``4%!0`&!@8`!P<'``@("``)"0D`"@H*``L+"P`,#`P`#0T-``X.
M#@`/#P\`$!`0`!$1$0`2$A(`$Q,3`!04%``5%14`%A86`!<7%P`8&!@`&1D9
M`!H:&@`;&QL`'!P<`!T='0`>'AX`'Q\?`"`@(``A(2$`(B(B`",C(P`D)"0`
M)24E`"8F)@`G)R<`*"@H`"DI*0`J*BH`*RLK`"PL+``M+2T`+BXN`"\O+P`P
M,#``,3$Q`#(R,@`S,S,`-#0T`#4U-0`V-C8`-S<W`#@X.``Y.3D`.CHZ`#L[
M.P`\/#P`/3T]`#X^/@`_/S\`0$!``$%!00!"0D(`0T-#`$1$1`!%144`1D9&
M`$='1P!(2$@`24E)`$I*2@!+2TL`3$Q,`$U-30!.3DX`3T]/`%!04`!145$`
M4E)2`%-34P!45%0`5555`%965@!75U<`6%A8`%E960!:6EH`6UM;`%Q<7`!=
M75T`7EY>`%]?7P!@8&``86%A`&)B8@!C8V,`9&1D`&5E90!F9F8`9V=G`&AH
M:`!I:6D`:FIJ`&MK:P!L;&P`;6UM`&YN;@!O;V\`<'!P`'%Q<0!R<G(`<W-S
M`'1T=`!U=74`=G9V`'=W=P!X>'@`>7EY`'IZ>@![>WL`?'Q\`'U]?0!^?GX`
M?W]_`("`@`"!@8$`@H*"`(.#@P"$A(0`A86%`(:&A@"'AX<`B(B(`(F)B0"*
MBHH`BXN+`(R,C`"-C8T`CHZ.`(^/CP"0D)``D9&1`)*2D@"3DY,`E)24`)65
ME0"6EI8`EY>7`)B8F`"9F9D`FIJ:`)N;FP"<G)P`G9V=`)Z>G@"?GY\`H*"@
M`*&AH0"BHJ(`HZ.C`*2DI`"EI:4`IJ:F`*>GIP"HJ*@`J:FI`*JJJ@"KJZL`
MK*RL`*VMK0"NKJX`KZ^O`+"PL`"QL;$`LK*R`+.SLP"TM+0`M;6U`+:VM@"W
MM[<`N+BX`+FYN0"ZNKH`N[N[`+R\O`"]O;T`OKZ^`+^_OP#`P,``P<'!`,+"
MP@##P\,`Q,3$`,7%Q0#&QL8`Q\?'`,C(R`#)R<D`RLK*`,O+RP#,S,P`S<W-
M`,[.S@#/S\\`T-#0`-'1T0#2TM(`T]/3`-34U`#5U=4`UM;6`-?7UP#8V-@`
MV=G9`-K:V@#;V]L`W-S<`-W=W0#>WMX`W]_?`.#@X`#AX>$`XN+B`./CXP#D
MY.0`Y>7E`.;FY@#GY^<`Z.CH`.GIZ0#JZNH`Z^OK`.SL[`#M[>T`[N[N`._O
M[P#P\/``\?'Q`/+R\@#S\_,`]/3T`/7U]0#V]O8`]_?W`/CX^`#Y^?D`^OKZ
M`/O[^P#\_/P`_?W]`/[^_@#___\`-#0T/#P\/#P\/#P\/#P\/#P\/#P\/#P\
M+`P$!`0$!`1<M-34U-34U-34U-3<U-34U-34U-34U-2T-%$$!`0$!&3,[/3T
M]/ST_/3T]/ST_/3\]/S\_/3T[,0T45$$!`0$9,3D[/3T]/3T]/3T]/3T]/3T
M]/3T].SLO#1145$$!`1DQ.3L[.SD[.SLY.3DY.SLY.SD[.SL[.2\-%%145$$
M!&2\Y.SLK(RLU,24E)R4C(R4E)2<G)SLW+PT45%145$$9+S<Y.SDQ%&4K&3D
MI$PT+%S$S%&TU.3<O#1145%145%DO-SDY.SDY%&$=.2,;*2<;(R\=*S4Y-R\
M-`0$!`0$!&2\W.3D[.3DO%&,U-2,K,QTS-RD4<3DW+PT!`0$!`0$9+S<Y.3D
MY.3<C'R<M'R$G&S,Y+QTQ.3<M#0$!`0$!`1<O-S<Y.3DY-2<I%R4C,P\E&3,
MO(Q1W-2T-`0$!`0$!&2TU-S<W.3<C(RL++RTS%K$9)QLE*3<U+0T!`0$!`0$
M7+34W-SDW-QTE-1,I/2,/-1TE'2<3-S,M#0$!`0$!`1DM-3<W-SDW'11]'2$
M[,QDY'QLI+1LI-2T-`0$!`0$!&2TS-S<S'R,A+3\S,3\]+3\O&3,K(11O*PT
M!`0$!`0$7+3,U,R,I-S<[/S\_/S\]/SLM.RL?)11K#0$!`0$!`1<K,S,;)ST
M_/S\_/S\_/S\_/S\_,24C%&L-`0$!`0$!%RLQ+Q\Y/S\_/S\_/S\_/S\_/S\
M[,RL4:PT!`0$!`0$7*RT3,3\_/SL]/S\_/S\_/S\_/STO%'$K#0$!`0$!`1<
MK+1,U/S\]+3<_/S\_/S\_/S\_,Q1S,2D-`0$!`0$!%RLQ*11_/S\M)ST_/S\
M_/S\_/STE%',Q*0L!`0$!`0$7*2D483L_/S,9+3L_/S\_/S\_-1LK,R\I#0$
M!`0$!`1<I&1TO/3\_,11A)SD_/S\_/SDC%',S,2D-`0$!`0$!%RD;(SD]/S\
MG%',493L_/S\_+QTS,S$O)PL!`0$!`0$7*2L9%QDI*11K,S$=)3D]/SD4;S,
MS,2\G#0$!`0$!`1<I+2\O(Q14<3$S,RL47Q\E&Q\O,S$Q+R<+`0$!`0$!%R<
MO,3$Q,3$Q,3$Q,3$45%14;S$Q,3$M)PT!`0$!`0$5)RTQ,3$Q,3$Q,3$Q,3$
MQ,3$Q,3$Q+RTG"P$!`0$!`14G+S$Q+R\O+R\O+R\O+R\O+R\O+R\O+2<+`0$
M!`0$!%R<M,2\M+2TM+2TM+2TM+2TM+2TM+2TK)PL!`0$!`0$5(24G)R4E)24
ME)24E)24E)24E)24E)24A"0$!`0$!`1$5%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<
M7%Q4-`0$!`0$!````#\````?````#P````<````#`````0`````````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\````_````/P```#\````_````/P```#\````_````/P```#\`
M```_*````!`````@`````0`(`````````0```````````````0````$```8&
M!@`$!`0``@("``````!"0D(`G9V=`&YN;@"!@8$`L;&Q`/___P!S<W,`GY^?
M`%A86`"RLK(`_?W]`&MK:P#\_/P`U]?7`!86%@#:VMH`EI:6`(Z.C@!&1D8`
MR\O+``L+"P"@H*``_O[^`,G)R0#!P<$``P,#`!$1$0"3DY,`L[.S`.[N[@#T
M]/0`X^/C`.7EY0"BHJ(`NKJZ`*^OKP`'!P<`<7%Q`+6UM0#U]?4`B8F)`(J*
MB@`T-#0`Z.CH`#4U-0!:6EH`M[>W`-C8V`#0T-``'1T=``P,#`"/CX\`OKZ^
M`&UM;0`)"0D`FYN;`"\O+P#-S<T`V=G9`!D9&0!-34T`1$1$`%!04``\/#P`
M9&1D`,+"P@`R,C(`"`@(``T-#0`/#P\`````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````!04%!04%!04%!04%!
M04$!`0$U0",%03%#04%$12X!`0$!`0$!!0$\%Q<!`1,!`0$!`0$!`14!`285
M`0$/`0$!`0$!`0$3`1$!!34!)@$!`0$!`0$!(A(0,0\%`2`!`0$!`0$!`1`*
M("(5%2XA+@$!`0$!'R`A$"(0(R,E)B<!`0$!)1`0$!`0$!`0$!=%`0$!01`0
M$!`0$!`0$!`0%P$!`0\0$!40$!`0$!`0$`$!`0$2$!`3'Q`0$!`0$`<!`0$!
M(!`0$`$/$!`0$!$!`0$!`0<@$`H!`040$!`Q`0$!`0$!`0$!`0$!004/`0$!
M`0$!`0$!`0$!`0$!`0$!`0$!@`$``,`!``#@`P``\`,``/`#``#P`P``X`,`
M`,`!``"``0``@`$``(`#``"``P````<````'``".#P``_Q\``"@````@````
M0`````$`"`````````0```````````````$````!`````````0$!``("`@`#
M`P,`!`0$``4%!0`&!@8`!P<'``@("``)"0D`"@H*``L+"P`,#`P`#0T-``X.
M#@`/#P\`$!`0`!$1$0`2$A(`$Q,3`!04%``5%14`%A86`!<7%P`8&!@`&1D9
M`!H:&@`;&QL`'!P<`!T='0`>'AX`'Q\?`"`@(``A(2$`(B(B`",C(P`D)"0`
M)24E`"8F)@`G)R<`*"@H`"DI*0`J*BH`*RLK`"PL+``M+2T`+BXN`"\O+P`P
M,#``,3$Q`#(R,@`S,S,`-#0T`#4U-0`V-C8`-S<W`#@X.``Y.3D`.CHZ`#L[
M.P`\/#P`/3T]`#X^/@`_/S\`0$!``$%!00!"0D(`0T-#`$1$1`!%144`1D9&
M`$='1P!(2$@`24E)`$I*2@!+2TL`3$Q,`$U-30!.3DX`3T]/`%!04`!145$`
M4E)2`%-34P!45%0`5555`%965@!75U<`6%A8`%E960!:6EH`6UM;`%Q<7`!=
M75T`7EY>`%]?7P!@8&``86%A`&)B8@!C8V,`9&1D`&5E90!F9F8`9V=G`&AH
M:`!I:6D`:FIJ`&MK:P!L;&P`;6UM`&YN;@!O;V\`<'!P`'%Q<0!R<G(`<W-S
M`'1T=`!U=74`=G9V`'=W=P!X>'@`>7EY`'IZ>@![>WL`?'Q\`'U]?0!^?GX`
M?W]_`("`@`"!@8$`@H*"`(.#@P"$A(0`A86%`(:&A@"'AX<`B(B(`(F)B0"*
MBHH`BXN+`(R,C`"-C8T`CHZ.`(^/CP"0D)``D9&1`)*2D@"3DY,`E)24`)65
ME0"6EI8`EY>7`)B8F`"9F9D`FIJ:`)N;FP"<G)P`G9V=`)Z>G@"?GY\`H*"@
M`*&AH0"BHJ(`HZ.C`*2DI`"EI:4`IJ:F`*>GIP"HJ*@`J:FI`*JJJ@"KJZL`
MK*RL`*VMK0"NKJX`KZ^O`+"PL`"QL;$`LK*R`+.SLP"TM+0`M;6U`+:VM@"W
MM[<`N+BX`+FYN0"ZNKH`N[N[`+R\O`"]O;T`OKZ^`+^_OP#`P,``P<'!`,+"
MP@##P\,`Q,3$`,7%Q0#&QL8`Q\?'`,C(R`#)R<D`RLK*`,O+RP#,S,P`S<W-
M`,[.S@#/S\\`T-#0`-'1T0#2TM(`T]/3`-34U`#5U=4`UM;6`-?7UP#8V-@`
MV=G9`-K:V@#;V]L`W-S<`-W=W0#>WMX`W]_?`.#@X`#AX>$`XN+B`./CXP#D
MY.0`Y>7E`.;FY@#GY^<`Z.CH`.GIZ0#JZNH`Z^OK`.SL[`#M[>T`[N[N`._O
M[P#P\/``\?'Q`/+R\@#S\_,`]/3T`/7U]0#V]O8`]_?W`/CX^`#Y^?D`^OKZ
M`/O[^P#\_/P`_?W]`/[^_@#___\`-#0T/#P\/#P\/#P\/#P\/#P\/#P\/#P\
M+`P$!`0$!`1<M-34U-34U-34U-3<U-34U-34U-34U-2T-%$$!`0$!&3,[/3T
M]/ST_/3T]/ST_/3\]/S\_/3T[,0T45$$!`0$9,3D[/3T]/3T]/3T]/3T]/3T
M]/3T].SLO#1145$$!`1DQ.3L[.SD[.SLY.3DY.SLY.SD[.SL[.2\-%%145$$
M!&2\Y.SLK(RLU,24E)R4C(R4E)2<G)SLW+PT45%145$$9+S<Y.SDQ%&4K&3D
MI$PT+%S$S%&TU.3<O#1145%145%DO-SDY.SDY%&$=.2,;*2<;(R\=*S4Y-R\
M-`0$!`0$!&2\W.3D[.3DO%&,U-2,K,QTS-RD4<3DW+PT!`0$!`0$9+S<Y.3D
MY.3<C'R<M'R$G&S,Y+QTQ.3<M#0$!`0$!`1<O-S<Y.3DY-2<I%R4C,P\E&3,
MO(Q1W-2T-`0$!`0$!&2TU-S<W.3<C(RL++RTS%K$9)QLE*3<U+0T!`0$!`0$
M7+34W-SDW-QTE-1,I/2,/-1TE'2<3-S,M#0$!`0$!`1DM-3<W-SDW'11]'2$
M[,QDY'QLI+1LI-2T-`0$!`0$!&2TS-S<S'R,A+3\S,3\]+3\O&3,K(11O*PT
M!`0$!`0$7+3,U,R,I-S<[/S\_/S\]/SLM.RL?)11K#0$!`0$!`1<K,S,;)ST
M_/S\_/S\_/S\_/S\_,24C%&L-`0$!`0$!%RLQ+Q\Y/S\_/S\_/S\_/S\_/S\
M[,RL4:PT!`0$!`0$7*RT3,3\_/SL]/S\_/S\_/S\_/STO%'$K#0$!`0$!`1<
MK+1,U/S\]+3<_/S\_/S\_/S\_,Q1S,2D-`0$!`0$!%RLQ*11_/S\M)ST_/S\
M_/S\_/STE%',Q*0L!`0$!`0$7*2D483L_/S,9+3L_/S\_/S\_-1LK,R\I#0$
M!`0$!`1<I&1TO/3\_,11A)SD_/S\_/SDC%',S,2D-`0$!`0$!%RD;(SD]/S\
MG%',493L_/S\_+QTS,S$O)PL!`0$!`0$7*2L9%QDI*11K,S$=)3D]/SD4;S,
MS,2\G#0$!`0$!`1<I+2\O(Q14<3$S,RL47Q\E&Q\O,S$Q+R<+`0$!`0$!%R<
MO,3$Q,3$Q,3$Q,3$45%14;S$Q,3$M)PT!`0$!`0$5)RTQ,3$Q,3$Q,3$Q,3$
MQ,3$Q,3$Q+RTG"P$!`0$!`14G+S$Q+R\O+R\O+R\O+R\O+R\O+R\O+2<+`0$
M!`0$!%R<M,2\M+2TM+2TM+2TM+2TM+2TM+2TK)PL!`0$!`0$5(24G)R4E)24
ME)24E)24E)24E)24E)24A"0$!`0$!`1$5%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<
M7%Q4-`0$!`0$!````#\````?````#P````<````#`````0`````````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\````_````/P```#\````_````/P```#\````_````/P```#\`
M```_```!``0`$!````$`"`!H!0```0`@(````0`(`*@(```"`!`0```!``@`
M:`4```,`("````$`"`"H"```!````%@"-````%8`4P!?`%8`10!2`%,`20!/
M`$X`7P!)`$X`1@!/``````"]!._^```!```````````````````````_````
M``````0`!``!````````````````````N`$```$`4P!T`'(`:0!N`&<`1@!I
M`&P`90!)`&X`9@!O````E`$```$`,``P`#``,``P`#0`0@`P````)``"``$`
M0P!O`&T`<`!A`&X`>0!.`&$`;0!E```````@````+``"``$`1@!I`&P`90!$
M`&4`<P!C`'(`:0!P`'0`:0!O`&X``````"`````P``@``0!&`&D`;`!E`%8`
M90!R`',`:0!O`&X``````#``+@`P`"X`,``N`#`````D``(``0!)`&X`=`!E
M`'(`;@!A`&P`3@!A`&T`90```"`````H``(``0!,`&4`9P!A`&P`0P!O`'``
M>0!R`&D`9P!H`'0````@````+``"``$`3`!E`&<`80!L`%0`<@!A`&0`90!M
M`&$`<@!K`',``````"`````L``(``0!/`'(`:0!G`&D`;@!A`&P`1@!I`&P`
M90!N`&$`;0!E````(````"0``@`!`%``<@!O`&0`=0!C`'0`3@!A`&T`90``
M````(````#0`"``!`%``<@!O`&0`=0!C`'0`5@!E`'(`<P!I`&\`;@```#``
M+@`P`"X`,``N`#````!$`````0!6`&$`<@!&`&D`;`!E`$D`;@!F`&\`````
M`"0`!````%0`<@!A`&X`<P!L`&$`=`!I`&\`;@````````"P!#P_>&UL('9E
M<G-I;VX](C$N,"(@96YC;V1I;F<](E541BTX(B!S=&%N9&%L;VYE/2)Y97,B
M/SX-"CQA<W-E;6)L>2!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S;V9T+6-O
M;3IA<VTN=C$B(&UA;FEF97-T5F5R<VEO;CTB,2XP(CX-"B`@("`\87-S96UB
M;'E)9&5N=&ET>2!V97)S:6]N/2(P+C`N,"XP(B!T>7!E/2)W:6XS,B(@;F%M
M93TB4$%2+5!A8VME<BY!<'!L:6-A=&EO;B(@+SX-"B`@("`\9&5S8W)I<'1I
M;VX^4$%2+5!A8VME<B!!<'!L:6-A=&EO;CPO9&5S8W)I<'1I;VX^#0H@("`@
M/&1E<&5N9&5N8WD^#0H@("`@("`@(#QD97!E;F1E;G1!<W-E;6)L>3X-"B`@
M("`@("`@("`@(#QA<W-E;6)L>4ED96YT:71Y('1Y<&4](G=I;C,R(B!N86UE
M/2)-:6-R;W-O9G0N5VEN9&]W<RY#;VUM;VXM0V]N=')O;',B('9E<G-I;VX]
M(C8N,"XP+C`B('!U8FQI8TME>51O:V5N/2(V-3DU8C8T,30T8V-F,61F(B!L
M86YG=6%G93TB*B(@<')O8V5S<V]R07)C:&ET96-T=7)E/2(J(B`O/@T*("`@
M("`@("`\+V1E<&5N9&5N=$%S<V5M8FQY/@T*("`@(#PO9&5P96YD96YC>3X-
M"B`@("`\=')U<W1);F9O('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R;W-O9G0M
M8V]M.F%S;2YV,R(^#0H@("`@("`@(#QS96-U<FET>3X-"B`@("`@("`@("`@
M(#QR97%U97-T9610<FEV:6QE9V5S/@T*("`@("`@("`@("`@("`@(#QR97%U
M97-T961%>&5C=71I;VY,979E;"!L979E;#TB87-);G9O:V5R(B!U:4%C8V5S
M<STB9F%L<V4B("\^#0H@("`@("`@("`@("`\+W)E<75E<W1E9%!R:79I;&5G
M97,^#0H@("`@("`@(#PO<V5C=7)I='D^#0H@("`@/"]T<G5S=$EN9F\^#0H@
M("`@/&-O;7!A=&EB:6QI='D@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O<V]F
M="UC;VTZ8V]M<&%T:6)I;&ET>2YV,2(^#0H@("`@("`@(#QA<'!L:6-A=&EO
M;CX-"B`@("`@("`@("`@(#PA+2T@5VEN9&]W<R`X+C$@+2T^#0H@("`@("`@
M("`@("`\<W5P<&]R=&5D3U,@260](GLQ9C8W-F,W-BTX,&4Q+30R,SDM.35B
M8BTX,V0P9C9D,&1A-SA](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@
M5FES=&$@+2T^#0H@("`@("`@("`@("`\<W5P<&]R=&5D3U,@260](GME,C`Q
M,30U-RTQ-30V+30S8S4M835F92TP,#AD965E,V0S9C!](B\^#0H@("`@("`@
M("`@("`\(2TM(%=I;F1O=W,@-R`M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T
M961/4R!)9#TB>S,U,3,X8CEA+35D.38M-&9B9"TX93)D+6$R-#0P,C(U9CDS
M87TB+SX-"B`@("`@("`@("`@(#PA+2T@5VEN9&]W<R`X("TM/@T*("`@("`@
M("`@("`@/'-U<'!O<G1E9$]3($ED/2)[-&$R9C(X93,M-3-B.2TT-#0Q+6)A
M.6,M9#8Y9#1A-&$V93,X?2(O/@T*("`@("`@("`@("`@/"$M+2!7:6YD;W=S
M(#$P("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED/2)[.&4P9C=A
M,3(M8F9B,RTT9F4X+6(Y834M-#AF9#4P83$U83EA?2(O/@T*("`@("`@("`\
M+V%P<&QI8V%T:6]N/@T*("`@(#PO8V]M<&%T:6)I;&ET>3X-"CPO87-S96UB
M;'D^#0H`````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````!```.@````,,)PPHC"V,+PPPC#*,-(PY##N,/TP!#$.,1TQ,C%7,6(Q
M:3%V,98Q33)F,N8RFC/X,V0TL32[--,T&S4@-3TUCS6J-=PU6#9M-L4V`C<M
M-SXW?S>Q-_0W`3@1.!DX(3@N.$8X3CA6.%XX=#B$.(PXE#C1..HX$SDI.4\Y
MX3GI.2DZ5CJ'.OPZ%#MM.W\[I3NW.^4[]SLE/#<\93QW/*4\MSSE//<\)3TW
M/64]=SVE/;<]Y3WW/24^-3Y#/E,^83YQ/GL^@#Z//IX^KC[T/OT^##\E/TD_
MAS^6/[0_Q#_F/P`@``"P````'C`L,#XP<C!],(8PE#"J,+4PP##+,.$P[#!A
M,7DQ*S(U,D$RF3*Y,@PS63.1,Z<SK#/%,]$SW3,*-!HT(C12-*8TR#0--3TU
M>36`-J0VM#:\-MTV`S<Y-\TWYC?O-_XW%C@R.%(X:CAZ.)HXNCC1..HX!SD?
M.2XY/CE/.60Y;CF=.;$YOCG).?$Y_3DA.D,Z3#I5.G<ZP3K@.OXZ&#LS.TT[
M<#M_.Z\[`%```/P```!>-K(ZNCK".LHZTCK:.N(ZZCKR.OHZ`CL*.Q([&CLB
M.RH[,CLZ.T([2CM2.UH[8CN6.YX[ICNN.[8[OCO&.\X[UCO>.^8[[COV._X[
M!CP./!8\'CPF/"X\-CP^/$8\3CQ6/%X\9CQN/'8\?CR&/(X\ECR>/*8\KCRV
M/+X\QCSN/"(]*3TP/3T]1#U+/5(]63UF/6T]=#U[/84]GCVL/;H]R#W6/>0]
M\CT`/@X^'#XD/BT^-CX_/D@^43Y:/G`^?3Z$/HL^E3Z?/JD^T#X&/PT_%#\>
M/R4_+#\S/SH_1S]./U4_7#]F/XP_G3^U/\4_U3_E/_4_`&```&0````%,"PP
M-#`],$8P?C".,)0PGC"C,+`P_3`3,9(QFC&B,:HQP#'7,?(Q^C$@,DDR43)8
M,F,R<#*%,IDRK3+C,NPR\S+Z,CXS3C-A,W(S@#.7,Z@SN3/0,]HS%#0```!P
M```,````##``````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
E````````````````````````````````````````````````````
