package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.26.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.26.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``3`$*
M`.F1[ED``````````.``+@,+`0(=`$X```#8````$@```!`````0````8```
M``!````0`````@``!`````$````$``````````!@`0``!```AML!``,``(``
M`"```!``````$```$````````!````````$`V`$````0`0!T"0```"`!`+`H
M`````````````````````````%`!`"0#````P```'```````````````````
M``````````````````````````````````````#$$0$`8`$`````````````
M`````````````````````"YT97AT````Y$T````0````3@````0`````````
M`````````&``4&`N9&%T80```$0`````8`````(```!2````````````````
M``!``&#`+G)D871A``!X0P```'````!$````5```````````````````0``P
M0"YB=6EL9&ED-0````#``````@```)@``````````````````$``,$`N96A?
M9G)A;50'````T`````@```":``````````````````!``#!`+F)S<P`````D
M$0```.``````````````````````````````@`!@P"YE9&%T80``V`$`````
M`0```@```*(``````````````````$``,$`N:61A=&$``'0)````$`$```H`
M``"D``````````````````!``##`+G)S<F,```"P*````"`!`+`H````K@``
M````````````````0``PP"YR96QO8P``)`,```!0`0``!````-@`````````
M`````````$``,$(`````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````%6)Y8/L&(/D\,<$
M)-!;0`#HBTD``,=$)`@`````QT0D!`````#'!"0`````Z-]*``#'1"0(````
M`,=$)`0`````QP0D`````.C32@``QT0D"`````#'1"0$`````,<$)`````#H
MQTH``,=$)`@`````QT0D!`````#'!"0`````Z+M*``#)PY"0D)"0D)"0D%6)
MY5=64X/L+(LU%!-!`,<$)`!P0`#_UH/L!(7`#X3-````B</'!"0`<$``_Q4<
M$T$`BQ48$T$`@^P$HR#Q0`#'1"0$#W!``(D<)(E5Y/_2@^P(B<>+5>3'1"0$
M)7!``(D<)/_2@^P(HP!@0`"%_W01QT0D!`#@0`#'!"0XT$``_]>A0&!``(7`
M=#C'!"0]<$``_]:#[`2%P+H`````=!;'1"0$2W!``(D$)/\5&!-!`(/L"(G"
MA=)T"<<$)$!@0`#_TL=$)`@$8$``QT0D!`````#'!"2@$4``Z-5#``"-9?1;
M7E]=PXGVC;PG`````,<%`&!```````"_`````.EI____C;8`````C;\`````
M58GE@^P8H0!@0`"%P'0)QP0D.-!``/_0H2#Q0`"%P'0,B00D_Q40$T$`@^P$
MR<.055=64XU8.('L#`,``(L5H!)!`(F4)/P"```QTHE$)$R)'"2)="0(B7PD
M#)"-="8`BSPDQT34?`````#'A-2$`````````(L$EXM,EP0/R(G+B434>(G(
MP>D8P>@(B=\E`/\``,'G"`G(B=G!X1@)R(GY@>$``/\`"<&)C-2`````@\("
M@_H0=:>-1"1XC90D>`(``(M8:(M(;(/`"#-8.#-(/#-8"#-(##-8^#-(_,=`
M?`````")WHG/B=D!\8G[B0PDB?$/K/D?"PPD#Z3S`8E<)`2)2'@YT'6YBT0D
M3(LXBV@$BW`8B?J+>`B)Z8MH#(ET)`B)RXMP((E,)!2)T8E\)!B+>!");"0<
MBV@4B70D6(G6B50D$(D\)(MX','F!0^LV1N);"0$B?4QTHE\)`R+>"0)S8M,
M)%@#3"1XB>B)?"1<BUPD7!-<)'R!P9EY@EJ#TP`!R(M,)!@1VHL<)(E$)"")
M5"0DBU0D"(G=,=6)[R'/B?J+?"0(BW0D((M$)!C'1"0D`````,=$)#0`````
M,?H!UHM4)!R)QXET)"")="0HP><>B<:)^8M\)`@/K-8"B<V+!"2+5"0D"?4#
MO"2`````B6PD,(ML)`P3K"2$````B?F!P9EY@EJ)ZXML)#")3"0X@],`BTPD
M.(E<)#S'1"0\`````#'%BT0D$(GO(<>+1"0@B?N+/"2)QC'[`=G!Y@4/K-`;
MBU0D%(G+"<:+1"00`?,Q_XE<)%")P<'A'@^LT`*+5"0@B<T)Q8L$)(EL)#B+
M;"0H]](C;"0X(U0D,`GJ`X0DB````,=$)"P`````B=&+5"0DQT0D1`````#'
M1"0D`````,=$)#0`````!9EY@EH!P8G8P>`%B40D*(G8BVPD*`^L^!N)1"1`
M"VPD0(M$)"`!Z8E,)"B)3"1@B<'!X1X/K-`"B<V+5"0X"<6+1"0PB6PD0(G=
M]]4AZHML)%`C;"1`">H#A"20````B=&+5"0L!9EY@EH!P8M$)"C!X`6)1"0@
MBT0D*(ML)"`/K-`;B=J)1"0P"VPD,`'IB?V)WP^L[P+!XAZ)3"10B4PD((G5
M"?V);"0PBVPD*(M4)$"+="0XQT0D/`````#'1"1$`````/?5(>J+;"1@(VPD
M,`GJ`[0DF````(G1BU0D)(GP!9EY@EH!P8M$)"")SXG&#ZS0&\'F!8M4)"S'
M1"0L``````G&BT0D*`'WB?Z)?"1@,?^)P<'A'@^LT`*)S8M4)#`)Q8M$)$")
M;"0XBVPD(/?5(>J+;"10(VPD.`GJ`X0DH````(G1BU0D)`69>8):`<&)\,'@
M!8E$)"B)\(ML)"@/K/@;B40D0`ML)$"+1"0@`>F)3"10B4PD*(G!P>$>#ZS0
M`HG-BU0D.`G%B6PD0(GU]]4AZHML)&`C;"1`BT0D,,=$)"0`````QT0D-```
M``#'1"0\``````GJ`X0DJ````(G1BU0D+`69>8):`<&+1"0HP>`%B40D((M$
M)"B+;"0@#ZS0&XGRB40D,`ML)#`!Z8G]B?</K.\"P>(>BW0D.(E,)"")U8M4
M)$")3"1@"?V);"0PBVPD*/?5(>J+;"10(VPD,`GJ`[0DL````(G1BU0D)(GP
M!9EY@EH!P8M$)"")SXG&#ZS0&\'F!8M4)"P)QHM$)"@!]XE\)%")_C'_B<'!
MX1X/K-`"B<T)Q8EL)#B+;"0@BU0D,(M$)$#'1"0L`````,=$)$0`````]]4A
MZHML)&`C;"0X">H#A"2X````B=&+5"0D!9EY@EH!P8GPP>`%B40D*(GPBVPD
M*`^L^!N)1"1`"VPD0(M$)"`!Z8E,)"B)3"1@B<'!X1X/K-`"B<V+5"0X"<6+
M1"0PB6PD0(GU]]4AZHML)%`C;"1`">H#A"3`````B=&+5"0L!9EY@EH!P8M$
M)"C!X`6)1"0@BT0D*(ML)"`/K-`;B?*)1"0P"VPD,`'IB?V)]P^L[P+!XAZ)
M3"10B4PD((G5QT0D)`````"+5"1`"?V+="0XQT0D-`````");"0PBVPD*,=$
M)#P`````]]4AZHML)&`C;"0P">H#M"3(````B=&+5"0DB?`%F7F"6@'!BT0D
M((G/B<8/K-`;P>8%BU0D+,=$)"P`````"<:+1"0H`?>)_HE\)&`Q_XG!P>$>
M#ZS0`HG-BU0D,`G%BT0D0(EL)#B+;"0@]]4AZHML)%`C;"0X">H#A"30````
MB=&+5"0D!9EY@EH!P8GPP>`%B40D*(GPBVPD*`^L^!N)1"1`"VPD0(M$)"`!
MZ8E,)%")3"0HB<'!X1X/K-`"B<T)Q8EL)$"+5"0XB?7WU8M$)##'1"0D````
M`,=$)$0`````QT0D-``````AZHML)&`C;"1`">H#A"38````B=&+5"0L!9EY
M@EH!P8M$)"C!X`6)1"0@BT0D*(ML)"`/K-`;B?*)1"0P"VPD,`'IB?V)]P^L
M[P+!XAZ+="0XB4PD((G5BU0D0(E,)&`)_8EL)#"+;"0H]]4AZHML)%`C;"0P
M">H#M"3@````B=&+5"0DB?`%F7F"6@'!BT0D((G/B<8/K-`;P>8%BU0D+`G&
MBT0D*`'WB7PD4(G^,?^)P<'A'@^LT`*)S0G%B6PD.,=$)#P`````BVPD((M4
M)#"+1"1`QT0D+`````#'1"1$`````/?5(>J+;"1@(VPD.`GJ`X0DZ````(G1
MBU0D)`69>8):`<&)\,'@!8E$)"B)\(ML)"@/K/@;B40D0`ML)$"+1"0@`>F)
M3"0HB4PD8(G!P>$>#ZS0`HG-BU0D.`G%BT0D,(EL)$")]??5(>J+;"10(VPD
M0`GJ`X0D\````(G1BU0D+`69>8):`<&+1"0HP>`%B40D((M$)"B+;"0@#ZS0
M&XGRB40D,`ML)#`!Z8G]B?</K.\"P>(>B4PD4(E,)"")U8M4)$"+="0X"?W'
M1"0D`````,=$)#0`````B6PD,(ML)"C'1"0\`````/?5(>J+;"1@(VPD,`GJ
M`[0D^````(G1BU0D)(GP!9EY@EH!P8M$)"")SXG&#ZS0&\'F!8M4)"S'1"0L
M``````G&BT0D*`'WB?Z)?"1@,?^)P<'A'@^LT`*)S8M4)#`)Q8M$)$");"0X
MBVPD(/?5(>J+;"10(VPD.`GJ`X0D``$``(G1!9EY@EH!P8GPP>`%B40D*(GP
MBVPD*`^L^!N)1"1`"VPD0(M$)"`!Z8E,)%")3"0HB<&+5"0DP>$>QT0D-```
M``")S<=$)"0`````QT0D1``````/K-`"BU0D.`G%BT0D,(EL)"")]??5(>J+
M;"1@(VPD(`GJ`X0D"`$``(G1BU0D+`69>8):`<&+1"0HP>`%B40D,(M$)"B+
M;"0P#ZS0&XGRB40D0`ML)$`!Z8G]B?</K.\"P>(>BW0D.(E,)#")U8M4)"")
M3"1@"?V+?"0HB6PD:(EL)$#WUR'ZBWPD4"'O"?H#M"00`0``B=&+5"0TB?`%
MF7F"6@'!BT0D,(G'P><%#ZS0&XG^"<:)\(G.`<:+1"0H,?^)="10BU0D+,=$
M)"0`````B<'!X1X/K-`"B<V+3"0@BU0D-`G%`XPD&`$``(EL)#B+;"1HBUPD
M.(G(!:'KV6XS;"0X,VPD8(GI`<&)\,'@!8E$)"")\(ML)"`/K/@;B40D*`ML
M)"B+1"0P`>F+;"10B4PD((E,)&")P0^LT`+!X1Z+5"0D"<&+1"1``X0D(`$`
M`(E,)$`%H>O9;C'+,=V)Z0'!BT0D(,'@!8E$)"B+1"0@BVPD*`^LT!N)\HE$
M)#`+;"0P`>G!XAX/K/X"B4PD4(E,)"B)U<=$)"P`````B[PD*`$```GUBTPD
M.(EL)#B+5"0HBUPD+(''H>O9;HGXBWPD0`'(BTPD*#'OBVPD8#']B>X!QL'B
M!8M$)"")5"0PBVPD,(GW#ZS9&XM4)"3'1"0D``````G-B<$![\'A'@^LT`*)
MS8N4)#`!``")_@G%BT0D0(E\)&`Q_XEL)#"+;"0X@<*AZ]EN`=`S;"0PBU0D
M+#-L)%")Z0'!B?#!X`6)1"0@B?"+;"0@#ZSX&XE$)$`+;"1`BT0D*`'IB4PD
M4(E,)"")P<'A'@^LT`*)RXN4)#@!```)PXM$)#B)W8E<)#B+7"0P@<*AZ]EN
M`=`QZXML)&"+5"0DQT0D+``````QW8GI`<&+1"0@P>`%B40D*(M$)""+;"0H
M#ZS0&XGRB40D0`ML)$"+1"0P`>G!XAX/K/X"B=6+E"1``0``B4PD8`GUBW0D
M.(E,)"B);"1`@<*AZ]EN`=`Q[HML)%"+5"0L,?6)Z0'!BT0D*(G'P><%#ZS0
M&XG^BU0D)`G&B?")S@'&BT0D(#'_B70D4(G!P>$>#ZS0`HG-BY0D2`$```G%
MBT0D.(EL)#"+;"1`@<*AZ]EN`=`S;"0P,VPD8(GI`<&)\,'@!8E$)"")\(ML
M)"`/K/@;B40D.`ML)#@!Z8E,)&")3"0@BT0D*(M4)"R+;"10QT0D)`````#'
M1"0L`````(G!P>$>#ZS0`HG+BY0D4`$``(M,)#`)PXM$)$")7"0X@<*AZ]EN
M`=`QV8M4)"0QS8GI`<&+1"0@P>`%B40D*(M$)""+;"0H#ZS0&XGRB40D0`ML
M)$"+1"0P`>G!XAX/K/X"B=6+E"18`0``B4PD*`GUBW0D.(E,)%");"1`@<*A
MZ]EN`=`Q[HML)&"+5"0L,?6)Z0'!BT0D*(G'P><%#ZS0&XG^BU0D)`G&B?")
MS@'&BT0D(#'_B70D8(G!P>$>#ZS0`HG-BY0D8`$```G%BT0D.(EL)#"+;"1`
M@<*AZ]EN`=`S;"0PBU0D+#-L)%#'1"0D`````,=$)"P`````B>D!P8GPP>`%
MB40D((GPBVPD(`^L^!N)1"0X"VPD.(M$)"@!Z8ML)&")3"0@B4PD4(G!P>$>
M#ZS0`HG+BY0D:`$``(M,)#`)PXM$)$")7"0X@<*AZ]EN`=`QV8M4)"0QS8GI
M`<&+1"0@P>`%B40D*(M$)""+;"0H#ZS0&XGRB40D0`ML)$"+1"0P`>G!XAX/
MK/X"B=6+E"1P`0``B4PD*`GUBW0D.(E,)&");"1`@<*AZ]EN`=`Q[HML)%`Q
M]8GI`<&+1"0HBU0D+(G'P><%#ZS0&XG^BU0D),=$)"0`````"<:)\(G.`<:+
M1"0@,?^)="10B<'!X1X/K-`"B<V+E"1X`0``"<6+1"0XB6PD,(ML)$"!PJ'K
MV6X!T#-L)#"+5"0L,VPD8(GI`<&)\,'@!8E$)"")\(ML)"`/K/@;B40D.`ML
M)#B+1"0H`>F+;"10B4PD((E,)&")P<'A'@^LT`*)RXN4)(`!``"+3"0P"<.+
M1"1`B5PD.('"H>O9;@'0,=F+5"0D,<V)Z0'!BT0D(,'@!8E$)"B+1"0@BVPD
M*`^LT!N)\HE$)$`+;"1``>G!XAX/K/X"B4PD4(G5B4PD*(N4)(@!```)]8M$
M)#"+="0XB6PD0,=$)"P`````@<*AZ]EN`=`Q[HML)&"+5"0L,?6)Z0'!BT0D
M*(G'P><%#ZS0&XG^BU0D),=$)"0`````"<:)\(G.`<:+1"0@,?^)="1@B<'!
MX1X/K-`"B<V+E"20`0``"<6+1"0XB6PD,(ML)$"!PJ'KV6X!T#-L)#"+5"0L
M,VPD4(GI`<&)\,'@!8E$)"")\(ML)"`/K/@;B40D.`ML)#B+1"0H`>F+;"1@
MB4PD((E,)%")P<'A'@^LT`*)RXN4))@!``"+3"0P"<.+1"1`B5PD.('"H>O9
M;@'0,=DQS8GI`<&+1"0@P>`%B40D*(M$)""+5"0DBVPD*,=$)"P`````QT0D
M-``````/K-`;B?*)1"1`"VPD0(M$)#`!Z<'B'@^L_@*)U8N4)*`!``")3"1@
M"?6+="0XB4PD*(EL)$"+7"1`@<*AZ]EN`=`Q[HML)%"+5"0L,?6)Z8ML)&`!
MP8M$)"B)Q\'G!0^LT!N)_HM4)"0)QHGPB<X!QHM$)"")="0PB70D4(G!P>$>
M#ZS0`HG.BY0DJ`$```G&BT0D.(GW@<*AZ]EN`=`Q\XM4)#0QW8G[B>D!P8M$
M)##!X`6)1"0@BT0D,(ML)"`/K-`;B40D.`ML)#@!Z8E,)&")3"0@BT0D*(M4
M)"R+;"10QT0D)`````")?"1HBWPD)(G!QT0D+``````/K-`"P>$>BY0DL`$`
M`(G."<:+1"1`@<*AZ]ENB70D.`'0,?.+="0@,=V)Z0'!B?#!X`6)RXE$)"B)
M\(ML)"@/K/@;BWPD,`G%BT0D8`'KBVPD-,=$)#0`````B?J)7"0HB5PD4,'B
M'HM<)"P/K.\"B=6+5"0X"?V);"1`B=<AZ@GO(?B+?"0DB<$)T8N4)+@!``")
MSHM,)&B!PMR\&X^)T(M4)"@!R(M,)"@!QL'B!8E4)#"+;"0P#ZS9&PG-`>Z)
M="1@B70D,(MT)""+7"1`BT0D4,=$)"0`````B?'!X1X/K/X"B<^+3"0X"?>)
M^HE\)%`)VB'0B?J+?"0L(=J)Q8M<)#0)U8N4),`!``#'1"0L`````(GN@<+<
MO!N/B="+5"0P`<B+3"0P`<;!X@6)5"0@BVPD(`^LV1N)\XMT)"@)S0'KB?&)
M7"0@B5PD:`^L_@+!X1Z+?"10BUPD)(GPB<Z+3"1`"<:+1"1@B?J)="0X"?(A
MT(GZBWPD,"'RB<4)U8N4),@!``")[H'"W+P;CXG0BU0D(`'(BTPD(`'&BT0D
M:,'B!8E4)"B+;"0H#ZS9&XGS"<V)^0'KBVPD-,'A'HE<)&")7"0H#ZSO`HG-
M"?V+?"0XB6PD0(M,)%#'1"0T`````(GZ">HAT(GZBWPD)"'JB</'1"0D````
M``G3BY0DT`$``(G>BUPD+('"W+P;CXG0BU0D*`'(BTPD*`'&BT0D8,'B!8E4
M)#"+;"0P#ZS9&XM<)$`)S0'NB70D,(ET)&B+="0@B?'!X1X/K/X"B<^+3"0X
M"?>)^HE\)%`)VB'0B?J+?"0L(=J)Q8M<)#0)U8N4)-@!``")[H'"W+P;CXG0
MBU0D,`'(BTPD,`'&P>(%B50D((ML)"`/K-D;"<T![HET)&")="0@BW0D*(GQ
M#ZS^`L'A'HM\)%")\(G.BTPD0`G&BT0D:(GZB70D.`GR(=")^B'RB<4)U8N4
M).`!``")[H'"W+P;CXG0BU0D((M<)"0!R(M,)""+?"0P`<:+1"1@QT0D+```
M``#!X@6)5"0HBVPD*`^LV1N)\PG-B?D!ZXML)#3!X1Z)7"0HB5PD:,=$)#0`
M````#ZSO`HG-BTPD4`G]BWPD.(EL)$")^@GJ(=")^HM\)"0AZHG#"=.+E"3H
M`0``B=Z+7"0L@<+<O!N/B="+5"0H`<B+3"0H`<:+1"1HP>(%B50D,(ML)#`/
MK-D;BUPD0`G-`>Z)="0PB70D8(MT)"")\<'A'@^L_@*)SXM,)#@)]XGZB7PD
M4`G:(=")^B':B<4)U8N4)/`!``")[H'"W+P;CXG0BU0D,`'(`<;!X@6)5"0@
MBTPD,(M<)#2+;"0@BWPD+,=$)"0`````QT0D+``````/K-D;BUPD)`G-`>Z)
M="0@B70D:(MT)"B)\0^L_@+!X1Z+?"10B?")SHM,)$`)QHM$)&")^HET)#@)
M\B'0B?J+?"0P(?*)Q0G5BY0D^`$``(GN@<+<O!N/B="+5"0@`<B+3"0@`<:+
M1"1HP>(%B50D*(ML)"@/K-D;B?,)S8GY`>N+;"0TP>$>B5PD*(E<)&`/K.\"
MB<V+3"10"?V+?"0XB6PD0(GZ">HAT(GZ(>J)PPG3BY0D``(``(G>BUPD+('"
MW+P;CXG0BU0D*`'(BTPD*`'&P>(%B50D,(ML)#`/K-D;BWPD)(M<)$`)S8M$
M)&#'1"0T``````'NQT0D)`````")="1HB70D,(MT)"")\<'A'@^L_@*)SXM,
M)#@)]XGZB7PD4`G:(=")^HM\)"PAVHG%BUPD-`G5BY0D"`(``(GN@<+<O!N/
MB="+5"0P`<B+3"0P`<;!X@6)5"0@BVPD(`^LV1N+7"0D"<T![HET)"")="1@
MBW0D*(GQ#ZS^`L'A'HM\)%")\(G.BTPD0`G&BT0D:(GZB70D.`GR(=")^B'R
MB<4)U8N4)!`"``")[H'"W+P;CXG0BU0D(`'(BTPD(`'&P>(%B50D*(ML)"@/
MK-D;B?,)S0'KB5PD:(E<)"B+?"0PBVPD-(M$)&#'1"0L`````,=$)#0`````
MB?G!X1X/K.\"B<V+3"10"?V+?"0XB6PD0(GZ">HAT(GZBWPD)"'JB</'1"0D
M``````G3BY0D&`(``(G>BUPD+('"W+P;CXG0BU0D*`'(BTPD*`'&BT0D:,'B
M!8E4)#"+;"0P#ZS9&XM<)$`)S0'NB70D,(ET)&"+="0@B?'!X1X/K/X"B<^+
M3"0X"?>)^HE\)%`)VB'0B?HAVHG%BUPD-`G5BY0D(`(``(GN@<+<O!N/B="+
M5"0P`<B+3"0P`<;!X@6)5"0@BVPD(`^LV1L)S0'NB70D:(ET)""+="0HBWPD
M+(M<)"3'1"0L`````(GQ#ZS^`L'A'HM\)%")\(G.BTPD0`G&BT0D8(GZB70D
M.`GR(=")^HM\)#`A\HG%"=6+E"0H`@``B>Z!PMR\&X^)T(M4)"`!R(M,)"`!
MQHM$)&C!X@6)5"0HBVPD*`^LV1N)\PG-B?D!ZXML)#3!X1Z)7"0HB5PD8,=$
M)#0`````#ZSO`HG-BTPD4`G]BWPD.(EL)$")^@GJ(=")^HM\)"0AZHG#"=.+
ME"0P`@``B=Z+7"0L@<+<O!N/B="+5"0H`<B+3"0H`<;!X@6)5"0PBVPD,`^L
MV1N+7"1`"<T![HET)&B)="0PBW0D((M$)&#'1"0D`````(GQP>$>#ZS^`HG/
MBTPD.`GWB?J)?"10"=HAT(GZBWPD+"':B<6+7"0T"=6+E"0X`@``QT0D+```
M``")[H'"W+P;CXG0BU0D,`'(BTPD,`'&P>(%B50D((ML)"`/K-D;BUPD)`G-
M`>Z)="0@B70D8(MT)"B)\0^L_@+!X1Z+?"10B?")SHM,)$`)QHM$)&B)^HET
M)#@)\B'0B?J+?"0P(?*)Q0G5BY0D0`(``(GN@<+<O!N/B="+5"0@`<B+3"0@
M`<:+1"1@P>(%B50D*(ML)"@/K-D;B?,)S8GY`>N+;"0TP>$>B5PD:(E<)"@/
MK.\"B<T)_8M\)#B);"1`B?H)ZB'0B?HAZHG#"=.+E"1(`@``BTPD4(G>BUPD
M+(M\)"3'1"0T`````,=$)"0`````@<+<O!N/B="+5"0H`<B+3"0H`<:+1"1H
MP>(%B50D,(ML)#`/K-D;BUPD0`G-`>Z)="0PB70D8(MT)"")\<'A'@^L_@*)
MSXN,)%0"```)]XGZ"=\A^(E4)%`AVHG%,?\)U8N4)%`"``")[H'"W+P;CX/1
M`(G0,=N)RHM,)#@!R(M,)#`1V@'&BUPD-!'7BU0D,`^LV1O!X@6)5"0@BVPD
M(#'2"<T![A'7B70D:(ET)"")?"1LBW0D*(M\)"R+E"18`@``B?$/K/X"P>$>
MBWPD4(GPB<Z+3"1`"<:!PM;!8LJ)="0XBVPD8(G0`<B+5"0@,?>+7"0DBTPD
M(#']BWPD,,=$)"P`````B>X!QL'B!8E4)"B+;"0H#ZS9&XGSBY0D8`(```G-
MB?D!ZXML)#3!X1Z)7"0HB5PD8(M<)&C'1"0T``````^L[P*)S8M,)%`)_8M\
M)#B!PM;!8LJ)T(M4)"B);"1``<B+3"0H,>\Q^XM\)"2)WHM<)"P!QL'B!8M$
M)#B)5"0PBVPD,`^LV1N+E"1H`@``"<T![HML)&")="0PB70D4(MT)"")\<'A
M'@^L_@*)SPGWBW0D0('"UL%BR@'0B7PD.#'^,?6)Z0'!BT0D,(M4)#2+?"0L
MB<O'1"0D`````,=$)"P`````B<8/K-`;P>8%BU0D)(E$)""+;"0@"?6+="0H
M`>N+;"10B5PD((E<)&")\0^L_@+!X1Z+?"0XB?")SHN,)'`"```)QHM$)$")
M="1`@<'6P6+*`<@Q]S']B>D!P8M$)"")Q\'G!0^LT!N)_HM4)#0)QHM$)#`!
M\3'_B<Z)3"10B<'!X1X/K-`"B<V+E"1X`@``"<6+1"0XB6PD,(ML)$"!PM;!
M8LH!T#-L)#`S;"1@B>D!P8GPP>`%B40D*(GPBVPD*`^L^!N)1"0X"VPD.(M$
M)"`!Z8E,)&")3"0HB<&+5"0DP>$>BVPD4(G+BTPD,,=$)"0`````#ZS0`HN4
M)(`"```)PXM$)$")7"0X@<+6P6+*`=`QV8M4)"PQS8GI`<&+1"0HP>`%B40D
M((M$)"B+;"0@#ZS0&XGRB40D0`ML)$"+1"0P`>G!XAX/K/X"B=6+E"2(`@``
MB4PD4`GUBW0D.(E,)"");"1`@<+6P6+*`=`Q[HML)&"+5"0D,?6)Z0'!BT0D
M((G'P><%#ZS0&XG^BU0D+`G&B?")S@'&BT0D*#'_B70D8(G!P>$>#ZS0`HG-
MBY0DD`(```G%BT0D.(EL)#"+;"1`@<+6P6+*`=`S;"0P,VPD4(GI`<&)\,'@
M!8E$)"B)\(ML)"@/K/@;BU0D),=$)"P`````B40D.`ML)#B+1"0@QT0D)```
M```!Z8ML)&")3"0HB4PD4(G!P>$>#ZS0`HG+BY0DF`(``(M,)#`)PXM$)$")
M7"0X@<+6P6+*`=`QV8M4)"PQS8GI`<&+1"0HP>`%B40D((M$)"B+;"0@#ZS0
M&XGRB40D0`ML)$"+1"0P`>G!XAX/K/X"B=6+E"2@`@``B4PD(`GUBW0D.(E,
M)&");"1`@<+6P6+*`=`Q[HML)%"+5"0D,?6)Z0'!BT0D((G'P><%#ZS0&XG^
M"<:)\(G.`<:+1"0H,?^)="10BU0D+,=$)"P`````B<'!X1X/K-`"B<V+E"2H
M`@``"<6+1"0XB6PD,(ML)$"!PM;!8LH!T#-L)#"+5"0D,VPD8,=$)"0`````
MB>D!P8GPP>`%B40D*(GPBVPD*`^L^!N)1"0X"VPD.(M$)"`!Z8ML)%")3"0H
MB4PD8(G!P>$>#ZS0`HG+BY0DL`(``(M,)#`)PXM$)$")7"0X@<+6P6+*`=`Q
MV8M4)"PQS8GI`<&+1"0HP>`%B40D((M$)"B+;"0@#ZS0&XGRB40D0`ML)$`!
MZ<'B'@^L_@*)U8N4)+@"``")3"10"?6)3"0@BT0D,(MT)#B);"1`@<+6P6+*
M`="+5"0D,>Z+;"1@,?6)Z0'!BT0D((G'P><%#ZS0&XG^BU0D+,=$)"P`````
M"<:)\(G.`<:+1"0H,?^)="1@B<'!X1X/K-`"B<V+E"3``@``"<6+1"0XB6PD
M,(ML)$"!PM;!8LH!T#-L)#"+5"0D,VPD4(GI`<&)\,'@!8E$)"B)\(ML)"@/
MK/@;B40D.`ML)#B+1"0@`>F+;"1@B4PD*(E,)%")P<'A'@^LT`*)RXN4),@"
M``"+3"0P"<.+1"1`B5PD.('"UL%BR@'0,=DQS8GI`<&+1"0HP>`%B40D((M$
M)"B+5"0LBVPD(,=$)"0`````#ZS0&XGRB40D0`ML)$"+1"0P`>G!XAZ)U8N4
M)-`"``")3"1@#ZS^`HE,)""+C"34`@``"?6!PM;!8LJ+="0XB6PD0(/1`(G+
MB=$QT@'($=HQ[HML)%`QVS'UB>D!P8M$)"`1TXM4)"2)Q\'G!0^LT!N)_C'2
MB=\)QHGPB<X!QHM$)"@1UXM4)"R)="10B7PD5#'_B<'!X1X/K-`"BY0DV`(`
M`(G-BXPDW`(```G%BT0D.(EL)#"+;"1`@<+6P6+*@]$`B<N)T3'2`<@1VC-L
M)#`QVS-L)&")Z0'!B?`1T\'@!3'2B40D*(GPBVPD*`^L^!N)1"0X"VPD.`'I
M$=.)RHE4)&")7"1DB50D*(M$)""+5"0DBVPD4,=$)"P`````QT0D)`````")
MP<'A'@^LT`*)RXN4).`"``"+3"0P"<.+1"1`B5PD.('"UL%BR@'0,=F+5"0L
M,<V)Z0'!BT0D*,'@!8E$)""+1"0HBVPD(`^LT!N)\HE$)$`+;"1`BT0D,`'I
MP>(>#ZS^`HG5BY0DZ`(``(E,)"`)]8MT)#B)3"1`B6PD,('"UL%BR@'0,>Z+
M;"1@BU0D)#'UB>D!P8M$)"")Q\'G!0^LT!N)_HM4)"P)QHGP`<@Q_XG&BT0D
M*(G!P>$>#ZS0`HG-BU0D((M,)"0)Q0^DT1[!XAZ)5"0HB4PD+(M4)""+3"0D
MBT0D.(M<)#");"10#ZS*`L'I`HE,)"2+C"3P`@``B50D((GZ@<'6P6+*`<@Q
MZXML)$`QW8GI`<&)\,'@!8E$)#B)\(ML)#@/K/@;#Z3R!8E$)$`+;"1`B50D
M/(GZP>H;B50D1(GH`<B)P0-,)!"+1"1,B<+'0`0`````B0B+1"08QT(,````
M`,="%``````!\(MT)"@+="0@B4((B?`#!"2)0A"+1"10`T0D"(E"&,="'```
M``"+1"0P`T0D6,=")`````")0B"+A"3\`@``,P6@$D$`=0N!Q`P#``!;7E]=
MP^B`'@``55=64X/L+(7`#X22````BPU$\$``A<F)3"08#X2`````#[8(A,ET
M6(#Y/713B<.#PP$/M@N$R70%@/D]=?$IPXM,)!B+*87M="R)5"0<B<:)SXE<
M)`B)="0$B2PDZ%(?``"%P'4'@'P=`#UT%X/'!(LOA>UUW(/$+(GH6UY?7<,Q
MV^N\B?HK5"08BT0D'(UL'0'!^@*)$(/$+(GH6UY?7<,Q[>O1C70F`(V\)P``
M``!3B<.#[!BAH!)!`(E$)`PQP(U4)`B)V.@T____A<!T,(M4)`BA1/!``(T$
MD(M0!(72B1!TW(/`!(M0!(72B1!U](U4)`B)V.@$____A<!UT(M$)`PS!:`2
M00!U!8/$&%O#Z&H=``"-=@"-O"<`````55>)QU93B=.#[#RAH!)!`(E$)"PQ
MP#'`@#H]#Y3``<.)'"3H9!X``(U4)"B)Q8GXZ*O^__^%P`^$Y@```(D$)(G&
MZ$4>```YQ0^&P0```(M$)"B-#(4`````#[8'A,`/A'D!```\/0^$<0$``(GX
MZP6`^CUT"H/``0^V$(32=?$I^(LU1/!``(U$*`*)!"0!SNAN'0``A<")!@^$
M40$``*%$\$``BU0D*(U/`8L4D`^V!X3`B`)U&.L:C;8`````@\$!#[9!_X/"
M`83`B`)T!#P]=>S&`CTQP(VT)@`````/M@P#B$P"`8/``83)=?$QP(M\)"PS
M/:`200`/A?H```"#Q#Q;7E]=PXVT)@````"#PP$/MD/_@\8!A,"(1O]U[^O,
MBQ5$\$``BS*%]@^$L@```#'`ZPN)]HV\)P````")\(UP`8L,LH7)=?2-!(4,
M````BPT@\$``A<ET-8E$)`2)%"3HZQP``(7`HT3P0`!T?XT,M0````")1"04
MBT0D%,=$L`0`````B70D*.G(_O__B00DB50D',<%(/!```$```#H7!P``(7`
MB40D%'1!BU0D'(T,M0````")!"2)3"0(B4PD&(E4)`3H31P``(M$)!2+3"08
MHT3P0`#KGS'`Z9W^__^X"````#'VZ6+___^X_____^GV_O__Z&`;``!55XG%
M5E.![)P```"AH!)!`(F$)(P````QP(7M=`:`?0``=1Z+A"2,````,P6@$D$`
M#X4"`0``@<2<````6UY?7<.)+"3HZAL``(7`B40D''32BT0D'(D$).CF&P``
MA<`/A*H```"-6!2_8'!``+D"````B=[SIG39OV)P0`"Y`P```(G>\Z9TR8DL
M).C[&P``B1PDB<;H\1L``(U$!@*)!"3H71L``(E<)`R);"0(B<;'1"0$97!`
M`(D$).BC&P``C40D*(DT)(E$)`3HFQL``(/X_W00BT0D."4`\```/0!```!T
M2HDT).C&&P``B30DZ-X:``"+1"0<B00DZ$(;``"%P`^%7/___XVV`````(M$
M)!R)!"3HH!H``(DL).@X&P``Z?O^__^-M"8`````B?#HR?[__^NUZ"(:``!F
MD%.)PX/L&*&L$D$`B00DZ`4:```%2`$``(D<)(E$)`3HF!H``(7`=`B#Q!@Q
MP%O#D*&L$D$`B00DZ-L9``"+@%@!```E`/```#T`0```=!^AK!)!`(D$).B\
M&0``BX!8`0``)0#P```]`*```'6ZB1PDQT0D!`(```#H[AD``(7`#Y3`@\08
M#[;`6\.0C70F`%575E-0N"@0``#HB1D``"G$B=6+!`2-="0<QT0D"/\/``#'
M!"1K<$``B70D!(G'H:`200")A"0<$```,<#H.AH``(7`?FJ-6/^`?!P<`'15
MB<.)7"0(QT0D!"\```")-"3HUAD``(7`=$:AK!)!`(D$).@1&0``B5PD#(ET
M)`B)?"0$B00DZ'T8``"+A"0<$```,P6@$D$`=36!Q"P0``!;7E]=PX7;=:F-
MM"8`````H:P200")!"3HRQ@``(EL)`B)?"0$B00DZ$,8``#KO.B\&```C;8`
M````C;\`````@^P<H:P200")!"3HF!@``,=$)`Q0<T``QT0D"'!40`#'1"0$
M>G!``(D$).@@&```@\0<PXVV`````(V_`````(/L+*&L$D$`B00DZ%@8``")
M!"3HJ/___Z&L$D$`B00DZ$,8``#'1"04`````,=$)!"6<$``QT0D#)9P0`#'
M1"0($%!``,=$)`27<$``B00DZ+,7``"#Q"S#C;0F`````#'`PXVV`````(V\
M)P````!55[A,$```5E/H^A<``"G$BYPD8!```(U4)#B+M"1D$```H:`200")
MA"0\$```,<"XK'!``.A&^?__QT0D!"\```")'"2)Q>C(&```A<!T+(G:N+5P
M0`#H1/K__XG8B[PD/!```#,]H!)!``^%7@$``('$3!```%M>7UW#B1PDZ*@8
M``")-"2)Q\=$)`3"<$``Z+88``")QHU'`8E$)"R0A?9TK(`^``^$R````(7M
M=!2)="0$B2PDZ%\8``"%P`^$E0```(DT).A?&```@'P&_R]U#H/H`8!\!O\O
MQ@0&`'3R`T0D+#W_#P``#X]?____C40D/(E<)!#'1"0,Q'!``(ET)`C'1"0$
MQG!``(D$).CO%P``H:P200")!"3HUA8```5(`0``B40D!(U$)#R)!"3HU1<`
M`(7`=1^AK!)!`(D$).BP%@``BX!8`0``)0#P```]`(```'0^QT0D!,)P0`#'
M!"0`````Z-X7``")QNDK____A>UT$+D"````OV!P0`")[O.F=-&^8'!``+@!
M````Z4K___^-1"0\QT0D!`$```")!"3HH!8``(7`=:J-5"0\N+5P0`#HXOC_
M_XU$)#R)!"3H6A<``.F/_O__Z"P6``"-M@````"-OP````"+1"0$#[8(A,ET
M%8U0`6:0@/DO#[8*#T3"@\(!A,EU\//#C;0F`````(V\)P````!64X/L%(M<
M)""%VP^$GP````^V,XGPA,`/A)(```")'"3H]A8``(U$`_\YPW()ZQR#Z`$Y
MPW09@#@O=/3K!V:0@#@O=#"#Z`$YPW+T.<-U)8GP/"]T6L=$)"#$<$``@\04
M6U[IK!8``)"-M"8`````@#@O=0>#Z`$YPW+T*=B-2`*)PC'`@?D`$```=QR-
M0@&)7"0$QP0D(.!``(E$)`CHB!8``+@@X$``@\046U[#D,=$)"!@<$``@\04
M6U[I4A8``(VT)@````"-O"<`````5U:#[#2AH!)!`(E$)"PQP*&$<4``B40D
M"J&(<4``B40D#J&,<4``B40D$J&0<4``B40D%J&4<4``B40D&J&8<4``B40D
M'J&<<4``B40D(J&@<4``B40D)@^W!:1Q0`!FB40D*KC-<$``Z.+V__^XUG!`
M`.C8]O__N-YP0`#HSO;__[CG<$``Z,3V__^X[G!``.BZ]O__N/YP0`#HL/;_
M_[BL<$``Z*;V__^X"G%``.B<]O__N!1Q0`#HDO;__[@><4``Z(CV__^XM7!`
M`.A^]O__C50D!+@H<4``Z,#U__^%P'0,B<*X%'%``.C0]O__C50D!+@Y<4``
MZ*+U__^%P'0,B<*X2W%``.BR]O__C50D!+A6<4``Z(3U__^%P'0PB<*XK'!`
M`.B4]O__NH)Q0`"X[G!``.B%]O__BT0D+#,%H!)!`'5+@\0T7E_#C78`C50D
M!+AF<4``Z$+U__^%P'0.B<*X"G%``.A2]O__Z[R-="0;OW=Q0`"Y"@```/.F
M=:J-5"0EN`IQ0`#H,/;__^N:Z(D3``")]HV\)P````"#[!RAH!)!`(E$)`PQ
MP*$(8$``@_C_=!B+3"0,,PV@$D$`=4"#Q!S#D(VT)@````"-5"0(N`IQ0`#H
MPO3__X7`=!X/MA"`^C`/E<"$T@^5PB'0#[;`HPA@0`#KO(UT)@`QP.ONZ!<3
M``"-M"8`````@^P<Z)03``")!"3HA!,``(7`=`R+`(/$',.-M@`````QP.OR
MC;8`````C;\`````5U93@^PPC50D*(MT)$"AH!)!`(E$)"PQP+BF<4``Z#WT
M__^%P'0J@#@`B<-T(XET)`2)!"3HZA,``(7`="*+1"0L,P6@$D$`=62#Q#!;
M7E_#B?*XIG%``.@C]?__Z]^0B30DZ*03``")'"2)Q^B:$P``C40'`HD$).@&
M$P``B5PD$(G'QT0D#,)P0`")="0(QT0D!,9P0`")!"3H1!,``(GZN*9Q0`#H
MU/3__^N0Z"T2``"-M@````"-O"<`````55>X3($``%93Z!H2```IQ(N$)&"!
M``"-5"1$QT0D2+=Q0`#'1"1,8'!``,=$)%``````QT0D8$MQ0`#'1"1DO'%`
M`(E$)##'1"1HPW%``*&@$D$`B80D/($``#'`N*QP0`#'1"1LRW%``,=$)'#0
M<4``QT0D=`````#'1"14JW%``,=$)%C4<4``QT0D7`````#H"?/__X7`=`F`
M.``/A2P$``#H9_[__X7`B<,/A#P$``")'"3HD1(``(U$``&)!"3H_1$``(`[
M`(G&="2)]P^V`XD\)(/#`<=$)`3=<4``@\<"B40D".@W$@``@#L`==Z-?"1D
MN$MQ0`#K#8UV`(/'!(M'_(7`="N-5"1$Z(WR__^%P(G#=.>`.`!TXNC=]O__
MA<!TV8D<).@5$@``A<")PW4\BUPD2#'_ZQ*-=@"#QP&+7+Q(A=L/A%8!``"`
M.P`/A$T!``")V.BA]O__A<!TW8D<).C9$0``A<!TT8G#B=J+"H/"!(V!__[^
M_O?1(<@E@("`@'3JB<'!Z1"I@(````]$P8U*`@]$T8G!`,&)\8/:`RG:BSF#
MP02-A__^_O[WUR'X)8"`@(!TZHG'B<W![Q"I@(````]$QXUY`HG!#T3O`,&#
MW0,I]8V\*@0$``")/"3HVA```(ET)!3'1"00XG%``(G&QT0D#,1P0`")7"0(
MQT0D!.=Q0`")!"2)1"0LZ`P1``#'1"0$P`$``(DT).B\$```@_C_='^AK!)!
M`(D$).C>#P``!4@!``")1"0$BT0D+(D$).AM$```A<")PP^$F@```(M$)#`Q
M[8L8Z*(/``"+3"0LQT0D!&AR0`")7"0(B4PD#(M`#(D$).@'$```BXPD/($`
M`#,-H!)!`(GH#X5T"@``@<1,@0``6UY?7<,QV^G%_O__Z*L/``"#.!$/A'/_
M___HG0\``(LPBT0D,#'MBQCH.@\``(M,)"R)="00B5PD",=$)`3P<4``B4PD
M#(M`#(D$).B;#P``ZY*AK!)!`(D$).@8#P``BX!8`0``)0#P```]`$````^%
M0____Z&L$D$`B00DZ/4.``"+L&`!``#H?@\``#G&#X4C____H:P200")!"3H
MU0X``(N`6`$``"7_`0``/<`!```/A0#___^)/"3H8P\``(U4)$2)Q;BF<4``
MZ"_P__^%P`^$UP$``(D$).C##P``B40D!(M$)#"+`(D$).B-]O__A<")1"0X
M#X2Q`0``BW0D.(DT).BA#P``@_@'?C")="0XC70&^+\K<D``N0D```#SIG4:
M@_@(#X3E!P``BW0D.(!\!O<O#X36!P``9I#HN_K__X7`=2"+1"0XQT0D!```
M`0")!"3H[PX``(7`B40D.`^%5@$``+J"<4``N`IQ0`#HJ/#__^A_#@``BW0D
M+,=$)!26<$``B40D$,=$)`S$<$``QT0D!$AR0`")+"2)="0(Z-L.``#K0Y"-
M="8`Z`<.``"#.!%U28/#`>@Z#@``QT0D&)9P0`")7"04B40D$,=$)`S$<$``
MB70D",=$)`16<D``B2PDZ)8.``#'1"0$P`$``(DL).A&#@``@_C_=*V+1"0L
MB00DZ.4-``")ZKBL<$``Z`7P__^)+"3H??K__^F]_?__D(VT)@````")!"2)
M1"0LZ&3Z__^+1"0LB00DZ%P.``")Q>F6_?__O@$```"XJW%``.L0D(UT)@"+
M1+14@\8!A<!T)8U4)$3HC.[__X7`=.B`.`!TXXD$).@?#@``A<!TUXG#Z8/[
M___'!"0-````N[!Q0`#H@PT``(G&Z8;[__^0C;0F`````(M$)#"+`(E$)#CI
M0/[__\=$)`P"````QT0D!.[___^)QL=$)`C_____B00DZ#,-``"-A"0\`0``
MQT0D"`8```")-"2)1"0$Z&`-``"`O"0\`0```'4.@+PD/0$``$,/A,,&``"+
M1"0XQT0D#`````#'1"0$`````,=$)`@`````B00DZ-P,``#'1"1X`2-%9\=$
M)'P`````QX0D@````(FKS>_'A"2$`````````,>$)(@```#^W+J8QX0DC```
M``````#'A"20````=E0R$,>$))0`````````QX0DF````/#ATL/'A"2<````
M`````,>$)*``````````QX0DI`````````#'A"2H`````````,>$)*P`````
M````QX0D\`````````");"0\D(UT)@"-A"0\`0``QT0D"`"```")1"0$BT0D
M.(D$).A=#```A<")PP^.4`(``(NT)*````"+O"2D````B=B9P>`#B5PD,(E4
M)#0!\(/_``^'X`$``#G&#X?8`0``B[0DJ````(N\)*P```"+5"0TB80DH```
M`(M$)##'A"2D``````````^LT!W!ZAT!QHN$)/`````1UXFT)*@```")O"2L
M````A<`/A'@!``"_0````(V,)+`````IQSG[#T[[`<B-C"0\`0``B7PD"(D$
M)(E,)`3H@`L``(N$)/`````!^(/X0(F$)/`````/A1/___^-A"0\`0``*?L!
MQXU$)'CH7\?__X/[/P^.W````(/K0(G]B=C!Z`:-<`&)1"0PP>8&`?Z+10"#
MQ4")A"2P````BT7$B80DM````(M%R(F$)+@```"+1<R)A"2\````BT70B80D
MP````(M%U(F$),0```"+1=B)A"3(````BT7<B80DS````(M%X(F$)-````"+
M1>2)A"34````BT7HB80DV````(M%[(F$)-P```"+1?")A"3@````BT7TB80D
MY````(M%^(F$).@```"+1?R)A"3L````C40D>.B4QO__.>X/A4S___^+;"0P
MB=B#X,"-?`=`P>4&*>N#^P2-E"2P````B?AS7#')]L,"=`L/MPAFB0JY`@``
M`/;#`70'#[8$"(@$"HF<)/````#IXOW__V:0C;PD/`$``.G5_O__C70F`(NT
M)*@```"+O"2L````@\8!@]<`Z1W^__^-M"8`````B=HQP(/B_(L,!XF,!+``
M``"#P`0YT'+OC8PDL````(T4`0'XZ7O___^+1"0XBVPD/(D$).AK"0``BX0D
MJ````(N<)*````"+M"2D````BY0DK````(E$)#")V`^L\`.)5"0T@^`_C4@!
MQH0$L````("#^3@/C[`"``"Z.````(V$)+`````IR@'(@_H$#X/6`@``A=)T
M#/;"`L8```^%CP,``(M4)#2+1"0PC;PD$P$``(B<).\````/K-`8BU0D-(B$
M).@```"+1"0P#ZS0$(M4)#2(A"3I````BT0D,`^LT`B(A"3J````#[9$)#"(
MA"3K````B=@/K/`8B(0D[````(G8#ZSP$(B$).T```")V`^L\`B(A"3N````
MC40D>.CJQ/__BTPD>(M<)'R)SHB,)`(!```/K-X8B?")S@^LWA"(A"3_````
MB?*)SHN,)(`````/K-X(B)0D``$``(GSB<Z(C"0&`0``B)PD`0$``(N<)(0`
M```/K-X8B?*)S@^LWA"(E"0#`0``B?*)SHN,)(@````/K-X(B)0D!`$``(GS
MB<Z(C"0*`0``B)PD!0$``(N<)(P````/K-X8B?*)S@^LWA"(E"0'`0``B?*)
MSHN,))`````/K-X(B)0D"`$``(GSB<Z(C"0.`0``B)PD"0$``(N<))0````/
MK-X8B?*)S@^LWA"(E"0+`0``B?*)SHN,))@````/K-X(B)0D#`$``(GSB<Z(
MC"02`0``B)PD#0$``(N<))P````/K-X8B?*)S@^LWA"(E"0/`0``B?*)S@^L
MW@B(E"00`0``B?.-M"0``0``B)PD$0$``(V<)#L!``#I"0```(UV``^V!H/&
M`0^VP(D\),=$)`3=<4``B40D"(/'`NBT!P``.?MUW(V$)!,!``#&A"0[`0``
M`,=$)!26<$``B40D$(M$)"S'1"0,Q'!``,=$)`0Y<D``B2PDB40D".AU!P``
MZ>_X__^+1"0PBU`$A=(/A!WX__^)%"2)5"0PZ'T'``"#^`,/A@CX__^+5"0P
MOS1R0`"Y!0```(UT`OSSI@]%5"0XB50D..GF]___C90DL````+A``````<HI
MR(G!=`TQP,8$`@"#P`$YR'+UC40D>(V\)+````#HM<+__[D.````,<#SJ^DZ
M_?__C4@$QP``````QT00_`````"#X?PIR`'0@^#\@_@$#X(6_?__@^#\B<(Q
MP,<$`0````"#P`0YT'+RZ?S\__^`O"0^`0``00^%+_G__X"\)#\!``!##X4A
M^?__@+PD0`$``$@/A1/Y__^`O"1!`0``10^%!?G__XMT)#C'1"0,`@```,=$
M)`3&____QT0D"/____^)-"3HX04``(V$)#P!``#'1"0(*0```(DT)(E$)`3H
M#@8``,=$)!26<$``C80D/`$``.F)_O__,?]FB7P0_NEE_/__Z`T%``"-M@``
M``"-O"<`````55=64X/L+*&@$D$`B40D'#'`H:P200")!"3HV00``(G#H:P2
M00")!"3HR@0```^V0&B-5"08B$-IN.YP0`#H/>;__X7`=`F`.#$/A,`"``#H
MB^___Z&L$D$`B00DZ)8$``#'1"0,"P```,=$)`@!````QT0D!+MR0`")!"3H
M+@0``(7`#X3(````@STH\$```8M`$(MX&`^.M0```#'VNP$```#K-(UV`(V\
M)P````"+11")1"0$H23P0`"+!)B)!"3H504``(7`=7>A*/!``(/#`8/&`3G8
M?FVAK!)!`(D$).@1!```QT0D#`````")="0(B7PD!(D$).C1`P``A<!T/XLH
MBT4()0`$(``]``0``'2<H:P200")!"3HUP,``,=$)`P"````QT0D"`````")
M;"0$B00DZ$L#``#I<?___Z$H\$``C5#_.?(/A((!``"AK!)!`(D$).B9`P``
MQT0D#`,```#'1"0(`0```,=$)`3`<D``B00DZ#$#``"%P(G#=#6+0!"+,*&L
M$D$`B00DZ&$#``#'1"0(PG)``(D$)(ET)`3HU0(``(M#$(L8]D,*0`^%O0$`
M`*&L$D$`B00DZ#$#``#'1"0,`P```,=$)`@!````QT0D!,=R0`")!"3HR0(`
M`(7`B<-T.HU4)!BXM7!``.B%Y/__A<")Q@^$2P$``(M#$(L8H:P200")!"3H
MX0(``(ET)`B)7"0$B00DZ%D"``"AK!)!`(D$).C$`@``QT0D"`$```#'1"0$
MR7)``(D$).AL`@``B<.AK!)!`(D$).B=`@``QT0D"-YR0`")!"2)7"0$Z!$"
M``"AK!)!`(D$).A\`@``C50D&,9`:0"XK'!``.CRX___A<"C+/!```^$A0``
M`,=$)`3``0``B00DZ!D#``"-4`&#^@%V"8/X$0^%W@```(M$)!PS!:`200`/
MA<D```"#Q"Q;7E]=PX/X`0^.=?[__[L!````H:P200"#PP&)!"3H!@(``(E\
M)`2)!"3HR@$``#D=*/!``'_<Z4?^__^0C70F`(!X`0`/A#O]___I,?W__Y"A
M)/!``(D$).BS[___A<"C+/!```^%8?___\<$).1R0`#H>@$``(UV`(V\)P``
M``"A)/!``(LPBT,0BQBAK!)!`(D$).B/`0``B?*)V.C^Y___Z;#^__^AK!)!
M`(D$).AT`0``B5PD!(D$).@8`0``Z27^___H9@$``(E$)`RA+/!``(E$)`BA
M)/!``(L`QP0D"'-``(E$)`3H`@$``&:05E.#[!2+="0@B30DZ#_K__\/M@B$
MR71$C5`!B<.-=@"`^2\/M@H/1-J#P@&$R77PZ(OM__^%]G0=A<!T&8`^`'04
MQT0D!.)Q0`")'"3H,@(``#G8=`J#Q!1;7L.)P^O0@\04B?!;7NETY?__D)"0
MD/\E"!-!`)"0_R4$$T$`D)#_)0`300"0D/\E_!)!`)"0_R7X$D$`D)#_)?02
M00"0D/\E\!)!`)"0_R7L$D$`D)#_)>@200"0D/\EY!)!`)"0_R7@$D$`D)#_
M)=P200"0D/\EV!)!`)"0_R74$D$`D)#_)=`200"0D/\ES!)!`)"0_R7($D$`
MD)#_)<0200"0D/\EP!)!`)"0_R6\$D$`D)#_);@200"0D/\EM!)!`)"0_R6P
M$D$`D)#_)<P100"0D/\E4!)!`)"0_R54$D$`D)#_)9P200"0D%%0/0`0``"-
M3"0,<A6!Z0`0``"#"0`M`!```#T`$```=^LIP8,)`%A9PY"0_R7$$4$`D)#_
M)<@100"0D/\ET!%!`)"0_R7T$4$`D)#_)?P100"0D/\E`!)!`)"0_R40$D$`
MD)#_)10200"0D/\E&!)!`)"0_R4<$D$`D)#_)=@100"0D/\EW!%!`)"0_R7D
M$4$`D)#_)>@100"0D/\E,!)!`)"0_R4T$D$`D)#_)3@200"0D/\E/!)!`)"0
M_R5`$D$`D)#_)>P100"0D/\E2!)!`)"0_R58$D$`D)#_)5P200"0D/\E8!)!
M`)"0_R5D$D$`D)#_)6@200"0D/\E;!)!`)"0_R7P$4$`D)#_)70200"0D/\E
M>!)!`)"0_R5\$D$`D)#_)8`200"0D/\EA!)!`)"0_R6($D$`D)#_)8P200"0
MD/\ED!)!`)"0_R64$D$`D)!FD&:05E.#[!2+7"0DQP0D"````(MT)"#HYP0`
M`(7;#X3/`0``BQ7@$4$`A<#'0P2H````QT,(V0<``,=##`````#'@X``````
M````QX.$````/@$``,=#+-!=0`#'0S#<74``QT,41/!``(E3$(ES*,=#2!!;
M0`#'0TP@6T``QT-0,%M``,=#5$!;0`#'0R1`\$``QP,`````QT,8M%5``,=#
M'(150`#'0R`$5D``QT-$8%M``,>#D````'A;0``/A,$```"+4'B%T@^$M@``
M`*%(\$``A<`/A:D```"A(&!``(7`#X0Y`@``H21@0`"%P`^$'`(``*$H8$``
MA<`/A/\!``"A+&!``(7`#X3B`0``H3!@0`"%P`^$Q0$``(LU-&!``(7V#X2G
M`0``BPTX8$``A<D/A(D!``"A/&!``(7`#X1L`0``H2!@0`")`J$D8$``B4($
MH2A@0`")0@BA+&!``(E"#*$P8$``B4(0H31@0`")0A2A.&!``(E"&*$\8$``
MB4(<QP0D`````.C)`P``@^P$B4-\QT,T`&!``,=#.$1@0`#'0SP`X$``QT-`
M)/%``,>#E````,"P0`#'@Y@```!XLT``QX.<``````!``,<$)`````#H!0,`
M`(/$%+@!````6U[""`"0C;0F`````(N(I````(L5X!%!`(7`QT`$J````,=`
M"-D'``#'0`P`````QX"``````````(D*QX"$````/@$``,=`+-!=0`#'0##<
M74``QT`41/!``(EP*,=`2!!;0`#'0$P@6T``QT!0,%M``,=`5$!;0`#'0"1`
M\$``QP``````QT`8M%5``,=`'(150`#'0"`$5D``QT!$8%M``,>`D````'A;
M0``/A.W^__^+4'B)PX72#X4J_O__Z=O^__^0C;0F`````(M"'*,\8$``Z8?^
M__^-=@"+0ABC.&!``.EJ_O__C78`BT(4HS1@0`#I3/[__XUV`(M"$*,P8$``
MZ2[^__^-=@"+0@RC+&!``.D1_O__C78`BT((HRA@0`#I]/W__XUV`(M"!*,D
M8$``Z=?]__^-=@"+`J,@8$``Z;O]__^0D)"0@^P<BT0D(,=$)`0`````B00D
MZ+G\__^#[`C_%=0100"#[!RA4/!``(E$)`BA5/!``(E$)`2A6/!``(D$).@>
M`0``@^P,@\0<PXVT)@````"#["R+1"0TBU0D,(M,)#B#^`$/A)@```!R5H/X
M`G0A@_@#=3R)3"0XQT0D-`,```")5"0P@\0LZ=0```"-="8`B4PD.,=$)#0"
M````B50D,(/$+.FX````D(VT)@````"X`0```(/$+,(,`)"-="8`B4PD",=$
M)`0`````B10DZ(P```"#[`R%P'3:BQ5,\$``B40D'(D4).C;````QP5,\$``
M_____XM$)!SKN(UV`#'`A<G'1"0$P%E```^4P(D4)(D56/!``,<%5/!```$`
M``")#5#P0`"C2/!``.BN````@_C_HTSP0``/E<"#Q"P/ML#"#`"-=@"-O"<`
M````Z?O^__^0D)"0D)"0D)"0D+@!````P@P`D)"0D)"0D)#SPY"0D)"0D)"0
MD)"0D)"0\\.0D)"0D)"0D)"0D)"0D//#D)"0D)"0D)"0D)"0D)#SPY"0D)"0
MD)"0D)"0D)"0\\.0D)"0D)"0D)"0D)"0D/\E^!%!`)"0_R4$$D$`D)#_)0@2
M00"0D/\E3!)!`)"04X/L&(M$)"2+7"0@QT0D!&#P0`")!"3HU/K__X/L"(E<
M)"#'1"0D8/!``(/$&%OI#````)"0D)"0D)"0D)"0D/\E#!)!`)"0_R44$T$`
MD)!5B>564X/D\(/L0(M%#(E$)"R+11")1"0HZ&;Y__^AH!)!`(E$)#PQP(U$
M)"B)1"0(C40D+(E$)`2-10B)!"3H;/C__\=$)`CH5$``QT0D!.A40`#'!"3P
M5$``Z-#X__^`/:@200``=2[H*OC__X7`HS#P0``/A%P!``")!"3H#?C__Z&L
M$D$`B00DZ*CX___&@+D%````H:P200")!"3HE/C__X"(1@,```*AK!)!`(D$
M).B`^/__@(A&`P```8M%"(T$A10```")!"3H__?__XMT)"R+70BC)/!``,<%
M*/!```0```"+%H/[`<=`!$ES0`#'0`Q,<T``B1"+%0Q@0`")4`BZ`0```'X2
MC70F`(L,EHE,D`R#P@$YVG7RQT28#`````")1"0,@\,#H3#P0`")7"0(QT0D
M$`````#'1"0$H#Q``(D$).@J]___A<")PW1&H3#P0`")!"3H)_?__XU4)#BX
M_G!``.A1V?__A<!T.*$P\$``B00DZ`#W___H(_?__XMT)#PS-:`200")V'50
MC67X6UY=PZ$P\$``B00DZ,GV__^)P^NIH2SP0`"%P'0*B00DZ$3V___KM8U4
M)#BXK'!``.CTV/__A<"C+/!``'2>Z]W'!"0!````Z,'W___H6/?__Y"0D)"0
MD)"058GE7>G'LO__D)"0D)"0D/_____`74```````/____\`````````````
M````````````````````````````````````````_____UQS0```````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````8WEG9V-C
M7W,M,2YD;&P`7U]R96=I<W1E<E]F<F%M95]I;F9O`%]?9&5R96=I<W1E<E]F
M<F%M95]I;F9O`&-Y9V=C:BTQ-BYD;&P`7TIV7U)E9VES=&5R0VQA<W-E<P``
M+@`N+@`E<R\E<P`O<')O8R]S96QF+V5X90!$>6YA3&]A9&5R.CIB;V]T7T1Y
M;F%,;V%D97(``$EN=&5R;F%L<SHZ4$%2.CI"3T]4`%!!4E]414U0`%!!4E]0
M4D]'3D%-10`Z`"\`)7,E<R5S`%!%4DPU3$E"`%!%4DQ,24(`4$523#5/4%0`
M4$523$E/`%!!4E])3DE424%,25I%1`!005)?4U!!5TY%1`!005)?0TQ%04X`
M4$%27T1%0E5'`%!!4E]#04-(10!005)?1TQ/0D%,7T1%0E5'`%!!4E]'3$]"
M04Q?5$U01$E2`%!!4E]435!$25(`4$%27T=,3T)!3%]414U0`%!!4E]'3$]"
M04Q?0TQ%04X`4$%27T-,14%./0`Q`%]?14Y67U!!4E]#3$5!3E]?("`@("`@
M("`@("`@("`@``!0051(`%5315(`4UE35$5-`"]T;7``5$U01$E2`%1%35!$
M25(`5$5-4`!435``55-%4DY!344`)3`R>`!P87(M`"5S)7,E<R5S`"5S.B!C
M<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R
M<FYO/25I*0H`<&%R;"YE>&4`+G!A<@`E<R5S8V%C:&4M)7,E<P`E<R5S=&5M
M<"TE=25S`"5S)7-T96UP+25U+25U)7,``"5S.B!P<FEV871E('-U8F1I<F5C
M=&]R>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y
M('EO=7(@;W!E<F%T:6]N*0H`05)'5@`8`'!E<FP`,`!005(Z.E!A8VME<CHZ
M5D524TE/3@`Q+C`T,`!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R
M>0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O
M<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*`"UE`"TM``!P97)L>'-I+F,```!E
M=F%L("=E>&5C("]U<W(O8FEN+W!E<FP@("U3("0P("1[,2LB)$`B?2<*:68@
M,#L@(R!N;W0@<G5N;FEN9R!U;F1E<B!S;VUE('-H96QL"G!A8VMA9V4@7U]P
M87)?<&P["FUY("@D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I;&4I.PI%
M3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]*2!["G)E<75I<F4@1FEL93HZ5&5M
M<#L*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*<F5Q=6ER92!&:6QE.CI3<&5C
M.PIM>2`D=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'!A<E]T
M96UP*3L*;W5T<RAQ<7M296UO=FEN9R!F:6QE<R!I;B`B)'!A<E]T96UP(GTI
M.PI&:6QE.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@*"`M9"`I(#\@<FUD:7(@
M.B!U;FQI;FL@?2P@)'!A<E]T96UP*3L*<FUD:7(@)'!A<E]T96UP.PII9B`H
M+60@)'!A<E]T96UP("8F("1>3R!N92`G35-7:6XS,B<I('L*;7D@)'1M<"`]
M(&YE=R!&:6QE.CI496UP*`I414U03$%412`]/B`G=&UP6%A86%@G+`I$25(@
M/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'1O<&1I<BDL"E-51D9)6"`]
M/B`G+F-M9"<L"E5.3$E.2R`]/B`P+`HI.PIP<FEN="`D=&UP("(C(2]B:6XO
M<V@*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R
M7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF
M:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G(B`N("1T;7`M
M/F9I;&5N86UE("X@(B<*(CL*8VAM;V0@,#<P,"PD=&UP+3YF:6QE;F%M93L*
M;7D@)&-M9"`]("1T;7`M/F9I;&5N86UE("X@)R`^+V1E=B]N=6QL(#(^)C$@
M)B<["F-L;W-E("1T;7`["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<2A3<&%W;F5D
M(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("D*+B`D
M=&UP+3YF:6QE;F%M92D["GT*?2!]"D)%1TE.('L*26YT97)N86QS.CI005(Z
M.D)/3U0H*2!I9B!D969I;F5D("9);G1E<FYA;',Z.E!!4CHZ0D]/5#L*979A
M;"!["E]P87)?:6YI=%]E;G8H*3L*;7D@)'%U:65T(#T@(21%3E9[4$%27T1%
M0E5'?3L*;7D@)4-O;F9I9R`]("@*<&%T:%]S97`@("`@/3X@*"1>3R`]?B`O
M7DU35VEN+R`_("<[)R`Z("<Z)RDL"E]E>&4@("`@("`@(#T^("@D7D\@/7X@
M+UXH/SI-4U=I;GQ/4S)\8WEG=VEN*2\@/R`G+F5X92<@.B`G)RDL"E]D96QI
M;2`@("`@(#T^("@D7D\@/7X@+UY-4U=I;GQ/4S(O(#\@)UQ<)R`Z("<O)RDL
M"BD["E]S971?<')O9VYA;64H*3L*7W-E=%]P87)?=&5M<"@I.PIM>2`H)'-T
M87)T7W!O<RP@)&1A=&%?<&]S*3L*>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]
M('-U8B![?3L*;W!E;B!?1D@L("<\)RP@)'!R;V=N86UE(&]R(&QA<W0["F)I
M;FUO9&4H7T9(*3L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D<')O9VYA;64[
M"FUY("1O9F9S970@/2`U,3(["FUY("1I9'@@/2`M,3L*=VAI;&4@*#$I"GL*
M)&]F9G-E="`]("1S:7IE(&EF("1O9F9S970@/B`D<VEZ93L*<V5E:R!?1D@L
M("TD;V9F<V5T+"`R(&]R(&1I92!Q<5MS965K(&9A:6QE9"!O;B`B)'!R;V=N
M86UE(CH@)"%=.PIM>2`D;G)E860@/2!R96%D(%]&2"P@)&)U9BP@)&]F9G-E
M=#L*9&EE('%Q6W)E860@9F%I;&5D(&]N("(D<')O9VYA;64B.B`D(5T@=6YL
M97-S("1N<F5A9"`]/2`D;V9F<V5T.PHD:61X(#T@<FEN9&5X*"1B=68L(")<
M;E!!4BYP;5QN(BD["FQA<W0@:68@)&ED>"`^/2`P('Q\("1O9F9S970@/3T@
M)'-I>F4@?'P@)&]F9G-E="`^(#$R."`J(#$P,C0["B1O9F9S970@*CT@,CL*
M?0IL87-T('5N;&5S<R`D:61X(#X](#`["B1O9F9S970@+3T@)&ED>"`M(#0[
M"G-E96L@7T9(+"`M)&]F9G-E="P@,CL*<F5A9"!?1D@L("1B=68L(#0["G-E
M96L@7T9(+"`M)&]F9G-E="`M('5N<&%C:R@B3B(L("1B=68I+"`R.PIR96%D
M(%]&2"P@)&)U9BP@-#L*)&1A=&%?<&]S(#T@*'1E;&P@7T9(*2`M(#0["FUY
M("5R97%U:7)E7VQI<W0["G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D
M(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B
M=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q*%5N<&%C:VEN9R!F
M:6QE("(D9G5L;&YA;64B+BXN*2D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]
M?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S
M96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\
M*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH
M(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN
M*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]T96UP9FEL
M92@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*)%!!4CHZ2&5A=GDZ.D9U;&Q#
M86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE.PHD4$%2.CI(96%V>3HZ1G5L
M;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA;64["GT*96QS:68@*"`D9G5L
M;&YA;64@/7X@;7Q>+S]S:&QI8B]\(&%N9"!D969I;F5D("1%3E9[4$%27U1%
M35!]("D@>PIM>2`D9FEL96YA;64@/2!?=&5M<&9I;&4H(B1B87-E;F%M921E
M>'0B+"`D8G5F+"`P-S4U*3L*;W5T<R@B4TA,24(Z("1F:6QE;F%M95QN(BD[
M"GT*96QS92!["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2`]"B1005(Z.DAE
M879Y.CI-;V1U;&5#86-H97LD9G5L;&YA;65](#T@>PIB=68@/3X@)&)U9BP*
M8W)C(#T^("1C<F,L"FYA;64@/3X@)&9U;&QN86UE+`I].PI]"G)E860@7T9(
M+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D
M;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U
M;&4["FUY("1F:6QE;F%M92`](&1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U
M;&5]('Q\(&1O('L*;7D@)&ME>3L*9F]R96%C:"`H:V5Y<R`E<F5Q=6ER95]L
M:7-T*2!["FYE>'0@=6YL97-S("]<421M;V1U;&5<120O.PHD:V5Y(#T@)%\[
M(&QA<W0["GT*9&5L971E("1R97%U:7)E7VQI<W1[)&ME>7T@:68@9&5F:6YE
M9"@D:V5Y*3L*?2!O<B!R971U<FX["B1)3D-[)&UO9'5L97T@/2`B+VQO861E
M<B\D9FEL96YA;64O)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!3GT@86YD
M(&1E9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/.CI&:6QE
M+3YN97=?=&UP9FEL92!O<B!D:64@)"$["F)I;FUO9&4H)&9H*3L*<')I;G0@
M)&9H("1F:6QE;F%M92T^>V)U9GT["G-E96LH)&9H+"`P+"`P*3L*<F5T=7)N
M("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W1E;7!F:6QE*"(D9FEL
M96YA;64M/GMC<F-]+G!M(BP@)&9I;&5N86UE+3Y[8G5F?2D["F]P96X@;7D@
M)&9H+"`G/"<L("1F:6QE;F%M92!O<B!D:64@(F-A;B=T(')E860@)&9I;&5N
M86UE.B`D(2(["F)I;FUO9&4H)&9H*3L*<F5T=7)N("1F:#L*?0ID:64@(D)O
M;W1S=')A<'!I;F<@9F%I;&5D.B!C86YN;W0@9FEN9"`D;6]D=6QE(5QN(CL*
M?2P@0$E.0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@4$%2.CI(96%V
M>3L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A
M=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)
M3SHZ1FEL93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E
M<75I<F5?;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE;F%M97T@;W(@
M)&9I;&5N86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA;64@/7X@+UPN
M<&UC/R0O:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I
M;&5N86UE('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L
M)"\["GT*?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA;65].PI]"FQA
M<W0@=6YL97-S("1B=68@97$@(E!+7#`P,UPP,#0B.PHD<W1A<G1?<&]S(#T@
M*'1E;&P@7T9(*2`M(#0["GT*;7D@0'!A<E]A<F=S.PIM>2`H)&]U="P@)&)U
M;F1L92P@)&QO9V9H+"`D8V%C:&5?;F%M92D["F1E;&5T92`D14Y6>U!!4E]!
M4%!?4D554T5].R`C('-A;FET:7IE("A2155312!M87D@8F4@82!S96-U<FET
M>2!P<F]B;&5M*0HD<75I970@/2`P('5N;&5S<R`D14Y6>U!!4E]$14)51WT[
M"FEF("@A)'-T87)T7W!O<R!O<B`H)$%21U9;,%T@97$@)RTM<&%R+6]P=&EO
M;G,G("8F('-H:69T*2D@>PIM>2`E9&ES=%]C;60@/2!Q=R@*<"`@(&)L:6)?
M=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU("`@=6YI;G-T86QL7W!A<@IS("`@
M<VEG;E]P87(*=B`@('9E<FEF>5]P87(**3L*:68@*$!!4D=6(&%N9"`D05)'
M5ELP72!E<2`G+2UR975S92<I('L*<VAI9G0@0$%21U8["B1%3E9[4$%27T%0
M4%]2155317T@/2!S:&EF="!`05)'5CL*?0IE;'-E('L@(R!N;W)M86P@<&%R
M;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I;F,["G=H:6QE("A`05)'5BD@>PHD
M05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q<&EU<U1V72DH+BHI+R!O<B!L87-T
M.PII9B`H)#$@97$@)TDG*2!["G!U<V@@0&%D9%]T;U]I;F,L("0R.PI]"F5L
M<VEF("@D,2!E<2`G32<I('L*979A;"`B=7-E("0R(CL*?0IE;'-I9B`H)#$@
M97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A<F=S+"`D,CL*?0IE;'-I9B`H)#$@
M97$@)T\G*2!["B1O=70@/2`D,CL*?0IE;'-I9B`H)#$@97$@)V(G*2!["B1B
M=6YD;&4@/2`G<VET92<["GT*96QS:68@*"0Q(&5Q("=")RD@>PHD8G5N9&QE
M(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q("=Q)RD@>PHD<75I970@/2`Q.PI]
M"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E;B`D;&]G9F@L("(^/B(L("0R(&]R
M(&1I92`B6%A8.B!#86YN;W0@;W!E;B!L;V<Z("0A(CL*?0IE;'-I9B`H)#$@
M97$@)U0G*2!["B1C86-H95]N86UE(#T@)#(["GT*<VAI9G0H0$%21U8I.PII
M9B`H;7D@)&-M9"`]("1D:7-T7V-M9'LD,7TI('L*9&5L971E("1%3E9[)U!!
M4E]414U0)WT["FEN:71?:6YC*"D["G)E<75I<F4@4$%2.CI$:7-T.PHF>R)0
M05(Z.D1I<W0Z.B1C;60B?2@I('5N;&5S<R!`05)'5CL*)GLB4$%2.CI$:7-T
M.CHD8VUD(GTH)%\I(&9O<B!`05)'5CL*97AI=#L*?0I]"G5N<VAI9G0@0$E.
M0RP@0&%D9%]T;U]I;F,["GT*?0II9B`H)&]U="D@>PI["G)E<75I<F4@24\Z
M.D9I;&4["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1P87(@/2!S:&EF
M="A`05)'5BD["FUY("1Z:7`["FEF("AD969I;F5D("1P87(I('L*07)C:&EV
M93HZ6FEP.CIS971#:'5N:U-I>F4H,3(X*C$P,C0I.PIO<&5N(&UY("1F:"P@
M)SPG+"`D<&%R(&]R(&1I92`B0V%N;F]T(&9I;F0@)R1P87(G.B`D(2(["F)I
M;FUO9&4H)&9H*3L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R
M8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D
M9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92`B
M4F5A9"`G)'!A<B<@97)R;W(Z("0A(CL*?0IM>2`E96YV(#T@9&\@>PII9B`H
M)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG
M*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N
M*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PIL
M;V-A;"`D+R`](%PT.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@4$%2+"`G
M/"<L("1P87(@;W(@9&EE("(D(3H@)'!A<B(["F)I;FUO9&4H4$%2*3L*9&EE
M("(D<&%R(&ES(&YO="!A(%!!4B!F:6QE(B!U;FQE<W,@/%!!4CX@97$@(E!+
M7#`P,UPP,#0B.PI]"D-R96%T95!A=&@H)&]U="D@.PIM>2`D9F@@/2!)3SHZ
M1FEL92T^;F5W*`HD;W5T+`I)3SHZ1FEL93HZ3U]#4D5!5"@I('P@24\Z.D9I
M;&4Z.D]?5U)/3DQ9*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI
M(&]R(&1I92`D(3L*8FEN;6]D92@D9F@I.PHD+R`]("AD969I;F5D("1D871A
M7W!O<RD@/R!<)&1A=&%?<&]S(#H@=6YD968["G-E96L@7T9(+"`P+"`P.PIM
M>2`D;&]A9&5R(#T@<V-A;&%R(#Q?1D@^.PII9B`H(21%3E9[4$%27U9%4D)!
M5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A?%Q`<F5M*2\I('L*<F5Q=6ER
M92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!4CHZ1FEL=&5R.CI0;V13=')I
M<"T^;F5W+3YA<'!L>2A<)&QO861E<BP@)#`I"GT*9F]R96%C:"!M>2`D:V5Y
M("AS;W)T(&ME>7,@)65N=BD@>PIM>2`D=F%L(#T@)&5N=GLD:V5Y?2!O<B!N
M97AT.PHD=F%L(#T@979A;"`D=F%L(&EF("1V86P@/7X@+UY;)R)=+SL*;7D@
M)&UA9VEC(#T@(E]?14Y67U!!4E\B("X@=6,H)&ME>2D@+B`B7U\B.PIM>2`D
M<V5T(#T@(E!!4E\B("X@=6,H)&ME>2D@+B`B/21V86PB.PHD;&]A9&5R(#U^
M('-[)&UA9VEC*"`K*7U["B1M86=I8R`N("1S970@+B`H)R`G('@@*&QE;F=T
M:"@D,2D@+2!L96YG=&@H)'-E="DI*0I]96<["GT*)&9H+3YP<FEN="@D;&]A
M9&5R*3L*)"\@/2!U;F1E9CL*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!005(Z
M.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN:71?
M:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![("$O
M0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D7R!N
M92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#;VYF
M:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*;7D@)69I;&5S.PHD
M9FEL97-[)%]]*RL@9F]R($!$>6YA3&]A9&5R.CID;%]S:&%R961?;V)J96-T
M<RP@=F%L=65S("5)3D,["FUY("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG
M>VQI8E]E>'1].PIM>2`E=W)I='1E;CL*9F]R96%C:"`H<V]R="!K97ES("5F
M:6QE<RD@>PIM>2`H)&YA;64L("1F:6QE*3L*9F]R96%C:"!M>2`D9&ER("A`
M:6YC*2!["FEF("@D;F%M92`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)%]]
M*2!["B1F:6QE(#T@)%\["FQA<W0["GT*96QS:68@*"]>*%Q1)&1I<EQ%7"\H
M+BI;7D-C72DI7%HO:2D@>PHH)&9I;&4L("1N86UE*2`]("@D,2P@)#(I.PIL
M87-T.PI]"F5L<VEF("AM(5XO;&]A9&5R+UM>+UTK+R@N*EM>0V-=*5Q:(2D@
M>PII9B`H;7D@)')E9B`]("1005(Z.DAE879Y.CI-;V1U;&5#86-H97LD,7TI
M('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"F5L<VEF
M("@M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B
M+"`D,2D["FQA<W0["GT*?0I]"FYE>'0@=6YL97-S(&1E9FEN960@)&YA;64@
M86YD(&YO="`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE
M*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"\["F]U=',H(&IO:6X@
M(B(L"G%Q*%!A8VMI;F<@(BDL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@
M.B`D9FEL92P*<7$H(BXN+BD**3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I
M;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIO<&5N
M($9)3$4L("<\)RP@)&9I;&4@;W(@9&EE(")#86XG="!O<&5N("1F:6QE.B`D
M(2(["F)I;FUO9&4H1DE,12D["B1C;VYT96YT(#T@/$9)3$4^.PIC;&]S92!&
M24Q%.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/FYE=RT^87!P;'DH7"1C;VYT
M96YT+"`D9FEL92D*:68@(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA;64@
M/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#;VYT
M96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*?0IO
M=71S*'%Q*%=R:71T96X@87,@(B1N86UE(BDI.PHD9F@M/G!R:6YT*")&24Q%
M(BD["B1F:"T^<')I;G0H<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2D[
M"B1F:"T^<')I;G0H<W!R:6YT9B@*(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z
M.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE"BDI.PHD9F@M/G!R:6YT
M*'!A8VLH)TXG+"!L96YG=&@H)&-O;G1E;G0I*2D["B1F:"T^<')I;G0H)&-O
M;G1E;G0I.PI]"GT*)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2!I9B`D
M>FEP.PHD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@-#`[
M"FEF("@A)&-A8VAE7VYA;64@86YD(&UY("1M=&EM92`]("AS=&%T*"1O=70I
M*5LY72D@>PIM>2`D8W1X(#T@979A;"![(')E<75I<F4@1&EG97-T.CI32$$[
M($1I9V5S=#HZ4TA!+3YN97<H,2D@?0I\?"!E=F%L('L@<F5Q=6ER92!$:6=E
M<W0Z.E-(03$[($1I9V5S=#HZ4TA!,2T^;F5W('T*?'P@979A;"![(')E<75I
M<F4@1&EG97-T.CI-1#4[($1I9V5S=#HZ340U+3YN97<@?3L*;7D@)'-H85]V
M97)S:6]N(#T@979A;"![("1$:6=E<W0Z.E-(03HZ5D524TE/3B!]('Q\(#`[
M"FEF("@D<VAA7W9E<G-I;VX@97$@)S4N,S@G(&]R("1S:&%?=F5R<VEO;B!E
M<2`G-2XS.2<I('L*)&-T>"T^861D9FEL92@D;W5T+"`B8B(I(&EF("@D8W1X
M*3L*?0IE;'-E('L*:68@*"1C='@@86YD(&]P96XH;7D@)&9H+"`B/"1O=70B
M*2D@>PIB:6YM;V1E*"1F:"D["B1C='@M/F%D9&9I;&4H)&9H*3L*8VQO<V4H
M)&9H*3L*?0I]"B1C86-H95]N86UE(#T@)&-T>"`_("1C='@M/FAE>&1I9V5S
M="`Z("1M=&EM93L*?0HD8V%C:&5?;F%M92`N/2`B7#`B('@@*#0Q("T@;&5N
M9W1H("1C86-H95]N86UE*3L*)&-A8VAE7VYA;64@+CT@(D-!0TA%(CL*)&9H
M+3YP<FEN="@D8V%C:&5?;F%M92D["B1F:"T^<')I;G0H<&%C:R@G3B<L("1F
M:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*2DI.PHD9F@M/G!R:6YT*")<;E!!
M4BYP;5QN(BD["B1F:"T^8VQO<V4["F-H;6]D(#`W-34L("1O=70["F5X:70[
M"GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O
M9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L
M;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV
M93HZ6FEP.PI]"FUY("1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL*;7D@)&9H
M(#T@24\Z.D9I;&4M/FYE=SL*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R
M)RD@;W(@9&EE("(D(3H@)$`B.PHD>FEP+3YR96%D1G)O;49I;&5(86YD;&4H
M)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2!O<B!D
M:64@(B0A.B`D0"(["G!U<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`["B1005(Z
M.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D>FEP.PHD<75I970@/2`A)$5.5GM0
M05)?1$5"54=].PIO=71S*'%Q*%PD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!
M4E]414U0?2(I*3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI('L@(R!S
M:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@)&UE;6)E
M<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^:7-$:7)E
M8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.86UE.PIN
M97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB+PHH/SHD
M0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\**#\Z)$-O;F9I9SHZ0V]N9FEG
M>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@
M/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.
M5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M
M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO
M=71S*'%Q*%-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A
M9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(BDI.PI](&5L<V4@>PIO=71S*'%Q
M*$5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B*2D[
M"B1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N86UE*3L*8VAM
M;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*
M?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\
M("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@
M+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L
M(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!
M4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!4D=6*3L*
M?0IS=6(@0W)E871E4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&
M:6QE.CI"87-E;F%M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@
M1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR
M97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@
M=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*
M<W5B(')E<75I<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U:7)E($1Y
M;F%,;V%D97(["G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I8W0["G)E
M<75I<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR
M97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U:7)E($5X
M<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT
M;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P96,["G)E
M<75I<F4@6%-,;V%D97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(
M86YD;&4["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL
M:6(["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E(%!!4CL*<F5Q=6ER
M92!005(Z.DAE879Y.PIR97%U:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!005(Z
M.D9I;'1E<CHZ4&]D4W1R:7`["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A=&-H
M0V]N=&5N=#L*<F5Q=6ER92!A='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER92!#
M=V0@?3L*979A;"![(')E<75I<F4@5VEN,S(@?3L*979A;"![(')E<75I<F4@
M4V-A;&%R.CI5=&EL('T["F5V86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP
M.CI"=7)S="!].PIE=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P
M='5R92!].PIE=F%L('L@<F5Q=6ER92!097)L24\[(')E<75I<F4@4&5R;$E/
M.CIS8V%L87(@?3L*979A;"![(')E<75I<F4@=71F."!].PI]"G-U8B!?<V5T
M7W!A<E]T96UP('L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'T@86YD("1%
M3E9[4$%27U1%35!](#U^("\H+BLI+RD@>PHD<&%R7W1E;7`@/2`D,3L*<F5T
M=7)N.PI]"F9O<F5A8V@@;7D@)'!A=&@@*`HH;6%P("1%3E9[)%]]+"!Q=R@@
M4$%27U1-4$1)4B!435!$25(@5$5-4$1)4B!414U0(%1-4"`I*2P*<7<H($,Z
M7%Q414U0("]T;7`@+B`I"BD@>PIN97AT('5N;&5S<R!D969I;F5D("1P871H
M(&%N9"`M9"`D<&%T:"!A;F0@+7<@)'!A=&@["FUY("1U<V5R;F%M93L*;7D@
M)'!W=6ED.PIE=F%L('LH)'!W=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D969I
M;F5D("0^.WT["FEF("@@9&5F:6YE9"@F5VEN,S(Z.DQO9VEN3F%M92D@*2![
M"B1U<V5R;F%M92`]("97:6XS,CHZ3&]G:6Y.86UE.PI]"F5L<VEF("AD969I
M;F5D("1P=W5I9"D@>PHD=7-E<FYA;64@/2`D<'=U:60["GT*96QS92!["B1U
M<V5R;F%M92`]("1%3E9[55-%4DY!345]('Q\("1%3E9[55-%4GT@?'P@)U-9
M4U1%32<["GT*)'5S97)N86UE(#U^(',O7%<O7R]G.PIM>2`D<W1M<&1I<B`]
M("(D<&%T:"1#;VYF:6=[7V1E;&EM?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E
M<FYA;64I.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*:68@*"$D14Y6>U!!4E]#
M3$5!3GT@86YD(&UY("1M=&EM92`]("AS=&%T*"1P<F]G;F%M92DI6SE=*2![
M"F]P96X@*&UY("1F:"P@(CPB+B`D<')O9VYA;64I.PIS965K("1F:"P@+3$X
M+"`R.PIS>7-R96%D("1F:"P@;7D@)&)U9BP@-CL*:68@*"1B=68@97$@(EPP
M0T%#2$4B*2!["G-E96L@)&9H+"`M-3@L(#(["G-Y<W)E860@)&9H+"`D8G5F
M+"`T,3L*)&)U9B`]?B!S+UPP+R]G.PHD<W1M<&1I<B`N/2`B)$-O;F9I9WM?
M9&5L:6U]8V%C:&4M(B`N("1B=68["GT*96QS92!["FUY("1C='@@/2!E=F%L
M('L@<F5Q=6ER92!$:6=E<W0Z.E-(03L@1&EG97-T.CI32$$M/FYE=R@Q*2!]
M"GQ\(&5V86P@>R!R97%U:7)E($1I9V5S=#HZ4TA!,3L@1&EG97-T.CI32$$Q
M+3YN97<@?0I\?"!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z.DU$-3L@1&EG97-T
M.CI-1#4M/FYE=R!].PIM>2`D<VAA7W9E<G-I;VX@/2!E=F%L('L@)$1I9V5S
M=#HZ4TA!.CI615)324].('T@?'P@,#L*:68@*"1S:&%?=F5R<VEO;B!E<2`G
M-2XS."<@;W(@)'-H85]V97)S:6]N(&5Q("<U+C,Y)RD@>PHD8W1X+3YA9&1F
M:6QE*"1P<F]G;F%M92P@(F(B*2!I9B`H)&-T>"D["GT*96QS92!["FEF("@D
M8W1X(&%N9"!O<&5N*&UY("1F:"P@(CPD<')O9VYA;64B*2D@>PIB:6YM;V1E
M*"1F:"D["B1C='@M/F%D9&9I;&4H)&9H*3L*8VQO<V4H)&9H*3L*?0I]"B1S
M=&UP9&ER("X]("(D0V]N9FEG>U]D96QI;7UC86-H92TB("X@*"`D8W1X(#\@
M)&-T>"T^:&5X9&EG97-T(#H@)&UT:6UE("D["GT*8VQO<V4H)&9H*3L*?0IE
M;'-E('L*)$5.5GM005)?0TQ%04Y](#T@,3L*)'-T;7!D:7(@+CT@(B1#;VYF
M:6=[7V1E;&EM?71E;7`M)"0B.PI]"B1%3E9[4$%27U1%35!](#T@)'-T;7!D
M:7(["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIL87-T.PI]"B1P87)?=&5M<"`]
M("0Q(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O
M*"XK*2\["GT*<W5B(%]T96UP9FEL92!["FUY("@D;F%M92P@)&-O;G1E;G1S
M+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M
M92(["G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D
M9G5L;&YA;64N)"0B.PIO<&5N(&UY("1F:"P@)SXG+"`D=&5M<&YA;64@;W(@
M9&EE(")C86XG="!W<FET92`D=&5M<&YA;64Z("0A(CL*8FEN;6]D92`D9F@[
M"G!R:6YT("1F:"`D8V]N=&5N=',["F-L;W-E("1F:#L*8VAM;V0@)&UO9&4L
M("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE
M+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D
M9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O
M*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*
M:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[
M4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA
M;64L(')I;F1E>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@*R`Q*3L*
M?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L
M("1#;VYF:6=[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\
M)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D
M<')O9VYA;64D0V]N9FEG>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)$-O;F9I
M9WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O
M7%$D0V]N9FEG>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF
M(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]4
M14U0?3L*)&1I<B`]?B!S+UQ1)$-O;F9I9WM?9&5L:6U]7$4D+R\["B@H)'!R
M;V=N86UE(#T@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA;64D0V]N9FEG
M>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R
M;V=N86UE)$-O;F9I9WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D0V]N
M9FEG>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D0V]N
M9FEG>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@
M>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H
M:6YD97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI(#P@,"D@>PHD<')O
M9VYA;64@/2`B+B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W
M9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@
M*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z
M(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN
M/UQ1)$-O;F9I9WM?9&5L:6U]7$4I+R1P=V0D0V]N9FEG>U]D96QI;7TO.PHD
M14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI
M=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2![
M"G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*
M?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/
M1TY!344@05)'0R!!4D=67S`@*2`I('L*9&5L971E("1%3E9[)U!!4E\G+B1?
M?3L*?0IF;W(@*'%W+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B1%
M3E9[)U!!4E\G+B1??2`]("1%3E9[)U!!4E]'3$]"04Q?)RXD7WT@:68@97AI
M<W1S("1%3E9[)U!!4E]'3$]"04Q?)RXD7WT["GT*;7D@)'!A<E]C;&5A;B`]
M(")?7T5.5E]005)?0TQ%04Y?7R`@("`@("`@("`@("`@("(["FEF("@D14Y6
M>U!!4E]414U0?2D@>PID96QE=&4@)$5.5GM005)?0TQ%04Y].PI]"F5L<VEF
M("@A97AI<W1S("1%3E9[4$%27T=,3T)!3%]#3$5!3GTI('L*;7D@)'9A;'5E
M(#T@<W5B<W1R*"1P87)?8VQE86XL(#$R("L@;&5N9W1H*")#3$5!3B(I*3L*
M)$5.5GM005)?0TQ%04Y](#T@)#$@:68@)'9A;'5E(#U^("]>4$%27T-,14%.
M/2A<4RLI+SL*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@
M*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN
M="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*
M<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[
M)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I
M=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K
M86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF
M:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q*'!A<BYP
M;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C
M:"!F:6QE(&]R(&1I<F5C=&]R>5QN*0IU;FQE<W,@+64@)'!R;V=N86UE.PID
M;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58
M251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]
M("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^
M+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ
M7U]%4E)/4CL*,3L*7U]%3D1?7PH`4%L``$=#0SH@*$=.52D@-BXT+C``````
M1T-#.B`H1TY5*2`V+C0N,`````!'0T,Z("A'3E4I(#8N-"XP`````$=#0SH@
M*$=.52D@-BXT+C``````1T-#.B`H1TY5*2`V+C0N,`````!'0T,Z("A'3E4I
M(#8N-"XP`````$=#0SH@*$=.52D@-BXT+C``````1T-#.B`H1TY5*2`V+C0N
M,`````!'0T,Z("A'3E4I(#8N-"XP`````$=#0SH@*$=.52D@-BXT+C``````
M1T-#.B`H1TY5*2`V+C0N,`````!'0T,Z("A'3E4I(#8N-"XP`````$=#0SH@
M*$=.52D@-BXT+C``````1T-#.B`H1TY5*2`V+C0N,`````!'0T,Z("A'3E4I
M(#8N-"XP`````````````````0```*`2`0#?$0``(````*`2`0!Z-@``(```
M`*`2`0!'-P``(````*`2`0"6-P``(````*`2`0"\-P``(````*`2`0"G.```
M(````*`2`0"].0``(````*`2`0#=.0``(````*`2`0"P.P``(````*`2`0`4
M/```(````*`2`0`S/0``(````*`2`0!W/0``(````*`2`0#V/P``(````*`2
M`0`Q00``(````*`2`0"400``(````*`2`0"N00``(````*`2`0`_0@``(```
M`*`2`0!T0@``(````*`2`0!`0P``(````*`2`0!^10``(````*`2`0`84```
M(````*`2`0#24@``(````*`2`0#O6P``(````*`2`0!;70``(````*P2`0#W
M.@``(````*P2`0`A.P``(````*P2`0!`.P``(````*P2`0#K.P``(````*P2
M`0`Q/```(````*P2`0!D/```(````*P2`0"D/```(````*P2`0"Y/```(```
M`*P2`0`F/@``(````*P2`0!,/@``(````*P2`0`>10``(````*P2`0#D10``
M(````*P2`0`'1@``(````*P2`0`G1@``(````*P2`0`C4```(````*P2`0`R
M4```(````*P2`0!F4```(````*P2`0#K4```(````*P2`0`E40``(````*P2
M`0!C40``(````*P2`0";40``(````*P2`0#+40``(````*P2`0`;4@``(```
M`*P2`0`X4@``(````*P2`0!?4@``(````*P2`0"`4@``(````*P2`0#S4@``
M(````*P2`0!M4P``(````*P2`0"(4P``(````*P2`0!47```(````*P2`0!H
M7```(````*P2`0!\7```(````*@2`0`R7```(```````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````"
M````&0```!S````<F```4E-$4WG,;<;;'47>:2:$!*I:J6X!````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````!0``````````7I2``%\"`$;#`0$B`$``!P````<````X#___X<`````
M00X(A0)"#04"@\4,!`0`%``````````!>E(``7P(`1L,!`2(`0``+````!P`
M```X0/__!`$```!!#@B%`D(-!4:'`X8$@P4"V@K#0<9!QT'%#`0$2@L`'```
M`$P````80?__+P````!!#@B%`D(-!6O%#`0$```4``````````%Z4@`!?`@!
M&PP$!(@!``!`````'````!!!___`)````$$."(4"00X,AP-!#A"&!$$.%(,%
M20Z@!@.I)`H.%$'##A!!Q@X,0<<."$'%#@1!"P```%0```!@````C&7__Z4`
M````00X(A0)!#@R'`T$.$(8$00X4@P5##D`"<@H.%$/##A!!Q@X,0<<."$'%
M#@1!"UH*#A1#PPX00<8.#$''#@A!Q0X$00L````@````N````.1E__]F````
M`$$."(,"10X@`ED*#@A!PPX$00L\````W````#!F__\``@```$$."(4"00X,
MAP-##A"&!$$.%(,%10Y0`OD*#A1!PPX00<8.#$''#@A!Q0X$2`L`/````!P!
M``#P9___/@$```!!#@B%`D$.#(<#0PX0A@1!#A2#!48.L`%Q"@X40<,.$$'&
M#@Q!QPX(0<4.!$$+`"@```!<`0``\&C__XL`````00X(@P)%#B!E"@X(0\,.
M!$(+`E8."$3##@0`0````(@!``!4:?__U`````!!#@B%`D$.#(<#00X0A@1!
M#A2#!4$.&$P.P"`"CPH.%$'##A!!Q@X,0<<."$'%#@1!"P`4````S`$``/!I
M__\T`````$,.('`.!``4````Y`$``!AJ__]9`````$,.,`)5#@00````_`$`
M`&!J__\#`````````#P````0`@``7&K__]0!````00X(A0)!#@R'`T8.$(8$
M00X4@P5'#N`@`F<*#A1!PPX00<8.#$''#@A!Q0X$00L0````4`(``/QK__\B
M`````````$0```!D`@``&&S__\(`````00X(A@)!#@R#`T,.(`)<"@X,0<,.
M"$'&#@1-"WP*#@Q!PPX(0<8.!$(+2PX,0<,."$'&#@0``"P```"L`@``H&S_
M_Y<!````00X(AP)!#@R&`T,.0`-%`0H.#$'&#@A!QPX$1`L``!@```#<`@``
M$&[__VD`````0PX@9`H.!$D+```8````^`(``&1N__\D`````$,.(%8*#@1'
M"P``-````!0#``!X;O__LP````!!#@B'`D$.#(8#00X0@P1##D`"1PH.$$'#
M#@Q!Q@X(0<<.!$$+``!`````3`,```!O__\3#0```$$."(4"00X,AP-&#A"&
M!$$.%(,%1P[@@@(#D`(*#A1!PPX00<8.#$''#@A!Q0X$00L``#P```"0`P``
MW'O__[X#````00X(A0)!#@R'`T$.$(8$00X4@P5##D`#R`(*#A1!PPX00<8.
M#$''#@A!Q0X$00LT````T`,``%Q___]L`````$$."(8"00X,@P-##B`"5`H.
M#$'##@A!Q@X$00M'#@Q#PPX(0<8.!"@````(!```)(?__^@!````00X(A0)"
M#05(A@.#!`.,`0K#0<9!Q0P$!$$+%``````````!>E(``7P(`1L,!`2(`0``
M-````!P```"`@?__+`,```!!#@B&`D$.#(,#0PX@`XH!#AQ##B`"3`H.#$;#
M#@A!Q@X$2PL````4``````````%Z4@`!?`@!&PP$!(@!```8````'````&"$
M__\@`````$,.(%0.&$,.(```%``````````!>E(``7P(`1L,!`2(`0``'```
M`!P```!,A/__*0````!##B!?#A1##B!##@0````L````/````%R$___V````
M`$,.,'0*#@1)"U,*#@1-"T@*#@1("U0.)$,.,`)I#@00````;````"R%__\%
M`````````!0``````````7I2``%\"`$;#`0$B`$``!`````<````$(7__P@`
M````````%``````````!>E(``7P(`1L,!`2(`0``$````!P```#TA/__`@``
M```````4``````````%Z4@`!?`@!&PP$!(@!```0````'````-B$__\"````
M`````!0``````````7I2``%\"`$;#`0$B`$``!`````<````O(3__P(`````
M````%``````````!>E(``7P(`1L,!`2(`0``$````!P```"@A/__`@``````
M```4``````````%Z4@`!?`@!&PP$!(@!```0````'````(2$__\"````````
M`!0``````````7I2``%\"`$;#`0$B`$``"0````<````B(3__S0`````00X(
M@P)##B!8#AA##B!/#@A!PPX$````````%``````````!>E(``7P(`1L,!`2(
M`0``'````!P```"$AO__"0````!!#@B%`D(-!4'%#`0$````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````.F1[ED`````T@`!``$`
M```1````$0```"@``0!L``$`L``!`!!0``#D%0$`]!4!`-`5`0``0@``T%L`
M``Q@``#P/@``T%,````]```@/P``D$$``!`]``#P/P``\$(``#!"``!@/```
MV@`!`/```0`"`0$`$@$!`"D!`0!$`0$`20$!`%,!`0!@`0$`;`$!`'T!`0")
M`0$`EP$!`*0!`0"Q`0$`O@$!`-`!`0````$``@`#``0`!0`&``<`"``)``H`
M"P`,``T`#@`/`!``<&%R+F5X90!84U]);G1E<FYA;'-?4$%27T)/3U0`7VYM
M7U]03%]D;U]U;F1U;7``7VYM7U]03%]T:')?:V5Y`%]N;5]?7U]S=&%C:U]C
M:&M?9W5A<F0`9V5T7W5S97)N86UE7V9R;VU?9V5T<'=U:60`;6%I;@!M>5]P
M87)?<&P`<&%R7V)A<V5N86UE`'!A<E]C;&5A;G5P`'!A<E]C=7)R96YT7V5X
M96,`<&%R7V1I<FYA;64`<&%R7V5N=E]C;&5A;@!P87)?9FEN9'!R;V<`<&%R
M7VEN:71?96YV`'!A<E]M:W1M<&1I<@!P87)?<V5T=7!?;&EB<&%T:`!X<U]I
M;FET``````````````````````````````````````````````````````!D
M$`$```````````"\&`$`Q!$!`#P1`0```````````-`8`0"<$@$`2!$!````
M````````1!D!`*@2`0"P$0$```````````!D&0$`$!,!````````````````
M````````````)!,!`#03`0!`$P$`4!,!`%P3`0!L$P$`?!,!`(@3`0"8$P$`
MI!,!`+`3`0"\$P$`R!,!`-03`0#@$P$`Z!,!`/03`0`(%`$`'!0!`"P4`0`T
M%`$`0!0!`$@4`0!4%`$`8!0!`&P4`0!T%`$`?!0!`(@4`0"4%`$`H!0!`*P4
M`0"T%`$`O!0!`,@4`0#<%`$`\!0!``@5`0`0%0$`'!4!`"@5`0`T%0$`/!4!
M`$@5`0!0%0$`7!4!`&@5`0!T%0$`@!4!`(P5`0"8%0$`I!4!`+`5`0``````
MO!4!`-`5`0``````Y!4!`/05`0`$%@$`&!8!`#`6`0!`%@$`4!8!`&@6`0!X
M%@$`C!8!`)P6`0"L%@$`P!8!`-@6`0#L%@$`_!8!`!`7`0`D%P$`-!<!`$@7
M`0!8%P$`;!<!`'P7`0"(%P$`F!<!``````"D%P$`LA<!`,87`0#8%P$`````
M`"03`0`T$P$`0!,!`%`3`0!<$P$`;!,!`'P3`0"($P$`F!,!`*03`0"P$P$`
MO!,!`,@3`0#4$P$`X!,!`.@3`0#T$P$`"!0!`!P4`0`L%`$`-!0!`$`4`0!(
M%`$`5!0!`&`4`0!L%`$`=!0!`'P4`0"(%`$`E!0!`*`4`0"L%`$`M!0!`+P4
M`0#(%`$`W!0!`/`4`0`(%0$`$!4!`!P5`0`H%0$`-!4!`#P5`0!(%0$`4!4!
M`%P5`0!H%0$`=!4!`(`5`0",%0$`F!4!`*05`0"P%0$``````+P5`0#0%0$`
M`````.05`0#T%0$`!!8!`!@6`0`P%@$`0!8!`%`6`0!H%@$`>!8!`(P6`0"<
M%@$`K!8!`,`6`0#8%@$`[!8!`/P6`0`0%P$`)!<!`#07`0!(%P$`6!<!`&P7
M`0!\%P$`B!<!`)@7`0``````I!<!`+(7`0#&%P$`V!<!```````:`%]?8WAA
M7V%T97AI=```*`!?7V5R<FYO````1P!?7V=E=')E96YT`````%$`7U]M86EN
M`````,T`7V1L;%]C<G0P0#````!R`5]G971P=W5I9#,R````>0%?9V5T=6ED
M,S(`A@%?:6UP=7)E7W!T<@```,$!7VQS965K-C0``,,!7VQS=&%T-C0``.0!
M7V]P96XV-````$L"7W-T870V-````+,"86-C97-S`````"L#8V%L;&]C````
M`&@#8VQO<V4`:0-C;&]S961I<@``K0-C>6=W:6Y?9&5T86-H7V1L;`"O`V-Y
M9W=I;E]I;G1E<FYA;````,4#9&QL7V1L;&-R=#````#_`V5X:70``&H$9G!R
M:6YT9@```'8$9G)E90``W@1G971P:60`````Z@1G971P=W5I9```^`1G971U
M:60`````O`5L<V5E:P"^!6QS=&%T`,,%;6%L;&]C`````-,%;65M8VAR````
M`-4%;65M8W!Y`````-<%;65M;6]V90```-L%;6MD:7(`%P9O<&5N```;!F]P
M96YD:7(````N!G!O<VEX7VUE;6%L:6=N`````%<&<'1H<F5A9%]A=&9O<FL`
M````BP9P=&AR96%D7V=E='-P96-I9FEC````W09R96%D``#>!G)E861D:7(`
M``#@!G)E861L:6YK``#C!G)E86QL;V,````1!W)M9&ER`*H'<W!R:6YT9@``
M`+,'<W1A=```O0=S=')C:'(`````OP=S=')C;7``````Q`=S=')D=7``````
MS@=S=')L96X`````TP=S=')N8VUP````U`=S=')N8W!Y````WP=S=')S='(`
M````Y@=S=')T;VL`````1PAU;FQI;FL`````"P!?7W-T86-K7V-H:U]F86EL
M```-`%]?<W1A8VM?8VAK7V=U87)D`!$`4$Q?9&]?=6YD=6UP``!9`%!,7W1H
M<E]K97D`````5P%097)L7V%T9F]R:U]L;V-K``!8`5!E<FQ?871F;W)K7W5N
M;&]C:P````!A`5!E<FQ?879?9F5T8V@`:0%097)L7V%V7W-H:69T`,D!4&5R
M;%]C<F]A:U]N;V-O;G1E>'0``%("4&5R;%]G971?<W8```"``E!E<FQ?9W9?
M9F5T8VAP=@```&L#4&5R;%]M9U]S970```#;`U!E<FQ?;F5W6%,`````W0-0
M97)L7VYE=UA37V9L86=S``"@!5!E<FQ?<V%F97-Y<VUA;&QO8P````#_!5!E
M<FQ?<W9?,G!V7V9L86=S`'$&4&5R;%]S=E]S971P=@!Z!E!E<FQ?<W9?<V5T
M<'9N`````*4&4&5R;%]S>7-?:6YI=#,`````I@9097)L7W-Y<U]T97)M`"('
M8F]O=%]$>6YA3&]A9&5R````+@=P97)L7V%L;&]C`````#`'<&5R;%]C;VYS
M=')U8W0`````,0=P97)L7V1E<W1R=6-T`#('<&5R;%]F<F5E`#,'<&5R;%]P
M87)S90`````T!W!E<FQ?<G5N``!E`49R965,:6)R87)Y`!8"1V5T36]D=6QE
M2&%N9&QE00``1@)'9710<F]C061D<F5S<P``*@-,;V%D3&EB<F%R>4$````0
M`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!
M```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`
M`!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``
M$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0
M`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`8WEG=VEN,2YD;&P`%!`!
M`!00`0!C>6=S<W`M,"YD;&P`````*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`
M*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H
M$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`&-Y9W!E<FPU7S(V+F1L
M;``\$`$`/!`!`#P0`0`\$`$`2T523D5,,S(N9&QL````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````$``,````P``"`#@```,```(`0````\```@!@````@`0"`````
M```````````````$``$```!@``"``@```'@``(`#````D```@`0```"H``"`
M```````````````````!``D$``!0`0`````````````````````!``D$``!@
M`0`````````````````````!``D$``!P`0`````````````````````!``D$
M``"``0```````````````````0```,`!`(#8``"````````````````````!
M``D$``"0`0`````````````````````!``$````(`0"`````````````````
M```!``D$``"@`0`````````````````````!``$````X`0"`````````````
M```````!``D$``"P`0``T"$!`&@%`````````````#@G`0"H"```````````
M``#@+P$`:`4`````````````2#4!`*@(`````````````/`]`0`^````````
M```````P/@$`6`(`````````````B$`!`#X#``````````````8`5P!)`$X`
M10!8`$4````H````$````"`````!``@````````!```````````````!````
M`0``!@8&``0$!``"`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S
M<P"?GY\`6%A8`+*RL@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.
M`$9&1@#+R\L`"PL+`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`
M[N[N`/3T]`#CX^,`Y>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")
MB8D`BHJ*`#0T-`#HZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/
MCP"^OKX`;6UM``D)"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0
M`#P\/`!D9&0`PL+"`#(R,@`("`@`#0T-``\/#P``````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````%!04%!04%!
M04%!04%!00$!`35`(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!
M%0$!)A4!`0\!`0$!`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!
M`0$!$`H@(A45+B$N`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!
M`0%!$!`0$!`0$!`0$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0
M!P$!`0$@$!`0`0\0$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!
M!0\!`0$!`0$!`0$!`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#
M``#@`P``P`$``(`!``"``0``@`,``(`#````!P````<``(X/``#_'P``*```
M`"````!``````0`(````````!````````````````0````$````````!`0$`
M`@("``,#`P`$!`0`!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-
M#0T`#@X.``\/#P`0$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8
M&``9&1D`&AH:`!L;&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C
M`"0D)``E)24`)B8F`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`
M+R\O`#`P,``Q,3$`,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z
M.CH`.SL[`#P\/``]/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%
M10!&1D8`1T='`$A(2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0
M`%%140!24E(`4U-3`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`
M7%Q<`%U=70!>7EX`7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G
M9V<`:&AH`&EI:0!J:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R
M<@!S<W,`='1T`'5U=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]
M`'Y^?@!_?W\`@("``(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`
MB8F)`(J*B@"+BXL`C(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4
ME)0`E965`):6E@"7EY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?
MGP"@H*``H:&A`**BH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ
M`*NKJP"LK*P`K:VM`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`
MMK:V`+>WMP"XN+@`N;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!
MP<$`PL+"`,/#PP#$Q,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,
MS`#-S<T`SL[.`,_/SP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7
M`-C8V`#9V=D`VMK:`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`
MX^/C`.3DY`#EY>4`YN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N
M[NX`[^_O`/#P\`#Q\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY
M^0#Z^OH`^_O[`/S\_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\
M/#P\/#PL#`0$!`0$!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$
M9,SL]/3T_/3\]/3T_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T
M]/3T]/3T]/3T[.R\-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT
M45%1400$9+SD[.RLC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$
M492L9.2D3#0L7,3,4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QT
MK-3DW+PT!`0$!`0$9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1D
MO-SDY.3DY-R,?)RT?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,
MS#R49,R\C%'<U+0T!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$
M!`0$!`1<M-3<W.3<W'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<
M=%'T=(3LS&3D?&RDM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RL
MA%&\K#0$!`0$!`1<M,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RL
MS,QLG/3\_/S\_/S\_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\
M_/S\_/SLS*Q1K#0$!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$
M!`0$!%RLM$S4_/STM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RT
MG/3\_/S\_/S\_/244<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RL
MS+RD-`0$!`0$!%RD9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1L
MC.3T_/R<4<Q1E.S\_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T
M_.11O,S,Q+R<-`0$!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$
M!`0$7)R\Q,3$Q,3$Q,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$
MQ,3$Q,3$Q,3$Q,3$O+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\
MM)PL!`0$!`0$7)RTQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<
MG)24E)24E)24E)24E)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<
M7%Q<7%Q<7%0T!`0$!`0$````/P```!\````/````!P````,````!````````
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\H````$````"`````!``@````````!```````````````!````
M`0``!@8&``0$!``"`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S
M<P"?GY\`6%A8`+*RL@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.
M`$9&1@#+R\L`"PL+`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`
M[N[N`/3T]`#CX^,`Y>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")
MB8D`BHJ*`#0T-`#HZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/
MCP"^OKX`;6UM``D)"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0
M`#P\/`!D9&0`PL+"`#(R,@`("`@`#0T-``\/#P``````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````%!04%!04%!
M04%!04%!00$!`35`(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!
M%0$!)A4!`0\!`0$!`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!
M`0$!$`H@(A45+B$N`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!
M`0%!$!`0$!`0$!`0$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0
M!P$!`0$@$!`0`0\0$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!
M!0\!`0$!`0$!`0$!`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#
M``#@`P``P`$``(`!``"``0``@`,``(`#````!P````<``(X/``#_'P``*```
M`"````!``````0`(````````!````````````````0````$````````!`0$`
M`@("``,#`P`$!`0`!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-
M#0T`#@X.``\/#P`0$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8
M&``9&1D`&AH:`!L;&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C
M`"0D)``E)24`)B8F`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`
M+R\O`#`P,``Q,3$`,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z
M.CH`.SL[`#P\/``]/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%
M10!&1D8`1T='`$A(2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0
M`%%140!24E(`4U-3`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`
M7%Q<`%U=70!>7EX`7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G
M9V<`:&AH`&EI:0!J:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R
M<@!S<W,`='1T`'5U=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]
M`'Y^?@!_?W\`@("``(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`
MB8F)`(J*B@"+BXL`C(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4
ME)0`E965`):6E@"7EY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?
MGP"@H*``H:&A`**BH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ
M`*NKJP"LK*P`K:VM`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`
MMK:V`+>WMP"XN+@`N;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!
MP<$`PL+"`,/#PP#$Q,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,
MS`#-S<T`SL[.`,_/SP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7
M`-C8V`#9V=D`VMK:`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`
MX^/C`.3DY`#EY>4`YN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N
M[NX`[^_O`/#P\`#Q\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY
M^0#Z^OH`^_O[`/S\_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\
M/#P\/#PL#`0$!`0$!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$
M9,SL]/3T_/3\]/3T_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T
M]/3T]/3T]/3T[.R\-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT
M45%1400$9+SD[.RLC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$
M492L9.2D3#0L7,3,4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QT
MK-3DW+PT!`0$!`0$9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1D
MO-SDY.3DY-R,?)RT?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,
MS#R49,R\C%'<U+0T!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$
M!`0$!`1<M-3<W.3<W'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<
M=%'T=(3LS&3D?&RDM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RL
MA%&\K#0$!`0$!`1<M,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RL
MS,QLG/3\_/S\_/S\_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\
M_/S\_/SLS*Q1K#0$!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$
M!`0$!%RLM$S4_/STM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RT
MG/3\_/S\_/S\_/244<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RL
MS+RD-`0$!`0$!%RD9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1L
MC.3T_/R<4<Q1E.S\_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T
M_.11O,S,Q+R<-`0$!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$
M!`0$7)R\Q,3$Q,3$Q,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$
MQ,3$Q,3$Q,3$Q,3$O+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\
MM)PL!`0$!`0$7)RTQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<
MG)24E)24E)24E)24E)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<
M7%Q<7%Q<7%0T!`0$!`0$````/P```!\````/````!P````,````!````````
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\````_
M````/P```#\```$`!``0$````0`(`&@%```!`"`@```!``@`J`@```(`$!``
M``$`"`!H!0```P`@(````0`(`*@(```$````6`(T````5@!3`%\`5@!%`%(`
M4P!)`$\`3@!?`$D`3@!&`$\``````+T$[_X```$`````````````````````
M`#\`````````!``$``$```````````````````"X`0```0!3`'0`<@!I`&X`
M9P!&`&D`;`!E`$D`;@!F`&\```"4`0```0`P`#``,``P`#``-`!"`#`````D
M``(``0!#`&\`;0!P`&$`;@!Y`$X`80!M`&4``````"`````L``(``0!&`&D`
M;`!E`$0`90!S`&,`<@!I`'``=`!I`&\`;@``````(````#``"``!`$8`:0!L
M`&4`5@!E`'(`<P!I`&\`;@``````,``N`#``+@`P`"X`,````"0``@`!`$D`
M;@!T`&4`<@!N`&$`;`!.`&$`;0!E````(````"@``@`!`$P`90!G`&$`;`!#
M`&\`<`!Y`'(`:0!G`&@`=````"`````L``(``0!,`&4`9P!A`&P`5`!R`&$`
M9`!E`&T`80!R`&L`<P``````(````"P``@`!`$\`<@!I`&<`:0!N`&$`;`!&
M`&D`;`!E`&X`80!M`&4````@````)``"``$`4`!R`&\`9`!U`&,`=`!.`&$`
M;0!E```````@````-``(``$`4`!R`&\`9`!U`&,`=`!6`&4`<@!S`&D`;P!N
M````,``N`#``+@`P`"X`,````$0````!`%8`80!R`$8`:0!L`&4`20!N`&8`
M;P``````)``$````5`!R`&$`;@!S`&P`80!T`&D`;P!N`````````+`$/#]X
M;6P@=F5R<VEO;CTB,2XP(B!E;F-O9&EN9STB551&+3@B('-T86YD86QO;F4]
M(GEE<R(_/@T*/&%S<V5M8FQY('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R;W-O
M9G0M8V]M.F%S;2YV,2(@;6%N:69E<W1697)S:6]N/2(Q+C`B/@T*("`@(#QA
M<W-E;6)L>4ED96YT:71Y('9E<G-I;VX](C`N,"XP+C`B('1Y<&4](G=I;C,R
M(B!N86UE/2)005(M4&%C:V5R+D%P<&QI8V%T:6]N(B`O/@T*("`@(#QD97-C
M<FEP=&EO;CY005(M4&%C:V5R($%P<&QI8V%T:6]N/"]D97-C<FEP=&EO;CX-
M"B`@("`\9&5P96YD96YC>3X-"B`@("`@("`@/&1E<&5N9&5N=$%S<V5M8FQY
M/@T*("`@("`@("`@("`@/&%S<V5M8FQY261E;G1I='D@='EP93TB=VEN,S(B
M(&YA;64](DUI8W)O<V]F="Y7:6YD;W=S+D-O;6UO;BU#;VYT<F]L<R(@=F5R
M<VEO;CTB-BXP+C`N,"(@<'5B;&EC2V5Y5&]K96X](C8U.35B-C0Q-#1C8V8Q
M9&8B(&QA;F=U86=E/2(J(B!P<F]C97-S;W)!<F-H:71E8W1U<F4](BHB("\^
M#0H@("`@("`@(#PO9&5P96YD96YT07-S96UB;'D^#0H@("`@/"]D97!E;F1E
M;F-Y/@T*("`@(#QT<G5S=$EN9F\@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O
M<V]F="UC;VTZ87-M+G8S(CX-"B`@("`@("`@/'-E8W5R:71Y/@T*("`@("`@
M("`@("`@/')E<75E<W1E9%!R:79I;&5G97,^#0H@("`@("`@("`@("`@("`@
M/')E<75E<W1E9$5X96-U=&EO;DQE=F5L(&QE=F5L/2)A<TEN=F]K97(B('5I
M06-C97-S/2)F86QS92(@+SX-"B`@("`@("`@("`@(#PO<F5Q=65S=&5D4')I
M=FEL96=E<SX-"B`@("`@("`@/"]S96-U<FET>3X-"B`@("`\+W1R=7-T26YF
M;SX-"CPO87-S96UB;'D^#0H`````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````!```#P````,
M,)LPHC"X,+XPQ##,,-0P[##Y,`4Q##$3,1XQ-#$],4TQ5S%F,8(QIS&R,;DQ
MQC'?,0```#```'@```!Z-J$V1S=E-Y8WO#<T.%$XISC5.`4Y'#E+.8@YO3G=
M.1HZ*CIE.O<Z(3M`.Z4[L#OK.Q0\,3QD/'0\?#R$/*0\N3S1/-D\X3SI/#,]
M03UC/7<]G3T-/AD^)CY,/FX^CSZ:/L4^>C^W/\4_U#_V/P```$```+@````!
M,`HP$S`<,"4P+C`W,$`P2S!5,%\P:3!S,'TPAS"1,)LPI3"O,+DPQS#7,.4P
M]3`#,1,Q'3$B,3$Q13%5,64Q=S&4,9\QKC'%,>8Q/S)*,G0RA#*X,L0RTS(/
M,Q<S)S,O,SLS0#-.,U8S7C-N,W8SS3/G,^(T[#3X-!XU7C5^-<XUY#4'-B<V
M6#:I-OHV_S85-R$W*3=6-V8W<C>B-^4W*#A0/G<^ASZ//LT^XC\```!0``"L
M`0``&#`C,#(P2C!F,(8PG##(,-PPZS`E,5,Q8S&#,9LQJS'+,>LQ`C(;,C@R
M4#)?,F\R@#*5,J$RTC+S,A`S,3-`,TTS83-M,X@SKS.X,\$S%C1"-$HT4C1:
M-&(T:C1R-'HT@C2*-)(TFC2B-*HTLC2Z-,(TRC32--HTXC3J-/(T^C0"-0HU
M$C5&-4XU5C5>-68U;C5V-7XUAC6.-98UGC6F-:XUMC6^-<8USC76-=XUYC7N
M-?8U_C4&-@XV%C8>-B8V+C8V-CXV1C9.-E8V7C9F-I,VQ3;,-M,VX#;G-NXV
M]3;\-@DW$#<7-QXW*#<^-TLW6#=E-W(W?S>--YLWJ#>U-[PWQ#?,-]0WW#?D
M-^PW"#@/.!8X'3@G.#$X.SAH.)PXHSBJ.+0XNSC".,DXT#C=..0XZSCR./PX
M)#DT.40Y5#ED.70YA#F3.;PYQ#G-.=8Y?3J/.J@ZM#JZ.L0ZR3K6.F([:CMR
M.WH[D#NG.\([RCOO.Q@\(#PG/#(\03Q4/&@\?#RI/*\\OSS&/,X\_CP2/24]
M-CU$/5L]:SU\/9,]GSW4/0```&````P````,,```````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
9````````````````````````````````````
