/*

Copyright (C) 2019 Olaf Till <i7tiol@t-online.de>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef __OCT_PARALLEL_CONTROL__

#define __OCT_PARALLEL_CONTROL__

#include <octave/oct.h>

#include "config.h"

#include <octave/defaults.h>
#include <octave/oct-syscalls.h>

#include <unistd.h>

namespace octave_parallel
{

  class
  instance
  {
  public:


    instance (int fd, std::string& octb);

    ~instance (void);

    bool good (void)
    {
      return store_errno == 0;
    }

    int send_octave_value (octave_value& ov)
    {
      if (minimal_write_data (*os, ov))
        {
          store_errno = -1;

          return -1;
        }

      return 0;
    }

    int recv_octave_value (octave_value& ov)
    {
      if (minimal_read_data (*is, ov))
        {
          store_errno = -1;

          return -1;
        }

      return 0;
    }

    int initialize_new_function_handle (octave_value& fcn,
                                        std::string& dir,
                                        octave_value& eh,
                                        octave_value& nargout);

  private:

    int store_errno;

    pipes pps;

#if defined _WIN32 && ! defined __CYGWIN__
    void *process_handle;
#else
    pid_t pid;
#endif

    octave_parallel_pipe_streambuf *inbuf, *outbuf;

    std::istream *is;

    std::ostream *os;

    // The current directory when the last function handle was
    // sent. This directory will be set as current again for the
    // __exit__ function to be executed at the slave. The rationale is
    // that for sending the function handle the functions of the
    // 'parallel' package must have been in the search path, so by
    // returning to this directory __exit__ also must be in the search
    // path.
    std::string cdir;
  };

  class
  control
  {
  public:

    control (void)
      : nproc (0), octave_fd (-1), unrecoverable_error (0)
    {

      // This is only test code.
      std::string empty_string;
      if (set_executable (empty_string))
        _p_error ("couldn't set executable");
      else
        instc = new instance (octave_fd, octave_binary);
    }


    ~control (void)
    {

      // This is only test code
      delete instc;

    }

    int set_executable (const std::string& path);

    // This is only for testing.
    instance *instc;

    // TODO: close_all ()

  private:

    octave_idx_type nproc;

    std::string octave_binary;

    int octave_fd;

    int unrecoverable_error;
  };

} // namespace octave_parallel

// For fexecve, we need to copy some const char* to non-const. Since
// fork() and exec() are not seperate commands at all operating
// systems, this has to be done before fork, in the parent process, so
// measures must be taken for deallocation.
class
copy_to_non_const
{
public:
  copy_to_non_const (const char *cstr)
    : str (NULL)
  {
    str = new char [strlen (cstr) + 1];

    strcpy (str, cstr);
  }

  ~copy_to_non_const (void)
  {
    delete [] str;
  }

  char *get_str (void)
  {
    return str;
  }

private:

  char *str;
};


#endif // __OCT_PARALLEL_CONTROL__
