/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifndef FIELD_H
#define FIELD_H

#include <cstdio>
#include "array.h"
#include "cdo_options.h"
#include "cdo_vlist.h"
#include "compare.h"

double varToStd(double rvar, double missval);

enum field_flag
{
  FIELD_VEC = 2,  // allocated memory
  FIELD_FLT = 4,  // 32-bit float
  FIELD_DBL = 8,  // 64-bit float
  FIELD_NAT = 16, // native: 32-bit float for 32-bit float data, otherweise 64-bit float
};

class // Field
#ifdef WARN_UNUSED
[[gnu::warn_unused]]
#endif
Field
{
public:
  int fpeRaised = 0;
  int nwpv = 1;         // number of words per value; real:1  complex:2
  int grid = -1;
  MemType memType = MemType::Native;  // MemType::Float or MemType::Double

  size_t gridsize = 0;
  size_t size = 0;
  size_t nsamp = 0;

  size_t nmiss = 0;
  double missval = 0;

  Varray<float> vec_f;
  Varray<double> vec_d;
  Varray<double> weightv;

  Field() {}
  void init(const CdoVar &var);
  void resize(size_t count);
  void resize(size_t count, double value);
  void resizef(size_t count);
  void resizef(size_t count, float value);
  bool empty() const;
  void check_gridsize() const;

  bool hasData() const
  {
    return (memType == MemType::Float) ? !vec_f.empty() : !vec_d.empty();
  }

private:
  size_t m_count = 0;
};

class // Field3D
#ifdef WARN_UNUSED
[[gnu::warn_unused]]
#endif
Field3D : public Field
{
public:
  size_t nlevels = 0;

  Field3D() {}
  void init(const CdoVar &var);
};

struct RecordInfo
{
  int varID = 0;
  int levelID = 0;
  bool lconst = false;
};

using FieldVector = std::vector<Field>;
using FieldVector2D = std::vector<FieldVector>;
using FieldVector3D = std::vector<FieldVector2D>;

using Field3DVector = std::vector<Field3D>;

void fieldFill(Field &field, double value);
void fieldCopy(const Field &field_src, Field &field_tgt);
void fieldCopy(const Field3D &field_src, int levelID, Field &field_tgt);
void fieldAdd(Field &field1, const Field3D &field2, int levelID);
size_t fieldNumMiss(const Field &field);
size_t fieldNumMV(Field &field);
MinMax fieldMinMax(const Field &field);

// fieldmem.cc
void fieldsFromVlist(int vlistID, FieldVector2D &field2D);
void fieldsFromVlist(int vlistID, FieldVector2D &field2D, int ptype);
void fieldsFromVlist(int vlistID, FieldVector2D &field2D, int ptype, double fillValue);

// field.cc
double fieldFunction(const Field &field, int function);
double fieldMin(const Field &field);
double fieldMax(const Field &field);
double fieldSum(const Field &field);
double fieldMean(const Field &field);
double fieldMeanw(const Field &field);
double fieldAvg(const Field &field);
double fieldAvgw(const Field &field);
double fieldStd(const Field &field);
double fieldStd1(const Field &field);
double fieldVar(const Field &field);
double fieldVar1(const Field &field);
double fieldStdw(const Field &field);
double fieldStd1w(const Field &field);
double fieldVarw(const Field &field);
double fieldVar1w(const Field &field);

// ENS validation
double vfldrank(Field &field);

double fieldPctl(Field &field, double pn);

// fieldzon.cc
void fieldZonFunction(const Field &field1, Field &field2, int function);
void fieldZonMin(const Field &field1, Field &field2);
void fieldZonMax(const Field &field1, Field &field2);
void fieldZonRange(const Field &field1, Field &field2);
void fieldZonSum(const Field &field1, Field &field2);
void fieldZonAvg(const Field &field1, Field &field2);
void fieldZonMean(const Field &field1, Field &field2);
void fieldZonStd(const Field &field1, Field &field2);
void fieldZonStd1(const Field &field1, Field &field2);
void fieldZonVar(const Field &field1, Field &field2);
void fieldZonVar1(const Field &field1, Field &field2);
void fieldZonPctl(const Field &field1, Field &field2, int p);

// fieldmer.cc
void fieldMerFunction(const Field &field1, Field &field2, int function);
void fieldMerPctl(const Field &field1, Field &field2, int p);

void vfldrms(const Field &field1, const Field &field2, Field &field3);

// fieldc.cc
void vfarcfun(Field &field, double rconst, int function);
void vfarcmul(Field &field, double rconst);
void vfarcdiv(Field &field, double rconst);
void vfarcadd(Field &field, double rconst);
void vfarcsub(Field &field, double rconst);
void vfarcmin(Field &field, double rconst);
void vfarcmax(Field &field, double rconst);
void vfarmod(Field &field, double divisor);

// fieldccplx.cc
void vfarcfuncplx(Field &field, const double rconstcplx[2], int function);

// field2.cc
void vfarfun(Field &field1, const Field &field2, int function);

void vfaradd(Field &field1, const Field &field2);
void vfarsum(Field &field1, const Field &field2);
void vfarsub(Field &field1, const Field &field2);
void vfarmul(Field &field1, const Field &field2);
void vfardiv(Field &field1, const Field &field2);
void vfarmin(Field &field1, const Field &field2);
void vfarmax(Field &field1, const Field &field2);
void vfaratan2(Field &field1, const Field &field2);

void vfarsumq(Field &field1, const Field &field2);
void vfarsumw(Field &field1, const Field &field2, double w);
void vfarsumqw(Field &field1, const Field &field2, double w);
void vfarsumtr(Field &field1, const Field &field2, double refval);
void vfarcpy(Field &field1, const Field &field2);
void vfarvinit(Field &field1, const Field &field2);
void vfarvincr(Field &field1, const Field &field2);

void vfarsumsumq(Field &field1, Field &field2, const Field &field3);
void vfarmaxmin(Field &field1, Field &field2, const Field &field3);
void vfarminidx(Field &field1, Field &field2, const Field &field3, int idx);
void vfarmaxidx(Field &field1, Field &field2, const Field &field3, int idx);
void vfarvar(Field &field1, const Field &field2, const Field &field3, int divisor);
void vfarstd(Field &field1, const Field &field2, const Field &field3, int divisor);
void vfarcvar(Field &field1, const Field &field2, int nsets, int divisor);
void vfarcstd(Field &field1, const Field &field2, int nsets, int divisor);
void vfarmoq(Field &field1, const Field &field2);
void vfarmoqw(Field &field1, const Field &field2, double w);

void vfarcount(Field &field1, const Field &field2);

// field2cplx.cc
void vfarfuncplx(Field &field1, const Field &field2, int function);

#endif /* FIELD_H */
