/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.1

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Zonstat    zonmin          Zonal minimum
      Zonstat    zonmax          Zonal maximum
      Zonstat    zonrange        Zonal range
      Zonstat    zonsum          Zonal sum
      Zonstat    zonmean         Zonal mean
      Zonstat    zonavg          Zonal average
      Zonstat    zonstd          Zonal standard deviation
      Zonstat    zonstd1         Zonal standard deviation [Normalize by (n-1)]
      Zonstat    zonvar          Zonal variance
      Zonstat    zonvar1         Zonal variance [Normalize by (n-1)]
      Zonstat    zonpctl         Zonal percentiles
*/

#include <cdi.h>

#include <utility>

#include "cdo_options.h"
#include "cdo_vlist.h"
#include "functs.h"
#include "process_int.h"
#include "param_conversion.h"
#include <mpim_grid.h>
#include "griddes.h"

void zonal_mean_weights(const int gridID1, const int gridID2, Varray2D<size_t> &remapIndices, Varray2D<double> &remapWeights);
void zonal_mean(const Varray2D<size_t> &remapIndices, const Varray2D<double> &remapWeights, const Field &field1, Field &field2);

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("zonmin",   func_min,   0, nullptr);
  cdoOperatorAdd("zonmax",   func_max,   0, nullptr);
  cdoOperatorAdd("zonrange", func_range, 0, nullptr);
  cdoOperatorAdd("zonsum",   func_sum,   0, nullptr);
  cdoOperatorAdd("zonmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("zonavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("zonvar",   func_var,   0, nullptr);
  cdoOperatorAdd("zonvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("zonstd",   func_std,   0, nullptr);
  cdoOperatorAdd("zonstd1",  func_std1,  0, nullptr);
  cdoOperatorAdd("zonskew",  func_skew,  0, nullptr);
  cdoOperatorAdd("zonkurt",  func_kurt,  0, nullptr);
  cdoOperatorAdd("zonpctl",  func_pctl,  0, nullptr);
  // clang-format on
}

void *
Zonstat(void *process)
{
  int gridIDdestroy = -1, gridID1 = -1, gridID2 = -1;
  int zongridID = -1;
  int sourceGridIsRegular = true;

  cdoInitialize(process);

  addOperators();

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);

  double pn = 0.0;
  if (operfunc == func_pctl)
    {
      operatorInputArg("percentile number");
      pn = parameter2double(cdoOperatorArgv(0));
    }
  else if (cdoOperatorArgc() == 1 && operfunc == func_mean)
    {
      sourceGridIsRegular = false;
      gridID2 = cdoDefineGrid(cdoOperatorArgv(0));
      const auto gridtype = gridInqType(gridID2);
      if (gridtype != GRID_GAUSSIAN && gridtype != GRID_LONLAT) cdoAbort("Target grid type must be Gaussian or LonLat!");
      if (!gridInqYbounds(gridID2, NULL)) cdoAbort("Target grid cell bounds missing!");
      if (gridInqXsize(gridID2) > 1) cdoAbort("Target grid must be zonal!");
    }
  else
    {
      operatorCheckArgc(0);
    }

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto ngrids = vlistNgrids(vlistID1);
  for (int index = 0; index < ngrids; index++)
    {
      const auto gridID = vlistGrid(vlistID1, index);
      const auto xsize = gridInqXsize(gridID);
      const auto ysize = gridInqYsize(gridID);
      const auto gridtype = gridInqType(gridID);
      if (xsize > 1 || gridtype == GRID_GAUSSIAN_REDUCED)
        {
          if (gridID1 == -1) gridID1 = gridID;
        }
      else
        {
          if (ysize > 1 && zongridID == -1) zongridID = gridID;
        }
    }

  int ndiffgrids = 0;
  for (int index = 0; index < ngrids; index++)
    {
      const auto gridID = vlistGrid(vlistID1, index);
      if (zongridID != -1 && zongridID == gridID) continue;
      if (gridID1 != gridID) ndiffgrids++;
    }
  if (ndiffgrids) cdoAbort("Too many different grids!");

  if (gridID1 != -1)
    {
      const auto gridtype = gridInqType(gridID1);
      if (sourceGridIsRegular)
        {
          if (gridtype == GRID_LONLAT || gridtype == GRID_GAUSSIAN ||
              gridtype == GRID_GAUSSIAN_REDUCED || gridtype == GRID_GENERIC)
            {
              gridID2 = (zongridID != -1 && gridInqYsize(zongridID) == gridInqYsize(gridID1)) ? zongridID : gridToZonal(gridID1);
            }
          else
            {
              cdoAbort("Unsupported gridtype: %s", gridNamePtr(gridtype));
            }
        }
      else
        {
          const auto gridID = generate_full_cell_grid(gridID1);
          if (gridID != gridID1) gridIDdestroy = gridID1 = gridID;
        }
    }
  else
    {
      gridID2 = zongridID;
      cdoWarning("Input stream already contains zonal data!");
    }

  for (int index = 0; index < ngrids; index++) vlistChangeGridIndex(vlistID2, index, gridID2);

  if (Options::cdoChunkType == CDI_UNDEFID)
    {
      const auto nvars = vlistNvars(vlistID2);
      for (int varID = 0; varID < nvars; ++varID) vlistDefVarChunkType(vlistID2, varID, CDI_CHUNK_AUTO);
    }

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  VarList varList1;
  varListInit(varList1, vlistID1);

  const auto nlatmax = gridInqYsize(gridID2);

  Field field1, field2;
  field2.resize(nlatmax, 0.0);
  field2.grid = gridID2;
  field2.memType = MemType::Double;

  Varray2D<size_t> remapIndices;
  Varray2D<double> remapWeights;

  if (!sourceGridIsRegular) zonal_mean_weights(gridID1, gridID2, remapIndices, remapWeights);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdoInqRecord(streamID1, &varID, &levelID);
          field1.init(varList1[varID]);
          cdoReadRecord(streamID1, field1);
          field1.grid = gridID1;

          field2.missval = field1.missval;

          if (zongridID != -1 && zongridID == field1.grid)
            {
              field2.nmiss = field1.nmiss;
              if (field1.memType == MemType::Float)
                varrayCopy(nlatmax, field1.vec_f, field2.vec_d);
              else
                varrayCopy(nlatmax, field1.vec_d, field2.vec_d);
            }
          else if (sourceGridIsRegular)
            {
              (operfunc == func_pctl) ? fieldZonPctl(field1, field2, pn) : fieldZonFunction(field1, field2, operfunc);
            }
          else
            {
              zonal_mean(remapIndices, remapWeights, field1, field2);
            }

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, field2);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  if (gridIDdestroy != -1) gridDestroy(gridIDdestroy);

  cdoFinish();

  return nullptr;
}
