/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Yseasstat  yseasrange      Multi-year seasonal range
      Yseasstat  yseasmin        Multi-year seasonal minimum
      Yseasstat  yseasmax        Multi-year seasonal maximum
      Yseasstat  yseassum        Multi-year seasonal sum
      Yseasstat  yseasmean       Multi-year seasonal mean
      Yseasstat  yseasavg        Multi-year seasonal average
      Yseasstat  yseasvar        Multi-year seasonal variance
      Yseasstat  yseasvar1       Multi-year seasonal variance [Normalize by (n-1)]
      Yseasstat  yseasstd        Multi-year seasonal standard deviation
      Yseasstat  yseasstd1       Multi-year seasonal standard deviation [Normalize by (n-1)]
*/

#include <cdi.h>

#include "cdo_season.h"
#include "datetime.h"
#include "functs.h"
#include "process_int.h"

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("yseasrange", func_range, 0, nullptr);
  cdoOperatorAdd("yseasmin",   func_min,   0, nullptr);
  cdoOperatorAdd("yseasmax",   func_max,   0, nullptr);
  cdoOperatorAdd("yseassum",   func_sum,   0, nullptr);
  cdoOperatorAdd("yseasmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("yseasavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("yseasvar",   func_var,   0, nullptr);
  cdoOperatorAdd("yseasvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("yseasstd",   func_std,   0, nullptr);
  cdoOperatorAdd("yseasstd1",  func_std1,  0, nullptr);
  // clang-format on
}

void *
Yseasstat(void *process)
{
  constexpr int MaxSeasons = 4;
  int seas_nsets[MaxSeasons] = { 0 };
  CdoDateTime datetime[MaxSeasons];
  FieldVector2D vars1[MaxSeasons], vars2[MaxSeasons], samp1[MaxSeasons];

  cdoInitialize(process);

  addOperators();

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);

  const auto lrange = (operfunc == func_range);
  const auto lmean = (operfunc == func_mean || operfunc == func_avg);
  const auto lstd = (operfunc == func_std || operfunc == func_std1);
  const auto lvarstd = (lstd || operfunc == func_var || operfunc == func_var1);
  const auto lvars2 = (lvarstd || lrange);
  const int divisor = (operfunc == func_std1 || operfunc == func_var1);

  auto vfarstdvar_func = lstd ? vfarstd : vfarvar;
  auto vfarcstdvar_func = lstd ? vfarcstd : vfarcvar;

  operatorCheckArgc(0);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID2)) taxisDeleteBounds(taxisID2);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  VarList varList;
  varListInit(varList, vlistID1);

  int VARS_MEMTYPE = 0;
  if ((operfunc == func_min) || (operfunc == func_max)) VARS_MEMTYPE = FIELD_NAT;

  Field field;

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      const auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      const auto vdate = taxisInqVdate(taxisID1);
      const auto vtime = taxisInqVtime(taxisID1);

      const auto seas = monthToSeason(decodeMonth(vdate));

      setDateTime(datetime[seas], vdate, vtime);

      if (!vars1[seas].size())
        {
          fieldsFromVlist(vlistID1, samp1[seas]);
          fieldsFromVlist(vlistID1, vars1[seas], FIELD_VEC | VARS_MEMTYPE);
          if (lvars2) fieldsFromVlist(vlistID1, vars2[seas], FIELD_VEC);
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = (varList[varID].timetype == TIME_CONSTANT);
            }

          auto &rsamp1 = samp1[seas][varID][levelID];
          auto &rvars1 = vars1[seas][varID][levelID];
          const auto nsets = seas_nsets[seas];

          if (nsets == 0)
            {
              cdoReadRecord(streamID1, rvars1);
              if (lrange)
                {
                  vars2[seas][varID][levelID].nmiss = rvars1.nmiss;
                  vars2[seas][varID][levelID].vec_d = rvars1.vec_d;
                }

              if (rvars1.nmiss || !rsamp1.empty())
                {
                  if (rsamp1.empty()) rsamp1.resize(rvars1.size);
                  vfarvinit(rsamp1, rvars1);
                }
            }
          else
            {
              field.init(varList[varID]);
              cdoReadRecord(streamID1, field);

              if (field.nmiss || !rsamp1.empty())
                {
                  if (rsamp1.empty()) rsamp1.resize(rvars1.size, nsets);
                  vfarvinit(rsamp1, rvars1);
                }

              // clang-format off
              if      (lvarstd) vfarsumsumq(rvars1, vars2[seas][varID][levelID], field);
              else if (lrange)  vfarmaxmin(rvars1, vars2[seas][varID][levelID], field);
              else              vfarfun(rvars1, field, operfunc);
              // clang-format on
            }
        }

      if (seas_nsets[seas] == 0 && lvarstd)
        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (recList[recID].lconst) continue;

            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            vfarmoq(vars2[seas][varID][levelID], vars1[seas][varID][levelID]);
          }

      seas_nsets[seas]++;
      tsID++;
    }

  for (int seas = 0; seas < MaxSeasons; seas++)
    if (seas_nsets[seas])
      {
        const auto nsets = seas_nsets[seas];
        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (recList[recID].lconst) continue;

            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            const auto &rsamp1 = samp1[seas][varID][levelID];
            auto &rvars1 = vars1[seas][varID][levelID];

            if (lmean)
              {
                if (!rsamp1.empty())
                  vfardiv(rvars1, rsamp1);
                else
                  vfarcdiv(rvars1, (double) nsets);
              }
            else if (lvarstd)
              {
                if (!rsamp1.empty())
                  vfarstdvar_func(rvars1, vars2[seas][varID][levelID], rsamp1, divisor);
                else
                  vfarcstdvar_func(rvars1, vars2[seas][varID][levelID], nsets, divisor);
              }
            else if (lrange)
              {
                vfarsub(rvars1, vars2[seas][varID][levelID]);
              }
          }

        taxisDefVdate(taxisID2, datetime[seas].date);
        taxisDefVtime(taxisID2, datetime[seas].time);
        cdoDefTimestep(streamID2, otsID);

        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (otsID && recList[recID].lconst) continue;

            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            auto &rvars1 = vars1[seas][varID][levelID];

            cdoDefRecord(streamID2, varID, levelID);
            cdoWriteRecord(streamID2, rvars1);
          }

        otsID++;
      }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
