/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Timstat    timrange        Time range
      Timstat    timmin          Time minimum
      Timstat    timmax          Time maximum
      Timstat    timsum          Time sum
      Timstat    timmean         Time mean
      Timstat    timavg          Time average
      Timstat    timvar          Time variance
      Timstat    timvar1         Time variance [Normalize by (n-1)]
      Timstat    timstd          Time standard deviation
      Timstat    timstd1         Time standard deviation [Normalize by (n-1)]
      Hourstat   hourrange       Hourly range
      Hourstat   hourmin         Hourly minimum
      Hourstat   hourmax         Hourly maximum
      Hourstat   hoursum         Hourly sum
      Hourstat   hourmean        Hourly mean
      Hourstat   houravg         Hourly average
      Hourstat   hourvar         Hourly variance
      Hourstat   hourvar1        Hourly variance [Normalize by (n-1)]
      Hourstat   hourstd         Hourly standard deviation
      Hourstat   hourstd1        Hourly standard deviation [Normalize by (n-1)]
      Daystat    dayrange        Daily range
      Daystat    daymin          Daily minimum
      Daystat    daymax          Daily maximum
      Daystat    daysum          Daily sum
      Daystat    daymean         Daily mean
      Daystat    dayavg          Daily average
      Daystat    dayvar          Daily variance
      Daystat    dayvar1         Daily variance [Normalize by (n-1)]
      Daystat    daystd          Daily standard deviation
      Daystat    daystd1         Daily standard deviation [Normalize by (n-1)]
      Monstat    monrange        Monthly range
      Monstat    monmin          Monthly minimum
      Monstat    monmax          Monthly maximum
      Monstat    monsum          Monthly sum
      Monstat    monmean         Monthly mean
      Monstat    monavg          Monthly average
      Monstat    monvar          Monthly variance
      Monstat    monvar1         Monthly variance [Normalize by (n-1)]
      Monstat    monstd          Monthly standard deviation
      Monstat    monstd1         Monthly standard deviation [Normalize by (n-1)]
      Yearstat   yearrange       Yearly range
      Yearstat   yearmin         Yearly minimum
      Yearstat   yearmax         Yearly maximum
      Yearstat   yearsum         Yearly sum
      Yearstat   yearmean        Yearly mean
      Yearstat   yearavg         Yearly average
      Yearstat   yearvar         Yearly variance
      Yearstat   yearvar1        Yearly variance [Normalize by (n-1)]
      Yearstat   yearstd         Yearly standard deviation
      Yearstat   yearstd1        Yearly standard deviation [Normalize by (n-1)]
*/

#include <cdi.h>

#include "cdo_options.h"
#include "cdo_vlist.h"
#include "functs.h"
#include "process_int.h"
#include "datetime.h"
#include "printinfo.h"
#include "util_date.h"
#include "param_conversion.h"
#include "progress.h"

enum
{
  HOUR_LEN = 4,
  DAY_LEN = 6,
  MON_LEN = 8,
  YEAR_LEN = 10
};

static void
vlistSetFrequency(const int vlistID, const int comparelen)
{
  const char *freq = nullptr;
  // clang-format off
  if      (comparelen == DAY_LEN)  freq = "day";
  else if (comparelen == MON_LEN)  freq = "mon";
  else if (comparelen == YEAR_LEN) freq = "year";
  // clang-format on
  if (freq) cdiDefAttTxt(vlistID, CDI_GLOBAL, "frequency", (int) strlen(freq), freq);
}

static void
set_missval(Field &field, const Field &samp, int nsets, double vfrac)
{
  const auto fieldsize = field.size;
  const auto missval = field.missval;

  size_t irun = 0;
  for (size_t i = 0; i < fieldsize; ++i)
    {
      if ((samp.vec_d[i] / nsets) < vfrac)
        {
          field.vec_d[i] = missval;
          irun++;
        }
    }

  if (irun) field.nmiss = fieldNumMiss(field);
}

static void
set_array_value(Varray<double> &array, double value)
{
  const auto size = array.size();
  for (size_t i = 0; i < size; ++i) array[i] = value;
}

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("timrange",   func_range,  DATE_LEN, nullptr);
  cdoOperatorAdd("timmin",     func_min,    DATE_LEN, nullptr);
  cdoOperatorAdd("timmax",     func_max,    DATE_LEN, nullptr);
  cdoOperatorAdd("timminidx",  func_minidx, DATE_LEN, nullptr);
  cdoOperatorAdd("timmaxidx",  func_maxidx, DATE_LEN, nullptr);
  cdoOperatorAdd("timsum",     func_sum,    DATE_LEN, nullptr);
  cdoOperatorAdd("timmean",    func_mean,   DATE_LEN, nullptr);
  cdoOperatorAdd("timavg",     func_avg,    DATE_LEN, nullptr);
  cdoOperatorAdd("timvar",     func_var,    DATE_LEN, nullptr);
  cdoOperatorAdd("timvar1",    func_var1,   DATE_LEN, nullptr);
  cdoOperatorAdd("timstd",     func_std,    DATE_LEN, nullptr);
  cdoOperatorAdd("timstd1",    func_std1,   DATE_LEN, nullptr);
  cdoOperatorAdd("yearrange",  func_range,  YEAR_LEN, nullptr);
  cdoOperatorAdd("yearmin",    func_min,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearmax",    func_max,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearminidx", func_minidx, YEAR_LEN, nullptr);
  cdoOperatorAdd("yearmaxidx", func_maxidx, YEAR_LEN, nullptr);
  cdoOperatorAdd("yearsum",    func_sum,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearmean",   func_mean,   YEAR_LEN, nullptr);
  cdoOperatorAdd("yearavg",    func_avg,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearvar",    func_var,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearvar1",   func_var1,   YEAR_LEN, nullptr);
  cdoOperatorAdd("yearstd",    func_std,    YEAR_LEN, nullptr);
  cdoOperatorAdd("yearstd1",   func_std1,   YEAR_LEN, nullptr);
  cdoOperatorAdd("monrange",   func_range,  MON_LEN, nullptr);
  cdoOperatorAdd("monmin",     func_min,    MON_LEN, nullptr);
  cdoOperatorAdd("monmax",     func_max,    MON_LEN, nullptr);
  cdoOperatorAdd("monsum",     func_sum,    MON_LEN, nullptr);
  cdoOperatorAdd("monmean",    func_mean,   MON_LEN, nullptr);
  cdoOperatorAdd("monavg",     func_avg,    MON_LEN, nullptr);
  cdoOperatorAdd("monvar",     func_var,    MON_LEN, nullptr);
  cdoOperatorAdd("monvar1",    func_var1,   MON_LEN, nullptr);
  cdoOperatorAdd("monstd",     func_std,    MON_LEN, nullptr);
  cdoOperatorAdd("monstd1",    func_std1,   MON_LEN, nullptr);
  cdoOperatorAdd("dayrange",   func_range,  DAY_LEN, nullptr);
  cdoOperatorAdd("daymin",     func_min,    DAY_LEN, nullptr);
  cdoOperatorAdd("daymax",     func_max,    DAY_LEN, nullptr);
  cdoOperatorAdd("daysum",     func_sum,    DAY_LEN, nullptr);
  cdoOperatorAdd("daymean",    func_mean,   DAY_LEN, nullptr);
  cdoOperatorAdd("dayavg",     func_avg,    DAY_LEN, nullptr);
  cdoOperatorAdd("dayvar",     func_var,    DAY_LEN, nullptr);
  cdoOperatorAdd("dayvar1",    func_var1,   DAY_LEN, nullptr);
  cdoOperatorAdd("daystd",     func_std,    DAY_LEN, nullptr);
  cdoOperatorAdd("daystd1",    func_std1,   DAY_LEN, nullptr);
  cdoOperatorAdd("hourrange",  func_range,  HOUR_LEN, nullptr);
  cdoOperatorAdd("hourmin",    func_min,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourmax",    func_max,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hoursum",    func_sum,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourmean",   func_mean,   HOUR_LEN, nullptr);
  cdoOperatorAdd("houravg",    func_avg,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourvar",    func_var,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourvar1",   func_var1,   HOUR_LEN, nullptr);
  cdoOperatorAdd("hourstd",    func_std,    HOUR_LEN, nullptr);
  cdoOperatorAdd("hourstd1",   func_std1,   HOUR_LEN, nullptr);
  // clang-format on
}

void *
Timstat(void *argument)
{
  const TimeStat timestat_date = TimeStat::MEAN;
  int64_t vdate0 = 0;
  int vtime0 = 0;
  CdoStreamID streamID3;
  int vlistID3, taxisID3 = -1;
  bool lvfrac = false;
  char indate1[DATE_LEN + 1], indate2[DATE_LEN + 1];
  double vfrac = 1;

  cdoInitialize(argument);

  addOperators();

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);
  const auto comparelen = cdoOperatorF2(operatorID);

  const auto lminidx = (operfunc == func_minidx);
  const auto lmaxidx = (operfunc == func_maxidx);
  const auto lrange = (operfunc == func_range);
  const auto lmean = (operfunc == func_mean || operfunc == func_avg);
  const auto lstd = (operfunc == func_std || operfunc == func_std1);
  const auto lvarstd = (lstd || operfunc == func_var || operfunc == func_var1);
  const auto lvars2 = (lvarstd || lrange || lminidx || lmaxidx);
  const int divisor = (operfunc == func_std1 || operfunc == func_var1);

  auto vfarstdvar_func = lstd ? vfarstd : vfarvar;
  auto vfarcstdvar_func = lstd ? vfarcstd : vfarcvar;

  if (operfunc == func_mean)
    {
      const auto oargc = operatorArgc();
      if (oargc == 1)
        {
          lvfrac = true;
          vfrac = parameter2double(cdoOperatorArgv(0));
          if (Options::cdoVerbose) cdoPrint("Set vfrac to %g", vfrac);
          if (vfrac < 0 || vfrac > 1) cdoAbort("vfrac out of range!");
        }
      else if (oargc > 1)
        cdoAbort("Too many arguments!");
    }
  else
    {
      operatorCheckArgc(0);
    }

  const int cmplen = DATE_LEN - comparelen;

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  vlistDefineTimestepType(vlistID2, operfunc);

  vlistDefNtsteps(vlistID2, (cmplen == 0) ? 1 : -1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID2);
  if (taxisInqType(taxisID2) == TAXIS_FORECAST) taxisDefType(taxisID2, TAXIS_RELATIVE);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto nvars = vlistNvars(vlistID1);

  if (lminidx || lmaxidx)
    for (int varID = 0; varID < nvars; ++varID) vlistDefVarDatatype(vlistID2, varID, CDI_DATATYPE_INT32);

  vlistSetFrequency(vlistID2, comparelen);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  if (Options::cdoDiag)
    {
      char filename[8192];
      strcpy(filename, cdoOperatorName(operatorID));
      strcat(filename, "_");
      strcat(filename, cdoGetStreamName(1));
      streamID3 = cdoOpenWrite(filename);

      vlistID3 = vlistDuplicate(vlistID1);

      for (int varID = 0; varID < nvars; ++varID)
        {
          vlistDefVarDatatype(vlistID3, varID, CDI_DATATYPE_INT32);
          vlistDefVarMissval(vlistID3, varID, -1);
          cdiDefKeyString(vlistID3, varID, CDI_KEY_UNITS, "");
          vlistDefVarAddoffset(vlistID3, varID, 0);
          vlistDefVarScalefactor(vlistID3, varID, 1);
        }

      taxisID3 = taxisDuplicate(taxisID1);
      taxisWithBounds(taxisID3);
      vlistDefTaxis(vlistID3, taxisID3);

      cdoDefVlist(streamID3, vlistID3);
    }

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  DateTimeList dtlist;
  dtlist.setStat(timestat_date);
  dtlist.setCalendar(taxisInqCalendar(taxisID1));

  VarList varList;
  varListInit(varList, vlistID1);

  int VARS_MEMTYPE = 0;
  if ((operfunc == func_min) || (operfunc == func_max)) VARS_MEMTYPE = FIELD_NAT;
  // if ((Options::CDO_???(--single) == MemType::Float) && (operfunc == func_mean)) VARS_MEMTYPE = FIELD_NAT;
  // if (Options::CDO_Memtype == MemType::Float) VARS_MEMTYPE = FIELD_FLT;
  if (Options::cdoDiag || (lvfrac && operfunc == func_mean)) VARS_MEMTYPE = FIELD_DBL;

  Field field;
  Varray<double> samp;

  FieldVector2D samp1, vars1, vars2;
  fieldsFromVlist(vlistID1, samp1);
  fieldsFromVlist(vlistID1, vars1, FIELD_VEC | VARS_MEMTYPE);
  if (lvars2) fieldsFromVlist(vlistID1, vars2, FIELD_VEC);

  const auto ntsteps1 = vlistNtsteps(vlistID1);

  if (!Options::cdoVerbose && ntsteps1 > 1) progress::init();

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      int nsets = 0;
      int nrecs = 0;
      while (true)
        {
          nrecs = cdoStreamInqTimestep(streamID1, tsID);
          if (nrecs == 0) break;

          if (!Options::cdoVerbose && ntsteps1 > 1) progress::update(0, 1, (tsID + 1.) / ntsteps1);

          dtlist.taxisInqTimestep(taxisID1, nsets);
          const auto vdate = dtlist.getVdate(nsets);
          const auto vtime = dtlist.getVtime(nsets);

          if (nsets == 0) SET_DATE(indate2, vdate, vtime);
          SET_DATE(indate1, vdate, vtime);

          if (DATE_IS_NEQ(indate1, indate2, cmplen)) break;

          for (int recID = 0; recID < nrecs; recID++)
            {
              int varID, levelID;
              cdoInqRecord(streamID1, &varID, &levelID);

              if (tsID == 0)
                {
                  recList[recID].varID = varID;
                  recList[recID].levelID = levelID;
                  recList[recID].lconst = (varList[varID].timetype == TIME_CONSTANT);
                }

              auto &rsamp1 = samp1[varID][levelID];
              auto &rvars1 = vars1[varID][levelID];

              if (nsets == 0)
                {
                  cdoReadRecord(streamID1, rvars1);
                  if (lrange || lminidx || lmaxidx)
                    {
                      vars2[varID][levelID].nmiss = rvars1.nmiss;
                      vars2[varID][levelID].vec_d = rvars1.vec_d;
                    }

                  if (lminidx || lmaxidx) fieldFill(rvars1, 0.0);

                  if (rvars1.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(rvars1.size);
                      vfarvinit(rsamp1, rvars1);
                    }
                }
              else
                {
                  field.init(varList[varID]);
                  cdoReadRecord(streamID1, field);

                  if (field.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(rvars1.size, nsets);
                      vfarvincr(rsamp1, field);
                    }

                  // clang-format off
                  if      (lvarstd) vfarsumsumq(rvars1, vars2[varID][levelID], field);
                  else if (lrange)  vfarmaxmin(rvars1, vars2[varID][levelID], field);
                  else if (lminidx) vfarminidx(rvars1, vars2[varID][levelID], field, nsets);
                  else if (lmaxidx) vfarmaxidx(rvars1, vars2[varID][levelID], field, nsets);
                  else              vfarfun(rvars1, field, operfunc);
                  // clang-format on
                }
            }

          if (nsets == 0 && lvarstd)
            for (int recID = 0; recID < maxrecs; recID++)
              {
                if (recList[recID].lconst) continue;

                const auto varID = recList[recID].varID;
                const auto levelID = recList[recID].levelID;
                vfarmoq(vars2[varID][levelID], vars1[varID][levelID]);
              }

          vdate0 = vdate;
          vtime0 = vtime;
          nsets++;
          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (recList[recID].lconst) continue;

          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          const auto &rsamp1 = samp1[varID][levelID];
          auto &rvars1 = vars1[varID][levelID];

          if (lmean)
            {
              if (!rsamp1.empty())
                vfardiv(rvars1, rsamp1);
              else
                vfarcdiv(rvars1, (double) nsets);
            }
          else if (lvarstd)
            {
              if (!rsamp1.empty())
                vfarstdvar_func(rvars1, vars2[varID][levelID], rsamp1, divisor);
              else
                vfarcstdvar_func(rvars1, vars2[varID][levelID], nsets, divisor);
            }
          else if (lrange)
            {
              vfarsub(rvars1, vars2[varID][levelID]);
            }
        }

      if (Options::cdoVerbose)
        cdoPrint("%s %s  vfrac = %g, nsets = %d", dateToString(vdate0).c_str(), timeToString(vtime0).c_str(), vfrac, nsets);

      if (lvfrac && operfunc == func_mean)
        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (recList[recID].lconst) continue;

            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            const auto &rsamp1 = samp1[varID][levelID];

            if (!rsamp1.empty()) set_missval(vars1[varID][levelID], rsamp1, nsets, vfrac);
          }

      dtlist.statTaxisDefTimestep(taxisID2, nsets);
      cdoDefTimestep(streamID2, otsID);

      if (Options::cdoDiag)
        {
          dtlist.statTaxisDefTimestep(taxisID3, nsets);
          cdoDefTimestep(streamID3, otsID);
        }

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recList[recID].lconst) continue;

          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          auto &rvars1 = vars1[varID][levelID];

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, rvars1);

          if (Options::cdoDiag)
            {
              auto &rsamp1 = samp1[varID][levelID];
              samp.resize(field.size);
              if (!rsamp1.empty())
                samp = rsamp1.vec_d;
              else
                set_array_value(samp, nsets);

              cdoDefRecord(streamID3, varID, levelID);
              cdoWriteRecord(streamID3, samp.data(), 0);
            }
        }

      if (nrecs == 0) break;
      otsID++;
    }

  if (!Options::cdoVerbose && ntsteps1 > 1) progress::update(0, 1, 1);

  if (Options::cdoDiag) cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
