/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

*/

#include <cdi.h>

#include "process_int.h"
#include "printinfo.h"
#include "cdo_zaxis.h"

void *
Pinfo(void *process)
{
  size_t imiss = 0;
  double arrmin, arrmax, arrmean;

  cdoInitialize(process);

  // clang-format off
  const auto PINFO  = cdoOperatorAdd("pinfo",  0, 0, nullptr);
  const auto PINFOV = cdoOperatorAdd("pinfov", 0, 0, nullptr);
  // clang-format on

  (void) (PINFO);  // CDO_UNUSED

  const auto operatorID = cdoOperatorID();

  operatorCheckArgc(0);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  VarList varList1;
  varListInit(varList1, vlistID1);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  Varray<double> array1(gridsizemax), array2(gridsizemax);

  int indg = 0;
  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      const auto vdate = taxisInqVdate(taxisID1);
      const auto vtime = taxisInqVtime(taxisID1);
      const auto vdateString = dateToString(vdate);
      const auto vtimeString = timeToString(vtime);

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          if (tsID == 0 && recID == 0)
            {
              if (operatorID == PINFOV)
                fprintf(stdout,
                        "   Rec :       Date  Time    Varname     Level    Size    Miss :     Minimum        Mean     Maximum\n");
              else
                fprintf(stdout, "   Rec :       Date  Time    Code  Level    Size    Miss :     Minimum        Mean     Maximum\n");
            }

          int varID, levelID;
          cdoInqRecord(streamID1, &varID, &levelID);
          size_t nmiss;
          cdoReadRecord(streamID1, array1.data(), &nmiss);

          indg += 1;
          auto gridsize = varList1[varID].gridsize;

          if (operatorID == PINFOV)
            fprintf(stdout, "%6d :%s %s %-8s ", indg, vdateString.c_str(), vtimeString.c_str(), varList1[varID].name);
          else
            fprintf(stdout, "%6d :%s %s %3d", indg, vdateString.c_str(), vtimeString.c_str(), varList1[varID].code);

          const auto level = cdoZaxisInqLevel(varList1[varID].zaxisID, levelID);
          fprintf(stdout, " %7g ", level);

          fprintf(stdout, "%7zu %7zu :", gridsize, nmiss);

          if (gridInqType(varList1[varID].gridID) == GRID_SPECTRAL || (gridsize == 1 && nmiss == 0))
            {
              fprintf(stdout, "            %#12.5g\n", array1[0]);
            }
          else
            {
              if (nmiss)
                {
                  auto mmm = varrayMinMaxMeanMV(gridsize, array1, varList1[varID].missval);
                  arrmin = mmm.min;
                  arrmax = mmm.max;
                  arrmean = mmm.mean;
                  auto ivals = mmm.n;
                  imiss = gridsize - ivals;
                  gridsize = ivals;
                }
              else
                {
                  auto mmm = varrayMinMaxMean(gridsize, array1);
                  arrmin = mmm.min;
                  arrmax = mmm.max;
                  arrmean = mmm.mean;
                }

              if (gridsize)
                {
                  fprintf(stdout, "%#12.5g%#12.5g%#12.5g\n", arrmin, arrmean, arrmax);
                }
              else
                {
                  fprintf(stdout, "                     nan\n");
                }

              if (imiss != nmiss && nmiss) fprintf(stdout, "Found %zu of %zu missing values!\n", imiss, nmiss);
            }

          varrayCopy(gridsize, array1, array2);

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, array2.data(), nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
