/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Timpctl    timpctl         Time percentiles
      Hourpctl   hourpctl        Hourly percentiles
      Daypctl    daypctl         Daily percentiles
      Monpctl    monpctl         Monthly percentiles
      Yearpctl   yearpctl        Yearly percentiles
*/

#include <cdi.h>

#include "util_date.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "param_conversion.h"
#include "percentiles_hist.h"
#include "datetime.h"

static void
timpctl(int operatorID)
{
  const auto timestat_date = TimeStat::MEAN;
  char indate1[DATE_LEN + 1], indate2[DATE_LEN + 1];

  operatorInputArg("percentile number");
  const auto pn = parameter2double(cdoOperatorArgv(0));

  const int cmplen = DATE_LEN - cdoOperatorF2(operatorID);

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);
  const auto streamID3 = cdoOpenRead(2);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = cdoStreamInqVlist(streamID3);
  const auto vlistID4 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);
  vlistCompare(vlistID1, vlistID3, CMP_ALL);

  if (cdoOperatorF2(operatorID) == 16) vlistDefNtsteps(vlistID4, 1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = vlistInqTaxis(vlistID2);
  const auto taxisID3 = vlistInqTaxis(vlistID3);
  // TODO - check that time axes 2 and 3 are equal

  const auto taxisID4 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID4);
  vlistDefTaxis(vlistID4, taxisID4);

  const auto streamID4 = cdoOpenWrite(3);
  cdoDefVlist(streamID4, vlistID4);

  const auto ntsteps = vlistNtsteps(vlistID1);
  const auto nvars = vlistNvars(vlistID1);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  FieldVector constFields(maxrecs);

  DateTimeList dtlist;
  dtlist.setStat(timestat_date);
  dtlist.setCalendar(taxisInqCalendar(taxisID1));

  Field field1, field2;

  VarList varList1;
  varListInit(varList1, vlistID1);

  HistogramSet hset(nvars, ntsteps);

  for (int varID = 0; varID < nvars; varID++)
    hset.createVarLevels(varID, varList1[varID].nlevels, varList1[varID].gridsize);

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      auto nrecs = cdoStreamInqTimestep(streamID2, otsID);
      if (nrecs != cdoStreamInqTimestep(streamID3, otsID))
        cdoAbort("Number of records at time step %d of %s and %s differ!", otsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      const auto vdate2 = taxisInqVdate(taxisID2);
      const auto vtime2 = taxisInqVtime(taxisID2);
      const auto vdate3 = taxisInqVdate(taxisID3);
      const auto vtime3 = taxisInqVtime(taxisID3);
      if (vdate2 != vdate3 || vtime2 != vtime3)
        cdoAbort("Verification dates at time step %d of %s and %s differ!", otsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdoInqRecord(streamID2, &varID, &levelID);
          field1.init(varList1[varID]);
          cdoReadRecord(streamID2, field1);

          cdoInqRecord(streamID3, &varID, &levelID);
          field2.init(varList1[varID]);
          cdoReadRecord(streamID3, field2);

          hset.defVarLevelBounds(varID, levelID, field1, field2);
        }

      int nsets = 0;
      while (nrecs && (nrecs = cdoStreamInqTimestep(streamID1, tsID)))
        {
          dtlist.taxisInqTimestep(taxisID1, nsets);
          const auto vdate1 = dtlist.getVdate(nsets);
          const auto vtime1 = dtlist.getVtime(nsets);

          if (nsets == 0) SET_DATE(indate2, vdate1, vtime1);
          SET_DATE(indate1, vdate1, vtime1);

          if (DATE_IS_NEQ(indate1, indate2, cmplen)) break;

          for (int recID = 0; recID < nrecs; recID++)
            {
              int varID, levelID;
              cdoInqRecord(streamID1, &varID, &levelID);

              if (tsID == 0)
                {
                  recList[recID].varID = varID;
                  recList[recID].levelID = levelID;
                  recList[recID].lconst = (varList1[varID].timetype == TIME_CONSTANT);
                }

              if (tsID == 0 && recList[recID].lconst)
                {
                  constFields[recID].init(varList1[varID]);
                  cdoReadRecord(streamID1, constFields[recID]);
                }
              else
                {
                  field1.init(varList1[varID]);
                  cdoReadRecord(streamID1, field1);

                  hset.addVarLevelValues(varID, levelID, field1);
                }
            }

          nsets++;
          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      dtlist.statTaxisDefTimestep(taxisID4, nsets);
      cdoDefTimestep(streamID4, otsID);

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recList[recID].lconst) continue;

          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          cdoDefRecord(streamID4, varID, levelID);

          if (recList[recID].lconst)
            {
              cdoWriteRecord(streamID4, constFields[recID]);
            }
          else
            {
              field1.init(varList1[varID]);
              hset.getVarLevelPercentiles(field1, varID, levelID, pn);
              cdoWriteRecord(streamID4, field1);
            }
        }

      if (nrecs == 0) break;
      otsID++;
    }

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);
}

void *
Timpctl(void *process)
{
  cdoInitialize(process);

  // clang-format off
  cdoOperatorAdd("timpctl",  func_pctl, 31, nullptr);
  cdoOperatorAdd("yearpctl", func_pctl, 10, nullptr);
  cdoOperatorAdd("monpctl",  func_pctl,  8, nullptr);
  cdoOperatorAdd("daypctl",  func_pctl,  6, nullptr);
  cdoOperatorAdd("hourpctl", func_pctl,  4, nullptr);
  // clang-format on

  timpctl(cdoOperatorID());

  cdoFinish();

  return nullptr;
}
