/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Splittime  splithour       Split hours
      Splittime  splitday        Split days
      Splittime  splitmon        Split months
      Splittime  splitseas       Split seasons
*/

#include <time.h>
#include <cdi.h>

#include "cdo_options.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_season.h"
#include "util_files.h"

constexpr int MaxStreams = 32;

struct tm
datetime_to_tm(int64_t date, int time)
{
  int year, month, day, hour, minute, second;
  cdiDecodeDate(date, &year, &month, &day);
  cdiDecodeTime(time, &hour, &minute, &second);

  struct tm stime;
  memset(&stime, 0, sizeof(struct tm));

  stime.tm_sec = second;
  stime.tm_min = minute;
  stime.tm_hour = hour;
  stime.tm_mday = day;
  stime.tm_mon = month - 1;
  stime.tm_year = year - 1900;

  return stime;
}

void *
Splittime(void *process)
{
  CdoStreamID streamID2;
  int varID, levelID;
  CdoStreamID streamIDs[MaxStreams];
  int tsIDs[MaxStreams];
  int index = 0;
  size_t nmiss;

  cdoInitialize(process);

  if (processSelf().m_ID != 0) cdoAbort("This operator can't be combined with other operators!");

  const auto lcopy = unchangedRecord();

  // clang-format off
  const auto SPLITHOUR = cdoOperatorAdd("splithour", func_time, 10000, nullptr);
  const auto SPLITDAY  = cdoOperatorAdd("splitday",  func_date,     1, nullptr);
  const auto SPLITMON  = cdoOperatorAdd("splitmon",  func_date,   100, nullptr);
  const auto SPLITSEAS = cdoOperatorAdd("splitseas", func_date,   100, nullptr);
  // clang-format on

  (void) (SPLITDAY);   // CDO_UNUSED
  (void) (SPLITHOUR);  // CDO_UNUSED

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);
  const auto operintval = cdoOperatorF2(operatorID);

  const char *format = nullptr;
  if (operatorID == SPLITMON && operatorArgc() == 1) format = cdoOperatorArgv(0).c_str();
  else  operatorCheckArgc(0);

  const char *seas_name[4];
  getSeasonName(seas_name);

  for (int i = 0; i < MaxStreams; i++) streamIDs[i] = CDO_STREAM_UNDEF;
  for (int i = 0; i < MaxStreams; i++) tsIDs[i] = 0;

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  char filename[8192];
  strcpy(filename, cdoGetObase());
  const int nchars = strlen(filename);

  auto refname = cdoGetStreamName(0);
  char filesuffix[32] = { 0 };
  cdoGenFileSuffix(filesuffix, sizeof(filesuffix), cdoInqFiletype(streamID1), vlistID1, refname);

  Varray<double> array;
  //  if ( ! lcopy )
  {
    auto gridsizemax = vlistGridsizeMax(vlistID1);
    if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;
    array.resize(gridsizemax);
  }

  const auto nvars = vlistNvars(vlistID1);
  int nconst = 0;
  for (varID = 0; varID < nvars; varID++)
    if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) nconst++;

  FieldVector2D vars;
  if (nconst)
    {
      vars.resize(nvars);

      for (varID = 0; varID < nvars; varID++)
        {
          if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
            {
              const auto gridID = vlistInqVarGrid(vlistID1, varID);
              const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
              const auto gridsize = gridInqSize(gridID);

              vars[varID].resize(nlevel);

              for (levelID = 0; levelID < nlevel; levelID++)
                {
                  vars[varID][levelID].grid = gridID;
                  vars[varID][levelID].resize(gridsize);
                }
            }
        }
    }

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      const auto vdate = taxisInqVdate(taxisID1);
      const auto vtime = taxisInqVtime(taxisID1);

      if (operfunc == func_date)
        {
          index = (vdate / operintval) % 100;
          if (index < 0) index = -index;

          if (operatorID == SPLITSEAS) index = monthToSeason(index);
        }
      else if (operfunc == func_time)
        {
          index = (vtime / operintval) % 100;
        }

      if (index < 0 || index >= MaxStreams) cdoAbort("Index out of range!");

      streamID2 = streamIDs[index];
      if (streamID2 == CDO_STREAM_UNDEF)
        {
          if (operatorID == SPLITSEAS)
            {
              sprintf(filename + nchars, "%3s", seas_name[index]);
              if (filesuffix[0]) sprintf(filename + nchars + 3, "%s", filesuffix);
            }
          else
            {
              size_t slen;
              char oformat[32];
              strcpy(oformat, "%02d");

              if (operatorID == SPLITMON && format)
                {
                  char sbuf[32];
                  auto stime = datetime_to_tm(vdate, vtime);
                  slen = strftime(sbuf, 32, format, &stime);

                  if (slen) strcpy(oformat, sbuf);
                }

              slen = sprintf(filename + nchars, oformat, index);
              if (filesuffix[0]) sprintf(filename + nchars + slen, "%s", filesuffix);
            }

          if (Options::cdoVerbose) cdoPrint("create file %s", filename);

          streamID2 = cdoOpenWrite(filename);
          cdoDefVlist(streamID2, vlistID2);
          streamIDs[index] = streamID2;
        }

      cdoDefTimestep(streamID2, tsIDs[index]);

      if (tsID > 0 && tsIDs[index] == 0 && nconst)
        {
          for (varID = 0; varID < nvars; varID++)
            {
              if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
                {
                  const auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
                  for (levelID = 0; levelID < nlevels; levelID++)
                    {
                      cdoDefRecord(streamID2, varID, levelID);
                      nmiss = vars[varID][levelID].nmiss;
                      cdoWriteRecord(streamID2, vars[varID][levelID].vec_d.data(), nmiss);
                    }
                }
            }
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoDefRecord(streamID2, varID, levelID);

          if (lcopy && !(tsID == 0 && nconst))
            {
              cdoCopyRecord(streamID2, streamID1);
            }
          else
            {
              cdoReadRecord(streamID1, array.data(), &nmiss);
              cdoWriteRecord(streamID2, array.data(), nmiss);

              if (tsID == 0 && nconst)
                {
                  if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
                    {
                      const auto gridID = vlistInqVarGrid(vlistID1, varID);
                      const auto gridsize = gridInqSize(gridID);
                      varrayCopy(gridsize, array, vars[varID][levelID].vec_d);
                      vars[varID][levelID].nmiss = nmiss;
                    }
                }
            }
        }

      tsIDs[index]++;
      tsID++;
    }

  cdoStreamClose(streamID1);

  for (index = 0; index < MaxStreams; index++)
    {
      if (streamIDs[index] != CDO_STREAM_UNDEF) cdoStreamClose(streamIDs[index]);
    }

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
