/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "process_int.h"
#include "readline.h"
#include "cdo_zaxis.h"

#define MAX_LINE_LEN 4096

void *
Setrcaname(void *process)
{
  const char *rcsnames;
  char line[MAX_LINE_LEN];
  char sname[CDI_MAX_NAME], sdescription[CDI_MAX_NAME], sunits[CDI_MAX_NAME];
  int scode, sltype, slevel;
  int zaxisID, ltype, code, nlev;
  int level;

  cdoInitialize(process);

  const auto lcopy = unchangedRecord();

  operatorInputArg("file name with RCA names");
  rcsnames = cdoOperatorArgv(0).c_str();

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto nvars = vlistNvars(vlistID2);

  auto fp = fopen(rcsnames, "r");
  if (fp != nullptr)
    {
      while (cdo::readline(fp, line, MAX_LINE_LEN))
        {
          sscanf(line, "%d\t%d\t%d\t%s\t%s\t%s", &scode, &sltype, &slevel, sname, sdescription, sunits);
          /*
          printf("%s\n", line);
          printf("%d:%d:%d:%s:%s:%s\n", scode, sltype, slevel, sname,
          sdescription, sunits);
          */
          for (int varID = 0; varID < nvars; varID++)
            {
              code = vlistInqVarCode(vlistID2, varID);
              zaxisID = vlistInqVarZaxis(vlistID2, varID);
              nlev = zaxisInqSize(zaxisID);

              ltype = zaxis2ltype(zaxisID);

              if (code == scode)
                {
                  if (ltype == 105)
                    {
                      if (nlev != 1)
                        {
                          cdoWarning("Number of levels should be 1 for level type 105!");
                          cdoWarning("Maybe environment variable SPLIT_LTYPE_105 is not set.");
                          continue;
                        }
                      level = (int) cdoZaxisInqLevel(zaxisID, 0);
                      if (sltype == 105 && slevel == level)
                        {
                          cdiDefKeyString(vlistID2, varID, CDI_KEY_NAME, sname);
                          cdiDefKeyString(vlistID2, varID, CDI_KEY_LONGNAME, sdescription);
                          cdiDefKeyString(vlistID2, varID, CDI_KEY_UNITS, sunits);
                          break;
                        }
                    }
                  else if (sltype != 105)
                    {
                      cdiDefKeyString(vlistID2, varID, CDI_KEY_NAME, sname);
                      cdiDefKeyString(vlistID2, varID, CDI_KEY_LONGNAME, sdescription);
                      cdiDefKeyString(vlistID2, varID, CDI_KEY_UNITS, sunits);
                      break;
                    }
                }
            }
        }

      fclose(fp);
    }
  else
    {
      perror(rcsnames);
    }

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);

  cdoDefVlist(streamID2, vlistID2);

  Field field;

  VarList varList1;
  varListInit(varList1, vlistID1);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoDefRecord(streamID2, varID, levelID);

          if (lcopy)
            {
              cdoCopyRecord(streamID2, streamID1);
            }
          else
            {
              field.init(varList1[varID]);
              cdoReadRecord(streamID1, field);
              cdoWriteRecord(streamID2, field);
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
