# Test file mode strings from Solaris /usr/ucb/ls (Berkeley compatibility).

use strict;
use warnings;
use Test::More;
plan tests => 4096;

use File::Listing;

while (my $line = <DATA>) {
    chomp $line;

    # Expected output value.

    my $expected = $. - 1;

    # This version of `ls' does not show whether the sticky bit (file mode bit
    # 01000 ) is set, so remove it from the expected output.

    $expected &= 06777;

    # Information text.

    my $text = sprintf('"%s" -> "%05o"', $line, $expected);

    # Get output and keep only permission (no file type info).

    my $got = File::Listing::file_mode($line);
    $got &= 07777;

    cmp_ok($got, '==', $expected, $text);
}

__DATA__
----------
---------x
--------w-
--------wx
-------r--
-------r-x
-------rw-
-------rwx
------x---
------x--x
------x-w-
------x-wx
------xr--
------xr-x
------xrw-
------xrwx
-----w----
-----w---x
-----w--w-
-----w--wx
-----w-r--
-----w-r-x
-----w-rw-
-----w-rwx
-----wx---
-----wx--x
-----wx-w-
-----wx-wx
-----wxr--
-----wxr-x
-----wxrw-
-----wxrwx
----r-----
----r----x
----r---w-
----r---wx
----r--r--
----r--r-x
----r--rw-
----r--rwx
----r-x---
----r-x--x
----r-x-w-
----r-x-wx
----r-xr--
----r-xr-x
----r-xrw-
----r-xrwx
----rw----
----rw---x
----rw--w-
----rw--wx
----rw-r--
----rw-r-x
----rw-rw-
----rw-rwx
----rwx---
----rwx--x
----rwx-w-
----rwx-wx
----rwxr--
----rwxr-x
----rwxrw-
----rwxrwx
---x------
---x-----x
---x----w-
---x----wx
---x---r--
---x---r-x
---x---rw-
---x---rwx
---x--x---
---x--x--x
---x--x-w-
---x--x-wx
---x--xr--
---x--xr-x
---x--xrw-
---x--xrwx
---x-w----
---x-w---x
---x-w--w-
---x-w--wx
---x-w-r--
---x-w-r-x
---x-w-rw-
---x-w-rwx
---x-wx---
---x-wx--x
---x-wx-w-
---x-wx-wx
---x-wxr--
---x-wxr-x
---x-wxrw-
---x-wxrwx
---xr-----
---xr----x
---xr---w-
---xr---wx
---xr--r--
---xr--r-x
---xr--rw-
---xr--rwx
---xr-x---
---xr-x--x
---xr-x-w-
---xr-x-wx
---xr-xr--
---xr-xr-x
---xr-xrw-
---xr-xrwx
---xrw----
---xrw---x
---xrw--w-
---xrw--wx
---xrw-r--
---xrw-r-x
---xrw-rw-
---xrw-rwx
---xrwx---
---xrwx--x
---xrwx-w-
---xrwx-wx
---xrwxr--
---xrwxr-x
---xrwxrw-
---xrwxrwx
--w-------
--w------x
--w-----w-
--w-----wx
--w----r--
--w----r-x
--w----rw-
--w----rwx
--w---x---
--w---x--x
--w---x-w-
--w---x-wx
--w---xr--
--w---xr-x
--w---xrw-
--w---xrwx
--w--w----
--w--w---x
--w--w--w-
--w--w--wx
--w--w-r--
--w--w-r-x
--w--w-rw-
--w--w-rwx
--w--wx---
--w--wx--x
--w--wx-w-
--w--wx-wx
--w--wxr--
--w--wxr-x
--w--wxrw-
--w--wxrwx
--w-r-----
--w-r----x
--w-r---w-
--w-r---wx
--w-r--r--
--w-r--r-x
--w-r--rw-
--w-r--rwx
--w-r-x---
--w-r-x--x
--w-r-x-w-
--w-r-x-wx
--w-r-xr--
--w-r-xr-x
--w-r-xrw-
--w-r-xrwx
--w-rw----
--w-rw---x
--w-rw--w-
--w-rw--wx
--w-rw-r--
--w-rw-r-x
--w-rw-rw-
--w-rw-rwx
--w-rwx---
--w-rwx--x
--w-rwx-w-
--w-rwx-wx
--w-rwxr--
--w-rwxr-x
--w-rwxrw-
--w-rwxrwx
--wx------
--wx-----x
--wx----w-
--wx----wx
--wx---r--
--wx---r-x
--wx---rw-
--wx---rwx
--wx--x---
--wx--x--x
--wx--x-w-
--wx--x-wx
--wx--xr--
--wx--xr-x
--wx--xrw-
--wx--xrwx
--wx-w----
--wx-w---x
--wx-w--w-
--wx-w--wx
--wx-w-r--
--wx-w-r-x
--wx-w-rw-
--wx-w-rwx
--wx-wx---
--wx-wx--x
--wx-wx-w-
--wx-wx-wx
--wx-wxr--
--wx-wxr-x
--wx-wxrw-
--wx-wxrwx
--wxr-----
--wxr----x
--wxr---w-
--wxr---wx
--wxr--r--
--wxr--r-x
--wxr--rw-
--wxr--rwx
--wxr-x---
--wxr-x--x
--wxr-x-w-
--wxr-x-wx
--wxr-xr--
--wxr-xr-x
--wxr-xrw-
--wxr-xrwx
--wxrw----
--wxrw---x
--wxrw--w-
--wxrw--wx
--wxrw-r--
--wxrw-r-x
--wxrw-rw-
--wxrw-rwx
--wxrwx---
--wxrwx--x
--wxrwx-w-
--wxrwx-wx
--wxrwxr--
--wxrwxr-x
--wxrwxrw-
--wxrwxrwx
-r--------
-r-------x
-r------w-
-r------wx
-r-----r--
-r-----r-x
-r-----rw-
-r-----rwx
-r----x---
-r----x--x
-r----x-w-
-r----x-wx
-r----xr--
-r----xr-x
-r----xrw-
-r----xrwx
-r---w----
-r---w---x
-r---w--w-
-r---w--wx
-r---w-r--
-r---w-r-x
-r---w-rw-
-r---w-rwx
-r---wx---
-r---wx--x
-r---wx-w-
-r---wx-wx
-r---wxr--
-r---wxr-x
-r---wxrw-
-r---wxrwx
-r--r-----
-r--r----x
-r--r---w-
-r--r---wx
-r--r--r--
-r--r--r-x
-r--r--rw-
-r--r--rwx
-r--r-x---
-r--r-x--x
-r--r-x-w-
-r--r-x-wx
-r--r-xr--
-r--r-xr-x
-r--r-xrw-
-r--r-xrwx
-r--rw----
-r--rw---x
-r--rw--w-
-r--rw--wx
-r--rw-r--
-r--rw-r-x
-r--rw-rw-
-r--rw-rwx
-r--rwx---
-r--rwx--x
-r--rwx-w-
-r--rwx-wx
-r--rwxr--
-r--rwxr-x
-r--rwxrw-
-r--rwxrwx
-r-x------
-r-x-----x
-r-x----w-
-r-x----wx
-r-x---r--
-r-x---r-x
-r-x---rw-
-r-x---rwx
-r-x--x---
-r-x--x--x
-r-x--x-w-
-r-x--x-wx
-r-x--xr--
-r-x--xr-x
-r-x--xrw-
-r-x--xrwx
-r-x-w----
-r-x-w---x
-r-x-w--w-
-r-x-w--wx
-r-x-w-r--
-r-x-w-r-x
-r-x-w-rw-
-r-x-w-rwx
-r-x-wx---
-r-x-wx--x
-r-x-wx-w-
-r-x-wx-wx
-r-x-wxr--
-r-x-wxr-x
-r-x-wxrw-
-r-x-wxrwx
-r-xr-----
-r-xr----x
-r-xr---w-
-r-xr---wx
-r-xr--r--
-r-xr--r-x
-r-xr--rw-
-r-xr--rwx
-r-xr-x---
-r-xr-x--x
-r-xr-x-w-
-r-xr-x-wx
-r-xr-xr--
-r-xr-xr-x
-r-xr-xrw-
-r-xr-xrwx
-r-xrw----
-r-xrw---x
-r-xrw--w-
-r-xrw--wx
-r-xrw-r--
-r-xrw-r-x
-r-xrw-rw-
-r-xrw-rwx
-r-xrwx---
-r-xrwx--x
-r-xrwx-w-
-r-xrwx-wx
-r-xrwxr--
-r-xrwxr-x
-r-xrwxrw-
-r-xrwxrwx
-rw-------
-rw------x
-rw-----w-
-rw-----wx
-rw----r--
-rw----r-x
-rw----rw-
-rw----rwx
-rw---x---
-rw---x--x
-rw---x-w-
-rw---x-wx
-rw---xr--
-rw---xr-x
-rw---xrw-
-rw---xrwx
-rw--w----
-rw--w---x
-rw--w--w-
-rw--w--wx
-rw--w-r--
-rw--w-r-x
-rw--w-rw-
-rw--w-rwx
-rw--wx---
-rw--wx--x
-rw--wx-w-
-rw--wx-wx
-rw--wxr--
-rw--wxr-x
-rw--wxrw-
-rw--wxrwx
-rw-r-----
-rw-r----x
-rw-r---w-
-rw-r---wx
-rw-r--r--
-rw-r--r-x
-rw-r--rw-
-rw-r--rwx
-rw-r-x---
-rw-r-x--x
-rw-r-x-w-
-rw-r-x-wx
-rw-r-xr--
-rw-r-xr-x
-rw-r-xrw-
-rw-r-xrwx
-rw-rw----
-rw-rw---x
-rw-rw--w-
-rw-rw--wx
-rw-rw-r--
-rw-rw-r-x
-rw-rw-rw-
-rw-rw-rwx
-rw-rwx---
-rw-rwx--x
-rw-rwx-w-
-rw-rwx-wx
-rw-rwxr--
-rw-rwxr-x
-rw-rwxrw-
-rw-rwxrwx
-rwx------
-rwx-----x
-rwx----w-
-rwx----wx
-rwx---r--
-rwx---r-x
-rwx---rw-
-rwx---rwx
-rwx--x---
-rwx--x--x
-rwx--x-w-
-rwx--x-wx
-rwx--xr--
-rwx--xr-x
-rwx--xrw-
-rwx--xrwx
-rwx-w----
-rwx-w---x
-rwx-w--w-
-rwx-w--wx
-rwx-w-r--
-rwx-w-r-x
-rwx-w-rw-
-rwx-w-rwx
-rwx-wx---
-rwx-wx--x
-rwx-wx-w-
-rwx-wx-wx
-rwx-wxr--
-rwx-wxr-x
-rwx-wxrw-
-rwx-wxrwx
-rwxr-----
-rwxr----x
-rwxr---w-
-rwxr---wx
-rwxr--r--
-rwxr--r-x
-rwxr--rw-
-rwxr--rwx
-rwxr-x---
-rwxr-x--x
-rwxr-x-w-
-rwxr-x-wx
-rwxr-xr--
-rwxr-xr-x
-rwxr-xrw-
-rwxr-xrwx
-rwxrw----
-rwxrw---x
-rwxrw--w-
-rwxrw--wx
-rwxrw-r--
-rwxrw-r-x
-rwxrw-rw-
-rwxrw-rwx
-rwxrwx---
-rwxrwx--x
-rwxrwx-w-
-rwxrwx-wx
-rwxrwxr--
-rwxrwxr-x
-rwxrwxrw-
-rwxrwxrwx
----------
---------x
--------w-
--------wx
-------r--
-------r-x
-------rw-
-------rwx
------x---
------x--x
------x-w-
------x-wx
------xr--
------xr-x
------xrw-
------xrwx
-----w----
-----w---x
-----w--w-
-----w--wx
-----w-r--
-----w-r-x
-----w-rw-
-----w-rwx
-----wx---
-----wx--x
-----wx-w-
-----wx-wx
-----wxr--
-----wxr-x
-----wxrw-
-----wxrwx
----r-----
----r----x
----r---w-
----r---wx
----r--r--
----r--r-x
----r--rw-
----r--rwx
----r-x---
----r-x--x
----r-x-w-
----r-x-wx
----r-xr--
----r-xr-x
----r-xrw-
----r-xrwx
----rw----
----rw---x
----rw--w-
----rw--wx
----rw-r--
----rw-r-x
----rw-rw-
----rw-rwx
----rwx---
----rwx--x
----rwx-w-
----rwx-wx
----rwxr--
----rwxr-x
----rwxrw-
----rwxrwx
---x------
---x-----x
---x----w-
---x----wx
---x---r--
---x---r-x
---x---rw-
---x---rwx
---x--x---
---x--x--x
---x--x-w-
---x--x-wx
---x--xr--
---x--xr-x
---x--xrw-
---x--xrwx
---x-w----
---x-w---x
---x-w--w-
---x-w--wx
---x-w-r--
---x-w-r-x
---x-w-rw-
---x-w-rwx
---x-wx---
---x-wx--x
---x-wx-w-
---x-wx-wx
---x-wxr--
---x-wxr-x
---x-wxrw-
---x-wxrwx
---xr-----
---xr----x
---xr---w-
---xr---wx
---xr--r--
---xr--r-x
---xr--rw-
---xr--rwx
---xr-x---
---xr-x--x
---xr-x-w-
---xr-x-wx
---xr-xr--
---xr-xr-x
---xr-xrw-
---xr-xrwx
---xrw----
---xrw---x
---xrw--w-
---xrw--wx
---xrw-r--
---xrw-r-x
---xrw-rw-
---xrw-rwx
---xrwx---
---xrwx--x
---xrwx-w-
---xrwx-wx
---xrwxr--
---xrwxr-x
---xrwxrw-
---xrwxrwx
--w-------
--w------x
--w-----w-
--w-----wx
--w----r--
--w----r-x
--w----rw-
--w----rwx
--w---x---
--w---x--x
--w---x-w-
--w---x-wx
--w---xr--
--w---xr-x
--w---xrw-
--w---xrwx
--w--w----
--w--w---x
--w--w--w-
--w--w--wx
--w--w-r--
--w--w-r-x
--w--w-rw-
--w--w-rwx
--w--wx---
--w--wx--x
--w--wx-w-
--w--wx-wx
--w--wxr--
--w--wxr-x
--w--wxrw-
--w--wxrwx
--w-r-----
--w-r----x
--w-r---w-
--w-r---wx
--w-r--r--
--w-r--r-x
--w-r--rw-
--w-r--rwx
--w-r-x---
--w-r-x--x
--w-r-x-w-
--w-r-x-wx
--w-r-xr--
--w-r-xr-x
--w-r-xrw-
--w-r-xrwx
--w-rw----
--w-rw---x
--w-rw--w-
--w-rw--wx
--w-rw-r--
--w-rw-r-x
--w-rw-rw-
--w-rw-rwx
--w-rwx---
--w-rwx--x
--w-rwx-w-
--w-rwx-wx
--w-rwxr--
--w-rwxr-x
--w-rwxrw-
--w-rwxrwx
--wx------
--wx-----x
--wx----w-
--wx----wx
--wx---r--
--wx---r-x
--wx---rw-
--wx---rwx
--wx--x---
--wx--x--x
--wx--x-w-
--wx--x-wx
--wx--xr--
--wx--xr-x
--wx--xrw-
--wx--xrwx
--wx-w----
--wx-w---x
--wx-w--w-
--wx-w--wx
--wx-w-r--
--wx-w-r-x
--wx-w-rw-
--wx-w-rwx
--wx-wx---
--wx-wx--x
--wx-wx-w-
--wx-wx-wx
--wx-wxr--
--wx-wxr-x
--wx-wxrw-
--wx-wxrwx
--wxr-----
--wxr----x
--wxr---w-
--wxr---wx
--wxr--r--
--wxr--r-x
--wxr--rw-
--wxr--rwx
--wxr-x---
--wxr-x--x
--wxr-x-w-
--wxr-x-wx
--wxr-xr--
--wxr-xr-x
--wxr-xrw-
--wxr-xrwx
--wxrw----
--wxrw---x
--wxrw--w-
--wxrw--wx
--wxrw-r--
--wxrw-r-x
--wxrw-rw-
--wxrw-rwx
--wxrwx---
--wxrwx--x
--wxrwx-w-
--wxrwx-wx
--wxrwxr--
--wxrwxr-x
--wxrwxrw-
--wxrwxrwx
-r--------
-r-------x
-r------w-
-r------wx
-r-----r--
-r-----r-x
-r-----rw-
-r-----rwx
-r----x---
-r----x--x
-r----x-w-
-r----x-wx
-r----xr--
-r----xr-x
-r----xrw-
-r----xrwx
-r---w----
-r---w---x
-r---w--w-
-r---w--wx
-r---w-r--
-r---w-r-x
-r---w-rw-
-r---w-rwx
-r---wx---
-r---wx--x
-r---wx-w-
-r---wx-wx
-r---wxr--
-r---wxr-x
-r---wxrw-
-r---wxrwx
-r--r-----
-r--r----x
-r--r---w-
-r--r---wx
-r--r--r--
-r--r--r-x
-r--r--rw-
-r--r--rwx
-r--r-x---
-r--r-x--x
-r--r-x-w-
-r--r-x-wx
-r--r-xr--
-r--r-xr-x
-r--r-xrw-
-r--r-xrwx
-r--rw----
-r--rw---x
-r--rw--w-
-r--rw--wx
-r--rw-r--
-r--rw-r-x
-r--rw-rw-
-r--rw-rwx
-r--rwx---
-r--rwx--x
-r--rwx-w-
-r--rwx-wx
-r--rwxr--
-r--rwxr-x
-r--rwxrw-
-r--rwxrwx
-r-x------
-r-x-----x
-r-x----w-
-r-x----wx
-r-x---r--
-r-x---r-x
-r-x---rw-
-r-x---rwx
-r-x--x---
-r-x--x--x
-r-x--x-w-
-r-x--x-wx
-r-x--xr--
-r-x--xr-x
-r-x--xrw-
-r-x--xrwx
-r-x-w----
-r-x-w---x
-r-x-w--w-
-r-x-w--wx
-r-x-w-r--
-r-x-w-r-x
-r-x-w-rw-
-r-x-w-rwx
-r-x-wx---
-r-x-wx--x
-r-x-wx-w-
-r-x-wx-wx
-r-x-wxr--
-r-x-wxr-x
-r-x-wxrw-
-r-x-wxrwx
-r-xr-----
-r-xr----x
-r-xr---w-
-r-xr---wx
-r-xr--r--
-r-xr--r-x
-r-xr--rw-
-r-xr--rwx
-r-xr-x---
-r-xr-x--x
-r-xr-x-w-
-r-xr-x-wx
-r-xr-xr--
-r-xr-xr-x
-r-xr-xrw-
-r-xr-xrwx
-r-xrw----
-r-xrw---x
-r-xrw--w-
-r-xrw--wx
-r-xrw-r--
-r-xrw-r-x
-r-xrw-rw-
-r-xrw-rwx
-r-xrwx---
-r-xrwx--x
-r-xrwx-w-
-r-xrwx-wx
-r-xrwxr--
-r-xrwxr-x
-r-xrwxrw-
-r-xrwxrwx
-rw-------
-rw------x
-rw-----w-
-rw-----wx
-rw----r--
-rw----r-x
-rw----rw-
-rw----rwx
-rw---x---
-rw---x--x
-rw---x-w-
-rw---x-wx
-rw---xr--
-rw---xr-x
-rw---xrw-
-rw---xrwx
-rw--w----
-rw--w---x
-rw--w--w-
-rw--w--wx
-rw--w-r--
-rw--w-r-x
-rw--w-rw-
-rw--w-rwx
-rw--wx---
-rw--wx--x
-rw--wx-w-
-rw--wx-wx
-rw--wxr--
-rw--wxr-x
-rw--wxrw-
-rw--wxrwx
-rw-r-----
-rw-r----x
-rw-r---w-
-rw-r---wx
-rw-r--r--
-rw-r--r-x
-rw-r--rw-
-rw-r--rwx
-rw-r-x---
-rw-r-x--x
-rw-r-x-w-
-rw-r-x-wx
-rw-r-xr--
-rw-r-xr-x
-rw-r-xrw-
-rw-r-xrwx
-rw-rw----
-rw-rw---x
-rw-rw--w-
-rw-rw--wx
-rw-rw-r--
-rw-rw-r-x
-rw-rw-rw-
-rw-rw-rwx
-rw-rwx---
-rw-rwx--x
-rw-rwx-w-
-rw-rwx-wx
-rw-rwxr--
-rw-rwxr-x
-rw-rwxrw-
-rw-rwxrwx
-rwx------
-rwx-----x
-rwx----w-
-rwx----wx
-rwx---r--
-rwx---r-x
-rwx---rw-
-rwx---rwx
-rwx--x---
-rwx--x--x
-rwx--x-w-
-rwx--x-wx
-rwx--xr--
-rwx--xr-x
-rwx--xrw-
-rwx--xrwx
-rwx-w----
-rwx-w---x
-rwx-w--w-
-rwx-w--wx
-rwx-w-r--
-rwx-w-r-x
-rwx-w-rw-
-rwx-w-rwx
-rwx-wx---
-rwx-wx--x
-rwx-wx-w-
-rwx-wx-wx
-rwx-wxr--
-rwx-wxr-x
-rwx-wxrw-
-rwx-wxrwx
-rwxr-----
-rwxr----x
-rwxr---w-
-rwxr---wx
-rwxr--r--
-rwxr--r-x
-rwxr--rw-
-rwxr--rwx
-rwxr-x---
-rwxr-x--x
-rwxr-x-w-
-rwxr-x-wx
-rwxr-xr--
-rwxr-xr-x
-rwxr-xrw-
-rwxr-xrwx
-rwxrw----
-rwxrw---x
-rwxrw--w-
-rwxrw--wx
-rwxrw-r--
-rwxrw-r-x
-rwxrw-rw-
-rwxrw-rwx
-rwxrwx---
-rwxrwx--x
-rwxrwx-w-
-rwxrwx-wx
-rwxrwxr--
-rwxrwxr-x
-rwxrwxrw-
-rwxrwxrwx
------S---
------S--x
------S-w-
------S-wx
------Sr--
------Sr-x
------Srw-
------Srwx
------s---
------s--x
------s-w-
------s-wx
------sr--
------sr-x
------srw-
------srwx
-----wS---
-----wS--x
-----wS-w-
-----wS-wx
-----wSr--
-----wSr-x
-----wSrw-
-----wSrwx
-----ws---
-----ws--x
-----ws-w-
-----ws-wx
-----wsr--
-----wsr-x
-----wsrw-
-----wsrwx
----r-S---
----r-S--x
----r-S-w-
----r-S-wx
----r-Sr--
----r-Sr-x
----r-Srw-
----r-Srwx
----r-s---
----r-s--x
----r-s-w-
----r-s-wx
----r-sr--
----r-sr-x
----r-srw-
----r-srwx
----rwS---
----rwS--x
----rwS-w-
----rwS-wx
----rwSr--
----rwSr-x
----rwSrw-
----rwSrwx
----rws---
----rws--x
----rws-w-
----rws-wx
----rwsr--
----rwsr-x
----rwsrw-
----rwsrwx
---x--S---
---x--S--x
---x--S-w-
---x--S-wx
---x--Sr--
---x--Sr-x
---x--Srw-
---x--Srwx
---x--s---
---x--s--x
---x--s-w-
---x--s-wx
---x--sr--
---x--sr-x
---x--srw-
---x--srwx
---x-wS---
---x-wS--x
---x-wS-w-
---x-wS-wx
---x-wSr--
---x-wSr-x
---x-wSrw-
---x-wSrwx
---x-ws---
---x-ws--x
---x-ws-w-
---x-ws-wx
---x-wsr--
---x-wsr-x
---x-wsrw-
---x-wsrwx
---xr-S---
---xr-S--x
---xr-S-w-
---xr-S-wx
---xr-Sr--
---xr-Sr-x
---xr-Srw-
---xr-Srwx
---xr-s---
---xr-s--x
---xr-s-w-
---xr-s-wx
---xr-sr--
---xr-sr-x
---xr-srw-
---xr-srwx
---xrwS---
---xrwS--x
---xrwS-w-
---xrwS-wx
---xrwSr--
---xrwSr-x
---xrwSrw-
---xrwSrwx
---xrws---
---xrws--x
---xrws-w-
---xrws-wx
---xrwsr--
---xrwsr-x
---xrwsrw-
---xrwsrwx
--w---S---
--w---S--x
--w---S-w-
--w---S-wx
--w---Sr--
--w---Sr-x
--w---Srw-
--w---Srwx
--w---s---
--w---s--x
--w---s-w-
--w---s-wx
--w---sr--
--w---sr-x
--w---srw-
--w---srwx
--w--wS---
--w--wS--x
--w--wS-w-
--w--wS-wx
--w--wSr--
--w--wSr-x
--w--wSrw-
--w--wSrwx
--w--ws---
--w--ws--x
--w--ws-w-
--w--ws-wx
--w--wsr--
--w--wsr-x
--w--wsrw-
--w--wsrwx
--w-r-S---
--w-r-S--x
--w-r-S-w-
--w-r-S-wx
--w-r-Sr--
--w-r-Sr-x
--w-r-Srw-
--w-r-Srwx
--w-r-s---
--w-r-s--x
--w-r-s-w-
--w-r-s-wx
--w-r-sr--
--w-r-sr-x
--w-r-srw-
--w-r-srwx
--w-rwS---
--w-rwS--x
--w-rwS-w-
--w-rwS-wx
--w-rwSr--
--w-rwSr-x
--w-rwSrw-
--w-rwSrwx
--w-rws---
--w-rws--x
--w-rws-w-
--w-rws-wx
--w-rwsr--
--w-rwsr-x
--w-rwsrw-
--w-rwsrwx
--wx--S---
--wx--S--x
--wx--S-w-
--wx--S-wx
--wx--Sr--
--wx--Sr-x
--wx--Srw-
--wx--Srwx
--wx--s---
--wx--s--x
--wx--s-w-
--wx--s-wx
--wx--sr--
--wx--sr-x
--wx--srw-
--wx--srwx
--wx-wS---
--wx-wS--x
--wx-wS-w-
--wx-wS-wx
--wx-wSr--
--wx-wSr-x
--wx-wSrw-
--wx-wSrwx
--wx-ws---
--wx-ws--x
--wx-ws-w-
--wx-ws-wx
--wx-wsr--
--wx-wsr-x
--wx-wsrw-
--wx-wsrwx
--wxr-S---
--wxr-S--x
--wxr-S-w-
--wxr-S-wx
--wxr-Sr--
--wxr-Sr-x
--wxr-Srw-
--wxr-Srwx
--wxr-s---
--wxr-s--x
--wxr-s-w-
--wxr-s-wx
--wxr-sr--
--wxr-sr-x
--wxr-srw-
--wxr-srwx
--wxrwS---
--wxrwS--x
--wxrwS-w-
--wxrwS-wx
--wxrwSr--
--wxrwSr-x
--wxrwSrw-
--wxrwSrwx
--wxrws---
--wxrws--x
--wxrws-w-
--wxrws-wx
--wxrwsr--
--wxrwsr-x
--wxrwsrw-
--wxrwsrwx
-r----S---
-r----S--x
-r----S-w-
-r----S-wx
-r----Sr--
-r----Sr-x
-r----Srw-
-r----Srwx
-r----s---
-r----s--x
-r----s-w-
-r----s-wx
-r----sr--
-r----sr-x
-r----srw-
-r----srwx
-r---wS---
-r---wS--x
-r---wS-w-
-r---wS-wx
-r---wSr--
-r---wSr-x
-r---wSrw-
-r---wSrwx
-r---ws---
-r---ws--x
-r---ws-w-
-r---ws-wx
-r---wsr--
-r---wsr-x
-r---wsrw-
-r---wsrwx
-r--r-S---
-r--r-S--x
-r--r-S-w-
-r--r-S-wx
-r--r-Sr--
-r--r-Sr-x
-r--r-Srw-
-r--r-Srwx
-r--r-s---
-r--r-s--x
-r--r-s-w-
-r--r-s-wx
-r--r-sr--
-r--r-sr-x
-r--r-srw-
-r--r-srwx
-r--rwS---
-r--rwS--x
-r--rwS-w-
-r--rwS-wx
-r--rwSr--
-r--rwSr-x
-r--rwSrw-
-r--rwSrwx
-r--rws---
-r--rws--x
-r--rws-w-
-r--rws-wx
-r--rwsr--
-r--rwsr-x
-r--rwsrw-
-r--rwsrwx
-r-x--S---
-r-x--S--x
-r-x--S-w-
-r-x--S-wx
-r-x--Sr--
-r-x--Sr-x
-r-x--Srw-
-r-x--Srwx
-r-x--s---
-r-x--s--x
-r-x--s-w-
-r-x--s-wx
-r-x--sr--
-r-x--sr-x
-r-x--srw-
-r-x--srwx
-r-x-wS---
-r-x-wS--x
-r-x-wS-w-
-r-x-wS-wx
-r-x-wSr--
-r-x-wSr-x
-r-x-wSrw-
-r-x-wSrwx
-r-x-ws---
-r-x-ws--x
-r-x-ws-w-
-r-x-ws-wx
-r-x-wsr--
-r-x-wsr-x
-r-x-wsrw-
-r-x-wsrwx
-r-xr-S---
-r-xr-S--x
-r-xr-S-w-
-r-xr-S-wx
-r-xr-Sr--
-r-xr-Sr-x
-r-xr-Srw-
-r-xr-Srwx
-r-xr-s---
-r-xr-s--x
-r-xr-s-w-
-r-xr-s-wx
-r-xr-sr--
-r-xr-sr-x
-r-xr-srw-
-r-xr-srwx
-r-xrwS---
-r-xrwS--x
-r-xrwS-w-
-r-xrwS-wx
-r-xrwSr--
-r-xrwSr-x
-r-xrwSrw-
-r-xrwSrwx
-r-xrws---
-r-xrws--x
-r-xrws-w-
-r-xrws-wx
-r-xrwsr--
-r-xrwsr-x
-r-xrwsrw-
-r-xrwsrwx
-rw---S---
-rw---S--x
-rw---S-w-
-rw---S-wx
-rw---Sr--
-rw---Sr-x
-rw---Srw-
-rw---Srwx
-rw---s---
-rw---s--x
-rw---s-w-
-rw---s-wx
-rw---sr--
-rw---sr-x
-rw---srw-
-rw---srwx
-rw--wS---
-rw--wS--x
-rw--wS-w-
-rw--wS-wx
-rw--wSr--
-rw--wSr-x
-rw--wSrw-
-rw--wSrwx
-rw--ws---
-rw--ws--x
-rw--ws-w-
-rw--ws-wx
-rw--wsr--
-rw--wsr-x
-rw--wsrw-
-rw--wsrwx
-rw-r-S---
-rw-r-S--x
-rw-r-S-w-
-rw-r-S-wx
-rw-r-Sr--
-rw-r-Sr-x
-rw-r-Srw-
-rw-r-Srwx
-rw-r-s---
-rw-r-s--x
-rw-r-s-w-
-rw-r-s-wx
-rw-r-sr--
-rw-r-sr-x
-rw-r-srw-
-rw-r-srwx
-rw-rwS---
-rw-rwS--x
-rw-rwS-w-
-rw-rwS-wx
-rw-rwSr--
-rw-rwSr-x
-rw-rwSrw-
-rw-rwSrwx
-rw-rws---
-rw-rws--x
-rw-rws-w-
-rw-rws-wx
-rw-rwsr--
-rw-rwsr-x
-rw-rwsrw-
-rw-rwsrwx
-rwx--S---
-rwx--S--x
-rwx--S-w-
-rwx--S-wx
-rwx--Sr--
-rwx--Sr-x
-rwx--Srw-
-rwx--Srwx
-rwx--s---
-rwx--s--x
-rwx--s-w-
-rwx--s-wx
-rwx--sr--
-rwx--sr-x
-rwx--srw-
-rwx--srwx
-rwx-wS---
-rwx-wS--x
-rwx-wS-w-
-rwx-wS-wx
-rwx-wSr--
-rwx-wSr-x
-rwx-wSrw-
-rwx-wSrwx
-rwx-ws---
-rwx-ws--x
-rwx-ws-w-
-rwx-ws-wx
-rwx-wsr--
-rwx-wsr-x
-rwx-wsrw-
-rwx-wsrwx
-rwxr-S---
-rwxr-S--x
-rwxr-S-w-
-rwxr-S-wx
-rwxr-Sr--
-rwxr-Sr-x
-rwxr-Srw-
-rwxr-Srwx
-rwxr-s---
-rwxr-s--x
-rwxr-s-w-
-rwxr-s-wx
-rwxr-sr--
-rwxr-sr-x
-rwxr-srw-
-rwxr-srwx
-rwxrwS---
-rwxrwS--x
-rwxrwS-w-
-rwxrwS-wx
-rwxrwSr--
-rwxrwSr-x
-rwxrwSrw-
-rwxrwSrwx
-rwxrws---
-rwxrws--x
-rwxrws-w-
-rwxrws-wx
-rwxrwsr--
-rwxrwsr-x
-rwxrwsrw-
-rwxrwsrwx
------S---
------S--x
------S-w-
------S-wx
------Sr--
------Sr-x
------Srw-
------Srwx
------s---
------s--x
------s-w-
------s-wx
------sr--
------sr-x
------srw-
------srwx
-----wS---
-----wS--x
-----wS-w-
-----wS-wx
-----wSr--
-----wSr-x
-----wSrw-
-----wSrwx
-----ws---
-----ws--x
-----ws-w-
-----ws-wx
-----wsr--
-----wsr-x
-----wsrw-
-----wsrwx
----r-S---
----r-S--x
----r-S-w-
----r-S-wx
----r-Sr--
----r-Sr-x
----r-Srw-
----r-Srwx
----r-s---
----r-s--x
----r-s-w-
----r-s-wx
----r-sr--
----r-sr-x
----r-srw-
----r-srwx
----rwS---
----rwS--x
----rwS-w-
----rwS-wx
----rwSr--
----rwSr-x
----rwSrw-
----rwSrwx
----rws---
----rws--x
----rws-w-
----rws-wx
----rwsr--
----rwsr-x
----rwsrw-
----rwsrwx
---x--S---
---x--S--x
---x--S-w-
---x--S-wx
---x--Sr--
---x--Sr-x
---x--Srw-
---x--Srwx
---x--s---
---x--s--x
---x--s-w-
---x--s-wx
---x--sr--
---x--sr-x
---x--srw-
---x--srwx
---x-wS---
---x-wS--x
---x-wS-w-
---x-wS-wx
---x-wSr--
---x-wSr-x
---x-wSrw-
---x-wSrwx
---x-ws---
---x-ws--x
---x-ws-w-
---x-ws-wx
---x-wsr--
---x-wsr-x
---x-wsrw-
---x-wsrwx
---xr-S---
---xr-S--x
---xr-S-w-
---xr-S-wx
---xr-Sr--
---xr-Sr-x
---xr-Srw-
---xr-Srwx
---xr-s---
---xr-s--x
---xr-s-w-
---xr-s-wx
---xr-sr--
---xr-sr-x
---xr-srw-
---xr-srwx
---xrwS---
---xrwS--x
---xrwS-w-
---xrwS-wx
---xrwSr--
---xrwSr-x
---xrwSrw-
---xrwSrwx
---xrws---
---xrws--x
---xrws-w-
---xrws-wx
---xrwsr--
---xrwsr-x
---xrwsrw-
---xrwsrwx
--w---S---
--w---S--x
--w---S-w-
--w---S-wx
--w---Sr--
--w---Sr-x
--w---Srw-
--w---Srwx
--w---s---
--w---s--x
--w---s-w-
--w---s-wx
--w---sr--
--w---sr-x
--w---srw-
--w---srwx
--w--wS---
--w--wS--x
--w--wS-w-
--w--wS-wx
--w--wSr--
--w--wSr-x
--w--wSrw-
--w--wSrwx
--w--ws---
--w--ws--x
--w--ws-w-
--w--ws-wx
--w--wsr--
--w--wsr-x
--w--wsrw-
--w--wsrwx
--w-r-S---
--w-r-S--x
--w-r-S-w-
--w-r-S-wx
--w-r-Sr--
--w-r-Sr-x
--w-r-Srw-
--w-r-Srwx
--w-r-s---
--w-r-s--x
--w-r-s-w-
--w-r-s-wx
--w-r-sr--
--w-r-sr-x
--w-r-srw-
--w-r-srwx
--w-rwS---
--w-rwS--x
--w-rwS-w-
--w-rwS-wx
--w-rwSr--
--w-rwSr-x
--w-rwSrw-
--w-rwSrwx
--w-rws---
--w-rws--x
--w-rws-w-
--w-rws-wx
--w-rwsr--
--w-rwsr-x
--w-rwsrw-
--w-rwsrwx
--wx--S---
--wx--S--x
--wx--S-w-
--wx--S-wx
--wx--Sr--
--wx--Sr-x
--wx--Srw-
--wx--Srwx
--wx--s---
--wx--s--x
--wx--s-w-
--wx--s-wx
--wx--sr--
--wx--sr-x
--wx--srw-
--wx--srwx
--wx-wS---
--wx-wS--x
--wx-wS-w-
--wx-wS-wx
--wx-wSr--
--wx-wSr-x
--wx-wSrw-
--wx-wSrwx
--wx-ws---
--wx-ws--x
--wx-ws-w-
--wx-ws-wx
--wx-wsr--
--wx-wsr-x
--wx-wsrw-
--wx-wsrwx
--wxr-S---
--wxr-S--x
--wxr-S-w-
--wxr-S-wx
--wxr-Sr--
--wxr-Sr-x
--wxr-Srw-
--wxr-Srwx
--wxr-s---
--wxr-s--x
--wxr-s-w-
--wxr-s-wx
--wxr-sr--
--wxr-sr-x
--wxr-srw-
--wxr-srwx
--wxrwS---
--wxrwS--x
--wxrwS-w-
--wxrwS-wx
--wxrwSr--
--wxrwSr-x
--wxrwSrw-
--wxrwSrwx
--wxrws---
--wxrws--x
--wxrws-w-
--wxrws-wx
--wxrwsr--
--wxrwsr-x
--wxrwsrw-
--wxrwsrwx
-r----S---
-r----S--x
-r----S-w-
-r----S-wx
-r----Sr--
-r----Sr-x
-r----Srw-
-r----Srwx
-r----s---
-r----s--x
-r----s-w-
-r----s-wx
-r----sr--
-r----sr-x
-r----srw-
-r----srwx
-r---wS---
-r---wS--x
-r---wS-w-
-r---wS-wx
-r---wSr--
-r---wSr-x
-r---wSrw-
-r---wSrwx
-r---ws---
-r---ws--x
-r---ws-w-
-r---ws-wx
-r---wsr--
-r---wsr-x
-r---wsrw-
-r---wsrwx
-r--r-S---
-r--r-S--x
-r--r-S-w-
-r--r-S-wx
-r--r-Sr--
-r--r-Sr-x
-r--r-Srw-
-r--r-Srwx
-r--r-s---
-r--r-s--x
-r--r-s-w-
-r--r-s-wx
-r--r-sr--
-r--r-sr-x
-r--r-srw-
-r--r-srwx
-r--rwS---
-r--rwS--x
-r--rwS-w-
-r--rwS-wx
-r--rwSr--
-r--rwSr-x
-r--rwSrw-
-r--rwSrwx
-r--rws---
-r--rws--x
-r--rws-w-
-r--rws-wx
-r--rwsr--
-r--rwsr-x
-r--rwsrw-
-r--rwsrwx
-r-x--S---
-r-x--S--x
-r-x--S-w-
-r-x--S-wx
-r-x--Sr--
-r-x--Sr-x
-r-x--Srw-
-r-x--Srwx
-r-x--s---
-r-x--s--x
-r-x--s-w-
-r-x--s-wx
-r-x--sr--
-r-x--sr-x
-r-x--srw-
-r-x--srwx
-r-x-wS---
-r-x-wS--x
-r-x-wS-w-
-r-x-wS-wx
-r-x-wSr--
-r-x-wSr-x
-r-x-wSrw-
-r-x-wSrwx
-r-x-ws---
-r-x-ws--x
-r-x-ws-w-
-r-x-ws-wx
-r-x-wsr--
-r-x-wsr-x
-r-x-wsrw-
-r-x-wsrwx
-r-xr-S---
-r-xr-S--x
-r-xr-S-w-
-r-xr-S-wx
-r-xr-Sr--
-r-xr-Sr-x
-r-xr-Srw-
-r-xr-Srwx
-r-xr-s---
-r-xr-s--x
-r-xr-s-w-
-r-xr-s-wx
-r-xr-sr--
-r-xr-sr-x
-r-xr-srw-
-r-xr-srwx
-r-xrwS---
-r-xrwS--x
-r-xrwS-w-
-r-xrwS-wx
-r-xrwSr--
-r-xrwSr-x
-r-xrwSrw-
-r-xrwSrwx
-r-xrws---
-r-xrws--x
-r-xrws-w-
-r-xrws-wx
-r-xrwsr--
-r-xrwsr-x
-r-xrwsrw-
-r-xrwsrwx
-rw---S---
-rw---S--x
-rw---S-w-
-rw---S-wx
-rw---Sr--
-rw---Sr-x
-rw---Srw-
-rw---Srwx
-rw---s---
-rw---s--x
-rw---s-w-
-rw---s-wx
-rw---sr--
-rw---sr-x
-rw---srw-
-rw---srwx
-rw--wS---
-rw--wS--x
-rw--wS-w-
-rw--wS-wx
-rw--wSr--
-rw--wSr-x
-rw--wSrw-
-rw--wSrwx
-rw--ws---
-rw--ws--x
-rw--ws-w-
-rw--ws-wx
-rw--wsr--
-rw--wsr-x
-rw--wsrw-
-rw--wsrwx
-rw-r-S---
-rw-r-S--x
-rw-r-S-w-
-rw-r-S-wx
-rw-r-Sr--
-rw-r-Sr-x
-rw-r-Srw-
-rw-r-Srwx
-rw-r-s---
-rw-r-s--x
-rw-r-s-w-
-rw-r-s-wx
-rw-r-sr--
-rw-r-sr-x
-rw-r-srw-
-rw-r-srwx
-rw-rwS---
-rw-rwS--x
-rw-rwS-w-
-rw-rwS-wx
-rw-rwSr--
-rw-rwSr-x
-rw-rwSrw-
-rw-rwSrwx
-rw-rws---
-rw-rws--x
-rw-rws-w-
-rw-rws-wx
-rw-rwsr--
-rw-rwsr-x
-rw-rwsrw-
-rw-rwsrwx
-rwx--S---
-rwx--S--x
-rwx--S-w-
-rwx--S-wx
-rwx--Sr--
-rwx--Sr-x
-rwx--Srw-
-rwx--Srwx
-rwx--s---
-rwx--s--x
-rwx--s-w-
-rwx--s-wx
-rwx--sr--
-rwx--sr-x
-rwx--srw-
-rwx--srwx
-rwx-wS---
-rwx-wS--x
-rwx-wS-w-
-rwx-wS-wx
-rwx-wSr--
-rwx-wSr-x
-rwx-wSrw-
-rwx-wSrwx
-rwx-ws---
-rwx-ws--x
-rwx-ws-w-
-rwx-ws-wx
-rwx-wsr--
-rwx-wsr-x
-rwx-wsrw-
-rwx-wsrwx
-rwxr-S---
-rwxr-S--x
-rwxr-S-w-
-rwxr-S-wx
-rwxr-Sr--
-rwxr-Sr-x
-rwxr-Srw-
-rwxr-Srwx
-rwxr-s---
-rwxr-s--x
-rwxr-s-w-
-rwxr-s-wx
-rwxr-sr--
-rwxr-sr-x
-rwxr-srw-
-rwxr-srwx
-rwxrwS---
-rwxrwS--x
-rwxrwS-w-
-rwxrwS-wx
-rwxrwSr--
-rwxrwSr-x
-rwxrwSrw-
-rwxrwSrwx
-rwxrws---
-rwxrws--x
-rwxrws-w-
-rwxrws-wx
-rwxrwsr--
-rwxrwsr-x
-rwxrwsrw-
-rwxrwsrwx
---S------
---S-----x
---S----w-
---S----wx
---S---r--
---S---r-x
---S---rw-
---S---rwx
---S--x---
---S--x--x
---S--x-w-
---S--x-wx
---S--xr--
---S--xr-x
---S--xrw-
---S--xrwx
---S-w----
---S-w---x
---S-w--w-
---S-w--wx
---S-w-r--
---S-w-r-x
---S-w-rw-
---S-w-rwx
---S-wx---
---S-wx--x
---S-wx-w-
---S-wx-wx
---S-wxr--
---S-wxr-x
---S-wxrw-
---S-wxrwx
---Sr-----
---Sr----x
---Sr---w-
---Sr---wx
---Sr--r--
---Sr--r-x
---Sr--rw-
---Sr--rwx
---Sr-x---
---Sr-x--x
---Sr-x-w-
---Sr-x-wx
---Sr-xr--
---Sr-xr-x
---Sr-xrw-
---Sr-xrwx
---Srw----
---Srw---x
---Srw--w-
---Srw--wx
---Srw-r--
---Srw-r-x
---Srw-rw-
---Srw-rwx
---Srwx---
---Srwx--x
---Srwx-w-
---Srwx-wx
---Srwxr--
---Srwxr-x
---Srwxrw-
---Srwxrwx
---s------
---s-----x
---s----w-
---s----wx
---s---r--
---s---r-x
---s---rw-
---s---rwx
---s--x---
---s--x--x
---s--x-w-
---s--x-wx
---s--xr--
---s--xr-x
---s--xrw-
---s--xrwx
---s-w----
---s-w---x
---s-w--w-
---s-w--wx
---s-w-r--
---s-w-r-x
---s-w-rw-
---s-w-rwx
---s-wx---
---s-wx--x
---s-wx-w-
---s-wx-wx
---s-wxr--
---s-wxr-x
---s-wxrw-
---s-wxrwx
---sr-----
---sr----x
---sr---w-
---sr---wx
---sr--r--
---sr--r-x
---sr--rw-
---sr--rwx
---sr-x---
---sr-x--x
---sr-x-w-
---sr-x-wx
---sr-xr--
---sr-xr-x
---sr-xrw-
---sr-xrwx
---srw----
---srw---x
---srw--w-
---srw--wx
---srw-r--
---srw-r-x
---srw-rw-
---srw-rwx
---srwx---
---srwx--x
---srwx-w-
---srwx-wx
---srwxr--
---srwxr-x
---srwxrw-
---srwxrwx
--wS------
--wS-----x
--wS----w-
--wS----wx
--wS---r--
--wS---r-x
--wS---rw-
--wS---rwx
--wS--x---
--wS--x--x
--wS--x-w-
--wS--x-wx
--wS--xr--
--wS--xr-x
--wS--xrw-
--wS--xrwx
--wS-w----
--wS-w---x
--wS-w--w-
--wS-w--wx
--wS-w-r--
--wS-w-r-x
--wS-w-rw-
--wS-w-rwx
--wS-wx---
--wS-wx--x
--wS-wx-w-
--wS-wx-wx
--wS-wxr--
--wS-wxr-x
--wS-wxrw-
--wS-wxrwx
--wSr-----
--wSr----x
--wSr---w-
--wSr---wx
--wSr--r--
--wSr--r-x
--wSr--rw-
--wSr--rwx
--wSr-x---
--wSr-x--x
--wSr-x-w-
--wSr-x-wx
--wSr-xr--
--wSr-xr-x
--wSr-xrw-
--wSr-xrwx
--wSrw----
--wSrw---x
--wSrw--w-
--wSrw--wx
--wSrw-r--
--wSrw-r-x
--wSrw-rw-
--wSrw-rwx
--wSrwx---
--wSrwx--x
--wSrwx-w-
--wSrwx-wx
--wSrwxr--
--wSrwxr-x
--wSrwxrw-
--wSrwxrwx
--ws------
--ws-----x
--ws----w-
--ws----wx
--ws---r--
--ws---r-x
--ws---rw-
--ws---rwx
--ws--x---
--ws--x--x
--ws--x-w-
--ws--x-wx
--ws--xr--
--ws--xr-x
--ws--xrw-
--ws--xrwx
--ws-w----
--ws-w---x
--ws-w--w-
--ws-w--wx
--ws-w-r--
--ws-w-r-x
--ws-w-rw-
--ws-w-rwx
--ws-wx---
--ws-wx--x
--ws-wx-w-
--ws-wx-wx
--ws-wxr--
--ws-wxr-x
--ws-wxrw-
--ws-wxrwx
--wsr-----
--wsr----x
--wsr---w-
--wsr---wx
--wsr--r--
--wsr--r-x
--wsr--rw-
--wsr--rwx
--wsr-x---
--wsr-x--x
--wsr-x-w-
--wsr-x-wx
--wsr-xr--
--wsr-xr-x
--wsr-xrw-
--wsr-xrwx
--wsrw----
--wsrw---x
--wsrw--w-
--wsrw--wx
--wsrw-r--
--wsrw-r-x
--wsrw-rw-
--wsrw-rwx
--wsrwx---
--wsrwx--x
--wsrwx-w-
--wsrwx-wx
--wsrwxr--
--wsrwxr-x
--wsrwxrw-
--wsrwxrwx
-r-S------
-r-S-----x
-r-S----w-
-r-S----wx
-r-S---r--
-r-S---r-x
-r-S---rw-
-r-S---rwx
-r-S--x---
-r-S--x--x
-r-S--x-w-
-r-S--x-wx
-r-S--xr--
-r-S--xr-x
-r-S--xrw-
-r-S--xrwx
-r-S-w----
-r-S-w---x
-r-S-w--w-
-r-S-w--wx
-r-S-w-r--
-r-S-w-r-x
-r-S-w-rw-
-r-S-w-rwx
-r-S-wx---
-r-S-wx--x
-r-S-wx-w-
-r-S-wx-wx
-r-S-wxr--
-r-S-wxr-x
-r-S-wxrw-
-r-S-wxrwx
-r-Sr-----
-r-Sr----x
-r-Sr---w-
-r-Sr---wx
-r-Sr--r--
-r-Sr--r-x
-r-Sr--rw-
-r-Sr--rwx
-r-Sr-x---
-r-Sr-x--x
-r-Sr-x-w-
-r-Sr-x-wx
-r-Sr-xr--
-r-Sr-xr-x
-r-Sr-xrw-
-r-Sr-xrwx
-r-Srw----
-r-Srw---x
-r-Srw--w-
-r-Srw--wx
-r-Srw-r--
-r-Srw-r-x
-r-Srw-rw-
-r-Srw-rwx
-r-Srwx---
-r-Srwx--x
-r-Srwx-w-
-r-Srwx-wx
-r-Srwxr--
-r-Srwxr-x
-r-Srwxrw-
-r-Srwxrwx
-r-s------
-r-s-----x
-r-s----w-
-r-s----wx
-r-s---r--
-r-s---r-x
-r-s---rw-
-r-s---rwx
-r-s--x---
-r-s--x--x
-r-s--x-w-
-r-s--x-wx
-r-s--xr--
-r-s--xr-x
-r-s--xrw-
-r-s--xrwx
-r-s-w----
-r-s-w---x
-r-s-w--w-
-r-s-w--wx
-r-s-w-r--
-r-s-w-r-x
-r-s-w-rw-
-r-s-w-rwx
-r-s-wx---
-r-s-wx--x
-r-s-wx-w-
-r-s-wx-wx
-r-s-wxr--
-r-s-wxr-x
-r-s-wxrw-
-r-s-wxrwx
-r-sr-----
-r-sr----x
-r-sr---w-
-r-sr---wx
-r-sr--r--
-r-sr--r-x
-r-sr--rw-
-r-sr--rwx
-r-sr-x---
-r-sr-x--x
-r-sr-x-w-
-r-sr-x-wx
-r-sr-xr--
-r-sr-xr-x
-r-sr-xrw-
-r-sr-xrwx
-r-srw----
-r-srw---x
-r-srw--w-
-r-srw--wx
-r-srw-r--
-r-srw-r-x
-r-srw-rw-
-r-srw-rwx
-r-srwx---
-r-srwx--x
-r-srwx-w-
-r-srwx-wx
-r-srwxr--
-r-srwxr-x
-r-srwxrw-
-r-srwxrwx
-rwS------
-rwS-----x
-rwS----w-
-rwS----wx
-rwS---r--
-rwS---r-x
-rwS---rw-
-rwS---rwx
-rwS--x---
-rwS--x--x
-rwS--x-w-
-rwS--x-wx
-rwS--xr--
-rwS--xr-x
-rwS--xrw-
-rwS--xrwx
-rwS-w----
-rwS-w---x
-rwS-w--w-
-rwS-w--wx
-rwS-w-r--
-rwS-w-r-x
-rwS-w-rw-
-rwS-w-rwx
-rwS-wx---
-rwS-wx--x
-rwS-wx-w-
-rwS-wx-wx
-rwS-wxr--
-rwS-wxr-x
-rwS-wxrw-
-rwS-wxrwx
-rwSr-----
-rwSr----x
-rwSr---w-
-rwSr---wx
-rwSr--r--
-rwSr--r-x
-rwSr--rw-
-rwSr--rwx
-rwSr-x---
-rwSr-x--x
-rwSr-x-w-
-rwSr-x-wx
-rwSr-xr--
-rwSr-xr-x
-rwSr-xrw-
-rwSr-xrwx
-rwSrw----
-rwSrw---x
-rwSrw--w-
-rwSrw--wx
-rwSrw-r--
-rwSrw-r-x
-rwSrw-rw-
-rwSrw-rwx
-rwSrwx---
-rwSrwx--x
-rwSrwx-w-
-rwSrwx-wx
-rwSrwxr--
-rwSrwxr-x
-rwSrwxrw-
-rwSrwxrwx
-rws------
-rws-----x
-rws----w-
-rws----wx
-rws---r--
-rws---r-x
-rws---rw-
-rws---rwx
-rws--x---
-rws--x--x
-rws--x-w-
-rws--x-wx
-rws--xr--
-rws--xr-x
-rws--xrw-
-rws--xrwx
-rws-w----
-rws-w---x
-rws-w--w-
-rws-w--wx
-rws-w-r--
-rws-w-r-x
-rws-w-rw-
-rws-w-rwx
-rws-wx---
-rws-wx--x
-rws-wx-w-
-rws-wx-wx
-rws-wxr--
-rws-wxr-x
-rws-wxrw-
-rws-wxrwx
-rwsr-----
-rwsr----x
-rwsr---w-
-rwsr---wx
-rwsr--r--
-rwsr--r-x
-rwsr--rw-
-rwsr--rwx
-rwsr-x---
-rwsr-x--x
-rwsr-x-w-
-rwsr-x-wx
-rwsr-xr--
-rwsr-xr-x
-rwsr-xrw-
-rwsr-xrwx
-rwsrw----
-rwsrw---x
-rwsrw--w-
-rwsrw--wx
-rwsrw-r--
-rwsrw-r-x
-rwsrw-rw-
-rwsrw-rwx
-rwsrwx---
-rwsrwx--x
-rwsrwx-w-
-rwsrwx-wx
-rwsrwxr--
-rwsrwxr-x
-rwsrwxrw-
-rwsrwxrwx
---S------
---S-----x
---S----w-
---S----wx
---S---r--
---S---r-x
---S---rw-
---S---rwx
---S--x---
---S--x--x
---S--x-w-
---S--x-wx
---S--xr--
---S--xr-x
---S--xrw-
---S--xrwx
---S-w----
---S-w---x
---S-w--w-
---S-w--wx
---S-w-r--
---S-w-r-x
---S-w-rw-
---S-w-rwx
---S-wx---
---S-wx--x
---S-wx-w-
---S-wx-wx
---S-wxr--
---S-wxr-x
---S-wxrw-
---S-wxrwx
---Sr-----
---Sr----x
---Sr---w-
---Sr---wx
---Sr--r--
---Sr--r-x
---Sr--rw-
---Sr--rwx
---Sr-x---
---Sr-x--x
---Sr-x-w-
---Sr-x-wx
---Sr-xr--
---Sr-xr-x
---Sr-xrw-
---Sr-xrwx
---Srw----
---Srw---x
---Srw--w-
---Srw--wx
---Srw-r--
---Srw-r-x
---Srw-rw-
---Srw-rwx
---Srwx---
---Srwx--x
---Srwx-w-
---Srwx-wx
---Srwxr--
---Srwxr-x
---Srwxrw-
---Srwxrwx
---s------
---s-----x
---s----w-
---s----wx
---s---r--
---s---r-x
---s---rw-
---s---rwx
---s--x---
---s--x--x
---s--x-w-
---s--x-wx
---s--xr--
---s--xr-x
---s--xrw-
---s--xrwx
---s-w----
---s-w---x
---s-w--w-
---s-w--wx
---s-w-r--
---s-w-r-x
---s-w-rw-
---s-w-rwx
---s-wx---
---s-wx--x
---s-wx-w-
---s-wx-wx
---s-wxr--
---s-wxr-x
---s-wxrw-
---s-wxrwx
---sr-----
---sr----x
---sr---w-
---sr---wx
---sr--r--
---sr--r-x
---sr--rw-
---sr--rwx
---sr-x---
---sr-x--x
---sr-x-w-
---sr-x-wx
---sr-xr--
---sr-xr-x
---sr-xrw-
---sr-xrwx
---srw----
---srw---x
---srw--w-
---srw--wx
---srw-r--
---srw-r-x
---srw-rw-
---srw-rwx
---srwx---
---srwx--x
---srwx-w-
---srwx-wx
---srwxr--
---srwxr-x
---srwxrw-
---srwxrwx
--wS------
--wS-----x
--wS----w-
--wS----wx
--wS---r--
--wS---r-x
--wS---rw-
--wS---rwx
--wS--x---
--wS--x--x
--wS--x-w-
--wS--x-wx
--wS--xr--
--wS--xr-x
--wS--xrw-
--wS--xrwx
--wS-w----
--wS-w---x
--wS-w--w-
--wS-w--wx
--wS-w-r--
--wS-w-r-x
--wS-w-rw-
--wS-w-rwx
--wS-wx---
--wS-wx--x
--wS-wx-w-
--wS-wx-wx
--wS-wxr--
--wS-wxr-x
--wS-wxrw-
--wS-wxrwx
--wSr-----
--wSr----x
--wSr---w-
--wSr---wx
--wSr--r--
--wSr--r-x
--wSr--rw-
--wSr--rwx
--wSr-x---
--wSr-x--x
--wSr-x-w-
--wSr-x-wx
--wSr-xr--
--wSr-xr-x
--wSr-xrw-
--wSr-xrwx
--wSrw----
--wSrw---x
--wSrw--w-
--wSrw--wx
--wSrw-r--
--wSrw-r-x
--wSrw-rw-
--wSrw-rwx
--wSrwx---
--wSrwx--x
--wSrwx-w-
--wSrwx-wx
--wSrwxr--
--wSrwxr-x
--wSrwxrw-
--wSrwxrwx
--ws------
--ws-----x
--ws----w-
--ws----wx
--ws---r--
--ws---r-x
--ws---rw-
--ws---rwx
--ws--x---
--ws--x--x
--ws--x-w-
--ws--x-wx
--ws--xr--
--ws--xr-x
--ws--xrw-
--ws--xrwx
--ws-w----
--ws-w---x
--ws-w--w-
--ws-w--wx
--ws-w-r--
--ws-w-r-x
--ws-w-rw-
--ws-w-rwx
--ws-wx---
--ws-wx--x
--ws-wx-w-
--ws-wx-wx
--ws-wxr--
--ws-wxr-x
--ws-wxrw-
--ws-wxrwx
--wsr-----
--wsr----x
--wsr---w-
--wsr---wx
--wsr--r--
--wsr--r-x
--wsr--rw-
--wsr--rwx
--wsr-x---
--wsr-x--x
--wsr-x-w-
--wsr-x-wx
--wsr-xr--
--wsr-xr-x
--wsr-xrw-
--wsr-xrwx
--wsrw----
--wsrw---x
--wsrw--w-
--wsrw--wx
--wsrw-r--
--wsrw-r-x
--wsrw-rw-
--wsrw-rwx
--wsrwx---
--wsrwx--x
--wsrwx-w-
--wsrwx-wx
--wsrwxr--
--wsrwxr-x
--wsrwxrw-
--wsrwxrwx
-r-S------
-r-S-----x
-r-S----w-
-r-S----wx
-r-S---r--
-r-S---r-x
-r-S---rw-
-r-S---rwx
-r-S--x---
-r-S--x--x
-r-S--x-w-
-r-S--x-wx
-r-S--xr--
-r-S--xr-x
-r-S--xrw-
-r-S--xrwx
-r-S-w----
-r-S-w---x
-r-S-w--w-
-r-S-w--wx
-r-S-w-r--
-r-S-w-r-x
-r-S-w-rw-
-r-S-w-rwx
-r-S-wx---
-r-S-wx--x
-r-S-wx-w-
-r-S-wx-wx
-r-S-wxr--
-r-S-wxr-x
-r-S-wxrw-
-r-S-wxrwx
-r-Sr-----
-r-Sr----x
-r-Sr---w-
-r-Sr---wx
-r-Sr--r--
-r-Sr--r-x
-r-Sr--rw-
-r-Sr--rwx
-r-Sr-x---
-r-Sr-x--x
-r-Sr-x-w-
-r-Sr-x-wx
-r-Sr-xr--
-r-Sr-xr-x
-r-Sr-xrw-
-r-Sr-xrwx
-r-Srw----
-r-Srw---x
-r-Srw--w-
-r-Srw--wx
-r-Srw-r--
-r-Srw-r-x
-r-Srw-rw-
-r-Srw-rwx
-r-Srwx---
-r-Srwx--x
-r-Srwx-w-
-r-Srwx-wx
-r-Srwxr--
-r-Srwxr-x
-r-Srwxrw-
-r-Srwxrwx
-r-s------
-r-s-----x
-r-s----w-
-r-s----wx
-r-s---r--
-r-s---r-x
-r-s---rw-
-r-s---rwx
-r-s--x---
-r-s--x--x
-r-s--x-w-
-r-s--x-wx
-r-s--xr--
-r-s--xr-x
-r-s--xrw-
-r-s--xrwx
-r-s-w----
-r-s-w---x
-r-s-w--w-
-r-s-w--wx
-r-s-w-r--
-r-s-w-r-x
-r-s-w-rw-
-r-s-w-rwx
-r-s-wx---
-r-s-wx--x
-r-s-wx-w-
-r-s-wx-wx
-r-s-wxr--
-r-s-wxr-x
-r-s-wxrw-
-r-s-wxrwx
-r-sr-----
-r-sr----x
-r-sr---w-
-r-sr---wx
-r-sr--r--
-r-sr--r-x
-r-sr--rw-
-r-sr--rwx
-r-sr-x---
-r-sr-x--x
-r-sr-x-w-
-r-sr-x-wx
-r-sr-xr--
-r-sr-xr-x
-r-sr-xrw-
-r-sr-xrwx
-r-srw----
-r-srw---x
-r-srw--w-
-r-srw--wx
-r-srw-r--
-r-srw-r-x
-r-srw-rw-
-r-srw-rwx
-r-srwx---
-r-srwx--x
-r-srwx-w-
-r-srwx-wx
-r-srwxr--
-r-srwxr-x
-r-srwxrw-
-r-srwxrwx
-rwS------
-rwS-----x
-rwS----w-
-rwS----wx
-rwS---r--
-rwS---r-x
-rwS---rw-
-rwS---rwx
-rwS--x---
-rwS--x--x
-rwS--x-w-
-rwS--x-wx
-rwS--xr--
-rwS--xr-x
-rwS--xrw-
-rwS--xrwx
-rwS-w----
-rwS-w---x
-rwS-w--w-
-rwS-w--wx
-rwS-w-r--
-rwS-w-r-x
-rwS-w-rw-
-rwS-w-rwx
-rwS-wx---
-rwS-wx--x
-rwS-wx-w-
-rwS-wx-wx
-rwS-wxr--
-rwS-wxr-x
-rwS-wxrw-
-rwS-wxrwx
-rwSr-----
-rwSr----x
-rwSr---w-
-rwSr---wx
-rwSr--r--
-rwSr--r-x
-rwSr--rw-
-rwSr--rwx
-rwSr-x---
-rwSr-x--x
-rwSr-x-w-
-rwSr-x-wx
-rwSr-xr--
-rwSr-xr-x
-rwSr-xrw-
-rwSr-xrwx
-rwSrw----
-rwSrw---x
-rwSrw--w-
-rwSrw--wx
-rwSrw-r--
-rwSrw-r-x
-rwSrw-rw-
-rwSrw-rwx
-rwSrwx---
-rwSrwx--x
-rwSrwx-w-
-rwSrwx-wx
-rwSrwxr--
-rwSrwxr-x
-rwSrwxrw-
-rwSrwxrwx
-rws------
-rws-----x
-rws----w-
-rws----wx
-rws---r--
-rws---r-x
-rws---rw-
-rws---rwx
-rws--x---
-rws--x--x
-rws--x-w-
-rws--x-wx
-rws--xr--
-rws--xr-x
-rws--xrw-
-rws--xrwx
-rws-w----
-rws-w---x
-rws-w--w-
-rws-w--wx
-rws-w-r--
-rws-w-r-x
-rws-w-rw-
-rws-w-rwx
-rws-wx---
-rws-wx--x
-rws-wx-w-
-rws-wx-wx
-rws-wxr--
-rws-wxr-x
-rws-wxrw-
-rws-wxrwx
-rwsr-----
-rwsr----x
-rwsr---w-
-rwsr---wx
-rwsr--r--
-rwsr--r-x
-rwsr--rw-
-rwsr--rwx
-rwsr-x---
-rwsr-x--x
-rwsr-x-w-
-rwsr-x-wx
-rwsr-xr--
-rwsr-xr-x
-rwsr-xrw-
-rwsr-xrwx
-rwsrw----
-rwsrw---x
-rwsrw--w-
-rwsrw--wx
-rwsrw-r--
-rwsrw-r-x
-rwsrw-rw-
-rwsrw-rwx
-rwsrwx---
-rwsrwx--x
-rwsrwx-w-
-rwsrwx-wx
-rwsrwxr--
-rwsrwxr-x
-rwsrwxrw-
-rwsrwxrwx
---S--S---
---S--S--x
---S--S-w-
---S--S-wx
---S--Sr--
---S--Sr-x
---S--Srw-
---S--Srwx
---S--s---
---S--s--x
---S--s-w-
---S--s-wx
---S--sr--
---S--sr-x
---S--srw-
---S--srwx
---S-wS---
---S-wS--x
---S-wS-w-
---S-wS-wx
---S-wSr--
---S-wSr-x
---S-wSrw-
---S-wSrwx
---S-ws---
---S-ws--x
---S-ws-w-
---S-ws-wx
---S-wsr--
---S-wsr-x
---S-wsrw-
---S-wsrwx
---Sr-S---
---Sr-S--x
---Sr-S-w-
---Sr-S-wx
---Sr-Sr--
---Sr-Sr-x
---Sr-Srw-
---Sr-Srwx
---Sr-s---
---Sr-s--x
---Sr-s-w-
---Sr-s-wx
---Sr-sr--
---Sr-sr-x
---Sr-srw-
---Sr-srwx
---SrwS---
---SrwS--x
---SrwS-w-
---SrwS-wx
---SrwSr--
---SrwSr-x
---SrwSrw-
---SrwSrwx
---Srws---
---Srws--x
---Srws-w-
---Srws-wx
---Srwsr--
---Srwsr-x
---Srwsrw-
---Srwsrwx
---s--S---
---s--S--x
---s--S-w-
---s--S-wx
---s--Sr--
---s--Sr-x
---s--Srw-
---s--Srwx
---s--s---
---s--s--x
---s--s-w-
---s--s-wx
---s--sr--
---s--sr-x
---s--srw-
---s--srwx
---s-wS---
---s-wS--x
---s-wS-w-
---s-wS-wx
---s-wSr--
---s-wSr-x
---s-wSrw-
---s-wSrwx
---s-ws---
---s-ws--x
---s-ws-w-
---s-ws-wx
---s-wsr--
---s-wsr-x
---s-wsrw-
---s-wsrwx
---sr-S---
---sr-S--x
---sr-S-w-
---sr-S-wx
---sr-Sr--
---sr-Sr-x
---sr-Srw-
---sr-Srwx
---sr-s---
---sr-s--x
---sr-s-w-
---sr-s-wx
---sr-sr--
---sr-sr-x
---sr-srw-
---sr-srwx
---srwS---
---srwS--x
---srwS-w-
---srwS-wx
---srwSr--
---srwSr-x
---srwSrw-
---srwSrwx
---srws---
---srws--x
---srws-w-
---srws-wx
---srwsr--
---srwsr-x
---srwsrw-
---srwsrwx
--wS--S---
--wS--S--x
--wS--S-w-
--wS--S-wx
--wS--Sr--
--wS--Sr-x
--wS--Srw-
--wS--Srwx
--wS--s---
--wS--s--x
--wS--s-w-
--wS--s-wx
--wS--sr--
--wS--sr-x
--wS--srw-
--wS--srwx
--wS-wS---
--wS-wS--x
--wS-wS-w-
--wS-wS-wx
--wS-wSr--
--wS-wSr-x
--wS-wSrw-
--wS-wSrwx
--wS-ws---
--wS-ws--x
--wS-ws-w-
--wS-ws-wx
--wS-wsr--
--wS-wsr-x
--wS-wsrw-
--wS-wsrwx
--wSr-S---
--wSr-S--x
--wSr-S-w-
--wSr-S-wx
--wSr-Sr--
--wSr-Sr-x
--wSr-Srw-
--wSr-Srwx
--wSr-s---
--wSr-s--x
--wSr-s-w-
--wSr-s-wx
--wSr-sr--
--wSr-sr-x
--wSr-srw-
--wSr-srwx
--wSrwS---
--wSrwS--x
--wSrwS-w-
--wSrwS-wx
--wSrwSr--
--wSrwSr-x
--wSrwSrw-
--wSrwSrwx
--wSrws---
--wSrws--x
--wSrws-w-
--wSrws-wx
--wSrwsr--
--wSrwsr-x
--wSrwsrw-
--wSrwsrwx
--ws--S---
--ws--S--x
--ws--S-w-
--ws--S-wx
--ws--Sr--
--ws--Sr-x
--ws--Srw-
--ws--Srwx
--ws--s---
--ws--s--x
--ws--s-w-
--ws--s-wx
--ws--sr--
--ws--sr-x
--ws--srw-
--ws--srwx
--ws-wS---
--ws-wS--x
--ws-wS-w-
--ws-wS-wx
--ws-wSr--
--ws-wSr-x
--ws-wSrw-
--ws-wSrwx
--ws-ws---
--ws-ws--x
--ws-ws-w-
--ws-ws-wx
--ws-wsr--
--ws-wsr-x
--ws-wsrw-
--ws-wsrwx
--wsr-S---
--wsr-S--x
--wsr-S-w-
--wsr-S-wx
--wsr-Sr--
--wsr-Sr-x
--wsr-Srw-
--wsr-Srwx
--wsr-s---
--wsr-s--x
--wsr-s-w-
--wsr-s-wx
--wsr-sr--
--wsr-sr-x
--wsr-srw-
--wsr-srwx
--wsrwS---
--wsrwS--x
--wsrwS-w-
--wsrwS-wx
--wsrwSr--
--wsrwSr-x
--wsrwSrw-
--wsrwSrwx
--wsrws---
--wsrws--x
--wsrws-w-
--wsrws-wx
--wsrwsr--
--wsrwsr-x
--wsrwsrw-
--wsrwsrwx
-r-S--S---
-r-S--S--x
-r-S--S-w-
-r-S--S-wx
-r-S--Sr--
-r-S--Sr-x
-r-S--Srw-
-r-S--Srwx
-r-S--s---
-r-S--s--x
-r-S--s-w-
-r-S--s-wx
-r-S--sr--
-r-S--sr-x
-r-S--srw-
-r-S--srwx
-r-S-wS---
-r-S-wS--x
-r-S-wS-w-
-r-S-wS-wx
-r-S-wSr--
-r-S-wSr-x
-r-S-wSrw-
-r-S-wSrwx
-r-S-ws---
-r-S-ws--x
-r-S-ws-w-
-r-S-ws-wx
-r-S-wsr--
-r-S-wsr-x
-r-S-wsrw-
-r-S-wsrwx
-r-Sr-S---
-r-Sr-S--x
-r-Sr-S-w-
-r-Sr-S-wx
-r-Sr-Sr--
-r-Sr-Sr-x
-r-Sr-Srw-
-r-Sr-Srwx
-r-Sr-s---
-r-Sr-s--x
-r-Sr-s-w-
-r-Sr-s-wx
-r-Sr-sr--
-r-Sr-sr-x
-r-Sr-srw-
-r-Sr-srwx
-r-SrwS---
-r-SrwS--x
-r-SrwS-w-
-r-SrwS-wx
-r-SrwSr--
-r-SrwSr-x
-r-SrwSrw-
-r-SrwSrwx
-r-Srws---
-r-Srws--x
-r-Srws-w-
-r-Srws-wx
-r-Srwsr--
-r-Srwsr-x
-r-Srwsrw-
-r-Srwsrwx
-r-s--S---
-r-s--S--x
-r-s--S-w-
-r-s--S-wx
-r-s--Sr--
-r-s--Sr-x
-r-s--Srw-
-r-s--Srwx
-r-s--s---
-r-s--s--x
-r-s--s-w-
-r-s--s-wx
-r-s--sr--
-r-s--sr-x
-r-s--srw-
-r-s--srwx
-r-s-wS---
-r-s-wS--x
-r-s-wS-w-
-r-s-wS-wx
-r-s-wSr--
-r-s-wSr-x
-r-s-wSrw-
-r-s-wSrwx
-r-s-ws---
-r-s-ws--x
-r-s-ws-w-
-r-s-ws-wx
-r-s-wsr--
-r-s-wsr-x
-r-s-wsrw-
-r-s-wsrwx
-r-sr-S---
-r-sr-S--x
-r-sr-S-w-
-r-sr-S-wx
-r-sr-Sr--
-r-sr-Sr-x
-r-sr-Srw-
-r-sr-Srwx
-r-sr-s---
-r-sr-s--x
-r-sr-s-w-
-r-sr-s-wx
-r-sr-sr--
-r-sr-sr-x
-r-sr-srw-
-r-sr-srwx
-r-srwS---
-r-srwS--x
-r-srwS-w-
-r-srwS-wx
-r-srwSr--
-r-srwSr-x
-r-srwSrw-
-r-srwSrwx
-r-srws---
-r-srws--x
-r-srws-w-
-r-srws-wx
-r-srwsr--
-r-srwsr-x
-r-srwsrw-
-r-srwsrwx
-rwS--S---
-rwS--S--x
-rwS--S-w-
-rwS--S-wx
-rwS--Sr--
-rwS--Sr-x
-rwS--Srw-
-rwS--Srwx
-rwS--s---
-rwS--s--x
-rwS--s-w-
-rwS--s-wx
-rwS--sr--
-rwS--sr-x
-rwS--srw-
-rwS--srwx
-rwS-wS---
-rwS-wS--x
-rwS-wS-w-
-rwS-wS-wx
-rwS-wSr--
-rwS-wSr-x
-rwS-wSrw-
-rwS-wSrwx
-rwS-ws---
-rwS-ws--x
-rwS-ws-w-
-rwS-ws-wx
-rwS-wsr--
-rwS-wsr-x
-rwS-wsrw-
-rwS-wsrwx
-rwSr-S---
-rwSr-S--x
-rwSr-S-w-
-rwSr-S-wx
-rwSr-Sr--
-rwSr-Sr-x
-rwSr-Srw-
-rwSr-Srwx
-rwSr-s---
-rwSr-s--x
-rwSr-s-w-
-rwSr-s-wx
-rwSr-sr--
-rwSr-sr-x
-rwSr-srw-
-rwSr-srwx
-rwSrwS---
-rwSrwS--x
-rwSrwS-w-
-rwSrwS-wx
-rwSrwSr--
-rwSrwSr-x
-rwSrwSrw-
-rwSrwSrwx
-rwSrws---
-rwSrws--x
-rwSrws-w-
-rwSrws-wx
-rwSrwsr--
-rwSrwsr-x
-rwSrwsrw-
-rwSrwsrwx
-rws--S---
-rws--S--x
-rws--S-w-
-rws--S-wx
-rws--Sr--
-rws--Sr-x
-rws--Srw-
-rws--Srwx
-rws--s---
-rws--s--x
-rws--s-w-
-rws--s-wx
-rws--sr--
-rws--sr-x
-rws--srw-
-rws--srwx
-rws-wS---
-rws-wS--x
-rws-wS-w-
-rws-wS-wx
-rws-wSr--
-rws-wSr-x
-rws-wSrw-
-rws-wSrwx
-rws-ws---
-rws-ws--x
-rws-ws-w-
-rws-ws-wx
-rws-wsr--
-rws-wsr-x
-rws-wsrw-
-rws-wsrwx
-rwsr-S---
-rwsr-S--x
-rwsr-S-w-
-rwsr-S-wx
-rwsr-Sr--
-rwsr-Sr-x
-rwsr-Srw-
-rwsr-Srwx
-rwsr-s---
-rwsr-s--x
-rwsr-s-w-
-rwsr-s-wx
-rwsr-sr--
-rwsr-sr-x
-rwsr-srw-
-rwsr-srwx
-rwsrwS---
-rwsrwS--x
-rwsrwS-w-
-rwsrwS-wx
-rwsrwSr--
-rwsrwSr-x
-rwsrwSrw-
-rwsrwSrwx
-rwsrws---
-rwsrws--x
-rwsrws-w-
-rwsrws-wx
-rwsrwsr--
-rwsrwsr-x
-rwsrwsrw-
-rwsrwsrwx
---S--S---
---S--S--x
---S--S-w-
---S--S-wx
---S--Sr--
---S--Sr-x
---S--Srw-
---S--Srwx
---S--s---
---S--s--x
---S--s-w-
---S--s-wx
---S--sr--
---S--sr-x
---S--srw-
---S--srwx
---S-wS---
---S-wS--x
---S-wS-w-
---S-wS-wx
---S-wSr--
---S-wSr-x
---S-wSrw-
---S-wSrwx
---S-ws---
---S-ws--x
---S-ws-w-
---S-ws-wx
---S-wsr--
---S-wsr-x
---S-wsrw-
---S-wsrwx
---Sr-S---
---Sr-S--x
---Sr-S-w-
---Sr-S-wx
---Sr-Sr--
---Sr-Sr-x
---Sr-Srw-
---Sr-Srwx
---Sr-s---
---Sr-s--x
---Sr-s-w-
---Sr-s-wx
---Sr-sr--
---Sr-sr-x
---Sr-srw-
---Sr-srwx
---SrwS---
---SrwS--x
---SrwS-w-
---SrwS-wx
---SrwSr--
---SrwSr-x
---SrwSrw-
---SrwSrwx
---Srws---
---Srws--x
---Srws-w-
---Srws-wx
---Srwsr--
---Srwsr-x
---Srwsrw-
---Srwsrwx
---s--S---
---s--S--x
---s--S-w-
---s--S-wx
---s--Sr--
---s--Sr-x
---s--Srw-
---s--Srwx
---s--s---
---s--s--x
---s--s-w-
---s--s-wx
---s--sr--
---s--sr-x
---s--srw-
---s--srwx
---s-wS---
---s-wS--x
---s-wS-w-
---s-wS-wx
---s-wSr--
---s-wSr-x
---s-wSrw-
---s-wSrwx
---s-ws---
---s-ws--x
---s-ws-w-
---s-ws-wx
---s-wsr--
---s-wsr-x
---s-wsrw-
---s-wsrwx
---sr-S---
---sr-S--x
---sr-S-w-
---sr-S-wx
---sr-Sr--
---sr-Sr-x
---sr-Srw-
---sr-Srwx
---sr-s---
---sr-s--x
---sr-s-w-
---sr-s-wx
---sr-sr--
---sr-sr-x
---sr-srw-
---sr-srwx
---srwS---
---srwS--x
---srwS-w-
---srwS-wx
---srwSr--
---srwSr-x
---srwSrw-
---srwSrwx
---srws---
---srws--x
---srws-w-
---srws-wx
---srwsr--
---srwsr-x
---srwsrw-
---srwsrwx
--wS--S---
--wS--S--x
--wS--S-w-
--wS--S-wx
--wS--Sr--
--wS--Sr-x
--wS--Srw-
--wS--Srwx
--wS--s---
--wS--s--x
--wS--s-w-
--wS--s-wx
--wS--sr--
--wS--sr-x
--wS--srw-
--wS--srwx
--wS-wS---
--wS-wS--x
--wS-wS-w-
--wS-wS-wx
--wS-wSr--
--wS-wSr-x
--wS-wSrw-
--wS-wSrwx
--wS-ws---
--wS-ws--x
--wS-ws-w-
--wS-ws-wx
--wS-wsr--
--wS-wsr-x
--wS-wsrw-
--wS-wsrwx
--wSr-S---
--wSr-S--x
--wSr-S-w-
--wSr-S-wx
--wSr-Sr--
--wSr-Sr-x
--wSr-Srw-
--wSr-Srwx
--wSr-s---
--wSr-s--x
--wSr-s-w-
--wSr-s-wx
--wSr-sr--
--wSr-sr-x
--wSr-srw-
--wSr-srwx
--wSrwS---
--wSrwS--x
--wSrwS-w-
--wSrwS-wx
--wSrwSr--
--wSrwSr-x
--wSrwSrw-
--wSrwSrwx
--wSrws---
--wSrws--x
--wSrws-w-
--wSrws-wx
--wSrwsr--
--wSrwsr-x
--wSrwsrw-
--wSrwsrwx
--ws--S---
--ws--S--x
--ws--S-w-
--ws--S-wx
--ws--Sr--
--ws--Sr-x
--ws--Srw-
--ws--Srwx
--ws--s---
--ws--s--x
--ws--s-w-
--ws--s-wx
--ws--sr--
--ws--sr-x
--ws--srw-
--ws--srwx
--ws-wS---
--ws-wS--x
--ws-wS-w-
--ws-wS-wx
--ws-wSr--
--ws-wSr-x
--ws-wSrw-
--ws-wSrwx
--ws-ws---
--ws-ws--x
--ws-ws-w-
--ws-ws-wx
--ws-wsr--
--ws-wsr-x
--ws-wsrw-
--ws-wsrwx
--wsr-S---
--wsr-S--x
--wsr-S-w-
--wsr-S-wx
--wsr-Sr--
--wsr-Sr-x
--wsr-Srw-
--wsr-Srwx
--wsr-s---
--wsr-s--x
--wsr-s-w-
--wsr-s-wx
--wsr-sr--
--wsr-sr-x
--wsr-srw-
--wsr-srwx
--wsrwS---
--wsrwS--x
--wsrwS-w-
--wsrwS-wx
--wsrwSr--
--wsrwSr-x
--wsrwSrw-
--wsrwSrwx
--wsrws---
--wsrws--x
--wsrws-w-
--wsrws-wx
--wsrwsr--
--wsrwsr-x
--wsrwsrw-
--wsrwsrwx
-r-S--S---
-r-S--S--x
-r-S--S-w-
-r-S--S-wx
-r-S--Sr--
-r-S--Sr-x
-r-S--Srw-
-r-S--Srwx
-r-S--s---
-r-S--s--x
-r-S--s-w-
-r-S--s-wx
-r-S--sr--
-r-S--sr-x
-r-S--srw-
-r-S--srwx
-r-S-wS---
-r-S-wS--x
-r-S-wS-w-
-r-S-wS-wx
-r-S-wSr--
-r-S-wSr-x
-r-S-wSrw-
-r-S-wSrwx
-r-S-ws---
-r-S-ws--x
-r-S-ws-w-
-r-S-ws-wx
-r-S-wsr--
-r-S-wsr-x
-r-S-wsrw-
-r-S-wsrwx
-r-Sr-S---
-r-Sr-S--x
-r-Sr-S-w-
-r-Sr-S-wx
-r-Sr-Sr--
-r-Sr-Sr-x
-r-Sr-Srw-
-r-Sr-Srwx
-r-Sr-s---
-r-Sr-s--x
-r-Sr-s-w-
-r-Sr-s-wx
-r-Sr-sr--
-r-Sr-sr-x
-r-Sr-srw-
-r-Sr-srwx
-r-SrwS---
-r-SrwS--x
-r-SrwS-w-
-r-SrwS-wx
-r-SrwSr--
-r-SrwSr-x
-r-SrwSrw-
-r-SrwSrwx
-r-Srws---
-r-Srws--x
-r-Srws-w-
-r-Srws-wx
-r-Srwsr--
-r-Srwsr-x
-r-Srwsrw-
-r-Srwsrwx
-r-s--S---
-r-s--S--x
-r-s--S-w-
-r-s--S-wx
-r-s--Sr--
-r-s--Sr-x
-r-s--Srw-
-r-s--Srwx
-r-s--s---
-r-s--s--x
-r-s--s-w-
-r-s--s-wx
-r-s--sr--
-r-s--sr-x
-r-s--srw-
-r-s--srwx
-r-s-wS---
-r-s-wS--x
-r-s-wS-w-
-r-s-wS-wx
-r-s-wSr--
-r-s-wSr-x
-r-s-wSrw-
-r-s-wSrwx
-r-s-ws---
-r-s-ws--x
-r-s-ws-w-
-r-s-ws-wx
-r-s-wsr--
-r-s-wsr-x
-r-s-wsrw-
-r-s-wsrwx
-r-sr-S---
-r-sr-S--x
-r-sr-S-w-
-r-sr-S-wx
-r-sr-Sr--
-r-sr-Sr-x
-r-sr-Srw-
-r-sr-Srwx
-r-sr-s---
-r-sr-s--x
-r-sr-s-w-
-r-sr-s-wx
-r-sr-sr--
-r-sr-sr-x
-r-sr-srw-
-r-sr-srwx
-r-srwS---
-r-srwS--x
-r-srwS-w-
-r-srwS-wx
-r-srwSr--
-r-srwSr-x
-r-srwSrw-
-r-srwSrwx
-r-srws---
-r-srws--x
-r-srws-w-
-r-srws-wx
-r-srwsr--
-r-srwsr-x
-r-srwsrw-
-r-srwsrwx
-rwS--S---
-rwS--S--x
-rwS--S-w-
-rwS--S-wx
-rwS--Sr--
-rwS--Sr-x
-rwS--Srw-
-rwS--Srwx
-rwS--s---
-rwS--s--x
-rwS--s-w-
-rwS--s-wx
-rwS--sr--
-rwS--sr-x
-rwS--srw-
-rwS--srwx
-rwS-wS---
-rwS-wS--x
-rwS-wS-w-
-rwS-wS-wx
-rwS-wSr--
-rwS-wSr-x
-rwS-wSrw-
-rwS-wSrwx
-rwS-ws---
-rwS-ws--x
-rwS-ws-w-
-rwS-ws-wx
-rwS-wsr--
-rwS-wsr-x
-rwS-wsrw-
-rwS-wsrwx
-rwSr-S---
-rwSr-S--x
-rwSr-S-w-
-rwSr-S-wx
-rwSr-Sr--
-rwSr-Sr-x
-rwSr-Srw-
-rwSr-Srwx
-rwSr-s---
-rwSr-s--x
-rwSr-s-w-
-rwSr-s-wx
-rwSr-sr--
-rwSr-sr-x
-rwSr-srw-
-rwSr-srwx
-rwSrwS---
-rwSrwS--x
-rwSrwS-w-
-rwSrwS-wx
-rwSrwSr--
-rwSrwSr-x
-rwSrwSrw-
-rwSrwSrwx
-rwSrws---
-rwSrws--x
-rwSrws-w-
-rwSrws-wx
-rwSrwsr--
-rwSrwsr-x
-rwSrwsrw-
-rwSrwsrwx
-rws--S---
-rws--S--x
-rws--S-w-
-rws--S-wx
-rws--Sr--
-rws--Sr-x
-rws--Srw-
-rws--Srwx
-rws--s---
-rws--s--x
-rws--s-w-
-rws--s-wx
-rws--sr--
-rws--sr-x
-rws--srw-
-rws--srwx
-rws-wS---
-rws-wS--x
-rws-wS-w-
-rws-wS-wx
-rws-wSr--
-rws-wSr-x
-rws-wSrw-
-rws-wSrwx
-rws-ws---
-rws-ws--x
-rws-ws-w-
-rws-ws-wx
-rws-wsr--
-rws-wsr-x
-rws-wsrw-
-rws-wsrwx
-rwsr-S---
-rwsr-S--x
-rwsr-S-w-
-rwsr-S-wx
-rwsr-Sr--
-rwsr-Sr-x
-rwsr-Srw-
-rwsr-Srwx
-rwsr-s---
-rwsr-s--x
-rwsr-s-w-
-rwsr-s-wx
-rwsr-sr--
-rwsr-sr-x
-rwsr-srw-
-rwsr-srwx
-rwsrwS---
-rwsrwS--x
-rwsrwS-w-
-rwsrwS-wx
-rwsrwSr--
-rwsrwSr-x
-rwsrwSrw-
-rwsrwSrwx
-rwsrws---
-rwsrws--x
-rwsrws-w-
-rwsrws-wx
-rwsrwsr--
-rwsrwsr-x
-rwsrwsrw-
-rwsrwsrwx
