"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEnsureDefaultDataView = void 0;
var _lodash = require("lodash");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const createEnsureDefaultDataView = (uiSettings, onRedirectNoDataView, canUpdateUiSetting, savedObjectsClient) => {
  /**
   * Checks whether a default index pattern is set and exists and defines
   * one otherwise.
   */
  return async function ensureDefaultDataView() {
    if (canUpdateUiSetting === false) {
      return;
    }
    let patterns = await this.getIds();
    let defaultId = await uiSettings.get('defaultIndex');
    let defined = !!defaultId;
    const exists = (0, _lodash.includes)(patterns, defaultId);
    if (defined && !exists) {
      await uiSettings.remove('defaultIndex');
      defaultId = defined = false;
    }
    if (defined) {
      const dataView = await this.get(defaultId);
      const dataSourceRef = dataView === null || dataView === void 0 ? void 0 : dataView.dataSourceRef;
      if (!dataSourceRef) {
        return;
      }
      let isDefaultDataViewReferenceValid = true;
      if (!dataSourceRef.id) {
        isDefaultDataViewReferenceValid = false;
      } else {
        try {
          var _result$error, _result$error2;
          const result = await this.getDataSource(dataSourceRef.id);
          isDefaultDataViewReferenceValid = !(((_result$error = result.error) === null || _result$error === void 0 ? void 0 : _result$error.statusCode) === 403 || ((_result$error2 = result.error) === null || _result$error2 === void 0 ? void 0 : _result$error2.statusCode) === 404);
        } catch (e) {
          // The logic below for updating the default index pattern only handles cases where the data source is not found or the user lacks access permissions
          // For other unexpected errors, we simply return to prevent infinite loops when updating the default index pattern.
          return;
        }
      }
      if (!isDefaultDataViewReferenceValid) {
        try {
          if (savedObjectsClient) {
            const datasources = await savedObjectsClient.find({
              type: 'data-source'
            });
            const dataViews = await savedObjectsClient.find({
              type: 'index-pattern'
            });
            const existDataSources = datasources.map(item => item.id);
            patterns = [];
            dataViews.forEach(item => {
              var _item$references;
              const sourceRef = (_item$references = item.references) === null || _item$references === void 0 ? void 0 : _item$references.find(ref => ref.type === 'data-source');
              let isDataSourceReferenceValid = false;
              /**
               * The reference is valid when either:
               * 1. No data source is referenced (using local cluster, which must be available for OpenSearch Dashboards to function)
               * 2. A data source is referenced with a valid ID
               */
              if (!sourceRef) {
                isDataSourceReferenceValid = true;
              }
              if (sourceRef !== null && sourceRef !== void 0 && sourceRef.id && existDataSources.includes(sourceRef.id)) {
                isDataSourceReferenceValid = true;
              }
              if (isDataSourceReferenceValid) {
                patterns.push(item.id);
              }
            });
          }
        } catch (e) {
          return;
        }
      } else {
        return;
      }
    }

    // If there is any index pattern created, set the first as default
    if (patterns.length >= 1) {
      defaultId = patterns[0];
      await uiSettings.set('defaultIndex', defaultId);
    } else {
      const isEnhancementsEnabled = await uiSettings.get('query:enhancements:enabled');
      const shouldRedirect = !isEnhancementsEnabled;
      if (shouldRedirect) return onRedirectNoDataView();else return;
    }
  };
};
exports.createEnsureDefaultDataView = createEnsureDefaultDataView;