﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class CreateGroupRequest : public IAMRequest {
 public:
  AWS_IAM_API CreateGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateGroup"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p> The path to the group. For more information about paths, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM
   * identifiers</a> in the <i>IAM User Guide</i>.</p> <p>This parameter is optional.
   * If it is not included, it defaults to a slash (/).</p> <p>This parameter allows
   * (through its <a href="http://wikipedia.org/wiki/regex">regex pattern</a>) a
   * string of characters consisting of either a forward slash (/) by itself or a
   * string that must begin and end with forward slashes. In addition, it can contain
   * any ASCII character from the ! (<code>\u0021</code>) through the DEL character
   * (<code>\u007F</code>), including most punctuation characters, digits, and upper
   * and lowercased letters.</p>
   */
  inline const Aws::String& GetPath() const { return m_path; }
  inline bool PathHasBeenSet() const { return m_pathHasBeenSet; }
  template <typename PathT = Aws::String>
  void SetPath(PathT&& value) {
    m_pathHasBeenSet = true;
    m_path = std::forward<PathT>(value);
  }
  template <typename PathT = Aws::String>
  CreateGroupRequest& WithPath(PathT&& value) {
    SetPath(std::forward<PathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the group to create. Do not include the path in this value.</p>
   * <p>IAM user, group, role, and policy names must be unique within the account.
   * Names are not distinguished by case. For example, you cannot create resources
   * named both "MyResource" and "myresource".</p>
   */
  inline const Aws::String& GetGroupName() const { return m_groupName; }
  inline bool GroupNameHasBeenSet() const { return m_groupNameHasBeenSet; }
  template <typename GroupNameT = Aws::String>
  void SetGroupName(GroupNameT&& value) {
    m_groupNameHasBeenSet = true;
    m_groupName = std::forward<GroupNameT>(value);
  }
  template <typename GroupNameT = Aws::String>
  CreateGroupRequest& WithGroupName(GroupNameT&& value) {
    SetGroupName(std::forward<GroupNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_path;

  Aws::String m_groupName;
  bool m_pathHasBeenSet = false;
  bool m_groupNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
