/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.search;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import javax.inject.Inject;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.LongField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.document.TextField;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.IndexableField;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryparser.classic.MultiFieldQueryParser;
import org.apache.lucene.queryparser.classic.ParseException;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.WildcardQuery;
import org.apache.lucene.search.highlight.Formatter;
import org.apache.lucene.search.highlight.Highlighter;
import org.apache.lucene.search.highlight.InvalidTokenOffsetsException;
import org.apache.lucene.search.highlight.QueryScorer;
import org.apache.lucene.search.highlight.Scorer;
import org.apache.lucene.search.highlight.SimpleHTMLFormatter;
import org.apache.lucene.search.highlight.TextFragment;
import org.apache.lucene.search.highlight.TokenSources;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.RAMDirectory;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.Paragraph;
import org.apache.zeppelin.search.SearchService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LuceneSearch
extends SearchService {
    private static final Logger LOGGER = LoggerFactory.getLogger(LuceneSearch.class);
    private static final String SEARCH_FIELD_TEXT = "contents";
    private static final String SEARCH_FIELD_TITLE = "header";
    private static final String PARAGRAPH = "paragraph";
    private static final String ID_FIELD = "id";
    private Path indexPath;
    private Directory indexDirectory;
    private Analyzer analyzer;
    private IndexWriterConfig indexWriterConfig;
    private IndexWriter indexWriter;

    @Inject
    public LuceneSearch(ZeppelinConfiguration conf) {
        super("LuceneSearch-Thread");
        if (conf.isZeppelinSearchUseDisk().booleanValue()) {
            try {
                this.indexPath = Paths.get(conf.getZeppelinSearchIndexPath(), new String[0]);
                this.indexDirectory = FSDirectory.open((Path)this.indexPath);
                LOGGER.info("Use {} for storing lucene search index", (Object)this.indexPath);
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to create index directory for search service. Use memory instead", e);
            }
        } else {
            this.indexDirectory = new RAMDirectory();
        }
        this.analyzer = new StandardAnalyzer();
        this.indexWriterConfig = new IndexWriterConfig(this.analyzer);
        try {
            this.indexWriter = new IndexWriter(this.indexDirectory, this.indexWriterConfig);
        }
        catch (IOException e) {
            LOGGER.error("Failed to create new IndexWriter", (Throwable)e);
        }
    }

    @Override
    public List<Map<String, String>> query(String queryStr) {
        if (null == this.indexDirectory) {
            throw new IllegalStateException("Something went wrong on instance creation time, index dir is null");
        }
        List<Map<String, String>> result = Collections.emptyList();
        try (DirectoryReader indexReader = DirectoryReader.open((Directory)this.indexDirectory);){
            IndexSearcher indexSearcher = new IndexSearcher((IndexReader)indexReader);
            StandardAnalyzer analyzer = new StandardAnalyzer();
            MultiFieldQueryParser parser = new MultiFieldQueryParser(new String[]{SEARCH_FIELD_TEXT, SEARCH_FIELD_TITLE}, (Analyzer)analyzer);
            Query query = parser.parse(queryStr);
            LOGGER.debug("Searching for: {}", (Object)query.toString(SEARCH_FIELD_TEXT));
            SimpleHTMLFormatter htmlFormatter = new SimpleHTMLFormatter();
            Highlighter highlighter = new Highlighter((Formatter)htmlFormatter, (Scorer)new QueryScorer(query));
            result = this.doSearch(indexSearcher, query, (Analyzer)analyzer, highlighter);
        }
        catch (IOException e) {
            LOGGER.error("Failed to open index dir {}, make sure indexing finished OK", (Object)this.indexDirectory, (Object)e);
        }
        catch (ParseException e) {
            LOGGER.error("Failed to parse query {}", (Object)queryStr, (Object)e);
        }
        return result;
    }

    private List<Map<String, String>> doSearch(IndexSearcher searcher, Query query, Analyzer analyzer, Highlighter highlighter) {
        ArrayList matchingParagraphs = Lists.newArrayList();
        try {
            ScoreDoc[] hits = searcher.search((Query)query, (int)20).scoreDocs;
            for (int i = 0; i < hits.length; ++i) {
                LOGGER.debug("doc={} score={}", (Object)hits[i].doc, (Object)Float.valueOf(hits[i].score));
                int id = hits[i].doc;
                Document doc = searcher.doc(id);
                String path = doc.get(ID_FIELD);
                if (path != null) {
                    TokenStream tokenTitle;
                    TextFragment[] frgTitle;
                    LOGGER.debug("{}. {}", (Object)(i + 1), (Object)path);
                    String title = doc.get("title");
                    if (title != null) {
                        LOGGER.debug("   Title: {}", (Object)doc.get("title"));
                    }
                    String text = doc.get(SEARCH_FIELD_TEXT);
                    String header = doc.get(SEARCH_FIELD_TITLE);
                    String fragment = "";
                    if (text != null) {
                        TokenStream tokenStream = TokenSources.getTokenStream((IndexReader)searcher.getIndexReader(), (int)id, (String)SEARCH_FIELD_TEXT, (Analyzer)analyzer);
                        TextFragment[] frag = highlighter.getBestTextFragments(tokenStream, text, true, 3);
                        LOGGER.debug("    {} fragments found for query '{}'", (Object)frag.length, (Object)query);
                        for (int j = 0; j < frag.length; ++j) {
                            if (frag[j] == null || !(frag[j].getScore() > 0.0f)) continue;
                            LOGGER.debug("    Fragment: {}", (Object)frag[j].toString());
                        }
                        String string = fragment = frag != null && frag.length > 0 ? frag[0].toString() : "";
                    }
                    header = header != null ? ((frgTitle = highlighter.getBestTextFragments(tokenTitle = TokenSources.getTokenStream((IndexReader)searcher.getIndexReader(), (int)id, (String)SEARCH_FIELD_TITLE, (Analyzer)analyzer), header, true, 3)) != null && frgTitle.length > 0 ? frgTitle[0].toString() : "") : "";
                    matchingParagraphs.add(ImmutableMap.of((Object)ID_FIELD, (Object)path, (Object)"name", (Object)title, (Object)"snippet", (Object)fragment, (Object)"text", (Object)text, (Object)SEARCH_FIELD_TITLE, (Object)header));
                    continue;
                }
                LOGGER.info("{}. No {} for this document", (Object)(i + 1), (Object)ID_FIELD);
            }
        }
        catch (IOException | InvalidTokenOffsetsException e) {
            LOGGER.error("Exception on searching for {}", (Object)query, (Object)e);
        }
        return matchingParagraphs;
    }

    @Override
    public void updateNoteIndex(Note note) throws IOException {
        this.updateIndexNoteName(note);
    }

    private void updateIndexNoteName(Note note) throws IOException {
        String noteName = note.getName();
        String noteId = note.getId();
        LOGGER.debug("Update note index: {}, '{}'", (Object)noteId, (Object)noteName);
        if (null == noteName || noteName.isEmpty()) {
            LOGGER.debug("Skipping empty notebook name");
            return;
        }
        this.updateDoc(noteId, noteName, null);
    }

    @Override
    public void updateParagraphIndex(Paragraph p) throws IOException {
        LOGGER.debug("Update paragraph index: {}", (Object)p.getId());
        this.updateDoc(p.getNote().getId(), p.getNote().getName(), p);
    }

    private void updateDoc(String noteId, String noteName, Paragraph p) throws IOException {
        String id = LuceneSearch.formatId(noteId, p);
        Document doc = this.newDocument(id, noteName, p);
        try {
            this.indexWriter.updateDocument(new Term(ID_FIELD, id), (Iterable)doc);
            this.indexWriter.commit();
        }
        catch (IOException e) {
            LOGGER.error("Failed to update index of notebook {}", (Object)noteId, (Object)e);
        }
    }

    static String formatId(String noteId, Paragraph p) {
        String id = noteId;
        if (null != p) {
            id = Joiner.on((char)'/').join((Object)id, (Object)PARAGRAPH, new Object[]{p.getId()});
        }
        return id;
    }

    static String formatDeleteId(String noteId, Paragraph p) {
        String id = noteId;
        id = null != p ? Joiner.on((char)'/').join((Object)id, (Object)PARAGRAPH, new Object[]{p.getId()}) : id + "*";
        return id;
    }

    private Document newDocument(String id, String noteName, Paragraph p) {
        Document doc = new Document();
        StringField pathField = new StringField(ID_FIELD, id, Field.Store.YES);
        doc.add((IndexableField)pathField);
        doc.add((IndexableField)new StringField("title", noteName, Field.Store.YES));
        if (null != p) {
            if (p.getText() != null) {
                doc.add((IndexableField)new TextField(SEARCH_FIELD_TEXT, p.getText(), Field.Store.YES));
            }
            if (p.getTitle() != null) {
                doc.add((IndexableField)new TextField(SEARCH_FIELD_TITLE, p.getTitle(), Field.Store.YES));
            }
            Date date = p.getDateStarted() != null ? p.getDateStarted() : p.getDateCreated();
            doc.add((IndexableField)new LongField("modified", date.getTime(), Field.Store.NO));
        } else {
            doc.add((IndexableField)new TextField(SEARCH_FIELD_TEXT, noteName, Field.Store.YES));
        }
        return doc;
    }

    @Override
    public void addNoteIndex(Note note) {
        try {
            this.addIndexDocAsync(note);
            this.indexWriter.commit();
        }
        catch (IOException e) {
            LOGGER.error("Failed to add note {} to index", (Object)note, (Object)e);
        }
    }

    @Override
    public void addParagraphIndex(Paragraph pararaph) throws IOException {
        this.updateDoc(pararaph.getNote().getId(), pararaph.getNote().getName(), pararaph);
    }

    private void addIndexDocAsync(Note note) throws IOException {
        this.indexNoteName(this.indexWriter, note.getId(), note.getName());
        for (Paragraph paragraph : note.getParagraphs()) {
            this.updateDoc(note.getId(), note.getName(), paragraph);
        }
    }

    @Override
    public void deleteNoteIndex(Note note) {
        if (note == null) {
            return;
        }
        this.deleteDoc(note.getId(), null);
        for (Paragraph paragraph : note.getParagraphs()) {
            this.deleteParagraphIndex(note.getId(), paragraph);
        }
    }

    @Override
    public void deleteParagraphIndex(String noteId, Paragraph p) {
        this.deleteDoc(noteId, p);
    }

    private void deleteDoc(String noteId, Paragraph p) {
        String fullNoteOrJustParagraph = LuceneSearch.formatDeleteId(noteId, p);
        LOGGER.debug("Deleting note {}, out of: {}", (Object)noteId, (Object)this.indexWriter.numDocs());
        try {
            this.indexWriter.deleteDocuments(new Query[]{new WildcardQuery(new Term(ID_FIELD, fullNoteOrJustParagraph))});
            this.indexWriter.commit();
        }
        catch (IOException e) {
            LOGGER.error("Failed to delete {} from index by '{}'", new Object[]{noteId, fullNoteOrJustParagraph, e});
        }
        LOGGER.debug("Done, index contains {} docs now {}", (Object)this.indexWriter.numDocs());
    }

    @Override
    public void close() {
        try {
            this.indexWriter.close();
        }
        catch (IOException e) {
            LOGGER.error("Failed to .close() the notebook index", (Throwable)e);
        }
    }

    private void indexNoteName(IndexWriter w, String noteId, String noteName) throws IOException {
        LOGGER.debug("Indexing Notebook {}, '{}'", (Object)noteId, (Object)noteName);
        if (null == noteName || noteName.isEmpty()) {
            LOGGER.debug("Skipping empty notebook name");
            return;
        }
        this.updateDoc(noteId, noteName, null);
    }

    @Override
    public void startRebuildIndex(Stream<Note> notes) {
        Thread thread = new Thread(() -> {
            LOGGER.info("Starting rebuild index");
            notes.forEach(note -> {
                this.addNoteIndex((Note)note);
                note.unLoad();
            });
            LOGGER.info("Finish rebuild index");
        });
        thread.setName("LuceneSearch-RebuildIndex-Thread");
        thread.start();
        try {
            thread.join();
        }
        catch (InterruptedException e) {
            e.printStackTrace();
        }
    }
}

