/*
 Licensed to the Apache Software Foundation (ASF) under one
 or more contributor license agreements.  See the NOTICE file
 distributed with this work for additional information
 regarding copyright ownership.  The ASF licenses this file
 to you under the Apache License, Version 2.0 (the
 "License"); you may not use this file except in compliance
 with the License.  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
*/

package placement

import (
	"testing"

	"gotest.tools/v3/assert"

	"github.com/apache/yunikorn-core/pkg/common/configs"
	"github.com/apache/yunikorn-core/pkg/common/security"
	"github.com/apache/yunikorn-core/pkg/scheduler/placement/types"
)

// basic test to check if no rules leave the manager unusable
func TestManagerNew(t *testing.T) {
	// basic info without rules, manager should not init
	man := NewPlacementManager(nil, queueFunc)
	assert.Equal(t, 2, len(man.rules), "wrong rule count for empty placement manager")
	assert.Equal(t, types.Provided, man.rules[0].getName(), "wrong name for implicit provided rule")
	assert.Equal(t, types.Recovery, man.rules[1].getName(), "wrong name for implicit recovery rule")
}

func TestManagerInit(t *testing.T) {
	// basic info without rules, manager should implicitly init
	man := NewPlacementManager(nil, queueFunc)
	assert.Equal(t, 2, len(man.rules), "wrong rule count for nil placement manager")

	// try to init with empty list should do the same
	var rules []configs.PlacementRule
	err := man.initialise(rules)
	assert.NilError(t, err, "Failed to initialize empty placement rules")
	assert.Equal(t, 2, len(man.rules), "wrong rule count for empty placement manager")

	rules = []configs.PlacementRule{
		{Name: "unknown"},
	}
	err = man.initialise(rules)
	if err == nil {
		t.Error("initialise with 'unknown' rule list should have failed")
	}

	// init the manager with one rule
	rules = []configs.PlacementRule{
		{Name: "test"},
	}
	err = man.initialise(rules)
	assert.NilError(t, err, "failed to init existing manager")

	// update the manager: remove rules implicit state is reverted
	rules = []configs.PlacementRule{}
	err = man.initialise(rules)
	assert.NilError(t, err, "Failed to re-initialize empty placement rules")
	assert.Equal(t, 2, len(man.rules), "wrong rule count for newly empty placement manager")

	// check if we handle a nil list
	err = man.initialise(nil)
	assert.NilError(t, err, "Failed to re-initialize nil placement rules")
	assert.Equal(t, 2, len(man.rules), "wrong rule count for nil placement manager")
}

func TestManagerUpdate(t *testing.T) {
	// basic info without rules, manager should not init
	man := NewPlacementManager(nil, queueFunc)
	// update the manager
	rules := []configs.PlacementRule{
		{Name: "test"},
	}
	err := man.UpdateRules(rules)
	assert.NilError(t, err, "failed to update existing manager")

	// update the manager: remove rules init state is reverted
	rules = []configs.PlacementRule{}
	err = man.UpdateRules(rules)
	assert.NilError(t, err, "Failed to re-initialize empty placement rules")
	assert.Equal(t, 2, len(man.rules), "wrong rule count for newly empty placement manager")

	// check if we handle a nil list
	err = man.UpdateRules(nil)
	assert.NilError(t, err, "Failed to re-initialize nil placement rules")
	assert.Equal(t, 2, len(man.rules), "wrong rule count for nil placement manager")
}

func TestManagerBuildRule(t *testing.T) {
	// basic with 1 rule
	man := NewPlacementManager(nil, queueFunc)
	rules := []configs.PlacementRule{
		{Name: "test"},
	}
	ruleObjs, err := man.buildRules(rules)
	if err != nil {
		t.Errorf("test rule build should not have failed, err: %v", err)
	}
	if len(ruleObjs) != 2 {
		t.Errorf("test rule build should have created 2 rules found: %d", len(ruleObjs))
	}

	// rule with a parent rule should only be 1 rule in the list
	rules = []configs.PlacementRule{
		{Name: "test",
			Parent: &configs.PlacementRule{
				Name: "test",
			},
		},
	}
	ruleObjs, err = man.buildRules(rules)
	if err != nil || len(ruleObjs) != 2 {
		t.Errorf("test rule build should not have failed and created 2 top level rule, err: %v, rules: %v", err, ruleObjs)
	} else {
		parent := ruleObjs[0].getParent()
		if parent == nil || parent.getName() != "test" {
			t.Error("test rule build should have created 2 rules: parent not found")
		}
	}

	// two rules in order: cannot use the same rule names as we can not check them
	rules = []configs.PlacementRule{
		{Name: "user"},
		{Name: "test"},
	}
	ruleObjs, err = man.buildRules(rules)
	if err != nil || len(ruleObjs) != 3 {
		t.Errorf("rule build should not have failed and created 3 rules, err: %v, rules: %v", err, ruleObjs)
	} else if ruleObjs[0].getName() != "user" || ruleObjs[1].getName() != "test" || ruleObjs[2].getName() != "recovery" {
		t.Errorf("rule build order is not preserved: %v", ruleObjs)
	}
}

func TestManagerPlaceApp(t *testing.T) {
	// Create the structure for the test
	data := `
partitions:
  - name: default
    queues:
      - name: root
        queues:
          - name: testparent
            submitacl: "*"
            queues:
              - name: testchild
          - name: fixed
            submitacl: "other-user "
            parent: true
`
	err := initQueueStructure([]byte(data))
	assert.NilError(t, err, "setting up the queue config failed")
	// basic info without rules, manager should init
	man := NewPlacementManager(nil, queueFunc)
	if man == nil {
		t.Fatal("placement manager create failed")
	}
	// update the manager
	rules := []configs.PlacementRule{
		{Name: "user",
			Create: false,
			Parent: &configs.PlacementRule{
				Name:  "fixed",
				Value: "testparent"},
		},
		{Name: "provided",
			Create: true},
		{Name: "tag",
			Value:  "namespace",
			Create: true},
	}
	err = man.UpdateRules(rules)
	assert.NilError(t, err, "failed to update existing manager")
	tags := make(map[string]string)
	user := security.UserGroup{
		User:   "testchild",
		Groups: []string{},
	}
	app := newApplication("app1", "default", "", user, tags, nil, "")

	// user rule existing queue, acl allowed
	err = man.PlaceApplication(app)
	queueName := app.GetQueuePath()
	if err != nil || queueName != "root.testparent.testchild" {
		t.Errorf("leaf exist: app should have been placed in user queue, queue: '%s', error: %v", queueName, err)
	}
	user = security.UserGroup{
		User:   "other-user",
		Groups: []string{},
	}

	// user rule new queue: fails on create flag
	app = newApplication("app1", "default", "", user, tags, nil, "")
	err = man.PlaceApplication(app)
	queueName = app.GetQueuePath()
	if err == nil || queueName != "" {
		t.Errorf("leaf to create, no create flag: app should not have been placed, queue: '%s', error: %v", queueName, err)
	}

	// provided rule (2nd rule): queue acl allowed, anyone create
	app = newApplication("app1", "default", "root.fixed.leaf", user, tags, nil, "")
	err = man.PlaceApplication(app)
	queueName = app.GetQueuePath()
	if err != nil || queueName != "root.fixed.leaf" {
		t.Errorf("leave create, acl allow: app should have been placed, queue: '%s', error: %v", queueName, err)
	}

	// provided rule (2rd): queue acl deny, queue does not exist
	user = security.UserGroup{
		User:   "unknown-user",
		Groups: []string{},
	}
	app = newApplication("app1", "default", "root.fixed.other", user, tags, nil, "")
	err = man.PlaceApplication(app)
	queueName = app.GetQueuePath()
	if err == nil || queueName != "" {
		t.Errorf("leaf to create, acl deny: app should not have been placed, queue: '%s', error: %v", queueName, err)
	}

	// tag rule (3rd) check queue acl deny, queue was created above)
	tags = map[string]string{"namespace": "root.fixed.leaf"}
	app = newApplication("app1", "default", "", user, tags, nil, "")
	err = man.PlaceApplication(app)
	queueName = app.GetQueuePath()
	if err == nil || queueName != "" {
		t.Errorf("existing leaf, acl deny: app should not have been placed, queue: '%s', error: %v", queueName, err)
	}

	// tag rule (3rd) queue acl allow, queue already exists
	user = security.UserGroup{
		User:   "other-user",
		Groups: []string{},
	}
	app = newApplication("app1", "default", "", user, tags, nil, "")
	err = man.PlaceApplication(app)
	queueName = app.GetQueuePath()
	if err != nil || queueName != "root.fixed.leaf" {
		t.Errorf("existing leaf, acl allow: app should have been placed, queue: '%s', error: %v", queueName, err)
	}

	// provided rule (2nd): submit to parent
	app = newApplication("app1", "default", "root.fixed", user, nil, nil, "")
	err = man.PlaceApplication(app)
	queueName = app.GetQueuePath()
	if err == nil || queueName != "" {
		t.Errorf("parent queue: app should not have been placed, queue: '%s', error: %v", queueName, err)
	}
}
