/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sis.math;

import java.io.Serializable;
import java.util.Iterator;
import java.util.function.DoubleBinaryOperator;
import org.apache.sis.math.ArrayVector;
import org.apache.sis.math.CompoundDirectPositions;
import org.apache.sis.math.Vector;
import org.apache.sis.util.ArgumentChecks;
import org.apache.sis.util.internal.DoubleDouble;
import org.apache.sis.util.internal.Numerics;
import org.apache.sis.util.internal.Strings;
import org.apache.sis.util.resources.Errors;
import org.opengis.geometry.DirectPosition;
import org.opengis.geometry.MismatchedDimensionException;

public class Plane
implements DoubleBinaryOperator,
Cloneable,
Serializable {
    private static final long serialVersionUID = 2956201711131316723L;
    private static final int DIMENSION = 3;
    private static final double ZERO_THRESHOLD = 1.0E-14;
    private double sx;
    private double sy;
    private double z0;

    public Plane() {
        this.z0 = Double.NaN;
        this.sy = Double.NaN;
        this.sx = Double.NaN;
    }

    public Plane(double sx, double sy, double z0) {
        this.sx = sx;
        this.sy = sy;
        this.z0 = z0;
    }

    public final double slopeX() {
        return this.sx;
    }

    public final double slopeY() {
        return this.sy;
    }

    public final double z0() {
        return this.z0;
    }

    public final double x(double y, double z) {
        return (z - (this.z0 + this.sy * y)) / this.sx;
    }

    public final double y(double x, double z) {
        return (z - (this.z0 + this.sx * x)) / this.sy;
    }

    public final double z(double x, double y) {
        return this.z0 + this.sx * x + this.sy * y;
    }

    @Override
    public double applyAsDouble(double x, double y) {
        return this.z(x, y);
    }

    public void setEquation(double sx, double sy, double z0) {
        this.sx = sx;
        this.sy = sy;
        this.z0 = z0;
    }

    public void setEquation(Number sx, Number sy, Number z0) {
        this.setEquation(sx.doubleValue(), sy.doubleValue(), z0.doubleValue());
    }

    public double fit(double[] x, double[] y, double[] z) {
        ArgumentChecks.ensureNonNull("x", x);
        ArgumentChecks.ensureNonNull("y", y);
        ArgumentChecks.ensureNonNull("z", z);
        return this.fit(new ArrayVector.Doubles(x), new ArrayVector.Doubles(y), (Vector)new ArrayVector.Doubles(z));
    }

    public double fit(Vector x, Vector y, Vector z) {
        ArgumentChecks.ensureNonNull("x", x);
        ArgumentChecks.ensureNonNull("y", y);
        ArgumentChecks.ensureNonNull("z", z);
        return this.fit(new CompoundDirectPositions(x, y, z));
    }

    public double fit(int nx, int ny, Vector z) {
        ArgumentChecks.ensureStrictlyPositive("nx", nx);
        ArgumentChecks.ensureStrictlyPositive("ny", ny);
        ArgumentChecks.ensureNonNull("z", z);
        int length = Math.multiplyExact(nx, ny);
        if (z.size() != length) {
            throw new IllegalArgumentException(Errors.format((short)133, length, z.size()));
        }
        Fit r = new Fit(nx, ny, z);
        r.resolve();
        double p = r.correlation(nx, length, z, null);
        this.setEquation(r.sx, r.sy, r.z0);
        return p;
    }

    public double fit(Iterable<? extends DirectPosition> points) {
        ArgumentChecks.ensureNonNull("points", points);
        Fit r = new Fit(points);
        r.resolve();
        double p = r.correlation(0, 0, null, points.iterator());
        this.setEquation(r.sx, r.sy, r.z0);
        return p;
    }

    public Plane clone() {
        try {
            return (Plane)super.clone();
        }
        catch (CloneNotSupportedException exception) {
            throw new AssertionError((Object)exception);
        }
    }

    public boolean equals(Object object) {
        if (object != null && this.getClass() == object.getClass()) {
            Plane that = (Plane)object;
            return Numerics.equals(this.z0, that.z0) && Numerics.equals(this.sx, that.sx) && Numerics.equals(this.sy, that.sy);
        }
        return false;
    }

    public int hashCode() {
        return Long.hashCode(0x290689CB2C6251F3L ^ Double.doubleToLongBits(this.z0) + 31L * (Double.doubleToLongBits(this.sx) + 31L * Double.doubleToLongBits(this.sy)));
    }

    public String toString() {
        StringBuilder buffer = new StringBuilder(60).append("z(x,y) = ");
        String separator = "";
        if (this.sx != 0.0) {
            buffer.append(this.sx).append("\u22c5x");
            separator = " + ";
        }
        if (this.sy != 0.0) {
            buffer.append(separator).append(this.sy).append("\u22c5y");
            separator = " + ";
        }
        return buffer.append(separator).append(this.z0).toString();
    }

    private static final class Fit {
        private DoubleDouble sum_x = DoubleDouble.ZERO;
        private DoubleDouble sum_y = DoubleDouble.ZERO;
        private DoubleDouble sum_z = DoubleDouble.ZERO;
        private DoubleDouble sum_xx = DoubleDouble.ZERO;
        private DoubleDouble sum_yy = DoubleDouble.ZERO;
        private DoubleDouble sum_xy = DoubleDouble.ZERO;
        private DoubleDouble sum_zx = DoubleDouble.ZERO;
        private DoubleDouble sum_zy = DoubleDouble.ZERO;
        private final int n;
        DoubleDouble sx;
        DoubleDouble sy;
        DoubleDouble z0;

        Fit(Iterable<? extends DirectPosition> points) {
            int i = 0;
            int n = 0;
            for (DirectPosition directPosition : points) {
                double z;
                double y;
                int dimension = directPosition.getDimension();
                if (dimension != 3) {
                    throw new MismatchedDimensionException(Errors.format((short)81, Strings.toIndexed("points", i), 3, dimension));
                }
                ++i;
                double x = directPosition.getOrdinate(0);
                if (Double.isNaN(x) || Double.isNaN(y = directPosition.getOrdinate(1)) || Double.isNaN(z = directPosition.getOrdinate(2))) continue;
                this.sum_x = this.sum_x.add(x, false);
                this.sum_y = this.sum_y.add(y, false);
                this.sum_z = this.sum_z.add(z, false);
                this.sum_xx = this.sum_xx.add(DoubleDouble.product(x, x));
                this.sum_yy = this.sum_yy.add(DoubleDouble.product(y, y));
                this.sum_xy = this.sum_xy.add(DoubleDouble.product(x, y));
                this.sum_zx = this.sum_zx.add(DoubleDouble.product(z, x));
                this.sum_zy = this.sum_zy.add(DoubleDouble.product(z, y));
                ++n;
            }
            this.n = n;
        }

        Fit(int nx, int ny, Vector vz) {
            int n = 0;
            for (int y = 0; y < ny; ++y) {
                for (int x = 0; x < nx; ++x) {
                    double z = vz.doubleValue(n);
                    if (Double.isNaN(z)) {
                        throw new IllegalArgumentException(Errors.format((short)110, Strings.toIndexed("z", n)));
                    }
                    this.sum_z = this.sum_z.add(z, false);
                    this.sum_zx = this.sum_zx.add(DoubleDouble.product(z, x));
                    this.sum_zy = this.sum_zy.add(DoubleDouble.product(z, y));
                    ++n;
                }
            }
            this.sum_x = DoubleDouble.of((double)n / 2.0, false).multiply(nx - 1);
            this.sum_y = DoubleDouble.of((double)n / 2.0, false).multiply(ny - 1);
            this.sum_xx = DoubleDouble.of(n).multiply((double)nx - 0.5, false).multiply(nx - 1).divide(3);
            this.sum_yy = DoubleDouble.of(n).multiply((double)ny - 0.5, false).multiply(ny - 1).divide(3);
            this.sum_xy = DoubleDouble.of((double)n / 4.0, false).multiply(ny - 1).multiply(nx - 1);
            this.n = n;
        }

        private void resolve() {
            DoubleDouble zx = this.sum_zx.subtract(this.sum_z.multiply(this.sum_x).divide(this.n));
            DoubleDouble zy = this.sum_zy.subtract(this.sum_z.multiply(this.sum_y).divide(this.n));
            DoubleDouble xx = this.sum_xx.subtract(this.sum_x.multiply(this.sum_x).divide(this.n));
            DoubleDouble xy = this.sum_xy.subtract(this.sum_x.multiply(this.sum_y).divide(this.n));
            DoubleDouble yy = this.sum_yy.subtract(this.sum_y.multiply(this.sum_y).divide(this.n));
            DoubleDouble den = xy.square().subtract(xx.multiply(yy));
            this.sx = zy.multiply(xy).subtract(zx.multiply(yy)).divide(den);
            this.sy = zx.multiply(xy).subtract(zy.multiply(xx)).divide(den);
            this.z0 = this.sum_z.subtract(this.sx.multiply(this.sum_x).add(this.sy.multiply(this.sum_y))).divide(this.n);
        }

        double correlation(int nx, int length, Vector vz, Iterator<? extends DirectPosition> points) {
            boolean detectZeroSx = true;
            boolean detectZeroSy = true;
            boolean detectZeroZ0 = true;
            double sx = this.sx.doubleValue();
            double sy = this.sy.doubleValue();
            double z0 = this.z0.doubleValue();
            double mean_x = this.sum_x.doubleValue() / (double)this.n;
            double mean_y = this.sum_y.doubleValue() / (double)this.n;
            double mean_z = this.sum_z.doubleValue() / (double)this.n;
            double offset = Math.abs(sx * mean_x + sy * mean_y + z0);
            int index = 0;
            double sum_ds2 = 0.0;
            double sum_dz2 = 0.0;
            double sum_dsz = 0.0;
            while (true) {
                double z;
                double y;
                double x;
                if (vz != null) {
                    if (index >= length) break;
                    x = index % nx;
                    y = index / nx;
                    z = vz.doubleValue(index++);
                } else {
                    if (!points.hasNext()) break;
                    DirectPosition p = points.next();
                    x = p.getOrdinate(0);
                    y = p.getOrdinate(1);
                    z = p.getOrdinate(2);
                }
                x = (x - mean_x) * sx;
                y = (y - mean_y) * sy;
                double s2 = x + y;
                if (!Double.isNaN(s2) && !Double.isNaN(z -= mean_z)) {
                    sum_ds2 += s2 * s2;
                    sum_dz2 += z * z;
                    sum_dsz += s2 * z;
                }
                if (detectZeroSx && Math.abs(x) >= Math.ulp(y * 1.0E-14)) {
                    detectZeroSx = false;
                }
                if (detectZeroSy && Math.abs(y) >= Math.ulp(x * 1.0E-14)) {
                    detectZeroSy = false;
                }
                if (!detectZeroZ0 || !(offset >= Math.ulp(s2 * 1.0E-14))) continue;
                detectZeroZ0 = false;
            }
            if (detectZeroSx) {
                this.sx = DoubleDouble.ZERO;
            }
            if (detectZeroSy) {
                this.sy = DoubleDouble.ZERO;
            }
            if (detectZeroZ0) {
                this.z0 = DoubleDouble.ZERO;
            }
            return Math.min(sum_dsz / Math.sqrt(sum_ds2 * sum_dz2), 1.0);
        }
    }
}

