/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.tika.server.core;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import org.apache.cxf.jaxrs.JAXRSServerFactoryBean;
import org.apache.cxf.jaxrs.client.WebClient;
import org.apache.cxf.jaxrs.lifecycle.ResourceProvider;
import org.apache.cxf.jaxrs.lifecycle.SingletonResourceProvider;
import org.junit.jupiter.api.Test;

import org.apache.tika.server.core.resource.DetectorResource;
import org.apache.tika.server.core.resource.MetadataResource;
import org.apache.tika.server.core.resource.RecursiveMetadataResource;
import org.apache.tika.server.core.resource.TikaResource;
import org.apache.tika.server.core.resource.UnpackerResource;
import org.apache.tika.server.core.writer.CSVMessageBodyWriter;
import org.apache.tika.server.core.writer.JSONMessageBodyWriter;
import org.apache.tika.server.core.writer.MetadataListMessageBodyWriter;
import org.apache.tika.server.core.writer.TextMessageBodyWriter;


/**
 * Test to make sure that no stack traces are returned
 * when the stack trace param is set to false.
 */
public class StackTraceOffTest extends CXFTestBase {

    private static final String TEST_HELLO_WORLD = "test-documents/mock/hello_world.xml";
    private static final String TEST_NULL = "test-documents/mock/null_pointer.xml";
    private static final String TEST_PASSWORD_PROTECTED =
            "test-documents/mock/encrypted_document_exception.xml";


    private static final String[] PATHS = new String[]{"/tika", "/rmeta", "/unpack", "/meta",};
    private static final int UNPROCESSEABLE = 422;

    @Override
    protected void setUpResources(JAXRSServerFactoryBean sf) {
        List<ResourceProvider> rCoreProviders = new ArrayList<>();
        rCoreProviders.add(new SingletonResourceProvider(new MetadataResource()));
        rCoreProviders.add(new SingletonResourceProvider(new RecursiveMetadataResource()));
        rCoreProviders
                .add(new SingletonResourceProvider(new DetectorResource(new ServerStatus("", 0))));
        rCoreProviders.add(new SingletonResourceProvider(new TikaResource()));
        rCoreProviders.add(new SingletonResourceProvider(new UnpackerResource()));
        sf.setResourceProviders(rCoreProviders);
    }

    @Override
    protected void setUpProviders(JAXRSServerFactoryBean sf) {
        List<Object> providers = new ArrayList<>();
        providers.add(new TikaServerParseExceptionMapper(false));
        providers.add(new JSONMessageBodyWriter());
        providers.add(new CSVMessageBodyWriter());
        providers.add(new TextMessageBodyWriter());
        providers.add(new MetadataListMessageBodyWriter());
        sf.setProviders(providers);
    }

    @Test
    public void testEncrypted() throws Exception {
        for (String path : PATHS) {
            if ("/rmeta".equals(path)) {
                continue;
            }
            String accept = "*/*";
            if ("/tika".equals(path)) {
                accept = "text/plain";
            }
            Response response = WebClient.create(endPoint + path).accept(accept)
                    .header("Content-Disposition",
                            "attachment; filename=" + TEST_PASSWORD_PROTECTED)
                    .put(ClassLoader.getSystemResourceAsStream(TEST_PASSWORD_PROTECTED));
            assertNotNull(response, "null response: " + path);
            assertEquals(UNPROCESSEABLE, response.getStatus(), "unprocessable: " + path);
            String msg = getStringFromInputStream((InputStream) response.getEntity());
            assertEquals("", msg, "should be empty: " + path);
        }
    }

    @Test
    public void testNullPointerOnTika() throws Exception {
        for (String path : PATHS) {
            if ("/rmeta".equals(path)) {
                continue;
            }
            String accept = "*/*";
            if ("/tika".equals(path)) {
                accept = "text/plain";
            }
            Response response = WebClient.create(endPoint + path).accept(accept)
                    .put(ClassLoader.getSystemResourceAsStream(TEST_NULL));
            assertNotNull(response, "null response: " + path);
            assertEquals(UNPROCESSEABLE, response.getStatus(), "unprocessable: " + path);
            String msg = getStringFromInputStream((InputStream) response.getEntity());
            assertEquals("", msg, "should be empty: " + path);

        }
    }

    @Test
    public void testEmptyParser() throws Exception {
        //As of Tika 1.23, we're no longer returning 415 for file types
        //that don't have a parser
        //no stack traces for 415
        for (String path : PATHS) {

            Response response = WebClient.create(endPoint + path).accept("*:*")
                    .put(getClass().getResourceAsStream("/test-documents/testDigilite.fdf"));
            if (path.equals("/unpack")) {
                //"NO CONTENT"
                assertEquals(204, response.getStatus(), "bad type: " + path);
            } else {
                assertEquals(200, response.getStatus(), "bad type: " + path);
                assertNotNull(response, "null response: " + path);
            }
        }
    }

    //For now, make sure that non-complete document
    //still returns BAD_REQUEST.  We may want to
    //make MetadataResource return the same types of parse
    //exceptions as the others...
    @Test
    public void testMeta() throws Exception {
        InputStream stream = ClassLoader.getSystemResourceAsStream(TEST_HELLO_WORLD);

        Response response =
                WebClient.create(endPoint + "/meta" + "/Author").type("application/mock+xml")
                        .accept(MediaType.TEXT_PLAIN).put(copy(stream, 100));
        assertEquals(Response.Status.BAD_REQUEST.getStatusCode(), response.getStatus());
        String msg = getStringFromInputStream((InputStream) response.getEntity());
        assertEquals("Failed to get metadata field Author", msg);
    }
}
