/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.syncope.core.persistence.jpa.entity.user;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSetter;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.syncope.core.persistence.api.dao.UserDAO;
import org.apache.syncope.core.persistence.api.entity.PlainAttrUniqueValue;
import org.apache.syncope.core.persistence.api.entity.PlainAttrValue;
import org.apache.syncope.core.persistence.api.entity.user.UMembership;
import org.apache.syncope.core.persistence.api.entity.user.UPlainAttr;
import org.apache.syncope.core.persistence.api.entity.user.UPlainAttrValue;
import org.apache.syncope.core.persistence.api.entity.user.User;
import org.apache.syncope.core.persistence.jpa.entity.AbstractPlainAttr;
import org.apache.syncope.core.spring.ApplicationContextProvider;

public class JSONUPlainAttr extends AbstractPlainAttr<User> implements UPlainAttr {

    private static final long serialVersionUID = 8066058729580952116L;

    @JsonIgnore
    private User owner;

    /**
     * The membership of this attribute; might be {@code NULL} if this attribute is not related to a membership.
     */
    @JsonProperty
    private String membership;

    /**
     * Values of this attribute (if schema is not UNIQUE).
     */
    private List<JSONUPlainAttrValue> values = new ArrayList<>();

    /**
     * Value of this attribute (if schema is UNIQUE).
     */
    @JsonProperty
    private JSONUPlainAttrUniqueValue uniqueValue;

    @Override
    public User getOwner() {
        return owner;
    }

    @Override
    public void setOwner(final User owner) {
        this.owner = (JPAUser) owner;
    }

    @JsonIgnore
    @Override
    public UMembership getMembership() {
        return ApplicationContextProvider.getBeanFactory().getBean(UserDAO.class).findMembership(membership);
    }

    @Override
    public void setMembership(final UMembership membership) {
        this.membership = Optional.ofNullable(membership).map(UMembership::getKey).orElse(null);
    }

    @JsonGetter("membership")
    @Override
    public String getMembershipKey() {
        return membership;
    }

    @JsonSetter("membership")
    public void setMembership(final String membership) {
        this.membership = membership;
    }

    @Override
    protected boolean addForMultiValue(final PlainAttrValue attrValue) {
        return values.add((JSONUPlainAttrValue) attrValue);
    }

    @Override
    public List<? extends UPlainAttrValue> getValues() {
        return values;
    }

    @Override
    public JSONUPlainAttrUniqueValue getUniqueValue() {
        return uniqueValue;
    }

    @JsonIgnore
    @Override
    public void setUniqueValue(final PlainAttrUniqueValue uniqueValue) {
        this.uniqueValue = (JSONUPlainAttrUniqueValue) uniqueValue;
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder().
                appendSuper(super.hashCode()).
                append(membership).
                append(values).
                append(uniqueValue).
                build();
    }

    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final JSONUPlainAttr other = (JSONUPlainAttr) obj;
        return new EqualsBuilder().
                appendSuper(super.equals(obj)).
                append(membership, other.membership).
                append(values, other.values).
                append(uniqueValue, other.uniqueValue).
                build();
    }
}
