/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.spark.sql.catalyst.analysis.resolver

import java.util.IdentityHashMap

import scala.collection.mutable

import org.apache.spark.SparkException
import org.apache.spark.sql.catalyst.expressions.{Alias, Attribute, Expression, NamedExpression}
import org.apache.spark.sql.catalyst.expressions.aggregate.AggregateExpression
import org.apache.spark.sql.catalyst.plans.logical.Aggregate

/**
 * Used to extract aggregate and grouping expressions in the context of underlying [[Aggregate]]
 * operator or collecting aggregate expressions based on the provided expression.
 */
class GroupingAndAggregateExpressionsExtractor(
    aggregate: Aggregate,
    autoGeneratedAliasProvider: AutoGeneratedAliasProvider) {
  private val aliasChildToAliasInAggregateExpressions = new IdentityHashMap[Expression, Alias]
  private val aggregateExpressionsSemanticComparator = new SemanticComparator(
    aggregate.aggregateExpressions.map {
      case alias: Alias =>
        aliasChildToAliasInAggregateExpressions.put(alias.child, alias)
        alias.child
      case other => other
    }
  )

  private val groupingExpressionsSemanticComparator = new SemanticComparator(
    aggregate.groupingExpressions
  )

  /**
   * Extracts grouping and aggregate expressions based on the provided expression. Instead of
   * provided expression use the extracted one (if any), update `referencedGroupingExpressions`
   * (grouping expressions used for extraction) and `extractedAggregateExpressionAliases` (aliases
   * of [[AggregateExpression]]s that are transformed to attributes during extraction) in order to
   * insert missing attributes to below operators.
   */
  def extractReferencedGroupingAndAggregateExpressions(
      expression: Expression,
      referencedGroupingExpressions: mutable.ArrayBuffer[NamedExpression],
      extractedAggregateExpressionAliases: mutable.ArrayBuffer[Alias]): Expression = {
    collectFirstAggregateExpression(expression) match {
      case (Some(attribute: Attribute), _)
          if !aliasChildToAliasInAggregateExpressions.containsKey(attribute) =>
        attribute
      case (Some(expression), alias) =>
        alias match {
          case None =>
            throw SparkException.internalError(
              s"No parent alias for expression $expression while extracting aggregate" +
              s"expressions in Sort operator."
            )
          case Some(alias) =>
            alias.toAttribute
        }
      case (None, _) if groupingExpressionsSemanticComparator.exists(expression) =>
        expression match {
          case attribute: Attribute =>
            referencedGroupingExpressions += attribute
            attribute
          case other =>
            val alias = autoGeneratedAliasProvider.newAlias(child = other)
            referencedGroupingExpressions += alias
            alias.toAttribute
        }
      case _ =>
        expression match {
          case aggregateExpression: AggregateExpression =>
            val alias = autoGeneratedAliasProvider.newAlias(child = aggregateExpression)
            extractedAggregateExpressionAliases += alias
            alias.toAttribute
          case other => other
        }
    }
  }

  /**
   * Get first aggregate expression's child that is semantically equal to the provided expression,
   * and it's parent [[Alias]] (if any).
   */
  def collectFirstAggregateExpression(
      expression: Expression): (Option[Expression], Option[Alias]) = {
    val referencedAggregateExpression =
      aggregateExpressionsSemanticComparator.collectFirst(expression)
    referencedAggregateExpression match {
      case Some(expression) =>
        (Some(expression), Option(aliasChildToAliasInAggregateExpressions.get(expression)))
      case None =>
        (None, None)
    }
  }

  /**
   * Returns true if the provided expression is semantically equal to any of the aggregate
   * expressions' child.
   */
  def exists(expression: Expression): Boolean = {
    aggregateExpressionsSemanticComparator.exists(expression)
  }
}
