/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.sis.metadata.iso.acquisition;

import java.util.Collection;
import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import org.opengis.metadata.Identifier;
import org.opengis.metadata.acquisition.Event;
import org.opengis.metadata.acquisition.Objective;
import org.opengis.metadata.acquisition.Operation;
import org.opengis.metadata.acquisition.OperationType;
import org.opengis.metadata.acquisition.Plan;
import org.opengis.metadata.acquisition.Platform;
import org.opengis.metadata.citation.Citation;
import org.opengis.metadata.identification.Progress;
import org.opengis.util.InternationalString;
import org.apache.sis.metadata.iso.ISOMetadata;


/**
 * Designations for the operation used to acquire the dataset.
 * The following properties are mandatory in a well-formed metadata according ISO 19115:
 *
 * <div class="preformat">{@code MI_Operation}
 * {@code   ├─identifier…………………} Unique identification of the operation.
 * {@code   │   └─code………………………} Alphanumeric value identifying an instance in the namespace.
 * {@code   ├─status……………………………} Status of the data acquisition.
 * {@code   └─parentOperation……} Heritage of the operation.</div>
 *
 * <h2>Limitations</h2>
 * <ul>
 *   <li>Instances of this class are not synchronized for multi-threading.
 *       Synchronization, if needed, is caller's responsibility.</li>
 *   <li>Serialized objects of this class are not guaranteed to be compatible with future Apache SIS releases.
 *       Serialization support is appropriate for short term storage or RMI between applications running the
 *       same version of Apache SIS. For long term storage, use {@link org.apache.sis.xml.XML} instead.</li>
 * </ul>
 *
 * @author  Cédric Briançon (Geomatys)
 * @author  Martin Desruisseaux (Geomatys)
 * @version 1.0
 * @since   0.3
 * @module
 */
@XmlType(name = "MI_Operation_Type", propOrder = {
    "description",
    "citation",
    "identifier",
    "status",
    "type",
    "childOperations",
    "objectives",
    "parentOperation",
    "plan",
    "platforms",
    "significantEvents"
})
@XmlRootElement(name = "MI_Operation")
public class DefaultOperation extends ISOMetadata implements Operation {
    /**
     * Serial number for inter-operability with different versions.
     */
    private static final long serialVersionUID = 4828650802232651791L;

    /**
     * Description of the mission on which the platform observations are made and the objectives of that mission.
     */
    @SuppressWarnings("serial")
    private InternationalString description;

    /**
     * Identification of the mission.
     */
    @SuppressWarnings("serial")
    private Citation citation;

    /**
     * Status of the data acquisition.
     */
    @SuppressWarnings("serial")
    private Progress status;

    /**
     * Collection technique for the operation.
     */
    @SuppressWarnings("serial")
    private OperationType type;

    /**
     * Sub-missions that make up part of a larger mission.
     */
    @SuppressWarnings("serial")
    private Collection<Operation> childOperations;

    /**
     * Object(s) or area(s) of interest to be sensed.
     */
    @SuppressWarnings("serial")
    private Collection<Objective> objectives;

    /**
     * Heritage of the operation.
     */
    @SuppressWarnings("serial")
    private Operation parentOperation;

    /**
     * Plan satisfied by the operation.
     */
    @SuppressWarnings("serial")
    private Plan plan;

    /**
     * Platform (or platforms) used in the operation.
     */
    @SuppressWarnings("serial")
    private Collection<Platform> platforms;

    /**
     * Record of an event occurring during an operation.
     */
    @SuppressWarnings("serial")
    private Collection<Event> significantEvents;

    /**
     * Constructs an initially empty operation.
     */
    public DefaultOperation() {
    }

    /**
     * Constructs a new instance initialized with the values from the specified metadata object.
     * This is a <em>shallow</em> copy constructor, because the other metadata contained in the
     * given object are not recursively copied.
     *
     * @param  object  the metadata to copy values from, or {@code null} if none.
     *
     * @see #castOrCopy(Operation)
     */
    public DefaultOperation(final Operation object) {
        super(object);
        if (object != null) {
            description       = object.getDescription();
            citation          = object.getCitation();
            identifiers       = singleton(object.getIdentifier(), Identifier.class);
            status            = object.getStatus();
            type              = object.getType();
            childOperations   = copyCollection(object.getChildOperations(), Operation.class);
            objectives        = copyCollection(object.getObjectives(), Objective.class);
            parentOperation   = object.getParentOperation();
            plan              = object.getPlan();
            platforms         = copyCollection(object.getPlatforms(), Platform.class);
            significantEvents = copyCollection(object.getSignificantEvents(), Event.class);
        }
    }

    /**
     * Returns a SIS metadata implementation with the values of the given arbitrary implementation.
     * This method performs the first applicable action in the following choices:
     *
     * <ul>
     *   <li>If the given object is {@code null}, then this method returns {@code null}.</li>
     *   <li>Otherwise if the given object is already an instance of
     *       {@code DefaultOperation}, then it is returned unchanged.</li>
     *   <li>Otherwise a new {@code DefaultOperation} instance is created using the
     *       {@linkplain #DefaultOperation(Operation) copy constructor} and returned.
     *       Note that this is a <em>shallow</em> copy operation, because the other
     *       metadata contained in the given object are not recursively copied.</li>
     * </ul>
     *
     * @param  object  the object to get as a SIS implementation, or {@code null} if none.
     * @return a SIS implementation containing the values of the given object (may be the
     *         given object itself), or {@code null} if the argument was null.
     */
    public static DefaultOperation castOrCopy(final Operation object) {
        if (object == null || object instanceof DefaultOperation) {
            return (DefaultOperation) object;
        }
        return new DefaultOperation(object);
    }

    /**
     * Returns the description of the mission on which the platform observations are made and the
     * objectives of that mission. {@code null} if unspecified.
     *
     * @return description of the mission, or {@code null}.
     */
    @Override
    @XmlElement(name = "description")
    public InternationalString getDescription() {
        return description;
    }

    /**
     * Sets the description of the mission on which the platform observations are made and the
     * objectives of that mission.
     *
     * @param  newValue  the new description value.
     */
    public void setDescription(final InternationalString newValue) {
        checkWritePermission(description);
        description = newValue;
    }

    /**
     * Returns the identification of the mission. {@code null} if unspecified.
     *
     * @return identification of the mission, or {@code null}.
     */
    @Override
    @XmlElement(name = "citation")
    public Citation getCitation() {
        return citation;
    }

    /**
     * Sets the identification of the mission.
     *
     * @param  newValue  the new citation value.
     */
    public void setCitation(final Citation newValue) {
        checkWritePermission(citation);
        citation = newValue;
    }

    /**
     * Returns the unique identification of the operation.
     *
     * @return unique identification of the operation, or {@code null}.
     */
    @Override
    @XmlElement(name = "identifier")
    public Identifier getIdentifier() {
        return super.getIdentifier();
    }

    /**
     * Sets the unique identification of the operation.
     *
     * @param  newValue  the new identifier value.
     */
    @Override
    public void setIdentifier(final Identifier newValue) {
        super.setIdentifier(newValue);
    }

    /**
     * Returns the status of the data acquisition.
     *
     * @return status of the data acquisition, or {@code null}.
     */
    @Override
    @XmlElement(name = "status", required = true)
    public Progress getStatus() {
        return status;
    }

    /**
     * Sets the status of the data acquisition.
     *
     * @param  newValue  the new status value.
     */
    public void setStatus(final Progress newValue) {
        checkWritePermission(status);
        status = newValue;
    }

    /**
     * Returns the collection technique for the operation.
     *
     * @return collection technique for the operation, or {@code null}.
     */
    @Override
    @XmlElement(name = "type")
    public OperationType getType() {
        return type;
    }

    /**
     * Sets the collection technique for the operation.
     *
     * @param  newValue  the new type value.
     */
    public void setType(final OperationType newValue) {
        checkWritePermission(type);
        type = newValue;
    }

    /**
     * Returns the sub-missions that make up part of a larger mission.
     *
     * @return sub-missions.
     */
    @Override
    @XmlElement(name = "childOperation")
    public Collection<Operation> getChildOperations() {
        return childOperations = nonNullCollection(childOperations, Operation.class);
    }

    /**
     * Sets the sub-missions that make up part of a larger mission.
     *
     * @param  newValues  the new child operations values.
     */
    public void setChildOperations(final Collection<? extends Operation> newValues) {
        childOperations = writeCollection(newValues, childOperations, Operation.class);
    }

    /**
     * Returns object(s) or area(s) of interest to be sensed.
     *
     * @return object(s) or area(s) of interest.
     */
    @Override
    @XmlElement(name = "objective")
    public Collection<Objective> getObjectives() {
        return objectives = nonNullCollection(objectives, Objective.class);
    }

    /**
     * Sets Object(s) or area(s) of interest to be sensed.
     *
     * @param  newValues  the new objectives values.
     */
    public void setObjectives(final Collection<? extends Objective> newValues) {
        objectives = writeCollection(newValues, objectives, Objective.class);
    }

    /**
     * Returns the heritage of the operation.
     *
     * @return heritage of the operation, or {@code null}.
     */
    @Override
    @XmlElement(name = "parentOperation")
    public Operation getParentOperation() {
        return parentOperation;
    }

    /**
     * Sets the heritage of the operation.
     *
     * @param  newValue  the new parent operation value.
     */
    public void setParentOperation(final Operation newValue) {
        checkWritePermission(parentOperation);
        parentOperation = newValue;
    }

    /**
     * Returns the plan satisfied by the operation.
     *
     * @return plan satisfied by the operation, or {@code null}.
     */
    @Override
    @XmlElement(name = "plan")
    public Plan getPlan() {
        return plan;
    }

    /**
     * Sets the plan satisfied by the operation.
     *
     * @param  newValue  the new plan value.
     */
    public void setPlan(final Plan newValue) {
        checkWritePermission(plan);
        plan = newValue;
    }

    /**
     * Returns the platform (or platforms) used in the operation.
     *
     * @return platforms used in the operation.
     */
    @Override
    @XmlElement(name = "platform")
    public Collection<Platform> getPlatforms() {
        return platforms = nonNullCollection(platforms, Platform.class);
    }

    /**
     * Sets the platform (or platforms) used in the operation.
     *
     * @param  newValues  the new platforms values.
     */
    public void setPlatforms(final Collection<? extends Platform> newValues) {
        platforms = writeCollection(newValues, platforms, Platform.class);
    }

    /**
     * Returns the record of an event occurring during an operation.
     *
     * @return record of an event occurring during an operation.
     */
    @Override
    @XmlElement(name = "significantEvent")
    public Collection<Event> getSignificantEvents() {
        return significantEvents = nonNullCollection(significantEvents, Event.class);
    }

    /**
     * Sets the record of an event occurring during an operation.
     *
     * @param  newValues  the new significant events value.
     */
    public void setSignificantEvents(final Collection<? extends Event> newValues) {
        significantEvents = writeCollection(newValues, significantEvents, Event.class);
    }
}
