/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sis.math;

import java.io.Serializable;
import java.util.Iterator;
import java.util.function.DoubleBinaryOperator;
import org.apache.sis.internal.util.DoubleDouble;
import org.apache.sis.internal.util.Numerics;
import org.apache.sis.internal.util.Strings;
import org.apache.sis.math.ArrayVector;
import org.apache.sis.math.CompoundDirectPositions;
import org.apache.sis.math.Vector;
import org.apache.sis.util.ArgumentChecks;
import org.apache.sis.util.resources.Errors;
import org.opengis.geometry.DirectPosition;
import org.opengis.geometry.MismatchedDimensionException;

public class Plane
implements DoubleBinaryOperator,
Cloneable,
Serializable {
    private static final long serialVersionUID = 2956201711131316723L;
    private static final int DIMENSION = 3;
    private static final double ZERO_THRESHOLD = 1.0E-14;
    private double sx;
    private double sy;
    private double z0;

    public Plane() {
        this.z0 = Double.NaN;
        this.sy = Double.NaN;
        this.sx = Double.NaN;
    }

    public Plane(double sx, double sy, double z0) {
        this.sx = sx;
        this.sy = sy;
        this.z0 = z0;
    }

    public final double slopeX() {
        return this.sx;
    }

    public final double slopeY() {
        return this.sy;
    }

    public final double z0() {
        return this.z0;
    }

    public final double x(double y, double z) {
        return (z - (this.z0 + this.sy * y)) / this.sx;
    }

    public final double y(double x, double z) {
        return (z - (this.z0 + this.sx * x)) / this.sy;
    }

    public final double z(double x, double y) {
        return this.z0 + this.sx * x + this.sy * y;
    }

    @Override
    public double applyAsDouble(double x, double y) {
        return this.z(x, y);
    }

    public void setEquation(double sx, double sy, double z0) {
        this.sx = sx;
        this.sy = sy;
        this.z0 = z0;
    }

    public void setEquation(Number sx, Number sy, Number z0) {
        this.setEquation(sx.doubleValue(), sy.doubleValue(), z0.doubleValue());
    }

    public double fit(double[] x, double[] y, double[] z) {
        ArgumentChecks.ensureNonNull("x", x);
        ArgumentChecks.ensureNonNull("y", y);
        ArgumentChecks.ensureNonNull("z", z);
        return this.fit(new ArrayVector.Doubles(x), new ArrayVector.Doubles(y), (Vector)new ArrayVector.Doubles(z));
    }

    public double fit(Vector x, Vector y, Vector z) {
        ArgumentChecks.ensureNonNull("x", x);
        ArgumentChecks.ensureNonNull("y", y);
        ArgumentChecks.ensureNonNull("z", z);
        return this.fit(new CompoundDirectPositions(x, y, z));
    }

    public double fit(int nx, int ny, Vector z) {
        ArgumentChecks.ensureStrictlyPositive("nx", nx);
        ArgumentChecks.ensureStrictlyPositive("ny", ny);
        ArgumentChecks.ensureNonNull("z", z);
        int length = Math.multiplyExact(nx, ny);
        if (z.size() != length) {
            throw new IllegalArgumentException(Errors.format((short)133, length, z.size()));
        }
        Fit r = new Fit(nx, ny, z);
        r.resolve();
        double p = r.correlation(nx, length, z, null);
        this.setEquation(r.sx, r.sy, r.z0);
        return p;
    }

    public double fit(Iterable<? extends DirectPosition> points) {
        ArgumentChecks.ensureNonNull("points", points);
        Fit r = new Fit(points);
        r.resolve();
        double p = r.correlation(0, 0, null, points.iterator());
        this.setEquation(r.sx, r.sy, r.z0);
        return p;
    }

    public Plane clone() {
        try {
            return (Plane)super.clone();
        }
        catch (CloneNotSupportedException exception) {
            throw new AssertionError((Object)exception);
        }
    }

    public boolean equals(Object object) {
        if (object != null && this.getClass() == object.getClass()) {
            Plane that = (Plane)object;
            return Numerics.equals(this.z0, that.z0) && Numerics.equals(this.sx, that.sx) && Numerics.equals(this.sy, that.sy);
        }
        return false;
    }

    public int hashCode() {
        return Long.hashCode(0x290689CB2C6251F3L ^ Double.doubleToLongBits(this.z0) + 31L * (Double.doubleToLongBits(this.sx) + 31L * Double.doubleToLongBits(this.sy)));
    }

    public String toString() {
        StringBuilder buffer = new StringBuilder(60).append("z(x,y) = ");
        String separator = "";
        if (this.sx != 0.0) {
            buffer.append(this.sx).append("\u22c5x");
            separator = " + ";
        }
        if (this.sy != 0.0) {
            buffer.append(separator).append(this.sy).append("\u22c5y");
            separator = " + ";
        }
        return buffer.append(separator).append(this.z0).toString();
    }

    private static final class Fit {
        private final DoubleDouble sum_x = new DoubleDouble();
        private final DoubleDouble sum_y = new DoubleDouble();
        private final DoubleDouble sum_z = new DoubleDouble();
        private final DoubleDouble sum_xx = new DoubleDouble();
        private final DoubleDouble sum_yy = new DoubleDouble();
        private final DoubleDouble sum_xy = new DoubleDouble();
        private final DoubleDouble sum_zx = new DoubleDouble();
        private final DoubleDouble sum_zy = new DoubleDouble();
        private final DoubleDouble xx = new DoubleDouble();
        private final DoubleDouble yy = new DoubleDouble();
        private final DoubleDouble xy = new DoubleDouble();
        private final DoubleDouble zx = new DoubleDouble();
        private final DoubleDouble zy = new DoubleDouble();
        private final int n;
        DoubleDouble sx;
        DoubleDouble sy;
        DoubleDouble z0;

        Fit(Iterable<? extends DirectPosition> points) {
            int i = 0;
            int n = 0;
            for (DirectPosition directPosition : points) {
                double z;
                double y;
                int dimension = directPosition.getDimension();
                if (dimension != 3) {
                    throw new MismatchedDimensionException(Errors.format((short)81, Strings.toIndexed("points", i), 3, dimension));
                }
                ++i;
                double x = directPosition.getOrdinate(0);
                if (Double.isNaN(x) || Double.isNaN(y = directPosition.getOrdinate(1)) || Double.isNaN(z = directPosition.getOrdinate(2))) continue;
                this.xx.setToProduct(x, x);
                this.yy.setToProduct(y, y);
                this.xy.setToProduct(x, y);
                this.zx.setToProduct(z, x);
                this.zy.setToProduct(z, y);
                this.sum_x.add(x);
                this.sum_y.add(y);
                this.sum_z.add(z);
                this.sum_xx.add(this.xx);
                this.sum_yy.add(this.yy);
                this.sum_xy.add(this.xy);
                this.sum_zx.add(this.zx);
                this.sum_zy.add(this.zy);
                ++n;
            }
            this.n = n;
        }

        Fit(int nx, int ny, Vector vz) {
            int n = 0;
            for (int y = 0; y < ny; ++y) {
                for (int x = 0; x < nx; ++x) {
                    double z = vz.doubleValue(n);
                    if (Double.isNaN(z)) {
                        throw new IllegalArgumentException(Errors.format((short)110, Strings.toIndexed("z", n)));
                    }
                    this.zx.setToProduct(z, x);
                    this.zy.setToProduct(z, y);
                    this.sum_z.add(z);
                    this.sum_zx.add(this.zx);
                    this.sum_zy.add(this.zy);
                    ++n;
                }
            }
            this.sum_x.value = (double)n / 2.0;
            this.sum_x.multiply(nx - 1);
            this.sum_y.value = (double)n / 2.0;
            this.sum_y.multiply(ny - 1);
            this.sum_xx.value = n;
            this.sum_xx.multiply((double)nx - 0.5);
            this.sum_xx.multiply(nx - 1);
            this.sum_xx.divide(3.0);
            this.sum_yy.value = n;
            this.sum_yy.multiply((double)ny - 0.5);
            this.sum_yy.multiply(ny - 1);
            this.sum_yy.divide(3.0);
            this.sum_xy.value = (double)n / 4.0;
            this.sum_xy.multiply(ny - 1);
            this.sum_xy.multiply(nx - 1);
            this.n = n;
        }

        private void resolve() {
            this.zx.setFrom(this.sum_x);
            this.zx.divide(-this.n);
            this.zx.multiply(this.sum_z);
            this.zx.add(this.sum_zx);
            this.zy.setFrom(this.sum_y);
            this.zy.divide(-this.n);
            this.zy.multiply(this.sum_z);
            this.zy.add(this.sum_zy);
            this.xx.setFrom(this.sum_x);
            this.xx.divide(-this.n);
            this.xx.multiply(this.sum_x);
            this.xx.add(this.sum_xx);
            this.xy.setFrom(this.sum_y);
            this.xy.divide(-this.n);
            this.xy.multiply(this.sum_x);
            this.xy.add(this.sum_xy);
            this.yy.setFrom(this.sum_y);
            this.yy.divide(-this.n);
            this.yy.multiply(this.sum_y);
            this.yy.add(this.sum_yy);
            DoubleDouble tmp = new DoubleDouble(this.xx);
            tmp.multiply(this.yy);
            DoubleDouble den = new DoubleDouble(this.xy);
            den.multiply(this.xy);
            den.subtract(tmp);
            this.sx = new DoubleDouble(this.zy);
            this.sx.multiply(this.xy);
            tmp.setFrom(this.zx);
            tmp.multiply(this.yy);
            this.sx.subtract(tmp);
            this.sx.divide(den);
            this.sy = new DoubleDouble(this.zx);
            this.sy.multiply(this.xy);
            tmp.setFrom(this.zy);
            tmp.multiply(this.xx);
            this.sy.subtract(tmp);
            this.sy.divide(den);
            this.z0 = new DoubleDouble(this.sy);
            this.z0.multiply(this.sum_y);
            tmp.setFrom(this.sx);
            tmp.multiply(this.sum_x);
            tmp.add(this.z0);
            this.z0.setFrom(this.sum_z);
            this.z0.subtract(tmp);
            this.z0.divide(this.n);
        }

        double correlation(int nx, int length, Vector vz, Iterator<? extends DirectPosition> points) {
            boolean detectZeroSx = true;
            boolean detectZeroSy = true;
            boolean detectZeroZ0 = true;
            double sx = this.sx.doubleValue();
            double sy = this.sy.doubleValue();
            double z0 = this.z0.doubleValue();
            double mean_x = this.sum_x.doubleValue() / (double)this.n;
            double mean_y = this.sum_y.doubleValue() / (double)this.n;
            double mean_z = this.sum_z.doubleValue() / (double)this.n;
            double offset = Math.abs(sx * mean_x + sy * mean_y + z0);
            int index = 0;
            double sum_ds2 = 0.0;
            double sum_dz2 = 0.0;
            double sum_dsz = 0.0;
            while (true) {
                double z;
                double y;
                double x;
                if (vz != null) {
                    if (index >= length) break;
                    x = index % nx;
                    y = index / nx;
                    z = vz.doubleValue(index++);
                } else {
                    if (!points.hasNext()) break;
                    DirectPosition p = points.next();
                    x = p.getOrdinate(0);
                    y = p.getOrdinate(1);
                    z = p.getOrdinate(2);
                }
                x = (x - mean_x) * sx;
                y = (y - mean_y) * sy;
                double s = x + y;
                if (!Double.isNaN(s) && !Double.isNaN(z -= mean_z)) {
                    sum_ds2 += s * s;
                    sum_dz2 += z * z;
                    sum_dsz += s * z;
                }
                if (detectZeroSx && Math.abs(x) >= Math.ulp(y * 1.0E-14)) {
                    detectZeroSx = false;
                }
                if (detectZeroSy && Math.abs(y) >= Math.ulp(x * 1.0E-14)) {
                    detectZeroSy = false;
                }
                if (!detectZeroZ0 || !(offset >= Math.ulp(s * 1.0E-14))) continue;
                detectZeroZ0 = false;
            }
            if (detectZeroSx) {
                this.sx.clear();
            }
            if (detectZeroSy) {
                this.sy.clear();
            }
            if (detectZeroZ0) {
                this.z0.clear();
            }
            return Math.min(sum_dsz / Math.sqrt(sum_ds2 * sum_dz2), 1.0);
        }
    }
}

