/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.shiro.crypto;

/**
 * {@code CipherService} using the {@code Blowfish} cipher algorithm for all encryption, decryption, and key operations.
 * <p/>
 * The Blowfish algorithm can support key sizes between {@code 32} and {@code 448} bits<b>*</b>, inclusive.  However,
 * modern cryptanalysis techniques render keys of 80 bits or less mostly worthless - use {@code 128} or more whenever
 * possible.
 * <p/>
 * Note that this class retains the parent class's default {@link OperationMode#CFB CFB} mode of operation
 * instead of the typical JDK default of {@link OperationMode#ECB ECB}.  {@code ECB} should not be used in
 * security-sensitive environments because {@code ECB} does not allow for initialization vectors, which are
 * considered necessary for strong encryption.  See the {@link DefaultBlockCipherService parent class}'s JavaDoc and the
 * {@link JcaCipherService JcaCipherService} JavaDoc for more on why the JDK default should not be used and is not
 * used in this implementation.
 * <p/>
 * <b>*</b> Generating and using Blowfish key sizes greater than 128 require installation of the
 * <a href="http://java.sun.com/javase/downloads/index.jsp">Java Cryptography Extension (JCE) Unlimited Strength
 * Jurisdiction Policy files</a>.
 *
 * @since 1.0
 */
public class BlowfishCipherService extends DefaultBlockCipherService {

    private static final String ALGORITHM_NAME = "Blowfish";
    private static final int BLOCK_SIZE = 64;

    /**
     * Creates a new {@link CipherService} instance using the {@code Blowfish} cipher algorithm with the following
     * important cipher default attributes:
     * <table>
     * <tr>
     * <th>Attribute</th>
     * <th>Value</th>
     * </tr>
     * <tr>
     * <td>{@link #setKeySize keySize}</td>
     * <td>{@code 128} bits</td>
     * </tr>
     * <tr>
     * <td>{@link #setBlockSize blockSize}</td>
     * <td>{@code 64} bits (required for {@code Blowfish})</td>
     * </tr>
     * <tr>
     * <td>{@link #setMode mode}</td>
     * <td>{@link OperationMode#CFB CFB}<b>*</b></td>
     * </tr>
     * <tr>
     * <td>{@link #setPaddingScheme paddingScheme}</td>
     * <td>{@link PaddingScheme#PKCS5 PKCS5}</td>
     * </tr>
     * <tr>
     * <td>{@link #setInitializationVectorSize(int) initializationVectorSize}</td>
     * <td>{@code 64} bits</td>
     * </tr>
     * <tr>
     * <td>{@link #setGenerateInitializationVectors(boolean) generateInitializationVectors}</td>
     * <td>{@code true}<b>**</b></td>
     * </tr>
     * </table>
     * <p/>
     * <b>*</b> The {@link OperationMode#CFB CFB} operation mode is used instead of the JDK default {@code ECB} to
     * ensure strong encryption.  {@code ECB} should not be used in security-sensitive environments - see the
     * {@link DefaultBlockCipherService DefaultBlockCipherService} class JavaDoc's &quot;Operation Mode&quot; section
     * for more.
     * <p/>
     * <b>**</b>In conjunction with the default {@code CFB} operation mode, initialization vectors are generated by
     * default to ensure strong encryption.  See the {@link JcaCipherService JcaCipherService} class JavaDoc for more.
     */
    public BlowfishCipherService() {
        super(ALGORITHM_NAME);
        setInitializationVectorSize(BLOCK_SIZE); //like most block ciphers, the IV size is the same as the block size
    }
}
