/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.proxy.config.util;

import org.apache.shardingsphere.infra.config.datasource.DataSourceConfiguration;
import org.apache.shardingsphere.infra.config.datasource.DataSourceParameter;
import org.apache.shardingsphere.proxy.config.yaml.YamlDataSourceParameter;
import org.junit.Test;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;

public final class DataSourceParameterConverterTest {
    
    @Test
    public void assertGetDataSourceConfigurationMap() {
        Map<String, DataSourceParameter> dataSourceParameterMap = new HashMap<>(2, 1);
        dataSourceParameterMap.put("ds_0", crateDataSourceParameter());
        dataSourceParameterMap.put("ds_1", crateDataSourceParameter());
        Map<String, DataSourceConfiguration> actual = DataSourceParameterConverter.getDataSourceConfigurationMap(dataSourceParameterMap);
        assertThat(actual.size(), is(2));
        assertParameter(actual.get("ds_0"));
        assertParameter(actual.get("ds_1"));
    }
    
    private DataSourceParameter crateDataSourceParameter() {
        DataSourceParameter result = new DataSourceParameter();
        result.setUrl("jdbc:mysql://localhost:3306/demo_ds");
        result.setUsername("root");
        result.setPassword("root");
        return result;
    }
    
    private void assertParameter(final DataSourceConfiguration actual) {
        Map<String, Object> props = actual.getProps();
        assertThat(props.get("maximumPoolSize"), is(50));
        assertThat(props.get("minimumIdle"), is(1));
        assertThat(props.get("connectionTimeout"), is(30 * 1000L));
        assertThat(props.get("idleTimeout"), is(60 * 1000L));
        assertThat(props.get("maxLifetime"), is(30 * 60 * 1000L));
        assertThat(props.get("jdbcUrl"), is("jdbc:mysql://localhost:3306/demo_ds"));
        assertThat(props.get("username"), is("root"));
        assertThat(props.get("password"), is("root"));
    }
    
    @Test
    public void assertGetDataSourceParameterMapFromYamlConfiguration() {
        YamlDataSourceParameter yamlDataSourceParameter0 = new YamlDataSourceParameter();
        yamlDataSourceParameter0.setUrl("jdbc:mysql://localhost:3306/t_order");
        yamlDataSourceParameter0.setCustomPoolProps(getCustomPoolProps());
        setYamlDataSourceParameterPropertyWithoutUrl(yamlDataSourceParameter0);
        YamlDataSourceParameter yamlDataSourceParameter1 = new YamlDataSourceParameter();
        yamlDataSourceParameter1.setUrl("jdbc:mysql://localhost:3306/t_order_item");
        yamlDataSourceParameter1.setCustomPoolProps(getCustomPoolProps());
        setYamlDataSourceParameterPropertyWithoutUrl(yamlDataSourceParameter1);
        Map<String, YamlDataSourceParameter> yamlDataSourceParameterMap = new HashMap<>(2, 1);
        yamlDataSourceParameterMap.put("ds_0", yamlDataSourceParameter0);
        yamlDataSourceParameterMap.put("ds_1", yamlDataSourceParameter1);
        Map<String, DataSourceParameter> actualDataSourceParameters = DataSourceParameterConverter.getDataSourceParameterMapFromYamlConfiguration(yamlDataSourceParameterMap);
        assertThat(actualDataSourceParameters.size(), is(2));
        assertThat(actualDataSourceParameters.get("ds_0").getUrl(), is("jdbc:mysql://localhost:3306/t_order"));
        assertThat(actualDataSourceParameters.get("ds_1").getUrl(), is("jdbc:mysql://localhost:3306/t_order_item"));
        assertDataSourceParameter(actualDataSourceParameters.get("ds_0"));
        assertDataSourceParameter(actualDataSourceParameters.get("ds_1"));
    }
    
    private void setYamlDataSourceParameterPropertyWithoutUrl(final YamlDataSourceParameter yamlDataSourceParameter) {
        yamlDataSourceParameter.setMaxPoolSize(50);
        yamlDataSourceParameter.setMinPoolSize(1);
        yamlDataSourceParameter.setConnectionTimeoutMilliseconds(30 * 1000L);
        yamlDataSourceParameter.setIdleTimeoutMilliseconds(60 * 1000L);
        yamlDataSourceParameter.setMaxLifetimeMilliseconds(0L);
        yamlDataSourceParameter.setUsername("root");
        yamlDataSourceParameter.setPassword("root");
    }
    
    private void assertDataSourceParameter(final DataSourceParameter dataSourceParameter) {
        assertThat(dataSourceParameter.getMaxPoolSize(), is(50));
        assertThat(dataSourceParameter.getMinPoolSize(), is(1));
        assertThat(dataSourceParameter.getConnectionTimeoutMilliseconds(), is(30 * 1000L));
        assertThat(dataSourceParameter.getIdleTimeoutMilliseconds(), is(60 * 1000L));
        assertThat(dataSourceParameter.getMaxLifetimeMilliseconds(), is(0L));
        assertThat(dataSourceParameter.getUsername(), is("root"));
        assertThat(dataSourceParameter.getPassword(), is("root"));
        assertThat(dataSourceParameter.getCustomPoolProps().size(), is(2));
        assertThat(dataSourceParameter.getCustomPoolProps().get("maxPoolSize"), is(30));
        assertThat(dataSourceParameter.getCustomPoolProps().get("idleTimeoutMilliseconds"), is("30000"));
    }
    
    private Properties getCustomPoolProps() {
        Properties result = new Properties();
        result.put("maxPoolSize", 30);
        result.put("idleTimeoutMilliseconds", "30000");
        return result;
    }
}
