/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.servicecomb.transport.rest.client;

import static javax.ws.rs.core.Response.Status.BAD_REQUEST;
import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.catchThrowable;

import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;

import org.apache.servicecomb.swagger.invocation.Response;
import org.apache.servicecomb.swagger.invocation.exception.CommonExceptionData;
import org.junit.jupiter.api.Test;

import io.vertx.core.buffer.Buffer;
import io.vertx.core.json.Json;

class RestClientDecoderTest extends RestClientTestBase {
  static RestClientDecoder decoder = new RestClientDecoder();

  @Test
  void should_decode_by_json_when_no_content_type() {
    init("query", null, false);

    Response response = Response.ok(Buffer.buffer("\"result\""));
    decoder.decode(invocation, response);

    assertThat(response.<String>getResult()).isEqualTo("result");
  }

  @Test
  void should_decode_2xx_body() {
    init("query", null, false);

    Response response = Response.ok(Buffer.buffer("\"result\""))
        .setHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON);
    decoder.decode(invocation, response);

    assertThat(response.<String>getResult()).isEqualTo("result");
  }

  @Test
  void should_throw_exception_when_decode_invalid_2xx_body() {
    init("query", null, false);

    Response response = Response.ok(Buffer.buffer("result"))
        .setHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON);
    Throwable throwable = catchThrowable(() -> decoder.decode(invocation, response));

    assertThat(throwable.toString()).isEqualTo(
        "InvocationException: code=400;msg=CommonExceptionData{code='scb_rest_client.40000002', message='failed to decode success response body.', dynamic={}}");
  }

  @Test
  void should_decode_by_content_type_with_charset() {
    init("query", null, false);

    Response response = Response.ok(Buffer.buffer("\"result\""))
        .setHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON + ";charset=UTF-8");
    decoder.decode(invocation, response);

    assertThat(response.<String>getResult()).isEqualTo("result");
  }

  @Test
  void should_throw_exception_when_decode_not_2xx_response() {
    init("query", null, false);

    CommonExceptionData data = new CommonExceptionData("error");
    String json = Json.encodePrettily(data);
    Response response = Response
        .status(BAD_REQUEST)
        .entity(Buffer.buffer(json))
        .setHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON);

    Throwable throwable = catchThrowable(() -> decoder.decode(invocation, response));

    assertThat(throwable.toString()).isEqualTo("InvocationException: code=400;msg={message=error}");
  }

  @Test
  void should_throw_exception_when_decode_invalid_not_2xx_body() {
    init("query", null, false);

    Response response = Response
        .status(BAD_REQUEST)
        .entity(Buffer.buffer("result"))
        .setHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON);
    Throwable throwable = catchThrowable(() -> decoder.decode(invocation, response));

    assertThat(throwable.toString()).isEqualTo(
        "InvocationException: code=400;msg=CommonExceptionData{code='scb_rest_client.40000003', message='failed to decode fail response body.', dynamic={}}");
  }
}