/*
 * Decompiled with CFR 0.152.
 */
package org.apache.samza.system.kafka;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import kafka.common.TopicAndPartition;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.common.Metric;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.TopicPartition;
import org.apache.samza.Partition;
import org.apache.samza.SamzaException;
import org.apache.samza.system.IncomingMessageEnvelope;
import org.apache.samza.system.SystemStreamPartition;
import org.apache.samza.system.kafka.KafkaSystemConsumer;
import org.apache.samza.system.kafka.KafkaSystemConsumerMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class KafkaConsumerProxy<K, V> {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaConsumerProxy.class);
    private static final int SLEEP_MS_WHILE_NO_TOPIC_PARTITION = 100;
    final Thread consumerPollThread;
    private final Consumer<K, V> kafkaConsumer;
    private final KafkaSystemConsumer.KafkaConsumerMessageSink sink;
    private final KafkaSystemConsumerMetrics kafkaConsumerMetrics;
    private final String metricName;
    private final String systemName;
    private final String clientId;
    private final Map<TopicPartition, SystemStreamPartition> topicPartitionToSSP = new HashMap<TopicPartition, SystemStreamPartition>();
    private final Map<SystemStreamPartition, MetricName> perPartitionMetrics = new HashMap<SystemStreamPartition, MetricName>();
    private final Map<SystemStreamPartition, Long> nextOffsets = new ConcurrentHashMap<SystemStreamPartition, Long>();
    private final Map<SystemStreamPartition, Long> latestLags = new HashMap<SystemStreamPartition, Long>();
    private volatile boolean isRunning = false;
    private volatile Throwable failureCause = null;
    private final CountDownLatch consumerPollThreadStartLatch = new CountDownLatch(1);

    KafkaConsumerProxy(Consumer<K, V> kafkaConsumer, String systemName, String clientId, KafkaSystemConsumer.KafkaConsumerMessageSink messageSink, KafkaSystemConsumerMetrics samzaConsumerMetrics, String metricName) {
        this.kafkaConsumer = kafkaConsumer;
        this.systemName = systemName;
        this.sink = messageSink;
        this.kafkaConsumerMetrics = samzaConsumerMetrics;
        this.metricName = metricName;
        this.clientId = clientId;
        this.kafkaConsumerMetrics.registerClientProxy(metricName);
        this.consumerPollThread = new Thread(this.createProxyThreadRunnable());
        this.consumerPollThread.setDaemon(true);
        this.consumerPollThread.setName("Samza KafkaConsumerProxy Poll " + this.consumerPollThread.getName() + " - " + systemName);
        LOG.info("Creating KafkaConsumerProxy with systeName={}, clientId={}, metricsName={}", new Object[]{systemName, clientId, metricName});
    }

    public void addTopicPartition(SystemStreamPartition ssp, long nextOffset) {
        LOG.info(String.format("Adding new topicPartition %s with offset %s to queue for consumer %s", ssp, nextOffset, this));
        this.topicPartitionToSSP.put(KafkaSystemConsumer.toTopicPartition(ssp), ssp);
        this.nextOffsets.put(ssp, nextOffset);
        this.kafkaConsumerMetrics.setNumTopicPartitions(this.metricName, this.nextOffsets.size());
    }

    public void stop(long timeoutMs) {
        LOG.info("Shutting down KafkaConsumerProxy poll thread {} for {}", (Object)this.consumerPollThread.getName(), (Object)this);
        this.isRunning = false;
        try {
            this.consumerPollThread.join(timeoutMs / 2L);
            if (this.consumerPollThread.isAlive()) {
                this.consumerPollThread.interrupt();
                this.consumerPollThread.join(timeoutMs / 2L);
            }
        }
        catch (InterruptedException e) {
            LOG.warn("Join in KafkaConsumerProxy has failed", (Throwable)e);
            this.consumerPollThread.interrupt();
        }
    }

    public void start() {
        if (!this.consumerPollThread.isAlive()) {
            LOG.info("Starting KafkaConsumerProxy polling thread for " + this.toString());
            this.consumerPollThread.start();
            while (!this.isRunning && this.failureCause == null) {
                try {
                    this.consumerPollThreadStartLatch.await(3000L, TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException e) {
                    LOG.info("Ignoring InterruptedException while waiting for consumer poll thread to start.", (Throwable)e);
                }
            }
        } else {
            LOG.warn("Tried to start an already started KafkaConsumerProxy (%s). Ignoring.", (Object)this.toString());
        }
        if (this.topicPartitionToSSP.size() == 0) {
            String msg = String.format("Cannot start KafkaConsumerProxy without any registered TopicPartitions for %s", this.systemName);
            LOG.error(msg);
            throw new SamzaException(msg);
        }
    }

    boolean isRunning() {
        return this.isRunning;
    }

    Throwable getFailureCause() {
        return this.failureCause;
    }

    private void initializeLags() {
        Map endOffsets = this.kafkaConsumer.endOffsets(this.topicPartitionToSSP.keySet());
        endOffsets.forEach((tp, offset) -> {
            SystemStreamPartition ssp = this.topicPartitionToSSP.get(tp);
            long startingOffset = this.nextOffsets.get(ssp);
            long initialLag = (Long)endOffsets.get(tp) - startingOffset;
            LOG.info("Initial lag for SSP {} is {} (end={}, startOffset={})", new Object[]{ssp, initialLag, endOffsets.get(tp), startingOffset});
            this.latestLags.put(ssp, initialLag);
            this.sink.setIsAtHighWatermark(ssp, initialLag == 0L);
        });
        this.refreshLagMetrics();
    }

    private Runnable createProxyThreadRunnable() {
        Runnable runnable = () -> {
            this.isRunning = true;
            try {
                this.consumerPollThreadStartLatch.countDown();
                LOG.info("Starting consumer poll thread {} for system {}", (Object)this.consumerPollThread.getName(), (Object)this.systemName);
                this.initializeLags();
                while (this.isRunning) {
                    this.fetchMessages();
                }
            }
            catch (Throwable throwable) {
                LOG.error(String.format("Error in KafkaConsumerProxy poll thread for system: %s.", this.systemName), throwable);
                this.failureCause = throwable;
                this.isRunning = false;
            }
            if (!this.isRunning) {
                LOG.info("KafkaConsumerProxy for system {} has stopped.", (Object)this.systemName);
            }
        };
        return runnable;
    }

    private void fetchMessages() {
        HashSet<SystemStreamPartition> sspsToFetch = new HashSet<SystemStreamPartition>();
        for (SystemStreamPartition ssp : this.nextOffsets.keySet()) {
            if (!this.sink.needsMoreMessages(ssp)) continue;
            sspsToFetch.add(ssp);
        }
        LOG.debug("pollConsumer for {} SSPs: {}", (Object)sspsToFetch.size(), sspsToFetch);
        if (!sspsToFetch.isEmpty()) {
            this.kafkaConsumerMetrics.incClientReads(this.metricName);
            Map<SystemStreamPartition, List<IncomingMessageEnvelope>> response = this.pollConsumer(sspsToFetch, 500L);
            for (Map.Entry<SystemStreamPartition, List<IncomingMessageEnvelope>> e : response.entrySet()) {
                List<IncomingMessageEnvelope> envelopes = e.getValue();
                if (envelopes == null) continue;
                this.moveMessagesToTheirQueue(e.getKey(), envelopes);
            }
            this.populateCurrentLags(sspsToFetch);
        } else {
            LOG.debug("No topic/partitions need to be fetched for system {} right now. Sleeping {}ms.", (Object)this.systemName, (Object)100);
            this.kafkaConsumerMetrics.incClientSkippedFetchRequests(this.metricName);
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                LOG.warn("Sleep in fetchMessages was interrupted");
            }
        }
        this.refreshLagMetrics();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Map<SystemStreamPartition, List<IncomingMessageEnvelope>> pollConsumer(Set<SystemStreamPartition> systemStreamPartitions, long timeoutMs) {
        ConsumerRecords records;
        ArrayList<TopicPartition> topicPartitionsToPause = new ArrayList<TopicPartition>();
        ArrayList<TopicPartition> topicPartitionsToPoll = new ArrayList<TopicPartition>();
        for (Map.Entry<TopicPartition, SystemStreamPartition> e : this.topicPartitionToSSP.entrySet()) {
            TopicPartition tp = e.getKey();
            SystemStreamPartition ssp = e.getValue();
            if (systemStreamPartitions.contains(ssp)) {
                topicPartitionsToPoll.add(tp);
                continue;
            }
            topicPartitionsToPause.add(tp);
        }
        try {
            Map.Entry<TopicPartition, SystemStreamPartition> e;
            e = this.kafkaConsumer;
            synchronized (e) {
                this.kafkaConsumer.pause(topicPartitionsToPause);
                this.kafkaConsumer.resume(topicPartitionsToPoll);
                records = this.kafkaConsumer.poll(timeoutMs);
            }
        }
        catch (Exception e) {
            LOG.error("Caught a Kafka exception in pollConsumer for system " + this.systemName, (Throwable)e);
            throw e;
        }
        return this.processResults(records);
    }

    private Map<SystemStreamPartition, List<IncomingMessageEnvelope>> processResults(ConsumerRecords<K, V> records) {
        if (records == null) {
            throw new SamzaException("Received null 'records' after polling consumer in KafkaConsumerProxy " + this);
        }
        HashMap<SystemStreamPartition, List<IncomingMessageEnvelope>> results = new HashMap<SystemStreamPartition, List<IncomingMessageEnvelope>>(records.count());
        for (ConsumerRecord record : records) {
            int partition = record.partition();
            String topic = record.topic();
            TopicPartition tp = new TopicPartition(topic, partition);
            this.updateMetrics(record, tp);
            SystemStreamPartition ssp = this.topicPartitionToSSP.get(tp);
            ArrayList<IncomingMessageEnvelope> messages = (ArrayList<IncomingMessageEnvelope>)results.get(ssp);
            if (messages == null) {
                messages = new ArrayList<IncomingMessageEnvelope>();
                results.put(ssp, messages);
            }
            Object key = record.key();
            Object value = record.value();
            IncomingMessageEnvelope imEnvelope = new IncomingMessageEnvelope(ssp, String.valueOf(record.offset()), key, value, this.getRecordSize(record));
            messages.add(imEnvelope);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("# records per SSP:");
            for (Map.Entry e : results.entrySet()) {
                List list = (List)e.getValue();
                LOG.debug(e.getKey() + " = " + (list == null ? 0 : list.size()));
            }
        }
        return results;
    }

    private int getRecordSize(ConsumerRecord<K, V> r) {
        int keySize = r.key() == null ? 0 : r.serializedKeySize();
        return keySize + r.serializedValueSize();
    }

    private void updateMetrics(ConsumerRecord<K, V> r, TopicPartition tp) {
        TopicAndPartition tap = KafkaSystemConsumer.toTopicAndPartition(tp);
        SystemStreamPartition ssp = new SystemStreamPartition(this.systemName, tp.topic(), new Partition(tp.partition()));
        Long lag = this.latestLags.get(ssp);
        if (lag == null) {
            throw new SamzaException("Unknown/unregistered ssp in latestLags. ssp=" + ssp + "; system=" + this.systemName);
        }
        long currentSSPLag = lag;
        if (currentSSPLag < 0L) {
            return;
        }
        long recordOffset = r.offset();
        long highWatermark = recordOffset + currentSSPLag;
        int size = this.getRecordSize(r);
        this.kafkaConsumerMetrics.incReads(tap);
        this.kafkaConsumerMetrics.incBytesReads(tap, size);
        this.kafkaConsumerMetrics.setOffsets(tap, recordOffset);
        this.kafkaConsumerMetrics.incClientBytesReads(this.metricName, size);
        this.kafkaConsumerMetrics.setHighWatermarkValue(tap, highWatermark);
    }

    private void moveMessagesToTheirQueue(SystemStreamPartition ssp, List<IncomingMessageEnvelope> envelopes) {
        long nextOffset = this.nextOffsets.get(ssp);
        for (IncomingMessageEnvelope env : envelopes) {
            this.sink.addMessage(ssp, env);
            LOG.trace("IncomingMessageEnvelope. got envelope with offset:{} for ssp={}", (Object)env.getOffset(), (Object)ssp);
            nextOffset = Long.valueOf(env.getOffset()) + 1L;
        }
        this.nextOffsets.put(ssp, nextOffset);
    }

    private void populateCurrentLags(Set<SystemStreamPartition> ssps) {
        Map consumerMetrics = this.kafkaConsumer.metrics();
        if (this.perPartitionMetrics.isEmpty()) {
            HashMap<String, String> tags = new HashMap<String, String>();
            tags.put("client-id", this.clientId);
            for (SystemStreamPartition ssp : ssps) {
                TopicPartition tp = KafkaSystemConsumer.toTopicPartition(ssp);
                this.perPartitionMetrics.put(ssp, new MetricName(tp + ".records-lag", "consumer-fetch-manager-metrics", "", tags));
            }
        }
        for (SystemStreamPartition ssp : ssps) {
            MetricName mn = this.perPartitionMetrics.get(ssp);
            Metric currentLagMetric = (Metric)consumerMetrics.get(mn);
            long currentLag = currentLagMetric != null ? (long)currentLagMetric.value() : -1L;
            this.latestLags.put(ssp, currentLag);
            this.sink.setIsAtHighWatermark(ssp, currentLag == 0L);
        }
    }

    private void refreshLagMetrics() {
        for (Map.Entry<SystemStreamPartition, Long> e : this.nextOffsets.entrySet()) {
            SystemStreamPartition ssp = e.getKey();
            Long offset = e.getValue();
            TopicAndPartition tp = new TopicAndPartition(ssp.getStream(), ssp.getPartition().getPartitionId());
            Long lag = this.latestLags.get(ssp);
            LOG.trace("Latest offset of {} is  {}; lag = {}", new Object[]{ssp, offset, lag});
            if (lag == null || offset == null || lag < 0L) continue;
            long streamEndOffset = offset + lag;
            this.kafkaConsumerMetrics.setHighWatermarkValue(tp, streamEndOffset);
            this.kafkaConsumerMetrics.setLagValue(tp, lag);
        }
    }

    public String toString() {
        return String.format("consumerProxy-%s-%s", this.systemName, this.clientId);
    }
}

