/*
 * Decompiled with CFR 0.152.
 */
package org.apache.samza.zk;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.I0Itec.zkclient.IZkChildListener;
import org.I0Itec.zkclient.IZkDataListener;
import org.I0Itec.zkclient.ZkClient;
import org.I0Itec.zkclient.exception.ZkBadVersionException;
import org.I0Itec.zkclient.exception.ZkInterruptedException;
import org.I0Itec.zkclient.exception.ZkNodeExistsException;
import org.apache.commons.lang3.StringUtils;
import org.apache.samza.SamzaException;
import org.apache.samza.job.model.JobModel;
import org.apache.samza.metrics.MetricsRegistry;
import org.apache.samza.serializers.model.SamzaObjectMapper;
import org.apache.samza.zk.ProcessorData;
import org.apache.samza.zk.ZkBarrierForVersionUpgrade;
import org.apache.samza.zk.ZkKeyBuilder;
import org.apache.samza.zk.ZkUtilsMetrics;
import org.apache.zookeeper.data.Stat;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZkUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ZkUtils.class);
    static final String ZK_PROTOCOL_VERSION = "1.0";
    private final ZkClient zkClient;
    private volatile String ephemeralPath = null;
    private final ZkKeyBuilder keyBuilder;
    private final int connectionTimeoutMs;
    private final AtomicInteger currentGeneration;
    private final ZkUtilsMetrics metrics;
    private final int sessionTimeoutMs;

    public void incGeneration() {
        this.currentGeneration.incrementAndGet();
    }

    public int getGeneration() {
        return this.currentGeneration.get();
    }

    public ZkUtils(ZkKeyBuilder zkKeyBuilder, ZkClient zkClient, int connectionTimeoutMs, int sessionTimeOutMs, MetricsRegistry metricsRegistry) {
        this.keyBuilder = zkKeyBuilder;
        this.connectionTimeoutMs = connectionTimeoutMs;
        this.zkClient = zkClient;
        this.metrics = new ZkUtilsMetrics(metricsRegistry);
        this.currentGeneration = new AtomicInteger(0);
        this.sessionTimeoutMs = sessionTimeOutMs;
    }

    public void connect() throws ZkInterruptedException {
        boolean isConnected = this.zkClient.waitUntilConnected((long)this.connectionTimeoutMs, TimeUnit.MILLISECONDS);
        if (!isConnected) {
            this.metrics.zkConnectionError.inc();
            throw new RuntimeException("Unable to connect to Zookeeper within connectionTimeout " + this.connectionTimeoutMs + "ms. Shutting down!");
        }
    }

    public synchronized void unregister() {
        this.ephemeralPath = null;
    }

    public ZkClient getZkClient() {
        return this.zkClient;
    }

    public ZkKeyBuilder getKeyBuilder() {
        return this.keyBuilder;
    }

    public synchronized String registerProcessorAndGetId(ProcessorData data) {
        long startTimeMs = System.currentTimeMillis();
        long retryTimeOutMs = 2 * this.sessionTimeoutMs;
        String processorId = data.getProcessorId();
        if (this.ephemeralPath == null) {
            ProcessorNode processorNode;
            this.ephemeralPath = this.zkClient.createEphemeralSequential(this.keyBuilder.getProcessorsPath() + "/", (Object)data.toString());
            LOG.info("Created ephemeral path: {} for processor: {} in zookeeper.", (Object)this.ephemeralPath, (Object)data);
            while (!this.isValidRegisteredProcessor(processorNode = new ProcessorNode(data, this.ephemeralPath))) {
                long currentTimeMs = System.currentTimeMillis();
                if (currentTimeMs - startTimeMs < retryTimeOutMs) {
                    LOG.info("Processor: {} is duplicate. Retrying registration again.", (Object)processorId);
                    this.timeDelay(1000);
                    continue;
                }
                LOG.info("Processor: {} is duplicate. Deleting zookeeper node at path: {}.", (Object)processorId, (Object)this.ephemeralPath);
                this.zkClient.delete(this.ephemeralPath);
                this.metrics.deletions.inc();
                throw new SamzaException(String.format("Processor: %s is duplicate in the group. Registration failed.", processorId));
            }
        } else {
            LOG.info("Ephemeral path: {} exists for processor: {} in zookeeper.", (Object)this.ephemeralPath, (Object)data);
        }
        return this.ephemeralPath;
    }

    public void timeDelay(int sleepTimeInMillis) {
        try {
            Thread.sleep(sleepTimeInMillis);
        }
        catch (InterruptedException e) {
            LOG.error("Interrupted exception on wait.", (Throwable)e);
            Thread.interrupted();
        }
    }

    public synchronized void deleteProcessorNode(String processorId) {
        try {
            if (this.ephemeralPath != null) {
                LOG.info("Deleting the ephemeral node: {} of the processor: {} in zookeeper.", (Object)this.ephemeralPath, (Object)processorId);
                this.zkClient.delete(this.ephemeralPath);
            }
        }
        catch (Exception e) {
            LOG.error("Exception occurred on deletion of ephemeral node: {}.", (Object)this.ephemeralPath, (Object)e);
        }
    }

    private boolean isValidRegisteredProcessor(ProcessorNode processor) {
        String processorId = processor.getProcessorData().getProcessorId();
        List processorNodes = this.getAllProcessorNodes().stream().filter(processorNode -> ((ProcessorNode)processorNode).processorData.getProcessorId().equals(processorId)).collect(Collectors.toList());
        if (processorNodes.size() > 1) {
            LOG.debug("Processor nodes in zookeeper: {} for processorId: {}.", processorNodes, (Object)processorId);
            TreeSet sortedProcessorPaths = processorNodes.stream().map(ProcessorNode::getEphemeralPath).collect(Collectors.toCollection(TreeSet::new));
            return ((String)sortedProcessorPaths.first()).equals(processor.getEphemeralPath());
        }
        return true;
    }

    List<ProcessorNode> getAllProcessorNodes() {
        List<String> processorZNodes = this.getSortedActiveProcessorsZnodes();
        LOG.debug("Active ProcessorZNodes in zookeeper: {}.", processorZNodes);
        ArrayList<ProcessorNode> processorNodes = new ArrayList<ProcessorNode>();
        for (String processorZNode : processorZNodes) {
            String ephemeralProcessorPath = String.format("%s/%s", this.keyBuilder.getProcessorsPath(), processorZNode);
            String data = this.readProcessorData(ephemeralProcessorPath);
            if (data == null) continue;
            processorNodes.add(new ProcessorNode(new ProcessorData(data), ephemeralProcessorPath));
        }
        return processorNodes;
    }

    public List<String> getSortedActiveProcessorsZnodes() {
        List znodeIds = this.zkClient.getChildren(this.keyBuilder.getProcessorsPath());
        if (znodeIds.size() > 0) {
            Collections.sort(znodeIds);
            LOG.info("Found these children - " + znodeIds);
        }
        return znodeIds;
    }

    private String readProcessorData(String fullPath) {
        try {
            String data = (String)this.zkClient.readData(fullPath, true);
            this.metrics.reads.inc();
            return data;
        }
        catch (Exception e) {
            throw new SamzaException(String.format("Cannot read ZK node: %s", fullPath), (Throwable)e);
        }
    }

    public List<String> getSortedActiveProcessorsIDs() {
        return this.getActiveProcessorsIDs(this.getSortedActiveProcessorsZnodes());
    }

    public List<String> getActiveProcessorsIDs(List<String> znodeIds) {
        String processorPath = this.keyBuilder.getProcessorsPath();
        ArrayList<String> processorIds = new ArrayList<String>(znodeIds.size());
        if (znodeIds.size() > 0) {
            for (String child : znodeIds) {
                String fullPath = String.format("%s/%s", processorPath, child);
                String processorData = this.readProcessorData(fullPath);
                if (processorData == null) continue;
                processorIds.add(new ProcessorData(processorData).getProcessorId());
            }
            Collections.sort(processorIds);
            LOG.info("Found these children - " + znodeIds);
            LOG.info("Found these processorIds - " + processorIds);
        }
        return processorIds;
    }

    public void unsubscribeDataChanges(String path, IZkDataListener dataListener) {
        this.zkClient.unsubscribeDataChanges(path, dataListener);
    }

    public void subscribeDataChanges(String path, IZkDataListener dataListener) {
        this.zkClient.subscribeDataChanges(path, dataListener);
        this.metrics.subscriptions.inc();
    }

    public void subscribeChildChanges(String path, IZkChildListener listener) {
        this.zkClient.subscribeChildChanges(path, listener);
        this.metrics.subscriptions.inc();
    }

    public void unsubscribeChildChanges(String path, IZkChildListener childListener) {
        this.zkClient.unsubscribeChildChanges(path, childListener);
    }

    public void writeData(String path, Object object) {
        this.zkClient.writeData(path, object);
        this.metrics.writes.inc();
    }

    public boolean exists(String path) {
        return this.zkClient.exists(path);
    }

    public void close() {
        try {
            this.zkClient.close();
        }
        catch (ZkInterruptedException e) {
            LOG.warn("Interrupted when closing zkClient. Clearing the interrupted status and retrying.", (Throwable)e);
            Thread.interrupted();
            this.zkClient.close();
            Thread.currentThread().interrupt();
        }
    }

    public void subscribeToJobModelVersionChange(GenerationAwareZkDataListener dataListener) {
        LOG.info(" subscribing for jm version change at:" + this.keyBuilder.getJobModelVersionPath());
        this.zkClient.subscribeDataChanges(this.keyBuilder.getJobModelVersionPath(), (IZkDataListener)dataListener);
        this.metrics.subscriptions.inc();
    }

    public void publishJobModel(String jobModelVersion, JobModel jobModel) {
        try {
            ObjectMapper mmapper = SamzaObjectMapper.getObjectMapper();
            String jobModelStr = mmapper.writerWithDefaultPrettyPrinter().writeValueAsString((Object)jobModel);
            LOG.info("jobModelAsString=" + jobModelStr);
            this.zkClient.createPersistent(this.keyBuilder.getJobModelPath(jobModelVersion), (Object)jobModelStr);
            LOG.info("wrote jobModel path =" + this.keyBuilder.getJobModelPath(jobModelVersion));
        }
        catch (Exception e) {
            LOG.error("JobModel publish failed for version=" + jobModelVersion, (Throwable)e);
            throw new SamzaException((Throwable)e);
        }
    }

    public JobModel getJobModel(String jobModelVersion) {
        JobModel jm;
        LOG.info("Read the model ver=" + jobModelVersion + " from " + this.keyBuilder.getJobModelPath(jobModelVersion));
        Object data = this.zkClient.readData(this.keyBuilder.getJobModelPath(jobModelVersion));
        this.metrics.reads.inc();
        ObjectMapper mmapper = SamzaObjectMapper.getObjectMapper();
        try {
            jm = (JobModel)mmapper.readValue((String)data, JobModel.class);
        }
        catch (IOException e) {
            throw new SamzaException("failed to read JobModel from ZK", (Throwable)e);
        }
        return jm;
    }

    public String getJobModelVersion() {
        String jobModelVersion = (String)this.zkClient.readData(this.keyBuilder.getJobModelVersionPath(), true);
        this.metrics.reads.inc();
        return jobModelVersion;
    }

    public String getNextJobModelVersion(String currentJobModelVersion) {
        if (currentJobModelVersion == null) {
            return "1";
        }
        List publishedJobModelVersions = this.zkClient.getChildren(this.keyBuilder.getJobModelPathPrefix());
        this.metrics.reads.inc((long)publishedJobModelVersions.size());
        String maxPublishedJMVersion = publishedJobModelVersions.stream().max(Comparator.comparingInt(Integer::valueOf)).orElse("0");
        return Integer.toString(Math.max(Integer.valueOf(currentJobModelVersion), Integer.valueOf(maxPublishedJMVersion)) + 1);
    }

    public void publishJobModelVersion(String oldVersion, String newVersion) {
        Stat stat = new Stat();
        String currentVersion = (String)this.zkClient.readData(this.keyBuilder.getJobModelVersionPath(), stat);
        this.metrics.reads.inc();
        LOG.info("publishing new version: " + newVersion + "; oldVersion = " + oldVersion + "(" + stat.getVersion() + ")");
        if (currentVersion != null && !currentVersion.equals(oldVersion)) {
            throw new SamzaException("Someone changed JobModelVersion while the leader was generating one: expected" + oldVersion + ", got " + currentVersion);
        }
        int dataVersion = stat.getVersion();
        try {
            stat = this.zkClient.writeDataReturnStat(this.keyBuilder.getJobModelVersionPath(), (Object)newVersion, dataVersion);
            this.metrics.writes.inc();
        }
        catch (Exception e) {
            String msg = "publish job model version failed for new version = " + newVersion + "; old version = " + oldVersion;
            LOG.error(msg, (Throwable)e);
            throw new SamzaException(msg, (Throwable)e);
        }
        LOG.info("published new version: " + newVersion + "; expected data version = " + (dataVersion + 1) + "(actual data version after update = " + stat.getVersion() + ")");
    }

    public void validateZkVersion() {
        Stat stat;
        String version;
        String rootPath = this.keyBuilder.getRootPath();
        if (!this.zkClient.exists(rootPath)) {
            try {
                this.zkClient.createPersistent(rootPath, (Object)ZK_PROTOCOL_VERSION);
                LOG.info("Created zk root node: " + rootPath + " with zk version " + ZK_PROTOCOL_VERSION);
                return;
            }
            catch (ZkNodeExistsException e) {
                LOG.warn("root path " + rootPath + " already exists.");
            }
        }
        if ((version = (String)this.zkClient.readData(rootPath, stat = new Stat())) == null) {
            try {
                this.zkClient.writeData(rootPath, (Object)ZK_PROTOCOL_VERSION, stat.getVersion());
            }
            catch (ZkBadVersionException zkBadVersionException) {
                // empty catch block
            }
            version = (String)this.zkClient.readData(rootPath);
        }
        LOG.info("Current version for zk root node: " + rootPath + " is " + version + ", expected version is " + ZK_PROTOCOL_VERSION);
        if (!version.equals(ZK_PROTOCOL_VERSION)) {
            throw new SamzaException("ZK Protocol mismatch. Expected 1.0; found " + version);
        }
    }

    public void validatePaths(String[] paths) {
        for (String path : paths) {
            if (this.zkClient.exists(path)) continue;
            this.zkClient.createPersistent(path, true);
        }
    }

    public void subscribeToProcessorChange(IZkChildListener listener) {
        LOG.info("Subscribing for child change at:" + this.keyBuilder.getProcessorsPath());
        this.zkClient.subscribeChildChanges(this.keyBuilder.getProcessorsPath(), listener);
        this.metrics.subscriptions.inc();
    }

    public void cleanupZK(int numVersionsToLeave) {
        this.deleteOldBarrierVersions(numVersionsToLeave);
        this.deleteOldJobModels(numVersionsToLeave);
    }

    void deleteOldJobModels(int numVersionsToLeave) {
        String path = this.keyBuilder.getJobModelPathPrefix();
        LOG.info("About to delete jm path=" + path);
        List znodeIds = this.zkClient.getChildren(path);
        this.deleteOldVersionPath(path, znodeIds, numVersionsToLeave, new Comparator<String>(){

            @Override
            public int compare(String o1, String o2) {
                return Integer.valueOf(o1) - Integer.valueOf(o2);
            }
        });
    }

    void deleteOldBarrierVersions(int numVersionsToLeave) {
        String path = this.keyBuilder.getJobModelVersionBarrierPrefix();
        LOG.info("About to delete old barrier paths from " + path);
        List znodeIds = this.zkClient.getChildren(path);
        LOG.info("List of all zkNodes: " + znodeIds);
        this.deleteOldVersionPath(path, znodeIds, numVersionsToLeave, new Comparator<String>(){

            @Override
            public int compare(String o1, String o2) {
                return ZkBarrierForVersionUpgrade.getVersion(o1) - ZkBarrierForVersionUpgrade.getVersion(o2);
            }
        });
    }

    void deleteOldVersionPath(String path, List<String> zNodeIds, int numVersionsToLeave, Comparator<String> c) {
        if (StringUtils.isEmpty((CharSequence)path) || zNodeIds == null) {
            LOG.warn("cannot cleanup empty path or empty list in ZK");
            return;
        }
        if (zNodeIds.size() > numVersionsToLeave) {
            Collections.sort(zNodeIds, c);
            int size = zNodeIds.size();
            List<String> zNodesToDelete = zNodeIds.subList(0, zNodeIds.size() - numVersionsToLeave);
            LOG.info("Starting cleanup of barrier version zkNodes. From size=" + size + " to size " + zNodesToDelete.size() + "; numberToLeave=" + numVersionsToLeave);
            for (String znodeId : zNodesToDelete) {
                String pathToDelete = path + "/" + znodeId;
                try {
                    LOG.info("deleting " + pathToDelete);
                    this.zkClient.deleteRecursive(pathToDelete);
                    this.metrics.deletions.inc();
                }
                catch (Exception e) {
                    LOG.warn("delete of node " + pathToDelete + " failed.", (Throwable)e);
                }
            }
        }
    }

    static class ProcessorNode {
        private final ProcessorData processorData;
        private final String ephemeralProcessorPath;

        ProcessorNode(ProcessorData processorData, String ephemeralProcessorPath) {
            this.processorData = processorData;
            this.ephemeralProcessorPath = ephemeralProcessorPath;
        }

        ProcessorData getProcessorData() {
            return this.processorData;
        }

        String getEphemeralPath() {
            return this.ephemeralProcessorPath;
        }

        public String toString() {
            return String.format("[ProcessorData: %s, ephemeralProcessorPath: %s]", this.processorData, this.ephemeralProcessorPath);
        }

        public int hashCode() {
            return Objects.hash(this.processorData, this.ephemeralProcessorPath);
        }

        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            ProcessorNode other = (ProcessorNode)obj;
            return Objects.equals(this.processorData, other.processorData) && Objects.equals(this.ephemeralProcessorPath, other.ephemeralProcessorPath);
        }
    }

    public static abstract class GenerationAwareZkDataListener
    implements IZkDataListener {
        private final int generation;
        private final ZkUtils zkUtils;
        private final String listenerName;

        public GenerationAwareZkDataListener(ZkUtils zkUtils, String listenerName) {
            this.generation = zkUtils.getGeneration();
            this.zkUtils = zkUtils;
            this.listenerName = listenerName;
        }

        public void handleDataChange(String path, Object data) {
            if (!this.isValid()) {
                LOG.warn(String.format("Skipping handleDataChange for %s from wrong generation. Current generation: %s; Callback generation: %s", this.listenerName, this.zkUtils.getGeneration(), this.generation));
            } else {
                this.doHandleDataChange(path, data);
            }
        }

        public void handleDataDeleted(String dataPath) throws Exception {
            if (!this.isValid()) {
                LOG.warn(String.format("Skipping handleDataDeleted for %s from wrong generation. Current generation: %s; Callback generation: %s", this.listenerName, this.zkUtils.getGeneration(), this.generation));
            } else {
                this.doHandleDataDeleted(dataPath);
            }
        }

        public abstract void doHandleDataChange(String var1, Object var2);

        public abstract void doHandleDataDeleted(String var1);

        private boolean isValid() {
            return this.generation == this.zkUtils.getGeneration();
        }
    }

    public static abstract class GenerationAwareZkChildListener
    implements IZkChildListener {
        private final int generation;
        private final ZkUtils zkUtils;
        private final String listenerName;

        public GenerationAwareZkChildListener(ZkUtils zkUtils, String listenerName) {
            this.generation = zkUtils.getGeneration();
            this.zkUtils = zkUtils;
            this.listenerName = listenerName;
        }

        public void handleChildChange(String barrierParticipantPath, List<String> participantIds) throws Exception {
            int currentGeneration = this.zkUtils.getGeneration();
            if (currentGeneration != this.generation) {
                LOG.warn(String.format("Skipping handleChildChange for %s from wrong generation. Current generation: %s; Callback generation: %s", this.listenerName, currentGeneration, this.generation));
                return;
            }
            this.doHandleChildChange(barrierParticipantPath, participantIds);
        }

        public abstract void doHandleChildChange(String var1, List<String> var2) throws Exception;
    }
}

