/*
 * Decompiled with CFR 0.152.
 */
package org.apache.samza.zk;

import com.google.common.annotations.VisibleForTesting;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import org.I0Itec.zkclient.IZkStateListener;
import org.apache.commons.lang3.StringUtils;
import org.apache.samza.checkpoint.CheckpointManager;
import org.apache.samza.config.ApplicationConfig;
import org.apache.samza.config.Config;
import org.apache.samza.config.ConfigException;
import org.apache.samza.config.JobConfig;
import org.apache.samza.config.MapConfig;
import org.apache.samza.config.MetricsConfig;
import org.apache.samza.config.TaskConfigJava;
import org.apache.samza.config.ZkConfig;
import org.apache.samza.container.TaskName;
import org.apache.samza.coordinator.JobCoordinator;
import org.apache.samza.coordinator.JobCoordinatorListener;
import org.apache.samza.coordinator.JobModelManager;
import org.apache.samza.coordinator.LeaderElectorListener;
import org.apache.samza.job.model.ContainerModel;
import org.apache.samza.job.model.JobModel;
import org.apache.samza.metrics.MetricsRegistry;
import org.apache.samza.metrics.MetricsReporter;
import org.apache.samza.metrics.ReadableMetricsRegistry;
import org.apache.samza.runtime.ProcessorIdGenerator;
import org.apache.samza.storage.ChangelogStreamManager;
import org.apache.samza.system.StreamMetadataCache;
import org.apache.samza.system.SystemAdmins;
import org.apache.samza.util.MetricsReporterLoader;
import org.apache.samza.util.SystemClock;
import org.apache.samza.util.Util;
import org.apache.samza.zk.ScheduleAfterDebounceTime;
import org.apache.samza.zk.ZkBarrierForVersionUpgrade;
import org.apache.samza.zk.ZkBarrierListener;
import org.apache.samza.zk.ZkJobCoordinatorMetrics;
import org.apache.samza.zk.ZkKeyBuilder;
import org.apache.samza.zk.ZkLeaderElector;
import org.apache.samza.zk.ZkUtils;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZkJobCoordinator
implements JobCoordinator {
    private static final Logger LOG = LoggerFactory.getLogger(ZkJobCoordinator.class);
    private static final int METADATA_CACHE_TTL_MS = 5000;
    private static final int NUM_VERSIONS_TO_LEAVE = 10;
    private static final String JOB_MODEL_VERSION_CHANGE = "JobModelVersionChange";
    private static final String ON_PROCESSOR_CHANGE = "OnProcessorChange";
    private static final String ON_ZK_CLEANUP = "OnCleanUp";
    private final ZkUtils zkUtils;
    private final String processorId;
    private final Config config;
    private final ZkBarrierForVersionUpgrade barrier;
    private final ZkJobCoordinatorMetrics metrics;
    private final Map<String, MetricsReporter> reporters;
    private final ZkLeaderElector leaderElector;
    private final AtomicBoolean initiatedShutdown = new AtomicBoolean(false);
    private final StreamMetadataCache streamMetadataCache;
    private final SystemAdmins systemAdmins;
    private final int debounceTimeMs;
    private final Map<TaskName, Integer> changeLogPartitionMap = new HashMap<TaskName, Integer>();
    private JobCoordinatorListener coordinatorListener = null;
    private JobModel newJobModel;
    private boolean hasCreatedStreams = false;
    private String cachedJobModelVersion = null;
    @VisibleForTesting
    ScheduleAfterDebounceTime debounceTimer;

    ZkJobCoordinator(Config config, MetricsRegistry metricsRegistry, ZkUtils zkUtils) {
        this.config = config;
        this.metrics = new ZkJobCoordinatorMetrics(metricsRegistry);
        this.processorId = this.createProcessorId(config);
        this.zkUtils = zkUtils;
        zkUtils.getZkClient().subscribeStateChanges((IZkStateListener)new ZkSessionStateChangedListener());
        this.leaderElector = new ZkLeaderElector(this.processorId, zkUtils);
        this.leaderElector.setLeaderElectorListener(new LeaderElectorListenerImpl());
        this.debounceTimeMs = new JobConfig(config).getDebounceTimeMs();
        this.reporters = MetricsReporterLoader.getMetricsReporters(new MetricsConfig(config), this.processorId);
        this.debounceTimer = new ScheduleAfterDebounceTime(this.processorId);
        this.debounceTimer.setScheduledTaskCallback(throwable -> {
            LOG.error("Received exception in debounce timer! Stopping the job coordinator", throwable);
            this.stop();
        });
        this.barrier = new ZkBarrierForVersionUpgrade(zkUtils.getKeyBuilder().getJobModelVersionBarrierPrefix(), zkUtils, new ZkBarrierListenerImpl(), this.debounceTimer);
        this.systemAdmins = new SystemAdmins(config);
        this.streamMetadataCache = new StreamMetadataCache(this.systemAdmins, 5000, SystemClock.instance());
    }

    @Override
    public void start() {
        ZkKeyBuilder keyBuilder = this.zkUtils.getKeyBuilder();
        this.zkUtils.validateZkVersion();
        this.zkUtils.validatePaths(new String[]{keyBuilder.getProcessorsPath(), keyBuilder.getJobModelVersionPath(), keyBuilder.getJobModelPathPrefix()});
        this.startMetrics();
        this.systemAdmins.start();
        this.leaderElector.tryBecomeLeader();
        this.zkUtils.subscribeToJobModelVersionChange(new ZkJobModelVersionChangeHandler(this.zkUtils));
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void stop() {
        if (this.initiatedShutdown.compareAndSet(false, true)) {
            LOG.info("Shutting down JobCoordinator.");
            boolean shutdownSuccessful = false;
            this.metrics.isLeader.set((Object)false);
            try {
                if (this.coordinatorListener != null) {
                    this.coordinatorListener.onJobModelExpired();
                }
                this.debounceTimer.stopScheduler();
                if (this.leaderElector.amILeader()) {
                    LOG.info("Resigning leadership for processorId: " + this.processorId);
                    this.leaderElector.resignLeadership();
                }
                LOG.info("Shutting down ZkUtils.");
                if (this.zkUtils != null) {
                    this.zkUtils.close();
                }
                LOG.debug("Shutting down system admins.");
                this.systemAdmins.stop();
                LOG.debug("Shutting down metrics.");
                this.shutdownMetrics();
                if (this.coordinatorListener != null) {
                    this.coordinatorListener.onCoordinatorStop();
                }
                shutdownSuccessful = true;
                return;
            }
            catch (Throwable t) {
                LOG.error("Encountered errors during job coordinator stop.", t);
                if (this.coordinatorListener == null) return;
                this.coordinatorListener.onCoordinatorFailure(t);
                return;
            }
            finally {
                LOG.info("Job Coordinator shutdown finished with ShutdownComplete=" + shutdownSuccessful);
            }
        } else {
            LOG.info("Job Coordinator shutdown is in progress!");
        }
    }

    private void startMetrics() {
        for (MetricsReporter reporter : this.reporters.values()) {
            reporter.register("job-coordinator-" + this.processorId, (ReadableMetricsRegistry)this.metrics.getMetricsRegistry());
            reporter.start();
        }
    }

    private void shutdownMetrics() {
        for (MetricsReporter reporter : this.reporters.values()) {
            reporter.stop();
        }
    }

    @Override
    public void setListener(JobCoordinatorListener listener) {
        this.coordinatorListener = listener;
    }

    @Override
    public JobModel getJobModel() {
        return this.newJobModel;
    }

    @Override
    public String getProcessorId() {
        return this.processorId;
    }

    public void onProcessorChange(List<String> processors) {
        if (this.leaderElector.amILeader()) {
            LOG.info("ZkJobCoordinator::onProcessorChange - list of processors changed. List size=" + processors.size());
            this.debounceTimer.scheduleAfterDebounceTime(ON_PROCESSOR_CHANGE, this.debounceTimeMs, () -> this.doOnProcessorChange(processors));
        }
    }

    void doOnProcessorChange(List<String> processors) {
        List<String> currentProcessorIds = this.zkUtils.getSortedActiveProcessorsIDs();
        HashSet<String> uniqueProcessorIds = new HashSet<String>(currentProcessorIds);
        if (currentProcessorIds.size() != uniqueProcessorIds.size()) {
            LOG.info("Processors: {} has duplicates. Not generating JobModel.", currentProcessorIds);
            return;
        }
        LOG.info("Generating new JobModel with processors: {}.", currentProcessorIds);
        JobModel jobModel = this.generateNewJobModel(currentProcessorIds);
        if (!this.hasCreatedStreams) {
            CheckpointManager checkpointManager = new TaskConfigJava(this.config).getCheckpointManager(this.metrics.getMetricsRegistry());
            if (checkpointManager != null) {
                checkpointManager.createResources();
            }
            ChangelogStreamManager.createChangelogStreams(this.config, jobModel.maxChangeLogStreamPartitions);
            this.hasCreatedStreams = true;
        }
        String currentJMVersion = this.zkUtils.getJobModelVersion();
        String nextJMVersion = this.zkUtils.getNextJobModelVersion(currentJMVersion);
        LOG.info("pid=" + this.processorId + "Generated new JobModel with version: " + nextJMVersion + " and processors: " + currentProcessorIds);
        this.zkUtils.publishJobModel(nextJMVersion, jobModel);
        this.barrier.create(nextJMVersion, currentProcessorIds);
        this.zkUtils.publishJobModelVersion(currentJMVersion, nextJMVersion);
        LOG.info("pid=" + this.processorId + "Published new Job Model. Version = " + nextJMVersion);
        this.debounceTimer.scheduleAfterDebounceTime(ON_ZK_CLEANUP, 0L, () -> this.zkUtils.cleanupZK(10));
    }

    private String createProcessorId(Config config) {
        ApplicationConfig appConfig = new ApplicationConfig(config);
        if (appConfig.getProcessorId() != null) {
            return appConfig.getProcessorId();
        }
        if (StringUtils.isNotBlank((CharSequence)appConfig.getAppProcessorIdGeneratorClass())) {
            ProcessorIdGenerator idGenerator = Util.getObj(appConfig.getAppProcessorIdGeneratorClass(), ProcessorIdGenerator.class);
            return idGenerator.generateProcessorId(config);
        }
        throw new ConfigException(String.format("Expected either %s or %s to be configured", "processor.id", "app.processor-id-generator.class"));
    }

    private JobModel generateNewJobModel(List<String> processors) {
        String zkJobModelVersion = this.zkUtils.getJobModelVersion();
        if (zkJobModelVersion != null && !Objects.equals(this.cachedJobModelVersion, zkJobModelVersion)) {
            JobModel jobModel = this.zkUtils.getJobModel(zkJobModelVersion);
            for (ContainerModel containerModel : jobModel.getContainers().values()) {
                containerModel.getTasks().forEach((taskName, taskModel) -> this.changeLogPartitionMap.put((TaskName)taskName, taskModel.getChangelogPartition().getPartitionId()));
            }
            this.cachedJobModelVersion = zkJobModelVersion;
        }
        JobModel model = JobModelManager.readJobModel(this.config, this.changeLogPartitionMap, null, this.streamMetadataCache, processors);
        return new JobModel((Config)new MapConfig(), model.getContainers());
    }

    @VisibleForTesting
    public ZkUtils getZkUtils() {
        return this.zkUtils;
    }

    @VisibleForTesting
    class ZkSessionStateChangedListener
    implements IZkStateListener {
        private static final String ZK_SESSION_ERROR = "ZK_SESSION_ERROR";
        private static final String ZK_SESSION_EXPIRED = "ZK_SESSION_EXPIRED";

        ZkSessionStateChangedListener() {
        }

        public void handleStateChanged(Watcher.Event.KeeperState state) {
            switch (state) {
                case Expired: {
                    LOG.warn("Got " + state.toString() + " event for processor=" + ZkJobCoordinator.this.processorId + ". Stopping the container and unregister the processor node.");
                    ZkJobCoordinator.this.zkUtils.incGeneration();
                    ZkJobCoordinator.this.zkUtils.unregister();
                    if (ZkJobCoordinator.this.leaderElector.amILeader()) {
                        ZkJobCoordinator.this.leaderElector.resignLeadership();
                    }
                    LOG.info("Cancelling all scheduled actions in session expiration for processorId: {}.", (Object)ZkJobCoordinator.this.processorId);
                    ZkJobCoordinator.this.debounceTimer.cancelAllScheduledActions();
                    ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime(ZK_SESSION_EXPIRED, 0L, () -> {
                        if (ZkJobCoordinator.this.coordinatorListener != null) {
                            ZkJobCoordinator.this.coordinatorListener.onJobModelExpired();
                        }
                    });
                    return;
                }
                case Disconnected: {
                    LOG.warn("Got " + state.toString() + " event for processor=" + ZkJobCoordinator.this.processorId + ". Scheduling a coordinator stop.");
                    ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime(ZK_SESSION_ERROR, new ZkConfig(ZkJobCoordinator.this.config).getZkSessionTimeoutMs(), () -> ZkJobCoordinator.this.stop());
                    return;
                }
                case AuthFailed: 
                case NoSyncConnected: 
                case Unknown: {
                    LOG.warn("Got unexpected failure event " + state.toString() + " for processor=" + ZkJobCoordinator.this.processorId + ". Stopping the job coordinator.");
                    ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime(ZK_SESSION_ERROR, 0L, () -> ZkJobCoordinator.this.stop());
                    return;
                }
                case SyncConnected: {
                    LOG.info("Got syncconnected event for processor=" + ZkJobCoordinator.this.processorId + ".");
                    ZkJobCoordinator.this.debounceTimer.cancelAction(ZK_SESSION_ERROR);
                    return;
                }
            }
            LOG.info("Got ZK event " + state.toString() + " for processor=" + ZkJobCoordinator.this.processorId + ". Continue");
        }

        public void handleNewSession() {
            LOG.info("Got new session created event for processor=" + ZkJobCoordinator.this.processorId);
            ZkJobCoordinator.this.debounceTimer.cancelAllScheduledActions();
            LOG.info("register zk controller for the new session");
            ZkJobCoordinator.this.leaderElector.tryBecomeLeader();
            ZkJobCoordinator.this.zkUtils.subscribeToJobModelVersionChange(new ZkJobModelVersionChangeHandler(ZkJobCoordinator.this.zkUtils));
        }

        public void handleSessionEstablishmentError(Throwable error) {
            LOG.info("handleSessionEstablishmentError received for processor=" + ZkJobCoordinator.this.processorId, error);
            ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime(ZK_SESSION_ERROR, 0L, () -> ZkJobCoordinator.this.stop());
        }
    }

    class ZkJobModelVersionChangeHandler
    extends ZkUtils.GenerationAwareZkDataListener {
        public ZkJobModelVersionChangeHandler(ZkUtils zkUtils) {
            super(zkUtils, "ZkJobModelVersionChangeHandler");
        }

        @Override
        public void doHandleDataChange(String dataPath, Object data) {
            ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime(ZkJobCoordinator.JOB_MODEL_VERSION_CHANGE, 0L, () -> {
                String jobModelVersion = (String)data;
                LOG.info("Got a notification for new JobModel version. Path = {} Version = {}", (Object)dataPath, data);
                ZkJobCoordinator.this.newJobModel = ZkJobCoordinator.this.zkUtils.getJobModel(jobModelVersion);
                LOG.info("pid=" + ZkJobCoordinator.this.processorId + ": new JobModel is available. Version =" + jobModelVersion + "; JobModel = " + ZkJobCoordinator.this.newJobModel);
                if (!ZkJobCoordinator.this.newJobModel.getContainers().containsKey(ZkJobCoordinator.this.processorId)) {
                    LOG.info("New JobModel does not contain pid={}. Stopping this processor. New JobModel: {}", (Object)ZkJobCoordinator.this.processorId, (Object)ZkJobCoordinator.this.newJobModel);
                    ZkJobCoordinator.this.stop();
                } else {
                    if (ZkJobCoordinator.this.coordinatorListener != null) {
                        ZkJobCoordinator.this.coordinatorListener.onJobModelExpired();
                    }
                    ZkJobCoordinator.this.barrier.join(jobModelVersion, ZkJobCoordinator.this.processorId);
                }
            });
        }

        @Override
        public void doHandleDataDeleted(String dataPath) {
            LOG.warn("JobModel version z-node has been deleted. Shutting down the coordinator" + dataPath);
            ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime("JOB_MODEL_VERSION_DELETED", 0L, () -> ZkJobCoordinator.this.stop());
        }
    }

    class ProcessorChangeHandler
    extends ZkUtils.GenerationAwareZkChildListener {
        public ProcessorChangeHandler(ZkUtils zkUtils) {
            super(zkUtils, "ProcessorChangeHandler");
        }

        @Override
        public void doHandleChildChange(String parentPath, List<String> currentChildren) throws Exception {
            if (currentChildren == null) {
                LOG.info("handleChildChange on path " + parentPath + " was invoked with NULL list of children");
            } else {
                LOG.info("ProcessorChangeHandler::handleChildChange - Path: {} Current Children: {} ", (Object)parentPath, currentChildren);
                ZkJobCoordinator.this.onProcessorChange(currentChildren);
            }
        }
    }

    class ZkBarrierListenerImpl
    implements ZkBarrierListener {
        private final String barrierAction = "BarrierAction";
        private long startTime = 0L;

        ZkBarrierListenerImpl() {
        }

        @Override
        public void onBarrierCreated(String version) {
            this.startTime = System.nanoTime();
            ((ZkJobCoordinator)ZkJobCoordinator.this).metrics.barrierCreation.inc();
            if (ZkJobCoordinator.this.leaderElector.amILeader()) {
                ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime("BarrierAction", new ZkConfig(ZkJobCoordinator.this.config).getZkBarrierTimeoutMs(), () -> ZkJobCoordinator.this.barrier.expire(version));
            }
        }

        @Override
        public void onBarrierStateChanged(String version, ZkBarrierForVersionUpgrade.State state) {
            LOG.info("JobModel version " + version + " obtained consensus successfully!");
            ((ZkJobCoordinator)ZkJobCoordinator.this).metrics.barrierStateChange.inc();
            ((ZkJobCoordinator)ZkJobCoordinator.this).metrics.singleBarrierRebalancingTime.update(System.nanoTime() - this.startTime);
            if (ZkBarrierForVersionUpgrade.State.DONE.equals((Object)state)) {
                ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime("BarrierAction", 0L, () -> {
                    LOG.info("pid=" + ZkJobCoordinator.this.processorId + "new version " + version + " of the job model got confirmed");
                    JobModel jobModel = ZkJobCoordinator.this.getJobModel();
                    if (ZkJobCoordinator.this.coordinatorListener != null) {
                        ZkJobCoordinator.this.coordinatorListener.onNewJobModel(ZkJobCoordinator.this.processorId, jobModel);
                    }
                });
            } else if (ZkBarrierForVersionUpgrade.State.TIMED_OUT.equals((Object)state)) {
                LOG.warn("Barrier for version " + version + " timed out.");
                if (ZkJobCoordinator.this.leaderElector.amILeader()) {
                    LOG.info("Leader will schedule a new job model generation");
                    ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime(ZkJobCoordinator.ON_PROCESSOR_CHANGE, ZkJobCoordinator.this.debounceTimeMs, () -> ZkJobCoordinator.this.doOnProcessorChange(new ArrayList<String>()));
                }
            }
        }

        @Override
        public void onBarrierError(String version, Throwable t) {
            LOG.error("Encountered error while attaining consensus on JobModel version " + version);
            ((ZkJobCoordinator)ZkJobCoordinator.this).metrics.barrierError.inc();
            ZkJobCoordinator.this.stop();
        }
    }

    class LeaderElectorListenerImpl
    implements LeaderElectorListener {
        LeaderElectorListenerImpl() {
        }

        @Override
        public void onBecomingLeader() {
            LOG.info("ZkJobCoordinator::onBecomeLeader - I became the leader");
            ((ZkJobCoordinator)ZkJobCoordinator.this).metrics.isLeader.set((Object)true);
            ZkJobCoordinator.this.zkUtils.subscribeToProcessorChange(new ProcessorChangeHandler(ZkJobCoordinator.this.zkUtils));
            ZkJobCoordinator.this.debounceTimer.scheduleAfterDebounceTime(ZkJobCoordinator.ON_PROCESSOR_CHANGE, ZkJobCoordinator.this.debounceTimeMs, () -> ZkJobCoordinator.this.doOnProcessorChange(new ArrayList<String>()));
        }
    }
}

