/*
 * Decompiled with CFR 0.152.
 */
package org.apache.samza.zk;

import com.google.common.collect.ImmutableList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.samza.zk.ScheduleAfterDebounceTime;
import org.apache.samza.zk.ZkBarrierListener;
import org.apache.samza.zk.ZkUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZkBarrierForVersionUpgrade {
    private static final Logger LOG = LoggerFactory.getLogger(ZkBarrierForVersionUpgrade.class);
    private final ZkUtils zkUtils;
    private final BarrierKeyBuilder keyBuilder;
    private final Optional<ZkBarrierListener> barrierListenerOptional;
    private final ScheduleAfterDebounceTime debounceTimer;

    public ZkBarrierForVersionUpgrade(String barrierRoot, ZkUtils zkUtils, ZkBarrierListener barrierListener, ScheduleAfterDebounceTime debounceTimer) {
        if (zkUtils == null) {
            throw new RuntimeException("Cannot operate ZkBarrierForVersionUpgrade without ZkUtils.");
        }
        this.zkUtils = zkUtils;
        this.keyBuilder = new BarrierKeyBuilder(barrierRoot);
        this.barrierListenerOptional = Optional.ofNullable(barrierListener);
        this.debounceTimer = debounceTimer;
    }

    public void create(String version, List<String> participants) {
        LOG.info(String.format("Creating barrier with version: %s, participants: %s.", version, participants));
        String barrierRoot = this.keyBuilder.getBarrierRoot();
        String barrierParticipantsPath = this.keyBuilder.getBarrierParticipantsPath(version);
        String barrierStatePath = this.keyBuilder.getBarrierStatePath(version);
        this.zkUtils.validatePaths(new String[]{barrierRoot, this.keyBuilder.getBarrierPath(version), barrierParticipantsPath, barrierStatePath});
        LOG.info("Marking the barrier state: {} as {}.", (Object)barrierStatePath, (Object)State.NEW);
        this.zkUtils.writeData(barrierStatePath, (Object)State.NEW);
        LOG.info("Subscribing child changes on the path: {} for barrier version: {}.", (Object)barrierParticipantsPath, (Object)version);
        this.zkUtils.subscribeChildChanges(barrierParticipantsPath, new ZkBarrierChangeHandler(version, participants, this.zkUtils));
        this.barrierListenerOptional.ifPresent(zkBarrierListener -> zkBarrierListener.onBarrierCreated(version));
    }

    public void join(String version, String participantId) {
        LOG.info("Joining the barrier version: {} as participant: {}.", (Object)version, (Object)participantId);
        String barrierStatePath = this.keyBuilder.getBarrierStatePath(version);
        LOG.info("Subscribing data changes on the path: {} for barrier version: {}.", (Object)barrierStatePath, (Object)version);
        this.zkUtils.subscribeDataChanges(barrierStatePath, new ZkBarrierReachedHandler(barrierStatePath, version, this.zkUtils));
        this.zkUtils.getZkClient().createPersistent(String.format("%s/%s", this.keyBuilder.getBarrierParticipantsPath(version), participantId));
    }

    public void expire(String version) {
        String barrierStatePath = this.keyBuilder.getBarrierStatePath(version);
        State barrierState = (State)((Object)this.zkUtils.getZkClient().readData(barrierStatePath));
        if (Objects.equals((Object)barrierState, (Object)State.NEW)) {
            LOG.info(String.format("Expiring the barrier version: %s. Marking the barrier state: %s as %s.", new Object[]{version, barrierStatePath, State.TIMED_OUT}));
            this.zkUtils.writeData(this.keyBuilder.getBarrierStatePath(version), (Object)State.TIMED_OUT);
        } else {
            LOG.debug(String.format("Barrier version: %s is at: %s state. Not marking barrier as %s.", new Object[]{version, barrierState, State.TIMED_OUT}));
        }
    }

    public static int getVersion(String barrierPath) {
        return Integer.valueOf(barrierPath.substring(barrierPath.lastIndexOf(95) + 1));
    }

    class BarrierKeyBuilder {
        private static final String BARRIER_PARTICIPANTS = "/barrier_participants";
        private static final String BARRIER_STATE = "/barrier_state";
        private final String barrierRoot;

        BarrierKeyBuilder(String barrierRoot) {
            if (barrierRoot == null || barrierRoot.trim().isEmpty() || !barrierRoot.trim().startsWith("/")) {
                throw new IllegalArgumentException("Barrier root path cannot be null or empty and the path has to start with '/'");
            }
            this.barrierRoot = barrierRoot;
        }

        String getBarrierRoot() {
            return this.barrierRoot;
        }

        String getBarrierPath(String version) {
            return String.format("%s/barrier_%s", this.barrierRoot, version);
        }

        String getBarrierParticipantsPath(String version) {
            return this.getBarrierPath(version) + BARRIER_PARTICIPANTS;
        }

        String getBarrierStatePath(String version) {
            return this.getBarrierPath(version) + BARRIER_STATE;
        }
    }

    class ZkBarrierReachedHandler
    extends ZkUtils.GenerationAwareZkDataListener {
        private final String barrierStatePath;
        private final String barrierVersion;

        public ZkBarrierReachedHandler(String barrierStatePath, String version, ZkUtils zkUtils) {
            super(zkUtils, "ZkBarrierReachedHandler");
            this.barrierStatePath = barrierStatePath;
            this.barrierVersion = version;
        }

        @Override
        public void doHandleDataChange(String dataPath, Object data) {
            LOG.info(String.format("Received barrierState change notification for barrier version: %s from zkNode: %s with data: %s.", this.barrierVersion, dataPath, data));
            State barrierState = (State)((Object)data);
            ImmutableList expectedBarrierStates = ImmutableList.of((Object)((Object)State.DONE), (Object)((Object)State.TIMED_OUT));
            if (barrierState != null && expectedBarrierStates.contains((Object)barrierState)) {
                ZkBarrierForVersionUpgrade.this.zkUtils.unsubscribeDataChanges(this.barrierStatePath, this);
                ZkBarrierForVersionUpgrade.this.barrierListenerOptional.ifPresent(zkBarrierListener -> zkBarrierListener.onBarrierStateChanged(this.barrierVersion, (State)((Object)((Object)data))));
            } else {
                LOG.debug("Barrier version: {} is at state: {}. Ignoring the barrierState change notification.", (Object)this.barrierVersion, (Object)barrierState);
            }
        }

        @Override
        public void doHandleDataDeleted(String path) {
            LOG.warn("Data deleted in path: " + path + " barrierVersion: " + this.barrierVersion);
        }
    }

    class ZkBarrierChangeHandler
    extends ZkUtils.GenerationAwareZkChildListener {
        private static final String ACTION_NAME = "ZkBarrierChangeHandler";
        private final String barrierVersion;
        private final List<String> expectedParticipantIds;

        public ZkBarrierChangeHandler(String barrierVersion, List<String> expectedParticipantIds, ZkUtils zkUtils) {
            super(zkUtils, ACTION_NAME);
            this.barrierVersion = barrierVersion;
            this.expectedParticipantIds = expectedParticipantIds;
        }

        @Override
        public void doHandleChildChange(String barrierParticipantPath, List<String> participantIds) {
            if (participantIds == null) {
                LOG.info("Received notification with null participants for barrier: {}. Ignoring it.", (Object)barrierParticipantPath);
                return;
            }
            LOG.info(String.format("Current participants in barrier version: %s = %s.", this.barrierVersion, participantIds));
            LOG.info(String.format("Expected participants in barrier version: %s = %s.", this.barrierVersion, this.expectedParticipantIds));
            if (participantIds.size() == this.expectedParticipantIds.size() && CollectionUtils.containsAll(participantIds, this.expectedParticipantIds)) {
                ZkBarrierForVersionUpgrade.this.debounceTimer.scheduleAfterDebounceTime(ACTION_NAME, 0L, () -> {
                    String barrierStatePath = ZkBarrierForVersionUpgrade.this.keyBuilder.getBarrierStatePath(this.barrierVersion);
                    State barrierState = (State)((Object)((Object)ZkBarrierForVersionUpgrade.this.zkUtils.getZkClient().readData(barrierStatePath)));
                    if (Objects.equals((Object)barrierState, (Object)State.NEW)) {
                        LOG.info(String.format("Expected participants has joined the barrier version: %s. Marking the barrier state: %s as %s.", new Object[]{this.barrierVersion, barrierStatePath, State.DONE}));
                        ZkBarrierForVersionUpgrade.this.zkUtils.writeData(barrierStatePath, (Object)State.DONE);
                    } else {
                        LOG.debug(String.format("Barrier version: %s is at: %s state. Not marking barrier as %s.", new Object[]{this.barrierVersion, barrierState, State.DONE}));
                    }
                    LOG.info("Unsubscribing child changes on the path: {} for barrier version: {}.", (Object)barrierParticipantPath, (Object)this.barrierVersion);
                    ZkBarrierForVersionUpgrade.this.zkUtils.unsubscribeChildChanges(barrierParticipantPath, this);
                });
            }
        }
    }

    public static enum State {
        NEW("NEW"),
        TIMED_OUT("TIMED_OUT"),
        DONE("DONE");

        private String str;

        private State(String str) {
            this.str = str;
        }

        public String toString() {
            return this.str;
        }
    }
}

