/*
 * Decompiled with CFR 0.152.
 */
package org.apache.samza.zk;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScheduleAfterDebounceTime {
    private static final Logger LOG = LoggerFactory.getLogger(ScheduleAfterDebounceTime.class);
    private static final String DEBOUNCE_THREAD_NAME_FORMAT = "Samza Debounce Thread-%s";
    private static final int TIMEOUT_MS = 10000;
    private Optional<ScheduledTaskCallback> scheduledTaskCallback;
    private final ScheduledExecutorService scheduledExecutorService;
    private final Map<String, ScheduledFuture> futureHandles = new ConcurrentHashMap<String, ScheduledFuture>();
    private volatile boolean isShuttingDown;

    public ScheduleAfterDebounceTime(String processorId) {
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat(String.format(DEBOUNCE_THREAD_NAME_FORMAT, processorId)).setDaemon(true).build();
        this.scheduledExecutorService = Executors.newSingleThreadScheduledExecutor(threadFactory);
        this.isShuttingDown = false;
    }

    public void setScheduledTaskCallback(ScheduledTaskCallback scheduledTaskCallback) {
        this.scheduledTaskCallback = Optional.ofNullable(scheduledTaskCallback);
    }

    public synchronized void scheduleAfterDebounceTime(String actionName, long delayInMillis, Runnable runnable) {
        if (!this.isShuttingDown) {
            this.tryCancelScheduledAction(actionName);
            ScheduledFuture<?> scheduledFuture = this.scheduledExecutorService.schedule(this.getScheduleableAction(actionName, runnable), delayInMillis, TimeUnit.MILLISECONDS);
            LOG.info("Scheduled action: {} to run after: {} milliseconds.", (Object)actionName, (Object)delayInMillis);
            this.futureHandles.put(actionName, scheduledFuture);
        } else {
            LOG.info("Scheduler is stopped. Not scheduling action: {} to run.", (Object)actionName);
        }
    }

    public synchronized void cancelAction(String action) {
        if (!this.isShuttingDown) {
            this.tryCancelScheduledAction(action);
        }
    }

    public synchronized void stopScheduler() {
        if (this.isShuttingDown) {
            LOG.debug("Debounce timer shutdown is already in progress!");
            return;
        }
        this.isShuttingDown = true;
        LOG.info("Shutting down debounce timer!");
        this.scheduledExecutorService.shutdown();
        this.cancelAllScheduledActions();
    }

    public synchronized void cancelAllScheduledActions() {
        if (!this.isShuttingDown) {
            this.futureHandles.keySet().forEach(this::tryCancelScheduledAction);
            this.futureHandles.clear();
        }
    }

    private void tryCancelScheduledAction(String actionName) {
        LOG.info("Trying to cancel the action: {}.", (Object)actionName);
        ScheduledFuture scheduledFuture = this.futureHandles.get(actionName);
        if (scheduledFuture != null && !scheduledFuture.isDone()) {
            LOG.info("Attempting to cancel the future of action: {}", (Object)actionName);
            if (!scheduledFuture.cancel(false)) {
                try {
                    scheduledFuture.get(10000L, TimeUnit.MILLISECONDS);
                }
                catch (Exception e) {
                    LOG.warn("Cancelling the future of action: {} failed.", (Object)actionName, (Object)e);
                }
            }
            this.futureHandles.remove(actionName);
        }
    }

    private Runnable getScheduleableAction(String actionName, Runnable runnable) {
        return () -> {
            try {
                if (!this.isShuttingDown) {
                    runnable.run();
                    if (Thread.currentThread().isInterrupted()) {
                        LOG.warn("Action: {} is interrupted.", (Object)actionName);
                        this.doCleanUpOnTaskException(new InterruptedException());
                    } else {
                        LOG.info("Action: {} completed successfully.", (Object)actionName);
                    }
                }
            }
            catch (Throwable throwable) {
                LOG.error("Execution of action: {} failed.", (Object)actionName, (Object)throwable);
                this.doCleanUpOnTaskException(throwable);
            }
        };
    }

    private void doCleanUpOnTaskException(Throwable throwable) {
        this.stopScheduler();
        this.scheduledTaskCallback.ifPresent(callback -> callback.onError(throwable));
    }

    static interface ScheduledTaskCallback {
        public void onError(Throwable var1);
    }
}

