/*
 * Decompiled with CFR 0.152.
 */
package org.apache.samza.table.caching;

import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.apache.samza.SamzaException;
import org.apache.samza.context.Context;
import org.apache.samza.storage.kv.Entry;
import org.apache.samza.table.ReadWriteTable;
import org.apache.samza.table.ReadableTable;
import org.apache.samza.table.Table;
import org.apache.samza.table.utils.DefaultTableReadMetrics;
import org.apache.samza.table.utils.DefaultTableWriteMetrics;
import org.apache.samza.table.utils.TableMetricsUtil;

public class CachingTable<K, V>
implements ReadWriteTable<K, V> {
    private final String tableId;
    private final ReadableTable<K, V> rdTable;
    private final ReadWriteTable<K, V> rwTable;
    private final ReadWriteTable<K, V> cache;
    private final boolean isWriteAround;
    private DefaultTableReadMetrics readMetrics;
    private DefaultTableWriteMetrics writeMetrics;
    private AtomicLong hitCount = new AtomicLong();
    private AtomicLong missCount = new AtomicLong();

    public CachingTable(String tableId, ReadableTable<K, V> table, ReadWriteTable<K, V> cache, boolean isWriteAround) {
        this.tableId = tableId;
        this.rdTable = table;
        this.rwTable = table instanceof ReadWriteTable ? (ReadWriteTable)table : null;
        this.cache = cache;
        this.isWriteAround = isWriteAround;
    }

    public void init(Context context) {
        this.readMetrics = new DefaultTableReadMetrics(context, (Table)this, this.tableId);
        this.writeMetrics = new DefaultTableWriteMetrics(context, (Table)this, this.tableId);
        TableMetricsUtil tableMetricsUtil = new TableMetricsUtil(context, (Table)this, this.tableId);
        tableMetricsUtil.newGauge("hit-rate", () -> this.hitRate());
        tableMetricsUtil.newGauge("miss-rate", () -> this.missRate());
        tableMetricsUtil.newGauge("req-count", () -> this.requestCount());
    }

    private List<K> lookupCache(List<K> keys, Map<K, V> records) {
        ArrayList missKeys = new ArrayList();
        records.putAll(this.cache.getAll(keys));
        keys.forEach(k -> {
            if (!records.containsKey(k)) {
                missKeys.add(k);
            }
        });
        return missKeys;
    }

    public V get(K key) {
        try {
            return this.getAsync(key).get();
        }
        catch (InterruptedException e) {
            throw new SamzaException((Throwable)e);
        }
        catch (Exception e) {
            throw (SamzaException)e.getCause();
        }
    }

    public CompletableFuture<V> getAsync(K key) {
        this.readMetrics.numGets.inc();
        Object value = this.cache.get(key);
        if (value != null) {
            this.hitCount.incrementAndGet();
            return CompletableFuture.completedFuture(value);
        }
        long startNs = System.nanoTime();
        this.missCount.incrementAndGet();
        return this.rdTable.getAsync(key).handle((result, e) -> {
            if (e != null) {
                throw new SamzaException("Failed to get the record for " + key, e);
            }
            if (result != null) {
                this.cache.put(key, result);
            }
            this.readMetrics.getNs.update(System.nanoTime() - startNs);
            return result;
        });
    }

    public Map<K, V> getAll(List<K> keys) {
        try {
            return this.getAllAsync(keys).get();
        }
        catch (InterruptedException e) {
            throw new SamzaException((Throwable)e);
        }
        catch (Exception e) {
            throw (SamzaException)e.getCause();
        }
    }

    public CompletableFuture<Map<K, V>> getAllAsync(List<K> keys) {
        this.readMetrics.numGetAlls.inc();
        HashMap getAllResult = new HashMap();
        List<K> missingKeys = this.lookupCache(keys, getAllResult);
        if (missingKeys.isEmpty()) {
            return CompletableFuture.completedFuture(getAllResult);
        }
        long startNs = System.nanoTime();
        return this.rdTable.getAllAsync(missingKeys).handle((records, e) -> {
            if (e != null) {
                throw new SamzaException("Failed to get records for " + keys, e);
            }
            if (records != null) {
                this.cache.putAll(records.entrySet().stream().map(r -> new Entry(r.getKey(), r.getValue())).collect(Collectors.toList()));
                getAllResult.putAll(records);
            }
            this.readMetrics.getAllNs.update(System.nanoTime() - startNs);
            return getAllResult;
        });
    }

    public void put(K key, V value) {
        try {
            this.putAsync(key, value).get();
        }
        catch (InterruptedException e) {
            throw new SamzaException((Throwable)e);
        }
        catch (Exception e) {
            throw (SamzaException)e.getCause();
        }
    }

    public CompletableFuture<Void> putAsync(K key, V value) {
        this.writeMetrics.numPuts.inc();
        Preconditions.checkNotNull(this.rwTable, (Object)("Cannot write to a read-only table: " + this.rdTable));
        long startNs = System.nanoTime();
        return this.rwTable.putAsync(key, value).handle((result, e) -> {
            if (e != null) {
                throw new SamzaException(String.format("Failed to put a record, key=%s, value=%s", key, value), e);
            }
            if (!this.isWriteAround) {
                if (value == null) {
                    this.cache.delete(key);
                } else {
                    this.cache.put(key, value);
                }
            }
            this.writeMetrics.putNs.update(System.nanoTime() - startNs);
            return result;
        });
    }

    public void putAll(List<Entry<K, V>> records) {
        try {
            this.putAllAsync(records).get();
        }
        catch (InterruptedException e) {
            throw new SamzaException((Throwable)e);
        }
        catch (Exception e) {
            throw (SamzaException)e.getCause();
        }
    }

    public CompletableFuture<Void> putAllAsync(List<Entry<K, V>> records) {
        this.writeMetrics.numPutAlls.inc();
        long startNs = System.nanoTime();
        Preconditions.checkNotNull(this.rwTable, (Object)("Cannot write to a read-only table: " + this.rdTable));
        return this.rwTable.putAllAsync(records).handle((result, e) -> {
            if (e != null) {
                throw new SamzaException("Failed to put records " + records, e);
            }
            if (!this.isWriteAround) {
                this.cache.putAll(records);
            }
            this.writeMetrics.putAllNs.update(System.nanoTime() - startNs);
            return result;
        });
    }

    public void delete(K key) {
        try {
            this.deleteAsync(key).get();
        }
        catch (InterruptedException e) {
            throw new SamzaException((Throwable)e);
        }
        catch (Exception e) {
            throw (SamzaException)e.getCause();
        }
    }

    public CompletableFuture<Void> deleteAsync(K key) {
        this.writeMetrics.numDeletes.inc();
        long startNs = System.nanoTime();
        Preconditions.checkNotNull(this.rwTable, (Object)("Cannot delete from a read-only table: " + this.rdTable));
        return this.rwTable.deleteAsync(key).handle((result, e) -> {
            if (e != null) {
                throw new SamzaException("Failed to delete the record for " + key, e);
            }
            if (!this.isWriteAround) {
                this.cache.delete(key);
            }
            this.writeMetrics.deleteNs.update(System.nanoTime() - startNs);
            return result;
        });
    }

    public void deleteAll(List<K> keys) {
        try {
            this.deleteAllAsync(keys).get();
        }
        catch (Exception e) {
            throw new SamzaException((Throwable)e);
        }
    }

    public CompletableFuture<Void> deleteAllAsync(List<K> keys) {
        this.writeMetrics.numDeleteAlls.inc();
        long startNs = System.nanoTime();
        Preconditions.checkNotNull(this.rwTable, (Object)("Cannot delete from a read-only table: " + this.rdTable));
        return this.rwTable.deleteAllAsync(keys).handle((result, e) -> {
            if (e != null) {
                throw new SamzaException("Failed to delete the record for " + keys, e);
            }
            if (!this.isWriteAround) {
                this.cache.deleteAll(keys);
            }
            this.writeMetrics.deleteAllNs.update(System.nanoTime() - startNs);
            return result;
        });
    }

    public synchronized void flush() {
        this.writeMetrics.numFlushes.inc();
        long startNs = System.nanoTime();
        Preconditions.checkNotNull(this.rwTable, (Object)("Cannot flush a read-only table: " + this.rdTable));
        this.rwTable.flush();
        this.writeMetrics.flushNs.update(System.nanoTime() - startNs);
    }

    public void close() {
        this.cache.close();
        this.rdTable.close();
    }

    double hitRate() {
        long reqs = this.requestCount();
        return reqs == 0L ? 1.0 : (double)this.hitCount.get() / (double)reqs;
    }

    double missRate() {
        long reqs = this.requestCount();
        return reqs == 0L ? 1.0 : (double)this.missCount.get() / (double)reqs;
    }

    long requestCount() {
        return this.hitCount.get() + this.missCount.get();
    }
}

