/*
 * Decompiled with CFR 0.152.
 */
package org.apache.samza.storage;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import java.io.File;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.samza.SamzaException;
import org.apache.samza.config.Config;
import org.apache.samza.container.TaskName;
import org.apache.samza.serializers.model.SamzaObjectMapper;
import org.apache.samza.storage.SideInputsProcessor;
import org.apache.samza.storage.StorageEngine;
import org.apache.samza.storage.StorageManagerUtil;
import org.apache.samza.storage.StoreProperties;
import org.apache.samza.storage.kv.KeyValueStore;
import org.apache.samza.system.IncomingMessageEnvelope;
import org.apache.samza.system.StreamMetadataCache;
import org.apache.samza.system.SystemAdmins;
import org.apache.samza.system.SystemStream;
import org.apache.samza.system.SystemStreamMetadata;
import org.apache.samza.system.SystemStreamPartition;
import org.apache.samza.util.Clock;
import org.apache.samza.util.FileUtil;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.ObjectWriter;
import org.codehaus.jackson.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.collection.JavaConverters;

public class TaskSideInputStorageManager {
    private static final Logger LOG = LoggerFactory.getLogger(TaskSideInputStorageManager.class);
    private static final String OFFSET_FILE = "SIDE-INPUT-OFFSETS";
    private static final long STORE_DELETE_RETENTION_MS = TimeUnit.DAYS.toMillis(1L);
    private static final ObjectMapper OBJECT_MAPPER = SamzaObjectMapper.getObjectMapper();
    private static final TypeReference<HashMap<SystemStreamPartition, String>> OFFSETS_TYPE_REFERENCE = new TypeReference<HashMap<SystemStreamPartition, String>>(){};
    private static final ObjectWriter OBJECT_WRITER = OBJECT_MAPPER.writerWithType(OFFSETS_TYPE_REFERENCE);
    private final Clock clock;
    private final Map<String, SideInputsProcessor> storeToProcessor;
    private final Map<String, StorageEngine> stores;
    private final String storeBaseDir;
    private final Map<String, Set<SystemStreamPartition>> storeToSSps;
    private final Map<SystemStreamPartition, Set<String>> sspsToStores;
    private final StreamMetadataCache streamMetadataCache;
    private final SystemAdmins systemAdmins;
    private final TaskName taskName;
    private final Map<SystemStreamPartition, String> lastProcessedOffsets = new ConcurrentHashMap<SystemStreamPartition, String>();
    private Map<SystemStreamPartition, String> startingOffsets;

    public TaskSideInputStorageManager(TaskName taskName, StreamMetadataCache streamMetadataCache, String storeBaseDir, Map<String, StorageEngine> sideInputStores, Map<String, SideInputsProcessor> storesToProcessor, Map<String, Set<SystemStreamPartition>> storesToSSPs, SystemAdmins systemAdmins, Config config, Clock clock) {
        this.clock = clock;
        this.stores = sideInputStores;
        this.storeBaseDir = storeBaseDir;
        this.storeToSSps = storesToSSPs;
        this.streamMetadataCache = streamMetadataCache;
        this.systemAdmins = systemAdmins;
        this.taskName = taskName;
        this.storeToProcessor = storesToProcessor;
        this.validateStoreConfiguration();
        this.sspsToStores = new HashMap<SystemStreamPartition, Set<String>>();
        storesToSSPs.forEach((store, ssps) -> {
            for (SystemStreamPartition ssp : ssps) {
                this.sspsToStores.computeIfAbsent(ssp, key -> new HashSet());
                this.sspsToStores.computeIfPresent(ssp, (key, value) -> {
                    value.add(store);
                    return value;
                });
            }
        });
    }

    public void init() {
        LOG.info("Initializing side input stores.");
        Map<SystemStreamPartition, String> fileOffsets = this.getFileOffsets();
        LOG.info("File offsets for the task {}: ", (Object)this.taskName, fileOffsets);
        Map<SystemStreamPartition, String> oldestOffsets = this.getOldestOffsets();
        LOG.info("Oldest offsets for the task {}: ", (Object)this.taskName, fileOffsets);
        this.startingOffsets = this.getStartingOffsets(fileOffsets, oldestOffsets);
        LOG.info("Starting offsets for the task {}: {}", (Object)this.taskName, this.startingOffsets);
        this.lastProcessedOffsets.putAll(fileOffsets);
        LOG.info("Last processed offsets for the task {}: {}", (Object)this.taskName, this.lastProcessedOffsets);
        this.initializeStoreDirectories();
    }

    public void flush() {
        LOG.info("Flushing the side input stores.");
        this.stores.values().forEach(StorageEngine::flush);
        this.writeOffsetFiles();
    }

    public void stop() {
        LOG.info("Stopping the side input stores.");
        this.stores.values().forEach(StorageEngine::stop);
        this.writeOffsetFiles();
    }

    public StorageEngine getStore(String storeName) {
        return this.stores.get(storeName);
    }

    public String getStartingOffset(SystemStreamPartition ssp) {
        return this.startingOffsets.get(ssp);
    }

    public String getLastProcessedOffset(SystemStreamPartition ssp) {
        return this.lastProcessedOffsets.get(ssp);
    }

    @VisibleForTesting
    void updateLastProcessedOffset(SystemStreamPartition ssp, String offset) {
        this.lastProcessedOffsets.put(ssp, offset);
    }

    public void process(IncomingMessageEnvelope message) {
        SystemStreamPartition ssp = message.getSystemStreamPartition();
        Set<String> storeNames = this.sspsToStores.get(ssp);
        for (String storeName : storeNames) {
            SideInputsProcessor sideInputsProcessor = this.storeToProcessor.get(storeName);
            KeyValueStore keyValueStore = (KeyValueStore)this.stores.get(storeName);
            Collection entriesToBeWritten = sideInputsProcessor.process(message, keyValueStore);
            keyValueStore.putAll((List)ImmutableList.copyOf((Collection)entriesToBeWritten));
        }
        this.lastProcessedOffsets.put(ssp, message.getOffset());
    }

    private void initializeStoreDirectories() {
        LOG.info("Initializing side input store directories.");
        this.stores.keySet().forEach(storeName -> {
            File storeLocation = this.getStoreLocation((String)storeName);
            String storePath = storeLocation.toPath().toString();
            if (!this.isValidSideInputStore((String)storeName, storeLocation)) {
                LOG.info("Cleaning up the store directory at {} for {}", (Object)storePath, storeName);
                FileUtil.rm(storeLocation);
            }
            if (this.isPersistedStore((String)storeName) && !storeLocation.exists()) {
                LOG.info("Creating {} as the store directory for the side input store {}", (Object)storePath, storeName);
                storeLocation.mkdirs();
            }
        });
    }

    @VisibleForTesting
    void writeOffsetFiles() {
        this.storeToSSps.entrySet().stream().filter(entry -> this.isPersistedStore((String)entry.getKey())).forEach(entry -> {
            String storeName = (String)entry.getKey();
            Map offsets = ((Set)entry.getValue()).stream().filter(this.lastProcessedOffsets::containsKey).collect(Collectors.toMap(Function.identity(), this.lastProcessedOffsets::get));
            try {
                String fileContents = OBJECT_WRITER.writeValueAsString(offsets);
                File offsetFile = new File(this.getStoreLocation(storeName), OFFSET_FILE);
                FileUtil.writeWithChecksum(offsetFile, fileContents);
            }
            catch (Exception e) {
                throw new SamzaException("Failed to write offset file for side input store: " + storeName, (Throwable)e);
            }
        });
    }

    @VisibleForTesting
    Map<SystemStreamPartition, String> getFileOffsets() {
        LOG.info("Loading initial offsets from the file for side input stores.");
        HashMap<SystemStreamPartition, String> fileOffsets = new HashMap<SystemStreamPartition, String>();
        this.stores.keySet().forEach(storeName -> {
            LOG.debug("Reading local offsets for store: {}", storeName);
            File storeLocation = this.getStoreLocation((String)storeName);
            if (this.isValidSideInputStore((String)storeName, storeLocation)) {
                try {
                    String fileContents = StorageManagerUtil.readOffsetFile(storeLocation, OFFSET_FILE);
                    Map offsets = (Map)OBJECT_MAPPER.readValue(fileContents, OFFSETS_TYPE_REFERENCE);
                    fileOffsets.putAll(offsets);
                }
                catch (Exception e) {
                    LOG.warn("Failed to load the offset file for side input store:" + storeName, (Throwable)e);
                }
            }
        });
        return fileOffsets;
    }

    @VisibleForTesting
    File getStoreLocation(String storeName) {
        return new File(this.storeBaseDir, (storeName + File.separator + this.taskName.toString()).replace(' ', '_'));
    }

    @VisibleForTesting
    Map<SystemStreamPartition, String> getStartingOffsets(Map<SystemStreamPartition, String> fileOffsets, Map<SystemStreamPartition, String> oldestOffsets) {
        HashMap<SystemStreamPartition, String> startingOffsets = new HashMap<SystemStreamPartition, String>();
        this.sspsToStores.keySet().forEach(ssp -> {
            String fileOffset = (String)fileOffsets.get(ssp);
            String oldestOffset = (String)oldestOffsets.get(ssp);
            startingOffsets.put((SystemStreamPartition)ssp, StorageManagerUtil.getStartingOffset(ssp, this.systemAdmins.getSystemAdmin(ssp.getSystem()), fileOffset, oldestOffset));
        });
        return startingOffsets;
    }

    @VisibleForTesting
    Map<SystemStreamPartition, String> getOldestOffsets() {
        HashMap<SystemStreamPartition, String> oldestOffsets = new HashMap<SystemStreamPartition, String>();
        Map<SystemStream, List<SystemStreamPartition>> systemStreamToSsp = this.sspsToStores.keySet().stream().collect(Collectors.groupingBy(SystemStreamPartition::getSystemStream));
        Map metadata = (Map)JavaConverters.mapAsJavaMapConverter(this.streamMetadataCache.getStreamMetadata((scala.collection.immutable.Set<SystemStream>)((scala.collection.mutable.Set)JavaConverters.asScalaSetConverter(systemStreamToSsp.keySet()).asScala()).toSet(), false)).asJava();
        metadata.forEach((systemStream, systemStreamMetadata) -> {
            Map partitionMetadata = systemStreamMetadata.getSystemStreamPartitionMetadata();
            Map offsets = ((List)systemStreamToSsp.get(systemStream)).stream().collect(Collectors.toMap(Function.identity(), ssp -> ((SystemStreamMetadata.SystemStreamPartitionMetadata)partitionMetadata.get(ssp.getPartition())).getOldestOffset()));
            oldestOffsets.putAll(offsets);
        });
        return oldestOffsets;
    }

    private boolean isValidSideInputStore(String storeName, File storeLocation) {
        return this.isPersistedStore(storeName) && !StorageManagerUtil.isStaleStore(storeLocation, OFFSET_FILE, STORE_DELETE_RETENTION_MS, this.clock.currentTimeMillis()) && StorageManagerUtil.isOffsetFileValid(storeLocation, OFFSET_FILE);
    }

    private boolean isPersistedStore(String storeName) {
        return Optional.ofNullable(this.stores.get(storeName)).map(StorageEngine::getStoreProperties).map(StoreProperties::isPersistedToDisk).orElse(false);
    }

    private void validateStoreConfiguration() {
        this.stores.forEach((storeName, storageEngine) -> {
            if (!this.storeToProcessor.containsKey(storeName)) {
                throw new SamzaException(String.format("Side inputs processor missing for store: %s.", storeName));
            }
            if (storageEngine.getStoreProperties().isLoggedStore()) {
                throw new SamzaException(String.format("Cannot configure both side inputs and a changelog for store: %s.", storeName));
            }
        });
    }
}

