/*
 * Decompiled with CFR 0.152.
 */
package org.apache.samza.clustermanager;

import com.google.common.base.Preconditions;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.samza.SamzaException;
import org.apache.samza.clustermanager.AbstractContainerAllocator;
import org.apache.samza.clustermanager.ClusterResourceManager;
import org.apache.samza.clustermanager.ContainerAllocator;
import org.apache.samza.clustermanager.HostAwareContainerAllocator;
import org.apache.samza.clustermanager.ResourceFailure;
import org.apache.samza.clustermanager.ResourceManagerFactory;
import org.apache.samza.clustermanager.SamzaApplicationState;
import org.apache.samza.clustermanager.SamzaResource;
import org.apache.samza.clustermanager.SamzaResourceStatus;
import org.apache.samza.config.ClusterManagerConfig;
import org.apache.samza.config.Config;
import org.apache.samza.config.JobConfig;
import org.apache.samza.coordinator.JobModelManager;
import org.apache.samza.metrics.ContainerProcessManagerMetrics;
import org.apache.samza.metrics.MetricsRegistryMap;
import org.apache.samza.util.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerProcessManager
implements ClusterResourceManager.Callback {
    private static final Logger log = LoggerFactory.getLogger(ContainerProcessManager.class);
    private final boolean hostAffinityEnabled;
    private final SamzaApplicationState state;
    private final ClusterManagerConfig clusterManagerConfig;
    private final JobConfig jobConfig;
    private final AbstractContainerAllocator containerAllocator;
    private final Thread allocatorThread;
    private final ClusterResourceManager clusterResourceManager;
    private volatile boolean tooManyFailedContainers = false;
    private volatile Throwable exceptionOccurred = null;
    private final Map<String, ResourceFailure> containerFailures = new HashMap<String, ResourceFailure>();
    private final ContainerProcessManagerMetrics metrics;

    ContainerProcessManager(Config config, SamzaApplicationState state, MetricsRegistryMap registry, AbstractContainerAllocator allocator, ClusterResourceManager manager) {
        this.state = state;
        this.clusterManagerConfig = new ClusterManagerConfig(config);
        this.jobConfig = new JobConfig(config);
        this.hostAffinityEnabled = this.clusterManagerConfig.getHostAffinityEnabled();
        this.clusterResourceManager = manager;
        this.metrics = new ContainerProcessManagerMetrics(config, state, registry);
        this.containerAllocator = allocator;
        this.allocatorThread = new Thread((Runnable)this.containerAllocator, "Container Allocator Thread");
    }

    public ContainerProcessManager(Config config, SamzaApplicationState state, MetricsRegistryMap registry) {
        this.state = state;
        this.clusterManagerConfig = new ClusterManagerConfig(config);
        this.jobConfig = new JobConfig(config);
        this.hostAffinityEnabled = this.clusterManagerConfig.getHostAffinityEnabled();
        ResourceManagerFactory factory = this.getContainerProcessManagerFactory(this.clusterManagerConfig);
        this.clusterResourceManager = (ClusterResourceManager)Preconditions.checkNotNull((Object)factory.getClusterResourceManager(this, state));
        this.metrics = new ContainerProcessManagerMetrics(config, state, registry);
        this.containerAllocator = this.hostAffinityEnabled ? new HostAwareContainerAllocator(this.clusterResourceManager, this.clusterManagerConfig.getContainerRequestTimeout(), config, state) : new ContainerAllocator(this.clusterResourceManager, config, state);
        this.allocatorThread = new Thread((Runnable)this.containerAllocator, "Container Allocator Thread");
        log.info("finished initialization of samza task manager");
    }

    ContainerProcessManager(Config config, SamzaApplicationState state, MetricsRegistryMap registry, ClusterResourceManager resourceManager) {
        JobModelManager jobModelManager = state.jobModelManager;
        this.state = state;
        this.clusterManagerConfig = new ClusterManagerConfig(config);
        this.jobConfig = new JobConfig(config);
        this.hostAffinityEnabled = this.clusterManagerConfig.getHostAffinityEnabled();
        this.clusterResourceManager = resourceManager;
        this.metrics = new ContainerProcessManagerMetrics(config, state, registry);
        this.containerAllocator = this.hostAffinityEnabled ? new HostAwareContainerAllocator(this.clusterResourceManager, this.clusterManagerConfig.getContainerRequestTimeout(), config, state) : new ContainerAllocator(this.clusterResourceManager, config, state);
        this.allocatorThread = new Thread((Runnable)this.containerAllocator, "Container Allocator Thread");
        log.info("finished initialization of samza task manager");
    }

    public boolean shouldShutdown() {
        log.debug(" TaskManager state: Completed containers: {}, Configured containers: {}, Is there too many FailedContainers: {}, Is AllocatorThread alive: {} ", new Object[]{this.state.completedContainers.get(), this.state.containerCount, this.tooManyFailedContainers ? "yes" : "no", this.allocatorThread.isAlive() ? "yes" : "no"});
        if (this.exceptionOccurred != null) {
            log.error("Exception in ContainerProcessManager", this.exceptionOccurred);
            throw new SamzaException(this.exceptionOccurred);
        }
        return this.tooManyFailedContainers || this.state.completedContainers.get() == this.state.containerCount.get() || !this.allocatorThread.isAlive();
    }

    public void start() {
        this.metrics.start();
        log.info("Starting Container Process Manager");
        this.clusterResourceManager.start();
        log.info("Starting the Samza task manager");
        int containerCount = this.jobConfig.getContainerCount();
        this.state.containerCount.set(containerCount);
        this.state.neededContainers.set(containerCount);
        Map<String, String> containerToHostMapping = this.state.jobModelManager.jobModel().getAllContainerLocality();
        this.containerAllocator.requestResources(containerToHostMapping);
        log.info("Starting the container allocator thread");
        this.allocatorThread.start();
    }

    public void stop() {
        log.info("Invoked stop of the Samza container process manager");
        this.containerAllocator.stop();
        try {
            this.allocatorThread.join();
            log.info("Stopped container allocator");
        }
        catch (InterruptedException ie) {
            log.error("Allocator Thread join() threw an interrupted exception", (Throwable)ie);
            Thread.currentThread().interrupt();
        }
        if (this.metrics != null) {
            try {
                this.metrics.stop();
                log.info("Stopped metrics reporters");
            }
            catch (Throwable e) {
                log.error("Exception while stopping metrics {}", e);
            }
        }
        try {
            this.clusterResourceManager.stop(this.state.status);
            log.info("Stopped cluster resource manager");
        }
        catch (Throwable e) {
            log.error("Exception while stopping cluster resource manager {}", e);
        }
        log.info("Finished stop of Container process manager");
    }

    public void onResourceAllocated(SamzaResource container) {
        log.info("Container allocated from RM on " + container.getHost());
        this.containerAllocator.addResource(container);
    }

    public void onResourceCompleted(SamzaResourceStatus containerStatus) {
        String containerIdStr = containerStatus.getResourceID();
        String containerId = null;
        for (Map.Entry entry : this.state.runningContainers.entrySet()) {
            if (!((SamzaResource)entry.getValue()).getResourceID().equals(containerStatus.getResourceID())) continue;
            log.info("Matching container ID found " + (String)entry.getKey() + " " + entry.getValue());
            containerId = (String)entry.getKey();
            break;
        }
        if (containerId == null) {
            log.info("No matching container id found for " + containerStatus.toString());
            this.state.redundantNotifications.incrementAndGet();
            return;
        }
        this.state.runningContainers.remove(containerId);
        int exitStatus = containerStatus.getExitCode();
        switch (exitStatus) {
            case 0: {
                log.info("Container {} completed successfully.", (Object)containerIdStr);
                this.state.completedContainers.incrementAndGet();
                this.state.finishedContainers.incrementAndGet();
                this.containerFailures.remove(containerId);
                if (this.state.completedContainers.get() != this.state.containerCount.get()) break;
                log.info("Setting job status to SUCCEEDED, since all containers have been marked as completed.");
                this.state.status = SamzaApplicationState.SamzaAppStatus.SUCCEEDED;
                break;
            }
            case -102: 
            case -101: 
            case -100: {
                log.info("Got an exit code of {}. This means that container {} was killed by YARN, either due to being released by the application master or being 'lost' due to node failures etc. or due to preemption by the RM", (Object)exitStatus, (Object)containerIdStr);
                this.state.releasedContainers.incrementAndGet();
                log.info("Released container {} was assigned task group ID {}. Requesting a new container for the task group.", (Object)containerIdStr, containerId);
                this.state.neededContainers.incrementAndGet();
                this.state.jobHealthy.set(false);
                this.containerAllocator.requestResource(containerId, "ANY_HOST");
                break;
            }
            default: {
                log.info("Container failed for some reason. Let's start it again");
                log.info("Container " + containerIdStr + " failed with exit code . " + exitStatus + " - " + containerStatus.getDiagnostics() + " containerID is " + containerId);
                this.state.failedContainers.incrementAndGet();
                this.state.failedContainersStatus.put(containerIdStr, containerStatus);
                this.state.jobHealthy.set(false);
                this.state.neededContainers.incrementAndGet();
                String lastSeenOn = this.state.jobModelManager.jobModel().getContainerToHostValue(containerId, "host");
                if (!this.hostAffinityEnabled || lastSeenOn == null) {
                    lastSeenOn = "ANY_HOST";
                }
                log.info("Container was last seen on " + lastSeenOn);
                int retryCount = this.clusterManagerConfig.getContainerRetryCount();
                int retryWindowMs = this.clusterManagerConfig.getContainerRetryWindowMs();
                if (retryCount == 0) {
                    log.error("Container ID {} ({}) failed, and retry count is set to 0, so shutting down the application master, and marking the job as failed.", (Object)containerId, (Object)containerIdStr);
                    this.tooManyFailedContainers = true;
                } else if (retryCount > 0) {
                    long lastFailureTime;
                    int currentFailCount;
                    if (this.containerFailures.containsKey(containerId)) {
                        ResourceFailure failure = this.containerFailures.get(containerId);
                        currentFailCount = failure.getCount() + 1;
                        lastFailureTime = failure.getLastFailure();
                    } else {
                        currentFailCount = 1;
                        lastFailureTime = 0L;
                    }
                    if (currentFailCount >= retryCount) {
                        long lastFailureMsDiff = System.currentTimeMillis() - lastFailureTime;
                        if (lastFailureMsDiff < (long)retryWindowMs) {
                            log.error("Container ID " + containerId + "(" + containerIdStr + ") has failed " + currentFailCount + " times, with last failure " + lastFailureMsDiff + "ms ago. This is greater than retry count of " + retryCount + " and window of " + retryWindowMs + "ms , so shutting down the application master, and marking the job as failed.");
                            this.tooManyFailedContainers = true;
                            this.state.status = SamzaApplicationState.SamzaAppStatus.FAILED;
                        } else {
                            log.info("Resetting fail count for container ID {} back to 1, since last container failure ({}) for this container ID was outside the bounds of the retry window.", (Object)containerId, (Object)containerIdStr);
                            this.containerFailures.put(containerId, new ResourceFailure(1, System.currentTimeMillis()));
                        }
                    } else {
                        log.info("Current fail count for container ID {} is {}.", (Object)containerId, (Object)currentFailCount);
                        this.containerFailures.put(containerId, new ResourceFailure(currentFailCount, System.currentTimeMillis()));
                    }
                }
                if (this.tooManyFailedContainers) break;
                log.info("Requesting a new container ");
                this.containerAllocator.requestResource(containerId, lastSeenOn);
            }
        }
    }

    @Override
    public void onResourcesAvailable(List<SamzaResource> resources) {
        for (SamzaResource resource : resources) {
            this.onResourceAllocated(resource);
        }
    }

    @Override
    public void onResourcesCompleted(List<SamzaResourceStatus> resourceStatuses) {
        for (SamzaResourceStatus resourceStatus : resourceStatuses) {
            this.onResourceCompleted(resourceStatus);
        }
    }

    @Override
    public void onStreamProcessorLaunchSuccess(SamzaResource resource) {
        String containerId = this.getPendingContainerId(resource.getResourceID());
        log.info("Successfully started container ID: {} on resource: {}", (Object)containerId, (Object)resource);
        if (containerId != null) {
            log.info("Moving containerID: {} on resource: {} from pending to running state", (Object)containerId, (Object)resource);
            this.state.pendingContainers.remove(containerId);
            this.state.runningContainers.put(containerId, resource);
            if (this.state.neededContainers.decrementAndGet() == 0) {
                this.state.jobHealthy.set(true);
            }
        } else {
            log.warn("SamzaResource {} was not in pending state. Got an invalid callback for a launch request that was not issued", (Object)resource);
        }
    }

    @Override
    public void onStreamProcessorLaunchFailure(SamzaResource resource, Throwable t) {
        log.error("Got a launch failure for SamzaResource {} with exception {}", (Object)resource, (Object)t);
        log.info("Releasing unstartable container {}", (Object)resource.getResourceID());
        this.clusterResourceManager.releaseResources(resource);
        String containerId = this.getPendingContainerId(resource.getResourceID());
        log.info("Failed container ID: {} for resourceId: {}", (Object)containerId, (Object)resource.getResourceID());
        if (containerId != null) {
            log.info("Launch of container ID: {} failed on host: {}. Falling back to ANY_HOST", (Object)containerId, (Object)resource.getHost());
            this.containerAllocator.requestResource(containerId, "ANY_HOST");
        } else {
            log.warn("SamzaResource {} was not in pending state. Got an invalid callback for a launch request that was not issued", (Object)resource);
        }
    }

    @Override
    public void onError(Throwable e) {
        log.error("Exception occured in callbacks in the Container Manager : {}", e);
        this.exceptionOccurred = e;
    }

    private ResourceManagerFactory getContainerProcessManagerFactory(ClusterManagerConfig clusterManagerConfig) {
        ResourceManagerFactory factory;
        String containerManagerFactoryClass = clusterManagerConfig.getContainerManagerClass();
        try {
            factory = Util.getObj(containerManagerFactoryClass, ResourceManagerFactory.class);
        }
        catch (Exception e) {
            log.error("Exception when creating ContainerManager", (Throwable)e);
            throw new SamzaException((Throwable)e);
        }
        return factory;
    }

    private String getPendingContainerId(String resourceId) {
        for (Map.Entry entry : this.state.pendingContainers.entrySet()) {
            if (!((SamzaResource)entry.getValue()).getResourceID().equals(resourceId)) continue;
            log.info("Matching container ID found " + (String)entry.getKey() + " " + entry.getValue());
            return (String)entry.getKey();
        }
        return null;
    }
}

