/*
 * Decompiled with CFR 0.152.
 */
package org.apache.samza.checkpoint.azure;

import com.google.common.collect.ImmutableMap;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.table.CloudTable;
import com.microsoft.azure.storage.table.TableBatchOperation;
import com.microsoft.azure.storage.table.TableEntity;
import com.microsoft.azure.storage.table.TableQuery;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import org.apache.samza.AzureClient;
import org.apache.samza.AzureException;
import org.apache.samza.Partition;
import org.apache.samza.SamzaException;
import org.apache.samza.checkpoint.Checkpoint;
import org.apache.samza.checkpoint.CheckpointManager;
import org.apache.samza.checkpoint.azure.TaskCheckpointEntity;
import org.apache.samza.config.AzureConfig;
import org.apache.samza.container.TaskName;
import org.apache.samza.serializers.JsonSerdeV2;
import org.apache.samza.system.SystemStreamPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.Option;

public class AzureCheckpointManager
implements CheckpointManager {
    private static final Logger LOG = LoggerFactory.getLogger((String)AzureCheckpointManager.class.getName());
    private static final String PARTITION_KEY = "PartitionKey";
    public static final String REGEX_INVALID_KEY = ".*[#?/\\\\].*";
    public static final String REGEX_TABLE_NAME = "[^A-Za-z0-9]";
    public static final int MAX_WRITE_BATCH_SIZE = 100;
    public static final String SYSTEM_PROP_NAME = "system";
    public static final String STREAM_PROP_NAME = "stream";
    public static final String PARTITION_PROP_NAME = "partition";
    private final String jobTableName;
    private final String storageConnectionString;
    private final AzureClient azureClient;
    private CloudTable cloudTable;
    private final Set<TaskName> taskNames = new HashSet<TaskName>();
    private final JsonSerdeV2<Map<String, String>> jsonSerde = new JsonSerdeV2();

    AzureCheckpointManager(AzureConfig azureConfig, Option<String> jobName) {
        if (!jobName.isDefined()) {
            throw new AzureException("Jobs must have a name to use Azure Checkpoint Manager");
        }
        this.jobTableName = ((String)jobName.get()).replaceAll(REGEX_TABLE_NAME, "");
        this.storageConnectionString = azureConfig.getAzureConnectionString();
        this.azureClient = new AzureClient(this.storageConnectionString);
    }

    public void start() {
        try {
            this.cloudTable = this.azureClient.getTableClient().getTableReference(this.jobTableName);
            this.cloudTable.createIfNotExists();
        }
        catch (URISyntaxException e) {
            LOG.error("Connection string {} specifies an invalid URI while creating checkpoint table.", (Object)this.storageConnectionString);
            throw new AzureException(e);
        }
        catch (StorageException e) {
            LOG.error("Azure Storage failed when creating checkpoint table", (Throwable)e);
            throw new AzureException(e);
        }
    }

    public void register(TaskName taskName) {
        this.taskNames.add(taskName);
    }

    public void writeCheckpoint(TaskName taskName, Checkpoint checkpoint) {
        if (!this.taskNames.contains(taskName)) {
            throw new SamzaException("writing checkpoint of unregistered task");
        }
        TableBatchOperation batchOperation = new TableBatchOperation();
        Iterator iterator = checkpoint.getOffsets().entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry entry = iterator.next();
            SystemStreamPartition ssp = (SystemStreamPartition)entry.getKey();
            String offset = (String)entry.getValue();
            String partitionKey = taskName.toString();
            this.checkValidKey(partitionKey, "Taskname");
            String rowKey = this.serializeSystemStreamPartition(ssp);
            this.checkValidKey(rowKey, "SystemStreamPartition");
            TaskCheckpointEntity taskCheckpoint = new TaskCheckpointEntity(partitionKey, rowKey, offset);
            batchOperation.insertOrReplace((TableEntity)taskCheckpoint);
            if (batchOperation.size() < 100 && iterator.hasNext()) continue;
            try {
                this.cloudTable.execute(batchOperation);
            }
            catch (StorageException e) {
                LOG.error("Executing batch failed for task: {}", (Object)taskName);
                throw new AzureException(e);
            }
            batchOperation.clear();
        }
    }

    private void checkValidKey(String key, String fieldUsed) {
        if (key == null || key.matches(REGEX_INVALID_KEY)) {
            throw new AzureException(String.format("Cannot insert to Azure Checkpoint Manager; %s %s contains invalid characters [*, /, \\\\, ?]", fieldUsed, key));
        }
    }

    private String serializeSystemStreamPartition(SystemStreamPartition ssp) {
        HashMap<String, String> sspMap = new HashMap<String, String>();
        sspMap.put(SYSTEM_PROP_NAME, ssp.getSystem());
        sspMap.put(STREAM_PROP_NAME, ssp.getStream());
        sspMap.put(PARTITION_PROP_NAME, String.valueOf(ssp.getPartition().getPartitionId()));
        return new String(this.jsonSerde.toBytes(sspMap));
    }

    private SystemStreamPartition deserializeSystemStreamPartition(String serializedSSP) {
        Map sspPropertiesMap = (Map)this.jsonSerde.fromBytes(serializedSSP.getBytes());
        String systemName = (String)sspPropertiesMap.get(SYSTEM_PROP_NAME);
        String streamName = (String)sspPropertiesMap.get(STREAM_PROP_NAME);
        Partition partition = new Partition(Integer.parseInt((String)sspPropertiesMap.get(PARTITION_PROP_NAME)));
        return new SystemStreamPartition(systemName, streamName, partition);
    }

    public Checkpoint readLastCheckpoint(TaskName taskName) {
        if (!this.taskNames.contains(taskName)) {
            throw new SamzaException("reading checkpoint of unregistered/unwritten task");
        }
        String partitionQueryKey = taskName.toString();
        String partitionFilter = TableQuery.generateFilterCondition((String)PARTITION_KEY, (String)"eq", (String)partitionQueryKey);
        TableQuery query = TableQuery.from(TaskCheckpointEntity.class).where(partitionFilter);
        ImmutableMap.Builder builder = ImmutableMap.builder();
        try {
            for (TaskCheckpointEntity taskCheckpointEntity : this.cloudTable.execute(query)) {
                String serializedSSP = taskCheckpointEntity.getRowKey();
                builder.put((Object)this.deserializeSystemStreamPartition(serializedSSP), (Object)taskCheckpointEntity.getOffset());
            }
        }
        catch (NoSuchElementException e) {
            LOG.warn("No checkpoints found found for registered taskName={}", (Object)taskName);
            return null;
        }
        LOG.debug("Received checkpoint state for taskName=%s", (Object)taskName);
        return new Checkpoint((Map)builder.build());
    }

    public void stop() {
    }

    public void clearCheckpoints() {
        LOG.debug("Clearing all checkpoints in Azure table");
        for (TaskName taskName : this.taskNames) {
            String partitionQueryKey = taskName.toString();
            String partitionFilter = TableQuery.generateFilterCondition((String)PARTITION_KEY, (String)"eq", (String)partitionQueryKey);
            TableQuery partitionQuery = TableQuery.from(TaskCheckpointEntity.class).where(partitionFilter);
            this.deleteEntities(this.cloudTable.execute(partitionQuery).iterator());
        }
    }

    private void deleteEntities(Iterator<TaskCheckpointEntity> entitiesToDelete) {
        TableBatchOperation batchOperation = new TableBatchOperation();
        while (entitiesToDelete.hasNext()) {
            TaskCheckpointEntity entity = entitiesToDelete.next();
            batchOperation.delete((TableEntity)entity);
            if (batchOperation.size() < 100 && entitiesToDelete.hasNext()) continue;
            try {
                this.cloudTable.execute(batchOperation);
            }
            catch (StorageException e) {
                LOG.error("Executing batch failed for deleting checkpoints");
                throw new AzureException(e);
            }
            batchOperation.clear();
        }
    }
}

