/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.logservice.tool;

import com.beust.jcommander.IParameterValidator;
import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;
import com.beust.jcommander.validators.PositiveInteger;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.ratis.logservice.api.LogInfo;
import org.apache.ratis.logservice.api.LogName;
import org.apache.ratis.logservice.api.LogReader;
import org.apache.ratis.logservice.api.LogServiceClient;
import org.apache.ratis.logservice.api.LogStream;
import org.apache.ratis.logservice.api.LogWriter;
import org.apache.ratis.logservice.common.LogNotFoundException;
import org.apache.ratis.logservice.server.LogStateMachine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VerificationTool {
    public static final Logger LOG = LoggerFactory.getLogger(LogStateMachine.class);
    @Parameter(names={"-q", "--metaQuorum"}, description="Metadata Service Quorum", required=true)
    private String metaQuorum;
    @Parameter(names={"-nl", "--numLogs"}, description="Number of logs", validateWith={NonZeroPositiveInteger.class})
    private int numLogs = 10;
    @Parameter(names={"-nr", "--numRecords"}, description="Number of records to write per log", validateWith={NonZeroPositiveInteger.class})
    private int numRecords = 1000;
    @Parameter(names={"-w", "--write"}, description="Write to the logs", arity=1)
    private boolean write = true;
    @Parameter(names={"-r", "--read"}, description="Read the logs", arity=1)
    private boolean read = true;
    @Parameter(names={"-l", "--logFrequency"}, description="Print update every N operations", validateWith={NonZeroPositiveInteger.class})
    private int logFrequency = 50;
    @Parameter(names={"-h", "--help"}, description="Help", help=true)
    private boolean help = false;
    @Parameter(names={"-s", "--size"}, description="Size in bytes of each value")
    private int recordSize = -1;
    @Parameter(names={"-bs", "--batchSize"}, description="Number of records in a batch, a value of 0 disables batching", validateWith={PositiveInteger.class})
    private int batchSize = 0;
    public static final String LOG_NAME_PREFIX = "testlog";
    public static final String MESSAGE_PREFIX = "message";

    public static void main(String[] args) throws IOException {
        VerificationTool tool = new VerificationTool();
        JCommander jc = JCommander.newBuilder().addObject((Object)tool).build();
        jc.parse(args);
        if (tool.help) {
            jc.usage();
            return;
        }
        System.out.println(tool.metaQuorum);
        LogServiceClient client = new LogServiceClient(tool.metaQuorum);
        ExecutorService executor = Executors.newCachedThreadPool();
        ArrayList futures = new ArrayList(tool.numLogs);
        if (tool.write) {
            int i;
            LOG.info("Executing parallel writes");
            HashSet<LogName> logsInSystem = new HashSet<LogName>();
            List<LogInfo> listOfLogs = client.listLogs();
            for (LogInfo logInfo : listOfLogs) {
                logsInSystem.add(logInfo.getLogName());
            }
            LOG.info("Observed logs already in system: {}", logsInSystem);
            for (i = 0; i < tool.numLogs; ++i) {
                LogName logName = VerificationTool.getLogName(i);
                if (!logsInSystem.contains(logName)) continue;
                LOG.info("Deleting {}", (Object)logName);
                client.deleteLog(logName);
            }
            if (tool.batchSize > 0) {
                int numBatches = tool.numRecords / tool.batchSize;
                for (int i2 = 0; i2 < tool.numLogs; ++i2) {
                    BatchWriter writer = new BatchWriter(VerificationTool.getLogName(i2), client, tool.numRecords, tool.logFrequency, tool.recordSize, tool.batchSize, numBatches);
                    futures.add(executor.submit(writer));
                }
            } else {
                for (i = 0; i < tool.numLogs; ++i) {
                    BulkWriter writer = new BulkWriter(VerificationTool.getLogName(i), client, tool.numRecords, tool.logFrequency, tool.recordSize);
                    futures.add(executor.submit(writer));
                }
            }
            VerificationTool.waitForCompletion(futures);
        }
        if (tool.read) {
            LOG.info("Executing parallel reads");
            futures = new ArrayList(tool.numLogs);
            for (int i = 0; i < tool.numLogs; ++i) {
                BulkReader reader = new BulkReader(VerificationTool.getLogName(i), client, tool.numRecords, tool.logFrequency, tool.recordSize);
                futures.add(executor.submit(reader));
            }
            VerificationTool.waitForCompletion(futures);
        }
        executor.shutdownNow();
    }

    private static LogName getLogName(int id) {
        return LogName.of(LOG_NAME_PREFIX + id);
    }

    private static void waitForCompletion(List<Future<?>> futures) {
        for (Future<?> future : futures) {
            try {
                Object object = future.get();
                if (object == null) continue;
                LOG.error("Operation failed with error ", object);
                System.exit(-1);
            }
            catch (Exception e) {
                LOG.error("Got exception ", (Throwable)e);
                System.exit(-1);
            }
        }
        LOG.info("All operations finished");
    }

    static class BulkReader
    extends Operation {
        BulkReader(LogName logName, LogServiceClient client, int numRecords, int logFreq, int valueSize) {
            super(logName, client, numRecords, logFreq, valueSize);
        }

        @Override
        public void run() {
            try {
                LogStream logStream = this.getClient().getLog(this.getLogName());
                LogReader reader = logStream.createReader();
                long size = logStream.getLength();
                if (size != (long)this.getNumRecords()) {
                    LOG.error("There is mismatch is number of records. Expected Records: " + this.getNumRecords() + ", Actual Records: " + size);
                    System.exit(-1);
                }
                int i = 0;
                while ((long)i < size) {
                    ByteBuffer buffer = reader.readNext();
                    String message = this.parseValue(buffer);
                    if (i % this.getLogFreq() == 0) {
                        LOG.info(this.getLogName() + " Read " + message);
                    }
                    if (!message.equals(VerificationTool.MESSAGE_PREFIX + i)) {
                        LOG.error("Message is not correct. Expected: " + VerificationTool.MESSAGE_PREFIX + i + ". Actual:" + message);
                        System.exit(-1);
                    }
                    ++i;
                }
                LOG.info("{} log entries read from log {} successfully.", (Object)this.getNumRecords(), (Object)this.getLogName());
                reader.close();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    static class BatchWriter
    extends Operation {
        private int batchSize;
        private int numBatches;

        BatchWriter(LogName logName, LogServiceClient client, int numRecords, int logFreq, int valueSize, int batchSize, int numBatches) {
            super(logName, client, numRecords, logFreq, valueSize);
            this.batchSize = batchSize;
            this.numBatches = numBatches;
        }

        @Override
        public void run() {
            try {
                LogWriter writer = this.getLogWriter();
                for (int i = 0; i < this.numBatches; ++i) {
                    ArrayList<ByteBuffer> messages = new ArrayList<ByteBuffer>(this.batchSize);
                    for (int j = 0; j < this.batchSize; ++j) {
                        String message = VerificationTool.MESSAGE_PREFIX + (i * this.batchSize + j);
                        messages.add(this.createValue(message));
                        if ((i * this.batchSize + j) % this.getLogFreq() != 0) continue;
                        LOG.info(this.getLogName() + " batching write " + message);
                    }
                    try {
                        writer.write(messages);
                        continue;
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
                if (this.getNumRecords() % this.batchSize != 0) {
                    ArrayList<ByteBuffer> lastBatch = new ArrayList<ByteBuffer>();
                    for (int i = this.numBatches * this.batchSize; i < this.getNumRecords(); ++i) {
                        String message = VerificationTool.MESSAGE_PREFIX + i;
                        lastBatch.add(this.createValue(message));
                    }
                    LOG.info(this.getLogName() + " writing last mini-batch of " + lastBatch.size() + " records");
                    try {
                        writer.write(lastBatch);
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
                LOG.info("{} entries written in batches to {} successfully.", (Object)this.getNumRecords(), (Object)this.getLogName());
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    static class BulkWriter
    extends Operation {
        BulkWriter(LogName logName, LogServiceClient client, int numRecords, int logFreq, int valueSize) {
            super(logName, client, numRecords, logFreq, valueSize);
        }

        @Override
        public void run() {
            try {
                LogWriter writer = this.getLogWriter();
                for (int i = 0; i < this.getNumRecords(); ++i) {
                    String message = VerificationTool.MESSAGE_PREFIX + i;
                    if (i % this.getLogFreq() == 0) {
                        LOG.info(this.getLogName() + " Writing " + message);
                    }
                    writer.write(this.createValue(message));
                }
                writer.close();
                LOG.info("{} entries written to {} successfully.", (Object)this.getNumRecords(), (Object)this.getLogName());
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    static abstract class Operation
    implements Runnable {
        static final byte DIVIDER_BYTE = 95;
        private final LogName logName;
        private final LogServiceClient client;
        private final int numRecords;
        private final int logFreq;
        private final int valueSize;

        public LogName getLogName() {
            return this.logName;
        }

        public int getNumRecords() {
            return this.numRecords;
        }

        public int getLogFreq() {
            return this.logFreq;
        }

        public LogServiceClient getClient() {
            return this.client;
        }

        Operation(LogName logName, LogServiceClient client, int numRecords, int logFreq, int valueSize) {
            this.logName = logName;
            this.client = client;
            this.numRecords = numRecords;
            this.logFreq = logFreq;
            this.valueSize = valueSize;
        }

        ByteBuffer createValue(String prefix) {
            if (this.valueSize == -1) {
                return ByteBuffer.wrap(prefix.getBytes(StandardCharsets.UTF_8));
            }
            byte[] value = new byte[this.valueSize];
            byte[] prefixBytes = prefix.getBytes(StandardCharsets.UTF_8);
            if (prefixBytes.length > this.valueSize) {
                System.arraycopy(prefixBytes, 0, value, 0, this.valueSize);
                return ByteBuffer.wrap(value);
            }
            System.arraycopy(prefixBytes, 0, value, 0, prefixBytes.length);
            int bytesWritten = prefixBytes.length;
            if (bytesWritten + 1 > this.valueSize) {
                return ByteBuffer.wrap(value);
            }
            value[bytesWritten] = 95;
            int bytesToGenerate = this.valueSize - ++bytesWritten;
            Random r = new Random();
            byte[] suffix = new byte[bytesToGenerate];
            r.nextBytes(suffix);
            System.arraycopy(suffix, 0, value, bytesWritten, suffix.length);
            return ByteBuffer.wrap(value);
        }

        String parseValue(ByteBuffer buff) {
            if (!buff.hasArray()) {
                throw new IllegalArgumentException("Require a ByteBuffer with a backing array");
            }
            if (this.valueSize == -1) {
                return new String(buff.array(), buff.arrayOffset(), buff.remaining(), StandardCharsets.UTF_8);
            }
            int length = buff.limit() - buff.arrayOffset();
            byte[] value = new byte[length];
            System.arraycopy(buff.array(), buff.arrayOffset(), value, 0, length);
            int dividerOffset = -1;
            for (int i = 0; i < value.length; ++i) {
                if (value[i] != 95) continue;
                dividerOffset = i;
                break;
            }
            if (dividerOffset < 0) {
                return new String(value, StandardCharsets.UTF_8);
            }
            return new String(value, 0, dividerOffset, StandardCharsets.UTF_8);
        }

        LogWriter getLogWriter() throws IOException {
            LogStream logStream = null;
            try {
                logStream = this.client.getLog(this.logName);
            }
            catch (LogNotFoundException e) {
                LOG.info("Creating {}", (Object)this.logName);
                logStream = this.client.createLog(this.logName);
            }
            return logStream.createWriter();
        }
    }

    public static class NonZeroPositiveInteger
    implements IParameterValidator {
        public void validate(String name, String value) throws ParameterException {
            int i = Integer.parseInt(value);
            if (i < 1) {
                throw new ParameterException("Parameter '" + name + "' must be positive and non-zero, but was " + value);
            }
        }
    }
}

