/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.RaftGroup;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.StateMachineException;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.ConfigurationManager;
import org.apache.ratis.server.impl.RaftConfiguration;
import org.apache.ratis.server.impl.RaftServerConstants;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.impl.StateMachineUpdater;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.raftlog.RaftLog;
import org.apache.ratis.server.raftlog.memory.MemoryRaftLog;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLog;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.SnapshotManager;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.util.TimeDuration;
import org.apache.ratis.util.Timestamp;

public class ServerState
implements Closeable {
    private final RaftGroupMemberId memberId;
    private final RaftServerImpl server;
    private final RaftLog log;
    private final ConfigurationManager configurationManager;
    private final StateMachineUpdater stateMachineUpdater;
    private final RaftStorage storage;
    private final SnapshotManager snapshotManager;
    private volatile Timestamp lastNoLeaderTime;
    private final TimeDuration noLeaderTimeout;
    private final AtomicLong currentTerm = new AtomicLong();
    private volatile RaftPeerId leaderId;
    private volatile RaftPeerId votedFor;
    private final AtomicReference<TermIndex> latestInstalledSnapshot = new AtomicReference();

    ServerState(RaftPeerId id, RaftGroup group, RaftProperties prop, RaftServerImpl server, StateMachine stateMachine) throws IOException {
        this.memberId = RaftGroupMemberId.valueOf(id, group.getGroupId());
        this.server = server;
        RaftConfiguration initialConf = RaftConfiguration.newBuilder().setConf(group.getPeers()).build();
        this.configurationManager = new ConfigurationManager(initialConf);
        RaftServerImpl.LOG.info("{}: {}", (Object)this.getMemberId(), (Object)this.configurationManager);
        File dir = ServerState.chooseStorageDir(RaftServerConfigKeys.storageDirs(prop), group.getGroupId().getUuid().toString());
        this.storage = new RaftStorage(dir, RaftServerConstants.StartupOption.REGULAR, RaftServerConfigKeys.Log.corruptionPolicy(prop));
        this.snapshotManager = new SnapshotManager(this.storage, id);
        long lastApplied = this.initStatemachine(stateMachine, group.getGroupId());
        this.leaderId = null;
        this.lastNoLeaderTime = Timestamp.currentTime();
        this.noLeaderTimeout = RaftServerConfigKeys.Notification.noLeaderTimeout(prop);
        this.log = ServerState.initRaftLog(this.getMemberId(), server, this.storage, this::setRaftConf, lastApplied, prop);
        RaftLog.Metadata metadata = this.log.loadMetadata();
        this.currentTerm.set(metadata.getTerm());
        this.votedFor = metadata.getVotedFor();
        this.stateMachineUpdater = new StateMachineUpdater(stateMachine, server, this, lastApplied, prop);
    }

    RaftGroupMemberId getMemberId() {
        return this.memberId;
    }

    static File chooseStorageDir(List<File> volumes, String targetSubDir) throws IOException {
        HashMap numberOfStorageDirPerVolume = new HashMap();
        File[] empty = new File[]{};
        ArrayList resultList = new ArrayList();
        volumes.stream().flatMap(volume -> {
            File[] dirs = Optional.ofNullable(volume.listFiles()).orElse(empty);
            numberOfStorageDirPerVolume.put(volume, dirs.length);
            return Arrays.stream(dirs);
        }).filter(dir -> targetSubDir.equals(dir.getName())).forEach(resultList::add);
        if (resultList.size() > 1) {
            throw new IOException("More than one directories found for " + targetSubDir + ": " + resultList);
        }
        if (resultList.size() == 1) {
            return (File)resultList.get(0);
        }
        return numberOfStorageDirPerVolume.entrySet().stream().min(Comparator.comparing(Map.Entry::getValue)).map(Map.Entry::getKey).map(v -> new File((File)v, targetSubDir)).orElseThrow(() -> new IOException("No storage directory found."));
    }

    private long initStatemachine(StateMachine sm, RaftGroupId gid) throws IOException {
        sm.initialize(this.server.getProxy(), gid, this.storage);
        SnapshotInfo snapshot = sm.getLatestSnapshot();
        if (snapshot == null || snapshot.getTermIndex().getIndex() < 0L) {
            return -1L;
        }
        RaftConfiguration raftConf = this.storage.readRaftConfiguration();
        if (raftConf != null) {
            this.setRaftConf(raftConf.getLogEntryIndex(), raftConf);
        }
        return snapshot.getIndex();
    }

    void writeRaftConfiguration(RaftProtos.LogEntryProto conf) {
        this.storage.writeRaftConfiguration(conf);
    }

    void start() {
        this.stateMachineUpdater.start();
    }

    private static RaftLog initRaftLog(RaftGroupMemberId memberId, RaftServerImpl server, RaftStorage storage, Consumer<RaftProtos.LogEntryProto> logConsumer, long lastIndexInSnapshot, RaftProperties prop) throws IOException {
        RaftLog log = RaftServerConfigKeys.Log.useMemory(prop) ? new MemoryRaftLog(memberId, lastIndexInSnapshot, prop) : new SegmentedRaftLog(memberId, server, storage, lastIndexInSnapshot, prop);
        log.open(lastIndexInSnapshot, logConsumer);
        return log;
    }

    RaftConfiguration getRaftConf() {
        return this.configurationManager.getCurrent();
    }

    public long getCurrentTerm() {
        return this.currentTerm.get();
    }

    boolean updateCurrentTerm(long newTerm) {
        long current = this.currentTerm.getAndUpdate(curTerm -> Math.max(curTerm, newTerm));
        if (newTerm > current) {
            this.votedFor = null;
            this.setLeader(null, "updateCurrentTerm");
            return true;
        }
        return false;
    }

    RaftPeerId getLeaderId() {
        return this.leaderId;
    }

    boolean hasLeader() {
        return this.leaderId != null;
    }

    long initElection() {
        this.votedFor = this.getMemberId().getPeerId();
        this.setLeader(null, "initElection");
        return this.currentTerm.incrementAndGet();
    }

    void persistMetadata() throws IOException {
        this.log.writeMetadata(this.currentTerm.get(), this.votedFor);
    }

    void grantVote(RaftPeerId candidateId) {
        this.votedFor = candidateId;
        this.setLeader(null, "grantVote");
    }

    void setLeader(RaftPeerId newLeaderId, String op) {
        if (!Objects.equals(this.leaderId, newLeaderId)) {
            String suffix;
            if (newLeaderId == null) {
                this.lastNoLeaderTime = Timestamp.currentTime();
                suffix = "";
            } else {
                Timestamp previous = this.lastNoLeaderTime;
                this.lastNoLeaderTime = null;
                suffix = ", leader elected after " + previous.elapsedTimeMs() + "ms";
                this.server.getStateMachine().notifyLeaderChanged(this.getMemberId(), newLeaderId);
            }
            RaftServerImpl.LOG.info("{}: change Leader from {} to {} at term {} for {}{}", this.getMemberId(), this.leaderId, newLeaderId, this.getCurrentTerm(), op, suffix);
            this.leaderId = newLeaderId;
        }
    }

    boolean shouldNotifyExtendedNoLeader() {
        return Optional.ofNullable(this.lastNoLeaderTime).map(Timestamp::elapsedTime).filter(t -> t.compareTo(this.noLeaderTimeout) > 0).isPresent();
    }

    public long getLastLeaderElapsedTimeMs() {
        Timestamp t = this.lastNoLeaderTime;
        return t == null ? 0L : t.elapsedTimeMs();
    }

    void becomeLeader() {
        this.setLeader(this.getMemberId().getPeerId(), "becomeLeader");
    }

    public RaftLog getLog() {
        return this.log;
    }

    void appendLog(TransactionContext operation) throws StateMachineException {
        this.log.append(this.currentTerm.get(), operation);
        Objects.requireNonNull(operation.getLogEntry());
    }

    boolean recognizeLeader(RaftPeerId leaderId, long leaderTerm) {
        long current = this.currentTerm.get();
        if (leaderTerm < current) {
            return false;
        }
        if (leaderTerm > current || this.leaderId == null) {
            return true;
        }
        return this.leaderId.equals(leaderId);
    }

    boolean recognizeCandidate(RaftPeerId candidateId, long candidateTerm) {
        if (!this.getRaftConf().containsInConf(candidateId)) {
            return false;
        }
        long current = this.currentTerm.get();
        if (candidateTerm > current) {
            return true;
        }
        if (candidateTerm == current) {
            return this.votedFor == null || this.votedFor.equals(candidateId);
        }
        return false;
    }

    boolean isLogUpToDate(TermIndex candidateLastEntry) {
        TermIndex local = this.log.getLastEntryTermIndex();
        SnapshotInfo snapshot = this.server.getStateMachine().getLatestSnapshot();
        if (local == null && snapshot == null) {
            return true;
        }
        if (candidateLastEntry == null) {
            return false;
        }
        if (local == null || snapshot != null && snapshot.getIndex() > local.getIndex()) {
            local = snapshot.getTermIndex();
        }
        return local.compareTo(candidateLastEntry) <= 0;
    }

    public String toString() {
        return this.getMemberId() + ":t" + this.currentTerm + ", leader=" + this.leaderId + ", voted=" + this.votedFor + ", raftlog=" + this.log + ", conf=" + this.getRaftConf();
    }

    boolean isConfCommitted() {
        return this.getLog().getLastCommittedIndex() >= this.getRaftConf().getLogEntryIndex();
    }

    void setRaftConf(RaftProtos.LogEntryProto entry) {
        if (entry.hasConfigurationEntry()) {
            this.setRaftConf(entry.getIndex(), ServerProtoUtils.toRaftConfiguration(entry));
        }
    }

    void setRaftConf(long logIndex, RaftConfiguration conf) {
        this.configurationManager.addConfiguration(logIndex, conf);
        this.server.getServerRpc().addPeers(conf.getPeers());
        RaftServerImpl.LOG.info("{}: set configuration {} at {}", this.getMemberId(), conf, logIndex);
        RaftServerImpl.LOG.trace("{}: {}", (Object)this.getMemberId(), (Object)this.configurationManager);
    }

    void updateConfiguration(RaftProtos.LogEntryProto[] entries) {
        if (entries != null && entries.length > 0) {
            this.configurationManager.removeConfigurations(entries[0].getIndex());
            Arrays.stream(entries).forEach(this::setRaftConf);
        }
    }

    boolean updateStatemachine(long majorityIndex, long currentTerm) {
        if (this.log.updateLastCommitted(majorityIndex, currentTerm)) {
            this.stateMachineUpdater.notifyUpdater();
            return true;
        }
        return false;
    }

    void reloadStateMachine(long lastIndexInSnapshot, long currentTerm) {
        this.log.updateLastCommitted(lastIndexInSnapshot, currentTerm);
        this.stateMachineUpdater.reloadStateMachine();
    }

    @Override
    public void close() throws IOException {
        try {
            this.stateMachineUpdater.stopAndJoin();
        }
        catch (InterruptedException e) {
            RaftServerImpl.LOG.warn("{}: Interrupted when joining stateMachineUpdater", (Object)this.getMemberId(), (Object)e);
        }
        RaftServerImpl.LOG.info("{}: closes. applyIndex: {}", (Object)this.getMemberId(), (Object)this.getLastAppliedIndex());
        this.log.close();
        this.storage.close();
    }

    public RaftStorage getStorage() {
        return this.storage;
    }

    void installSnapshot(RaftProtos.InstallSnapshotRequestProto request) throws IOException {
        StateMachine sm = this.server.getStateMachine();
        sm.pause();
        this.snapshotManager.installSnapshot(sm, request);
        this.updateInstalledSnapshotIndex(ServerProtoUtils.toTermIndex(request.getSnapshotChunk().getTermIndex()));
    }

    void updateInstalledSnapshotIndex(TermIndex lastTermIndexInSnapshot) {
        this.log.syncWithSnapshot(lastTermIndexInSnapshot.getIndex());
        this.latestInstalledSnapshot.set(lastTermIndexInSnapshot);
    }

    SnapshotInfo getLatestSnapshot() {
        return this.server.getStateMachine().getLatestSnapshot();
    }

    public long getLatestInstalledSnapshotIndex() {
        TermIndex ti = this.latestInstalledSnapshot.get();
        return ti != null ? ti.getIndex() : 0L;
    }

    long getSnapshotIndex() {
        SnapshotInfo s2 = this.getLatestSnapshot();
        long latestSnapshotIndex = s2 != null ? s2.getIndex() : 0L;
        return Math.max(latestSnapshotIndex, this.getLatestInstalledSnapshotIndex());
    }

    public long getNextIndex() {
        long logNextIndex = this.log.getNextIndex();
        long snapshotNextIndex = this.getSnapshotIndex() + 1L;
        return Math.max(logNextIndex, snapshotNextIndex);
    }

    public long getLastAppliedIndex() {
        return this.stateMachineUpdater.getStateMachineLastAppliedIndex();
    }

    boolean containsTermIndex(TermIndex ti) {
        Objects.requireNonNull(ti, "ti == null");
        if (Optional.ofNullable(this.latestInstalledSnapshot.get()).filter(ti::equals).isPresent()) {
            return true;
        }
        if (Optional.ofNullable(this.getLatestSnapshot()).map(SnapshotInfo::getTermIndex).filter(ti::equals).isPresent()) {
            return true;
        }
        return this.log.contains(ti);
    }
}

