/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.UUID;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.FollowerInfo;
import org.apache.ratis.server.impl.LeaderState;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.raftlog.RaftLog;
import org.apache.ratis.server.raftlog.RaftLogIOException;
import org.apache.ratis.server.storage.FileInfo;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.thirdparty.com.google.protobuf.GeneratedMessageV3;
import org.apache.ratis.util.Daemon;
import org.apache.ratis.util.DataQueue;
import org.apache.ratis.util.IOUtils;
import org.apache.ratis.util.LifeCycle;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.SizeInBytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogAppender {
    public static final Logger LOG = LoggerFactory.getLogger(LogAppender.class);
    private final String name;
    protected final RaftServerImpl server;
    private final LeaderState leaderState;
    protected final RaftLog raftLog;
    protected final FollowerInfo follower;
    private final DataQueue<RaftLog.EntryWithData> buffer;
    private final int snapshotChunkMaxSize;
    protected final long halfMinTimeoutMs;
    private final AppenderDaemon daemon;

    public LogAppender(RaftServerImpl server, LeaderState leaderState, FollowerInfo f) {
        this.follower = f;
        this.name = this.follower.getName() + "-" + this.getClass().getSimpleName();
        this.server = server;
        this.leaderState = leaderState;
        this.raftLog = server.getState().getLog();
        RaftProperties properties = server.getProxy().getProperties();
        this.snapshotChunkMaxSize = RaftServerConfigKeys.Log.Appender.snapshotChunkSizeMax(properties).getSizeInt();
        this.halfMinTimeoutMs = server.getMinTimeoutMs() / 2;
        SizeInBytes bufferByteLimit = RaftServerConfigKeys.Log.Appender.bufferByteLimit(properties);
        int bufferElementLimit = RaftServerConfigKeys.Log.Appender.bufferElementLimit(properties);
        this.buffer = new DataQueue<RaftLog.EntryWithData>(this, bufferByteLimit, bufferElementLimit, RaftLog.EntryWithData::getSerializedSize);
        this.daemon = new AppenderDaemon();
    }

    public String toString() {
        return this.name;
    }

    void startAppender() {
        this.daemon.start();
    }

    public boolean isAppenderRunning() {
        return this.daemon.isRunning();
    }

    void stopAppender() {
        this.daemon.stop();
    }

    public FollowerInfo getFollower() {
        return this.follower;
    }

    protected RaftPeerId getFollowerId() {
        return this.getFollower().getPeer().getId();
    }

    private TermIndex getPrevious(long nextIndex) {
        TermIndex snapshotTermIndex;
        if (nextIndex == 0L) {
            return null;
        }
        long previousIndex = nextIndex - 1L;
        TermIndex previous = this.raftLog.getTermIndex(previousIndex);
        if (previous != null) {
            return previous;
        }
        SnapshotInfo snapshot = this.server.getState().getLatestSnapshot();
        if (snapshot != null && (snapshotTermIndex = snapshot.getTermIndex()).getIndex() == previousIndex) {
            return snapshotTermIndex;
        }
        return null;
    }

    protected RaftProtos.AppendEntriesRequestProto createRequest(long callId) throws RaftLogIOException {
        TermIndex previous = this.getPrevious(this.follower.getNextIndex());
        long heartbeatRemainingMs = this.getHeartbeatRemainingTime();
        if (heartbeatRemainingMs <= 0L) {
            return this.leaderState.newAppendEntriesRequestProto(this.getFollowerId(), previous, Collections.emptyList(), !this.follower.isAttendingVote(), callId);
        }
        Preconditions.assertTrue(this.buffer.isEmpty(), () -> "buffer has " + this.buffer.getNumElements() + " elements.");
        long leaderNext = this.raftLog.getNextIndex();
        long followerNext = this.follower.getNextIndex();
        long halfMs = heartbeatRemainingMs / 2L;
        long next = followerNext;
        while (leaderNext > next && this.getHeartbeatRemainingTime() - halfMs > 0L && this.buffer.offer(this.raftLog.getEntryWithData(next++))) {
        }
        if (this.buffer.isEmpty()) {
            return null;
        }
        List<RaftProtos.LogEntryProto> protos = this.buffer.pollList(this.getHeartbeatRemainingTime(), RaftLog.EntryWithData::getEntry, (entry, time, exception) -> LOG.warn("{}: Failed to get {} in {}: {}", this.follower.getName(), entry, time, exception));
        this.buffer.clear();
        this.assertProtos(protos, followerNext, previous);
        return this.leaderState.newAppendEntriesRequestProto(this.getFollowerId(), previous, protos, !this.follower.isAttendingVote(), callId);
    }

    private void assertProtos(List<RaftProtos.LogEntryProto> protos, long nextIndex, TermIndex previous) {
        if (protos.isEmpty()) {
            return;
        }
        long firstIndex = protos.get(0).getIndex();
        Preconditions.assertTrue(firstIndex == nextIndex, () -> this.follower.getName() + ": firstIndex = " + firstIndex + " != nextIndex = " + nextIndex);
        if (firstIndex > 0L) {
            Objects.requireNonNull(previous, () -> this.follower.getName() + ": Previous TermIndex not found for firstIndex = " + firstIndex);
            Preconditions.assertTrue(previous.getIndex() == firstIndex - 1L, () -> this.follower.getName() + ": Previous = " + previous + " but firstIndex = " + firstIndex);
        }
    }

    private RaftProtos.AppendEntriesReplyProto sendAppendEntriesWithRetries() throws InterruptedException, InterruptedIOException, RaftLogIOException {
        int retry = 0;
        RaftProtos.AppendEntriesRequestProto request = null;
        while (this.isAppenderRunning()) {
            try {
                if (request == null || request.getEntriesCount() == 0) {
                    request = this.createRequest(0L);
                }
                if (request == null) {
                    LOG.trace("{} no entries to send now, wait ...", (Object)this);
                    return null;
                }
                if (!this.isAppenderRunning()) {
                    LOG.info("{} is stopped. Skip appendEntries.", (Object)this);
                    return null;
                }
                this.follower.updateLastRpcSendTime();
                RaftProtos.AppendEntriesReplyProto r = this.server.getServerRpc().appendEntries(request);
                this.follower.updateLastRpcResponseTime();
                this.updateCommitIndex(r.getFollowerCommit());
                return r;
            }
            catch (InterruptedIOException | RaftLogIOException e) {
                throw e;
            }
            catch (IOException ioe) {
                if (retry++ % 10 == 0) {
                    LOG.warn("{}: Failed to appendEntries (retry={}): {}", this, retry++, ioe);
                }
                this.handleException(ioe);
                if (!this.isAppenderRunning()) continue;
                this.leaderState.getSyncInterval().sleep();
            }
        }
        return null;
    }

    protected void updateCommitIndex(long commitIndex) {
        if (this.follower.updateCommitIndex(commitIndex)) {
            this.leaderState.commitIndexChanged();
        }
    }

    protected RaftProtos.InstallSnapshotRequestProto createInstallSnapshotNotificationRequest(TermIndex firstLogStartTermIndex) {
        return this.server.createInstallSnapshotRequest(this.getFollowerId(), firstLogStartTermIndex);
    }

    private RaftProtos.FileChunkProto readFileChunk(FileInfo fileInfo, FileInputStream in, byte[] buf, int length, long offset, int chunkIndex) throws IOException {
        RaftProtos.FileChunkProto.Builder builder = RaftProtos.FileChunkProto.newBuilder().setOffset(offset).setChunkIndex(chunkIndex);
        IOUtils.readFully(in, buf, 0, length);
        Path relativePath = this.server.getState().getStorage().getStorageDir().relativizeToRoot(fileInfo.getPath());
        builder.setFilename(relativePath.toString());
        builder.setDone(offset + (long)length == fileInfo.getFileSize());
        builder.setFileDigest(ByteString.copyFrom(fileInfo.getFileDigest().getDigest()));
        builder.setData(ByteString.copyFrom(buf, 0, length));
        return builder.build();
    }

    private RaftProtos.InstallSnapshotReplyProto installSnapshot(SnapshotInfo snapshot) throws InterruptedIOException {
        String requestId = UUID.randomUUID().toString();
        RaftProtos.InstallSnapshotReplyProto reply = null;
        try {
            for (RaftProtos.InstallSnapshotRequestProto request : new SnapshotRequestIter(snapshot, requestId)) {
                this.follower.updateLastRpcSendTime();
                reply = this.server.getServerRpc().installSnapshot(request);
                this.follower.updateLastRpcResponseTime();
                if (reply.getServerReply().getSuccess()) continue;
                return reply;
            }
        }
        catch (InterruptedIOException iioe) {
            throw iioe;
        }
        catch (Exception ioe) {
            LOG.warn("{}: Failed to installSnapshot {}: {}", this, snapshot, ioe);
            this.handleException(ioe);
            return null;
        }
        if (reply != null) {
            this.follower.setSnapshotIndex(snapshot.getTermIndex().getIndex());
            LOG.info("{}: installSnapshot {} successfully", (Object)this, (Object)snapshot);
            this.server.getRaftServerMetrics().getCounter("numInstallSnapshot").inc();
        }
        return reply;
    }

    protected SnapshotInfo shouldInstallSnapshot() {
        long logStartIndex = this.raftLog.getStartIndex();
        if (this.follower.getNextIndex() < this.raftLog.getNextIndex()) {
            SnapshotInfo snapshot = this.server.getState().getLatestSnapshot();
            if (this.follower.getNextIndex() < logStartIndex || logStartIndex == -1L && snapshot != null) {
                return snapshot;
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void runAppenderImpl() throws InterruptedException, IOException {
        while (this.isAppenderRunning()) {
            long waitTime;
            if (this.shouldSendRequest()) {
                GeneratedMessageV3 r;
                SnapshotInfo snapshot = this.shouldInstallSnapshot();
                if (snapshot != null) {
                    LOG.info("{}: followerNextIndex = {} but logStartIndex = {}, send snapshot {} to follower", this, this.follower.getNextIndex(), this.raftLog.getStartIndex(), snapshot);
                    r = this.installSnapshot(snapshot);
                    if (r != null && ((RaftProtos.InstallSnapshotReplyProto)r).getResult() == RaftProtos.InstallSnapshotResult.NOT_LEADER) {
                        this.checkResponseTerm(((RaftProtos.InstallSnapshotReplyProto)r).getTerm());
                    }
                } else {
                    r = this.sendAppendEntriesWithRetries();
                    if (r != null) {
                        this.handleReply((RaftProtos.AppendEntriesReplyProto)r);
                    }
                }
            }
            if (this.isAppenderRunning() && !this.shouldAppendEntries(this.follower.getNextIndex()) && (waitTime = this.getHeartbeatRemainingTime()) > 0L) {
                LogAppender logAppender = this;
                synchronized (logAppender) {
                    this.wait(waitTime);
                }
            }
            this.checkSlowness();
        }
    }

    private void handleReply(RaftProtos.AppendEntriesReplyProto reply) {
        if (reply != null) {
            switch (reply.getResult()) {
                case SUCCESS: {
                    long oldNextIndex = this.follower.getNextIndex();
                    long nextIndex = reply.getNextIndex();
                    if (nextIndex < oldNextIndex) {
                        throw new IllegalStateException("nextIndex=" + nextIndex + " < oldNextIndex=" + oldNextIndex + ", reply=" + ServerProtoUtils.toString(reply));
                    }
                    if (nextIndex <= oldNextIndex) break;
                    this.follower.updateMatchIndex(nextIndex - 1L);
                    this.follower.increaseNextIndex(nextIndex);
                    this.submitEventOnSuccessAppend();
                    break;
                }
                case NOT_LEADER: {
                    this.checkResponseTerm(reply.getTerm());
                    break;
                }
                case INCONSISTENCY: {
                    this.follower.decreaseNextIndex(reply.getNextIndex());
                    break;
                }
                case UNRECOGNIZED: {
                    LOG.warn("{}: received {}", (Object)this, (Object)reply.getResult());
                }
            }
        }
    }

    private void handleException(Exception e) {
        LOG.trace("TRACE", e);
        this.server.getServerRpc().handleException(this.follower.getPeer().getId(), e, false);
    }

    protected void submitEventOnSuccessAppend() {
        if (this.follower.isAttendingVote()) {
            this.leaderState.submitUpdateCommitEvent();
        } else {
            this.leaderState.submitCheckStagingEvent();
        }
    }

    protected void checkSlowness() {
        if (this.follower.isSlow()) {
            this.server.getStateMachine().notifySlowness(this.server.getRoleInfoProto());
        }
        this.leaderState.recordFollowerHeartbeatElapsedTime(this.follower.getPeer(), this.follower.getLastRpcResponseTime().elapsedTime().getDuration());
    }

    public synchronized void notifyAppend() {
        this.notify();
    }

    protected boolean shouldSendRequest() {
        return this.shouldAppendEntries(this.follower.getNextIndex()) || this.shouldHeartbeat();
    }

    private boolean shouldAppendEntries(long followerIndex) {
        return followerIndex < this.raftLog.getNextIndex();
    }

    protected boolean shouldHeartbeat() {
        return this.getHeartbeatRemainingTime() <= 0L;
    }

    protected long getHeartbeatRemainingTime() {
        return this.halfMinTimeoutMs - this.follower.getLastRpcTime().elapsedTimeMs();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected boolean checkResponseTerm(long responseTerm) {
        RaftServerImpl raftServerImpl = this.server;
        synchronized (raftServerImpl) {
            if (this.isAppenderRunning() && this.follower.isAttendingVote() && responseTerm > this.leaderState.getCurrentTerm()) {
                this.leaderState.submitStepDownEvent(responseTerm);
                return true;
            }
        }
        return false;
    }

    protected class SnapshotRequestIter
    implements Iterable<RaftProtos.InstallSnapshotRequestProto> {
        private final SnapshotInfo snapshot;
        private final List<FileInfo> files;
        private FileInputStream in;
        private int fileIndex = 0;
        private FileInfo currentFileInfo;
        private byte[] currentBuf;
        private long currentFileSize;
        private long currentOffset = 0L;
        private int chunkIndex = 0;
        private final String requestId;
        private int requestIndex = 0;

        public SnapshotRequestIter(SnapshotInfo snapshot, String requestId) throws IOException {
            this.snapshot = snapshot;
            this.requestId = requestId;
            this.files = snapshot.getFiles();
            if (this.files.size() > 0) {
                this.startReadFile();
            }
        }

        private void startReadFile() throws IOException {
            this.currentFileInfo = this.files.get(this.fileIndex);
            File snapshotFile = this.currentFileInfo.getPath().toFile();
            this.currentFileSize = snapshotFile.length();
            int bufLength = this.getSnapshotChunkLength(this.currentFileSize);
            this.currentBuf = new byte[bufLength];
            this.currentOffset = 0L;
            this.chunkIndex = 0;
            this.in = new FileInputStream(snapshotFile);
        }

        private int getSnapshotChunkLength(long len) {
            return len < (long)LogAppender.this.snapshotChunkMaxSize ? (int)len : LogAppender.this.snapshotChunkMaxSize;
        }

        @Override
        public Iterator<RaftProtos.InstallSnapshotRequestProto> iterator() {
            return new Iterator<RaftProtos.InstallSnapshotRequestProto>(){

                @Override
                public boolean hasNext() {
                    return SnapshotRequestIter.this.fileIndex < SnapshotRequestIter.this.files.size();
                }

                @Override
                public RaftProtos.InstallSnapshotRequestProto next() {
                    if (SnapshotRequestIter.this.fileIndex >= SnapshotRequestIter.this.files.size()) {
                        throw new NoSuchElementException();
                    }
                    int targetLength = SnapshotRequestIter.this.getSnapshotChunkLength(SnapshotRequestIter.this.currentFileSize - SnapshotRequestIter.this.currentOffset);
                    try {
                        RaftProtos.FileChunkProto chunk = LogAppender.this.readFileChunk(SnapshotRequestIter.this.currentFileInfo, SnapshotRequestIter.this.in, SnapshotRequestIter.this.currentBuf, targetLength, SnapshotRequestIter.this.currentOffset, SnapshotRequestIter.this.chunkIndex);
                        boolean done = SnapshotRequestIter.this.fileIndex == SnapshotRequestIter.this.files.size() - 1 && chunk.getDone();
                        RaftProtos.InstallSnapshotRequestProto request = LogAppender.this.server.createInstallSnapshotRequest(LogAppender.this.follower.getPeer().getId(), SnapshotRequestIter.this.requestId, SnapshotRequestIter.this.requestIndex++, SnapshotRequestIter.this.snapshot, Collections.singletonList(chunk), done);
                        SnapshotRequestIter.this.currentOffset = SnapshotRequestIter.this.currentOffset + (long)targetLength;
                        SnapshotRequestIter.this.chunkIndex++;
                        if (SnapshotRequestIter.this.currentOffset >= SnapshotRequestIter.this.currentFileSize) {
                            SnapshotRequestIter.this.in.close();
                            SnapshotRequestIter.this.fileIndex++;
                            if (SnapshotRequestIter.this.fileIndex < SnapshotRequestIter.this.files.size()) {
                                SnapshotRequestIter.this.startReadFile();
                            }
                        }
                        return request;
                    }
                    catch (IOException e) {
                        if (SnapshotRequestIter.this.in != null) {
                            try {
                                SnapshotRequestIter.this.in.close();
                            }
                            catch (IOException iOException) {
                                // empty catch block
                            }
                        }
                        LOG.warn("{}: Failed to prepare installSnapshot request", (Object)LogAppender.this, (Object)e);
                        throw new RuntimeException(e);
                    }
                }
            };
        }
    }

    class AppenderDaemon {
        private final String name;
        private final LifeCycle lifeCycle;
        private final Daemon daemon;

        AppenderDaemon() {
            this.name = LogAppender.this + "-" + this.getClass().getSimpleName();
            this.lifeCycle = new LifeCycle(this.name);
            this.daemon = new Daemon(this::run);
        }

        void start() {
            if (this.lifeCycle.compareAndTransition(LifeCycle.State.NEW, LifeCycle.State.STARTING)) {
                this.daemon.start();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void run() {
            LifeCycle lifeCycle = this.lifeCycle;
            synchronized (lifeCycle) {
                if (!this.isRunning()) {
                    return;
                }
                this.lifeCycle.transition(LifeCycle.State.RUNNING);
            }
            try {
                LogAppender.this.runAppenderImpl();
            }
            catch (InterruptedIOException | InterruptedException e) {
                LOG.info(this + " was interrupted: " + e);
            }
            catch (RaftLogIOException e) {
                LOG.error(this + " failed RaftLog", e);
                this.lifeCycle.transition(LifeCycle.State.EXCEPTION);
            }
            catch (IOException e) {
                LOG.error(this + " failed IOException", e);
                this.lifeCycle.transition(LifeCycle.State.EXCEPTION);
            }
            catch (Throwable e) {
                LOG.error(this + " unexpected exception", e);
                this.lifeCycle.transition(LifeCycle.State.EXCEPTION);
            }
            finally {
                if (!this.lifeCycle.compareAndTransition(LifeCycle.State.CLOSING, LifeCycle.State.CLOSED)) {
                    this.lifeCycle.transitionIfNotEqual(LifeCycle.State.EXCEPTION);
                }
                if (this.lifeCycle.getCurrentState() == LifeCycle.State.EXCEPTION) {
                    LogAppender.this.leaderState.restartSender(LogAppender.this);
                }
            }
        }

        boolean isRunning() {
            return !LifeCycle.States.CLOSING_OR_CLOSED_OR_EXCEPTION.contains((Object)this.lifeCycle.getCurrentState());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void stop() {
            LifeCycle lifeCycle = this.lifeCycle;
            synchronized (lifeCycle) {
                if (!this.isRunning()) {
                    return;
                }
                if (this.lifeCycle.compareAndTransition(LifeCycle.State.NEW, LifeCycle.State.CLOSED)) {
                    return;
                }
                this.lifeCycle.transition(LifeCycle.State.CLOSING);
            }
            this.daemon.interrupt();
        }

        public String toString() {
            return this.name;
        }
    }
}

