/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.examples.filestore;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.ratis.examples.filestore.FileInfo;
import org.apache.ratis.examples.filestore.FileStoreCommon;
import org.apache.ratis.proto.ExamplesProtos;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.CollectionUtils;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.LogUtils;
import org.apache.ratis.util.StringUtils;
import org.apache.ratis.util.function.CheckedSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileStore
implements Closeable {
    public static final Logger LOG = LoggerFactory.getLogger(FileStore.class);
    private final Supplier<RaftPeerId> idSupplier;
    private final Supplier<Path> rootSupplier;
    private final FileMap files;
    private final ExecutorService writer = Executors.newFixedThreadPool(10);
    private final ExecutorService committer = Executors.newFixedThreadPool(3);
    private final ExecutorService reader = Executors.newFixedThreadPool(10);
    private final ExecutorService deleter = Executors.newFixedThreadPool(3);

    public FileStore(Supplier<RaftPeerId> idSupplier, Path dir) {
        this.idSupplier = idSupplier;
        this.rootSupplier = JavaUtils.memoize(() -> dir.resolve(this.getId().toString()).normalize().toAbsolutePath());
        this.files = new FileMap(JavaUtils.memoize(() -> idSupplier.get() + ":files"));
    }

    public RaftPeerId getId() {
        return Objects.requireNonNull(this.idSupplier.get(), this.getClass().getSimpleName() + " is not initialized.");
    }

    public Path getRoot() {
        return this.rootSupplier.get();
    }

    static Path normalize(String path) {
        Objects.requireNonNull(path, "path == null");
        return Paths.get(path, new String[0]).normalize();
    }

    Path resolve(Path relative) throws IOException {
        Path root = this.getRoot();
        Path full = root.resolve(relative).normalize().toAbsolutePath();
        if (full.equals(root)) {
            throw new IOException("The file path " + relative + " resolved to " + full + " is the root directory " + root);
        }
        if (!full.startsWith(root)) {
            throw new IOException("The file path " + relative + " resolved to " + full + " is not a sub-path under root directory " + root);
        }
        return full;
    }

    CompletableFuture<ExamplesProtos.ReadReplyProto> read(String relative, long offset, long length) {
        Supplier<String> name = () -> "read(" + relative + ", " + offset + ", " + length + ") @" + this.getId();
        CheckedSupplier task = LogUtils.newCheckedSupplier(LOG, () -> {
            FileInfo info = this.files.get(relative);
            ExamplesProtos.ReadReplyProto.Builder reply = ExamplesProtos.ReadReplyProto.newBuilder().setResolvedPath(FileStoreCommon.toByteString(info.getRelativePath())).setOffset(offset);
            ByteString bytes = info.read(this::resolve, offset, length);
            return reply.setData(bytes).build();
        }, name);
        return FileStore.submit(task, this.reader);
    }

    CompletableFuture<Path> delete(long index, String relative) {
        Supplier<String> name = () -> "delete(" + relative + ") @" + this.getId() + ":" + index;
        CheckedSupplier task = LogUtils.newCheckedSupplier(LOG, () -> {
            FileInfo info = this.files.remove(relative);
            FileUtils.delete(this.resolve(info.getRelativePath()));
            return info.getRelativePath();
        }, name);
        return FileStore.submit(task, this.deleter);
    }

    static <T> CompletableFuture<T> submit(CheckedSupplier<T, IOException> task, ExecutorService executor) {
        CompletableFuture f = new CompletableFuture();
        executor.submit(() -> {
            try {
                f.complete(task.get());
            }
            catch (IOException e) {
                f.completeExceptionally(new IOException("Failed " + task, e));
            }
        });
        return f;
    }

    CompletableFuture<ExamplesProtos.WriteReplyProto> submitCommit(long index, String relative, boolean close, long offset, int size) {
        FileInfo.UnderConstruction uc;
        Function<FileInfo.UnderConstruction, FileInfo.ReadOnly> converter = close ? this.files::close : null;
        try {
            uc = this.files.get(relative).asUnderConstruction();
        }
        catch (FileNotFoundException e) {
            return FileStoreCommon.completeExceptionally(index, "Failed to write to " + relative, e);
        }
        return uc.submitCommit(offset, size, converter, this.committer, this.getId(), index).thenApply(n -> ExamplesProtos.WriteReplyProto.newBuilder().setResolvedPath(FileStoreCommon.toByteString(uc.getRelativePath())).setOffset(offset).setLength(n.intValue()).build());
    }

    CompletableFuture<Integer> write(long index, String relative, boolean close, long offset, ByteString data) {
        FileInfo.UnderConstruction uc;
        boolean createNew;
        int size = data != null ? data.size() : 0;
        LOG.trace("write {}, offset={}, size={}, close? {} @{}:{}", relative, offset, size, close, this.getId(), index);
        boolean bl = createNew = offset == 0L;
        if (createNew) {
            uc = new FileInfo.UnderConstruction(FileStore.normalize(relative));
            this.files.putNew(uc);
        } else {
            try {
                uc = this.files.get(relative).asUnderConstruction();
            }
            catch (FileNotFoundException e) {
                return FileStoreCommon.completeExceptionally(index, "Failed to write to " + relative, e);
            }
        }
        return size == 0 && !close ? CompletableFuture.completedFuture(0) : (createNew ? uc.submitCreate(this::resolve, data, close, this.writer, this.getId(), index) : uc.submitWrite(offset, data, close, this.writer, this.getId(), index));
    }

    @Override
    public void close() {
        this.writer.shutdownNow();
        this.committer.shutdownNow();
        this.reader.shutdownNow();
        this.deleter.shutdownNow();
    }

    static class FileMap {
        private final Object name;
        private final Map<Path, FileInfo> map = new ConcurrentHashMap<Path, FileInfo>();

        FileMap(Supplier<String> name) {
            this.name = StringUtils.stringSupplierAsObject(name);
        }

        FileInfo get(String relative) throws FileNotFoundException {
            return this.applyFunction(relative, this.map::get);
        }

        FileInfo remove(String relative) throws FileNotFoundException {
            LOG.trace("{}: remove {}", this.name, (Object)relative);
            return this.applyFunction(relative, this.map::remove);
        }

        private FileInfo applyFunction(String relative, Function<Path, FileInfo> f) throws FileNotFoundException {
            FileInfo info = f.apply(FileStore.normalize(relative));
            if (info == null) {
                throw new FileNotFoundException("File " + relative + " not found in " + this.name);
            }
            return info;
        }

        void putNew(FileInfo.UnderConstruction uc) {
            LOG.trace("{}: putNew {}", this.name, (Object)uc.getRelativePath());
            CollectionUtils.putNew(uc.getRelativePath(), uc, this.map, this.name::toString);
        }

        FileInfo.ReadOnly close(FileInfo.UnderConstruction uc) {
            LOG.trace("{}: close {}", this.name, (Object)uc.getRelativePath());
            FileInfo.ReadOnly ro = new FileInfo.ReadOnly(uc);
            CollectionUtils.replaceExisting(uc.getRelativePath(), uc, ro, this.map, this.name::toString);
            return ro;
        }
    }
}

