/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.raftlog.segmented;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.Consumer;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.metrics.RaftLogMetrics;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.raftlog.segmented.CacheInvalidationPolicy;
import org.apache.ratis.server.raftlog.segmented.LogSegment;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.RaftStorageDirectory;
import org.apache.ratis.util.AutoCloseableLock;
import org.apache.ratis.util.AutoCloseableReadWriteLock;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SegmentedRaftLogCache {
    public static final Logger LOG = LoggerFactory.getLogger(SegmentedRaftLogCache.class);
    private final String name;
    private volatile LogSegment openSegment;
    private final LogSegmentList closedSegments;
    private final RaftStorage storage;
    private final RaftLogMetrics raftLogMetrics;
    private final int maxCachedSegments;
    private final CacheInvalidationPolicy evictionPolicy = new CacheInvalidationPolicy.CacheInvalidationPolicyDefault();

    SegmentedRaftLogCache(Object name, RaftStorage storage, RaftProperties properties) {
        this(name, storage, properties, null);
    }

    SegmentedRaftLogCache(Object name, RaftStorage storage, RaftProperties properties, RaftLogMetrics raftLogMetrics) {
        this.name = name + "-" + this.getClass().getSimpleName();
        this.closedSegments = new LogSegmentList(name);
        this.storage = storage;
        this.maxCachedSegments = RaftServerConfigKeys.Log.maxCachedSegmentNum(properties);
        this.raftLogMetrics = raftLogMetrics;
    }

    int getMaxCachedSegments() {
        return this.maxCachedSegments;
    }

    void loadSegment(RaftStorageDirectory.LogPathAndIndex pi, boolean keepEntryInCache, Consumer<RaftProtos.LogEntryProto> logConsumer) throws IOException {
        LogSegment logSegment = LogSegment.loadSegment(this.storage, pi.getPath().toFile(), pi.startIndex, pi.endIndex, pi.isOpen(), keepEntryInCache, logConsumer, this.raftLogMetrics);
        if (logSegment != null) {
            this.addSegment(logSegment);
        }
    }

    long getCachedSegmentNum() {
        return this.closedSegments.countCached();
    }

    boolean shouldEvict() {
        return this.closedSegments.countCached() > (long)this.maxCachedSegments;
    }

    void evictCache(long[] followerIndices, long flushedIndex, long lastAppliedIndex) {
        List<LogSegment> toEvict = this.evictionPolicy.evict(followerIndices, flushedIndex, lastAppliedIndex, this.closedSegments, this.maxCachedSegments);
        for (LogSegment s : toEvict) {
            s.evictCache();
        }
    }

    private void validateAdding(LogSegment segment) {
        LogSegment lastClosed = this.closedSegments.getLast();
        if (lastClosed != null) {
            Preconditions.assertTrue((!lastClosed.isOpen() ? 1 : 0) != 0);
            Preconditions.assertTrue((lastClosed.getEndIndex() + 1L == segment.getStartIndex() ? 1 : 0) != 0);
        }
    }

    void addSegment(LogSegment segment) {
        this.validateAdding(segment);
        if (segment.isOpen()) {
            this.setOpenSegment(segment);
        } else {
            this.closedSegments.add(segment);
        }
    }

    void addOpenSegment(long startIndex) {
        this.setOpenSegment(LogSegment.newOpenSegment(this.storage, startIndex, this.raftLogMetrics));
    }

    private void setOpenSegment(LogSegment openSegment) {
        LOG.trace("{}: setOpenSegment to {}", (Object)this.name, (Object)openSegment);
        Preconditions.assertTrue((this.openSegment == null ? 1 : 0) != 0);
        this.openSegment = Objects.requireNonNull(openSegment);
    }

    private void clearOpenSegment() {
        LOG.trace("{}: clearOpenSegment {}", (Object)this.name, (Object)this.openSegment);
        Objects.requireNonNull(this.openSegment);
        this.openSegment = null;
    }

    LogSegment getOpenSegment() {
        return this.openSegment;
    }

    void rollOpenSegment(boolean createNewOpen) {
        Preconditions.assertTrue((this.openSegment != null && this.openSegment.numOfEntries() > 0 ? 1 : 0) != 0);
        long nextIndex = this.openSegment.getEndIndex() + 1L;
        this.openSegment.close();
        this.closedSegments.add(this.openSegment);
        this.clearOpenSegment();
        if (createNewOpen) {
            this.addOpenSegment(nextIndex);
        }
    }

    LogSegment getSegment(long index) {
        if (this.openSegment != null && index >= this.openSegment.getStartIndex()) {
            return this.openSegment;
        }
        return this.closedSegments.search(index);
    }

    LogSegment.LogRecord getLogRecord(long index) {
        LogSegment segment = this.getSegment(index);
        return segment == null ? null : segment.getLogRecord(index);
    }

    TermIndex[] getTermIndices(long startIndex, long endIndex) {
        if (startIndex < 0L || startIndex < this.getStartIndex()) {
            throw new IndexOutOfBoundsException("startIndex = " + startIndex + ", log cache starts from index " + this.getStartIndex());
        }
        if (startIndex > endIndex) {
            throw new IndexOutOfBoundsException("startIndex(" + startIndex + ") > endIndex(" + endIndex + ")");
        }
        long realEnd = Math.min(this.getEndIndex() + 1L, endIndex);
        if (startIndex >= realEnd) {
            return TermIndex.EMPTY_TERMINDEX_ARRAY;
        }
        return this.closedSegments.getTermIndex(startIndex, realEnd, this.openSegment);
    }

    private static void getFromSegment(LogSegment segment, long startIndex, TermIndex[] entries, int offset, int size) {
        long endIndex = segment.getEndIndex();
        endIndex = Math.min(endIndex, startIndex + (long)size - 1L);
        int index = offset;
        for (long i = startIndex; i <= endIndex; ++i) {
            LogSegment.LogRecord r = segment.getLogRecord(i);
            entries[index++] = r == null ? null : r.getTermIndex();
        }
    }

    boolean isConfigEntry(TermIndex ti) {
        LogSegment segment = this.getSegment(ti.getIndex());
        return segment != null && segment.isConfigEntry(ti);
    }

    long getStartIndex() {
        if (this.closedSegments.isEmpty()) {
            return this.openSegment != null ? this.openSegment.getStartIndex() : -1L;
        }
        return this.closedSegments.get(0).getStartIndex();
    }

    long getEndIndex() {
        return this.openSegment != null ? this.openSegment.getEndIndex() : (this.closedSegments.isEmpty() ? -1L : this.closedSegments.get(this.closedSegments.size() - 1).getEndIndex());
    }

    TermIndex getLastTermIndex() {
        return this.openSegment != null && this.openSegment.numOfEntries() > 0 ? this.openSegment.getLastTermIndex() : (this.closedSegments.isEmpty() ? null : this.closedSegments.get(this.closedSegments.size() - 1).getLastTermIndex());
    }

    void appendEntry(RaftProtos.LogEntryProto entry) {
        Preconditions.assertTrue((this.openSegment != null ? 1 : 0) != 0);
        this.openSegment.appendToOpenSegment(entry);
    }

    TruncationSegments truncate(long index) {
        return this.closedSegments.truncate(index, this.openSegment, this::clearOpenSegment);
    }

    TruncationSegments purge(long index) {
        return this.closedSegments.purge(index);
    }

    Iterator<TermIndex> iterator(long startIndex) {
        return new EntryIterator(startIndex);
    }

    TruncateIndices computeTruncateIndices(Consumer<TermIndex> failClientRequest, RaftProtos.LogEntryProto ... entries) {
        int arrayIndex;
        long truncateIndex = -1L;
        try (AutoCloseableLock readLock = this.closedSegments.readLock();){
            Iterator<TermIndex> i = this.iterator(entries[0].getIndex());
            for (arrayIndex = 0; i.hasNext() && arrayIndex < entries.length; ++arrayIndex) {
                TermIndex storedEntry = i.next();
                Preconditions.assertTrue((storedEntry.getIndex() == entries[arrayIndex].getIndex() ? 1 : 0) != 0, (String)"The stored entry's index %s is not consistent with the received entries[%s]'s index %s", (Object[])new Object[]{storedEntry.getIndex(), arrayIndex, entries[arrayIndex].getIndex()});
                if (storedEntry.getTerm() == entries[arrayIndex].getTerm()) continue;
                truncateIndex = storedEntry.getIndex();
                if (LOG.isTraceEnabled()) {
                    LOG.trace("{}: truncate to {}, arrayIndex={}, ti={}, storedEntry={}, entries={}", new Object[]{this.name, truncateIndex, arrayIndex, ServerProtoUtils.toTermIndex(entries[arrayIndex]), storedEntry, ServerProtoUtils.toString(entries)});
                }
                failClientRequest.accept(storedEntry);
                while (i.hasNext()) {
                    failClientRequest.accept(i.next());
                }
                break;
            }
        }
        return new TruncateIndices(arrayIndex, truncateIndex);
    }

    int getNumOfSegments() {
        return this.closedSegments.size() + (this.openSegment == null ? 0 : 1);
    }

    boolean isEmpty() {
        return this.closedSegments.isEmpty() && this.openSegment == null;
    }

    void clear() {
        if (this.openSegment != null) {
            this.openSegment.clear();
            this.clearOpenSegment();
        }
        this.closedSegments.clear();
    }

    private class EntryIterator
    implements Iterator<TermIndex> {
        private long nextIndex;
        private LogSegment currentSegment;
        private int segmentIndex;

        EntryIterator(long start) {
            this.nextIndex = start;
            this.segmentIndex = SegmentedRaftLogCache.this.closedSegments.binarySearch(this.nextIndex);
            if (this.segmentIndex >= 0) {
                this.currentSegment = SegmentedRaftLogCache.this.closedSegments.get(this.segmentIndex);
            } else {
                this.segmentIndex = -this.segmentIndex - 1;
                if (this.segmentIndex == SegmentedRaftLogCache.this.closedSegments.size()) {
                    this.currentSegment = SegmentedRaftLogCache.this.openSegment;
                } else {
                    Preconditions.assertTrue((this.segmentIndex == 0 ? 1 : 0) != 0);
                    throw new IndexOutOfBoundsException();
                }
            }
        }

        @Override
        public boolean hasNext() {
            return this.currentSegment != null && this.currentSegment.getLogRecord(this.nextIndex) != null;
        }

        @Override
        public TermIndex next() {
            LogSegment.LogRecord record;
            if (this.currentSegment == null || (record = this.currentSegment.getLogRecord(this.nextIndex)) == null) {
                throw new NoSuchElementException();
            }
            if (++this.nextIndex > this.currentSegment.getEndIndex() && this.currentSegment != SegmentedRaftLogCache.this.openSegment) {
                ++this.segmentIndex;
                this.currentSegment = this.segmentIndex == SegmentedRaftLogCache.this.closedSegments.size() ? SegmentedRaftLogCache.this.openSegment : SegmentedRaftLogCache.this.closedSegments.get(this.segmentIndex);
            }
            return record.getTermIndex();
        }
    }

    static class TruncateIndices {
        final int arrayIndex;
        final long truncateIndex;

        TruncateIndices(int arrayIndex, long truncateIndex) {
            this.arrayIndex = arrayIndex;
            this.truncateIndex = truncateIndex;
        }

        int getArrayIndex() {
            return this.arrayIndex;
        }

        long getTruncateIndex() {
            return this.truncateIndex;
        }
    }

    static class LogSegmentList {
        private final Object name;
        private final List<LogSegment> segments = new ArrayList<LogSegment>();
        private final AutoCloseableReadWriteLock lock;

        LogSegmentList(Object name) {
            this.name = name;
            this.lock = new AutoCloseableReadWriteLock(name);
        }

        AutoCloseableLock readLock() {
            StackTraceElement caller = LOG.isTraceEnabled() ? JavaUtils.getCallerStackTraceElement() : null;
            return this.lock.readLock(caller, arg_0 -> ((Logger)LOG).trace(arg_0));
        }

        AutoCloseableLock writeLock() {
            StackTraceElement caller = LOG.isTraceEnabled() ? JavaUtils.getCallerStackTraceElement() : null;
            return this.lock.writeLock(caller, arg_0 -> ((Logger)LOG).trace(arg_0));
        }

        boolean isEmpty() {
            try (AutoCloseableLock readLock = this.readLock();){
                boolean bl = this.segments.isEmpty();
                return bl;
            }
        }

        int size() {
            try (AutoCloseableLock readLock = this.readLock();){
                int n = this.segments.size();
                return n;
            }
        }

        long countCached() {
            try (AutoCloseableLock readLock = this.readLock();){
                long l = this.segments.stream().filter(LogSegment::hasCache).count();
                return l;
            }
        }

        LogSegment getLast() {
            try (AutoCloseableLock readLock = this.readLock();){
                LogSegment logSegment = this.segments.isEmpty() ? null : this.segments.get(this.segments.size() - 1);
                return logSegment;
            }
        }

        LogSegment get(int i) {
            try (AutoCloseableLock readLock = this.readLock();){
                LogSegment logSegment = this.segments.get(i);
                return logSegment;
            }
        }

        int binarySearch(long index) {
            try (AutoCloseableLock readLock = this.readLock();){
                int n = Collections.binarySearch(this.segments, index);
                return n;
            }
        }

        LogSegment search(long index) {
            try (AutoCloseableLock readLock = this.readLock();){
                int i = Collections.binarySearch(this.segments, index);
                LogSegment logSegment = i < 0 ? null : this.segments.get(i);
                return logSegment;
            }
        }

        TermIndex[] getTermIndex(long startIndex, long realEnd, LogSegment openSegment) {
            int searchIndex;
            TermIndex[] entries = new TermIndex[Math.toIntExact(realEnd - startIndex)];
            long index = startIndex;
            try (AutoCloseableLock readLock = this.readLock();){
                searchIndex = Collections.binarySearch(this.segments, startIndex);
                if (searchIndex >= 0) {
                    int numberFromSegment;
                    for (int i = searchIndex; i < this.segments.size() && index < realEnd; index += (long)numberFromSegment, ++i) {
                        LogSegment s = this.segments.get(i);
                        numberFromSegment = Math.toIntExact(Math.min(realEnd - index, s.getEndIndex() - index + 1L));
                        SegmentedRaftLogCache.getFromSegment(s, index, entries, Math.toIntExact(index - startIndex), numberFromSegment);
                    }
                }
            }
            if (searchIndex < 0) {
                SegmentedRaftLogCache.getFromSegment(openSegment, startIndex, entries, 0, entries.length);
            } else if (index < realEnd) {
                SegmentedRaftLogCache.getFromSegment(openSegment, index, entries, Math.toIntExact(index - startIndex), Math.toIntExact(realEnd - index));
            }
            return entries;
        }

        boolean add(LogSegment logSegment) {
            try (AutoCloseableLock writeLock = this.writeLock();){
                boolean bl = this.segments.add(logSegment);
                return bl;
            }
        }

        void clear() {
            try (AutoCloseableLock writeLock = this.writeLock();){
                this.segments.forEach(LogSegment::clear);
                this.segments.clear();
            }
        }

        TruncationSegments truncate(long index, LogSegment openSegment, Runnable clearOpenSegment) {
            try (AutoCloseableLock writeLock = this.writeLock();){
                int segmentIndex = this.binarySearch(index);
                if (segmentIndex == -this.segments.size() - 1) {
                    if (openSegment != null && openSegment.getEndIndex() >= index) {
                        long oldEnd = openSegment.getEndIndex();
                        if (index == openSegment.getStartIndex()) {
                            SegmentFileInfo deleted = LogSegmentList.deleteOpenSegment(openSegment, clearOpenSegment);
                            TruncationSegments truncationSegments = new TruncationSegments(null, Collections.singletonList(deleted));
                            return truncationSegments;
                        }
                        openSegment.truncate(index);
                        Preconditions.assertTrue((!openSegment.isOpen() ? 1 : 0) != 0);
                        SegmentFileInfo info = new SegmentFileInfo(openSegment.getStartIndex(), oldEnd, true, openSegment.getTotalSize(), openSegment.getEndIndex());
                        this.segments.add(openSegment);
                        clearOpenSegment.run();
                        TruncationSegments truncationSegments = new TruncationSegments(info, Collections.emptyList());
                        return truncationSegments;
                    }
                } else if (segmentIndex >= 0) {
                    LogSegment ts = this.segments.get(segmentIndex);
                    long oldEnd = ts.getEndIndex();
                    ArrayList<SegmentFileInfo> list = new ArrayList<SegmentFileInfo>();
                    ts.truncate(index);
                    int size = this.segments.size();
                    for (int i = size - 1; i >= (ts.numOfEntries() == 0 ? segmentIndex : segmentIndex + 1); --i) {
                        LogSegment s = this.segments.remove(i);
                        long endOfS = i == segmentIndex ? oldEnd : s.getEndIndex();
                        s.clear();
                        list.add(new SegmentFileInfo(s.getStartIndex(), endOfS, false, 0L, s.getEndIndex()));
                    }
                    if (openSegment != null) {
                        list.add(LogSegmentList.deleteOpenSegment(openSegment, clearOpenSegment));
                    }
                    SegmentFileInfo t = ts.numOfEntries() == 0 ? null : new SegmentFileInfo(ts.getStartIndex(), oldEnd, false, ts.getTotalSize(), ts.getEndIndex());
                    TruncationSegments truncationSegments = new TruncationSegments(t, list);
                    return truncationSegments;
                }
                TruncationSegments truncationSegments = null;
                return truncationSegments;
            }
        }

        TruncationSegments purge(long index) {
            try (AutoCloseableLock writeLock = this.writeLock();){
                int segmentIndex = this.binarySearch(index);
                ArrayList<SegmentFileInfo> list = new ArrayList<SegmentFileInfo>();
                if (segmentIndex == -this.segments.size() - 1) {
                    for (LogSegment ls : this.segments) {
                        list.add(SegmentFileInfo.newClosedSegmentFileInfo(ls));
                    }
                    this.segments.clear();
                } else if (segmentIndex >= 0) {
                    for (int i = segmentIndex - 1; i >= 0; --i) {
                        list.add(SegmentFileInfo.newClosedSegmentFileInfo(this.segments.remove(i)));
                    }
                } else {
                    throw new IllegalStateException("Unexpected gap in segments: binarySearch(" + index + ") returns " + segmentIndex + ", segments=" + this.segments);
                }
                TruncationSegments truncationSegments = list.isEmpty() ? null : new TruncationSegments(null, list);
                return truncationSegments;
            }
        }

        static SegmentFileInfo deleteOpenSegment(LogSegment openSegment, Runnable clearOpenSegment) {
            long oldEnd = openSegment.getEndIndex();
            openSegment.clear();
            SegmentFileInfo info = new SegmentFileInfo(openSegment.getStartIndex(), oldEnd, true, 0L, openSegment.getEndIndex());
            clearOpenSegment.run();
            return info;
        }
    }

    static class TruncationSegments {
        final SegmentFileInfo toTruncate;
        final SegmentFileInfo[] toDelete;

        TruncationSegments(SegmentFileInfo toTruncate, List<SegmentFileInfo> toDelete) {
            this.toDelete = toDelete == null ? null : toDelete.toArray(new SegmentFileInfo[toDelete.size()]);
            this.toTruncate = toTruncate;
        }

        long maxEndIndex() {
            long max = Long.MIN_VALUE;
            if (this.toTruncate != null) {
                max = this.toTruncate.endIndex;
            }
            for (SegmentFileInfo d : this.toDelete) {
                max = Math.max(max, d.endIndex);
            }
            return max;
        }

        public String toString() {
            return "toTruncate: " + this.toTruncate + "\n  toDelete: " + Arrays.toString(this.toDelete);
        }
    }

    static final class SegmentFileInfo {
        final long startIndex;
        final long endIndex;
        final boolean isOpen;
        final long targetLength;
        final long newEndIndex;

        static SegmentFileInfo newClosedSegmentFileInfo(LogSegment ls) {
            Objects.requireNonNull(ls, "ls == null");
            Preconditions.assertTrue((!ls.isOpen() ? 1 : 0) != 0, (Object)(ls + " is OPEN"));
            return new SegmentFileInfo(ls.getStartIndex(), ls.getEndIndex(), ls.isOpen(), 0L, 0L);
        }

        private SegmentFileInfo(long start, long end, boolean isOpen, long targetLength, long newEndIndex) {
            this.startIndex = start;
            this.endIndex = end;
            this.isOpen = isOpen;
            this.targetLength = targetLength;
            this.newEndIndex = newEndIndex;
        }

        public String toString() {
            return "(" + this.startIndex + ", " + this.endIndex + ") isOpen? " + this.isOpen + ", length=" + this.targetLength + ", newEndIndex=" + this.newEndIndex;
        }
    }
}

