/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.raftlog.segmented;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.raftlog.RaftLogIOException;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogFormat;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogInputStream;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.ratis.thirdparty.com.google.common.cache.CacheLoader;
import org.apache.ratis.thirdparty.com.google.protobuf.CodedOutputStream;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class LogSegment
implements Comparable<Long> {
    static final Logger LOG = LoggerFactory.getLogger(LogSegment.class);
    private volatile boolean isOpen;
    private long totalSize;
    private final long startIndex;
    private volatile long endIndex;
    private final RaftStorage storage;
    private final CacheLoader<LogRecord, RaftProtos.LogEntryProto> cacheLoader = new LogEntryLoader();
    private final AtomicInteger loadingTimes = new AtomicInteger();
    private final List<LogRecord> records = new ArrayList<LogRecord>();
    private final Map<TermIndex, RaftProtos.LogEntryProto> entryCache = new ConcurrentHashMap<TermIndex, RaftProtos.LogEntryProto>();
    private final Set<TermIndex> configEntries = new HashSet<TermIndex>();

    static long getEntrySize(RaftProtos.LogEntryProto entry) {
        int serialized = ServerProtoUtils.removeStateMachineData(entry).getSerializedSize();
        return serialized + CodedOutputStream.computeUInt32SizeNoTag((int)serialized) + 4;
    }

    static LogSegment newOpenSegment(RaftStorage storage, long start) {
        Preconditions.assertTrue((start >= 0L ? 1 : 0) != 0);
        return new LogSegment(storage, true, start, start - 1L);
    }

    @VisibleForTesting
    static LogSegment newCloseSegment(RaftStorage storage, long start, long end) {
        Preconditions.assertTrue((start >= 0L && end >= start ? 1 : 0) != 0);
        return new LogSegment(storage, false, start, end);
    }

    private static int readSegmentFile(File file, long start, long end, boolean isOpen, Consumer<RaftProtos.LogEntryProto> entryConsumer) throws IOException {
        int count = 0;
        try (SegmentedRaftLogInputStream in = new SegmentedRaftLogInputStream(file, start, end, isOpen);){
            RaftProtos.LogEntryProto next;
            RaftProtos.LogEntryProto prev = null;
            while ((next = in.nextEntry()) != null) {
                if (prev != null) {
                    Preconditions.assertTrue((next.getIndex() == prev.getIndex() + 1L ? 1 : 0) != 0, (String)"gap between entry %s and entry %s", (Object[])new Object[]{prev, next});
                }
                if (entryConsumer != null) {
                    entryConsumer.accept(next);
                }
                ++count;
                prev = next;
            }
        }
        return count;
    }

    static LogSegment loadSegment(RaftStorage storage, File file, long start, long end, boolean isOpen, boolean keepEntryInCache, Consumer<RaftProtos.LogEntryProto> logConsumer) throws IOException {
        LogSegment segment = isOpen ? LogSegment.newOpenSegment(storage, start) : LogSegment.newCloseSegment(storage, start, end);
        int entryCount = LogSegment.readSegmentFile(file, start, end, isOpen, entry -> {
            segment.append(keepEntryInCache || isOpen, (RaftProtos.LogEntryProto)entry);
            if (logConsumer != null) {
                logConsumer.accept((RaftProtos.LogEntryProto)entry);
            }
        });
        LOG.info("Successfully read {} entries from segment file {}", (Object)entryCount, (Object)file);
        if (entryCount == 0) {
            FileUtils.deleteFile((File)file);
            return null;
        }
        if (file.length() > segment.getTotalSize()) {
            FileUtils.truncateFile((File)file, (long)segment.getTotalSize());
        }
        Preconditions.assertTrue((start == segment.getStartIndex() ? 1 : 0) != 0);
        if (!segment.records.isEmpty()) {
            Preconditions.assertTrue((start == segment.records.get(0).getTermIndex().getIndex() ? 1 : 0) != 0);
        }
        if (!isOpen) {
            Preconditions.assertTrue((segment.getEndIndex() == end ? 1 : 0) != 0);
        }
        return segment;
    }

    private File getSegmentFile() {
        return this.isOpen ? this.storage.getStorageDir().getOpenLogFile(this.startIndex) : this.storage.getStorageDir().getClosedLogFile(this.startIndex, this.endIndex);
    }

    private LogSegment(RaftStorage storage, boolean isOpen, long start, long end) {
        this.storage = storage;
        this.isOpen = isOpen;
        this.startIndex = start;
        this.endIndex = end;
        this.totalSize = SegmentedRaftLogFormat.getHeaderLength();
    }

    long getStartIndex() {
        return this.startIndex;
    }

    long getEndIndex() {
        return this.endIndex;
    }

    boolean isOpen() {
        return this.isOpen;
    }

    int numOfEntries() {
        return Math.toIntExact(this.endIndex - this.startIndex + 1L);
    }

    void appendToOpenSegment(RaftProtos.LogEntryProto entry) {
        Preconditions.assertTrue((boolean)this.isOpen(), (String)"The log segment %s is not open for append", (Object[])new Object[]{this});
        this.append(true, entry);
    }

    private void append(boolean keepEntryInCache, RaftProtos.LogEntryProto entry) {
        LogRecord currentLast;
        Objects.requireNonNull(entry, "entry == null");
        if (this.records.isEmpty()) {
            Preconditions.assertTrue((entry.getIndex() == this.startIndex ? 1 : 0) != 0, (String)"gap between start index %s and first entry to append %s", (Object[])new Object[]{this.startIndex, entry.getIndex()});
        }
        if ((currentLast = this.getLastRecord()) != null) {
            Preconditions.assertTrue((entry.getIndex() == currentLast.getTermIndex().getIndex() + 1L ? 1 : 0) != 0, (String)"gap between entries %s and %s", (Object[])new Object[]{entry.getIndex(), currentLast.getTermIndex().getIndex()});
        }
        LogRecord record = new LogRecord(this.totalSize, entry);
        this.records.add(record);
        if (keepEntryInCache) {
            this.entryCache.put(record.getTermIndex(), entry);
        }
        if (entry.hasConfigurationEntry()) {
            this.configEntries.add(record.getTermIndex());
        }
        this.totalSize += LogSegment.getEntrySize(entry);
        this.endIndex = entry.getIndex();
    }

    RaftProtos.LogEntryProto getEntryFromCache(TermIndex ti) {
        return this.entryCache.get(ti);
    }

    synchronized RaftProtos.LogEntryProto loadCache(LogRecord record) throws RaftLogIOException {
        RaftProtos.LogEntryProto entry = this.entryCache.get(record.getTermIndex());
        if (entry != null) {
            return entry;
        }
        try {
            return (RaftProtos.LogEntryProto)this.cacheLoader.load((Object)record);
        }
        catch (Exception e) {
            throw new RaftLogIOException(e);
        }
    }

    LogRecord getLogRecord(long index) {
        if (index >= this.startIndex && index <= this.endIndex) {
            return this.records.get(Math.toIntExact(index - this.startIndex));
        }
        return null;
    }

    private LogRecord getLastRecord() {
        return this.records.isEmpty() ? null : this.records.get(this.records.size() - 1);
    }

    TermIndex getLastTermIndex() {
        LogRecord last = this.getLastRecord();
        return last == null ? null : last.getTermIndex();
    }

    boolean isConfigEntry(TermIndex ti) {
        return this.configEntries.contains(ti);
    }

    long getTotalSize() {
        return this.totalSize;
    }

    void truncate(long fromIndex) {
        Preconditions.assertTrue((fromIndex >= this.startIndex && fromIndex <= this.endIndex ? 1 : 0) != 0);
        for (long index = this.endIndex; index >= fromIndex; --index) {
            LogRecord removed = this.records.remove(Math.toIntExact(index - this.startIndex));
            this.entryCache.remove(removed.getTermIndex());
            this.configEntries.remove(removed.getTermIndex());
            this.totalSize = removed.offset;
        }
        this.isOpen = false;
        this.endIndex = fromIndex - 1L;
    }

    void close() {
        Preconditions.assertTrue((boolean)this.isOpen());
        this.isOpen = false;
    }

    public String toString() {
        return this.isOpen() ? "log_inprogress_" + this.startIndex : "log-" + this.startIndex + "_" + this.endIndex;
    }

    @Override
    public int compareTo(Long l) {
        return l >= this.getStartIndex() && l <= this.getEndIndex() ? 0 : (this.getEndIndex() < l ? -1 : 1);
    }

    void clear() {
        this.records.clear();
        this.entryCache.clear();
        this.configEntries.clear();
        this.endIndex = this.startIndex - 1L;
    }

    public int getLoadingTimes() {
        return this.loadingTimes.get();
    }

    void evictCache() {
        this.entryCache.clear();
    }

    boolean hasCache() {
        return this.isOpen || !this.entryCache.isEmpty();
    }

    boolean containsIndex(long index) {
        return this.startIndex <= index && this.endIndex >= index;
    }

    class LogEntryLoader
    extends CacheLoader<LogRecord, RaftProtos.LogEntryProto> {
        LogEntryLoader() {
        }

        public RaftProtos.LogEntryProto load(LogRecord key) throws IOException {
            File file = LogSegment.this.getSegmentFile();
            LogSegment.readSegmentFile(file, LogSegment.this.startIndex, LogSegment.this.endIndex, LogSegment.this.isOpen, entry -> LogSegment.this.entryCache.put(ServerProtoUtils.toTermIndex(entry), entry));
            LogSegment.this.loadingTimes.incrementAndGet();
            return (RaftProtos.LogEntryProto)Objects.requireNonNull(LogSegment.this.entryCache.get(key.getTermIndex()));
        }
    }

    static class LogRecord {
        private final long offset;
        private final TermIndex termIndex;

        LogRecord(long offset, RaftProtos.LogEntryProto entry) {
            this.offset = offset;
            this.termIndex = ServerProtoUtils.toTermIndex(entry);
        }

        TermIndex getTermIndex() {
            return this.termIndex;
        }

        long getOffset() {
            return this.offset;
        }
    }
}

