/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.storage;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.LogSegment;
import org.apache.ratis.server.storage.RaftLog;
import org.apache.ratis.server.storage.RaftLogCache;
import org.apache.ratis.server.storage.RaftLogIOException;
import org.apache.ratis.server.storage.RaftLogWorker;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.RaftStorageDirectory;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.util.AutoCloseableLock;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Preconditions;

public class SegmentedRaftLog
extends RaftLog {
    private final Optional<RaftServerImpl> server;
    private final RaftStorage storage;
    private final RaftLogCache cache;
    private final RaftLogWorker fileLogWorker;
    private final long segmentMaxSize;
    private final boolean stateMachineCachingEnabled;

    public SegmentedRaftLog(RaftPeerId selfId, RaftServerImpl server, RaftStorage storage, long lastIndexInSnapshot, RaftProperties properties) {
        this(selfId, server, server != null ? server.getStateMachine() : null, server != null ? server::submitUpdateCommitEvent : null, storage, lastIndexInSnapshot, properties);
    }

    SegmentedRaftLog(RaftPeerId selfId, RaftServerImpl server, StateMachine stateMachine, Runnable submitUpdateCommitEvent, RaftStorage storage, long lastIndexInSnapshot, RaftProperties properties) {
        super(selfId, lastIndexInSnapshot, RaftServerConfigKeys.Log.Appender.bufferByteLimit(properties).getSizeInt());
        this.server = Optional.ofNullable(server);
        this.storage = storage;
        this.segmentMaxSize = RaftServerConfigKeys.Log.segmentSizeMax(properties).getSize();
        this.cache = new RaftLogCache(selfId, storage, properties);
        this.fileLogWorker = new RaftLogWorker(selfId, stateMachine, submitUpdateCommitEvent, storage, properties);
        this.stateMachineCachingEnabled = RaftServerConfigKeys.Log.StateMachineData.cachingEnabled(properties);
    }

    @Override
    protected void openImpl(long lastIndexInSnapshot, Consumer<RaftProtos.LogEntryProto> consumer) throws IOException {
        this.loadLogSegments(lastIndexInSnapshot, consumer);
        File openSegmentFile = null;
        LogSegment openSegment = this.cache.getOpenSegment();
        if (openSegment != null) {
            openSegmentFile = this.storage.getStorageDir().getOpenLogFile(openSegment.getStartIndex());
        }
        this.fileLogWorker.start(Math.max(this.cache.getEndIndex(), lastIndexInSnapshot), openSegmentFile);
    }

    @Override
    public long getStartIndex() {
        return this.cache.getStartIndex();
    }

    private void loadLogSegments(long lastIndexInSnapshot, Consumer<RaftProtos.LogEntryProto> logConsumer) throws IOException {
        try (AutoCloseableLock writeLock = this.writeLock();){
            List<RaftStorageDirectory.LogPathAndIndex> paths = this.storage.getStorageDir().getLogSegmentFiles();
            int i = 0;
            for (RaftStorageDirectory.LogPathAndIndex pi : paths) {
                boolean keepEntryInCache = paths.size() - i++ <= this.cache.getMaxCachedSegments();
                this.cache.loadSegment(pi, keepEntryInCache, logConsumer);
            }
            if (!this.cache.isEmpty() && this.cache.getEndIndex() < lastIndexInSnapshot) {
                LOG.warn("End log index {} is smaller than last index in snapshot {}", (Object)this.cache.getEndIndex(), (Object)lastIndexInSnapshot);
                this.cache.clear();
            }
        }
    }

    @Override
    public RaftProtos.LogEntryProto get(long index) throws RaftLogIOException {
        LogSegment.LogRecordWithEntry recordAndEntry;
        LogSegment segment;
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            segment = this.cache.getSegment(index);
            if (segment == null) {
                RaftProtos.LogEntryProto logEntryProto = null;
                return logEntryProto;
            }
            recordAndEntry = segment.getEntryWithoutLoading(index);
            if (recordAndEntry == null) {
                RaftProtos.LogEntryProto logEntryProto = null;
                return logEntryProto;
            }
            if (recordAndEntry.hasEntry()) {
                RaftProtos.LogEntryProto logEntryProto = recordAndEntry.getEntry();
                return logEntryProto;
            }
        }
        this.checkAndEvictCache();
        return segment.loadCache(recordAndEntry.getRecord());
    }

    @Override
    public RaftLog.EntryWithData getEntryWithData(long index) throws RaftLogIOException {
        RaftProtos.LogEntryProto entry = this.get(index);
        if (!ServerProtoUtils.shouldReadStateMachineData(entry)) {
            return new RaftLog.EntryWithData(entry, null);
        }
        try {
            return new RaftLog.EntryWithData(entry, this.server.map(s -> s.getStateMachine().readStateMachineData(entry)).orElse(null));
        }
        catch (Throwable e) {
            String err = this.getSelfId() + ": Failed readStateMachineData for " + ServerProtoUtils.toLogEntryString(entry);
            LOG.error(err, e);
            throw new RaftLogIOException(err, JavaUtils.unwrapCompletionException((Throwable)e));
        }
    }

    private void checkAndEvictCache() {
        if (this.server.isPresent() && this.cache.shouldEvict()) {
            RaftServerImpl s = this.server.get();
            this.cache.evictCache(s.getFollowerNextIndices(), this.fileLogWorker.getFlushedIndex(), s.getState().getLastAppliedIndex());
        }
    }

    @Override
    public TermIndex getTermIndex(long index) {
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            LogSegment.LogRecord record = this.cache.getLogRecord(index);
            TermIndex termIndex = record != null ? record.getTermIndex() : null;
            return termIndex;
        }
    }

    @Override
    public TermIndex[] getEntries(long startIndex, long endIndex) {
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            TermIndex[] termIndexArray = this.cache.getTermIndices(startIndex, endIndex);
            return termIndexArray;
        }
    }

    @Override
    public TermIndex getLastEntryTermIndex() {
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            TermIndex termIndex = this.cache.getLastTermIndex();
            return termIndex;
        }
    }

    @Override
    CompletableFuture<Long> truncateImpl(long index) {
        this.checkLogState();
        try (AutoCloseableLock writeLock = this.writeLock();){
            RaftLogCache.TruncationSegments ts = this.cache.truncate(index);
            if (ts != null) {
                Task task = this.fileLogWorker.truncate(ts, index);
                CompletableFuture<Long> completableFuture = task.getFuture();
                return completableFuture;
            }
        }
        return CompletableFuture.completedFuture(index);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    CompletableFuture<Long> appendEntryImpl(RaftProtos.LogEntryProto entry) {
        this.checkLogState();
        if (LOG.isTraceEnabled()) {
            LOG.trace("{}: appendEntry {}", (Object)this.getSelfId(), (Object)ServerProtoUtils.toLogEntryString(entry));
        }
        try (AutoCloseableLock writeLock = this.writeLock();){
            this.validateLogEntry(entry);
            LogSegment currentOpenSegment = this.cache.getOpenSegment();
            if (currentOpenSegment == null) {
                this.cache.addOpenSegment(entry.getIndex());
                this.fileLogWorker.startLogSegment(entry.getIndex());
            } else if (this.isSegmentFull(currentOpenSegment, entry)) {
                this.cache.rollOpenSegment(true);
                this.fileLogWorker.rollLogSegment(currentOpenSegment);
                this.checkAndEvictCache();
            } else if (currentOpenSegment.numOfEntries() > 0 && currentOpenSegment.getLastTermIndex().getTerm() != entry.getTerm()) {
                long currentTerm = currentOpenSegment.getLastTermIndex().getTerm();
                Preconditions.assertTrue((currentTerm < entry.getTerm() ? 1 : 0) != 0, (String)"open segment's term %s is larger than the new entry's term %s", (Object[])new Object[]{currentTerm, entry.getTerm()});
                this.cache.rollOpenSegment(true);
                this.fileLogWorker.rollLogSegment(currentOpenSegment);
                this.checkAndEvictCache();
            }
            CompletableFuture<Long> writeFuture = this.fileLogWorker.writeLogEntry(entry).getFuture();
            if (this.stateMachineCachingEnabled) {
                this.cache.appendEntry(ServerProtoUtils.removeStateMachineData(entry));
            } else {
                this.cache.appendEntry(entry);
            }
            CompletableFuture<Long> completableFuture = writeFuture;
            return completableFuture;
        }
        catch (Throwable throwable) {
            LOG.error(this.getSelfId() + ": Failed to append " + ServerProtoUtils.toLogEntryString(entry), throwable);
            throw throwable;
        }
    }

    private boolean isSegmentFull(LogSegment segment, RaftProtos.LogEntryProto entry) {
        if (segment.getTotalSize() >= this.segmentMaxSize) {
            return true;
        }
        long entrySize = LogSegment.getEntrySize(entry);
        return entrySize <= this.segmentMaxSize && segment.getTotalSize() + entrySize > this.segmentMaxSize;
    }

    private void failClientRequest(TermIndex ti) {
        if (!this.server.isPresent()) {
            return;
        }
        try {
            RaftProtos.LogEntryProto entry = this.get(ti.getIndex());
            this.server.get().failClientRequest(entry);
        }
        catch (RaftLogIOException e) {
            LOG.error(this.getName() + ": Failed to read log " + ti, (Throwable)((Object)e));
        }
    }

    @Override
    public List<CompletableFuture<Long>> appendImpl(RaftProtos.LogEntryProto ... entries) {
        this.checkLogState();
        if (entries == null || entries.length == 0) {
            return Collections.emptyList();
        }
        try (AutoCloseableLock writeLock = this.writeLock();){
            ArrayList<CompletableFuture<Long>> futures;
            RaftLogCache.TruncateIndices ti = this.cache.computeTruncateIndices(this::failClientRequest, entries);
            long truncateIndex = ti.getTruncateIndex();
            int index = ti.getArrayIndex();
            LOG.debug("truncateIndex={}, arrayIndex={}", (Object)truncateIndex, (Object)index);
            if (truncateIndex != -1L) {
                futures = new ArrayList(entries.length - index + 1);
                futures.add(this.truncate(truncateIndex));
            } else {
                futures = new ArrayList<CompletableFuture<Long>>(entries.length - index);
            }
            for (int i = index; i < entries.length; ++i) {
                futures.add(this.appendEntry(entries[i]));
            }
            ArrayList<CompletableFuture<Long>> arrayList = futures;
            return arrayList;
        }
    }

    @Override
    public long getLatestFlushedIndex() {
        return this.fileLogWorker.getFlushedIndex();
    }

    @Override
    public void writeMetadata(long term, RaftPeerId votedFor) throws IOException {
        this.storage.getMetaFile().set(term, votedFor != null ? votedFor.toString() : null);
    }

    @Override
    public RaftLog.Metadata loadMetadata() throws IOException {
        return new RaftLog.Metadata(RaftPeerId.getRaftPeerId((String)this.storage.getMetaFile().getVotedFor()), this.storage.getMetaFile().getTerm());
    }

    @Override
    public void syncWithSnapshot(long lastSnapshotIndex) {
        this.fileLogWorker.syncWithSnapshot(lastSnapshotIndex);
    }

    @Override
    public boolean isConfigEntry(TermIndex ti) {
        return this.cache.isConfigEntry(ti);
    }

    @Override
    public void close() throws IOException {
        try (AutoCloseableLock writeLock = this.writeLock();){
            super.close();
            this.cache.clear();
        }
        this.fileLogWorker.close();
        this.storage.close();
    }

    RaftLogCache getRaftLogCache() {
        return this.cache;
    }

    @Override
    public String toString() {
        try (AutoCloseableLock readLock = this.readLock();){
            if (this.isOpened()) {
                String string = super.toString() + ",f" + this.getLatestFlushedIndex() + ",i" + Optional.ofNullable(this.getLastEntryTermIndex()).map(TermIndex::getIndex).orElse(0L);
                return string;
            }
            String string = super.toString();
            return string;
        }
    }

    static abstract class Task {
        private final CompletableFuture<Long> future = new CompletableFuture();

        Task() {
        }

        CompletableFuture<Long> getFuture() {
            return this.future;
        }

        void done() {
            this.future.complete(this.getEndIndex());
        }

        abstract void execute() throws IOException;

        abstract long getEndIndex();

        int getSerializedSize() {
            return 0;
        }

        public String toString() {
            return this.getClass().getSimpleName() + ":" + this.getEndIndex();
        }
    }
}

