/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.storage;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.metrics.RatisMetricsRegistry;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.TimeoutIOException;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.storage.LogOutputStream;
import org.apache.ratis.server.storage.LogSegment;
import org.apache.ratis.server.storage.RaftLogCache;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.SegmentedRaftLog;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.util.DataBlockingQueue;
import org.apache.ratis.util.ExitUtils;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.IOUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.SizeInBytes;
import org.apache.ratis.util.TimeDuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RaftLogWorker
implements Runnable {
    static final Logger LOG = LoggerFactory.getLogger(RaftLogWorker.class);
    static final TimeDuration ONE_SECOND = TimeDuration.valueOf((long)1L, (TimeUnit)TimeUnit.SECONDS);
    private final String name;
    private final DataBlockingQueue<SegmentedRaftLog.Task> queue;
    private volatile boolean running = true;
    private final Thread workerThread;
    private final RaftStorage storage;
    private volatile LogOutputStream out;
    private final Runnable submitUpdateCommitEvent;
    private final StateMachine stateMachine;
    private final Supplier<Timer> logFlushTimer;
    private int pendingFlushNum = 0;
    private long lastWrittenIndex;
    private volatile long flushedIndex;
    private final int forceSyncNum;
    private final long segmentMaxSize;
    private final long preallocatedSize;
    private final int bufferSize;
    private final StateMachineDataPolicy stateMachineDataPolicy;

    RaftLogWorker(RaftPeerId selfId, StateMachine stateMachine, Runnable submitUpdateCommitEvent, RaftStorage storage, RaftProperties properties) {
        this.name = selfId + "-" + this.getClass().getSimpleName();
        LOG.info("new {} for {}", (Object)this.name, (Object)storage);
        this.submitUpdateCommitEvent = submitUpdateCommitEvent;
        this.stateMachine = stateMachine;
        this.storage = storage;
        SizeInBytes queueByteLimit = RaftServerConfigKeys.Log.queueByteLimit(properties);
        int queueElementLimit = RaftServerConfigKeys.Log.queueElementLimit(properties);
        this.queue = new DataBlockingQueue((Object)this.name, queueByteLimit, queueElementLimit, SegmentedRaftLog.Task::getSerializedSize);
        this.segmentMaxSize = RaftServerConfigKeys.Log.segmentSizeMax(properties).getSize();
        this.preallocatedSize = RaftServerConfigKeys.Log.preallocatedSize(properties).getSize();
        this.bufferSize = RaftServerConfigKeys.Log.writeBufferSize(properties).getSizeInt();
        this.forceSyncNum = RaftServerConfigKeys.Log.forceSyncNum(properties);
        this.stateMachineDataPolicy = new StateMachineDataPolicy(properties);
        this.workerThread = new Thread((Runnable)this, this.name);
        this.logFlushTimer = JavaUtils.memoize(() -> RatisMetricsRegistry.getRegistry().timer(MetricRegistry.name(RaftLogWorker.class, (String[])new String[]{selfId.toString(), "flush-time"})));
    }

    void start(long latestIndex, File openSegmentFile) throws IOException {
        LOG.trace("{} start(latestIndex={}, openSegmentFile={})", new Object[]{this.name, latestIndex, openSegmentFile});
        this.lastWrittenIndex = latestIndex;
        this.flushedIndex = latestIndex;
        if (openSegmentFile != null) {
            Preconditions.assertTrue((boolean)openSegmentFile.exists());
            this.out = new LogOutputStream(openSegmentFile, true, this.segmentMaxSize, this.preallocatedSize, this.bufferSize);
        }
        this.workerThread.start();
    }

    void close() {
        this.running = false;
        this.workerThread.interrupt();
        try {
            this.workerThread.join(3000L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        IOUtils.cleanup((Logger)LOG, (Closeable[])new Closeable[]{this.out});
        LOG.info("{} close()", (Object)this.name);
    }

    void syncWithSnapshot(long lastSnapshotIndex) {
        this.queue.clear();
        this.lastWrittenIndex = lastSnapshotIndex;
        this.flushedIndex = lastSnapshotIndex;
        this.pendingFlushNum = 0;
    }

    public String toString() {
        return this.name;
    }

    private SegmentedRaftLog.Task addIOTask(SegmentedRaftLog.Task task) {
        LOG.debug("{} adds IO task {}", (Object)this.name, (Object)task);
        try {
            while (!this.queue.offer((Object)task, ONE_SECOND)) {
                Preconditions.assertTrue((boolean)this.isAlive(), (Object)"the worker thread is not alive");
            }
        }
        catch (Throwable t) {
            if (t instanceof InterruptedException && !this.running) {
                LOG.info("Got InterruptedException when adding task " + task + ". The RaftLogWorker already stopped.");
            }
            ExitUtils.terminate((int)2, (String)("Failed to add IO task " + task), (Throwable)t, (Logger)LOG);
        }
        return task;
    }

    boolean isAlive() {
        return this.running && this.workerThread.isAlive();
    }

    @Override
    public void run() {
        while (this.running) {
            try {
                SegmentedRaftLog.Task task = (SegmentedRaftLog.Task)this.queue.poll(ONE_SECOND);
                if (task == null) continue;
                try {
                    task.execute();
                }
                catch (IOException e) {
                    if (task.getEndIndex() < this.lastWrittenIndex) {
                        LOG.info("Ignore IOException when handling task " + task + " which is smaller than the lastWrittenIndex. There should be a snapshot installed.", (Throwable)e);
                    }
                    throw e;
                }
                task.done();
            }
            catch (InterruptedException e) {
                if (this.running) {
                    LOG.warn("{} got interrupted while still running", (Object)Thread.currentThread().getName());
                }
                LOG.info(Thread.currentThread().getName() + " was interrupted, exiting. There are " + this.queue.getNumElements() + " tasks remaining in the queue.");
                Thread.currentThread().interrupt();
                return;
            }
            catch (Throwable t) {
                if (!this.running) {
                    LOG.info("{} got closed and hit exception", (Object)Thread.currentThread().getName(), (Object)t);
                    continue;
                }
                ExitUtils.terminate((int)1, (String)(Thread.currentThread().getName() + " failed."), (Throwable)t, (Logger)LOG);
            }
        }
    }

    private boolean shouldFlush() {
        return this.pendingFlushNum >= this.forceSyncNum || this.pendingFlushNum > 0 && this.queue.isEmpty();
    }

    private void flushWrites() throws IOException {
        if (this.out != null) {
            LOG.debug("{}: flush {}", (Object)this.name, (Object)this.out);
            Timer.Context timerContext = this.logFlushTimer.get().time();
            try {
                CompletableFuture<Object> f;
                CompletableFuture<Object> completableFuture = f = this.stateMachine != null ? this.stateMachine.flushStateMachineData(this.lastWrittenIndex) : CompletableFuture.completedFuture(null);
                if (this.stateMachineDataPolicy.isSync()) {
                    this.stateMachineDataPolicy.getFromFuture(f, () -> this + "-flushStateMachineData");
                }
                this.out.flush();
                if (!this.stateMachineDataPolicy.isSync()) {
                    IOUtils.getFromFuture(f, () -> this + "-flushStateMachineData");
                }
            }
            finally {
                timerContext.stop();
            }
            this.updateFlushedIndex();
        }
    }

    private void updateFlushedIndex() {
        LOG.debug("{}: updateFlushedIndex {} -> {}", new Object[]{this.name, this.flushedIndex, this.lastWrittenIndex});
        this.flushedIndex = this.lastWrittenIndex;
        this.pendingFlushNum = 0;
        Optional.ofNullable(this.submitUpdateCommitEvent).ifPresent(Runnable::run);
    }

    void startLogSegment(long startIndex) {
        LOG.info("{}: Starting segment from index:{}", (Object)this.name, (Object)startIndex);
        this.addIOTask(new StartLogSegment(startIndex));
    }

    void rollLogSegment(LogSegment segmentToClose) {
        LOG.info("{}: Rolling segment {} to index:{}", new Object[]{this.name, segmentToClose.toString(), segmentToClose.getEndIndex()});
        this.addIOTask(new FinalizeLogSegment(segmentToClose));
        this.addIOTask(new StartLogSegment(segmentToClose.getEndIndex() + 1L));
    }

    SegmentedRaftLog.Task writeLogEntry(RaftProtos.LogEntryProto entry) {
        return this.addIOTask(new WriteLog(entry));
    }

    SegmentedRaftLog.Task truncate(RaftLogCache.TruncationSegments ts, long index) {
        LOG.info("{}: Truncating segments {}, start index {}", new Object[]{this.name, ts, index});
        return this.addIOTask(new TruncateLog(ts, index));
    }

    long getFlushedIndex() {
        return this.flushedIndex;
    }

    private class TruncateLog
    extends SegmentedRaftLog.Task {
        private final RaftLogCache.TruncationSegments segments;
        private final long truncateIndex;

        TruncateLog(RaftLogCache.TruncationSegments ts, long index) {
            this.segments = ts;
            this.truncateIndex = index;
        }

        @Override
        void execute() throws IOException {
            IOUtils.cleanup(null, (Closeable[])new Closeable[]{RaftLogWorker.this.out});
            RaftLogWorker.this.out = null;
            CompletableFuture<Void> stateMachineFuture = null;
            if (RaftLogWorker.this.stateMachine != null) {
                stateMachineFuture = RaftLogWorker.this.stateMachine.truncateStateMachineData(this.truncateIndex);
            }
            if (this.segments.toTruncate != null) {
                File fileToTruncate = this.segments.toTruncate.isOpen ? RaftLogWorker.this.storage.getStorageDir().getOpenLogFile(this.segments.toTruncate.startIndex) : RaftLogWorker.this.storage.getStorageDir().getClosedLogFile(this.segments.toTruncate.startIndex, this.segments.toTruncate.endIndex);
                Preconditions.assertTrue((boolean)fileToTruncate.exists(), (String)"File %s to be truncated does not exist", (Object[])new Object[]{fileToTruncate});
                FileUtils.truncateFile((File)fileToTruncate, (long)this.segments.toTruncate.targetLength);
                File dstFile = RaftLogWorker.this.storage.getStorageDir().getClosedLogFile(this.segments.toTruncate.startIndex, this.segments.toTruncate.newEndIndex);
                Preconditions.assertTrue((!dstFile.exists() ? 1 : 0) != 0, (String)"Truncated file %s already exists ", (Object[])new Object[]{dstFile});
                FileUtils.move((File)fileToTruncate, (File)dstFile);
                LOG.info("{}: Truncated log file {} to length {} and moved it to {}", new Object[]{RaftLogWorker.this.name, fileToTruncate, this.segments.toTruncate.targetLength, dstFile});
                RaftLogWorker.this.lastWrittenIndex = this.segments.toTruncate.newEndIndex;
            }
            if (this.segments.toDelete != null && this.segments.toDelete.length > 0) {
                long minStart = this.segments.toDelete[0].startIndex;
                for (RaftLogCache.SegmentFileInfo del : this.segments.toDelete) {
                    File delFile = del.isOpen ? RaftLogWorker.this.storage.getStorageDir().getOpenLogFile(del.startIndex) : RaftLogWorker.this.storage.getStorageDir().getClosedLogFile(del.startIndex, del.endIndex);
                    Preconditions.assertTrue((boolean)delFile.exists(), (String)"File %s to be deleted does not exist", (Object[])new Object[]{delFile});
                    FileUtils.deleteFile((File)delFile);
                    LOG.info("{}: Deleted log file {}", (Object)RaftLogWorker.this.name, (Object)delFile);
                    minStart = Math.min(minStart, del.startIndex);
                }
                if (this.segments.toTruncate == null) {
                    RaftLogWorker.this.lastWrittenIndex = minStart - 1L;
                }
            }
            if (stateMachineFuture != null) {
                IOUtils.getFromFuture(stateMachineFuture, () -> this + "-truncateStateMachineData");
            }
            RaftLogWorker.this.updateFlushedIndex();
        }

        @Override
        long getEndIndex() {
            if (this.segments.toTruncate != null) {
                return this.segments.toTruncate.newEndIndex;
            }
            if (this.segments.toDelete.length > 0) {
                return this.segments.toDelete[this.segments.toDelete.length - 1].endIndex;
            }
            return -1L;
        }

        @Override
        public String toString() {
            return super.toString() + ": " + this.segments;
        }
    }

    private class StartLogSegment
    extends SegmentedRaftLog.Task {
        private final long newStartIndex;

        StartLogSegment(long newStartIndex) {
            this.newStartIndex = newStartIndex;
        }

        @Override
        void execute() throws IOException {
            File openFile = RaftLogWorker.this.storage.getStorageDir().getOpenLogFile(this.newStartIndex);
            Preconditions.assertTrue((!openFile.exists() ? 1 : 0) != 0, (String)"open file %s exists for %s", (Object[])new Object[]{openFile, RaftLogWorker.this.name});
            Preconditions.assertTrue((RaftLogWorker.this.out == null && RaftLogWorker.this.pendingFlushNum == 0 ? 1 : 0) != 0);
            RaftLogWorker.this.out = new LogOutputStream(openFile, false, RaftLogWorker.this.segmentMaxSize, RaftLogWorker.this.preallocatedSize, RaftLogWorker.this.bufferSize);
            Preconditions.assertTrue((boolean)openFile.exists(), (String)"Failed to create file %s for %s", (Object[])new Object[]{openFile.getAbsolutePath(), RaftLogWorker.this.name});
            LOG.info("{}: created new log segment {}", (Object)RaftLogWorker.this.name, (Object)openFile);
        }

        @Override
        long getEndIndex() {
            return this.newStartIndex;
        }
    }

    private class FinalizeLogSegment
    extends SegmentedRaftLog.Task {
        private final long startIndex;
        private final long endIndex;

        FinalizeLogSegment(LogSegment segmentToClose) {
            Preconditions.assertTrue((segmentToClose != null ? 1 : 0) != 0, (Object)"Log segment to be rolled is null");
            this.startIndex = segmentToClose.getStartIndex();
            this.endIndex = segmentToClose.getEndIndex();
        }

        @Override
        public void execute() throws IOException {
            IOUtils.cleanup((Logger)LOG, (Closeable[])new Closeable[]{RaftLogWorker.this.out});
            RaftLogWorker.this.out = null;
            File openFile = RaftLogWorker.this.storage.getStorageDir().getOpenLogFile(this.startIndex);
            Preconditions.assertTrue((boolean)openFile.exists(), () -> RaftLogWorker.this.name + ": File " + openFile + " to be rolled does not exist");
            if (this.endIndex - this.startIndex + 1L > 0L) {
                File dstFile = RaftLogWorker.this.storage.getStorageDir().getClosedLogFile(this.startIndex, this.endIndex);
                Preconditions.assertTrue((!dstFile.exists() ? 1 : 0) != 0);
                FileUtils.move((File)openFile, (File)dstFile);
                LOG.info("{}: Rolled log segment from {} to {}", new Object[]{RaftLogWorker.this.name, openFile, dstFile});
            } else {
                FileUtils.deleteFile((File)openFile);
                LOG.info("{}: Deleted empty log segment {}", (Object)RaftLogWorker.this.name, (Object)openFile);
            }
            RaftLogWorker.this.updateFlushedIndex();
        }

        @Override
        long getEndIndex() {
            return this.endIndex;
        }

        @Override
        public String toString() {
            return super.toString() + ": startIndex=" + this.startIndex + " endIndex=" + this.endIndex;
        }
    }

    private class WriteLog
    extends SegmentedRaftLog.Task {
        private final RaftProtos.LogEntryProto entry;
        private final CompletableFuture<?> stateMachineFuture;
        private final CompletableFuture<Long> combined;

        WriteLog(RaftProtos.LogEntryProto entry) {
            this.entry = ServerProtoUtils.removeStateMachineData(entry);
            if (this.entry == entry || RaftLogWorker.this.stateMachine == null) {
                this.stateMachineFuture = null;
            } else {
                try {
                    this.stateMachineFuture = RaftLogWorker.this.stateMachine.writeStateMachineData(entry);
                }
                catch (Throwable e) {
                    LOG.error(RaftLogWorker.this.name + ": writeStateMachineData failed for index " + entry.getIndex() + ", entry=" + ServerProtoUtils.toLogEntryString(entry), e);
                    throw e;
                }
            }
            this.combined = this.stateMachineFuture == null ? super.getFuture() : super.getFuture().thenCombine(this.stateMachineFuture, (index, stateMachineResult) -> index);
        }

        @Override
        int getSerializedSize() {
            return ServerProtoUtils.getSerializedSize(this.entry);
        }

        @Override
        CompletableFuture<Long> getFuture() {
            return this.combined;
        }

        @Override
        public void execute() throws IOException {
            if (RaftLogWorker.this.stateMachineDataPolicy.isSync() && this.stateMachineFuture != null) {
                RaftLogWorker.this.stateMachineDataPolicy.getFromFuture(this.stateMachineFuture, () -> this + "-writeStateMachineData");
            }
            Preconditions.assertTrue((RaftLogWorker.this.out != null ? 1 : 0) != 0);
            Preconditions.assertTrue((RaftLogWorker.this.lastWrittenIndex + 1L == this.entry.getIndex() ? 1 : 0) != 0, (String)"lastWrittenIndex == %s, entry == %s", (Object[])new Object[]{RaftLogWorker.this.lastWrittenIndex, this.entry});
            RaftLogWorker.this.out.write(this.entry);
            RaftLogWorker.this.lastWrittenIndex = this.entry.getIndex();
            RaftLogWorker.this.pendingFlushNum++;
            if (RaftLogWorker.this.shouldFlush()) {
                RaftLogWorker.this.flushWrites();
            }
        }

        @Override
        long getEndIndex() {
            return this.entry.getIndex();
        }

        @Override
        public String toString() {
            return super.toString() + ": " + ServerProtoUtils.toLogEntryString(this.entry);
        }
    }

    static class StateMachineDataPolicy {
        private final boolean sync;
        private final TimeDuration syncTimeout;
        private final int syncTimeoutRetry;

        StateMachineDataPolicy(RaftProperties properties) {
            this.sync = RaftServerConfigKeys.Log.StateMachineData.sync(properties);
            this.syncTimeout = RaftServerConfigKeys.Log.StateMachineData.syncTimeout(properties);
            this.syncTimeoutRetry = RaftServerConfigKeys.Log.StateMachineData.syncTimeoutRetry(properties);
            Preconditions.assertTrue((this.syncTimeoutRetry >= -1 ? 1 : 0) != 0);
        }

        boolean isSync() {
            return this.sync;
        }

        void getFromFuture(CompletableFuture<?> future, Supplier<Object> getName) throws IOException {
            Preconditions.assertTrue((boolean)this.isSync());
            TimeoutIOException lastException = null;
            for (int retry = 0; this.syncTimeoutRetry == -1 || retry <= this.syncTimeoutRetry; ++retry) {
                try {
                    IOUtils.getFromFuture(future, getName, (TimeDuration)this.syncTimeout);
                    return;
                }
                catch (TimeoutIOException e) {
                    LOG.warn("Timeout " + retry + (this.syncTimeoutRetry == -1 ? "/~" : "/" + this.syncTimeoutRetry), (Throwable)e);
                    lastException = e;
                    continue;
                }
            }
            Objects.requireNonNull(lastException, "lastException == null");
            throw lastException;
        }
    }
}

